/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef HEADERLISTVIEW_H
#define HEADERLISTVIEW_H

#include "klistview.h"
#include "mailodybaselistview.h"
#include "messagedata.h"

namespace Mailody {

class HeaderListViewItem;

/**
 * @class HeaderListView
 * This class is a KListView which is meant to hold all the headers
 * in a particular folder. It is reimplemented to get some control over
 * the keys
 * @author Tom Albers <tomalbers@kde.nl>
 */
class HeaderListView : public KListView
{
    Q_OBJECT

    public:
        /** Constructor */
        explicit HeaderListView( QWidget* );

        /** dragobject */
        QDragObject *dragObject();

        /** Indicates if deleted messages should be shown or not */
        void setHideDeleted(bool i);

        /** Returns if deleted messages are hidden */
        bool getHideDeleted() { return m_hideDeleted; };

        /** Clears the view and only show @p item and its parents */
        void isolateItem( HeaderListViewItem* );

        /** Revert to the complete view */
        void clearIsolation();

        /** Query to find out if there is an isolation active */
        bool isolationActive() const {return m_isolationActive; };

        /**
         * Standard headerlistviewitems will only parse the message when
         * there is a draw to do on the screen. If you want to search, we
         * need to force all headers of that mailbox to be parsed.
         */
        void allHeadersVisible();

        /**
         * go to the next unread message, if there is one selected it will
         * use that as starting point and start from the top if none is found
         * in the remaining part.
         */
        HeaderListViewItem* nextUnread();

    protected:
        /**
         * Reimplemented for key nav. Up will emit scrollUp(), Down key
         * will emit scrollDown()
         */
        void keyPressEvent( QKeyEvent * );

    signals:
        /**
         * Emitted when user presses the down key, so you can scoll a message
         * down for example.
         */
        void scrollDown();

        /**
         * Emitted when user presses the up key, so you can scoll a message
         * up for example.
         */
        void scrollUp();

        /**
         * Emitted when user presses the page down key, so you can scoll a
         * message down for example.
         */
        void scrollPageDown();

        /**
         * Emitted when user presses the page up key, so you can scoll a
         * message up for example.
         */
        void scrollPageUp();

    private:
        void dragInfo(HeaderListViewItem*, QString&, bool);
        bool                                m_hideDeleted;
        bool                                m_isolationActive;
        QValueList< HeaderListViewItem* >   m_hiddenItems;

};

/**
 * @class HeaderListViewItem
 * This class is a KListViewItem which is meant to hold a header
 * in a particular folder. It is reimplemented to get some control over
 * the keys, paint the cells based on the labels or state of a message.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class HeaderListViewItem : public MailodyBaseListViewItem
{
    public:
        /**
         * Constructor
         * @param lv The listview to which this item should be inserted into
         * @param md The message which this headerlistviewitem should be
         *           created for
         */
        explicit HeaderListViewItem( KListView* lv, MessageData* md);

        /**
         * Constructor
         * @param lvi The headerlistviewitem to which this item should be
         *           a child.
         * @param md The message which this headerlistviewitem should be
         *           created for
         */
        explicit HeaderListViewItem( HeaderListViewItem* lvi,
                                     MessageData* md);

        /**
         * Destructor. Please note that this will delete the MessageData.
         */
        ~HeaderListViewItem();

        /**
         * By default headers are not parsed untill the paintCell() call.
         * A call to this method will parse them.
         */
        void parseHeaders();

        /**
         * Reimplemented to give it a color of the message has a label,
         * strike through if deleted, etc.
         */
        virtual void paintCell( QPainter * p, const QColorGroup & cg,
                                int column, int width, int align );

        /**
         * Reimplemented to arrange a good order. This way the user gets to
         * see a date, but it is sorted on an internal data, or get a
         * To: message in the headerlist, but ignore that when sorting.
         */
        virtual int compare( QListViewItem *i, int col,
                             bool ascending ) const;

        /**
         * Returns the MessageData for which this item was created
         */
        MessageData* msg() const { return m_msg; };

    private:
        MessageData  *m_msg;
        bool         m_dataNotSet;
};

}

#endif
