/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This file contains data structures and functions used for grammatical 
 * analysis. */

/* types ====================================================================*/

#ifndef ANALYSIS
#define ANALYSIS
typedef enum {MORPHOLOGY, SYNTAX} grammar_t; /* types of grammatic analysis */

typedef enum {BREAK_NODE, FINAL_NODE, 
	      INTER_NODE, PRUNED_NODE} tree_node_type_t;

typedef struct /* a node of the analysis tree */
{
  int_t index; /* index of this analysis tree node */
  tree_node_type_t type;
  int_t mother_index; /* -1 or index of mother analysis tree node */
  string_t rule_name; /* NULL or name of the rule that created result */
  string_t right_surf; /* NULL or right surface */
  value_t right_cat; /* NULL or right cat */
  string_t result_surf; /* result surface */
  value_t result_cat; /* NULL or result cat */
  string_t rule_set; /* NULL or successor rules */
} analysis_node_t;
#endif

typedef enum {ROBUST_OPTION, PRUNING_OPTION, CACHE_OPTION, 
	      MOR_OUT_FILTER_OPTION, SYN_IN_FILTER_OPTION, 
	      SYN_OUT_FILTER_OPTION, NUM_ANALYSIS_OPTIONS} analysis_option_t;

typedef enum {LEFT_SURFACE, RIGHT_SURFACE, RESULT_SURFACE} surface_t;
/* Kinds of surfaces that can be obtained by "get_rule_info" in "rules.h" */

/* variables ================================================================*/

GLOBAL rule_sys_t *rule_system[2];
/* rule_system[MORPHOLOGY] and rule_system[SYNTAX]. READ ONLY! */

GLOBAL grammar_t top_grammar; /* grammar of last analysis. READ ONLY! */

GLOBAL bool_t recognised_by_combi_rules;
/* TRUE if last analysis was recognised by combi rules */

GLOBAL bool_t recognised_by_robust_rule; 
/* TRUE iflast analysis was recognised by robust rule */

GLOBAL void (*debug_state) (int_t index);
/* Callback function for "analyse", is called when successor rules for
 * state with analysis node inde <index> will be executed. */

GLOBAL string_t (*get_surface) (surface_t surface_type);
/* Return surface <surface_type> for currently executed rule.
 * The result must be freed after use. */

GLOBAL string_t last_analysis_input; 
/* Start of top level input string. READ ONLY! */

/* functions ================================================================*/

extern void init_analysis (string_t morphology_file, string_t syntax_file);
/* Initialise the analysis module.
 * <morphology_file> and <syntax_file> are the rule files to load. 
 * <syntax_file> may be NULL. */

extern void terminate_analysis (void);
/* Free analysis module. */

extern void preprocess_input (string_t input);
/* Delete heading and trailing spaces in <input>
 * and compress all whitespace sequences to a single space */

extern void set_analysis_option (analysis_option_t selected, bool_t setting);
/* set analysis option <selected> to <setting>. */

extern bool_t get_analysis_option (analysis_option_t selected);
/* return the current setting of analysis option <selected>. */

extern void analyse (grammar_t grammar, 
		     string_t input, 
		     bool_t build_tree,
		     bool_t analyse_all);
/* Perform a LAG analysis of <input> using <grammar> (MORPHOLOGY or SYNTAX).
 * An analysis tree will be built if <build_tree> == TRUE.
 * The whole input will be analysed if <analyse_all> == TRUE. */

extern bool_t reset_analysis_results (void);
/* Restart to read analysis results. 
 * Return TRUE iff there are any analysis results. */

extern value_t get_next_analysis_result (void);
/* Return the category of the next analysis result.
 * Return NULL if there are no more results. */

extern bool_t reset_analysis_nodes (void);
/* Restart to read analysis nodes. 
 * Return TRUE iff there are any analysis nodes. */

extern analysis_node_t *get_next_analysis_node (void);
/* Return the next analysis tree node of the last call of "analyse_item".
 * Return NULL if there is no more node. 
 * The node must be freed with "free_analysis_node" after use. */

extern void free_analysis_node (analysis_node_t **node);
/* Free the memory occupied by <node>. */

/* end of file ==============================================================*/
