/*
 *  The ManaPlus Client
 *  Copyright (C) 2009  The Mana World Development Team
 *  Copyright (C) 2009-2010  The Mana Developers
 *  Copyright (C) 2011-2014  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gui/widgets/tabs/whispertab.h"

#include "chatlogger.h"
#include "commands.h"

#include "being/localplayer.h"

#include "net/chathandler.h"
#include "net/net.h"

#include "gui/windows/chatwindow.h"

#include "gui/widgets/windowcontainer.h"

#include "debug.h"

WhisperTab::WhisperTab(const Widget2 *const widget,
                       const std::string &nick) :
    ChatTab(widget, nick, ""),
    mNick(nick)
{
    setWhisperTabColors();
}

WhisperTab::~WhisperTab()
{
    if (chatWindow)
        chatWindow->removeWhisper(mNick);
}

void WhisperTab::handleInput(const std::string &msg)
{
    std::string newMsg;
    newMsg = ChatWindow::doReplace(msg);
    Net::getChatHandler()->privateMessage(mNick, newMsg);

    if (localPlayer)
        chatLog(localPlayer->getName(), newMsg);
    else
        chatLog("?", newMsg);
}

void WhisperTab::handleCommand(const std::string &msg)
{
    if (msg == "close")
    {
        delete this;
        return;
    }

    const size_t pos = msg.find(' ');
    const std::string type(msg, 0, pos);
    const std::string args(msg, pos == std::string::npos
        ? msg.size() : pos + 1);

    if (type == "me")
    {
        std::string str = strprintf("*%s*", args.c_str());
        Net::getChatHandler()->privateMessage(mNick, str);
        if (localPlayer)
            chatLog(localPlayer->getName(), str);
        else
            chatLog("?", str);
    }
    else
    {
        ChatTab::handleCommand(msg);
    }
}

bool WhisperTab::handleCommand(const std::string &restrict type,
                               const std::string &restrict args A_UNUSED)
{
    if (type == "close")
    {
        if (windowContainer)
            windowContainer->scheduleDelete(this);
        else
            delete this;
        if (chatWindow)
            chatWindow->defaultTab();
    }
    else if (type == "ignore")
    {
        Commands::ignore(mNick, this);
    }
    else if (type == "unignore")
    {
        Commands::unignore(mNick, this);
    }
    else
    {
        return false;
    }

    return true;
}

void WhisperTab::saveToLogFile(const std::string &msg) const
{
    if (chatLogger)
        chatLogger->log(getNick(), msg);
}

void WhisperTab::getAutoCompleteList(StringVect &names) const
{
    names.push_back(mNick);
}

void WhisperTab::setWhisperTabColors()
{
    setTabColor(&getThemeColor(Theme::WHISPER_TAB),
        &getThemeColor(Theme::WHISPER_TAB_OUTLINE));
    setHighlightedTabColor(&getThemeColor(Theme::WHISPER_TAB_HIGHLIGHTED),
        &getThemeColor(Theme::WHISPER_TAB_HIGHLIGHTED_OUTLINE));
    setSelectedTabColor(&getThemeColor(Theme::WHISPER_TAB_SELECTED),
        &getThemeColor(Theme::WHISPER_TAB_SELECTED_OUTLINE));
}

void WhisperTab::setWhisperTabOfflineColors()
{
    setTabColor(&getThemeColor(Theme::WHISPER_TAB_OFFLINE),
        &getThemeColor(Theme::WHISPER_TAB_OFFLINE_OUTLINE));
    setHighlightedTabColor(&getThemeColor(
        Theme::WHISPER_TAB_OFFLINE_HIGHLIGHTED), &getThemeColor(
        Theme::WHISPER_TAB_OFFLINE_HIGHLIGHTED_OUTLINE));
    setSelectedTabColor(&getThemeColor(Theme::WHISPER_TAB_OFFLINE_SELECTED),
        &getThemeColor(Theme::WHISPER_TAB_OFFLINE_SELECTED_OUTLINE));
}
