#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/strexp.h"

#include "guiutils.h"
#include "cdialog.h"

#include "editor.h"
#include "editorfip.h"
#include "editorfipop.h"
#include "editorop.h"

#include "manedit.h"


/*
 *      Tempory structure to return matched information.
 */
typedef struct {
	
	int line_num;
	int start_pos, end_pos;
	
} fip_match_struct;


static fip_match_struct *EditorFIPDoSearchInArray(
	const char **line, int total_lines,
	const char *string, int *matches_rtn
);

static int EditorFIPDoFindAnyBranchIteration(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	GtkCTreeNode *branch,
	char **strv, int strc, gbool case_sensitive,
	const char *manual_page_name
);
static int EditorFIPDoFindAllBranchIteration(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	GtkCTreeNode *branch,
	char **strv, int strc, gbool case_sensitive,
	const char *manual_page_name
);

static int EditorFIPDoFindAny(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	const char *string, gbool case_sensitive, gbool explode_string
);
static int EditorFIPDoFindAll(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	const char *string, gbool case_sensitive, gbool explode_string
);
int EditorFIPDoFind(
	editor_fip_struct *fip,
	editor_struct *editor,
	int throughness,        /* 0 = match all, 1 = match any. */
	const char *string, 
	gbool case_sensitive
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Searches for the given string in the given line arrays.
 *
 *	Returns a dynamically allocated list of fip_match_struct or NULL
 *	for no matches. Return value needs to be free'ed by the calling
 *	function.
 *
 *	Value for matches_rtn will indicate the number of returned
 *	matches (or 0 if return is NULL).
 */
static fip_match_struct *EditorFIPDoSearchInArray(
	const char **line, int total_lines,
	const char *string, int *matches_rtn
)
{
	int i, n;
	const char *cstrptr;
	const char *line_ptr;
	fip_match_struct *match = NULL;
	int buf_len, total_matches = 0;


	/* Update returns. */
	if(matches_rtn != NULL)
	    (*matches_rtn) = total_matches;

	if((line == NULL) || (string == NULL) || (total_lines < 1))
	    return(match);

	if((*string) == '\0')
	    return(match);

	for(i = 0, buf_len = 0; i < total_lines; i++)
	{
	    line_ptr = (const char *)line[i];
	    if(line_ptr == NULL)
		continue;

	    /* Is string found on this line? */
	    cstrptr = (const char *)strstr(line_ptr, string);
	    if(cstrptr == NULL)
	    {
		/* Add the length of this line plus one for the newline
		 * character that is not there.
		 */
		buf_len += strlen(line_ptr) + 1;
	    }
	    else
	    {
		/* Got match! */
		int sp, ep;

		sp = buf_len + (int)(cstrptr - line_ptr);
		ep = sp + strlen(string);

		/* Allocate a new return. */
		n = total_matches;
		total_matches = n + 1;

		match = (fip_match_struct *)realloc(
		    match,
		    total_matches * sizeof(fip_match_struct)
		);
		if(match == NULL)
		{
		    total_matches = 0;
		    break;
		}
		else
		{
		    /* Set new match structure values. */
		    fip_match_struct *match_ptr = &(match[n]);

		    match_ptr->line_num = i;
		    match_ptr->start_pos = sp;
		    match_ptr->end_pos = ep;
		}

		/* Add the length of this line plus one for the newline
		 * character that is not there.   
		 */
		buf_len += strlen(line_ptr) + 1;
	    }
	}

	/* Update returns. */
	if(matches_rtn != NULL)
	    (*matches_rtn) = total_matches;

	return(match);
}

/*
 *      Called by EditorFIPDoFindAny() to do find on a branch and
 *      recurse into all child branches if any.
 *
 *	Inputs assumed valid, will test stop_find_count on dialog and
 *	react appropriatly.
 *
 *      Returns the number of matches.
 */
static int EditorFIPDoFindAnyBranchIteration(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	GtkCTreeNode *branch,
	char **strv, int strc, gbool case_sensitive,
	const char *manual_page_name
)
{
	int i, n, new_matches, match_count = 0;
	char **line;
	GtkCTreeRow *branch_row;
	editor_item_struct *item_ptr;
	fip_match_struct *match, *match_ptr;


	if(branch == NULL)
	    return(match_count);

	if(fip->stop_find_count > 0)
	    return(match_count);

	EditorFIPSetStatusProgress(fip, -1.0);

	/* Get item data from branch. */
	item_ptr = EditorBranchGetData(layout_ctree, branch);
	if(item_ptr != NULL)
	{
	    char *tmp_buf;
	    int tmp_buf_len;

	    /* Search through item data by its type. */
	    switch(item_ptr->type)
	    {
	      case EditorItemTypeFile:
		/* Allocate message. */
		tmp_buf_len = ((item_ptr->file_name == NULL) ?
		    0 : strlen(item_ptr->file_name)) + 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: Manual Page \"%s\"",
			item_ptr->file_name
		    );

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);
		tmp_buf = NULL;

		/* Nothing to search. */

		break;

	       case EditorItemTypeHeader:
		/* Allocate message. */
		tmp_buf_len = 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: Header");

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);
		tmp_buf = NULL;


		/* Create a line array with just one pointer. */
		line = (char **)malloc(1 * sizeof(char *));
		if(line != NULL)
		{
		    /* Search through header name. */
		    line[0] = ((item_ptr->header_name == NULL) ?
			NULL : strdup(item_ptr->header_name)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0, n = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match != NULL)
			{
			    n++;
			    free(match);
			}
		    }
		    if(n > 0)
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_name,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }

		    /* Search through header section number. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_section_number == NULL) ?
			NULL : strdup(item_ptr->header_section_number)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0, n = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match != NULL)
			{
			    n++;
			    free(match);
			}
		    }
		    if(n > 0)
		    {
			EditorFIPAddItem(   
			    fip,
			    item_ptr->header_section_number,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }
		
		    /* Search through header version. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_version == NULL) ?
			NULL : strdup(item_ptr->header_version)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0, n = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match != NULL)
			{
			    n++;
			    free(match);
			}
		    }
		    if(n > 0)
		    {
			EditorFIPAddItem(   
			    fip,
			    item_ptr->header_version,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }
		
		    /* Search through header author. */ 
		    free(line[0]);
		    line[0] = ((item_ptr->header_author == NULL) ? 
			NULL : strdup(item_ptr->header_author) 
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0, n = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match != NULL)
			{   
			    n++;
			    free(match);
			}
		    }
		    if(n > 0)
		    {
			EditorFIPAddItem(   
			    fip,
			    item_ptr->header_author, 
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }
		
		    /* Search through header catagory. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_catagory == NULL) ?
			NULL : strdup(item_ptr->header_catagory)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0, n = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match != NULL)
			{   
			    n++;
			    free(match);
			}
		    }
		    if(n > 0)
		    {
			EditorFIPAddItem(   
			    fip,
			    item_ptr->header_catagory,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }

		    /* Free coppied lines array. */
		    free(line[0]);
		    free(line);
		    line = NULL;
		}


		/* Begin searching through header comments. */

		/* Make a duplicate of each line. */
		line = strlistcopy(
		    (const char **)item_ptr->line, item_ptr->total_lines
		);
		if(!case_sensitive && (line != NULL))
		{
		    for(i = 0; i < item_ptr->total_lines; i++)
			strtoupper(line[i]);
		}

		/* Search through each coppied line, recording any
		 * occurance of each search string
		 */
		for(i = 0; i < strc; i++)
		{
		    EditorFIPSetStatusProgress(fip, -1.0);

		    match = EditorFIPDoSearchInArray(
			(const char **)line, item_ptr->total_lines,
			strv[i], &new_matches
		    );
		    match_count += new_matches;
		    for(n = 0; n < new_matches; n++)
		    {
			match_ptr = &(match[n]);

			EditorFIPAddItem(
			    fip,
			    ((match_ptr->line_num > -1) ?
				item_ptr->line[match_ptr->line_num] : ""
			    ),
			    match_ptr->line_num,
			    "Header",
			    manual_page_name,
			    branch,		/* Found branch. */
			    (void *)item_ptr,	/* Branch data. */
			    match_ptr->start_pos, match_ptr->end_pos
			);
		    }

		    /* Deallocate match structures list return. */
		    free(match);
		    match = NULL;
		    new_matches = 0;
		}

		/* Deallocate coppied lines. */
		strlistfree(line, item_ptr->total_lines);
		line = NULL;
		break;

	      case EditorItemTypeSection:
		/* Allocate message. */
		tmp_buf_len = ((item_ptr->section_name == NULL) ?
		    0 : strlen(item_ptr->section_name)
		) + 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: \"%s\"", item_ptr->section_name);

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);
		tmp_buf = NULL;

		/* Make a duplicate of each line. */
		line = strlistcopy(
		    (const char **)item_ptr->line, item_ptr->total_lines
		);
		if(!case_sensitive && (line != NULL))
		{
		    for(i = 0; i < item_ptr->total_lines; i++)
			strtoupper(line[i]);
		}

		/* Search through each coppied line, recording any
		 * occurance of each search string.
		 */
		for(i = 0; i < strc; i++)
		{
		    EditorFIPSetStatusProgress(fip, -1.0);

		    match = EditorFIPDoSearchInArray(
			(const char **)line, item_ptr->total_lines,
			strv[i], &new_matches
		    );
		    match_count += new_matches;
		    for(n = 0; n < new_matches; n++)
		    {
			match_ptr = &(match[n]);

			EditorFIPAddItem(
			    fip,
			    ((match_ptr->line_num > -1) ?
				item_ptr->line[match_ptr->line_num] : ""
			    ),
			    match_ptr->line_num,
			    ((item_ptr->section_name == NULL) ?
				"(null)" : item_ptr->section_name
			    ),
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    match_ptr->start_pos, match_ptr->end_pos
			);
		    }

		    /* Deallocate match structures list return. */
		    free(match);
		    match = NULL;
		    new_matches = 0;
		}

		/* Deallocate coppied lines. */
		strlistfree(line, item_ptr->total_lines);
		line = NULL;

		break;
	    }
	}

	/* Get branch row. */
	branch_row = GTK_CTREE_ROW(branch);

	/* Branch has children? */
	if(branch_row->children != NULL)
	{
	    /* Set branch to point to first child. */
	    branch = branch_row->children;

	    while(branch != NULL)
	    {
		match_count += EditorFIPDoFindAnyBranchIteration(
		    fip, editor,
		    results_clist, layout_ctree,
		    branch,
		    strv, strc, case_sensitive,
		    manual_page_name
		);
		branch_row = GTK_CTREE_ROW(branch);
		branch = ((branch_row == NULL) ? NULL : branch_row->sibling);
	    }
	}

	return(match_count);
}

/*
 *	Called by EditorFIPDoFindAll() to do find on a branch and
 *	recurse into all child branches if any.
 *
 *      Inputs assumed valid, will test stop_find_count on dialog and
 *      react appropriatly.
 *
 *	Returns the number of matches.
 */
static int EditorFIPDoFindAllBranchIteration(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	GtkCTreeNode *branch,
	char **strv, int strc, gbool case_sensitive,
	const char *manual_page_name
)
{
	int i, new_matches, match_count = 0;
	char **line;
	GtkCTreeRow *branch_row;
	editor_item_struct *item_ptr;
	fip_match_struct *match;


	if(branch == NULL)
	    return(match_count);

	if(fip->stop_find_count > 0)
	    return(match_count);

	EditorFIPSetStatusProgress(fip, -1.0);

	/* Get item data from branch. */
	item_ptr = EditorBranchGetData(layout_ctree, branch);
	if(item_ptr != NULL)
	{
	    char *tmp_buf;
	    int tmp_buf_len;

	    /* Search through item data by its type. */
	    switch(item_ptr->type)
	    {
	      case EditorItemTypeFile:
		/* Allocate message. */
		tmp_buf_len = ((item_ptr->file_name == NULL) ?
		    0 : strlen(item_ptr->file_name)) + 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: Manual Page \"%s\"",
			item_ptr->file_name
		    );

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);   
		tmp_buf = NULL;

		/* Nothing to search. */

		break;

	      case EditorItemTypeHeader:
		/* Allocate message. */
		tmp_buf_len = 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: Header");

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);
		tmp_buf = NULL;


		/* Create a line array with just one pointer. */
		line = (char **)malloc(1 * sizeof(char *));
		if(line != NULL)
		{
		    /* Search through header name. */
		    line[0] = ((item_ptr->header_name == NULL) ?
			NULL : strdup(item_ptr->header_name)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match == NULL)
			    break;

			/* Deallocate match structures list return. */
			free(match);
		    }
		    if(i >= strc)
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_name,
			    -1,			/* N/a. */
			    "Header",
			    manual_page_name,
			    branch,		/* Found branch. */
			    (void *)item_ptr,	/* Branch data. */
			    -1, -1
			);
		    }

		    /* Search through header section number. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_section_number == NULL) ?
			NULL : strdup(item_ptr->header_section_number)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match == NULL)
			    break;
		   
			/* Deallocate match structures list return. */
			free(match);
		    }
		    if(i >= strc)
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_section_number,
			    -1,			/* N/a. */
			    "Header",
			    manual_page_name,
			    branch,		/* Found branch. */
			    (void *)item_ptr,	/* Branch data. */
			    -1, -1
			);
		    }

		    /* Search through header version. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_version == NULL) ?
			NULL : strdup(item_ptr->header_version)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray(
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match == NULL)
			    break;

			/* Deallocate match structures list return. */
			free(match);
		    }
		    if(i >= strc)
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_version,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }

		    /* Search through header author. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_author == NULL) ?
			NULL : strdup(item_ptr->header_author)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray( 
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match == NULL)
			    break;

			/* Deallocate match structures list return. */
			free(match);
		    }
		    if(i >= strc)
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_author,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }

		    /* Search through header catagory. */
		    free(line[0]);
		    line[0] = ((item_ptr->header_catagory == NULL) ?
			NULL : strdup(item_ptr->header_catagory)
		    );
		    if(!case_sensitive)
			strtoupper(line[0]);
		    for(i = 0; i < strc; i++)
		    {
			match = EditorFIPDoSearchInArray( 
			    (const char **)line, 1,
			    strv[i], &new_matches
			);
			if(match == NULL)
			    break;

			/* Deallocate match structures list return. */
			free(match);
		    }
		    if(i >= strc) 
		    {
			EditorFIPAddItem(
			    fip,
			    item_ptr->header_catagory,
			    -1,                 /* N/a. */
			    "Header",
			    manual_page_name,
			    branch,             /* Found branch. */
			    (void *)item_ptr,   /* Branch data. */
			    -1, -1
			);
		    }

		    /* Free coppied lines array. */
		    free(line[0]);
		    free(line);
		    line = NULL;
		}


		/* Begin searching through header comments. */

		/* Make a duplicate of each line. */
		line = strlistcopy(
		    (const char **)item_ptr->line, item_ptr->total_lines
		);
		if(!case_sensitive && (line != NULL))
		{
		    for(i = 0; i < item_ptr->total_lines; i++)
			strtoupper(line[i]);
		}

		/* Search through each coppied line, recording only if
		 * all occurances all the search strings are in the line.
		 */
		for(i = 0; i < strc; i++)
		{
		    EditorFIPSetStatusProgress(fip, -1.0);

		    match = EditorFIPDoSearchInArray(
			(const char **)line, item_ptr->total_lines,
			strv[i], &new_matches
		    );
		    /* Break if one of the search strings is not in the
		     * lines array.
		     */
		    if(match == NULL)
			break;

		    /* Deallocate match structures list return. */
		    free(match);
		    match = NULL;
		    new_matches = 0;
		}
		if(i >= strc)
		{
		    /* All search strings matched! */
		    match_count++;

		    /* Consider this as one match. */
		    EditorFIPAddItem(
			fip,
			((item_ptr->total_lines > 0) ?
			    item_ptr->line[0] : ""
			),
			-1,		/* N/a. */
			"Header",
			manual_page_name,
			branch,			/* Found branch. */
			(void *)item_ptr,	/* Branch data. */
			-1, -1
		   );
		}

		/* Deallocate coppied lines. */
		strlistfree(line, item_ptr->total_lines);
		line = NULL;

		break;

	      case EditorItemTypeSection:
		/* Allocate message. */ 
		tmp_buf_len = ((item_ptr->section_name == NULL) ?
		    0 : strlen(item_ptr->section_name)
		) + 256;
		tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		if(tmp_buf != NULL)
		    sprintf(tmp_buf, "Searching: \"%s\"", item_ptr->section_name);

		EditorFIPSetStatusMessage(fip, tmp_buf);

		/* Deallocate message. */
		free(tmp_buf);
		tmp_buf = NULL;

		/* Make a duplicate of each line. */
		line = strlistcopy(
		    (const char **)item_ptr->line, item_ptr->total_lines
		);
		if(!case_sensitive && (line != NULL))
		{
		    for(i = 0; i < item_ptr->total_lines; i++)
			strtoupper(line[i]);
		}

		/* Search through each coppied line. */
		for(i = 0; i < strc; i++)
		{
		    EditorFIPSetStatusProgress(fip, -1.0);

		    match = EditorFIPDoSearchInArray(
			(const char **)line, item_ptr->total_lines,
			strv[i], &new_matches
		    );
		    /* Break if one of the search strings is not in the
		     * lines array.
		     */
		    if(match == NULL)
			break;

		    /* Deallocate match structures list return. */
		    free(match);
		    match = NULL;
		    new_matches = 0;
		}
		if(i >= strc)
		{
		    /* All search strings matched! */
		    match_count++;

		    /* Consider this as one match. */
		    EditorFIPAddItem(
			fip,
			((item_ptr->total_lines > 0) ?
			    item_ptr->line[0] : ""
			),
			-1,              /* N/a. */
			((item_ptr->section_name == NULL) ?
			    "(null)" : item_ptr->section_name
			),
			manual_page_name,
			branch,			/* Found branch. */
			(void *)item_ptr,	/* Branch data. */
			-1, -1
		    );
		}

		/* Deallocate coppied lines. */
		strlistfree(line, item_ptr->total_lines);
		line = NULL;

		break;
	    }
	}

	/* Get branch row. */
	branch_row = GTK_CTREE_ROW(branch);

	/* Branch has children? */
	if(branch_row->children != NULL)
	{
	    /* Set branch to point to first child. */
	    branch = branch_row->children;

	    while(branch != NULL)
	    {
		match_count += EditorFIPDoFindAllBranchIteration(
		    fip, editor,
		    results_clist, layout_ctree,
		    branch,
		    strv, strc, case_sensitive,
		    manual_page_name
		);
		branch_row = GTK_CTREE_ROW(branch);
		branch = ((branch_row == NULL) ? NULL : branch_row->sibling);
	    }
	}

	return(match_count);
}

/*
 *	Procedure to find any occurance of string on editor.
 *
 *	Inputs assumed valid.
 *
 *	Returns number of matches made.
 */
static int EditorFIPDoFindAny(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	const char *string, gbool case_sensitive, gbool explode_string
)
{
	int match_count = 0;
	GtkCTreeNode *branch;
	GtkCTreeRow *branch_row;
	char **strv;
	int strc;
	editor_item_struct *item_ptr;
	const char *manual_page_name;


	/* Get first branch on layout ctree. */
	branch = gtk_ctree_node_nth(layout_ctree, 0);
	if(branch == NULL)
	    return(match_count);

	/* Explode search string? */
	if(explode_string)
	{
	    strv = strexp(string, &strc);
	    if(strv == NULL)
		return(match_count);
	}
	else
	{
	    strc = 1;
	    strv = (char **)malloc(strc * sizeof(char *));
	    if(strv == NULL)
		return(match_count);
	    else
		strv[0] = strdup(string);
	}

	/* Convert each exploded string to upper case if not case
	 * sensitive.
	 */
	if(!case_sensitive)
	{
	    int i;

	    for(i = 0; i < strc; i++)
		strtoupper(strv[i]);
	}   

	/* Begin searching for all occurances through each branch. */
	while(branch != NULL)
	{
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();

	    if(fip->stop_find_count > 0)
		break;

	    /* Get item data pointer for branch. */
	    item_ptr = EditorBranchGetData(layout_ctree, branch);

	    /* Get manual page name from (assumed) toplevel branch
	     * item data.
	     */
	    manual_page_name = (const char *)(
		(item_ptr == NULL) ? NULL : item_ptr->file_name
	    );

	    /* Do find any for this branch and all its child branches. */
	    match_count += EditorFIPDoFindAnyBranchIteration(
		fip, editor,
		results_clist, layout_ctree,
		branch,
		strv, strc, case_sensitive,
		manual_page_name
	    );

	    /* Get branch row. */
	    branch_row = GTK_CTREE_ROW(branch);

	    /* Get sibling of branch and set it as new branch. */
	    branch = ((branch_row == NULL) ? NULL : branch_row->sibling);
	}

	/* Deallocate exploded search string. */
	strlistfree(strv, strc);

	return(match_count);
}

/*
 *	Procedure to find all occurances of string on editor.
 *
 *	Inputs assumed valid.
 *
 *	Returns number of matches made.
 */
static int EditorFIPDoFindAll(
	editor_fip_struct *fip, editor_struct *editor,
	GtkCList *results_clist, GtkCTree *layout_ctree,
	const char *string, gbool case_sensitive, gbool explode_string
)
{
	int match_count = 0;
	GtkCTreeNode *branch;
	GtkCTreeRow *branch_row;
	char **strv;
	int strc;
	editor_item_struct *item_ptr;
	const char *manual_page_name;


	/* Get first branch on layout ctree. */
	branch = gtk_ctree_node_nth(layout_ctree, 0);
	if(branch == NULL)
	    return(match_count);

	/* Explode search string? */
	if(explode_string)
	{
	    strv = strexp(string, &strc);
	    if(strv == NULL)
		return(match_count);
	}
	else
	{
	    strc = 1;
	    strv = (char **)malloc(strc * sizeof(char *));
	    if(strv == NULL)
		return(match_count);
	    else
		strv[0] = strdup(string);
	}

	/* Convert each exploded string to upper case if not case
	 * sensitive.
	 */
	if(!case_sensitive)
	{
	    int i;

	    for(i = 0; i < strc; i++)
		strtoupper(strv[i]);
	}

	/* Begin searching for all occurances through each branch. */
	while(branch != NULL)
	{
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();

	    if(fip->stop_find_count > 0)
		break;              

	    /* Get item data pointer for branch. */
	    item_ptr = EditorBranchGetData(layout_ctree, branch);
	    
	    /* Get manual page name from (assumed) toplevel branch
	     * item data.
	     */
	    manual_page_name = (const char *)(
		(item_ptr == NULL) ? NULL : item_ptr->file_name
	    );

	    /* Do find all for this branch and all its child branches. */
	    match_count += EditorFIPDoFindAllBranchIteration(
		fip, editor,
		results_clist, layout_ctree,
		branch,
		strv, strc, case_sensitive,
		manual_page_name
	    );

	    /* Get branch row. */
	    branch_row = GTK_CTREE_ROW(branch);

	    /* Get sibling of branch and set it as new branch. */
	    branch = ((branch_row == NULL) ? NULL : branch_row->sibling);
	}

	/* Deallocate exploded search string. */
	strlistfree(strv, strc);

	return(match_count);
}


/*
 *	Procedure to find currances of string in the given editor using
 *	the Find In Pages dialog procedure.
 *
 *	Does not clear the results clist.
 *
 *	Returns the number of matches.
 */
int EditorFIPDoFind(
	editor_fip_struct *fip, editor_struct *editor,
	int throughness,        /* 0 = match all, 1 = match any. */
	const char *string,
	gbool case_sensitive
)
{
	int matches_count = 0;
	GtkCList *results_clist;
	GtkCTree *layout_ctree;


	if((fip == NULL) || (editor == NULL) || (string == NULL))
	    return(matches_count);

	if(!fip->initialized)
	    return(matches_count);

	/* Dialog already processing? */
	if(fip->processing)
	    return(matches_count);

	/* Be sure to apply values of currently selected branch on the
	 * editor. This ensures we save the most up to date data.
	 */
	EditorDoApplyValues(editor, editor->selected_branch);


	/* Mark dialog as processing. */
	fip->processing = TRUE;

	/* Reset stop_find_count on dialog. */
	fip->stop_find_count = 0;

	/* Update menus and buttons to reflect processing. */
	EditorFIPUpdateMenus(fip);


	/* Get pointer to results clist on dialog (must be valid). */
	results_clist = (GtkCList *)fip->results_clist;
	if(results_clist == NULL)
	{
	    fip->processing = FALSE;
	    return(matches_count);
	}

	/* Get pointer to layout ctree on editor (must be valid). */
	layout_ctree = (GtkCTree *)editor->layout_ctree;
	if(layout_ctree == NULL)
	{
	    fip->processing = FALSE;
	    return(matches_count);
	}


	/* If throughness is set to 0 (match all) or 2 (match phrase),
	 * then check if given search string has only one value.
	 */
	if((throughness == 0) || (throughness == 2))
	{
	    const char *cstrptr = (const char *)strchr(string, ' ');
	    if(cstrptr == NULL)
	    {
		/* Only one argument, so there is no need to match all or
		 * match phrase, so change throughness to 1 to match any.
		 */
		throughness = 1;
	    }
	}

	/* Handle by throughness. */
	switch(throughness)
	{
	  case 2:	/* Match phrase. */
	    matches_count += EditorFIPDoFindAny(  
		fip, editor,
		results_clist, layout_ctree,
		string, case_sensitive, FALSE
	    );
	    break;

	  case 1:	/* Match any occurance. */
	    matches_count += EditorFIPDoFindAny(
		fip, editor,
		results_clist, layout_ctree,
		string, case_sensitive, TRUE
	    );
	    break;

	  case 0:	/* Match all occurances. */
	    matches_count += EditorFIPDoFindAll(
		fip, editor,
		results_clist, layout_ctree,
		string, case_sensitive, TRUE
	    );
	    break;
	}


	/* Mark dialog as done processing. */
	fip->processing = FALSE;

	return(matches_count);
}
