/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.filomace.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QtGui>


/////////////////////// Std includes
#include <math.h>
#include <algorithm>
#include <limits> // for std::numeric_limits

using namespace std;

/////////////////////// Local includes
#include "isotopicPatternCalculationDlg.hpp"
#include "isotopicPatternCalculator.hpp"
#include "isotopicPeak.hpp"
#include "application.hpp"


namespace massXpert
{

  IsotopicPatternCalculationDlg::IsotopicPatternCalculationDlg 
 (QWidget *parent, const QList<Atom *> &atomList)
   : QDialog(parent), m_atomList(atomList),
     m_polChemDef(static_cast<CalculatorWnd *>(parent)->polChemDef())
  {
    Q_ASSERT(parent);

    m_aborted = false;
    
    // When the window is created the formula line edit is empty, so
    // at least that data has an error. The ionization data is most
    // probably not erroneous because it comes from the polymer
    // chemistry definition or from the caller.
    m_validationErrors = MXP_VALIDATION_FORMULA_ERRORS;
    
    m_filePath = "";
  
    m_ui.setupUi(this);

    // Set the name of the polymer chemistry definition.

    m_ui.polChemDefLabel->setText(m_polChemDef.name());
  
    // By default we want a gaussian-type curve.
    m_curveType = MXP_CURVE_TYPE_GAUSSIAN;
    
    // Charge stuff.
    m_ui.chargeSpinBox->setRange(1, 10000000);
    m_ui.chargeSpinBox->setValue(1);
    m_charge = m_ui.chargeSpinBox->value();

    // Isotopic peak probability stuff.
    m_ui.minimumProbabilityLineEdit->setText("0.0000001");
    m_minimumProbability = 0.0000001;
  
    // Max number of peaks in the isotopic cluster.
    m_ui.maximumPeaksSpinBox->setRange(0, 1000);
    m_ui.maximumPeaksSpinBox->setValue(100);
    m_maximumPeaks = 100;
  
    // Resolution and FWHM stuff
    m_ui.resolutionSpinBox->setRange(0, 1000000);
    m_ui.resolutionSpinBox->setValue(0);
    m_resolution = 0;
    m_fwhm = 0;
    
    // Get the number of points used to craft the curve.
    m_points = m_ui.pointsSpinBox->value();

    // Since both values above are not correct, set the corresponding
    // bits to the validation error int.
    m_validationErrors |= MXP_VALIDATION_RESOLUTION_ERRORS;
    m_validationErrors |= MXP_VALIDATION_FWHM_ERRORS;
        
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
  
    settings.beginGroup("isotopic_pattern_calculation_dlg");

    restoreGeometry(settings.value("geometry").toByteArray());

    m_ui.splitter->restoreState(settings.value("splitter").toByteArray());

    settings.endGroup();

    connect(m_ui.chargeSpinBox,
            SIGNAL(valueChanged(int)),
            this,
            SLOT(chargeChanged(int)));
    
    connect(m_ui.formulaLineEdit,
            SIGNAL(textChanged(const QString &)),
            this,
            SLOT(formulaChanged(const QString &)));

    connect(m_ui.pointsSpinBox,
            SIGNAL(valueChanged(int)),
            this,
            SLOT(pointsChanged(int)));

    connect(m_ui.resolutionSpinBox,
            SIGNAL(valueChanged(int)),
            this,
            SLOT(resolutionChanged(int)));

    connect(m_ui.fwhmLineEdit,
            SIGNAL(textEdited(const QString &)),
            this,
            SLOT(fwhmEdited(const QString &)));

    connect(m_ui.executePushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(execute()));
  
    connect(m_ui.abortPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(abort()));

    connect(m_ui.outputFilePushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(outputFile()));

    // For debugging purposes:
    
  }


  void 
  IsotopicPatternCalculationDlg::closeEvent(QCloseEvent *event)
  {
    if (event)
      printf("%s", "");
  
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
  
    settings.beginGroup("isotopic_pattern_calculation_dlg");
  
    settings.setValue("geometry", saveGeometry());
  
    settings.setValue("splitter", m_ui.splitter->saveState());

    settings.endGroup();
  }


  IsotopicPatternCalculationDlg::~IsotopicPatternCalculationDlg()
  {
  }


  bool
  IsotopicPatternCalculationDlg::fetchValidateInputData()
  {
    Application *application = static_cast<Application *>(qApp);

    // The ionization stuff, the formula for which the isotopic
    // pattern is to be calculated, the mz ratio (in m_mono) all have
    // been checked previously by automated procedures (see the
    // "changed" slots). We have to make sure that either the
    // resolution or the FWHM values are OK. All the other bits must
    // be clear. Only either the RESOLUTION or the FWHM bit might be
    // set.

    int testBitset = 0;
    testBitset |= MXP_VALIDATION_RESOLUTION_ERRORS;
    testBitset |= MXP_VALIDATION_FWHM_ERRORS;
    
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      testBitset, "testBitset");

    if(m_validationErrors == testBitset)
      return false;
      
    if((m_validationErrors & MXP_VALIDATION_FORMULA_ERRORS) ==
       MXP_VALIDATION_FORMULA_ERRORS)
      return false;

    // Now that we know that the formula was validated, we can ask
    // that all the isotopes be deep-copied into the formula.

    // The list of atomCount objects has to be made with fully detailed
    // isotopes...
    if (!m_formula.deepAtomCopy(m_atomList))
      {
	QMessageBox::warning(0,
			      tr("massXpert: Isotopic Pattern Calculation"),
			      tr("Failed to deep-copy atom list."),
			      QMessageBox::Ok);

	return false;
      }
    
    // We still have to fetch a number of parameters.

    // Get to know if we want gaussian or lorentzian curves.
    if(m_ui.gaussianRadioButton->isChecked())
      {
        m_curveType = MXP_CURVE_TYPE_GAUSSIAN;
      }
    else
      {
        m_curveType = MXP_CURVE_TYPE_LORENTZIAN;
      }
  
  
    // Shall we use localization for all the numerical output?
    bool isUsingLocale = m_ui.localeCheckBox->checkState() == Qt::Checked ?
      true : false;
    
    int totAtoms = m_formula.totalAtoms();
    m_ui.progressBar->setRange(0, totAtoms);
    m_ui.progressBar->setValue(0);
    
    int totIsotopes = m_formula.totalIsotopes(m_atomList);

    QString textEditText;

    if (isUsingLocale)
      textEditText += tr("INPUT\n=====\n\n"
                         "Formula: %1\n"
                         "Charge: %2\n"
                         "Mono Mass: %3 \t Avg mass: %4\n"
                         "Total number of atoms: %5\t"
                         "Total number of isotopes: %6\n\n")
        .arg(m_formula.formula())
	.arg(m_charge)
        .arg(application->locale().
	      toString(m_mono, 'f', MXP_OLIGOMER_DEC_PLACES))
	.arg(application->locale().
	      toString(m_avg, 'f', MXP_OLIGOMER_DEC_PLACES))
        .arg(totAtoms)
	.arg(totIsotopes);
    else
      textEditText += tr("INPUT\n=====\n\n"
                         "Formula: %1\n"
                         "Charge: %2\n"
                         "Mono Mass: %3 \t Avg mass: %4\n"
                         "Total number of atoms: %5\t"
                         "Total number of isotopes: %6\n\n")
	.arg(m_formula.formula())
        .arg(m_charge)
	.arg(QString().setNum(m_mono, 'f', MXP_OLIGOMER_DEC_PLACES))
	.arg(QString().setNum(m_avg, 'f', MXP_OLIGOMER_DEC_PLACES))
	.arg(totAtoms)
	.arg(totIsotopes);
    
    // Put that description in the text edit widget.
    m_ui.resultTextEdit->append(textEditText);
    
    // qDebug() << __FILE__ << __LINE__
    //          << textEditText;
    
    // At this point we have to sort out wether the user wants to use
    // the resolution or the FWHM for the calculations. This is
    // automatically known by looking at which value is 0 (see the
    // corresponding changed() slots).

    // Get the points value.
    m_points = m_ui.pointsSpinBox->value();
    
    // How about setting the increment, that is the size of the gap
    // between two points ? If it is not set (its value is 0), then
    // increment is fwhm() / m_points;

    // First however, see if the user wants to use the FWHM value or
    // the resolution.
    
    if(m_fwhm)
      {
        // FWHM is not zero, which means the user has set a value for
        // it. Keep it.
      }
    else
      {
        // We have to compute the FWHM starting from the mz ratio and
        // the resolution.
        
        m_fwhm = m_mono / m_resolution;
      }

    // Set the maximum number of peaks in the isotopic curve.
    m_maximumPeaks = m_ui.maximumPeaksSpinBox->value();
    QString fwhmString = m_ui.fwhmLineEdit->text();
    
    // Set the minimum probability each isotopic peak must have to be
    // retained in the final curve.

    QString minProbString = m_ui.minimumProbabilityLineEdit->text();
    bool ok = false;
    
    double minProb = minProbString.toDouble(&ok);
    
    if(!minProb && !ok)
      {
	QMessageBox::warning(0,
                             tr("massXpert: Isotopic Pattern Calculation"),
                             tr("Please, fix the minimum probability."),
                             QMessageBox::Ok);
        
	return false;
      }
      
    m_minimumProbability = minProb;
    
    return true;
  }

  
// Returns false if formula is bad or aborts if calculation fails..
  bool
  IsotopicPatternCalculationDlg::fetchFormulaMass(double *mono, double *avg)
  {
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ fetchFormulaMass");

    // Check if there are currently errors set.
    
    if((m_validationErrors & MXP_VALIDATION_FORMULA_ERRORS) ==
       MXP_VALIDATION_FORMULA_ERRORS)
      return false;
    
    double monoMass = 0;
    double avgMass = 0;
    
    // It is impossible that we have an error here, since the formula
    // was previously validated (MXP_VALIDATION_FORMULA_ERRORS above).
    if (!m_formula.accountMasses(m_atomList, &monoMass, &avgMass, 1))
      qFatal("Fatal error at %s@%d.Aborting.", __FILE__, __LINE__);
    
    if(mono)
      *mono = monoMass;
    
    if(avg)
      *avg = avgMass;

    // qDebug() << __FILE__ << __LINE__
    //          << "Formula mono/avg masses:"
    //          << monoMass << "/" << avgMass;

    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ fetchFormulaMass");

    return true;
  }
    
  // Returns false if the formula is not validated and aborts if an
  // error occurs while the formula was validated.
  bool
  IsotopicPatternCalculationDlg::fetchMzRatio(double *mono, double *avg)
  {
    // Check if there are currently errors set either in the formula
    // or in the ionization rule.

    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ ENTER fetchMzRatio");
        
    if((m_validationErrors & MXP_VALIDATION_FORMULA_ERRORS) ==
       MXP_VALIDATION_FORMULA_ERRORS)
      {
        return false;
      }
    
    // The formula data have already been automatically fetched and
    // set to m_formula when the data in the widgets changed using the
    // correponding slot.
    
    // We can compute the m/z ratio.
    
    double monoMass = 0;
    double avgMass = 0;

    fetchFormulaMass(&monoMass, &avgMass);

    // qDebug() << __FILE__ << __LINE__ 
    //          << "monoMass:" << monoMass
    //          << "avgMass:" << avgMass;
    
    // Note that the only acceptable value returned here is mono > 0,
    // because the formula was validated and thus mass calculation
    // should not fail. Indeed, if there is a failure in the called
    // function, qFatal is triggered. No need, thus, to check the
    // status of the call. It cannot fail.
    
    // Compute the m/z ratio.
    monoMass = monoMass / m_charge;
    avgMass = avgMass / m_charge;
    
    // At this point the ionizable has the right mzRatio.
    
    if(mono)
      *mono = monoMass;
    
    if(avg)
      *avg = avgMass;

    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ EXIT fetchMzRatio");
        
    return true;
  }
  

  void 
  IsotopicPatternCalculationDlg::formulaChanged(const QString &text)
  {
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ ENTER formulaChanged");

    // qDebug() << __FILE__ << __LINE__
    //          << "formulaChanged has text:" << text;
    
    // New text contains a new formula. Check it.
    
    if(text.isEmpty())
      {
        // Set the error bit.
        m_validationErrors |= MXP_VALIDATION_FORMULA_ERRORS;

        m_ui.monoMzRatioLineEdit->setText("0.000");
        m_ui.inputDataFeedbackLineEdit->setText(tr("Formula error"));

        return;
      }
          
    Formula formula(text);

    // Do not bother storing all the atoms in the formula, this is
    // only a check (false param below).  m_atomList is a reference to
    // the atom list of the polymer chemistry definition currently
    // used in the caller calculator window.

    if (!formula.validate(m_atomList, false, true))
      {
        // Set the error bit.
        m_validationErrors |= MXP_VALIDATION_FORMULA_ERRORS;

        m_ui.monoMzRatioLineEdit->setText("0.000");
        m_ui.inputDataFeedbackLineEdit->setText(tr("Formula error"));

        return;
      }
    
    // qDebug() << __FILE__ << __LINE__
    //          << "Formula:" << formula.formula() << "validated." ;

    // At this point we know we could actually validate the formula,
    // so do that work with the member formula:

    // We want to validate the formula and in the mean time construct
    // the list of all the AtomCount objects(first true), and since
    // the formula is reused we also ensure that that list is reset
    // (second true). m_atomList is a reference to the atom list of
    // the polymer chemistry definition currently used in the caller
    // calculator window.

    m_formula.setFormula(text);
    m_formula.validate(m_atomList, true, true);
    
    // Clear the bit as there is no error here.
    m_validationErrors &= ~MXP_VALIDATION_FORMULA_ERRORS;
    
    m_ui.inputDataFeedbackLineEdit->setText(tr("Formula fine"));
    
    // qDebug() << __FILE__ << __LINE__
    //          << "Going to call updateMzRatio.";
    
    updateMzRatio();
    updateIncrement();
    
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ EXIT formulaChanged");

    return;
  }
  

  void 
  IsotopicPatternCalculationDlg::chargeChanged(int value)
  {
    //The charge has changed, we should compute the mzRatio.
    m_charge = m_ui.chargeSpinBox->value();
    
    updateMzRatio();
    updateIncrement();
  }
  

  void 
  IsotopicPatternCalculationDlg::pointsChanged(int value)
  {
    //The points have changed, we should compute the increment.
    m_points = m_ui.pointsSpinBox->value();
    
    updateIncrement();
  }
  

  void 
  IsotopicPatternCalculationDlg::resolutionChanged(int value)
  {
    // If the value changed, that means that the user wants the
    // resolution to be taken into account for calculation of the peak
    // width, and not the FWHM. Only if resolution is set to 0, FWHM
    // will be the value taken into account.

    // Show what's the value of the errors.
    
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ ENTER resolutionChanged");
    
    if(value <= 0)
      {
        // Tell the user to set a valid FWHM value, then.
        m_ui.inputDataFeedbackLineEdit->setText
          (tr("Set a valid FWHM value"));
            
        m_ui.resolutionSpinBox->setValue(0);
        
        // Set the error bit so that we can let the increment
        // calculation know that we can not perform that computation
        // using FWHM.

        m_validationErrors |= MXP_VALIDATION_RESOLUTION_ERRORS;

        return;
      }
    
    m_resolution = value;
    
    m_ui.inputDataFeedbackLineEdit->setText
      (tr("Will use the resolution"));

    // Clear the bit as there is no error here.
    m_validationErrors &= ~MXP_VALIDATION_RESOLUTION_ERRORS;

    // Clear the fwhm value, as we want to use the resolution and set
    // the error bit so that when we update the increment we know that
    // we should use the resolution for the calculation and not the
    // FWHM.
    m_ui.fwhmLineEdit->setText("0");
    m_fwhm = 0;
    m_validationErrors |= MXP_VALIDATION_FWHM_ERRORS;
    
    // Show what's the value of the errors.

    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ EXIT resolutionChanged");

    updateIncrement();
    
    return;
  }
  

  void 
  IsotopicPatternCalculationDlg::fwhmEdited(const QString &text)
  {
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ ENTER fwhmEdited");

    // If the text was edited in the line edit, that means that the
    // user wants the FWHM to be taken into account for calculation of
    // the peak width, and not the resolution. Only if FWHM is set to
    // 0, resolution will be the value taken into account.
    
    QString txt = m_ui.fwhmLineEdit->text();
    
    bool ok = false;
    
    double fwhmValue = txt.toDouble(&ok);
    
    if(!fwhmValue && !ok)
      {
        // Set the error bit.
        m_validationErrors |= MXP_VALIDATION_FWHM_ERRORS;

        m_ui.inputDataFeedbackLineEdit->setText
          (tr("Fix the FWHM value, please"));
        
        return;
      }
    
    // But the value might be faithfully 0, if the user is telling us
    // that she wants to take the resolution into account and not the
    // FWHM.

    if(!fwhmValue)
      {
        m_ui.inputDataFeedbackLineEdit->setText
          (tr("Set a valid resolution value"));

        // Set the error bit to let the increment calculation know
        // that it cannot base the calculation on the FWHM value.

        m_validationErrors |= MXP_VALIDATION_FWHM_ERRORS;

        return;
      }
    
    // At this point we know that fwhmValue contains a proper value.
    
    m_ui.inputDataFeedbackLineEdit->setText
      (tr("Will use the FWHM"));
    
    m_validationErrors &= ~MXP_VALIDATION_FWHM_ERRORS;

    m_fwhm = fwhmValue;

    // Set the resolution spinbox to 0 and set the error bit
    // associated with it to let the increment calculation know that
    // it cannot be based on that value.

    m_resolution = 0;
    m_ui.resolutionSpinBox->setValue(0);
    m_validationErrors |= MXP_VALIDATION_RESOLUTION_ERRORS;
    
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ EXIT fwhmEdited");

    updateIncrement();
    
    return;
  }

    
  bool 
  IsotopicPatternCalculationDlg::updateMzRatio()
  {
    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ ENTER updateMzRatio");

    // Set the mz ratio to our member data variable.
    bool res = fetchMzRatio(&m_mono, &m_avg);
    
    // If res is falsed, that means that the formula is not validated
    // at present. Silently return.
    if(!res)
      {
        QMessageBox::warning(0, 
                             tr("massXpert - Isotopic Pattern Calculation"),
                             tr("Failed to fetch the mz ratio."),
                             QMessageBox::Ok);
        
      return false;
      }
    
    Application *application = static_cast<Application *>(qApp);
    
    QString mzRatio = application->locale().
      toString(m_mono, 'f', MXP_OLIGOMER_DEC_PLACES);
    
    m_ui.monoMzRatioLineEdit->setText(mzRatio);

    // Show what's the value of the errors.
    // debugPutStdErrBitset(__FILE__, __LINE__,
    //                      m_validationErrors, 
    //                      "m_validationErrors @ EXIT updateMzRatio");

    return true;
  }
  

  bool 
  IsotopicPatternCalculationDlg::updateIncrement()
  {
    // We can only calculate the increment if we have the number of
    // points to construct the curve and either the resolution and the
    // mzRatio or the FWHM.

    // If we know the FWHM, then, fine, we can compute the increment
    // right away.
    
    double increment = 0;
    
    if((m_validationErrors & MXP_VALIDATION_FWHM_ERRORS) !=
       MXP_VALIDATION_FWHM_ERRORS)
      {
        increment = (MXP_FWHM_PEAK_SPAN_FACTOR * m_fwhm) / m_points;
      }
    else if((m_validationErrors & MXP_VALIDATION_RESOLUTION_ERRORS) !=
       MXP_VALIDATION_RESOLUTION_ERRORS)
      {
        // If we know the resolution and we can get the mzRatio, we'll
        // be able to first compute the FWHM as (mzRatio / FWHM). From
        // there, we'll be able to compute the increment.
        
        double mono = 0;
        double avg = 0;
        
        // We compute the increment by using the mass of the formula,
        // not its mzRatio.
        if(fetchFormulaMass(&mono, &avg))
          {
            double fwhm = mono / m_resolution;
            
            increment = (MXP_FWHM_PEAK_SPAN_FACTOR * fwhm) / m_points;

            // Because we'll need more points when the charge
            // increases... Formally, we have to check that charge is
            // not 0, but we should not be required to do it because
            // the charge can never be 0 (gui-limited values for the
            // charge spin box).
            if(m_charge)
              increment = increment / m_charge;
          }
      }

    if(increment)
      {
        m_increment = increment;
        
        QString text;
        text.setNum(m_increment);
        
        m_ui.incrementLabel->setText(text);
        
        return true;
      }
    else
      m_ui.incrementLabel->setText("0");
    
    return false;
  }
  

  void 
  IsotopicPatternCalculationDlg::execute()
  {
    Application *application = static_cast<Application *>(qApp);

    // Clear the textEdit widget so that we can start out-putting text
    // into it, note that fetchValidateInputData() puts text in it.
    m_ui.resultTextEdit->clear();

    // Clear the results string for the creation of the calculator
    // below to receive an empty string.
    m_resultsString.clear();
    
    // We are asked to launch the calculation. But we ought to make
    // sure that all the required data are set fine.

    if(!fetchValidateInputData())
      {
        QMessageBox::warning(0, 
                             tr("massXpert - Isotopic Pattern Calculation"),
                             tr("Please fix the input data first."),
                             QMessageBox::Ok);
        return;
      }
    

    // Send to the text edit widget the data we are going to use for
    // the calculation.

    QString textEditText;

    bool isUsingLocale = m_ui.localeCheckBox->checkState() == Qt::Checked ?
      true : false;
    
    if (isUsingLocale)
      textEditText += tr("FWHM: %1 \t Max. peaks %2 "
                         "\t Min. probability: %3\n\n")
	.arg(application->locale().toString(m_fwhm, 'f', 4))
	.arg(application->locale().toString(m_maximumPeaks))
	.arg(application->locale().toString(m_minimumProbability, 'f', 15));
    else
      textEditText += tr("FWHM: %1 \t Max. peaks %2 "
                         "\t Min. probability: %3\n\n")
	.arg(QString().setNum(m_fwhm, 'f', 4))
	.arg(QString().setNum(m_maximumPeaks))
	.arg(QString().setNum(m_minimumProbability, 'f', 15));
        
    m_ui.resultTextEdit->append(textEditText);


    // We will allocate a IsotopicPatternCalculator instance and let
    // it do the work.

    IsotopicPatternCalculator *calculator = 
      new IsotopicPatternCalculator(m_formula, m_charge, m_fwhm, 
                                    m_maximumPeaks, m_minimumProbability,
                                    m_atomList, &m_resultsString);
    
    connect(calculator,
            SIGNAL(isotopicCalculationProgressValueChanged(int)),
            this,
            SLOT(isotopicCalculationProgressValueChanged(int)));

    connect(calculator,
            SIGNAL(isotopicCalculationMessageChanged(QString)),
            this,
            SLOT(isotopicCalculationMessageChanged(QString)));
    
    connect(this,
            SIGNAL(isotopicCalculationAborted()),
            calculator,
            SLOT(isotopicCalculationAborted()));
    
    calculator->calculateCentroidPeakList();
    const QList<IsotopicPeak *> &peakList = calculator->centroidPeakList();
    

    // For each centroidPeak, let's print the mass, the relative
    // intensity and the probability.

    textEditText.clear();
    
    textEditText.append("\nList of centroid peaks:\n");
    
    for(int iter = 0, size = peakList.size();
        iter < size; ++iter)
      {
        IsotopicPeak *peak = peakList.at(iter);
        textEditText += QString().setNum(peak->mass(), 'f', 6);
        textEditText += " ";
        textEditText += QString().setNum(peak->relativeIntensity(), 'f', 15);
        textEditText += "\n";
      }

    m_ui.resultTextEdit->append(textEditText);
    
    // qDebug() << __FILE__ << __LINE__
    //          << textEditText;
    
    // For each Centroid peak, we should make a real curve that is
    // created either a gaussian or a lorentzian curve.

    QList<IsotopicCurve *>curveList;
    
    for (int iter = 0, peakListSize = peakList.size();
         iter < peakListSize; ++iter)
      {
        IsotopicPeak *peak = peakList.at(iter);
        
	double mass = peak->mass();
	double relInt = peak->relativeIntensity();
        
        IsotopicCurve *curve = new IsotopicCurve(mass, 
                                                 relInt,
                                                 m_fwhm, 
                                                 m_points, 
                                                 m_increment,
                                                 1 /* normFactor */,
                                                 m_curveType);
        
        // qDebug() << __FILE__ << __LINE__
        //          << "Start curve calculation.";
        
        curve->calculateCurve();

        // We want to get the calculation configuration only once, for
        // the first iteration in this loop.

        if (!iter)
          {
            // Let's get the detailed calculation configuration
            // string, so that we can append it to the textEdit
            // widget.
            textEditText.clear();
            textEditText.append(curve->config());
            m_ui.resultTextEdit->append(textEditText);
          }
        
        curveList.append(curve);

        // qDebug() << __FILE__ << __LINE__
        //          << "\nAppended new curve:\n"
        //          << curve->dataAsString();
      }
    
    // At this point, each Centroid peak, that is each mzRatio/intensity
    // pair in centroidPeakList has given rise to a full
    // gaussian/lorentzian curve that's been appended in curveList.

    // We now have to perform the sum of all these curves so as to
    // generate a single curve that encompasses all the isotopic
    // cluster.

    // First off, get the min/max mzRatio of all the curves.

    // qDebug() << __FILE__ << __LINE__
    //          << "Number of isotopic curves: " << curveList.size();
    
    double tempMzRatio = 0;
    double maxMzRatio = 0;
    double minMzRatio = 1000000000;

    for(int iter = 0, curveListSize = curveList.size();
        iter < curveListSize ; ++iter)
      {
        IsotopicCurve *curve = curveList.at(iter);
      
        // Get to the list of points in 
        const QList<IsotopicPeak *> &peakList =
          curve->isotopicPeakList();

        // Get first point of the curve.
        IsotopicPeak *peak = peakList.first();

        // Get the minimum mzRatio
        tempMzRatio = peak->mass();
      
        if(tempMzRatio < minMzRatio)
          minMzRatio = tempMzRatio;

        // Get last point of the curve.
        peak = peakList.last();

        // Get the maximum mzRatio
        tempMzRatio = peak->mass();
      
        if(tempMzRatio > maxMzRatio)
          maxMzRatio = tempMzRatio;
      }

    // qDebug() << __FILE__ << __LINE__
    //          << "Whole isotopic cluster mass spectrum spans: "
    //          <<"[" << minMzRatio << "--" << maxMzRatio << "]";
    
    // At this point we know that our mass spectrum should range
    // [minMzRatio -- maxMzRatio]. We can perform the actual sum of
    // all the isotopic curves. First off, empty the m_isotopicCurve.
    
    m_isotopicCurve.emptyPeakList();
    
    double curMzRatio = minMzRatio;
  
    while(curMzRatio <= maxMzRatio)
      {
        // curMzRatio now has a mzRatio value for which we have to
        // sum all the intensities in each curve.
      
        double summedIntensity = 0;
      
        for(int iter = 0, curveListSize = curveList.size(); 
            iter < curveListSize ; ++iter)
          {
            IsotopicCurve *curve = curveList.at(iter);
          
            bool ok = false;
          
            summedIntensity += curve->intensityAt(curMzRatio, 
                                                  m_increment, &ok);
          }
      
        IsotopicPeak *newPeak = 
          new IsotopicPeak(curMzRatio, summedIntensity, 0, 0);
        
        m_isotopicCurve.appendPeak(newPeak);

        curMzRatio += m_increment;
      }
    
    if(!m_filePath.isEmpty())
      {
        // Send all the data to the output file.
        if (!m_isotopicCurve.dataToFile(m_filePath))
          QMessageBox::warning(this, 
                               tr("massXpert - Isotopic Pattern Calculation"),
                               tr("Failed to export the curve data to file."),
                               QMessageBox::Ok);
        
        return;
      }
    
    // Send the data to the text edit widget.
    m_ui.resultTextEdit->append(m_isotopicCurve.dataAsString());

    delete calculator;
  }
  
  
  void 
  IsotopicPatternCalculationDlg::isotopicCalculationProgressValueChanged
  (int value)
  {
    m_ui.progressBar->setValue(value);
    // qDebug() << __FILE__ << __LINE__
    //          << "isotopicCalculationProgressValueChanged:" << value;
    qApp->processEvents();
  }
  

  void 
  IsotopicPatternCalculationDlg::isotopicCalculationMessageChanged
  (QString text)
  {
    m_ui.feedbackLineEdit->setText(text);
    // qDebug() << __FILE__ << __LINE__
    //          << "isotopicCalculationMessageChanged:" << text;
    qApp->processEvents();
  }
  

  void 
  IsotopicPatternCalculationDlg::abort()
  {
    m_aborted = true;
    emit(isotopicCalculationAborted());
  }
  

  void 
  IsotopicPatternCalculationDlg::outputFile()
  {
    QString name;
  
    m_filePath = 
      QFileDialog::getSaveFileName(this, tr("Export Raw Text File"),
				    QDir::homePath(),
				    tr("Any file type(*)"));
  }

  void 
  IsotopicPatternCalculationDlg::debugPutStdErrBitset(QString file, int line,
                                                      int value,
                                                      const QString &text)
  {
    qDebug() << file << line << "\n"
             << GlobBinaryRepresentation(MXP_VALIDATION_ERRORS_NONE)
             << ": MXP_VALIDATION_ERRORS_NONE" << "\n"
             << GlobBinaryRepresentation(MXP_VALIDATION_FORMULA_ERRORS)
             << ": MXP_VALIDATION_FORMULA_ERRORS" << "\n"
             << GlobBinaryRepresentation(MXP_VALIDATION_RESOLUTION_ERRORS)
             << ": MXP_VALIDATION_RESOLUTION_ERRORS" << "\n"
             << GlobBinaryRepresentation(MXP_VALIDATION_FWHM_ERRORS)
             << ": MXP_VALIDATION_FWHM_ERRORS" << "\n"
             << "\n"
             << GlobBinaryRepresentation(value)
             << ":" << text << ":" << value
             << "\n";
  }
  
} // namespace massXpert



#if 0
  /***************************************************************************
   *  file                 :  ipc.c                                          *
   *  copyright            :(C) 2001-2005 by Dirk Nolting                   *
   *  email                : nolting@uni-duesseldorf.de                      *
   ***************************************************************************/

  /***************************************************************************
   *                                                                         *
   *   This program is free software; you can redistribute it and/or modify  *
   *   it under the terms of the GNU General Public License as published by  *
   *   the Free Software Foundation; either version~2 of the License, or     *
   *  (at your option) any later version.                                   *
   *                                                                         *
   ***************************************************************************/

#include "global.h"
#include "ipc.h"
#include "pars.h"
#include "element.h"
#include "gp_out.h"
#include <time.h>
#include <signal.h>
#include <math.h>

//#define SUMMARIZE_LEVEL 0.00000001
//#define SUMMARIZE_LEVEL 0.0001
//#define SUMMARIZE_LEVEL 0.001
#define SUMMARIZE_LEVEL 0.01

  compound *verbindung=NULL;
  isotope *m_peakList;
  int fast_calc=0;

  void free_list(isotope *target)
  {
    while(target->next)
      {
	target=target->next;
	free(target->previous);
      }
    free(target);
  }

  void cut_peaks(isotope *spectrum)
  {
    int dummy=1;

    while((spectrum->next) &&(dummy<fast_calc) )
      {
	++dummy;
	spectrum=spectrum->next;
      }

    if(spectrum->next)
      {
	free_list(spectrum->next);
	spectrum->next=NULL;
      }
  }

  void summarize_peaks()
  {  isotope *dummy,*d2;

    for(dummy=m_peakList;dummy;dummy=dummy->next)
      /* Differenz wegen Rundungsfehlern */
      while( dummy->next &&(dummy->next->mass - dummy->mass < SUMMARIZE_LEVEL) )
	{
	  d2=dummy->next;
	  dummy->next=d2->next;
	  if(dummy->next)
	    dummy->next->previous=dummy;
	  dummy->p+=d2->p;
	  free(d2);
	}
  }

  isotope *add_peak(isotope *base,isotope *peak)
  {
    static isotope *IsotopeIterator;

    if(!(base->mass))
      {
	peak->next=NULL;
	peak->previous=NULL;
	IsotopeIterator = peak;
	return peak;
      }

    if( peak->mass >= IsotopeIterator->mass )
      while((IsotopeIterator->next) &&(IsotopeIterator->mass < peak->mass))
	IsotopeIterator=IsotopeIterator->next;
    else
      {
	while((IsotopeIterator->previous) &&(IsotopeIterator->mass > peak->mass) )
	  IsotopeIterator=IsotopeIterator->previous;
	IsotopeIterator=IsotopeIterator->next;
      }

    if((IsotopeIterator->mass) >=(peak->mass) )
      {
	peak->next=IsotopeIterator;
	peak->previous=IsotopeIterator->previous;
	peak->previous->next=peak;
	IsotopeIterator->previous=peak;
	return base;
      }
    else
      {
	IsotopeIterator->next=peak;
	peak->next=NULL;
	peak->previous=IsotopeIterator;
	return base;
      }
    return 0;
  }

  int calculate_peaks(){
    compound *c;
    isotope *newPeakList,*p,*i,*np1;
    int amount;

    if(!(m_peakList=malloc(sizeof(isotope))))
      return 0;
    m_peakList->mass=0;
    m_peakList->p=1;
    m_peakList->previous=NULL;
    m_peakList->next=NULL;

    for (c = verbindung; c; c = c->next)
      {
	for(amount = 0; amount < c->amount; ++amount)
	  {
	    if (!(newPeakList=malloc(sizeof(isotope))))
	      return 0;
	  
	    newPeakList->mass = 0;
	  
	    for (p = m_peakList; p; p = p->next)
	      {
		for(i = c->isotopes; i; i = i->next)
		  {
		    //printf("working on isotope of mass %f\n", i->mass);

		    if (!(np1 = malloc(sizeof(isotope))))
		      return 0;
		  
		    np1->mass=p->mass + i->mass;
		    np1->p=p->p * i->p;
		  
		    if(!(newPeakList = add_peak(newPeakList,np1)))
		      return 0;
		  }
	      }
	  
	    free_list(m_peakList);
	    m_peakList = newPeakList;
	    summarize_peaks();
	  
	    if (fast_calc)
	      cut_peaks(m_peakList);
	  }
      }
  
    return 1;
  }


  void print_result(int digits,int charge){
    isotope *d;
    double maxp=0,relint=0,sump=0;
    int permutationen=0;

    printf("\n");
 
    for(d=m_peakList;d;d=d->next)
      {
	++permutationen;
	sump+=d->p;
	d->mass=d->mass / charge;
	d->mass=(rint( d->mass * pow(10,digits) ) / pow(10,digits) );
      }

    summarize_peaks();
    for(d=m_peakList;d;d=d->next)
      if(d->p > maxp)
	maxp=d->p;

    for(d=m_peakList;d;d=d->next)
      {
	if(( relint=(d->p/maxp)*100) > MIN_INT )
	  printf("M= %f, p= %e, rel. Int.= %f%%\n",
		 d->mass,d->p,relint);
      }
    if(!(fast_calc))
      printf("\nNumber of  permutations: %i\n",permutationen);
    else
      {
	sump=(rint(sump*10000)/100); 
	printf("\nCovered Intensity: %2.2f%% \n",sump);
      }
  }

  int main(int argc,char **argv){
    long seconds;
    int d=1,zeig_summenformel=0,calc_peaks=1,gnuplot=0,use_digits=USE_DIGITS,charge=1;
    char *gnuplotfile=NULL;
  
    if(!argv[d])
      {
	usage();
	return 1;
      }

#ifdef HAVE_SIGNAL
    signal(SIGHUP,SIG_IGN);
#endif

    if(!init_elements()){
      printf("Error in init_elements\n");
      return 1;
    }
    while(argv[d])
      {
	if(!strcmp(argv[d],"-f"))
	  {
	    ++d;
	    if(!argv[d])
	      {
		printf("Missing argument for -f\n");
		return 1;
	      }
	    fast_calc=strtol(argv[d],NULL,10);
	  }

	else if(!strcmp(argv[d],"-z"))
	  {
	    ++d;
	    if(!argv[d])
	      {
		printf("Missing argument for -z\n");
		return 1;
	      }
	    charge=strtol(argv[d],NULL,10);
	  }

	else if(!strcmp(argv[d],"-d"))
	  {
	    ++d;
	    if(!argv[d])
	      {
		printf("Missing argument for -d\n");
		return 1;
	      }
	    use_digits=strtol(argv[d],NULL,10);
	  }

	else if(!strcmp(argv[d],"-c")){
	  ++d;
	  if(!argv[d])
	    {
	      printf("Missing argument for -c\n");
	      return 1;
	    }
	  if(!pars_chem_form(argv[d])){
	    printf("Parser error.\n");
	    return 1;
	  }
	}

	else if(!strcmp(argv[d],"-g")){
	  ++d;
	  if(!argv[d]){
	    printf("Missing argument for -g\n");
	    return 1;
	  }
	  gnuplot=1;
	  if(!(gnuplotfile=strdup(argv[d]))){
	    printf("Not enough memory\n");
	    return 1;
	  }
	}


	else if(!strcmp(argv[d],"-p")){
	  ++d;
	  if(!argv[d]){
	    printf("Missing argument for -p\n");
	    return 1;
	  }
	  if(!(pars_peptid(argv[d]))){
	    printf("Error in peptid parser.\n");
	    return 1;
	  }
	}

	else if(!strcmp(argv[d],"-a")){
	  ++d;
	  if(!argv[d]){
	    printf("Missing argument for -a\n");
	    return 1;
	  }
	  if(!pars_amino_acid(argv[d])){
	    printf("Error in pars_amino_acid.\n");
	    return 1;
	  }
	}

	else if(!strcmp(argv[d],"-s"))
	  zeig_summenformel=1;

	else if(!strcmp(argv[d],"-h"))
	  {
	    usage();
	    return 1;
	  }

	else if(!strcmp(argv[d],"-x"))
	  calc_peaks=0;

	else
	  {
	    printf("Unknown flag: %s\n",argv[d]);
	    usage();
	    return 1;
	  }
	++d;
      }

    if(zeig_summenformel)
      {
	if(!print_sum())
	  {
	    printf("error while showing chemical formula.\n");
	    return 1;
	  }
      }
#ifdef HAVE_TIME
    seconds=time(0);
#endif

    if(calc_peaks)
      if(!calculate_peaks()){
	printf("Error in calculate_peaks\n");
	return 1;
      }

    print_result(use_digits,charge);

#ifdef HAVE_TIME
    seconds=time(0)-seconds;
    printf("Computing time: %li seconds.\n",seconds);
#endif

    if(gnuplot)
      if(!(make_gnuplot_output(gnuplotfile))){
	printf("Error while generating gnuplot file\n");
	return 1;
      }

    return 0;
  }


  void usage()
  {
    printf("\nThis is IPC v %s\nCopyright Dirk Nolting 2001-2005\n\n",VERSION);
    printf("\nSynopsis:\n ipc -d <int> -z <int> -f <int> <-a <amino acid> -c <chemical formula> -p <File> -g <name> -s -x -h\n\n");

    printf("-c <chemical formula> calculates the isotopic pattern of the \n");
    printf("   given chemical formula. Additive with -p\n");

    printf("-p <File> reads peptide sequenz(one letter notation) from the \n");
    printf("   given file and calculates the isotopic pattern. Additive with -c\n");

    printf("-a <amino acids> calculate isotopic pattern from given amino acids\n");
    printf("   in one letter notation\n");

    printf("-s gives the chemical formula. Usefull for -a or -p\n");
    printf("-g creates gnuplot output and stores it in the file <name> and <name>.gnu\n");

    printf("-x no calculation of the isotopic pattern. Usefull for -s\n");
    printf("-f fast calc, calculates only first <int> peaks\n");
    printf("-d <int> digits significant\n");
    printf("-z assume <int> charges on ion \n");
    printf("-h show this text\n\n");
  }
#endif
