/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Local includes
#include "isotopicPeak.hpp"


namespace massXpert
{

  //! Constructs an isotopic peak.
  /*! All the parameters default to 0.
  
    \param mass mass initializer. Defaults to 0. Cannot be negative.

    \param abundance abundance initializer. Defaults to 0. Cannot be
    negative.

    \param probability probability. Defaults to 0. Cannot be
    negative.

    \param relIntensity relative intensity. Defaults to 0. Cannot be
    negative.
  */
  IsotopicPeak::IsotopicPeak(double mass, double relIntensity,
                             double probability, double abundance)
  {
    Q_ASSERT(mass >= 0);
    Q_ASSERT(relIntensity >= 0);
    Q_ASSERT(probability >= 0);
    Q_ASSERT(abundance >= 0);

    m_mass = mass;
    m_relativeIntensity = relIntensity;
    m_probability = probability;
    m_abundance = abundance;
  }


  //! Constructs a copy of \p other.
  /*!
    \param other isotopic peak to be used as a mold.
  */
  IsotopicPeak::IsotopicPeak(const IsotopicPeak &other)
    : Isotope(other),
      m_probability(other.m_probability),
      m_relativeIntensity(other.m_relativeIntensity)
  {
  }


  //! Creates a new isotopic peak.
  /*! The newly created peak is initialized using \c this.
  
    \return The new isotopic peak, which must be deleted when no longer in
    use.
  */
  IsotopicPeak * 
  IsotopicPeak::clone() const
  {
    IsotopicPeak *other = new IsotopicPeak(*this);
  
    return other;
  }


  //! Modifies \p other to be identical to \p this.
  /*!  \param other isotopic peak.
   */
  void 
  IsotopicPeak::clone(IsotopicPeak *other) const
  {
    Q_ASSERT(other);
  
    if (other == this)
      return;
  
    Isotope::clone(other);
  
    other->m_probability = m_probability;
    other->m_relativeIntensity = m_relativeIntensity;
  }


  //! Modifies \p this  to be identical to \p other.
  /*!  \param other isotope to be used as a mold.
   */
  void 
  IsotopicPeak::mold(const IsotopicPeak &other)
  { 
    if (&other == this)
      return;
  
    Isotope::mold(other);
  
    m_probability = other.m_probability;
    m_relativeIntensity = other.m_relativeIntensity;
  }


  //! Assigns other to \p this isotopic peak.
  /*! \param other isotopic peak used as the mold to set values to \p
    this instance.
  
    \return a reference to \p this isotopic peak.
  */
  IsotopicPeak &
  IsotopicPeak::operator =(const IsotopicPeak &other)
  {
    if (&other != this)
      mold(other);
  
    return *this;
  }


  //! Sets the probability.
  /*!  \param value probability. Cannot be negative.
   */
  void
  IsotopicPeak::setProbability(double value)
  {
    Q_ASSERT(value >= 0);

    m_probability = value;
  }


  //! Increments the probability.
  /*!  \param value value by which to increment the probability. Cannot
    be negative.
  */
  void 
  IsotopicPeak::incrementProbability(double value)
  {
    Q_ASSERT(value >= 0);
  
    m_probability += value;
  }


  //! Returns the probability.
  /*!  
    \return the probability.
  */
  double
  IsotopicPeak::probability() const
  {
    return m_probability;
  }


  //! Returns the probability as a string.
  /*! The string is created by taking into account the locale passed as
    argument. Check QLocale doc in case QLocale() is passed as argument.
  
    \param locale locale used to construct the string.
  
    \return the probability as a locale-complying string.
  */
  QString 
  IsotopicPeak::probability(const QLocale &locale, int decimalPlaces) const
  {
    if (decimalPlaces < 0)
      decimalPlaces = 0;
    
    return locale.toString(m_probability, 'f', decimalPlaces);
  }


  //! Sets the relative intensity.
  /*!  \param value relative intensity. Cannot be negative.
   */
  void
  IsotopicPeak::setRelativeIntensity(double value)
  {
    Q_ASSERT(value >= 0);

    m_relativeIntensity = value;
  }


  //! Returns the relative intensity.
  /*!  \return the relative intensity.
   */
  double
  IsotopicPeak::relativeIntensity() const
  {
    return m_relativeIntensity;
  }


  //! Returns the relative intensity as a string.
  /*! The string is created by taking into account the locale passed as
    argument. Check QLocale doc in case QLocale() is passed as argument.
  
    \return the relative intensity as a string.
  */
  QString 
  IsotopicPeak::relativeIntensity(const QLocale &locale, 
				   int decimalPlaces) const
  {
    if (decimalPlaces < 0)
      decimalPlaces = 0;
    
    return locale.toString(m_relativeIntensity, 'f', decimalPlaces);
  }

} // namespace massXpert
