/*
 * Algebraic manipulator differentiation routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

/*
 * Compute the derivative of an equation side, with respect to the variable "v",
 * using the fast transform method.
 * The result must be simplified by the caller.
 *
 * Return true if successful.
 */
int
differentiate(equation, np, v)
token_type	*equation;	/* source and destination expression */
int		*np;		/* pointer to length of "equation" */
long		v;
{
	int	i;

	organize(equation, np);
/* First put every times and divide on a level by itself. */
	for (i = 1; i < *np; i += 2) {
		switch (equation[i].token.operatr) {
		case TIMES:
		case DIVIDE:
			binary_parenthesize(equation, *np, i);
		}
	}
	return d_recurse(equation, np, 0, 1, v);
}

/*
 * Recursive differentiation routine.
 *
 * Symbolically differentiate expression in "equation"
 * (which is a standard equation side) starting at "loc".
 * The current level of parentheses is "level" and
 * do the differentation with respect to variable "v".
 *
 * Return "true" if successful, "false" if it is beyond this program's
 * capabilities or an error was encountered.
 */
int
d_recurse(equation, np, loc, level, v)
token_type	*equation;
int		*np, loc, level;
long		v;
{
	int	i, j;
	int	n;
	int	op;
	int	oploc;
	int	endloc;
	double	d;

	if (equation[loc].level < level) {
/* First differentiate if it is a single variable or constant. */
/* If it is the specified variable, change it to the constant "1.0", */
/* otherwise change it to "0.0". */
		if (equation[loc].kind == VARIABLE
		    && ((v == MATCH_ANY && (equation[loc].token.variable & VAR_MASK) > SIGN)
		    || equation[loc].token.variable == v)) {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 1.0;
		} else {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 0.0;
		}
		return true;
	}
	op = 0;
	for (endloc = loc + 1;; endloc += 2) {
		if (endloc >= *np || equation[endloc].level < level)
			break;
		if (equation[endloc].level == level) {
			switch (op) {
			case 0:
			case PLUS:
			case MINUS:
				break;
			default:
/* Oops.  More than one operator on the same level in this expression. */
				printf(_("Error in d_recurse().\n"));
				return false;
			}
			op = equation[endloc].token.operatr;
			oploc = endloc;
		}
	}
	switch (op) {
	case 0:
	case PLUS:
	case MINUS:
		break;
	case TIMES:
		goto d_times;
	case DIVIDE:
		goto d_divide;
	case POWER:
		goto d_power;
	default:
/* Differentiate an unsupported operator. */
/* This is possible if the expression doesn't contain the specified variable. */
/* In that case, the expression is replaced with "0", otherwise return false. */
		for (i = loc; i < endloc; i += 2) {
			if (equation[i].kind == VARIABLE
			    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
			    || equation[i].token.variable == v)) {
				return false;
			}
		}
		blt(&equation[loc+1], &equation[endloc], (*np - endloc) * sizeof(token_type));
		*np -= (endloc - (loc + 1));
		equation[loc].level = level;
		equation[loc].kind = CONSTANT;
		equation[loc].token.constant = 0.0;
		return true;
	}
/* Differentiate PLUS and MINUS operators. */
/* Use rule: d(u+v) = d(u) + d(v), */
/* where "d()" is the derivative function, */
/* "u" and "v" are expressions. */
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].kind != OPERATOR) {
			if (!d_recurse(equation, np, i, level + 1, v))
				return false;
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	return true;
d_times:
/* Differentiate TIMES operator. */
/* Use rule: d(u*v) = u*d(v) + v*d(u). */
	if (*np + 1 + (endloc - loc) > n_tokens) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level++;
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += 1 + (endloc - loc);
	equation[endloc].level = level;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = PLUS;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 2, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 2, v))
		return false;
	return true;
d_divide:
/* Differentiate DIVIDE operator. */
/* Use rule: d(u/v) = (v*d(u) - u*d(v))/v^2. */
	if (*np + 3 + (endloc - loc) + (endloc - oploc) > n_tokens) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level += 2;
	equation[oploc].token.operatr = TIMES;
	j = 1 + (endloc - loc);
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += j;
	equation[endloc].level = level + 1;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = MINUS;
	j += endloc;
	blt(&equation[j+2+(endloc-oploc)], &equation[j], (*np - j) * sizeof(token_type));
	*np += 2 + (endloc - oploc);
	equation[j].level = level;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = DIVIDE;
	blt(&equation[j+1], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	j += endloc - oploc;
	equation[j].level = level + 1;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = POWER;
	j++;
	equation[j].level = level + 1;
	equation[j].kind = CONSTANT;
	equation[j].token.constant = 2.0;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 3, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 3, v))
		return false;
	return true;
d_power:
/* Differentiate POWER operator. */
/* Since we don't have logarithms, just do what we can without them. */
	for (i = oploc; i < endloc; i++) {
		if (equation[i].kind == VARIABLE
		    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
		    || equation[i].token.variable == v)) {
			if ((oploc - loc) == 1) {
				switch (equation[loc].kind) {
				case CONSTANT:
					d = equation[loc].token.constant;
					break;
				case VARIABLE:
					if (!var_is_const(equation[loc].token.variable, &d))
						return false;
					break;
				default:
					return false;
				}
				n = (endloc - oploc) + 2;
				if (*np + n > n_tokens) {
					error_huge();
				}
				blt(&equation[endloc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
				*np += n;
				n = endloc;
				equation[n].level = level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				equation[n].level = level;
				equation[n].kind = CONSTANT;
				equation[n].token.constant = log(d);
				n++;
				equation[n].level = level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				blt(&equation[n], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
				for (i = loc; i < endloc; i++) {
					equation[i].level++;
				}
				if (!d_recurse(equation, np, n, level + 1, v))
					return false;
				return true;
			} else {
				return false;
			}
		}
	}
	blt(&scratch[0], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	n = endloc - (oploc + 1);
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	if (n + (endloc - loc) + 2 > n_tokens) {
		error_huge();
	}
	blt(&scratch[n], &equation[loc], (endloc - loc) * sizeof(token_type));
	i = n;
	n += oploc + 1 - loc;
	for (; i < n; i++)
		scratch[i].level++;
	n += endloc - (oploc + 1);
	for (; i < n; i++)
		scratch[i].level += 2;
	scratch[n].level = level + 2;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = MINUS;
	n++;
	scratch[n].level = level + 2;
	scratch[n].kind = CONSTANT;
	scratch[n].token.constant = 1.0;
	n++;
	if (n + (oploc - loc) + 1 > n_tokens) {
		error_huge();
	}
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	j = n;
	blt(&scratch[n], &equation[loc], (oploc - loc) * sizeof(token_type));
	n += oploc - loc;
	if (*np - (endloc - loc) + n > n_tokens) {
		error_huge();
	}
	blt(&equation[loc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
	*np += loc + n - endloc;
	blt(&equation[loc], &scratch[0], n * sizeof(token_type));
	if (!d_recurse(equation, np, loc + j, level + 1, v))
		return false;
	return true;
}

/*
 * The derivative command.
 */
int
derivative(cp)
char	*cp;
{
	long	v;
	int	i, j;
	long	l;
	int	found;
	int	er;
	int	order;

	v = 0;
	order = 1;
	if (notdefined(cur_equation)) {
		return false;
	}
	if (*cp) {
		if (is_all(cp)) {
			v = MATCH_ANY;
			cp = skip_param(cp);
		} else {
			cp = parse_var2(&v, cp);
			if (cp == NULL) {
				return false;
			}
		}
		if (isascii(*cp) && isdigit(*cp)) {
			order = atoi(cp);
			cp = skip_num(cp);
		}
	}
	if (extra_garbage(cp)) {
		return false;
	}
	if (order < 1) {
		printf(_("Order out of range.\n"));
		usage_flag = true;
		return false;
	}
	if (v == 0) {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	i = next_espace();
	found = false;
	er = !solved_equation(cur_equation);
	er |= ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN);
	for (j = 0; j < n_rhs[cur_equation]; j += 2) {
		if (rhs[cur_equation][j].kind == VARIABLE) {
			if (rhs[cur_equation][j].token.variable == v)
				found = true;
		}
	}
	if (v && v != MATCH_ANY && !found) {
		printf(_("Variable not found in RHS.  Result would be zero.\n"));
		return false;
	}
	blt(rhs[i], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
	simp_loop(&rhs[i][0], &n_rhs[i]);
	for (j = 0; j < order; j++) {
		if (!differentiate(&rhs[i][0], &n_rhs[i], v)) {
			printf(_("Differentiation failed.\n"));
			return false;
		}
#if	true
		simpa_side(rhs[i], &n_rhs[i], false, true);
#else
		simp_loop(&rhs[i][0], &n_rhs[i]);
#endif
	}
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	if (!er && order < 10) {
/* increment the number of primes in the LHS variable */
		l = lhs[i][0].token.variable;
		for (j = 1; j <= order; j++) {
			l += PRIME_INCREMENT;
			if (l < 0)
				break;
		}
		if (l >= 0) {
			lhs[i][0].token.variable = l;
		}
	}
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

/*
 * The taylor command.
 */
int
taylor(cp)
char	*cp;
{
	long	v;
	int	i, j, k;
	int	i1, j1;
	int	level;
	int	found;
	int	n;
	int	order;
	token_type	*ep;
	double	d;
	char	*cp1;
	int	our;
	int	our_nlhs, our_nrhs;

	if (notdefined(cur_equation)) {
		return false;
	}
	if (*cp == '\0') {
		if (!prompt_var(&v))
			return false;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
	}
	i = next_espace();
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	n_rhs[i] = 0;
	our = next_espace();
	n_lhs[i] = 0;
	found = false;
	for (j = 0; j < n_rhs[cur_equation]; j += 2) {
		if (rhs[cur_equation][j].kind == VARIABLE) {
			if (rhs[cur_equation][j].token.variable == v)
				found = true;
		}
	}
	if (!found) {
		printf(_("Variable not found in RHS.\n"));
		return false;
	}
	printf(_("Taylor approximation of current equation about "));
	list_var(v, true, false);
	printf(_(" = point.\n"));
	blt(rhs[our], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	our_nrhs = n_rhs[cur_equation];
	if (!differentiate(&rhs[our][0], &our_nrhs, v)) {
		printf(_("Differentiation failed.\n"));
		return false;
	}
	strcpy(prompt_str, _("Enter point: "));
	if (!get_expr(&lhs[our][0], &our_nlhs)) {
		return false;
	}
oops2:
	strcpy(prompt_str, _("Enter order (number of derivatives to take): "));
	if ((cp1 = getstring((char *) &scratch[0], n_tokens * sizeof(token_type))) == NULL)
		return false;
	if (*cp1 == '\0' || *cp1 == '\n') {
		printf(_("Derivatives will be taken until they reach zero.\n"));
		order = 32000;
	} else {
		if (!isascii(*cp1) || !isdigit(*cp1))
			goto oops2;
		order = atoi(cp1);
		if (order < 0) {
			goto oops2;
		}
	}
	n = 0;
	i1 = 0;
	blt(&rhs[i][0], &rhs[cur_equation][0], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
loop_again:
	for (k = i1; k < n_rhs[i]; k += 2) {
		if (rhs[i][k].kind == VARIABLE && rhs[i][k].token.variable == v) {
			level = rhs[i][k].level;
			if ((n_rhs[i] + our_nlhs - 1) > n_tokens)
				error_huge();
			blt(&rhs[i][k+our_nlhs], &rhs[i][k+1], (n_rhs[i] - (k + 1)) * sizeof(token_type));
			n_rhs[i] += our_nlhs - 1;
			j1 = k;
			blt(&rhs[i][k], &lhs[our][0], our_nlhs * sizeof(token_type));
			k += our_nlhs;
			for (; j1 < k; j1++)
				rhs[i][j1].level += level;
			k--;
		}
	}
	if ((n_rhs[i] + our_nlhs + 7) > n_tokens)
		error_huge();
	for (k = i1; k < n_rhs[i]; k++)
		rhs[i][k].level++;
	k = n_rhs[i];
	ep = &rhs[i][k];
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = TIMES;
	ep++;
	ep->level = 3;
	ep->kind = VARIABLE;
	ep->token.variable = v;
	ep++;
	ep->level = 3;
	ep->kind = OPERATOR;
	ep->token.operatr = MINUS;
	k += 3;
	j1 = k;
	blt(&rhs[i][k], &lhs[our][0], our_nlhs * sizeof(token_type));
	k += our_nlhs;
	for (; j1 < k; j1++)
		rhs[i][j1].level += 3;
	ep = &rhs[i][k];
	ep->level = 2;
	ep->kind = OPERATOR;
	ep->token.operatr = POWER;
	ep++;
	ep->level = 2;
	ep->kind = CONSTANT;
	ep->token.constant = n;
	ep++;
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = DIVIDE;
	ep++;
	for (d = 1.0, j = 2; j <= n; j++)
		d *= j;
	ep->level = 1;
	ep->kind = CONSTANT;
	ep->token.constant = d;
	k += 4;
	n_rhs[i] = k;
	for (; i1 < k; i1++)
		rhs[i][i1].level++;
	simp_loop(&rhs[i][0], &n_rhs[i]);
	if (n < order) {
		if (n > 0) {
			if (!differentiate(&rhs[our][0], &our_nrhs, v)) {
				printf(_("Differentiation failed.\n"));
				return false;
			}
		}
		simpa_side(&rhs[our][0], &our_nrhs, false, true);
		side_debug(3, rhs[our], our_nrhs);
		if (our_nrhs != 1 || rhs[our][0].kind != CONSTANT
		    || rhs[our][0].token.constant != 0.0) {
			k = n_rhs[i];
			if ((k + 1 + our_nrhs) > n_tokens)
				error_huge();
			for (i1 = 0; i1 < k; i1++)
				rhs[i][i1].level++;
			rhs[i][k].level = 1;
			rhs[i][k].kind = OPERATOR;
			rhs[i][k].token.operatr = PLUS;
			k++;
			i1 = k;
			blt(&rhs[i][k], &rhs[our][0], our_nrhs * sizeof(token_type));
			n_rhs[i] = k + our_nrhs;
			n++;
			goto loop_again;
		}
	}
	n_lhs[i] = n_lhs[cur_equation];
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}
