(* Copyright (C) 2005, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

(* $Id: dependenciesParser.ml 7922 2007-11-25 13:12:25Z tassi $ *)

(* FG
 * From Cambridge dictionary
 * Dependency:
 * a country which is supported and governed by another country
 * Dependence:
 * when you need something or someone all the time, especially in order to
 * continue existing or operating
 *
 * Fate vobis ...
 *)

exception UnableToInclude of string

  (* statements meaningful for matitadep *)
type dependency =
  | IncludeDep of string
  | BaseuriDep of string
  | UriDep of UriManager.uri
  
let pp_dependency = function
  | IncludeDep str -> "include \"" ^ str ^ "\""
  | BaseuriDep str -> "set \"baseuri\" \"" ^ str ^ "\""
  | UriDep uri -> "uri \"" ^ UriManager.string_of_uri uri ^ "\""

let parse_dependencies lexbuf = 
  let tok_stream,_ =
    CicNotationLexer.level2_ast_lexer.Token.tok_func (Obj.magic lexbuf)
  in
  let rec parse acc = 
   try
    (parser
    | [< '("QSTRING", s) >] ->
        (* because of alias id qstring = qstring :-( *)
        (try
          parse (UriDep (UriManager.uri_of_string s) :: acc)
         with
          UriManager.IllFormedUri _ -> parse acc)
    | [< '("URI", u) >] ->
        parse (UriDep (UriManager.uri_of_string u) :: acc)
    | [< '("IDENT", "include"); '("QSTRING", fname) >] ->
        parse (IncludeDep fname :: acc)
    | [< '("IDENT", "set"); '("QSTRING", "baseuri"); '("QSTRING", baseuri) >] ->
        parse (BaseuriDep baseuri :: acc)
    | [< '("EOI", _) >] -> acc
    | [< 'tok >] -> parse acc
    | [<  >] -> acc) tok_stream
   with
      Stream.Error _ -> parse acc
    | CicNotationLexer.Error _ -> parse acc
  in
  List.rev (parse [])

let make_absolute paths path =
   let rec aux = function
   | [] -> ignore (Unix.stat path); path
   | p :: tl ->
      let path = p ^ "/" ^ path in
       try
         ignore (Unix.stat path);
         HLog.debug ("Including "^path^" with path: " ^ p);
         path
       with Unix.Unix_error _ -> aux tl
   in
   try
     aux paths
   with Unix.Unix_error _ -> raise (UnableToInclude path)
;;
       
let baseuri_of_script ~include_paths file = 
 let file = make_absolute include_paths file in
 let ic = open_in file in
 let istream = Ulexing.from_utf8_channel ic in
 let rec find_baseuri =
  function
     [] -> failwith ("No baseuri defined in " ^ file)
   | BaseuriDep s::_ -> s
   | _::tl -> find_baseuri tl in
 let buri = find_baseuri (parse_dependencies istream) in
 let uri = Http_getter_misc.strip_trailing_slash buri in
 if String.length uri < 5 || String.sub uri 0 5 <> "cic:/" then
   HLog.error (file ^ " sets an incorrect baseuri: " ^ buri);
 uri,file
