/** Configure a Matrox G550 for TV output.
 * @author Shaun Jackman <sjackman@debian.org>
 * @copyright Copyright 2004 Shaun Jackman
 */

#include "config.h"
#include <fcntl.h>
#include <getopt.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <unistd.h>


#define PALWTADD 0x3C00
#define X_DATAREG 0x3C0A

#define XTVO_IDX 0x87
#define XTVO_DATA 0x88


static volatile unsigned char* base;

static int option_verbose;


#define read_crtc2(reg) base[reg]


static void
write_crtc2( int reg, int data)
{
	int in;
	base[reg] = data;
	in = base[reg];
	if( in != data)
		fprintf( stderr,
				PACKAGE ": CRTC2(%02x): wrote %02x, read %02x\n",
				reg, data, in);
}


static void
write_crtc2_16( int reg, int data)
{
	uint16_t* p = (uint16_t*)&base[reg];
	*p = data;
}


static void
write_crtc2_32( int reg, int data)
{
	uint32_t* p = (uint32_t*)&base[reg];
	*p = data;
}


static int
read_dac( int reg)
{
	write_crtc2( PALWTADD, reg);
	return read_crtc2( X_DATAREG);
}


static void
write_dac( int reg, int data)
{
	int in;
	write_crtc2( PALWTADD, reg);
	write_crtc2( X_DATAREG, data);
	in = read_crtc2( X_DATAREG);
	if( in != data)
		fprintf( stderr,
				PACKAGE ": DAC(%02x): wrote %02x, read %02x\n",
				reg, data, in);
}


static int
read_tvo( int reg)
{
	write_dac( XTVO_IDX, reg);
	return read_dac( XTVO_DATA);
}


static void
write_tvo( int reg, int data)
{
	int in;
	write_dac( XTVO_IDX, reg);
	write_dac( XTVO_DATA, data);
	in = read_dac( XTVO_DATA);
	if( in != data)
		fprintf( stderr,
				PACKAGE ": TVO(%02x): wrote %02x, read %02x\n",
				reg, data, in);
}


#define print_crtc2( reg) \
	printf( "%04x %02x\n", reg, read_crtc2( reg))


#define print_dac( reg) \
	printf( "%02x %02x\n", reg, read_dac( reg))


#define print_tvo( reg) \
	printf( "%02x %02x\n", reg, read_tvo( reg))


static void
print_registers( void)
{
	int i;
	for( i = 0x3c10; i < 0x3c14; i++)
		print_crtc2( i);
	for( i = 0x3c44; i < 0x3c48; i++)
		print_crtc2( i);
	for( i = 0x3c4c; i < 0x3c50; i++)
		print_crtc2( i);

	for( i = 0; i < 0x90; i++)
		print_dac( i);
	// 0x90-0x9b (read-only)
	for( i = 0x9c; i < 0xb0; i++)
		print_dac( i);

	for( i = 0; i < 0x40; i++)
		print_tvo( i);
	for( i = 0x80; i < 0x86; i++)
		print_tvo( i);
}


static void
write_registers( void)
{
	// ModeLine "720x480i" 13.5 720 736 800 858 480 482 486 525 Interlace
	// ModeLine "480i" 13.5 720 736 800 856 480 484 492 525 Interlace
	// ModeLine "NTSC-DTV-59.94i" 13.469 720 760 816 856 480 482 488 525 Interlace
	double dotclock = 13.5e6;
	int hdsp = 720, hbeg = 736, hend = 800, httl = 858,
		vdsp = 480, vbeg = 482, vend = 486, vttl = 525,
		pitch = 736;
	double fh = dotclock / httl;
	double fv = fh / vttl;

	if( option_verbose) {
		printf( "Section \"Monitor\"\n"
				"\tIdentifier \"NTSC TV\"\n"
				"\tHorizSync %.3f # kHz\n"
				"\tVertRefresh %.3f # Hz\n"
				"\tModeLine \"%dx%d@%.2f\" "
					"%.2f %d %d %d %d %d %d %d %d\n"
				"EndSection\n",
				2 * fh / 1000, 2 * fv,
				hdsp, vdsp, 2 * fv, 2 * dotclock / 1000000,
				hdsp, hbeg, hend, httl, vdsp, vbeg, vend, vttl);
		printf( "Section \"Device\"\n"
				"\tIdentifier \"G550 2\"\n"
				"\tDriver     \"mga\"\n"
				"\tOption     \"HWcursor\" \"off\"\n"
				"\tOption     \"AGPMode\" \"4\"\n"
				"\tBusID      \"PCI:1:0:0\"\n"
				"\tScreen     1\n"
				"EndSection\n"
				"Section \"Screen\"\n"
				"\tIdentifier   \"Screen 2\"\n"
				"\tDevice       \"G550 2\"\n"
				"\tMonitor      \"NTSC TV\"\n"
				"\tDefaultDepth 16\n"
				"\tSubSection \"Display\"\n"
				"\t\tDepth 16\n"
				"\t\tModes \"%dx%d@%.2f\"\n"
				"\tEndSubSection\n"
				"EndSection\n", hdsp, vdsp, 2 * fv);
	}

	switch( httl % 8) {
		case 0:
			fputs( PACKAGE ": c2ntscen is unnecessary\n", stderr);
			break;
		case 2:
			// c2ntscen adds two pixels to the back porch.
			httl -= 2;
			break;
		default:
			fputs( PACKAGE
					": invalid horizontal total (httl % 8 == 2)\n",
					stderr);
	}

	write_crtc2( 0x3c10, 0x23); // C2CTL0
	// c2en=1 c2pixclksel=01 c2pixclkdis=0 c2hiprilvl=010
	write_crtc2( 0x3c11, 0x51); // C2CTL1 c2maxhipri=001 reserved=01010
	write_crtc2( 0x3c12, 0x40); // C2CTL2 crtcdacsel=0 c2depth=010
	write_crtc2( 0x3c13, 0x02); // C2CTL3
	// c2vcbcr=0 c2interlace=1 c2fieldlength=0 c2fieldpol=0
	// c2vidrstmod=00 c2hploaden=0 c2vploaden=0
	write_crtc2( 0x3c44, 0x00); // C2MISC0
	// c2fieldline0=000 c2fieldline1=000
	write_crtc2( 0x3c45, 0x00); // C2MISC1 c2hsyncpol=0 c2vsyncpol=0
	write_crtc2( 0x3c46, 0xf1); // C2MISC2 c2vlinecomp=000011110001
	write_crtc2( 0x3c47, 0x00); // C2MISC3
	write_crtc2( 0x3c4c, 0x10); // C2DATACTL0
	// c2dithen=0 c2yfilten=0 c2cbcrfilten=0 c2subpicen=0
	// c2ntscen=1 c2statickeyen=0 c2offsetdiven=0 c2uyvyfmt=0
	write_crtc2( 0x3c4d, 0x00); // C2DATACTL1 c2bpp15h=00000000
	write_crtc2( 0x3c4e, 0x00); // C2DATACTL2 c2bpp15l=00000000
	write_crtc2( 0x3c4f, 0x00); // C2DATACTL3 reserved=00000000
	
	// 0x3c10 CTCTL (read-write)
	write_crtc2_16( 0x3c14, httl-8); // C2HPARAM  c2htotal
	write_crtc2_16( 0x3c16, hdsp-8); // C2HPARAM  c2hdispen
	write_crtc2_16( 0x3c18, hbeg-8); // C2HSYNC   c2hsyncstr
	write_crtc2_16( 0x3c1a, hend-8); // C2HSYNC   c2hsyncend
	write_crtc2_16( 0x3c1c, vttl/2-1); // C2VPARAM  c2vtotal
	write_crtc2_16( 0x3c1e, vdsp/2-1); // C2VPARAM  c2vdispend
	write_crtc2_16( 0x3c20, vbeg/2-1); // C2VSYNC   c2vsyncstr
	write_crtc2_16( 0x3c22, vend/2-1); // C2VSYNC   c2vsyncend
	write_crtc2_16( 0x3c24, hbeg); // C2PRELOAD c2hpreload
	write_crtc2_16( 0x3c26, vbeg/2); // C2PRELOAD c2vpreload
	write_crtc2_32( 0x3c28, 0x01800000 + pitch*2); // C2STARTADD0
	write_crtc2_32( 0x3c2c, 0x01800000); // C2STARTADD1
	//write_crtc2_32( 0x3c30, 0); // C2PL2STARTADD0
	//write_crtc2_32( 0x3c34, 0); // C2PL2STARTADD1
	//write_crtc2_32( 0x3c38, 0); // C2PL3STARTADD0
	//write_crtc2_32( 0x3c3c, 0); // C2PL3STARTADD1
	write_crtc2_16( 0x3c40, pitch*4); // C2OFFSET
	// 0x3c44 C2MISC (read-write)
	// 0x3c48 C2VCOUNT (read-only)
	// 0x3c4c C2DATACTL (read-write)
	//write_crtc2( 0x3c50, 0); // C2SUBPICLUT c2subpiclutx
	//write_crtc2( 0x3c51, 0); // C2SUBPICLUT c2subpicylut
	//write_crtc2( 0x3c52, 0); // C2SUBPICLUT c2subpiccblut
	//write_crtc2( 0x3c53, 0); // C2SUBPICLUT c2subpiccrlut
	//write_crtc2_32( 0x3c54, 0); // C2PICSTARTADD0
	//write_crtc2_32( 0x3c58, 0); // C2PICSTARTADD1
	
	write_dac( 0x04, 0x03); // XCURADDL curadrl
	write_dac( 0x05, 0x0a); // XCURADDH curadrh
	//write_dac( 0x06, 0x02); // XCURCTRL curmod=010
	write_dac( 0x0c, 0xff); // XCURCOL1RED curcol
	write_dac( 0x0d, 0xff); // XCURCOL1GREEN curcol
	write_dac( 0x0e, 0xff); // XCURCOL1BLUE curcol
	write_dac( 0x10, 0x80); // XCURCOL2RED curcol
	write_dac( 0x11, 0x80); // XCURCOL2GREEN curcol
	write_dac( 0x12, 0x80); // XCURCOL2BLUE curcol
	write_dac( 0x1a, 0x0b); // XPIXCLKCTRL pixclksl=11 pixclkdis=0 pixpllpdN=1
	write_dac( 0x1d, 0x00); // XGENCTRL alphaen=0 pedon=0 iogsyncdis=0
	write_dac( 0x1e, 0x1e); // XMISCCTRL
	// dacpdN=0 mfcsel=11 vga8dac=1 ramcs=1 vdoutsel=000
	write_dac( 0x1f, 0x00); // XPANELMODE
	// panelsel=0 panelctl=000 panhsoff=0 panvsoff=0 panhspol=0 panvspol=0
	write_dac( 0x20, 0x08); // XMAFCDELAY mafcclkdel=1000
	write_dac( 0x2a, 0x40); // XGENIOCTRL ddcoe=0000 miscoe=100
	//write_dac( 0x2c, 0x02); // XSYSPLLM syspllm=00010
	//write_dac( 0x2d, 0x23); // XSYSPLLN sysplln=0100011
	write_dac( 0x2e, 0x00); // XSYSPLLP syspllp=000 sysplls=00
	//write_dac( 0x3a, 0x28); // XSENSETEST
	write_dac( 0x3e, 0x1f); // XCRCBITSEL crcsel=11111
	write_dac( 0x40, 0x53); // XCOLMSK colmsk
	write_dac( 0x42, 0xfd); // XCOLKEY colkey
	write_dac( 0x4c, 0x01); // XPIXPLLCM pixpllm=00001
	write_dac( 0x4d, 0x0d); // XPIXPLLCN pixplln=0001101
	write_dac( 0x4e, 0x03); // XPIXPLLCP pixpllp=111 pixplls=00
	write_dac( 0x60, 0xfe);
	write_dac( 0x80, 0xff);
	write_dac( 0x81, 0xff);
	write_dac( 0x82, 0xff);
	write_dac( 0x83, 0xff);
	write_dac( 0x84, 0xbf);
	write_dac( 0x85, 0xff);
	write_dac( 0x86, 0x7f);
	write_dac( 0x8a, read_dac( 0x8a) | 0x0c); // XOUTPUTCONN
	write_dac( 0x8b, 0x00); // XSYNCCTRL
	//write_dac( 0x8c, 0x40); // XVIDPLLSTAT
	write_dac( 0x8d, 0x0a); // XVIDPLLP pixpllp=010 pixplls=01
	write_dac( 0x8e, 0x00); // XVIDPLLM pixpllm=00000
	write_dac( 0x8f, 0x0e); // XVIDPLLN pixplln=0001110
	write_dac( 0xa0, 0x1f); // XPWRCTRL
	write_dac( 0xa2, 0x2f); // XPANMODE

	write_tvo( 0x3e, 0x01); // PROGRAMMING
	write_tvo( 0x00, 0x21); // XCHROMA0
	write_tvo( 0x01, 0xf0); // XCHROMA1
	write_tvo( 0x02, 0x7c); // XCHROMA2
	write_tvo( 0x03, 0x1f); // XCHROMA3
	write_tvo( 0x06, 0x09);
	write_tvo( 0x08, 0x7e); // horizontal sync
	write_tvo( 0x09, 0x44); // burst length
	write_tvo( 0x0a, 0x76); // horizontal back porch
	write_tvo( 0x0b, 0x49);
	write_tvo( 0x0e, 0x48); // BLACK_LEVEL0
	write_tvo( 0x0f, 0x00); // BLACK_LEVEL1
	write_tvo( 0x10, 0x42);
	write_tvo( 0x11, 0x03);
	//write_tvo( 0x17, 0x83); // v_total / 4
	//write_tvo( 0x18, 0x01); // v_total & 3
	write_tvo( 0x19, 0x00);
	write_tvo( 0x1e, 0x9c); // WHITE_LEVEL0
	write_tvo( 0x1f, 0x02); // WHITE_LEVEL1
	write_tvo( 0x20, 0x82); // SATURATION0
	write_tvo( 0x22, 0x82); // SATURATION1
	write_tvo( 0x29, 0x11);
	write_tvo( 0x2c, 0x20); // horizontal front porch
	write_tvo( 0x31, 0xb4); // hvis / 8
	//write_tvo( 0x32, 0x00); // hvis & 7
	write_tvo( 0x33, 0x14); // upper blanking
	write_tvo( 0x35, 0x00); // written multiple times
	write_tvo( 0x37, 0xbd);
	write_tvo( 0x38, 0xda);
	write_tvo( 0x3b, 0x15);
	write_tvo( 0x3c, 0x42);
	write_tvo( 0x3d, 0x03);
	write_tvo( 0x80, 0x03); // SCART is 0x43
	write_tvo( 0x82, 0x14); // YSYNCL
	//write_tvo( 0x83, 0x00); // YSYNCH
	//write_tvo( 0x84, 0x01); // XSYNCL
	//write_tvo( 0x85, 0x00); // XSYNCH
	write_tvo( 0x3e, 0x00); // PROGRAMMING
	read_tvo( 0x1f);
}


static void
init( unsigned int address)
{
	int fd = open( "/dev/mem", O_RDWR);
	if (fd < 0) {
		perror( PACKAGE ": /dev/mem");
		exit( EXIT_FAILURE);
	}
	base = mmap( NULL, 16384, PROT_READ|PROT_WRITE, MAP_SHARED,
			fd, (off_t)address);
	if( base == NULL) {
		perror( PACKAGE ": mmap failed");
		exit( EXIT_FAILURE);
	}
	close( fd);
}


int
main( int argc, char* argv[])
{
	static int option_address, option_read, option_write, option_help, option_version;
	static struct option options[] = {
		{ "address", required_argument, NULL, 'a' },
		{ "read", no_argument, &option_read, 'r' },
		{ "write", no_argument, &option_write, 'w' },
		{ "verbose", no_argument, &option_verbose, 'v' },
		{ "help", no_argument, &option_help, 'H' },
		{ "version", no_argument, &option_version, 'V' },
		{ 0, 0, 0, 0}
	};

	for(;;) {
		char c = getopt_long( argc, argv, "a:rwv", options, NULL);
		if( c == -1)
			break;
		switch( c) {
			case 'a': option_address = strtoul( optarg, NULL, 0); break;
			case 'r': option_read = 1; break;
			case 'w': option_write = 1; break;
			case 'v': option_verbose = 1; break;
			case '?': exit( EXIT_FAILURE);
		}
	}

	if( option_version) {
		puts(
PACKAGE_STRING "\n"
"Written by Shaun Jackman.\n\n"
"Copyright 2004 Shaun Jackman\n"
"This is free software; see the source for copying conditions. There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.");
		exit( EXIT_SUCCESS);
	}

	if( option_help) {
		puts(
"Usage:  " PACKAGE " --address=ADDRESS [OPTION]...\n"
"Configure the Matrox card at ADDRESS for NTSC TV output.\n\n"
"  -a, --address=ADDRESS  the base memory-mapped input/output address\n"
"  -r, --read             read configuration from the card\n"
"  -w, --write            write TV out configuration to the card\n"
"  -v, --verbose          display verbose output\n"
"      --help             display this help and exit\n"
"      --version          display version information and exit\n\n"
"Report bugs to " PACKAGE_BUGREPORT ".");
		exit( EXIT_SUCCESS);
	}

	if( option_address == 0) {
		fputs(
PACKAGE ": missing mandatory option --address\n"
"Try `" PACKAGE " --help' for more information.\n", stderr);
		exit( EXIT_FAILURE);
	}

	if( !option_read && !option_write)
		option_write = 1;

	init( option_address);

	if( option_write)
		write_registers();

	if( option_read)
		print_registers();

	exit( EXIT_SUCCESS);
}
