/*

    ========== licence begin  GPL
    Copyright (c) 2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/
#include "Oms/OMS_ClassIdEntry.hpp"
#include "Oms/OMS_Session.hpp"
#include "Oms/OMS_ObjectContainer.hpp"

/*----------------------------------------------------------------------*/
// Implementation OMS_ClassIdEntry
/*----------------------------------------------------------------------*/

OMS_ClassIdEntry::OMS_ClassIdEntry (OMS_ContainerInfo* pContainerInfo,OMS_Context* c) 
  : m_clsidHashNext(NULL), 
    m_guidHashNext(NULL), 
    m_containerInfo(pContainerInfo), 
    m_index(), 
    m_freeHead(NULL), 
    m_freeHeadInVersion(NULL), 
    m_useCachedKeys(false),  
    m_cacheMiss(),           
    m_cacheMissCmp(NULL)     
{
  m_index.AdviseCompare(this);
  m_index.AdviseAllocator(c);
  pContainerInfo->IncRefCnt(c->GetCurrentSession()->m_lcSink);
}

/*----------------------------------------------------------------------*/

void OMS_ClassIdEntry::chainFree (OMS_Context&c, OmsAbstractObject* po, int caller) 
{
  if (po) {
    OmsObjectContainerPtr p = OmsObjectContainer::containerPtr(po);

#ifdef USE_SYSTEM_ALLOC_CO13
    if (p->VarObjFlag()) {
      REINTERPRET_CAST(OMS_VarObjInfo*, po)->freeVarObj(&c);
    }
    c.deallocate(p);
#else
    if (p->IsVerNewObject()) {
#ifdef _ASSERT_OMS
      OmsObjectContainerPtr curr = m_freeHeadInVersion->free;
      while (curr) {
        if (curr == p) {
          throw DbpError (-10, "Duplicate in freelist", __MY_FILE__, __LINE__); 
        }
        curr = curr->GetNextFreeList();
      }
#endif
      p->InitializeForFreeList(caller);
      p->SetNextFreeList(m_freeHeadInVersion->free);
      m_freeHeadInVersion->free = p;
    } else {
#ifdef _ASSERT_OMS
      OmsObjectContainerPtr curr = m_freeHead->free;
      while (curr) {
        if (curr == p) {
          throw DbpError (-10, "Duplicate in freelist", __MY_FILE__, __LINE__); 
        }
        curr = curr->GetNextFreeList();
      }
#endif
      if (p->VarObjFlag()) {
        REINTERPRET_CAST(OMS_VarObjInfo*, po)->freeVarObj(&c);
      }
      p->InitializeForFreeList(caller);
      p->SetNextFreeList(m_freeHead->free);
      m_freeHead->free = p;
    }
#endif
  }
}

/*----------------------------------------------------------------------*/

void OMS_ClassIdEntry::chainFree (OMS_Context&c, OmsObjectContainerPtr& p, int caller) 
{
  if (NULL != p) {

#ifdef USE_SYSTEM_ALLOC_CO13
    if (p->VarObjFlag()) {
      REINTERPRET_CAST(OMS_VarObjInfo*, &p->m_pobj)->freeVarObj(&c);
    }
    c.deallocate(p);
    p = NULL;
#else
    if (p->IsVerNewObject()) {
#ifdef _ASSERT_OMS
      OmsObjectContainerPtr curr = m_freeHeadInVersion->free;
      while (curr) {
        if (curr == p) {
          throw DbpError (-10, "Duplicate in freelist", __MY_FILE__, __LINE__); 
        }
        curr = curr->GetNextFreeList();
      }
#endif
      p->InitializeForFreeList(caller);
      p->SetNextFreeList(m_freeHeadInVersion->free);
      m_freeHeadInVersion->free = p;
      p = NULL;
    } else {
#ifdef _ASSERT_OMS
      OmsObjectContainerPtr curr = m_freeHead->free;
      while (curr) {
        if (curr == p) {
          throw DbpError (-10, "Duplicate in freelist", __MY_FILE__, __LINE__); 
        }
        curr = curr->GetNextFreeList();
      }
#endif
      if (p->VarObjFlag()) {
        REINTERPRET_CAST(OMS_VarObjInfo*, &p->m_pobj)->freeVarObj(&c);
      }
      p->InitializeForFreeList(caller);
      p->SetNextFreeList(m_freeHead->free);
      m_freeHead->free = p;
      p = NULL;
    }
#endif
  }
}

/*----------------------------------------------------------------------*/
// PTS 1117571
// Delete complete cache miss structure                           
void OMS_ClassIdEntry::DropCacheMisses(OMS_Context* pContext) 
{ 
  const char* msg = "OMS_ClassIdEntry::DropCacheMisses";

  // Delete dynamically created cache-miss-entries 
  cgg250AvlTree<OMS_CacheMissEntry*,OMS_CacheMissEntry,OMS_Context>::Iterator nodeIter;
  OMS_CacheMissEntry*const* pCurrEntry;

  // Delete the cache miss entries
  nodeIter = m_cacheMiss.First();
  while (nodeIter) {
    pCurrEntry = nodeIter();
    (*pCurrEntry)->deleteSelf(pContext); 

    ++nodeIter;
  }

  // Delete the tree itself
  m_cacheMiss.DeleteAll();
  OMS_DETAIL_TRACE(omsTrKey, pContext->m_session->m_lcSink, msg << " DeleteAll");
}

/*----------------------------------------------------------------------*/

void OMS_ClassIdEntry::VersionDelIndex(bool cachedKeysOnly, OMS_Context* pContext) 
{
  const char* msg = "OMS_ClassIdEntry::VersionDelIndex";

  if (!cachedKeysOnly || !pContext->IsVersion()){
    // Delete the complete tree. This is possible, when either
    // - all entries should be deleted and not only those which are added by the
    //   cached key feature, or
    // - when no version is active
    m_index.DeleteAll();
    OMS_DETAIL_TRACE(omsTrKey, pContext->m_session->m_lcSink, msg << " DeleteAll");
  }
  else {
    // Delete only those entries which were added by the cached key feature
    // As it is not possible to delete entries while looping over an iterator
    // a temporary list is created with those entries which should be remain,
    // and after the whole index is deleted those entries are inserted again.
    cgg250AvlTree<unsigned char*, OMS_ClassIdEntry, OMS_Context>::Iterator objectIter;
    cgg251DCList<OmsObjectContainer*, OMS_Context> tmpList;    
    OmsObjectContainer* pCurrObject;

    tmpList.advise_allocator(pContext);

    objectIter = m_index.First();
    while (objectIter) {
      pCurrObject = VersionGetInfoFromNode(objectIter());          
      if (!pCurrObject) {
        throw DbpError (DbpError::DB_ERROR, e_system_error, msg, __MY_FILE__, __LINE__); 
      }

      if (pContext->IsVersionOid (pCurrObject->m_oid)) {
        tmpList.push_back(pCurrObject);
      }
      else {
        OMS_DETAIL_TRACE(omsTrKey, pContext->m_session->m_lcSink, msg << " Delete Object " << pCurrObject->m_oid);
      }

      ++objectIter;
    }

    m_index.DeleteAll();
    for (cgg251dclIterator<OmsObjectContainer*,OMS_Context> listIter 
      = tmpList.begin(); listIter; ++listIter) {
        VersionAddKey(*listIter(), pContext);
      }
      tmpList.remove_all();
  }
}

/*----------------------------------------------------------------------*/

// PTS 1117571
bool OMS_ClassIdEntry::SetCachedKeys(bool on, OMS_Context* pContext) 
{
  // Cached Keys are only defined on keyed objects
  //if (!IsKeyedObject()){
  if (!GetKeyDesc().IsKeyed()){
    return false;
  }

  // No change => nothing to do
  if (on == UseCachedKeys()){
    return false;
  }

  if (on) {
    // Cached Keys are switched on

    // Initialization of structures for cache misses
    m_cacheMissCmp = new (GetKeyDesc().GetLen(), pContext) OMS_CacheMissEntry (NULL, GetKeyDesc().GetLen());
    m_cacheMiss.AdviseCompare(m_cacheMissCmp);               
    m_cacheMiss.AdviseAllocator(pContext);  

    m_useCachedKeys = true;
  }
  else {
    // Cached Keys are switched off

    // Delete all entries which are not created in a version
    VersionDelIndex(true, pContext);

    // Reset of structures for cache misses
    DropCacheMisses(pContext);
    if (m_cacheMissCmp != NULL){
      m_cacheMissCmp->deleteSelf(pContext); 
      m_cacheMissCmp = NULL;
    }
    m_cacheMiss.AdviseCompare(NULL);               
    m_cacheMiss.AdviseAllocator(NULL);  

    m_useCachedKeys = false;
  }

  return true;
}

