/*!
  @file           Tools_Arguments.hpp
  @author         Bernd Vorsprach - bernd.vorsprach@sap.com
  @brief          argument parsing - Specification

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/

#ifndef _Tools_Arguments_HPP_
#define _Tools_Arguments_HPP_

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#include "SAPDB/ToolsCommon/Tools_DynamicUTF8String.hpp"
#include "SAPDB/ToolsCommon/Tools_Array.hpp"

/*!
  @brief     argument object

  This class may parse a string with arguments or options. The
  class builts a list of name/value pairs from the tokens 
  of input string. A name value/pair is a token with a assign 
  character ('=') inside. To enclose withspaces in a token use 
  doublequotes ('"'). To mask special characters use the
  backslash ('\'). Please regard there are some shells which
  will interpret and remove special characters like doublequotes 
  and backslashes too.

  Examples for tokens and their results:

  <ul>
  <li><b>name=value</b>&nbsp;&nbsp;      This creates a name/value pair.</li>
  <li><b>value</b>&nbsp;&nbsp;           A token without assign character ('=') creates a value without name.</li>
  <li><b>"name=value"</b>&nbsp;&nbsp;    This create a value without name, because the assign character is inside the doublequotes.</li>
  <li><b>"name"="value"</b>&nbsp;&nbsp;  This create a name/value pair. There may be whitespaces inside name or value.</li>
  <li><b>=value</b>&nbsp;&nbsp;          This creates a value without name.</li>
  <li><b>name=</b>&nbsp;&nbsp;           This creates a name without value.</li>
  <li><b>-name</b>&nbsp;&nbsp;           This creates a name without value.</li>
  <li><b>-name=value</b>&nbsp;&nbsp;     This creates a name/value pair.</li>
  </ul>
 */
class Tools_Arguments {

public:
  /*!
    @brief   Default constructor 
  */
  Tools_Arguments
    ( );

  /*!
    @brief   Constructor 
  */
  Tools_Arguments
    ( Tools_DynamicUTF8String::ConstPointer src );

  /*!
    @brief   Destructor
  */
  ~Tools_Arguments
    (  );

  /*!
    @brief   Set arguments
  */
  void setArguments
    ( Tools_DynamicUTF8String::ConstPointer src );

  /*!
    @brief  read a value by name
    @param  sName         [IN] - name of value
    @param  bCasesensitiv [IN] - case sensitivity flag
  */
  Tools_DynamicUTF8String getValue
    ( const Tools_DynamicUTF8String & sName ,
            bool                      bCasesensitiv = false );

  /*!
    @brief  read a value by number
    @param  nValue    [IN] - number 
  */
  Tools_DynamicUTF8String getValue
    ( SAPDB_Int nValue );

  /*!
    @brief  check existence of a value without name 
    @param  sValue        [IN] - name of value
    @param  bCasesensitiv [IN] - case sensitivity flag
  */
  bool existsValue
    ( const Tools_DynamicUTF8String & sValue,
            bool                      bCasesensitiv = false );

 /*!
    @brief  read aoption by number
    @param  nOptions   [IN] - number 
  */
  Tools_DynamicUTF8String getOption
    ( int nOption );

 /*!
    @brief  check for existence of a name
    @param  sName         [IN] - name of option
    @param  bCasesensitiv [IN] - case sensitivity flag
  */
  bool getOption
    ( const Tools_DynamicUTF8String & sName,
            bool                      bCasesensitiv = false );

 /*!
    @brief  check for existence of a name and read one following
            value.
    @param  sName         [IN]  - name of option
    @param  sArg1         [OUT] - Argument
    @param  bCasesensitiv [IN]  - case sensitivity flag
  */
  bool getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            bool                      bCasesensitiv = false );

 /*!
    @brief  check for existence of a name and read two following
            values.
    @param  sName         [IN]  - name of option
    @param  sArg1         [OUT] - Argument
    @param  sArg2         [OUT] - Argument
    @param  bCasesensitiv [IN]  - case sensitivity flag
  */
  bool getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2,
            bool                      bCasesensitiv = false );

 /*!
    @brief  check for existence of a name and read three following
            values.
    @param  sName         [IN]  - name of option
    @param  sArg1         [OUT] - Argument
    @param  sArg2         [OUT] - Argument
    @param  sArg3         [OUT] - Argument
    @param  bCasesensitiv [IN]  - case sensitivity flag
  */
  bool getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2,
            Tools_DynamicUTF8String & sArg3,
            bool                      bCasesensitiv = false );

  /*!
    @brief  read the name by number
    @param  nValue    [IN] - number 
  */
  Tools_DynamicUTF8String getName
    ( SAPDB_Int nValue );

  /*!
    @brief  read the number by name
    @param  sName         [IN] - name of value
    @param  bCasesensitiv [IN] - case sensitivity flag
  */
  SAPDB_Int getNumber
    ( const Tools_DynamicUTF8String & sName,
            bool                      bCasesensitiv = false);

  /*!
    @brief  read count of mane/value pairs
  */
  SAPDB_Int getCount
    ( );

private:
 /*!
    @brief  read a option with three arguments and argument count
    @param  sName         [IN]  - name of option
    @param  nArgs         [IN]  - Argument count
    @param  sArg1         [OUT] - Argument
    @param  sArg2         [OUT] - Argument
    @param  sArg3         [OUT] - Argument
    @param  bCasesensitiv [IN]  - case sensitivity flag
  */
  bool getOption
    ( const Tools_DynamicUTF8String & sName,
            SAPDB_Int                 nArgs,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2,
            Tools_DynamicUTF8String & sArg3,
            bool                      bCasesensitiv = false);

  Tools_Array<Tools_DynamicUTF8String> oNames;
  Tools_Array<Tools_DynamicUTF8String> oValues;
  SAPDB_Int                            nCount;
}; // end class Tools_Arguments


#endif /* _Tools_Arguments_HPP_ */
