/* ==================================================================
 *
 * responsible:   Bernd Vorsprach (D025588 - Bernd.Vorsprach@SAP-AG.de)
 *
 * created:       1998-02-06 by Bernd Vorsprach
 *
 * purpose:       - read and analyze old XParam help file
 *                - read and analyze new XParam help file
 *
 * ==================================================================
 *

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



 */

/* ------------------------------------------------------------------
 * IMPORT                                                                    
 * ------------------------------------------------------------------
 */
#include <ctype.h>
#include <string.h>
#include <stdio.h>

#include "heo02.h"
#include "heo06.h"

#include "hcn21.h"
#include "hcn29.h"
#include "hcn90.h"

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_ReadNewHelpFileContents
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_ReadNewHelpFileContents (
      tcn002_XParamData * pXParamData,
      char              * replyData,
      int               * replyLen,
      char              * szFile,
      tsp00_Int4            hHelpFile );

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_CheckLineSequence
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static bool    cn21_CheckLineSequence (
      tcn002_LineType currLine,
      tcn002_LineType lastLine);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_GetSecondToken
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_GetSecondToken (
      char             * szBuffer,
      char            ** pValue,
      tcn002_XPError   * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_AddParameter
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_AddParameter (
      tcn002_XParamData * pXParamData,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_ReadType
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_ReadType (
      tcn002_XParameter * pParameter,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_ReadDefault
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_ReadDefault (
      tcn002_XParameter * pParameter,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_ReadDefaultFor
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_ReadDefaultFor (
      tcn002_XParameter * pParameter,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_ReadProperty
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_ReadProperty (
      tcn002_XParameter * pParameter,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_InitProperties
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static void cn21_InitProperties (
      tcn002_XParameter * pParameter);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_AddLine
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_AddLine (
      tcn002_XParamData * pXParamData,
      char              * szBuffer,
      tcn002_LineType     xpLine,
      long                nNumber,
      tcn002_XPLine    ** pOut);

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_AddFollowLine
 * ------------------------------------------------------------------
 * purpose: reading new helpfile
 */
static tcn00_Error cn21_AddFollowLine (
      tcn002_XPLine     * pLine,
      char              * szBuffer);

/* ------------------------------------------------------------------
 * START OF IMPLEMENTATION
 * ------------------------------------------------------------------
 */

/* ------------------------------------------------------------------
 * SPECIFICATION PRIVATE FUNCTION cn21_SearchParameter
 * ------------------------------------------------------------------
 * purpose: search a parameter
 */
tcn002_XParameter * cn21_SearchParameter
      ( tcn002_XParameter * pParameter,
        const char        * szName );

/* ------------------------------------------------------------------
 * IMPLEMENATION PUBLIC FUNCTION cn21ReadHelpFile
 * ------------------------------------------------------------------
 */
tcn00_Error cn21ReadHelpFile (
      tcn002_XParamData * pXParamData,
      char              * replyData,
      int               * replyLen,
      char              * szFile,
      bool                bNewSyntax )
{
  tcn00_Error        nFuncStatus    = OK_CN00;
  tsp05_RteFileError rteFileErr;
  bool               bHelpFileOpen  = false;
  tsp00_Int4           hHelpFile;

  /* open XParam help file */
  if (nFuncStatus == OK_CN00) {
    sqlfopenc (szFile, sp5vf_text, sp5vf_read, sp5bk_buffered, &hHelpFile, &rteFileErr);
    if (rteFileErr.sp5fe_result != vf_ok) {
      nFuncStatus = ERR_RTE_CN00;
      cn90AnswerRTEError (replyData, replyLen, nFuncStatus, rteFileErr.sp5fe_text, rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_result );
    } else {
      bHelpFileOpen = true;
    } /* end if */
  } /* end if */

  /* File abarbeiten */
  if (nFuncStatus == OK_CN00) {
    if (bNewSyntax) {
      nFuncStatus = cn21_ReadNewHelpFileContents(pXParamData, replyData, replyLen, szFile, hHelpFile);
    } /* end if */
  } /* end if */

  /* close XParam help file (close always if open, independent from nFuncStatus) */
  if (bHelpFileOpen) {
    sqlfclosec ( hHelpFile, sp5vf_close_normal, &rteFileErr );
    /* set errortext only, if there no preceding error */
    if  ( (rteFileErr.sp5fe_result != vf_ok) && (nFuncStatus == OK_CN00) ) {
      nFuncStatus = ERR_RTE_CN00;
      cn90AnswerRTEError (replyData, replyLen, nFuncStatus, rteFileErr.sp5fe_text, rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_result );
    } /* end if */
  } /* end if */

  return nFuncStatus;

} /* cn21ReadHelpFile */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_ReadNewHelpFileContents
 * ------------------------------------------------------------------
 * purpose: read and analyze the contents of new XParam help file
 *
 */
static tcn00_Error cn21_ReadNewHelpFileContents (
      tcn002_XParamData * pXParamData,
      char              * replyData,
      int               * replyLen,
      char              * szFile,
      tsp00_Int4            hHelpFile )
{
  tcn00_Error           nFuncStatus  = OK_CN00;
  tcn002_XPError        nSyntaxError = XPOK_CN002;
  tsp05_RteFileError    rteFileErr;
  tsp00_Longint           nRead;
  char                  szBuffer[XP_LINE_LENGTH_MXCN002 + 1];
  char                  szTemp[XP_LINE_LENGTH_MXCN002 + 1];
  tcn002_XPLine       * pLine    = NULL;
  tcn002_LineType       currLine = XPLineUnknown;
  tcn002_LineType       lastLine = XPLineBOF;      /* start at Begin Of File */
  bool                  bFollow  = false;
  bool                  bString  = false;
  bool                  bIfOpen  = false;
  long                  nLine    = 0;
  struct {
    tcn002_LineType xpLineType;
    const char     * szKeyword;
  }                    LineToKeyTable[] = XP_KEYWORD_LINE_CN002;
  long                 i                = 0;


  /* looping around the help file */
  do {
    sqlfreadc (hHelpFile, szBuffer, XP_LINE_LENGTH_MXCN002, &nRead, &rteFileErr);

    currLine = XPLineUnknown;

    if  ( rteFileErr.sp5fe_result == vf_ok) {

      /* count of lines */
      nLine++;
      /* get a copy for analysis */
      SAPDB_strcpy(szTemp, szBuffer);
      /* remove tabs */
      char * pTab = &szTemp[0];
      while (*pTab != '\0') {
        *pTab = (*pTab == '\t') ? ' ' : *pTab;
        ++pTab;
      } // end while
      /* strip leading and trailing whitespaces */
      cn90Strip(szTemp);

      /* if in EXPLAIN or HELP check only for ENDEXPLAIN and ENDHELP */
      if ( ( (lastLine == XPLineEXPLAIN) || (lastLine == XPLineEXPLAINFollow) ) &&
             !cn90CheckKeyword(szTemp, XP_KEY_ENDEXPLAIN_CN002)                 ) {

        lastLine = XPLineEXPLAINFollow;
        nFuncStatus = cn21_AddLine(pXParamData,  szBuffer, XPLineEXPLAINFollow, nLine, &pLine);

      } else if ( ( (lastLine == XPLineHELP) || (lastLine == XPLineHELPFollow) ) &&
                     !cn90CheckKeyword(szTemp, XP_KEY_ENDHELP_CN002)         ) {

        lastLine = XPLineHELPFollow;
        nFuncStatus = cn21_AddLine(pXParamData,  szBuffer, XPLineHELPFollow, nLine, &pLine);

      } else {

        /* strip comment */
        i = 0;
        bString = false;
        while (szTemp[i] != CHAR_STRINGTERM_CN90) {
          if (szTemp[i] == XP_CHAR_STRING_CN002) {
            bString = !bString;
          } /* end if */
          if ((szTemp[i] == XP_CHAR_COMMENT_CN002) && !bString) {
            szTemp[i] = CHAR_STRINGTERM_CN90;
          } /* end if */
          i++;
        } /* end while */

        /* not empty? */
        if (strlen(szTemp) > 0) {

          /* its a follow line? */
          if (!bFollow) {

            /* detect line type by looping through the Keyword-To-Linetype-Table */
            i = 0;
            while ( (LineToKeyTable[i].xpLineType != XPLineUnknown) &&
                    !cn90CheckKeyword(szTemp, LineToKeyTable[i].szKeyword) ) {
              i++;
            } /* end while */
            currLine = LineToKeyTable[i].xpLineType;

            /* now we have a valid type of line or an syntax error */
            if (currLine == XPLineUnknown) {
              nFuncStatus  =  ERR_XPSYNTAX_CN00;
              nSyntaxError =  XPERR_NOTKEY_CN002;
            } else if (!cn21_CheckLineSequence(currLine, lastLine)) {
              nFuncStatus  =  ERR_XPSYNTAX_CN00;
              nSyntaxError =  XPERR_LINESEQ_CN002;
            } else {

              /* look for follow sign */
              if (szTemp[strlen(szTemp) - 1] == XP_CHAR_NEWFOLLOW_CN002) {
                szTemp[strlen(szTemp) - 1] = CHAR_STRINGTERM_CN90;

                /* its only valid for lines with expressions, otherwise error */
                switch (currLine) {
                  case XPLineCodeIF:
                  case XPLineCodeELSIF:
                  case XPLineCodeCALC:
                  case XPLineCodeCONSTRAINT:
                    bFollow = true;
                    break;
                  default:
                    nFuncStatus  =  ERR_XPSYNTAX_CN00;
                    nSyntaxError =  XPERR_INVFOLLOW_CN002;
                    break;
                } /* end switch */
              } else {
                bFollow = false;
              } /* end if */

              /* checking IF-ELSIF-ELSE */
              if (nFuncStatus == OK_CN00) {
                switch (currLine) {
                  case XPLineCodeIF:
                    bIfOpen = true;
                    break;
                  case XPLineCodeELSIF:
                    nFuncStatus  = bIfOpen ? nFuncStatus : ERR_XPSYNTAX_CN00;
                    nSyntaxError = bIfOpen ? nSyntaxError : XPERR_IFSEQ_CN002;
                    bIfOpen = true;
                    break;
                  case XPLineCodeELSE:
                    nFuncStatus  = bIfOpen ? nFuncStatus : ERR_XPSYNTAX_CN00;
                    nSyntaxError = bIfOpen ? nSyntaxError : XPERR_IFSEQ_CN002;
                    bIfOpen = false;
                    break;
                  case XPLineCodeCALC:
                  case XPLineCodeCONSTRAINT:
                    if ((lastLine == XPLineCodeCALC) || (lastLine == XPLineCodeCONSTRAINT)) {
                      bIfOpen = false;
                    } /* end if */
                    break;
                } /* end switch */
              } /* end if */

              if (nFuncStatus == OK_CN00) {
                /* do something with the line */
                switch (currLine) {
                  case XPLineID:
                    /* read identifier */
                    nFuncStatus = cn21_AddParameter(pXParamData, szTemp, &nSyntaxError);
                    break;
                  case XPLineTYPE:
                    /* read type */
                    nFuncStatus = cn21_ReadType(pXParamData->pLast, szTemp, &nSyntaxError);
                    break;
                  case XPLineDEFAULT:
                    /* read default */
                    nFuncStatus = cn21_ReadDefault(pXParamData->pLast, szTemp, &nSyntaxError);
                    break;
                  case XPLineDEFAULTFOR:
                    /* read default */
                    nFuncStatus = cn21_ReadDefaultFor(pXParamData->pLast, szTemp, &nSyntaxError);
                    break;
                  case XPLineProperty:
                    /* read property */
                    nFuncStatus = cn21_ReadProperty(pXParamData->pLast, szTemp, &nSyntaxError);
                  case XPLineCODE:
                  case XPLineENDCODE:
                  case XPLineEXPLAIN:
                  case XPLineENDEXPLAIN:
                  case XPLineHELP:
                  case XPLineENDHELP:
                    /* do nothing */
                    break;
                  case XPLineCodeIF:
                  case XPLineCodeELSIF:
                  case XPLineCodeELSE:
                  case XPLineCodeCALC:
                    /* add line */
                    nFuncStatus = cn21_AddLine(pXParamData,  szTemp, currLine, nLine, &pLine);
                    break;
                  case XPLineCodeCONSTRAINT:
                    if (pXParamData->pLast->pConstraint != NULL) {
                      nFuncStatus  =  ERR_XPSYNTAX_CN00;
                      nSyntaxError =  XPERR_DUPCONSTR_CN002;
                    } else {
                      /* add line */
                      nFuncStatus = cn21_AddLine(pXParamData,  szTemp, currLine, nLine, &pLine);
                    } // end if
                    break;
                  default:
                    nFuncStatus  =  ERR_XPSYNTAX_CN00;
                    nSyntaxError =  XPERR_LINEDET_CN002;
                    break;
                } /* end switch */
              } /* end if */
            } /* end if */

            lastLine = currLine;
          } else {
            /* look also for follow sign */
            if (szTemp[strlen(szTemp) - 1] == XP_CHAR_NEWFOLLOW_CN002) {
              szTemp[strlen(szTemp) - 1] = CHAR_STRINGTERM_CN90;
              bFollow = true;
            } else {
              bFollow = false;
            } /* end if */
            /* follow line - add to the previuos line */
            nFuncStatus = cn21_AddFollowLine(pLine, szTemp);
          } /* end if (bFollow) */
        } /* end if (empty line) */
      } /* end if (in HELP or EXPLAIN) */
    } /* end if (RTE error) */

  } while ( ( rteFileErr.sp5fe_result  == vf_ok             ) &&
            ( rteFileErr.sp5fe_warning == sp5vfw_no_warning ) &&
            ( nFuncStatus              == OK_CN00           )     );

  /* provide information about RTE error */
  if ( ( (rteFileErr.sp5fe_result != vf_ok)     &&
         (rteFileErr.sp5fe_result != vf_eof)        ) ||
     (rteFileErr.sp5fe_warning != sp5vfw_no_warning )    ) {
    nFuncStatus = ERR_RTE_CN00;
    cn90AnswerRTEError (replyData, replyLen, nFuncStatus, rteFileErr.sp5fe_text, rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_result );
  } /* end if */

  /* is EOF ok? */
  if (nFuncStatus == OK_CN00) {
    if (!cn21_CheckLineSequence(XPLineEOF, lastLine)) {
      nFuncStatus  =  ERR_XPSYNTAX_CN00;
      nSyntaxError =  XPERR_EOF_CN002;
    } /* end if */
  } /* end if */

  if ((nFuncStatus != OK_CN00) && (nFuncStatus != ERR_RTE_CN00)) {
    /* internal error */
    cn90AnswerIError(replyData, replyLen, nFuncStatus);

    if (nFuncStatus == ERR_XPSYNTAX_CN00) {
      /* add information about syntax error*/
      cn29SyntaxError(replyData, replyLen, nSyntaxError, szBuffer, szFile, nLine);
    } /* end if */
  } /* end if */

  return nFuncStatus;

} /* end cn21_ReadNewHelpFileContents */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_CheckLineSequence
 * ------------------------------------------------------------------
 * purpose: checks the sequence of line in help file using the
 *          transition table
 *
 */
static bool    cn21_CheckLineSequence (
      tcn002_LineType currLine,
      tcn002_LineType lastLine)
{
  bool    bValid = false;
  long    i      = 0;
  struct {
    tcn002_LineType  xpLast;
    tcn002_LineType  xpCurr;
  }       LineTable[] = XP_LINE_TRANSITIONS_CN002;

  while (!bValid && (LineTable[i].xpLast != XPLineUnknown)) {
    bValid = (LineTable[i].xpLast == lastLine) && (LineTable[i].xpCurr == currLine);
    i++;
  } /* end while */

  return bValid;
} /* end cn21_CheckLineSequence */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_GetSecondToken
 * ------------------------------------------------------------------
 * purpose: looks for the second (and last!) token
 */
static tcn00_Error cn21_GetSecondToken (
      char             * szBuffer,
      char            ** pValue,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error        nFuncStatus = OK_CN00;
  static char        szToken[XP_STRING_LENGTH_MXCN002 + 1];

  if (cn90GetToken(szBuffer, NULL, 3, -1)) {
    nFuncStatus = ERR_XPSYNTAX_CN00;
    *pSyntaxError = XPERR_TOKEN_CN002;
  } else if (cn90GetToken(szBuffer, szToken, 2, XP_STRING_LENGTH_MXCN002 + 1)) {
    if (strlen(szToken) == 0) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVVALUE_CN002;
    } /* end if */
  } else {
    nFuncStatus = ERR_XPSYNTAX_CN00;
    *pSyntaxError = XPERR_INVVALUE_CN002;
  } /* end if */

  if (nFuncStatus == OK_CN00) {
    *pValue = szToken;
  } /* end if */

  return nFuncStatus;
} /* end cn21_GetSecondToken */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_AddParameter
 * ------------------------------------------------------------------
 * purpose: create a new Parameter struct an read indentifier
 */
static tcn00_Error cn21_AddParameter (
      tcn002_XParamData * pXParamData,
      char             * szBuffer,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error        nFuncStatus = OK_CN00;
  char             * pName       = NULL;
  tcn002_XParameter * pParameter  = NULL;
  bool               bMallocOK   = true;

  // read the name
  nFuncStatus = cn21_GetSecondToken(szBuffer, &pName, pSyntaxError);

  // check for duplicates
  if (nFuncStatus == OK_CN00) {
    if (cn21_SearchParameter(pXParamData->pFirst, pName) != NULL) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_DUPLICATE_CN002;
    } // end if
  } /* end if */
  
  // list handling
  if (nFuncStatus == OK_CN00) {
    cn00MyMalloc (sizeof (tcn002_XParameter), &pParameter, &bMallocOK);
    if ((pParameter == NULL) || (!bMallocOK)) {
      nFuncStatus = ERR_MEM_CN00;
    } else {
      /* add the parameter in the list of parameters in the XParam section */
      if (pXParamData->pFirst == NULL) {
        pXParamData->pFirst = pParameter;
      } /* end if */
      if (pXParamData->pLast != NULL) {
        pXParamData->pLast->pNext = pParameter;
      } /* end if */
      pParameter->pNext = NULL;
      pParameter->pPrev = pXParamData->pLast;
      pXParamData->pLast = pParameter;
    } /* end if */
  } /* end if */

  // save data to item
  if (nFuncStatus == OK_CN00) {
    /* check length of identifier */
    if ((int) strlen(pName) > RTECONF_MAXNAMELENGTH ) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_LONGID_CN002;
    } else {
       if (isalpha(pName[0]) || (pName[0] == XP_CHAR_UNDERLINE_CN002) ) {
         /* copy identifier */
         cn90SaveStrcpy(pParameter->szParameterName, pName, sizeof(tsp00_XpKeyTypec));
        /* init properties */
         cn21_InitProperties (pParameter);
         /* init flags */
         pParameter->bHelpExist  = true;
      } else {
        nFuncStatus = ERR_XPSYNTAX_CN00;
        *pSyntaxError = XPERR_INVID_CN002;
      } /* end if */
    } /* end if */
  } /* end if */

  return nFuncStatus;
} /* end cn21_AddParameter */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_ReadType
 * ------------------------------------------------------------------
 * purpose: analyze the type line
 */
static tcn00_Error cn21_ReadType (
      tcn002_XParameter * pParameter,
      char             * szBuffer,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error        nFuncStatus = OK_CN00;
  char             * pType       = NULL;
  struct {
    RTEConf_Parameter::Type   nValue;
    const char              * szText;
  }                 aTable[]    = XP_TYPES_CN002;
  long              i            = 0;

  nFuncStatus = cn21_GetSecondToken(szBuffer, &pType, pSyntaxError);

  if (nFuncStatus == OK_CN00) {
    /* run through the type table */
    while ( (strlen(aTable[i].szText) > 0) && (strcmp(pType, aTable[i].szText) != 0) ) {
      i++;
    } /* end while */

    if (strlen(aTable[i].szText) == 0) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVTYPE_CN002;
    } /* end if */

    pParameter->xpRecord.xpType = aTable[i].nValue;
  } /* end if */

  return nFuncStatus;
} /* end cn21_ReadType */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_ReadDefault
 * ------------------------------------------------------------------
 * purpose: analyze the Default line
 */
static tcn00_Error cn21_ReadDefault (
      tcn002_XParameter * pParameter,
      char             * szBuffer,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error        nFuncStatus  = OK_CN00;
  char             * pDefault     = NULL;
  long               nValue       = 0;
  bool               bValid       = false;

  nFuncStatus = cn21_GetSecondToken(szBuffer, &pDefault, pSyntaxError); 

  if (nFuncStatus == OK_CN00) {
    switch(pParameter->xpRecord.xpType) {
      case RTEConf_Parameter::t_Integer:
        bValid = cn90AnalyzeNumber(pDefault, &nValue);
        if (bValid) {
          cn90SaveStrcpy(pParameter->szDefValue, pDefault, XP_STRING_LENGTH_MXCN002 + 1);
          cn90SaveStrcpy(pParameter->szNewValue, pDefault, XP_STRING_LENGTH_MXCN002 + 1);
        } /* end if */
        break ;
      case RTEConf_Parameter::t_Real:
        bValid = false;
        break ;
      case RTEConf_Parameter::t_CryptInfo:
        bValid = false;
        break;
      case RTEConf_Parameter::t_String:
        bValid = cn90AnalyzeString(pDefault,   pParameter->szDefValue, XP_STRING_LENGTH_MXCN002);
        cn90SaveStrcpy(pParameter->szNewValue, pParameter->szDefValue, XP_STRING_LENGTH_MXCN002 + 1);
        break ;
    } /* end switch */

    if (!bValid) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVDEF_CN002;
    } /* end if */
  } /* end if */

  return nFuncStatus;
} /* end cn21_ReadDefault */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_ReadDefaultFor
 * ------------------------------------------------------------------
 * purpose: analyze the Default line
 */
static tcn00_Error cn21_ReadDefaultFor (
      tcn002_XParameter * pParameter,
      char              * szBuffer,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error        nFuncStatus  = OK_CN00;
  _TCHAR             szInstance[XP_STRING_LENGTH_MXCN002 + 1];
  _TCHAR             szValue[XP_STRING_LENGTH_MXCN002 + 1];
  int                nIndex = 0;

  long               nValue       = 0;
  bool               bValid       = false;

  // read token
  if (!cn90GetToken(szBuffer, szInstance, 2, XP_STRING_LENGTH_MXCN002 + 1) || !cn90GetToken(szBuffer, szValue, 3, XP_STRING_LENGTH_MXCN002 + 1)) {
    nFuncStatus  = ERR_XPSYNTAX_CN00;
    *pSyntaxError = XPERR_INVDEF_CN002;
  } // end if

  // check free DefaultFor-Entry
  if (nFuncStatus == OK_CN00) {
    for (nIndex = 0; nIndex < XP_DEFAULTFOR_MXCN002; ++nIndex) {
      if (_tcslen(pParameter->szDefaultFor[nIndex][XP_DEFAULTFORINSTANCE_CN002]) == 0) {
        break;
      } // end if
    } // end for
    if (nIndex == XP_DEFAULTFOR_MXCN002) {
      nFuncStatus  = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVDEF_CN002;
    } // end if
  } // end if

  // copy Value
  if (nFuncStatus == OK_CN00) {
    switch(pParameter->xpRecord.xpType) {
      case RTEConf_Parameter::t_Integer:
        bValid = cn90AnalyzeNumber(szValue, &nValue);
        if (bValid) {
          cn90SaveStrcpy(pParameter->szDefaultFor[nIndex][XP_DEFAULTFORVALUE_CN002], szValue, XP_STRING_LENGTH_MXCN002 + 1);
        } /* end if */
        break ;
      case RTEConf_Parameter::t_Real:
        bValid = false;
        break ;
      case RTEConf_Parameter::t_CryptInfo:
        bValid = false;
        break;
      case RTEConf_Parameter::t_String:
        bValid = cn90AnalyzeString(szValue, pParameter->szDefaultFor[nIndex][XP_DEFAULTFORVALUE_CN002], XP_STRING_LENGTH_MXCN002);
        break ;
    } /* end switch */

    if (!bValid) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVDEF_CN002;
    } else {
      bValid = cn90AnalyzeString(szInstance, pParameter->szDefaultFor[nIndex][XP_DEFAULTFORINSTANCE_CN002], XP_STRING_LENGTH_MXCN002);
      if (!bValid) {
        nFuncStatus = ERR_XPSYNTAX_CN00;
        *pSyntaxError = XPERR_INVDEF_CN002;
      } /* end if */
    } /* end if */
  } /* end if */

  return nFuncStatus;
} /* end cn21_ReadDefaultFor */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_ReadProperty
 * ------------------------------------------------------------------
 * purpose: analyze the property line
 */
static tcn00_Error cn21_ReadProperty (
      tcn002_XParameter * pParameter,
      char             * szBuffer,
      tcn002_XPError    * pSyntaxError)
{
  tcn00_Error           nFuncStatus      = OK_CN00;
  char                * pProperty     = NULL;
  char                * pValue        = NULL;
  char                * pToken        = NULL;
  tcn002_XPPropValue     nValue        = XPPropValueUnknown;
  tcn002_XPPropClass     nClass        = XPPropClassUnknown;
  struct {
    tcn002_XPPropValue   nValue;
    const char        * szText;
  }                     aValueTable[] = XP_PROP_VALUES_CN002;
  struct tcn002_PropClassTable {
    tcn002_XPPropClass   nClass;
    tcn002_XPPropValue   nDefault;
    const char        * szText;
  }                     aClassTable[] = XP_PROP_CLASSES_CN002;
  struct {
    tcn002_XPPropClass   nClass;
    tcn002_XPPropValue   nValue;
  }                     aValidTable[] = XP_PROP_VALID_CN002;
  long                  i             = 0;
  long                  j             = 0;
  long                  k             = 0;

  /* knock down to tokens */
  pProperty = strtok(szBuffer, XP_KEY_TOKENSEP_CN002);
  /* look for value */
  pValue = strtok(NULL, "\n");
  cn90Strip(pValue);
  /* look for unexpected token */
//  pToken = strtok(NULL, XP_KEY_TOKENSEP_CN002);

  if (pToken != NULL) {
    nFuncStatus = ERR_XPSYNTAX_CN00;
    *pSyntaxError = XPERR_TOKEN_CN002;
  } else {
    /* check existence of value */
    if (pValue == NULL) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVVALUE_CN002;
    } else if (strlen(pValue) == 0) {
      nFuncStatus = ERR_XPSYNTAX_CN00;
      *pSyntaxError = XPERR_INVVALUE_CN002;
    } else {

      /* search class */
      while ( (aClassTable[i].nClass != XPPropClassUnknown) && (strcmp(aClassTable[i].szText, pProperty) != 0) ) {
        i++;
      } /* end while */
      /* search value */
      while ( (aValueTable[j].nValue != XPPropValueUnknown) && (strcmp(aValueTable[j].szText, pValue) != 0) ) {
        j++;
      } /* end while */

      if (aClassTable[i].nClass == XPPropClassUnknown) {
        nFuncStatus = ERR_XPSYNTAX_CN00;
        *pSyntaxError = XPERR_INVPROP_CN002;
      } else {
        /* is combination valid */
        while (  ( aValidTable[k].nClass != XPPropClassUnknown           ) &&
                !( ( aValidTable[k].nClass == aClassTable[i].nClass ) &&
                   ( aValidTable[k].nValue == aValueTable[j].nValue )    )    ) {
          k++;
        } /* end while */

        if (aValidTable[k].nClass != XPPropClassUnknown) {
          pParameter->xpProperties[aClassTable[i].nClass] = aValueTable[j].nValue;
          SAPDB_memmove(
              pParameter->szProperties[aClassTable[i].nClass],
              pValue,
              strlen(pValue)+1);
        } else {
          nFuncStatus = ERR_XPSYNTAX_CN00;
          *pSyntaxError = XPERR_INVPROP_CN002;
        } /* end if */
      } /* end if */
    } /* end if */
  } /* end if */

  return nFuncStatus;
} /* end cn21_ReadProperty */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_InitProperties
 * ------------------------------------------------------------------
 * purpose: init properties of parameter
 */
static void cn21_InitProperties (
      tcn002_XParameter * pParameter)
{
  tcn00_Error           nFuncStatus      = OK_CN00;
  struct tcn002_PropClassTable {
    tcn002_XPPropClass   nClass;
    tcn002_XPPropValue   nDefault;
    const char        * szText;
  }                     aClassTable[] = XP_PROP_CLASSES_CN002;
  long                  i             = 0;

  /* search class */
  for (i = 0; i < XPPropClassMAX; i++) {
    pParameter->xpProperties[i] = aClassTable[i].nDefault;
    SAPDB_strcpy(pParameter->szProperties[i], "");
  } /* end for */

} /* end cn21_InitProperties */

/* IMPLEMENATION PRIVATE FUNCTION cn21_AddLine
 * ------------------------------------------------------------------
 * purpose: add a line of specified type
 */
static tcn00_Error cn21_AddLine (
      tcn002_XParamData * pXParamData,
      char              * szBuffer,
      tcn002_LineType     xpLine,
      long                nNumber,
      tcn002_XPLine    ** pOut)
{
  tcn00_Error        nFuncStatus = OK_CN00;
  tcn002_XParameter * pParameter  = pXParamData->pLast;
  tcn002_XPLine     * pLine       = NULL;
  char             * pLineText   = NULL;
  bool               bMallocOK   = true;
  tcn002_XPLine     * pCurrent    = NULL;

  cn00MyMalloc (sizeof (tcn002_XPLine), &pLine, &bMallocOK);
  if ((pLine == NULL) || (!bMallocOK)) {
    nFuncStatus = ERR_MEM_CN00;
  } else {
    cn00MyMalloc (strlen(szBuffer) + 1, &pLineText, &bMallocOK);
    if ((pLineText == NULL) || (!bMallocOK)) {
      nFuncStatus = ERR_MEM_CN00;
      cn00MyFree(pLine);
    } else {

      pLine->szLine  = pLineText;
      pLine->xpLine  = xpLine;
      pLine->nNumber = nNumber;
      SAPDB_strcpy(pLine->szLine, szBuffer);

      *pOut = pLine;

      /* use the right list */
      switch (pLine->xpLine) {
        case XPLineCodeCALC:
        case XPLineCodeIF:
        case XPLineCodeELSIF:
        case XPLineCodeELSE:
          pCurrent = pXParamData->pLast->pCode;
          if (pCurrent == NULL) {
            pXParamData->pLast->pCode = pLine;
          } /* end if */
          break;
        case XPLineCodeCONSTRAINT:
          pCurrent = pXParamData->pLast->pConstraint;
          if (pCurrent == NULL) {
            pXParamData->pLast->pConstraint = pLine;
          } /* end if */
          break;
        case XPLineEXPLAINFollow:
          pCurrent = pXParamData->pLast->pExplain;
          if (pCurrent == NULL) {
            pXParamData->pLast->pExplain = pLine;
          } /* end if */
          break;
        case XPLineHELPFollow:
          pCurrent = pXParamData->pLast->pHelp;
          if (pCurrent == NULL) {
            pXParamData->pLast->pHelp = pLine;
          } /* end if */
          break;
        default:
          /* line detection error */
          nFuncStatus = ERR_XPSYNTAX_CN00;
          break;
      } /* end switch */

      if (pCurrent != NULL) {
        while (pCurrent->pNext != NULL) {
          pCurrent = pCurrent->pNext;
        } /* end while */
        pCurrent->pNext = pLine;
      } /* end if */
    } /* end if */
  } /* end if */

  return nFuncStatus;
} /* end cn21_AddLine */

/* ------------------------------------------------------------------
 * IMPLEMENATION PRIVATE FUNCTION cn21_AddFollowLine
 * ------------------------------------------------------------------
 * purpose: add a line of specified type
 */
static tcn00_Error cn21_AddFollowLine (
      tcn002_XPLine     * pLine,
      char             * szBuffer)
{
  tcn00_Error        nFuncStatus = OK_CN00;
  char             * pLineText   = NULL;
  bool               bMallocOK   = true;

  if (pLine != NULL) {
    cn00MyMalloc (strlen(szBuffer) + strlen(pLine->szLine) + 2, &pLineText, &bMallocOK);
    if ((pLineText == NULL) || (!bMallocOK)) {
      nFuncStatus = ERR_MEM_CN00;
    } else {

      sprintf(pLineText, "%s %s", pLine->szLine, szBuffer);

      cn00MyFree(pLine->szLine);

      pLine->szLine = pLineText;
    } /* end if */
  } else {
    nFuncStatus = ERR_MEM_CN00;
  } /* end if */

  return nFuncStatus;
} /* end cn21_AddFollowLine */

/* IMPLEMENTATION INTERN FUNCTION cn21_SearchParameter
 * ------------------------------------------------------------------
 */
tcn002_XParameter * cn21_SearchParameter
      ( tcn002_XParameter * pParameter,
        const char        * szName )
{
  bool                 bFound = false;

  /* search for dynamic match */
  bFound = false;
  while ((pParameter != NULL) && (bFound == false)) {
    if (strcmp(pParameter->szParameterName, szName) == 0) {
      bFound = true;
    } else {
      pParameter = pParameter->pNext;
    } /* end if */
  } /* end while */

  return pParameter;
} /* end cn21_SearchParameter */
