/*!
  @file           veo03.c
  @author         JoergM
  @special area   communication
  @brief          RTE independent functions (USER communication)
  @see            example.html ...

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/




/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "gsp01.h"
#include "geo001.h"
#include "geo007_1.h"
#include "geo007_2.h"
#include "heo46.h"
#include "geo57.h"
#include "geo60.h"
#include "heo03.h"
#include "geo03.h"
#include "geo43_0.h"
#include "heo420.h"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/



/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  GLOBAL VARIABLES                                                         *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL VARIABLES                                                          *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/

static tsp00_Uint     
eo03_GetServicePort ( char          *szSapRouterString, 
                      tsp00_Uint2   *pusServicePort,
                      tsp00_ErrTextc VAR_ARRAY_REF  pErrText) ;

static tsp00_Uint     
eo03_NiUpdateConnInfoAfterConnect ( teo003_ConnectParamRecord    *pConnParam,
                                    teo40_NiConnectInfoRecord    *pNIConnInfo,
                                    tsp00_ErrTextc VAR_ARRAY_REF  pErrText);

static tsp00_Uint     
eo03_InitNIConnInfo               ( teo003_ConnectParamRecord    *pConnParam,
                                    teo40_NiConnectInfoRecord    *pNIConnInfo,
                                    tsp00_ErrTextc VAR_ARRAY_REF  pErrText);

static tsp00_Uint     
eo03NiSqlConnect  ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                    tsp00_ErrTextc VAR_ARRAY_REF  pErrText );

static void 
eo03InitConnectParamRec ( teo40_NiConnectInfoRecord *pNIConnInfo,
                          teo003_ConPktParamRecord  *pConPktParamRec,
                          tsp00_Uint                 ulMessClass);

static tsp00_Uint     
eo03InfoRequest         ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                          tsp00_ErrTextc VAR_ARRAY_REF  ErrText );

/*===========================================================================*
 *  PUBLIC FUNCTIONS (CODE)                                                  *
 *===========================================================================*/

/* PTS 1104899 */

externC tsp00_Bool 
SqlIsSaprouterString ( char *szServerNode )	/* '\0' terminated node or saprouterstring */
{
  return ( (tsp00_Bool ) eo40NiIsSaprouterSring ( szServerNode ) );
}

/*------------------------------*/

/* PTS 1105295 */

externC void 
eo03NiBuildTracefileName    (tsp00_Pathc VAR_ARRAY_REF  TracefileName)
{
  tsp01_RteError  RteError;

  if ( sqlGetIndependentWrkPath(TracefileName, TERM_WITH_DELIMITER_EO01, &RteError ) )
    strcat( TracefileName, NI_TRACEFILE_EO40 ) ;
  else
    strcpy( TracefileName, NI_TRACEFILE_EO40 ) ; /* use current directory ( x_ping ) */

  return ;
}

/*------------------------------*/

externC tsp00_Uint     
eo03NiInit(tsp00_ErrTextc  VAR_ARRAY_REF  pErrText)
{

  tsp00_Uint      ulCommState = commErrOk_esp01;
  tsp00_Pathc     TracefileName ;

  eo03NiBuildTracefileName (TracefileName) ;

  if ( !eo40NiInit(TracefileName, NI_TRACE_ERRORS_ONLY_EO40, pErrText) )
    ulCommState = commErrNotOk_esp01 ;

 return ( ulCommState ) ;
}

/*------------------------------*/

externC tsp00_Uint     
eo03NiConnect ( teo003_ConnectParamRecord    *pConnParam,
                teo40_NiConnectInfoRecord    *pNIConnInfo,
                tsp00_ErrTextc VAR_ARRAY_REF  pErrText  )
{
  #undef  MF__
  #define MF__ MOD__"eo03NiConnect"

  tsp00_Uint      ulCommState;

  ulCommState = eo03_InitNIConnInfo( pConnParam, pNIConnInfo, pErrText ); 
  ulCommState = eo03NiSqlConnect( pNIConnInfo, pErrText );
  if ( ulCommState == commErrOk_esp01 )
    ulCommState = eo03_NiUpdateConnInfoAfterConnect( pConnParam, pNIConnInfo, pErrText ); 

  return ( ulCommState );
}

/*------------------------------*/

externC tsp00_Uint     
eo03NiRelease ( teo40_NiConnectInfoRecord *pNIConnInfo,
                tsp00_ErrTextc  VAR_ARRAY_REF  pErrText  )
  {

  tsp00_Uint      ulMessClass;
  tsp00_Uint      CommState = commErrOk_esp01;
  tsp00_Int       rc = NO_ERROR_EO001;

  switch ( pNIConnInfo->ulServiceType )
    {
    case srvControl_esp01 :
      ulMessClass = RSQL_USER_RELEASE_REQUEST_EO003;
      break;

    default:
      ulMessClass = RSQL_USER_RELEASE_REQUEST_EO003;
      break;
    }

  CommState = eo420SendReleasePacket ( pNIConnInfo->NiHandle, (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                       pNIConnInfo->ulClientRef, pNIConnInfo->ulServerRef, 
                                       ulMessClass, commErrOk_esp01 );

  eo40NiClose ( &pNIConnInfo->NiHandle );

  if ( pNIConnInfo->pCommPacketList[0] != NULL ) 
  {
    rc = FREE_MEM_EO57 ( pNIConnInfo->pCommPacketList[0] );

    if ( NO_ERROR_EO001 == rc  )
      pNIConnInfo->pCommPacketList[0] = NULL;
  }

  if ( pNIConnInfo->szSapRouterString != NULL )
  {
    if ( NO_ERROR_EO001 == rc )
      rc = FREE_MEM_EO57 ( pNIConnInfo->szSapRouterString ) ;
    else
      FREE_MEM_EO57 ( pNIConnInfo->szSapRouterString ) ; /* don't overwrite previous error */

    pNIConnInfo->szSapRouterString = NULL ;
  }

  if ( rc != NO_ERROR_EO001 )
  {
    eo46BuildErrorStringRC ( pErrText, (char *)ERRMSG_COM_CANT_FREE_MEM, rc );
    CommState = commErrNotOk_esp01 ;
  }

  return ( CommState );
  }

/*------------------------------*/

externC tsp00_Uint     
eo03NiRequest  ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                 tsp00_Uint                    ulCmdPacket,
                 tsp00_Uint                    ulCmdDataLen,
                 tsp00_ErrTextc  VAR_ARRAY_REF  pErrText )
  {

  tsp00_Uint                      ulCommState;
  teo003_CommPacketRecord         *pCommPacket = NULL;
  tsp00_Uint                      ulMessClass;

  if ( pNIConnInfo->ulServiceType == srvDistribution_esp01 )
    ulMessClass = RSQL_KERN_DATA_REQUEST_EO003;
  else
    ulMessClass = RSQL_USER_DATA_REQUEST_EO003;

  pCommPacket                  = pNIConnInfo->pCommPacketList[ulCmdPacket];
  pNIConnInfo->ulCurrentPacket = ulCmdPacket;
  pNIConnInfo->ulCmdDataLen    = ulCmdDataLen;


  ulCommState = eo420SendCommPacket ( pNIConnInfo->NiHandle, 
                                      (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                      pNIConnInfo->ulMaxSegmentSize,
                                      pCommPacket,
                                      ulCmdDataLen,
                                      pNIConnInfo->ulClientRef,
                                      pNIConnInfo->ulServerRef,
                                      ulMessClass,
                                      commErrOk_esp01,
                                      pErrText );

  return ( ulCommState );
  }

/*===========================================================================*/

externC tsp00_Uint     
eo03NiReceive  ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                    teo003_CommPacketRecord     **ppResPacket,
                    tsp00_Uint                   *pulResDataLen,
                    tsp00_ErrTextc  VAR_ARRAY_REF  pErrText )
{

  tsp00_Uint                              ulCommState;
  teo003_CommPacketRecord                *pCmdCommPacket;
  tsp00_Uint                              ulCurrentPacket;
  tsp00_Uint                              ulSenderRef;
  tsp00_Uint                              ulReceiverRef;
  tsp00_Uint                              ulMessClass;

  ulCurrentPacket = pNIConnInfo->ulCurrentPacket;
  pCmdCommPacket  = pNIConnInfo->pCommPacketList[ulCurrentPacket];
  *ppResPacket    = (teo003_CommPacketRecord *) ((char *)pCmdCommPacket->pDataPart +
                                        pNIConnInfo->ulCmdDataLen);

  ulCommState = eo420ReceiveCommPacket ( pNIConnInfo->NiHandle, 
                                         (RECV_PKT_FUNC_PTR_EO003)eo40NiReceive, 
                                         pNIConnInfo->ulSwapType,
                                        *ppResPacket,
                                         pNIConnInfo->ulMaxDataLen -
                                         pNIConnInfo->ulCmdDataLen,
                                         pulResDataLen,
                                        &ulSenderRef,
                                        &ulReceiverRef,
                                        &ulMessClass,
                                         pErrText );
 
  return ( ulCommState );
}
/*===========================================================================*/

/*===========================================================================*/

/* PTS 1105295 */
externC tsp00_Uint     
eo03NiXserverPing( char                         *szSapRouterString,
                   tsp00_Versionc VAR_ARRAY_REF  szServerVersion,
                   tsp00_ErrTextc VAR_ARRAY_REF  pErrText )
{

  teo40_NiConnectInfoRecord       TmpNIConnInfo;
  tsp00_Uint                      ulCommState = commErrOk_esp01 ;

  if ( szSapRouterString )
  {
    TmpNIConnInfo.szSapRouterString = szSapRouterString ;
    TmpNIConnInfo.usServicePort     = 0 ;

    ulCommState = eo03_GetServicePort( szSapRouterString, 
                                      &TmpNIConnInfo.usServicePort, 
                                       pErrText);

    if ( ulCommState == commErrOk_esp01 )
      ulCommState = eo40NiConnectToServer ( &TmpNIConnInfo, pErrText ) ;
 
    if ( ulCommState == commErrOk_esp01 )
    {
      teo003_ConPktParamRecord         ConPktParamRec;

      memset(&ConPktParamRec, 0, sizeof(teo003_ConPktParamRecord) ) ;

      /* send PING request */
      ConPktParamRec.ulMessClass         = RSQL_PING_REQUEST_EO003;

      ulCommState = eo420SendConnectPacket ( TmpNIConnInfo.NiHandle, (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                            &ConPktParamRec,  pErrText );

      if ( ulCommState == commErrOk_esp01 )
      {
        /* receive server reply */
        ulCommState = eo420ReceiveConnectPacket ( TmpNIConnInfo.NiHandle, 
                                                 (RECV_PKT_FUNC_PTR_EO003)eo40NiReceive,
                                                 &ConPktParamRec, pErrText );

        if ( ulCommState == commErrOk_esp01 )
        {  
          if ( ConPktParamRec.ulMessClass != RSQL_PING_REPLY_EO003 )
          {
            MSGD (( ERR_WRONG_MESSCLASS_REPL, ConPktParamRec.ulMessClass ));
            strcpy(pErrText, ERRMSG_PROTOCOL_ERROR);
            ulCommState = commErrNotOk_esp01 ;
          }
          else
          {
            memcpy (szServerVersion, &ConPktParamRec.szVersion[0], sizeof(tsp00_Versionc) -1 ) ;
            szServerVersion[sizeof(tsp00_Versionc) -1] = '\0' ;
          }
        }
      }
      eo40NiClose(&TmpNIConnInfo.NiHandle) ;
    }
  }
  return ( ulCommState );
}

/*===========================================================================*/

externC tsp00_Uint     
eo03NiSqlCancelDump       ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                            char                         *szSapRouterString,
                            tsp00_Uint                    ulReqType,
                            tsp00_ErrTextc VAR_ARRAY_REF  pErrText )
{

  teo40_NiConnectInfoRecord       TmpNIConnInfo;
  tsp00_Uint                      ulMessClass;
  tsp00_Uint                      ulCommState = commErrOk_esp01 ;

  TmpNIConnInfo = *pNIConnInfo;

   switch ( ulReqType )
     {
     case SQL_RTE_CANCEL_EO003:
       if (pNIConnInfo->ulServiceType == srvControl_esp01)
         ulMessClass = RSQL_CTRL_CANCEL_REQUEST_EO003;
       else
         ulMessClass = RSQL_USER_CANCEL_REQUEST_EO003;
       break;

     case SQL_RTE_DUMP_EO003:
       ulMessClass = RSQL_DUMP_REQUEST_EO003;
       break;

     default:
       MSGD (( ERR_PROTOCOL_ERROR, "REQUEST TYPE" )); 
       strcpy((char *)pErrText, ERRMSG_PROTOCOL_ERROR);
       ulCommState = commErrNotOk_esp01 ;
     }

  if ( ulCommState == commErrOk_esp01 )
  {
    teo003_ConPktParamRecord         ConPktParamRec;
    /* connect to server */
    TmpNIConnInfo.ulMaxSegmentSize  = sizeof(teo003_RteHeaderRecord) +
                                      sizeof(teo003_RteConnectPacketRec);
    TmpNIConnInfo.ulMinSegmentSize  = sizeof(teo003_RteHeaderRecord);

    /* PTS 1105116 */
    if ( szSapRouterString )
      TmpNIConnInfo.szSapRouterString = szSapRouterString ;

    ulCommState = eo40NiConnectToServer ( &TmpNIConnInfo, pErrText ) ;

    if ( ulCommState == commErrOk_esp01 )
    {
      /* send cancel request */
      ConPktParamRec.ulMessClass         = ulMessClass;
      ConPktParamRec.ulSenderRef         = TmpNIConnInfo.ulClientRef;
      ConPktParamRec.ulReceiverRef       = TmpNIConnInfo.ulServerRef;
      ConPktParamRec.ulServiceType       = TmpNIConnInfo.ulServiceType;
      ConPktParamRec.ulMaxSegmentSize    = TmpNIConnInfo.ulMaxSegmentSize;
      ConPktParamRec.ulPacketSize        = TmpNIConnInfo.ulPacketSize;
      ConPktParamRec.ulMaxDataLen        = TmpNIConnInfo.ulMaxDataLen;
      ConPktParamRec.ulMinReplySize      = TmpNIConnInfo.ulMinReplySize;
      ConPktParamRec.ulSwapType          = (tsp00_Uint    )undef_ceo00;
      ConPktParamRec.szSenderServerDB[0] = '\0';
      ConPktParamRec.pidSenderPID        = TmpNIConnInfo.pidClientPID;
      ConPktParamRec.usServicePort       = TmpNIConnInfo.usServicePort;
      ConPktParamRec.ulCommState         = commErrOk_esp01;
      ConPktParamRec.fAcknowledge        = false;

      ulCommState = eo420SendConnectPacket ( TmpNIConnInfo.NiHandle, (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                             &ConPktParamRec,  pErrText );                  

      eo40NiClose(&TmpNIConnInfo.NiHandle) ;
    }

  }
  return ( ulCommState );
}

/*===========================================================================*/

externC tsp00_Uint     
eo03NiSqlReplyAvailable   ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                            tsp00_ErrTextc  VAR_ARRAY_REF  pErrText )
{
  tsp00_Uint     ulCommState = eo40NiDataAvailable(pNIConnInfo->NiHandle, pErrText);

  return(ulCommState);
}

/*===========================================================================*
 *  LOCAL/PRIVATE/PROTECTED FUNCTIONS (CODE)                                 *
 *===========================================================================*/

static tsp00_Uint     
eo03NiSqlConnect  ( teo40_NiConnectInfoRecord    *pNIConnInfo,
                    tsp00_ErrTextc VAR_ARRAY_REF  pErrText )
{

  tsp00_Uint              CommState = commErrOk_esp01;

  pNIConnInfo->ulMaxSegmentSize = sizeof(teo003_RteHeaderRecord) +
                                  sizeof(teo003_RteConnectPacketRec);
  pNIConnInfo->ulMinSegmentSize = sizeof(teo003_RteHeaderRecord);

  /* connect to server */
  CommState = eo40NiConnectToServer ( pNIConnInfo, pErrText ) ;

  if ( CommState == commErrOk_esp01 )
  {
    if (pNIConnInfo->ulServiceType != srvControl_esp01)
    {
      /* info request to server, get packet sizes */
      CommState = eo03InfoRequest (pNIConnInfo, pErrText) ; 
    }
    else
    {
      /* packet sizes are fixed, no info request necessary */
      pNIConnInfo->ulPacketSize   = RSQL_CTRL_PACKET_SIZE_EO003 ;
      pNIConnInfo->ulMinReplySize = RSQL_CTRL_MIN_REPLY_SIZE_EO003;
      pNIConnInfo->ulMaxDataLen   = pNIConnInfo->ulPacketSize
                                  - (2 * RTE_HEADER_SIZE_EO003);
    }
  }

  if ( CommState == commErrOk_esp01 )
  {
    teo003_ConPktParamRecord   ConPktParamRec;

    pNIConnInfo->ulMaxSegmentSize = pNIConnInfo->ulPacketSize;
    pNIConnInfo->ulMinSegmentSize = 1024;

    /* try to set socket buffers to get maximum performance */
    CommState = eo420SetNiSocketBufferSize ( pNIConnInfo->NiHandle,
                                             pNIConnInfo->ulMinSegmentSize,
                                            &pNIConnInfo->ulMaxSegmentSize,
                                             pErrText ) ;

    if ( CommState == commErrOk_esp01 )
    {
      eo03InitConnectParamRec(pNIConnInfo, &ConPktParamRec,
                              eo420ServiceTypeToMessClass (pNIConnInfo->ulServiceType ));

      if ( ConPktParamRec.ulMessClass == RSQL_CTRL_CONN_REQUEST_EO003 )
      {
        if (pNIConnInfo->pszServerPgm != NULL) 
          strcpy ( ConPktParamRec.szServerPgm, pNIConnInfo->pszServerPgm);

        if (pNIConnInfo->pszServerDBRoot != NULL) 
          strcpy ( ConPktParamRec.szServerDBRoot, pNIConnInfo->pszServerDBRoot);
      }

      /* send connect packet to server */
      CommState = eo420SendConnectPacket ( pNIConnInfo->NiHandle, (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                          &ConPktParamRec,  pErrText );

      if ( CommState != commErrOk_esp01 )
        eo40NiClose(&pNIConnInfo->NiHandle) ;
      else
      {
        /* receive server reply */
        CommState = eo420ReceiveConnectPacket ( pNIConnInfo->NiHandle, 
                                                (RECV_PKT_FUNC_PTR_EO003)eo40NiReceive,
                                               &ConPktParamRec, pErrText );

        if ( CommState != commErrOk_esp01 )
          eo40NiClose(&pNIConnInfo->NiHandle) ;
        else
        if ( ConPktParamRec.ulMaxSegmentSize > pNIConnInfo->ulMaxSegmentSize )
        {
          strcpy(pErrText, ERRMSG_COM_ILL_MAXSEGMENTSIZE );
          CommState = commErrNotOk_esp01 ;
        }
        else
        {
          pNIConnInfo->ulServerRef = ConPktParamRec.ulSenderRef;
          pNIConnInfo->ulSwapType  = ConPktParamRec.ulSwapType;
        }
      }
    }
  }

  return ( CommState );
}
/*===========================================================================*/

static tsp00_Uint     
eo03InfoRequest( teo40_NiConnectInfoRecord    *pNIConnInfo,
                 tsp00_ErrTextc VAR_ARRAY_REF  pErrText )
{

  teo003_ConPktParamRecord ConPktParamRec;
  tsp00_Uint               CommState;

  eo03InitConnectParamRec ( pNIConnInfo, &ConPktParamRec, RSQL_INFO_REQUEST_EO003 ) ;

  /* send connect packet to server */
  CommState = eo420SendConnectPacket ( pNIConnInfo->NiHandle, (SEND_PKT_FUNC_PTR_EO003)eo40NiSend,
                                       &ConPktParamRec,  pErrText );

  if ( CommState != commErrOk_esp01 )
    eo40NiClose(&pNIConnInfo->NiHandle) ;
  else
  {
    /* receive server reply */
    CommState = eo420ReceiveConnectPacket ( pNIConnInfo->NiHandle, 
                                            (RECV_PKT_FUNC_PTR_EO003)eo40NiReceive,
                                           &ConPktParamRec, pErrText );

    if ( CommState == commErrOk_esp01 )
    {  
      if ( ConPktParamRec.ulMessClass != RSQL_INFO_REPLY_EO003 )
      {
       MSGD (( ERR_WRONG_MESSCLASS_REPL, ConPktParamRec.ulMessClass ));
       strcpy(pErrText, ERRMSG_PROTOCOL_ERROR);
       CommState = commErrNotOk_esp01 ;
      }
      else
      if ( ConPktParamRec.ulServiceType != pNIConnInfo->ulServiceType )
      {
        MSGD (( ERR_PROTOCOL_ERROR, "SERVICE" ));
        strcpy(pErrText, ERRMSG_PROTOCOL_ERROR);
        CommState = commErrNotOk_esp01 ;
      }
      else
      if ( strcmp ( pNIConnInfo->szServerDB,
                    ConPktParamRec.szSenderServerDB ) )
      {
        MSGD (( ERR_PROTOCOL_ERROR, "SERVERDB" ));
        strcpy(pErrText, ERRMSG_PROTOCOL_ERROR);
        CommState = commErrNotOk_esp01 ;
      }
      else
      {
        tsp00_Uint     ulPeerHeaderSize = ConPktParamRec.ulPacketSize -
                                          ConPktParamRec.ulMaxDataLen;
        tsp00_Uint     ulMyHeaderSize   = RTE_HEADER_SIZE_EO003 * 2;

        if ( ulPeerHeaderSize < ulMyHeaderSize )
             ConPktParamRec.ulMaxDataLen -= ulMyHeaderSize - ulPeerHeaderSize;

        if ( ConPktParamRec.usServicePort != 0 )
          pNIConnInfo->usServicePort = ConPktParamRec.usServicePort;

        pNIConnInfo->ulPacketSize    = ConPktParamRec.ulPacketSize;
        pNIConnInfo->ulMaxDataLen    = ConPktParamRec.ulMaxDataLen;
        pNIConnInfo->ulMinReplySize  = ConPktParamRec.ulMinReplySize;
      }
    }
  }
  return ( CommState );
}

/*===========================================================================*/

static void 
eo03InitConnectParamRec ( teo40_NiConnectInfoRecord *pNIConnInfo,
                          teo003_ConPktParamRecord  *pConPktParamRec,
                          tsp00_Uint                 ulMessClass)
{
  eo420InitConPktParamRec ( pConPktParamRec );
  pConPktParamRec->ulMessClass         = ulMessClass;
  pConPktParamRec->ulSenderRef         = pNIConnInfo->ulClientRef;
  pConPktParamRec->ulServiceType       = pNIConnInfo->ulServiceType;
  pConPktParamRec->ulMaxSegmentSize    = pNIConnInfo->ulMaxSegmentSize;
  pConPktParamRec->ulPacketSize        = pNIConnInfo->ulPacketSize;
  pConPktParamRec->ulMaxDataLen        = pNIConnInfo->ulMaxDataLen;
  pConPktParamRec->ulMinReplySize      = pNIConnInfo->ulMinReplySize;
  pConPktParamRec->pidSenderPID        = pNIConnInfo->pidClientPID;
  pConPktParamRec->usServicePort       = pNIConnInfo->usServicePort;
  strcpy ( pConPktParamRec->szReceiverServerDB, pNIConnInfo->szServerDB );

  if ( ulMessClass != RSQL_INFO_REQUEST_EO003 )
    pConPktParamRec->fAcknowledge        = pNIConnInfo->fAcknowledge;

}


/*===========================================================================*/

static tsp00_Uint     
eo03_NiUpdateConnInfoAfterConnect ( teo003_ConnectParamRecord    *pConnParam,
                                    teo40_NiConnectInfoRecord    *pNIConnInfo,
                                    tsp00_ErrTextc VAR_ARRAY_REF  pErrText)
{

  char                           *pCommPackets;
  tsp00_Uint                      CommState = commErrOk_esp01;
  tsp00_Int                       rc = NO_ERROR_EO001;

  rc = ALLOC_MEM_EO57 ( (void **)&pCommPackets,
                        pNIConnInfo->ulPacketCnt * pNIConnInfo->ulPacketSize );

  if ( NO_ERROR_EO001 == rc)
  {
    char *Ptr ;
    /* Saprouter string is the original pointer given as a DbNode from the application */
    /* if the NI-connect before was successfull, we have to save the saprouter string  */
    /* because we need it for canceling (or sqladump) the connection                   */
    rc = ALLOC_MEM_EO57 ( (void **)&Ptr, (tsp00_Uint)strlen ( pNIConnInfo->szSapRouterString ) + 1);
    if ( NO_ERROR_EO001 == rc )
    {
      strcpy ( Ptr, pNIConnInfo->szSapRouterString ) ;
      pNIConnInfo->szSapRouterString = Ptr ;
    }
    else
    {
      if ( pCommPackets != NULL )
        FREE_MEM_EO57 ( pCommPackets );
    }
  }

  if ( rc != NO_ERROR_EO001 )
    {
    eo40NiClose(&pNIConnInfo->NiHandle) ;
    eo46BuildErrorStringRC ( pErrText, (char *)ERRMSG_ALLOC_MEMORY, rc );
    CommState = commErrNotOk_esp01;
    }
  else
    {
      tsp00_Uint               ulCnt;

      for ( ulCnt = 0; ulCnt < pNIConnInfo->ulPacketCnt; ulCnt++ )
      {
        pNIConnInfo->pCommPacketList[ulCnt] = (teo003_CommPacketRecord *)pCommPackets;
        pConnParam->pCommPacketList[ulCnt]  = (teo003_CommPacketRecord *)pCommPackets;
        pCommPackets += pNIConnInfo->ulPacketSize;
      }
    }
  
  if ( CommState == commErrOk_esp01 )
  {
    pConnParam->ulPacketSize      = pNIConnInfo->ulPacketSize;
    pConnParam->ulMaxDataLen      = pNIConnInfo->ulMaxDataLen;
    pConnParam->ulMinReplySize    = pNIConnInfo->ulMinReplySize;
    pConnParam->ulServerRef       = pNIConnInfo->ulServerRef;
  }

  return( CommState ) ;
}

/*------------------------------*/

/* PTS 1105116 */
static tsp00_Uint     
eo03_GetServicePort ( char          *szSapRouterString, 
                      tsp00_Uint2   *pusServicePort,
                      tsp00_ErrTextc VAR_ARRAY_REF  pErrText)
{
  tsp00_Uint                      CommState = commErrOk_esp01;

  eo40NiExtractServiceFromSaprouterSring ( szSapRouterString, pusServicePort );

  if ( *pusServicePort <= 0 )
    if ( sql43_get_service_by_name  ( (char *)SOCKET_SERVICE_NI72_EO003, pusServicePort ) != NO_ERROR_EO001 )
    {
      CommState = commErrNotOk_esp01 ;
      strcpy( pErrText, ERRMSG_SERV_BY_NAME_ERROR);
      MSGD (( ERR_GET_SERV_BY_NAME, SOCKET_SERVICE_NI72_EO003 ));
    }
  return ( CommState ) ;
}

/*------------------------------*/

static tsp00_Uint     
eo03_InitNIConnInfo ( teo003_ConnectParamRecord    *pConnParam,
                      teo40_NiConnectInfoRecord    *pNIConnInfo,
                      tsp00_ErrTextc VAR_ARRAY_REF  pErrText)
{

  tsp00_Uint                      CommState = commErrOk_esp01;
  
  pNIConnInfo->pidClientPID     = pConnParam->pidClientPID;
  pNIConnInfo->ulServerRef      = (tsp00_Uint    )undef_ceo00;
  pNIConnInfo->ulClientRef      = pConnParam->ulClientRef;
  pNIConnInfo->ulServiceType    = pConnParam->ulServiceType;
  
  pNIConnInfo->ulCurrentPacket  = (tsp00_Uint    )undef_ceo00;
  pNIConnInfo->ulCmdDataLen     = (tsp00_Uint    )undef_ceo00;
  pNIConnInfo->ulPacketCnt      = pConnParam->ulPacketCnt;
  
  pNIConnInfo->ulPacketSize     = pConnParam->ulPacketSize;
  pNIConnInfo->ulMaxDataLen     = pConnParam->ulMaxDataLen;
  pNIConnInfo->ulMinReplySize   = (tsp00_Uint    )undef_ceo00;
  pNIConnInfo->ulMaxSegmentSize = (tsp00_Uint    )undef_ceo00;
  pNIConnInfo->fAcknowledge     = pConnParam->fAcknowledge;
  
  pNIConnInfo->pszServerDBRoot  = pConnParam->pszServerDBRoot ;
  pNIConnInfo->pszServerPgm     = pConnParam->pszServerPgm ;
  
  if ( pConnParam->pszServerNode )
  {
    pNIConnInfo->szSapRouterString = pConnParam->pszServerNode  ; /* Node is Saprouter string */
    eo40NiExtractDBNodeFromSaprouterSring  ( pNIConnInfo->szSapRouterString, pNIConnInfo->szServerNode );
  }
  else
  {
    pNIConnInfo->szSapRouterString  = NULL;
    pNIConnInfo->szServerNode[0]    = '\0';
  }
  strcpy ( pNIConnInfo->szServerDB,   pConnParam->pszServerDB );

  /* PTS 1105116 */
  CommState = eo03_GetServicePort( pNIConnInfo->szSapRouterString, 
                                  &pNIConnInfo->usServicePort, 
                                   pErrText);
  return ( CommState ) ;
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
