/*!**************************************************************************

  module      : Log_ActionObject.cpp
  special area: Logging
  responsible : UweH
  last changed: 2000-09-25  12:00
  copyright:    (c) 2000-2004 SAP AG
  description : implements class Log_ActionObject



    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*****************************************************************************/

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "KernelCommon/Kernel_VTrace.hpp"
#include "Logging/Log_ActionObject.hpp"
#include "Logging/Log_AfterImage.hpp"
#include "Logging/Log_BeforeImage.hpp"
#include "hbd04.h"
#include "hbd95.h"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/
/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/
/*===========================================================================*
 *  EXTERNAL VARIABLES                                                       *
 *===========================================================================*/
/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/* --------------------------------------------------------------------------*/
void Log_ActionObject::WritePersistentFormat
        (Data_SplitSpaceWriter &Writer,
         bool                  &isOK) const
{
    Head                          *pHead;
    Data_SplitSpaceWriter::Result  result;
    // set the structured data

    Writer.Reserve (sizeof(Head), REINTERPRET_CAST(SAPDB_Byte*&, pHead), result);
    isOK = Data_SplitSpaceWriter::moreSpaceAvailable == result ||
           Data_SplitSpaceWriter::ok                 == result;
    if ( ! isOK ) return;

    *pHead = m_Head;

    SAPDBERR_ASSERT_STATE( m_Head.keySize == 0 || m_Head.bodySize == 0 );
    
    if ( m_Head.bodySize > 0 )
    {
        if ( m_Head.bodySize > sizeof(tbd_node) )
        {
            // PTS 1117126 UH 2002-08-07 check added
            WriteToTrace ("m_Head.bodySize > sizeof(tbd_node)");
            RTE_Crash(SAPDBErr_Exception(__CONTEXT__,
                                         SAPDBERR_ASSERT_STATE_FAILED,
                                         "m_Head.bodySize > sizeof(tbd_node)"));        
        }
        Writer.Write (m_pBody, m_Head.bodySize, result);
    }
    else
        if ( m_Head.keySize > 0 )
        {
            if ( m_Head.keySize > sizeof(tbd_node) )
            {
                WriteToTrace ("m_Head.keySize > sizeof(tbd_node)");
                RTE_Crash(SAPDBErr_Exception(__CONTEXT__,
                                             SAPDBERR_ASSERT_STATE_FAILED,
                                             "m_Head.keySize > sizeof(tbd_node)"));
            }
            Writer.Write (m_pKey, m_Head.keySize, result);
        }
    isOK = Data_SplitSpaceWriter::moreSpaceAvailable == result ||
           Data_SplitSpaceWriter::ok                 == result;
    if ( ! isOK ) return;

    SAPDB_UInt alignmentLength = GetAlignmentLength();
    
    if ( alignmentLength > 0 )
    {
        SAPDB_Byte filler [5] = "\0\0\0\0";
        Writer.Write (filler, alignmentLength, result);
    }

    isOK = Data_SplitSpaceWriter::ok == result;
}

/* --------------------------------------------------------------------------*/
void Log_ActionObject::ReadPersistentFormat
        (Data_SplitSpaceReader  &Reader,
         SAPDBMem_IRawAllocator &Allocator,
         bool                   &isOK)
{
    isOK = true;

    Head                          *pHead;
    Data_SplitSpaceReader::Result  result;

    // set the structured data

    Reader.Reserve (sizeof(Head), REINTERPRET_CAST(SAPDB_Byte*&, pHead), result);
    isOK = Data_SplitSpaceReader::moreSpaceAvailable == result ||
           Data_SplitSpaceReader::ok                 == result;
    if ( ! isOK ) return;

    m_Head = *pHead;

    if ( m_Head.bodySize > 0 )
    {
        bool bAllocatorWasUsed = false;
        Reader.Read (m_Head.bodySize, m_pBody,
                     Allocator, bAllocatorWasUsed,
                     result);
        if ( bAllocatorWasUsed )
            m_pAllocator = &Allocator;
    }
    else if ( m_Head.keySize > 0 )
    {
        bool bAllocatorWasUsed = false;
        Reader.Read (m_Head.keySize, m_pKey,
                     Allocator, bAllocatorWasUsed,
                     result);
        if ( bAllocatorWasUsed )
            m_pAllocator = &Allocator;
    }

    isOK = Data_SplitSpaceReader::moreSpaceAvailable == result ||
           Data_SplitSpaceReader::ok                 == result;
    if ( ! isOK )
    {
        Kernel_VTrace() << "NOT OK: result: " << result;
        WriteToTrace("NOT OK");
        return;
    }

    SAPDB_UInt alignmentLength = GetAlignmentLength();
    
    if ( alignmentLength > 0 )
    {
        SAPDB_Byte filler[5];
        Reader.Read (filler, alignmentLength, result);
    }

    isOK = Data_SplitSpaceReader::ok == result;
}

/* --------------------------------------------------------------------------*/
void Log_ActionObject::Undo (tgg00_TransContext    &Trans,
                             const Log_BeforeImage &BeforeImage,
                             Data_PageNo            pageno,
                             Data_PageOffset        offset)
{
    if (e_ok != Trans.trError_gg00) return;
    
    switch ( BeforeImage.GetActionType() )
    {
    case Log_NewObject:
        {
            bd04RollbackObj (Trans, m_new,
                             m_Head.oid(), m_Head.fileno,
                             BeforeImage.GetPreviousState(),
                             BeforeImage.GetPreviousImage(),
                             BeforeImage.GetPreviousUpdateTransNo(),
                             0, NULL);

            if ( e_ok != Trans.trError_gg00 )
                WriteToTrace ( "bd04RollbackObj: ERROR" );
            break;
        }
    case Log_LockObject: // PTS 1119250 UH 2002-12-03 should do nothing
        break;
    case Log_DeleteObject:
    case Log_InsertObject:
    case Log_UpdateObject: 
         {
            tgg00_MessType LogType;
            switch (BeforeImage.GetActionType())
            {
            case Log_DeleteObject: LogType.becomes(m_delete); break;
            case Log_InsertObject: LogType.becomes(m_insert); break;
            case Log_UpdateObject: LogType.becomes(m_update); break;
            case Log_LockObject:   LogType.becomes(m_lock  ); break;
            default:
                LogType.becomes(m_nil); // error
            }
            bd04RollbackObj (Trans, LogType,
                             m_Head.oid(), m_Head.fileno,
                             BeforeImage.GetPreviousState(),
                             BeforeImage.GetPreviousImage(),
                             BeforeImage.GetPreviousUpdateTransNo(),
                             m_Head.bodySize, m_pBody);

            if ( e_ok != Trans.trError_gg00 )
                WriteToTrace ( "bd04RollbackObj: ERROR" );
            else
                if ( LogAction_Trace.TracesLevel(6) )
                {
                    BeforeImage.WriteToTrace ( "bd04RollbackObj: OK" );
                    WriteToTrace (NULL);
                }
            break;
        }
    default:
        Trans.trError_gg00 = e_not_implemented;
    }
}
    
/* --------------------------------------------------------------------------*/
void Log_ActionObject::Redo (tgg00_TransContext   &Trans,
                             const Log_AfterImage &AfterImage)
{
    //trace schreiben
    switch ( AfterImage.GetActionType() )
    {
    case Log_NewObject:    
		bd95RollforwardObj_NewObj (Trans,
			m_Head.oid(),
			m_Head.version(),
			m_Head.fileno,
			m_Head.keySize,
			m_pKey,
			AfterImage.GetSequence(),
			AfterImage.GetTransNo());
		break;
    case Log_InsertObject: 
    case Log_UpdateObject: 
		bd95Rollforward_UpdObj (Trans,
			m_Head.oid(),
			m_Head.version(),
			m_Head.fileno,
			m_Head.bodySize,
			m_pBody,
			AfterImage.GetSequence(),
			AfterImage.GetTransNo());
		break;
    case Log_DeleteObject: 
		bd95RollforwardObj_DelObj (Trans,
			m_Head.oid(),
			m_Head.version(),
			m_Head.fileno,
			AfterImage.GetSequence(),
			AfterImage.GetTransNo()); 
		break;
    default:
        Trans.trError_gg00 = e_not_implemented;
        return;
    }
    //trace schreiben
}

/* --------------------------------------------------------------------------*/
void Log_ActionObject::WriteToTrace (const char * Title) const
{
    Kernel_VTrace trace;
    
    if ( Title != NULL )
        trace << Title << " ";

    trace << "ObjFileNo: " << m_Head.fileno;

    tgg92_KernelOid oid = m_Head.const_oid();

    if ( oid.gg92IsNilPno() )
        trace << ", Oid: nil";
    else
        trace << ", Oid: " << oid.gg92GetPno()
                           << "." << oid.gg92GetPos()
                           << ":" << oid.gg92GetFrameVers();
    trace << ", BodySize: "  << m_Head.bodySize << ", KeySize: " << m_Head.keySize << NewLine;
    #ifdef SAPDB_SLOW
    trace << "PersistentLength: "
          << GetPersistentLength() << " ("
          << GetMinimalPersistentLength() << ")\n";
	#endif
}
    
