#! /usr/bin/perl
#
#    LCAEnv.pm - Creates and syncs a complete APO/LC make environment.
#
#    @(#)LCEnv     2003-05-12
#
#    U. Jansen, SAP AG
#
#    ========== licence begin LGPL
#    Copyright (C) 2002 SAP AG
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#    ========== licence end
#

##############################################################################
# Static includes:

use Exporter;

@ISA        = qw(Exporter);
@EXPORT_OK  = qw(ParseProtocols GetModListFromErrProt GetResponsibles DUserToRealName GenerateProtName WriteProtToWebDav TargetsToString SendErrorMail);

use strict;
use Mail::Sendmail;

##############################################################################
# ParseProtocols() - Parses a list of protocols and returns an error list.
##############################################################################

sub ParseProtocols
{
    my @ErrLines;
    my @ProtLines;
    my $ErrFound;

    foreach (@_)
    {
        print "Checking make protocol '$_'..";

        if (!open(PROT_IN, "<$_"))
        {
            print "..not found (ignored)\n";
            next;
        }

        @ProtLines = <PROT_IN>;
        close (PROT_IN);

        $ErrFound = grep { /END:\s*MAKE\s*TARGET:\s*\S*\s*RESULT:\s*ERROR/i } @ProtLines;

        if ($ErrFound)
        {
            push @ErrLines, @ProtLines;
            print "..ERRORS FOUND!\n";
        }
        else
        {
            print "..OK.\n";
        }
    }

    return @ErrLines;
}

##############################################################################
# GetModListFromErrProt() - Returns module details from an error protocol.
##############################################################################

sub GetModListFromErrProt
{
    my $href = {};
    my @ErrLines = @_;

    my %targets;
    my %depends;
    my %layers;
    my $target_count = 0;
    my $depend_count = 0;
    my $layer_count  = 0;

    while ($_ = $ErrLines[0])
    {
        shift @ErrLines;

        if (/^TARGET:\s*'(.*)'\s*.*ERROR.*/)
        {
            my $trg = $1;
            unless (defined $targets{$trg})
            {
                if ($ErrLines[1] =~ /not made because of dependency errors of/)
                {
                    $depends{$trg} = ++$depend_count;
                }
                else
                {
                    $targets{$trg} = ++$target_count;
                    $layers{$1}    = ++$layer_count if (/::?(.*)\/[^\/]*$/);
                }
            }
        }
    }

    $href->{'TARGETS'} = ();
    $href->{'DEPENDS'} = ();
    $href->{'LAYERS'}  = ();

    @{$href->{'TARGETS'}} = sort {$targets{$a} <=> $targets{$b}} keys %targets;
    $href->{'TRG_COUNT'}  = $target_count;
    $href->{'USER_TARGETS'} = GetResponsibles(@{$href->{'TARGETS'}});

    @{$href->{'DEPENDS'}} = sort {$depends{$a} <=> $depends{$b}} keys %depends;
    $href->{'DEP_COUNT'}  = $depend_count;

    @{$href->{'LAYERS'}}  = sort {$layers{$a} <=> $layers{$b}} keys %layers;
    $href->{'LAY_COUNT'}  = $layer_count;

    return $href;
}

##############################################################################
# GetResponsibles() - Retrieves the responsibles of all affected targets.
##############################################################################

sub GetResponsibles
{
    my @TrgList = @_;
    my $href = {};

    my $actuser;
    my $first;

    foreach my $target (@TrgList)
    {
        $actuser = "";
        $first = 1;

        if ($^O =~ /.*win32.*/i)
        {
            open (CMD_OUT, "python $ENV{TOOL}\\bin\\whose.py -format duser $target |") or print "ERROR! Can't open 'python $ENV{TOOL}\\bin\\whose.py -format duser $target'!\n$@\n";
        }
        else
        {
            open (CMD_OUT, "$ENV{TOOL}/bin/whose.py -format duser $target |") or print "ERROR! Can't open '$ENV{TOOL}/bin/whose.py -format duser $target'!\n$@\n";
        }

        while (<CMD_OUT>)
        {
            if ($first)
            {
                if (/(d\d+)/i)
                {
                    $actuser = $1;
                    $first   = 0;
                }
            }

            if (/(d\d+).*last\schange\sat\s(\S+)$/i)
            {
                $actuser = $1;
            }
        }

        close (CMD_OUT);

        $href->{$target} = $actuser;
    }

    return $href;
}

##############################################################################
# DUserToRealName() - Converts a D-User number into a real name.
##############################################################################

sub DUserToRealName
{
    my $duser = shift;
    my $realname = "";

    if ($^O =~ /.*win32.*/i)
    {
        open (CMD_OUT, "perl $ENV{'TOOL'}\\bin\\tel.pl $duser |") or print "ERROR! Can't open 'perl $ENV{'TOOL'}\\bin\\tel.pl $duser'!\n$!\n";
    }
    else
    {
        open (CMD_OUT, "$ENV{'TOOL'}/bin/tel.pl $duser |") or print "ERROR! Can't open '$ENV{'TOOL'}/bin/tel.pl $duser'!\n$!\n";
    }

    while (<CMD_OUT>)
    {
        $realname = $1 if (/^\"(.*)\".*/);
    }

    close (CMD_OUT);

    return $realname;
}

##############################################################################
# GenerateProtName() - Generate a useful name for the protocol file.
##############################################################################

sub GenerateProtName
{
    my $relname = shift;
    my ($os, $host) = _GetOSName();
    my @ltime   = localtime(time);

    return $os . "_" . $relname . "_" . sprintf("%04d%02d%02d%02d%02d", $ltime[5] + 1900, $ltime[4] + 1, $ltime[3], $ltime[2], $ltime[1]);
}

##############################################################################
# _GetOSName() - Returns the actual OS-Name
##############################################################################

sub _GetOSName
{
    my $hostname;
    my $os;

    $hostname =  `hostname`;
    $hostname =~ s/\n$//;
    $os = "SUN"     if ( $hostname =~ /^u.*/i );
    $os = "DEC"     if ( $hostname =~ /^d.*/i );
    $os = "AIX"     if ( $hostname =~ /^i.*/i );
    $os = "LINUX"   if ( $hostname =~ /^l.*/i );
    $os = "SNI"     if ( $hostname =~ /^s.*/i );
    $os = "HP"      if ( $hostname =~ /^h.*/i );
    $os = "WINDOWS" if ( $hostname =~ /^p.*/i );

    $os .= "(IA64)" if (($os eq "WINDOWS") and ($ENV{'CPU'} eq "IA64"));

    return ($os, $hostname);
}

##############################################################################
# WriteProtToWebDav() - Uploads a protocol to the WebDav server.
##############################################################################

sub WriteProtToWebDav
{
    my ($protname, $prot) = @_;

    open (PROT_OUT, ">$protname") or print "ERROR! Can't write protocol file '$protname'!\n$!\n";
    print PROT_OUT $prot;
    close PROT_OUT;

    if ($^O =~ /.*win32.*/i)
    {
        system("python $ENV{'TOOL'}\\bin\\webdav.py $protname");
    }
    else
    {
        system("$ENV{'TOOL'}/bin/webdav.py $protname");
    }

    unlink ($protname);

    return "http://pts:1080/webdav/daily_makes/$protname";
}

##############################################################################
# TargetsToString() - Copies all target lists into a string.
##############################################################################

sub TargetsToString
{
    my $ErrRef = shift;
    my $msg = "";

    if ($ErrRef->{'TRG_COUNT'} > 0)
    {
        $msg .= "\nAffected targets:\n\n";

        foreach my $target (keys %{$ErrRef->{'USER_TARGETS'}})
        {
            if ($ErrRef->{'USER_TARGETS'}->{$target} eq "")
            {
                $msg .= "$target\n";
            }
            else
            {
                my $uname = DUserToRealName($ErrRef->{'USER_TARGETS'}->{$target});
                $msg .= "$target ($uname)\n";
            }
        }
    }

    if ($ErrRef->{'DEP_COUNT'} > 0)
    {
        $msg .= "\nDependency errors:\n\n";

        foreach my $deperr (@{$ErrRef->{'DEPENDS'}})
        {
            $msg .= "$deperr\n";
        }
    }

    if ($ErrRef->{'LAY_COUNT'} > 0)
    {
        $msg .= "\nAffected Layers:\n\n";

        foreach my $layer (@{$ErrRef->{'LAYERS'}})
        {
            $msg .= "$layer\n";
        }
    }

    return $msg;
}

##############################################################################
# SendErrorMail() - Send the error mail.
##############################################################################

sub SendErrorMail
{
    my ($rel, $msg, $to_list, $cc_list, $ErrRef) = @_;
    my $tinylist;
    my $subject;
    my $qamode = $ENV{'LC_STATE'} ? $ENV{'LC_STATE'} : "DEV";
    my ($os, $hostname) = _GetOSName();
    my %mail;
    my $email = "";

    $tinylist = join (', ', @{$ErrRef->{'TARGETS'}});
    $tinylist = substr ($tinylist, 0, 50) . "..." if (length($tinylist) > 50);

    $subject = "Make problem on $os ($hostname) $rel $qamode ($tinylist)";

    %mail = ( To      => "$to_list",
              From    => 'automake@sap.corp',
              Message => "$msg",
              Cc      => "$cc_list",
              Smtp    => 'mail.sap-ag.de',
              Subject => "$subject" );

    if (!sendmail(%mail))
    {
        print "FATAL: sendmail error $Mail::Sendmail::error\n";
    }

    $email  = "To     : $to_list\n";
    $email .= "Cc     : $cc_list\n";
    $email .= "Subject: $subject\n\n";
    $email .= "Content:\n\n$msg\n";

    return $email;
}