#!/usr/bin/env python
# allsources.py

import string
import os

_platforms = [
    'linux',
    'aix64',
    'sun64',
    'hp64',
    'dec',
    'ia64']

_platformVars = [
    # MACH,  OS,      OSSPEC,  BIT64, other
    ('I386', 'WIN32', 'WIN32', None,  None),
    ('I386', 'WIN32', 'WIN32', 1,     None),
    ('I386', 'UNIX',  'LINUX', None,  'COREPERL_INCLUDE=YES'),
    ('IA64', 'UNIX',  'LINUX', 1,     'COREPERL_INCLUDE=YES'),
    ('SUN',  'UNIX',  'SVR4',  None,  'COREPERL_INCLUDE=YES'),
    ('SUN',  'UNIX',  'SVR4',  1,     'COREPERL_INCLUDE=YES'),
    ('_IBMR2', 'UNIX','AIX',   None,  'COREPERL_INCLUDE=YES'),
    ('_IBMR2', 'UNIX','AIX',   1,     'COREPERL_INCLUDE=YES'),
    ('PA20W', 'UNIX', 'HPUX',  None,  'COREPERL_INCLUDE=YES'),
    ('PA20W', 'UNIX', 'HPUX',  1,     'COREPERL_INCLUDE=YES'),
    ('IA64',  'UNIX', 'HPUX',  1,     'COREPERL_INCLUDE=YES'),
    ('ALPHA', 'UNIX', 'OSF1',  1,     'COREPERL_INCLUDE=YES'),
]

_fastTargets = ['all']
_quickTargets = ['quickknl']
_slowTargets = ['slowknl']
_cmdPrefix = 'perl %(TOOL)s/bin/' % os.environ

_platformSpecific = [
    'src/en/ven38NMP.cpp',
    'src/eo/heo35k.h',
    'src/eo/veo35k.cpp',
]

_additionalSources = [
]

_intentionalMissingSources = [
    'src/redist/bin/ALPHA_OSF1_64/python',
    'src/redist/bin/PA20W_HPUX_64/python',
    'src/redist/bin/PA20W_HPUX_32/python',
    'src/redist/bin/SUN_SVR4_64/python',
    'src/redist/bin/SUN_SVR4_32/python',
    'src/redist/bin/_IBMR2_AIX_64/python',
    'src/redist/bin/_IBMR2_AIX_32/python',
    'src/redist/bin/IA64_LINUX_64/python',
]

class AllSources:
    def __init__ (self, verbose, fastTargets, quickTargets, slowTargets):
        self.verbose = verbose
        self.fastTargets = fastTargets
        self.quickTargets = quickTargets
        self.slowTargets = slowTargets
        self.sources = {}
        for source in _additionalSources:
            self.sources [source] = 1

    def addLocalSources (self):
        if self.verbose:
            print 'local',
        self.readSourcesFromPipe ('imf.pl', self.fastTargets)
        self.readSourcesFromPipe ('imq.pl', self.quickTargets)
        self.readSourcesFromPipe ('ims.pl', self.slowTargets)
        if self.verbose:
            print '=>', len (self.sources.keys ())

    def addRemoteSources (self, platform):
        if self.verbose:
            print platform,
        self.readSourcesFromPipe ('remimf.pl ' + platform, self.fastTargets)
        self.readSourcesFromPipe ('remimq.pl ' + platform, self.quickTargets)
        self.readSourcesFromPipe ('remims.pl ' + platform, self.slowTargets)
        if self.verbose:
            print '=>', len (self.sources.keys ())

    def addPlatformSources (self, mach, osvar, osspec, isbit64, other):
        envvar = 'MACH=%s;OS=%s;OSSPEC=%s' % (mach, osvar, osspec)
        if isbit64:
            envvar = envvar + ';BIT64=1'
            bit = 64
        else:
            bit = 32
        if self.verbose:
            print '%-6s %-6s %d' % (osspec, mach, bit),
        if other:
            envvar = envvar + ';' + other
        os.putenv ('TOOLVARS_PATCH', envvar)
        self.readSourcesFromPipe ('imf.pl', self.fastTargets)
        self.readSourcesFromPipe ('imq.pl', self.quickTargets)
        self.readSourcesFromPipe ('ims.pl', self.slowTargets)
        if self.verbose:
            print '=>', len (self.sources.keys ())


    def writeResult (self, outfile):
        for source in _intentionalMissingSources:
            try:
                del self.sources [source]
            except KeyError:
                pass
        sources = self.sources.keys ()
        sources.sort ()
        stream = open (outfile, 'w')
        for source in sources:
            stream.write (source + '\n')
        stream.close ()

    def readSourcesFromPipe (self, cmd, targets):
        cmdString = '%s -m %s' % (cmd, string.join (targets))
        lines = os.popen (cmdString, 'r').readlines ()
        for line in lines:
            startDir = string.split (line, '/') [0]
            if startDir in ['src', 'desc']:
                self.sources [string.strip (line)] = 1

def writeMissingSources ():
    stream = open ('missing.lst', 'w')
    for fname in _intentionalMissingSources:
        stream.write (fname + '\n')
    stream.close ()


def checkOptions (options):
    if not options.ftarget and not options.qtarget and not options.starget:
        options.ftarget = _fastTargets
        options.starget = _slowTargets
        global _additionalSources
        _additionalSources = _platformSpecific

def main (options, args):
    if os.environ.has_key ('SAPDB_INTERNAL'):
        os.putenv ('SAPDB_INTERNAL', '')
    checkOptions (options)
    if options.missing:
        writeMissingSources ()
    else:
        sources = AllSources (options.verbose, options.ftarget,
            options.qtarget, options.starget)
        #sources.addLocalSources ()
        #for platform in _platforms:
        #    sources.addRemoteSources (platform)
        for mach, osvar, osspec, isbit64, other in _platformVars:
            sources.addPlatformSources (mach, osvar, osspec, isbit64, other)
        sources.writeResult (options.outfile)

def _options ():
    return [
        # (optstring, varname, typechar, default, help)
        ('o', 'outfile', ':', 'sources.lst', 'name of output file'),
        ('ftarget', None, '::', None, 'fast vmake targets to list'),
        ('qtarget', None, '::', None, 'fast vmake targets to list'),
        ('starget', None, '::', None, 'fast vmake targets to list'),
        ('missing', None, '', None, 'write list of missing files only'),
        ('v', 'verbose', '', None, 'verbose output'),
        ]

if __name__ == '__main__':
    import optlib
    optlib.optMain2 (main, _options ())

