#-----------------------------------------------------------------------------
#
#  Copyright (c) 2007 by Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

"""
A base class for Action declarations.

"""

# Standard library imports
import re

# Enthought library imports
from enthought.traits.api import Str
from enthought.envisage.action.action_plugin_definition import Action


class DefaultAction(Action):
    """
    A base class for Action declarations.

    Deriving an action class from this base class allows you to
    minimize the number of attributes that must be typed as the name
    of the derived class is used to generate the id, class_name, and
    image attributes.

    For an action class called MyCustomAction:
        id = 'MyCustom'
        class_name = '<ROOT PATH>.action.my_custom_action.' + \
                     'MyCustomAction
        image = 'my_custom'

    Where <ROOT_PATH> can either be specified, or is assumed to be
    the path to the package containing the module which contains the
    class declaration.

    Additionally, if the tooltip attribute is not set, it is set to the
    description attribute value.

    """

    # The path to the package that the action implementation will
    # be found within.
    root_path = Str


    def __init__(self, *args, **kw):
        """
        Constructor.

        Overridden to set our attributes based on our class's name.

        """

        super(Action, self).__init__(*args, **kw)

        # If no root path was specified, then assume we want the
        # same package that contained the module this class was
        # declared within.
        if self.root_path == '':
            module = self.__class__.__module__
            package = re.sub('\.\w+$', '', module)
            self.root_path = package

        # If names aren't specified for some of the attributes,
        # we create default ones.
        camel_class_name = self.__class__.__name__
        underscore_class_name = \
            camel_case_to_lowercase_underscore(camel_class_name)

        if self.class_name == '':
            self.class_name = '%s.action.%s.%s' % (self.root_path,
                underscore_class_name, camel_class_name)

        # The id and image strings don't use the 'Action' part
        # of the class name in their string.
        id_name = camel_class_name.replace('Action', '')
        image_name = camel_case_to_lowercase_underscore(id_name)

        if self.id == '':
            self.id = id_name

        if self.image == '':
            self.image = image_name

        # If the tooltip was not specified, then set it equal to the
        # description value.
        if self.tooltip == '':
            self.tooltip = self.description

        return


def camel_case_to_lowercase_underscore(s):
    """
    Convert a CamelCase name to lower cased underscore format.

    ie. CamelCase->camel_case

    This is useful for translating class names to module names.

    """

    # Replace a upper case letters with an _ and their same
    # character.  ie. CamelCase -> _Camel_Case
    s = re.sub('([A-Z])', '_\\1', s)

    # Remove the leading underscore if it exists.
    # ie. _Camel_Case -> Camel_Case
    s = re.sub('^_', '', s)

    # Finally lower case it.
    # ie. Camel_Case -> camel_case
    result = s.lower()

    return result

