""" Adds a menu bar and tool bar to every workbench window. """


# Enthought library imports.
from enthought.envisage import Plugin
from enthought.envisage.action import ActionSetManager
from enthought.envisage.workbench.services import IWORKBENCH, IWORKBENCH_UI

# Local imports.
from menu_bar_manager import MenuBarManager
from menu_builder import MenuBuilder
from tool_bar_manager import ToolBarManager
from tool_bar_builder import ToolBarBuilder
from view_menu import ViewMenu

# Plugin definition imports.
from action_plugin_definition import ActionSet


class ActionPlugin(Plugin):
    """ Adds a menu bar and tool bar to every workbench window.

    This plugin listens for workbench window creation, and adds on a menu bar
    and a tool bar built from action set contributions with the following
    roots:-

    'enthought.envisage.workbench.menubar'
    'enthought.envisage.workbench.toolbar'
    
    """

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin. """

        # Find the workbench service.
        workbench = self.get_service(IWORKBENCH)

        # Listen for windows being created.
        workbench.on_trait_change(self._on_window_created, 'window_created')

        return

    ###########################################################################
    # 'Private' interface.
    ###########################################################################

    def _create_menu_bar_manager(self, window, action_set_manager):
        """ Creates a menu bar manager from contributed action sets. """

        menu_bar_manager = MenuBarManager(window=window)

        # Initialize the menu bar manager with the contributed actions, groups
        # and menus.
        self._initialize_menu_bar(menu_bar_manager, action_set_manager)

        return menu_bar_manager

    def _initialize_menu_bar(self, menu_bar_manager, action_set_manager):
        """ Initializes a menu bar manager with all contributions. """

        menu_builder = MenuBuilder(window=menu_bar_manager.window)
        menu_builder.initialize_menu_manager(
            menu_bar_manager,
            action_set_manager,
            'enthought.envisage.workbench.menubar'
        )

        return

    def _create_tool_bar_manager(self, window, action_set_manager):
        """ Creates a tool bar manager from contributed action sets. """

        tool_bar_manager = ToolBarManager(
            window=window, show_tool_names=window.workbench.show_tool_names
        )

        # Initialize the too bar  manager with the contributed actions and
        # groups.
        self._initialize_tool_bar(tool_bar_manager, action_set_manager)

        return tool_bar_manager

    def _initialize_tool_bar(self, tool_bar_manager, action_set_manager):
        """ Initializes a tool bar manager with all contributions. """

        tool_bar_builder = ToolBarBuilder(window=tool_bar_manager.window)
        tool_bar_builder.initialize_menu_manager(
            tool_bar_manager,
            action_set_manager,
            'enthought.envisage.workbench.toolbar'
        )

        return

    def _add_view_menu(self, window):
        """ Adds the view menu to the window. """
        
        # Add the view menu.
        view_menu = ViewMenu(window=window)

        # fixme: This is required for two reasons.
        #
        # 1) It is called by the window layout when a view is closed.
        # 2) The view menu listens for changes to the window's active
        #    perspective, and so if we don't save a reference, it gets GC'd
        #    and the trait change handler never gets called.
        window.menu_bar_manager._view_menu = view_menu

        return
    
    def _get_action_sets(self):
        """ Returns all action set contributions. """

        action_sets = self.application.load_extensions(ActionSet)

        return action_sets

    #### Trait change handlers ################################################

    def _on_window_created(self, event):
        """ Dynamic trait change handler. """

        # The window that has just been created.
        window = event.window

        # Get all action, group and menu extensions.
        action_sets        = self._get_action_sets()
        action_set_manager = ActionSetManager(action_sets=action_sets)

        # Add the menu bar.
        #
        # fixme: We just completely overwrite for now!
        window.menu_bar_manager = self._create_menu_bar_manager(
            window, action_set_manager
        )

        # Add the view menu.
        #
        # fixme: This seems quite ugly!
        self._add_view_menu(window)
        
        # Add the toolbar.
        #
        # fixme: We just completely overwrite for now!
        window.tool_bar_manager = self._create_tool_bar_manager(
            window, action_set_manager
        )

        return
    
#### EOF ######################################################################
