""" Provides the ability to define menu bars, tool bars and menus. """

# Enthought library imports.
from enthought.envisage import ExtensionItem, ExtensionPoint, PluginDefinition
from enthought.traits.api import Bool, Dict, Enum, Instance, List, Str


# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = "enthought.envisage.action"


###############################################################################
# Extension points.
###############################################################################

class Location(ExtensionItem):
    """ The location of a group, menu, or action, within an action hierarchy."""

    # A forward-slash--separated path through the action hierarchy, to
    # either:
    #
    # * A menu: For groups *only*. This is the menu that the group is added to.
    #
    # * A group: For menus and actions *only*. This is the group that the menu 
    #   or action is added to.
    #
    # Examples
    # --------
    #
    # * To add a group to the menu bar: ``path = "MenuBar"``
    #
    # * To add a group to the tool bar: ``path = "ToolBar"``
    #
    # * To add a group to a sub-menu: ``path = "MenuBar/FooMenu/BarMenu"``
    #
    # * To add a menu to the menu bar: ``path = "MenuBar/BazGroup"``
    #
    # * To add a menu or action to a menu: 
    #   ``path = "MenuBar/FooMenu/BarMenu/BazGroup"``
    #
    path = Str

    # Placement of the action within the menu or group specified by the location 
    # path.
    
    # The item appears after the item with this ID.
    #
    # For groups, this is the ID of another group.
    # For menus and actions, this is the ID of another menu or action.
    after = Str

    # The action appears before the item with this ID.
    #
    # For groups, this is the ID of another group.
    # For menus and actions, this is the ID of another menu or action.
    before = Str


class Group(ExtensionItem):
    """ A group in a tool bar or menu. """

    # The group's unique identifier (unique within the tool bar, menu bar or
    # menu that the group is to be added to).
    id = Str

    # Does this group require a separator?
    separator = Bool(True)

    # The location of the group.
    location = Instance(Location)

    # The optional name of a class that implements the group. The class must
    # support the **enthought.pyface.action.Group** interface.
    class_name = Str


class Menu(ExtensionItem):
    """ A menu in a menu bar or menu. """

    # The menu's unique identifier (unique within the group that the menu is to
    # be added to).
    id = Str

    # The menu name (appears on the menu bar or menu).
    name = Str

    # The groups in the menu.
    groups = List(Group)

    # The location of the menu.
    location = Instance(Location)

    # The optional name of a class that implements the group. The class must
    # support the **enthought.pyface.action.MenuManager** interface.
    class_name = Str


class EnabledWhen(ExtensionItem):
    """ Controls when actions are enabled. """

    # The action is enabled only if *all* selected objects have the specified
    # cookie.
    cookie = Str

    # The action is enabled only if the *parents* of all selected objects have
    # the specified cookie.
    parent_cookie = Str

    # The action will be enabled iff ALL selected objects are of the specified
    # resource type.
    resource_type = Str


class DisabledWhen(ExtensionItem):
    """ Controls when actions are disabled. """

    # The action is disabled only if any of the selected objects has the
    # specified cookie.
    cookie = Str

    # The action is disabled only if any of the *parents* of the selected
    # objects has the specified cookie.
    parent_cookie = Str

    # The action is enabled only if *all* selected objects are of the 
    # specified resource type.
    resource_type = Str


class Action(ExtensionItem):
    """ An action in a menu bar, menu, or tool bar. """

    #### General ##############################################################

    # The action's globally unique identifier.
    id = Str

    # The action's name (displayed on menus, and tool bar tools).
    name = Str

    # Keyboard accelerator. By default the action has no accelerator.
    accelerator = Str

    # Is the action initially checked?
    #
    # This is relevant only if the action's **style** is 'radio' or 'toggle'.
    checked = Bool(False)

    # A longer description of the action (e.g., for context sensitive help).
    #
    # If no description is specified, then the **tooltip** text is used 
    # If there is no tooltip, then nothing is displayed.
    description = Str

    # Is the action initially enabled?
    enabled = Bool(True)

    # The action's image (displayed on tool bar tools).
    image = Str

    # A list of paths to be searched for the action's image.  If not specified,
    # the plugin's location is used.
    image_path = List(Str)

    # The action's style.
    style = Enum('push', 'radio', 'toggle')

    # A short description of the action used for tooltip text (and for the 
    # description if no **description** is specified).
    tooltip = Str

    #### Action implementation ################################################

    # The name of the class that implements the action.
    class_name = Str

    # Should the implementation class be be lazy-loaded?
    lazy_load = Bool(True)

    # The name of the function that implements the action.
    function_name = Str

    # The method to invoke on the target object.
    method_name = Str

    # The object that this action invokes a method on.
    #
    # This value is a UOL (Universal Object Locator) string reference, which is 
    # resolved to locate the actual object. The currently supported 'protocols'
    # are:
    #
    # * ``'service://a_service_identifier'``
    # * ``'name:/a/path/through/the/naming/service'``
    # * ``'file:/the/pathname/of/a/file/containing/a/UOL'``
    # * ``'http://a/URL/pointing/to/a/text/document/containing/a/UOL'``
    object = Str

    #### Placement ############################################################

    # The locations of the action. Unlike groups and menus, actions can appear
    # in multiple locations, e.g., on a menu *and* on the tool bar.
    locations = List(Location)

    #### fixme: These are hacks ####

    # Controls when the action is enabled.
    enabled_when = Instance(EnabledWhen)

    # Controls when the actions is disabled.
    disabled_when = Instance(DisabledWhen)


class ActionSet(ExtensionPoint):
    """ An action set is a collection of menus, groups, and actions. """

    # The action set's globally unique identifier.
    id = Str

    # The action set's name.
    #
    # fixme: This is not currently used, but in future it will be the name that
    # is shown to the user when they are customizing perspectives by adding or
    # removing action sets etc.
    name = Str

    # The actions in this set.
    actions = List(Action)

    # The groups in this set.
    groups = List(Group)

    # The menus in this set.
    menus = List(Menu)

    # A mapping from human-readable names to globally unique IDs.
    #
    # This mapping is used when interpreting the first item in a location path
    # (i.e., the **path** trait of a **Location** instance).
    #
    # When the path is intepreted, the first component (i.e., the first item 
    # before any '/') is checked to see if it is in the mapping, and if so it
    # is replaced with the value in the map.
    #
    # This technique allows paths to start with human readable names, as 
    # opposed to IDs (which are required in order to manage the namespace of 
    # all action sets).
    #
    # For example, in the Envisage workbench, the menu bar ID is:
    #
    # ``'enthought.envisage.workbench.menubar'``
    #
    # Without aliases, you must specify a location like this:
    #
    # ``Location(path='enthought.envisage.workbench.menubar/ASubMenu/AGroup')``
    #
    # This is a bit long-winded! Instead, you can define an alias:
    #
    #     ``aliases = { 'MenuBar' : 'enthought.envisage.workbench.menubar' }``
    #
    # In that case, you can specify a location like this:
    #
    #     ``Location(path='MenuBar/ASubMenu/AGroup')``
    #
    aliases = Dict(Str, Str)

###############################################################################
# The plugin definition.
###############################################################################

class ActionPluginDefinition(PluginDefinition):
    """ The definition of the action plugin. """

    # The plugin's globally unique identifier.
    id = ID

    # General information about the plugin.
    name          = "Envisage Action Plugin"
    version       = "1.0.0"
    provider_name = "Enthought Inc"
    provider_url  = "www.enthought.com"

    # The Id's of the plugins that this plugin requires.
    requires = ["enthought.envisage.core"]

    # The extension points offered by this plugin.
    extension_points = [ActionSet]

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = []

#### EOF ######################################################################
