""" The base class for all templates. """


# Standard library imports.
import os
from os.path import basename, dirname, join

# Enthought library imports.
from enthought.io import File
from enthought.pyface.api import ImageResource
from enthought.traits.api import HasPrivateTraits, Instance, Str


class Template(HasPrivateTraits):
    """ The base class for all templates. """

    #### 'Template' interface #################################################
    
    # The name of the template.
    name = Str

    # The path to the icon that represents the template.
    image = Instance(ImageResource, ImageResource('template'))

    # A short description of the template.
    description = Str

    ###########################################################################
    # 'Template' interface.
    ###########################################################################

    def generate(self):
        """ Create an object. """

        raise NotImplementedError


class FileCopyTemplate(Template):
    """ A template that creates 'objects' by copying files etc. """

    #### 'FileCopyTemplate' interface #########################################

    # The path name of the root folder to copy.
    source = Str

    # The target location.
    target = Str
    
    ###########################################################################
    # 'Template' interface.
    ###########################################################################

    def generate(self, force=False):
        """ Create an object. """

        # fixme: Should src and trg be 'File's?!?
        src = File(self.source)
        trg = File(self.target)

        if not force:
            # If the target already exists then add 'Copy of' to the name and
            # try again.
            if trg.exists:
                trg = self._uniquify(trg)

        else:
            if trg.exists:
                trg.delete()
                
        src.copy(trg)
        
        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _uniquify(self, f):
        """ Uniquify a file within its parent directory. """

        dir  = dirname(f.path)
        base = basename(f.path)

        f = File(join(dir, 'Copy of %s' % base))
        id = 2
        while f.exists:
            f = File(join(dir, 'Copy (%d) of %s' % (id, base)))
            id += 1

        return f
        
#### EOF ######################################################################
