""" The Envisage UI plugin. """


# Enthought library imports.
from enthought.envisage import ExtensionItem, ExtensionPoint, PluginDefinition
from enthought.traits.api import Any, Bool, Enum, Instance, Int, List, Str, Tuple

# Plugin definition imports.
from enthought.envisage.core.core_plugin_definition import Preferences


# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = "enthought.envisage.ui"


###############################################################################
# Extension points.
###############################################################################

#### UI branding ##############################################################

class UIBranding(ExtensionPoint):
    """ UI branding (the application icon, 'About' box image etc). """

    # The about box image.
    about_image = Any

    # The application icon.
    application_icon = Any

    # The application name (used for window titles etc).
    application_name = Str

    # The size for images on the toolbar
    tool_bar_image_size = Tuple()

#### Library version ##########################################################

class LibraryVersions(ExtensionPoint):
    """ Library versions that need acknowledgement in the About dialog. """

    # Version strings to include in the about dialog.
    versions = List(Str)

#### Actions ##################################################################

class EnabledWhen(ExtensionItem):
    """ Action enablement. """

    # The action will be enabled iff ALL selected objects have the specified
    # cookie.
    cookie = Str

    # The action will be enabled iff the PARENTS of ALL selected objects have
    # the specified cookie.
    parent_cookie = Str

    # The action will be enabled iff ALL selected objects are of the specified
    # resource type.
    resource_type = Str

    # fixme: A hack to ask the actual aciton to hanlde its enablement!
    hack = Bool(False)

class DisabledWhen(ExtensionItem):
    """ Action disablement. """

    # The action will be disabled iff any of the selected objects have the
    # specified cookie.
    cookie = Str

    # The action will be disabled iff any of the PARENTS of the selected
    # objects have the specified cookie.
    parent_cookie = Str

    # The action will be enabled iff ALL selected objects are of the specified
    # resource type.
    resource_type = Str

class Action(ExtensionItem):
    """ An action in a menu bar, menu or tool bar. """

    # Keyboard accelerator (by default the action has NO accelerator).
    accelerator = Str

    # Is the action checked?  This is only relevant if the action style is
    # 'radio' or 'toggle'.
    checked = Bool(False)

    # The name of the class that implements the action.
    class_name = Str

    # The name of the function that implements the action.
    function_name = Str

    # A longer description of the action (used for context sensitive help etc).
    # If no description is specified, the tooltip is used instead (and if there
    # is no tooltip, then well, maybe you just hate your users ;^).
    description = Str

    # Is the action enabled?
    enabled = Bool(True)

    # Enablement.
    enabled_when = Instance(EnabledWhen)

    # Disablement.
    disabled_when = Instance(DisabledWhen)

    # The action's unique identifier (may be None).
    id = Str

    # The action's image (displayed on tool bar tools etc).
    image = Str

    # The position of the action on the menu hierarchy (the empty string means
    # the action will not appear on a menu).
    menu_bar_path = Str

    # The action's name (displayed on menus/tool bar tools etc).
    name = Str

    # The action's style.
    style = Enum('push', 'radio', 'toggle')

    # The position of the action on the tool bar (the empty string means that
    # the aciton will not appear on the tool bar).
    tool_bar_path = Str

    # A short description of the action used for tooltip text etc.
    tooltip = Str


class Group(ExtensionItem):
    """ A group in a menu bar, menu or tool bar. """

    # The group's unique identifier.
    id = Str

    # Does this group require a separator?
    separator = Bool(True)

    # Placement.
    #
    # Put the group after the group with this Id.  If no such group exists the
    # group will simply be appended to the menu/menu bar/tool bar.
    after = Str

    # Put the group before the group with this Id.  If no such group exists the
    # group will simply be appended to the menu/menu bar/tool bar.
    before = Str

    # Put the group at the specifed index.  If the index is invalid the group
    # will simply be appended to the menu/menu bar/tool bar.
    index = Int

    # The path to the menu that the group should appear in.
    path = Str

class Menu(ExtensionItem):
    """ A menu in a menu bar, or menu. """

    # The menu's unique identifier.
    id = Str

    # The menu name (appears on the menu bar, or menu).
    name = Str

    # The path.
    path = Str

    # The groups in the menu.
    groups = List(Group)

    # The optional name of a class that implements MenuManager interface
    class_name = Str


class UIActions(ExtensionPoint):
    """ Allows contributions to the menu bar and tool bar. """

    # Action contributions.
    actions = List(Action)

    # Group contributions.
    groups = List(Group)

    # Menu contributions.
    menus = List(Menu)

#### Views ####################################################################

class Category(ExtensionItem):
    """ A view category. """

    # The category's unique identifier.
    id = Str

    # The categeory name.
    name = Str


class View(ExtensionItem):
    """ A view. """

    # The view's unique identifier.
    id = Str

    # The name of the class that implements the view.
    class_name = Str

    # The view name.
    name = Str

    # The image used to present the view visually.
    image = Str

    # The category that the view belongs to.
    category = Str

    # The default position of the view in a workbench window.
    position = Str('left')

    # The default visibility setting until the user saves a preference.
    visible = Bool(True)


class UIViews(ExtensionPoint):
    """ Allows for view contributions. """

    # View category contributions (for organising views hierarchically).
    categories = List(Category)

    # View contributions.
    views      = List(View)

###############################################################################
# Extensions.
###############################################################################

#### Preferences ##############################################################

preferences = Preferences(
    defaults = {
        # Should we prompt the user before exiting the application?
        'prompt_on_exit' : True,

        # Should we display the name of each tool bar tool under its image?
        'show_tool_names'  : True,
    }
)

#### UI Actions ###############################################################

ui_actions = UIActions(
    menus = [
        Menu(
            id     = "FileMenu",
            name   = "&File",
            path   = "FileGroup",

            groups = [
                Group(id = "Start"),
                Group(id = "End"),
            ]
        ),

        Menu(
            id     = "ViewMenu",
            name   = "&View",
            path   = "HelpGroup",

            groups = [
                Group(id = "Start"),
                Group(id = "End"),
            ]
        ),

        Menu(
            id     = "HelpMenu",
            name   = "&Help",
            path   = "HelpGroup",

            groups = [
                Group(id = "Start"),
                Group(id = "End"),
            ]
        )
    ],

    actions = [
        Action(
            id            = ID + ".action.about_action.AboutAction",
            class_name    = ID + ".action.about_action.AboutAction",
            name          = "&About",
            image         = "images/about.png",
            tooltip       = "Display information about the application",
            menu_bar_path = "HelpMenu/",
        ),

        Action(
            id            = ID + ".action.exit_action.ExitAction",
            class_name    = ID + ".action.exit_action.ExitAction",
            name          = "E&xit",
            tooltip       = "Exit the application",
            image         = "images/exit.png",
            menu_bar_path = "FileMenu/",
        )
    ]
)

###############################################################################
# The plugin!
###############################################################################

PluginDefinition(
    # The plugin's globally unique identifier.
    id = ID,

    # The name of the class that implements the plugin.
    class_name = "enthought.envisage.ui.UIPlugin",

    # General information about the plugin.
    name          = "Envisage User Interface",
    version       = "1.0.0",
    provider_name = "Enthought Inc",
    provider_url  = "www.enthought.com",
    enabled       = True,
    autostart     = True,

    # The Id's of the plugins that this plugin requires.
    requires = ["enthought.envisage.core"],

    # The extension points offered by this plugin.
    extension_points = [UIBranding, UIActions, UIViews],

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = [ui_actions, preferences]
)

#### EOF ######################################################################
