/* Copyright (C) 2004,2005 Andi Kleen, SuSE Labs.
   Decode IA32/x86-64 machine check events in /dev/mcelog. 

   mcelog is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; version
   2.

   mcelog is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should find a copy of v2 of the GNU General Public License somewhere
   on your Linux system; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#define _GNU_SOURCE 1
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <asm/types.h>
#include <asm/ioctls.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <stdarg.h>
#include <ctype.h>
#include "mcelog.h"
#include "k8.h"

enum {
	CPU_GENERIC,
	CPU_K8
} cpu = CPU_GENERIC;	

char *logfn = "/dev/mcelog";

int use_syslog;

void Wprintf(char *fmt, ...)
{
	va_list ap;
	va_start(ap,fmt);
	if (use_syslog) 
		vsyslog(LOG_NOTICE, fmt, ap);
	else
		vprintf(fmt, ap);
	va_end(ap);
}

char *bankname(unsigned bank) 
{ 
	static char numeric[64];
	switch (cpu) { 
	case CPU_K8:
		return k8_bank_name(bank);
		/* add banks of other cpu types here */
	default:
		sprintf(numeric, "BANK %d", bank); 
		return numeric;
	}
} 

void dump_mce(struct mce *m) 
{
	/* should not happen */
	if (!m->finished)
		Wprintf("not finished?\n");
	Wprintf("CPU %d %s ", m->cpu, bankname(m->bank));
	if (m->tsc)
		Wprintf("TSC %Lx %s\n", 
			m->tsc, 
			(m->mcgstatus & MCI_STATUS_UC) ? 
			"(upper bound, found by polled driver)" : "");
	else
		Wprintf("from boot or resume\n");
	if (m->rip) 
		Wprintf("RIP%s %02x:%Lx ", 
		       !(m->mcgstatus & MCG_STATUS_EIPV) ? " !INEXACT!" : "",
		       m->cs, m->rip);
	if (m->misc)
		Wprintf("MISC %Lx ", m->misc);
	if (m->addr)
		Wprintf("ADDR %Lx ", m->addr);
	if (m->rip | m->misc | m->addr)	
		Wprintf("\n");
	switch (cpu) { 
	case CPU_K8:
		decode_k8_mc(m); 
		break;
	/* add handlers for other CPUs here */
	default:
		break;
	} 
	/* decode all status bits here */
	Wprintf("STATUS %Lx MCGSTATUS %Lx\n", m->status, m->mcgstatus);
}

void check_cpu(void)
{ 
	FILE *f;
	f = fopen("/proc/cpuinfo","r");
	if (f != NULL) { 
		int found = 0; 
		int family; 
		char vendor[64];
		char *line = NULL;
		size_t linelen = 0; 
		while (getdelim(&line, &linelen, '\n', f) > 0 && found < 2) { 
			if (sscanf(line, "vendor_id : %63[^\n]", vendor) == 1) 
				found++; 
			if (sscanf(line, "cpu family : %d", &family) == 1)
				found++;
		} 
		if (found == 2) { 
			if (!strcmp(vendor,"AuthenticAMD") && family == 15)
				cpu = CPU_K8;
			/* Add checks for other CPUs here */	
		} else {
			fprintf(stderr, "mcelog: warning: Cannot parse /proc/cpuinfo\n"); 
		} 
		fclose(f);
		free(line);
	} else
		fprintf(stderr, "mcelog: warning: Cannot open /proc/cpuinfo\n");
} 

char *skipgunk(char *s)
{
	while (isspace(*s))
		++s; 
	if (*s == '<') { 
		s += strcspn(s, ">"); 
		if (*s == '>') 
			++s; 
	}
	while (isspace(*s))
		++s; 
	return s;
}

void dump_mce_final(struct mce *m, char *symbol, int missing)
{
	m->finished = 1;
	dump_mce(m);
	if (symbol[0])
		Wprintf("RIP: %s\n", symbol);
}

/* Decode ASCII input for fatal messages */
void decodefatal(FILE *inf)
{
	struct mce m;
	char *line = NULL; 
	size_t linelen = 0;
	int k;
	int missing = 0; 
	char symbol[100];
	int data = 0;

	k = 0;
	memset(&m, 0, sizeof(struct mce));
	symbol[0] = '\0';
	while (getdelim(&line, &linelen, '\n', inf) > 0) { 
		int n = 0;
		char *s = skipgunk(line);

		if (!strncmp(s, "CPU", 3)) { 
			unsigned cpu = 0, bank = 0;
			n = sscanf(s,
	       "CPU %u: Machine Check Exception: %16Lx Bank %d: %016Lx",
			       &cpu,
			       &m.mcgstatus,
			       &bank,
			       &m.status
			       );
			m.cpu = cpu;
			m.bank = bank;
			if (n != 4) 
				missing++; 
		} 

		else if (!strncmp(s, "RIP", 3)) { 
			unsigned cs = 0; 

			if (!strncmp(s, "RIP!INEXACT!", 12))
				s += 12; 
			else
				s += 3; 

			n = sscanf(s, "%02x:<%016Lx> {%100s}",
				   &cs,
				   &m.rip, 
				   symbol); 
			m.cs = cs;
			if (n < 2) 
				missing++; 
		} 

		else if (!strncmp(s, "TSC",3)) { 
			if ((n = sscanf(s, "TSC %Lx", &m.tsc)) != 1) 
				missing++; 
		}
		else if (!strncmp(s, "ADDR",4)) { 
			if ((n = sscanf(s, "ADDR %Lx", &m.addr)) != 1) 
				missing++; 
		}
		else if (!strncmp(s, "MISC",4)) { 
			if ((n = sscanf(s, "MISC %Lx", &m.misc)) != 1) 
				missing++; 
		}
		else { 
			if (*s && data) { 
				dump_mce_final(&m, symbol, missing); 
				data = 0;
			} 
			Wprintf("%s", line); 
		} 
		if (n > 0) 
			data = 1;
	} 
	free(line);
	if (data)
		dump_mce_final(&m, symbol, missing);
}

void usage(void)
{
	fprintf(stderr, 
		"Usage:\n"
		"  mcelog [--k8] [--generic] [--syslog] [mcelogdevice]\n"
		"  mcelog [--k8] [--generic] --ascii\n"
		"Decode machine check error records\n");	
	exit(1);
}

int main(int ac, char **av) 
{ 
	int recordlen = 0;
	int loglen = 0;

	if (av[1] && !strcmp(av[1],"--syslog")) { 
		av++;
		openlog("mcelog", 0, LOG_DAEMON);
		use_syslog = 1;
	}

	check_cpu();
	if (av[1]) { 
		if (!strcmp(av[1], "--k8")) {
			cpu = CPU_K8;
			av++;
		} else if (!strcmp(av[1], "--generic")) { 
			cpu = CPU_GENERIC;
			av++;
		}
	}		
	
	if (av[1] && !strcmp(av[1], "--ascii")) { 
		av++;
		if (av[1]) 
			usage();
		decodefatal(stdin); 
		exit(0);
	} 

	if (av[1]) {
		logfn = av[1];
		av++;
	}
	if (av[1])
		usage();
		
	int fd = open(logfn, O_RDONLY); 
	if (fd < 0) {
		fprintf(stderr, "Cannot open %s\n", logfn); 
		exit(1);
	}
	
	if (ioctl(fd, MCE_GET_RECORD_LEN, &recordlen) < 0)
		err("MCE_GET_RECORD_LEN");
	if (ioctl(fd, MCE_GET_LOG_LEN, &loglen) < 0)
		err("MCE_GET_LOG_LEN");

	if (recordlen > sizeof(struct mce))
		fprintf(stderr, 
    "mcelog: warning: record length longer than expected. Consider update.\n");

	char *buf = calloc(recordlen, loglen); 
	if (!buf) 
		exit(100);
	
	int len = read(fd, buf, recordlen * loglen); 
	if (len < 0) 
		err("read"); 

	int i; 
	for (i = 0; i < len / recordlen; i++) { 
		struct mce *mce = (struct mce *)(buf + i*recordlen);
		printf("MCE %d\n", i); 
		dump_mce(mce); 
	}

	if (recordlen < sizeof(struct mce))  {
		fprintf(stderr, 
			"mcelog: warning: %lu bytes ignored in each record\n",
				(unsigned long)recordlen - sizeof(struct mce)); 
		fprintf(stderr, "mcelog: consider an update\n"); 
	}	

	exit(0); 
} 
