/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_TASK_H_
#define MCLOUD_API_TASK_H_

#include <mcloud/api/visibility.h>
#include <functional>
#include <string>

namespace mcloud {
namespace api {

/*!
    \class Task
    \brief Task is an abstract class that defines common interfaces for upload and download task.
 */

class MCLOUD_API_DLL_PUBLIC Task {
public:
    /*!
     * \brief The Status enum indicates current status of this task.
     */
    enum class Status {
        Unstart,        ///< Task is in queue and sync manager does not run this task yet.
        Running,        ///< Task is being processed by sync manager.
        Canceled,       ///< Task is in canceled status, it is caused by the call from Client::cancel_sync().
        Paused,         ///< Task is in paused status, it is caused by the call from Client::pause_sync().
        Broken,         ///< Task is broken when sync manager failed to download/upload a task after several times trying.
        Complete,       ///< Task is completed.
    };

    typedef std::function<void(float)> ProgressHandler;

    typedef std::function<void(Status)> StatusHandler;

    typedef std::function<size_t(void *dest, size_t buf_size)> Buffer_Callback;

    virtual ~Task() = default;

    /*!
     * \brief Returns an unique id of task.
     */
    virtual const std::string & task_id() const = 0;

    /*!
     * \brief Returns an unique id of cloud content on mcloud.
     */
    virtual const std::string & content_id() const = 0;

    /*!
     * \brief Returns a display name of cloud content on mcloud.
     */
    virtual const std::string & content_name() const = 0;

    /*!
     * \brief Returns cloud content local storage file path.
     */
    virtual const std::string & file_path() const = 0;

    /*!
     * \brief Contains the error string if an error occurs during task running.
     */
    virtual const std::string & error_string() const = 0;

    /*!
     * \brief Returns a item url assigned by mcloud.
     * \note the url will be expried after a period of time.
     */
    virtual const std::string & task_url() const = 0;

    /*!
     * \brief Returns current download or upload item sync-up status.
     */
    virtual Task::Status status() const = 0;

    /*!
     * \brief Handler for download or upload progress of a task.
     * \sa Task::ProgressHandler
     */
    virtual Task::ProgressHandler & progress_changed() = 0;

    /*!
     * \brief Handler for download or upload progress of a task.
     * \sa Task::StatusHandler
     */
    virtual Task::StatusHandler & status_changed() = 0;

    /*!
     * \brief Cancels the task.
     * \sa Task::StatusHandler
     */
    virtual void cancel() = 0;
};

}
}

#endif // MCLOUD_API_TASK_H_
