\name{MCMCirtHier1d}
\alias{MCMCirtHier1d}
\title{Markov Chain Monte Carlo for Hierarchical One Dimensional Item Response Theory
   Model, Covariates Predicting Latent Ideal Point (Ability)}
\description{
  This function generates a sample from the posterior distribution of a one
  dimensional item response theory (IRT) model, with
  multivariate Normal priors on the item parameters, and a
  Normal-Inverse Gamma hierarchical prior on subject ideal points (abilities).  The user
  supplies item-response data, subject covariates, and priors. Note that
  this identification strategy obviates the constraints used on
  theta in \code{\link[MCMCpack]{MCMCirt1d}}. A sample from the posterior
  distribution is returned as an mcmc object, which can be
  subsequently analyzed with functions provided in the coda
  package.

  If you are interested in fitting  K-dimensional item response theory
  models, or would rather identify the model by placing constraints
  on the item parameters, please see \code{\link[MCMCpack]{MCMCirtKd}}.
  }

\usage{
MCMCirtHier1d(datamatrix, Xjdata,
           burnin = 1000, mcmc = 20000, thin=1,
           verbose = 0, seed = NA,
           theta.start = NA, a.start = NA, b.start = NA,
           beta.start=NA, b0=0, B0=.01, c0=.001, d0=.001,
           ab0=0, AB0=.25, store.item = FALSE, store.ability=TRUE,
           drop.constant.items=TRUE,
           marginal.likelihood=c("none","Chib95"),
	   px=TRUE,px_a0 = 10, px_b0=10,
           ... )  }

\arguments{
    \item{datamatrix}{The matrix of data.  Must be 0, 1, or missing values.  
     The rows of \code{datamatrix} correspond to subjects and the
     columns correspond to items.}

   \item{Xjdata}{A \code{data.frame} containing second-level predictor covariates for
     ideal points \eqn{\theta}{theta}. Predictors are modeled as a
     linear regression on the mean vector of
     \eqn{\theta}{theta}; the posterior sample contains regression
     coefficients \eqn{\beta}{beta} and common variance
     \eqn{\sigma^2}{sigma^2}. See Rivers (2003) for a
    thorough discussion of identification of IRT models.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Gibbs iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    Gibbs iterations must be divisible by this value.}

    \item{verbose}{A switch which determines whether or not the progress of
      the sampler is printed to the screen. If \code{verbose} is greater
      than 0 then every \code{verbose}th iteration will be printed to the
      screen.}
    
    \item{seed}{The seed for the random number generator.  If NA, the Mersenne
    Twister generator is used with default seed 12345; if an integer is 
    passed it is used to seed the Mersenne twister.  The user can also
    pass a list of length two to use the L'Ecuyer random number generator,
    which is suitable for parallel computation.  The first element of the
    list is the L'Ecuyer seed, which is a vector of length six or NA (if NA 
    a default seed of \code{rep(12345,6)} is used).  The second element of 
    list is a positive substream number. See the MCMCpack 
    specification for more details.}

    \item{theta.start}{The starting values for the subject
    abilities (ideal points). This can either be a scalar or a
    column vector with dimension equal to the number of voters.  
    If this takes a scalar value, then that value will serve as
    the starting value for all of the thetas.  The default value
    of NA will choose the starting values based on an
    eigenvalue-eigenvector decomposition of the agreement score matrix
    formed from the \code{datamatrix}.}

    \item{a.start}{The starting values for the
    \eqn{a}{a} difficulty parameters. This can either be
    a scalar or a column vector with dimension equal to the
    number of items.   If this takes a scalar value, then that
    value will serve as the starting value for all \eqn{a}{a}.
     The default value of NA will set the starting values based on
     a series of probit regressions that condition on the starting
     values of theta.}

    \item{b.start}{The starting values for the
    \eqn{b}{b} discrimination parameters. This can either
    be a scalar or a column vector with dimension equal to the
    number of items.   If this takes a scalar value, then that
    value will serve as the starting value for all \eqn{b}{b}. 
    The default value of NA will set the starting values based on a
    series of probit regressions that condition on the starting values
    of theta.}
    
  
  \item{beta.start}{The starting values for the \eqn{\beta}{beta}
    regression coefficients that predict the means of ideal points
    \eqn{\theta}{theta}. This can either
    be a scalar or a column vector with length equal to the
    number of covariates. If this takes a scalar value, then that
    value will serve as the starting value for all of the betas. 
    The default value of NA will set the starting values based on a
    linear regression of the covariates on (either provided or
    generated) \code{theta.start}.
  }


  \item{b0}{The prior mean of \eqn{\beta}{beta}. Can be either a scalar or a
  vector of length equal to the number of subject covariates. If a
  scalar all means with be set to the passed value.}

\item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a 
    scalar or a square matrix with dimensions equal to the number of betas.  
    If this  takes a scalar value, then that value times an identity matrix serves
    as the prior precision of beta. A default proper but diffuse value
    of .01 ensures finite marginal likelihood for model comparison.
    A value of 0 is equivalent to an improper uniform prior for beta.}
  
  \item{c0}{\eqn{c_0/2}{c0/2} is the shape parameter for the inverse
    Gamma prior on \eqn{\sigma^2}{sigma^2} (the variance of \eqn{\theta}{theta}). The amount of information in the inverse Gamma prior
    is something like that from \eqn{c_0}{c0} pseudo-observations.} 
  
  \item{d0}{\eqn{d_0/2}{d0/2} is the scale parameter for the
    inverse Gamma prior on \eqn{\sigma^2}{sigma^2} (the variance of 
    \eqn{\theta}{theta}). In constructing the inverse Gamma prior,
    \eqn{d_0}{d0} acts like the sum of squared errors from the
    \eqn{c_0}{c0} pseudo-observations.}
  
  
  \item{ab0}{The prior mean of \code{(a, b)}. Can be either a
    scalar or a 2-vector. If a scalar both means will be set to the
    passed value. The prior mean is assumed to be the same across all
    items.} 
  
  \item{AB0}{The prior precision of \code{(a, b)}.This can
    either be ascalar or a 2 by 2 matrix. If this takes a scalar
    value, then that value times an identity matrix serves as the
    prior precision. The prior precision is assumed to be the same
    across all items.}
  
  \item{store.item}{A switch that determines whether or not to
    store the item parameters for posterior analysis. 
    \emph{NOTE: In situations with many items storing the item
      parameters takes an enormous amount of memory, so
      \code{store.item} should only be \code{TRUE} if the chain is thinned
      heavily, or for applications with a small number of items}.
    By default, the item parameters are not stored.}
  
  \item{store.ability}{A switch that determines whether or not to
    store the ability parameters for posterior analysis. 
    \emph{NOTE: In situations with many individuals storing the ability
      parameters takes an enormous amount of memory, so
      \code{store.ability} should only be \code{TRUE} if the chain is thinned
      heavily, or for applications with a small number of individuals}.
    By default, ability parameters are stored.}
  
  \item{drop.constant.items}{A switch that determines whether or not
    items that have no variation
    should be deleted before fitting the model. Default = TRUE.}  
  
  \item{marginal.likelihood}{Should the marginal likelihood of the
  second-level model on ideal points be calculated using the method of
  Chib (1995)? It is stored as an attribute of the posterior \code{mcmc}
  object and suitable for comparison using \code{\link[MCMCpack]{BayesFactor}}.}
  
  \item{px}{Use Parameter Expansion to reduce autocorrelation in the chain? 
  PX introduces an unidentified parameter \eqn{alpha} for the residual
  variance in the latent data (Liu and Wu 1999). Default = TRUE
}
  \item{px_a0}{Prior shape parameter for the inverse-gamma distribution on \eqn{alpha}, the residual variance of the latent data. Default=10.}

\item{px_b0}{ Prior scale parameter for the inverse-gamma distribution on \eqn{alpha}, the residual variance of the latent data. Default = 10 }

  \item{...}{further arguments to be passed}       
}

\value{
  An \code{mcmc} object that contains the sample from the posterior
  distribution. This object can be summarized by functions provided by
  the coda package. If \code{marginal.likelihood = "Chib95"} the object will have attribute \code{logmarglike}.
}

\details{
  \code{MCMCirtHier1d} simulates from the posterior distribution using
  standard Gibbs sampling using data augmentation (a Normal draw
  for the subject abilities, a multivariate Normal draw for
  (second-level) subject
  ability predictors, an Inverse-Gamma draw for the (second-level)
  variance of subject abilities, a multivariate Normal
  draw for the item parameters, and a truncated Normal draw for
  the latent utilities). The simulation proper is done in
  compiled C++ code to maximize efficiency.  Please consult the
  coda documentation for a comprehensive list of functions that
  can be used to analyze the posterior sample.
  
  The model takes the following form.  We assume that each
  subject has an subject ability (ideal point) denoted
  \eqn{\theta_j}{theta_j} and that each item has a difficulty
  parameter \eqn{a_i}{a_i} and discrimination parameter
  \eqn{b_i}{b_i}.  The observed choice by subject
  \eqn{j}{j} on item \eqn{i}{i} is the observed data matrix which
  is \eqn{(I \times J)}{(I * J)}.  We assume that the choice is
  dictated by an unobserved utility: \deqn{z_{i,j} = -\alpha_i +
  \beta_i \theta_j + \varepsilon_{i,j}}{z_ij = -a_i +
  b_i*theta_j + epsilon_ij} Where the errors are assumed to be
distributed standard Normal. This constitutes the measurement or level-1
  model. The subject abilities (ideal points) are modeled by a second
  level Normal linear predictor for subject covariates \code{Xjdata}, with
  common variance \eqn{\sigma^2}{sigma^2}.
The parameters of interest are
  the subject abilities (ideal points), item parameters, and
  second-level coefficients.

  We assume the following priors.  For the subject abilities (ideal points):
  \deqn{\theta_j \sim \mathcal{N}(\mu_{\theta}  ,T_{0}^{-1})}{theta_j ~ N(t0, T0^{-1})}
  For the item parameters, the prior is:
  \deqn{\left[a_i, b_i \right]' \sim \mathcal{N}_2 
  (ab_{0},AB_{0}^{-1})}{[alpha_i beta_i]' ~ N_2 (ab0, AB0^{-1})}
   
  The model is identified by the proper priors on the item parameters
  and constraints placed on the ability parameters.
  
    As is the case with all measurement models, make sure that you have plenty
  of free memory, especially when storing the item parameters.  
}
  \author{Michael Malecki, \email{mike@crunch.io}, \url{https://github.com/malecki}.}

\references{
   James H. Albert. 1992. ``Bayesian Estimation of Normal Ogive Item Response 
   Curves Using Gibbs Sampling." \emph{Journal of Educational Statistics}.  
   17: 251--269.
   
   Joshua Clinton, Simon Jackman, and Douglas Rivers. 2004. ``The Statistical 
   Analysis of Roll Call Data."  \emph{American Political Science Review}
   98: 355--370.
   
   Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
   Springer: New York.

   Liu, Jun S. and Ying Nian Wu. 1999. ``Parameter Expansion for Data Augmentation.'' \emph{Journal of the American Statistical Association} 94: 1264--1274.

 Andrew D. Martin, Kevin M. Quinn, and Jong Hee Park. 2011.
  ``MCMCpack: Markov Chain Monte Carlo in R.'',
  \emph{Journal of Statistical Software}. 42(9): 1-21.
  \url{http://www.jstatsoft.org/v42/i09/}.

   Daniel Pemstein, Kevin M. Quinn, and Andrew D. Martin.  2007.  
   \emph{Scythe Statistical Library 1.0.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2006.
  ``Output Analysis and Diagnostics for MCMC (CODA)'',
   \emph{R News}. 6(1): 7-11.
  \url{http://CRAN.R-project.org/doc/Rnews/Rnews_2006-1.pdf}.
  
   Douglas Rivers.  2004.  ``Identification of Multidimensional Item-Response
   Models."  Stanford University, typescript.
}


\examples{
   \dontrun{
data(SupremeCourt)

Xjdata <- data.frame(presparty= c(1,1,0,1,1,1,1,0,0),
                     sex= c(0,0,1,0,0,0,0,1,0))

## Parameter Expansion reduces autocorrelation.
  posterior1 <- MCMCirtHier1d(t(SupremeCourt),
                   burnin=50000, mcmc=10000, thin=20, 
                   verbose=10000,
                   Xjdata=Xjdata,
                   marginal.likelihood="Chib95",
		   px=TRUE)

## But, you can always turn it off.
  posterior2 <- MCMCirtHier1d(t(SupremeCourt),
                   burnin=50000, mcmc=10000, thin=20, 
                   verbose=10000,
                   Xjdata=Xjdata,
                   #marginal.likelihood="Chib95",
		   px=FALSE)
## Note that the hierarchical model has greater autocorrelation than 
## the naive IRT model.
  posterior0 <- MCMCirt1d(t(SupremeCourt),
                        theta.constraints=list(Scalia="+", Ginsburg="-"),
                        B0.alpha=.2, B0.beta=.2,
                        burnin=50000, mcmc=100000, thin=100, verbose=10000,
                        store.item=FALSE)

## Randomly 10% Missing -- this affects the expansion parameter, increasing
## the variance of the (unidentified) latent parameter alpha.

   scMiss <- SupremeCourt
   scMiss[matrix(as.logical(rbinom(nrow(SupremeCourt)*ncol(SupremeCourt), 1, .1)), 
      dim(SupremeCourt))] <- NA

   posterior1.miss <- MCMCirtHier1d(t(scMiss),
                   burnin=80000, mcmc=10000, thin=20, 
                   verbose=10000,
                   Xjdata=Xjdata,
                   marginal.likelihood="Chib95",
		   px=TRUE)

   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},
\code{\link[MCMCpack]{MCMCirtKd}}}

