/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#ifndef __EPG_H__
#define __EPG_H__

#include <libgnome/libgnome.h>
#include "xml.hh"
#include "dvb_si.hh"
#include "date_time.hh"
#include "integer.hh"
#include "log.hh"
#include "channel.hh"

#define UNKNOWN_TEXT _("Unknown")

class EpgEvent
{
private:
	glong			event_id;
	glong			start_time;
	glong			duration;
	String	start_time_text;
	String	duration_text;
	String	title;
	String	description;
	const Channel*	channel;
	
	void fix_data();

public:
	EpgEvent();
	EpgEvent(const Channel* channel, const Event& event);
	EpgEvent(const EpgEvent& event);
	EpgEvent(const XmlNode& node);

	static String start_time_to_string(int start_time);
	static String duration_to_string(int duration);
	static gint compare_events (xmlNodePtr a, xmlNodePtr b)
	{
		XmlNode event_a(a);
		XmlNode event_b(b);
		int start_time_a = event_a.get_int_attribute_value("start_time");
		int start_time_b = event_b.get_int_attribute_value("start_time");
		return start_time_a - start_time_b;
	}

	void load(const XmlNode& node);
	
	glong get_event_id()						const { return event_id; }
	glong get_start_time()						const { return start_time; }
	glong get_duration()						const { return duration; }
	const String& get_start_time_text()	const { return start_time_text; }
	const String& get_duration_text()	const { return duration_text; }
	const String& get_title()			const { return title; }
	const String& get_description()		const { return description; }
	const Channel& get_channel() const;

	gboolean operator==(const EpgEvent& e);
};

class Epg
{	
private:
	XmlDocument document;
	xmlNodePtr get_channel_node(const String& channel_name);
	xmlNodePtr get_channel_node(const Channel& channel);
	xmlNodePtr get_transponder_node(Transponder& transponder);

public:	
	void load(const String& path);
	int add_event(const EpgEvent& event);
	int prune();
	void save();
	
	xmlNodePtr get_current_event(const Channel& channel);
	xmlNodePtr get_event_at(const Channel& channel, glong when);
	xmlNodePtr get_event(const Channel& channel, int event_id);
	xmlNodePtr get_event(const String& channel_name, int event_id);
	xmlNodePtr get_event(int event_id);
	GList* search(const String& query);
	xmlXPathObjectPtr get_events(const Channel& channel);
};

#endif
