/* Mednafen - Multi-system Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "mapinc.h"

static uint8 WRAM[8192];
static void AYSoundHQ(void);
static void DoAYSQHQ(int x);

static uint16 IRQCount;
static uint8 IRQa;

static uint8 reg_select;
static uint8 wram_control;
static uint8 sr[0x10];
static uint8 sr_index;


static DECLFW(SUN5BWRAM)
{
 if((wram_control&0xC0)==0xC0)
  (WRAM-0x6000)[A]=V;
}

static DECLFR(SUN5AWRAM)
{
 if((wram_control&0xC0)==0x40)
  return X.DB; 
 return CartBR(A);
}

static DECLFW(Mapper69_SWL)
{
  sr_index= V & 0xF;
}

static DECLFW(Mapper69_SWH)
{
	     int x;

	     if(FSettings.SndRate)
             switch(sr_index)
             {
              case 0:
              case 1:
              case 8:DoAYSQHQ(0);break;
              case 2:
              case 3:
              case 9:DoAYSQHQ(1);;break;
              case 4:
              case 5:
              case 10:DoAYSQHQ(2);break;
              case 7:
		     for(x=0;x<2;x++)
  		      DoAYSQHQ(x); 
		     break;
             }
             sr[sr_index]=V; 
}

static DECLFW(Mapper69_write)
{
 switch(A&0xE000)
 {
  case 0x8000:reg_select=V;break;
  case 0xa000:
              reg_select&=0xF;
              if(reg_select < 8)
               setchr1(reg_select<<10,V);
              else
               switch(reg_select&0x0f)
               {
                case 8:
                       wram_control = V;
                       if(V&0x40)
                        {
                         if(V&0x80) // Select WRAM
                          setprg8r(0x10,0x6000,0);
                        }
                        else
                         setprg8(0x6000,V);
                        break;
                case 9:setprg8(0x8000,V);break;
                case 0xa:setprg8(0xa000,V);break;
                case 0xb:setprg8(0xc000,V);break;
                case 0xc:
                         switch(V&3)
                         {
                          case 0:setmirror(MI_V);break;
                          case 1:setmirror(MI_H);break;
                          case 2:setmirror(MI_0);break;
                          case 3:setmirror(MI_1);break;
                         }
                         break;
             case 0xd:IRQa=V; if(!(V&0x80)) X6502_IRQEnd(MDFN_IQEXT); break;
             case 0xe:IRQCount&=0xFF00;IRQCount|=V;break;
             case 0xf:IRQCount&=0x00FF;IRQCount|=V<<8;break;
             }
             break;
 }
}

static int32 vcount[3];
static int32 dcount[3];
static uint32 CAYBC[3];

static void DoAYSQHQ(int x) 
{
 uint32 V;
 int32 freq=((sr[x<<1]|((sr[(x<<1)+1]&15)<<8))+1)<<4;
 int32 amp=(sr[0x8+x]&15)<<6;

 amp+=amp>>1;

 if(!(sr[0x7]&(1<<x)))
 {
  for(V=CAYBC[x];V<SOUNDTS;V++)
  {
   if(dcount[x])
    WaveHiEx[V]+=amp;
   vcount[x]--;
   if(vcount[x]<=0)
   {
    dcount[x]^=1;
    vcount[x]=freq;
   }
  } 
 }
 CAYBC[x]=SOUNDTS;
}

static void AYSoundHQ(void)
{
    DoAYSQHQ(0);
    DoAYSQHQ(1);
    DoAYSQHQ(2);
}

static void AYHiSync(int32 ts)
{
 int x;

 for(x=0;x<3;x++)
  CAYBC[x]=ts;
}

static void SunIRQHook(int a)
{
  if(IRQa & 1)
  {
   uint16 prev = IRQCount;

   IRQCount-=a;

   if(IRQCount > prev && (IRQa & 0x80))
   {
    X6502_IRQBegin(MDFN_IQEXT);
   }
  }
}

static int StateAction(StateMem *sm, int load, int data_only)
{
 SFORMAT StateRegs[] = {
        SFVARN(reg_select, "FM7S"),
        SFVARN(wram_control, "FM7W"),
        SFARRAYN(sr, 0x10, "FM7S"),
        SFVARN(sr_index, "FM7I"),
	SFEND
 };

 int ret = MDFNSS_StateAction(sm, load, data_only, StateRegs, "MAPR");

 if(load)
 {
   if(wram_control&0x40)
   {
    if(wram_control&0x80) // Select WRAM
     setprg8r(0x10,0x6000,0);
   }
   else
    setprg8(0x6000,wram_control);
 }

 return(ret);
}

void Mapper69_ESI(EXPSOUND *ep)
{
 ep->HiFill=AYSoundHQ;
 ep->HiSync=AYHiSync;
 memset(dcount,0,sizeof(dcount));
 memset(vcount,0,sizeof(vcount));
 memset(CAYBC,0,sizeof(CAYBC));
}

int NSFAY_Init(EXPSOUND *ep)
{
 sr_index=0;
 SetWriteHandler(0xc000,0xdfff,Mapper69_SWL);
 SetWriteHandler(0xe000,0xffff,Mapper69_SWH);
 Mapper69_ESI(ep);
 return(1);
}

static void Reset(CartInfo *info)
{
 sr_index = 0;
 wram_control = 0;
 memset(sr, 0xFF, sizeof(sr));  // Setting all bits will cause sound output to be disabled on reset.

 setprg8(0x8000, 0x3F);
 setprg8(0xa000, 0x3F);
 setprg8(0xc000, 0x3F);
 setprg8(0xe000, 0x3F);
}

int BTR_Init(CartInfo *info)
{
 SetupCartPRGMapping(0x10,WRAM,8192,1);

 SetWriteHandler(0x8000,0xbfff,Mapper69_write);
 SetWriteHandler(0xc000,0xdfff,Mapper69_SWL);
 SetWriteHandler(0xe000,0xffff,Mapper69_SWH);
 SetWriteHandler(0x6000,0x7fff,SUN5BWRAM);
 SetReadHandler(0x6000,0x7fff,SUN5AWRAM);
 SetReadHandler(0x8000, 0xFFFF, CartBR);

 info->Power = info->Reset = Reset;
 info->StateAction = StateAction;

 Mapper69_ESI(&GameExpSound);
 MapIRQHook = SunIRQHook;

 return(1);
}

