// Copyright (c) 2010, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// Shell process container and utilities.

class Process
{
	public static function runButExitOnError
	(
		command : String,
		args : Array<String>,
		? workingDirectory : String
	)
	: Iter<String>
	{
		var cwd = neko.Sys.getCwd();
		if( workingDirectory != null )
		{			
			neko.Sys.setCwd( workingDirectory );
		}
		var p = new Process( command, args );
		// Collect all output immediately to avoid the process blocking.
		var output = new List<String>();
		for( line in p.stdout() )
		{
			output.add( line );
		}
		neko.Sys.setCwd( cwd );
		if( p.process.exitCode() != 0 )
		{
			// Process failed, abort.
			Util.writeInfos( output );
			Util.writeError( command + " " + args.join(" ") + " failed:" );
			Util.writeErrors( p.stderr() );
			neko.Sys.exit( 1 );
		}
		// Process succeeded.
		return new MakeIter( output.iterator() );
	}

	public var process (default,null) : neko.io.Process;
	
	public function new( command: String, args: Array<String> )
	{
		this.command = command + " " + args.join( " " );
		process = new neko.io.Process( command, args );
	}

	public function stdout() : Iter<String>
	{
		return new ProcessOutputIterator( process.stdout );
	}
	
	public function stderr() : Iter<String>
	{
		return new ProcessOutputIterator( process.stderr );
	}

	var command (default,null) : String;	
}
