/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//*********************************************************************************************
//
// File:		MEStatusDlg.cpp
//
// Description:		Implementation for MEStatusDlg class - The GUI of the MEStatus application.
//
//**********************************************************************************************

#include "MEStatusDlg.h"
#include "IMEGuiCommands.h"
#include "ConfigFile.h"
#include <stdlib.h>
#include <string>
#include <fstream>
#include <gtk/gtk.h>
#include <glibmm/i18n.h>

using namespace std;

//ctor
MEStatusDlg::MEStatusDlg(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
  : Gtk::Window(cobject), //Base class ctor
    m_refGlade(refGlade)  //Reference to Glade Xml object
{
  m_pInitFailMsgDlg = NULL;
  m_pBrowserNotFoundDlg = NULL;
  m_pBrowserNotFoundOk = NULL;
  m_pOkButton = NULL;
  m_pLinkButtonBox = NULL;
  m_pLinkButton = NULL;
  m_pMenuPopup = NULL;
  m_pDontShowMsgChk = NULL;
  m_pRenotifyChk = NULL;
  m_pMEGuiCommands = NULL;
}

//dtor
MEStatusDlg::~MEStatusDlg()
{
  delete m_pInitFailMsgDlg;
  m_statusIcon->set_visible(false);
}

/*Init: Initialize the dialog. pMEGuiCommands is a pointer to the commands object, which must
  already be initialized. strError is a reference to a string that will hold an error message
  in case initialization fails.*/
bool MEStatusDlg::Init(IMEGuiCommands *pMEGuiCommands, Glib::ustring &strError)
{
  Gtk::Entry* pUrl;
  Glib::ustring url;
  //Reset strError
  strError = "";

  //Check parameters
  if (pMEGuiCommands == NULL)
  {
    strError = _("pMEGuiCommands is NULL");
    return false;
  }
  m_pMEGuiCommands = pMEGuiCommands;

  url = GetOEMUrl();

  /***** Initialize Widgets *****/

  //Init initialization failure message dialog
  m_pInitFailMsgDlg = new Gtk::MessageDialog::MessageDialog(
    *this, 
    _("The UNS daemon could not be contacted. Please make sure it is running."),
    false,
    Gtk::MESSAGE_WARNING);

  if (!m_pInitFailMsgDlg)
  {
    strError = _("Failed to create 'Init failure' message dialog.");
    return false;
  }
  m_pInitFailMsgDlg->set_title(_("Intel(R) AMT Status:"));
  g_signal_connect(m_pInitFailMsgDlg->gobj(), "response", GTK_SIGNAL_FUNC(OnInitFailOkButton), this);

  //Init browser not found dialog
  m_refGlade->get_widget("BrowserNotFoundDlg", m_pBrowserNotFoundDlg);
  if (!m_pBrowserNotFoundDlg)
  {
    strError = _("Couldn't get Browser Not Found dialog");
    return false;
  }

  //Init browser not found dialog OK button
  m_refGlade->get_widget("BrowserNotFoundOk", m_pBrowserNotFoundOk);
  if (!m_pBrowserNotFoundOk)
  {
    strError = _("Couldn't get Browser Not Found dialog ok button");
    return false;
  }
  m_pBrowserNotFoundOk->signal_clicked().connect(sigc::mem_fun(*this, &MEStatusDlg::OnBrowserNotFoundOkButton));

  //Init AMT Status label
  m_refGlade->get_widget("lblAMTStatus", m_pAMTStatusLabel);
  if (!m_pAMTStatusLabel)
  {
    strError = _("Couldn't get AMT status label");
    return false;
  }

  //Init OK button
  m_refGlade->get_widget("btnOk", m_pOkButton);
  if (!m_pOkButton)
  {
    strError = _("Couldn't get OK button");
    return false;
  }
  m_pOkButton->signal_clicked().connect(sigc::mem_fun(*this, &MEStatusDlg::OnOkButton));

  //Init "Don't show this message" checkbox
  m_refGlade->get_widget("chkDontShowMsg", m_pDontShowMsgChk);
  if (!m_pDontShowMsgChk)
  {
    strError = _("Couldn't get chkDontShowMsg");
    return false;
  }

  //Init "Reactivate notification" checkbox
  m_refGlade->get_widget("chkRenotify", m_pRenotifyChk);
  if (!m_pRenotifyChk)
  {
    strError = _("Couldn't get chkRenotify");
    return false;
  }

  //Init "click here" link button
  //(This button is placed in a button box that comes from the glade file).
  m_refGlade->get_widget("LinkBtnBox", m_pLinkButtonBox);
  if (!m_pLinkButtonBox)
  {
    strError = _("Couldn't get link button box");
    return false;
  }

  m_refGlade->get_widget("BrowserNotFoundUrl", pUrl);
  if (!pUrl) {
    strError = _("Couldn't get browser not found url");
    return false;
  }
  if (url == GetDefaultOEMUrl()) {
    url = pUrl->get_text();
  } else {
    pUrl->set_text(url);
  }
  
  m_pLinkButton = new Gtk::LinkButton(url, _("click here."));
  if (!m_pLinkButton)
  {
    strError = _("Couldn't create link button");
    return false;
  }
  m_pLinkButton->signal_clicked().connect(sigc::mem_fun(*this, &MEStatusDlg::OnLinkButton));
  m_pLinkButtonBox->add(*m_pLinkButton);

  //Load icons for status icon from files
  try
  {
    m_iconOn = Gdk::Pixbuf::create_from_file("res/ATChk-on.ico");
    m_iconOff = Gdk::Pixbuf::create_from_file("res/ATChk-off.ico");
    m_iconUnknown = Gdk::Pixbuf::create_from_file("res/ATChk-unk.ico");
  }
  catch (Glib::FileError &)
  {
    strError = _("Couldn't load icon file");
    return false;
  }
  
  //Initialize status icon
  m_statusIcon = Gtk::StatusIcon::create(m_iconOn);
  if (!m_statusIcon)
  {
    strError = _("Error creating status icon");
    return false;
  }
  /*I had to connect the signal handlers for the status icon the ugly way with GTK+, since
    gtkmm doesn't provide a way to do it.*/
  g_signal_connect(m_statusIcon->gobj(), "popup-menu", GTK_SIGNAL_FUNC(OnStatusIconPopupMenu), this);
  g_signal_connect(m_statusIcon->gobj(), "activate", GTK_SIGNAL_FUNC(OnStatusIconClicked), this);

  //Create actions for popup menu
  m_refActionGroup = Gtk::ActionGroup::create();
  m_refActionGroup->add(Gtk::Action::create("ShowStatus", _("Show Status")),
			sigc::mem_fun(*this, &MEStatusDlg::OnMenuShowStatus));
  m_refActionGroup->add(Gtk::Action::create("Exit", _("Exit")),
			 sigc::mem_fun(*this, &MEStatusDlg::OnMenuExit));

  //Create UI Manager and connect it to action group
  m_refUIManager = Gtk::UIManager::create();
  m_refUIManager->insert_action_group(m_refActionGroup);

  //Layout the actions in menu:
  Glib::ustring ui_info = 
    "<ui>"
    "  <popup name='PopupMenu'>"
    "    <menuitem action='ShowStatus'/>"
    "    <menuitem action='Exit'/>"
    "  </popup>"
    "</ui>";
  try
  {      
    m_refUIManager->add_ui_from_string(ui_info);
  }
  catch(const Glib::Error& ex)
  {
    strError = _("Building menus failed: ") + ex.what();
  }

  //Initialize popup menu
  m_pMenuPopup = dynamic_cast<Gtk::Menu*>( m_refUIManager->get_widget("/PopupMenu") ); 
  if (!m_pMenuPopup)
  {
    strError = _("Menu not found.");
    return false;
  }

  //All done
  return true;
}

/*SetEnablement sets the enablement state in the GUI: In the staus icon, and in the status label.*/
void MEStatusDlg::SetEnablement(IPCAMTStatus::Enablement enablement)
{
  m_pMEGuiCommands->SetEnablement(enablement);
}

//Overriden from Gtk::Widget
void MEStatusDlg::on_show()
{
  if (GetPopupForCurrentUser())
  {
    m_pDontShowMsgChk->show();
    m_pRenotifyChk->hide();
  }
  else
  {
    m_pDontShowMsgChk->hide();
    m_pRenotifyChk->show();
  }
  m_pLinkButtonBox->show_all();

  Gtk::Window::on_show();
}

//Notification of enablement change from IMEGuiNotifications
void MEStatusDlg::OnAMTChangeEnablement(IPCAMTStatus::Enablement enablement)
{
  switch (enablement)
  {
    case IPCAMTStatus::STATUS_DISABLED:
      m_statusIcon->set(m_iconOff);
      m_pAMTStatusLabel->set_label(_("Disabled."));
      break;
    case IPCAMTStatus::STATUS_ENABLED:
      m_statusIcon->set(m_iconOn);
      m_pAMTStatusLabel->set_label(_("Enabled."));
      break;
  case IPCAMTStatus:: STATUS_UNKNOWN:
      m_statusIcon->set(m_iconUnknown);
      m_pAMTStatusLabel->set_label(_("Unknown."));
      break;
  }
}

//Notification from IMEGuiNotifications that we should now display the popup
void MEStatusDlg::OnAMTEnablementPopup(IPCAMTStatus::Enablement enablement)
{
  present();
  if (enablement == IPCAMTStatus::STATUS_UNKNOWN)
  {
    m_pInitFailMsgDlg->run();
  }
}

//Main window Ok button was clicked
void MEStatusDlg::OnOkButton()
{
  if (m_pDontShowMsgChk->get_active())
  {
    SetPopupForCurrentUser(false);
  }
  else if (m_pRenotifyChk->get_active())
  {
    SetPopupForCurrentUser(true);
  }
  m_pDontShowMsgChk->set_active(false);
  m_pRenotifyChk->set_active(false);

  hide();
}

//"Browser not found" message dialog's Ok button was clicked
void MEStatusDlg::OnBrowserNotFoundOkButton()
{
  m_pBrowserNotFoundDlg->hide();
}

//The "click here" link button was clicked
void MEStatusDlg::OnLinkButton()
{
  bool bBrowserConfigured = false;
  std::string browser;

  ifstream infile("~/.IntelAMTCheck.browser.conf");
  if (infile.is_open())
  {
    ConfigFile browserConfig;
    infile >> browserConfig;
    if (browserConfig.readInto(browser, "browser"))
      bBrowserConfigured = true;
  }

  static const char* browsers[] = {"firefox", "opera", "konqueror", "epiphany", "mozilla", "netscape"};
  static const int num_browsers = sizeof(browsers) / sizeof(browsers[0]);

  int i = 0;
  bool bInvokeSuccess = false;
  if (!bBrowserConfigured)
    browser = browsers[i++];

  do {
    bInvokeSuccess = InvokeBrowser(browser, m_pLinkButton->get_uri());
    browser = browsers[i++];
  } while ((!bInvokeSuccess) && (i < num_browsers));

  if (!bInvokeSuccess)
    m_pBrowserNotFoundDlg->show();
}

//The "show status" option from the popup menu was selected
void MEStatusDlg::OnMenuShowStatus()
{
  //Show window and bring it to front.
  present();
}

//The "exit" option from the popup menu was selected
void MEStatusDlg::OnMenuExit()
{
  Gtk::Main::quit();
}

//Init fail dialog's Ok button was clicked
void MEStatusDlg::OnInitFailOkButton(GtkDialog *dialog,
				     gint       arg1,
				     gpointer   user_data)
{
  MEStatusDlg* pDlg = (MEStatusDlg*)user_data;
  if (pDlg)
    pDlg->m_pInitFailMsgDlg->hide();
}

//The status icon was clicked
void MEStatusDlg::OnStatusIconClicked(GtkStatusIcon* icon, gpointer func_data)
{
  MEStatusDlg* pDlg = (MEStatusDlg*)func_data;
  if (pDlg)
    pDlg->present();
}

//The status icon was right-clicked
void MEStatusDlg::OnStatusIconPopupMenu(GtkStatusIcon *status_icon,
					guint          button,
					guint          activate_time,
					gpointer       user_data)
{
  MEStatusDlg* pDlg = (MEStatusDlg*)user_data;
  if (pDlg && pDlg->m_pMenuPopup)
  {
	pDlg->m_pMenuPopup->popup(button, activate_time);
  }
}

//Get the popup preference for the current user
bool MEStatusDlg::GetPopupForCurrentUser() const
{
  return m_pMEGuiCommands->GetPopupForCurrentUser();
}

//Set the popup preference for the current user
void MEStatusDlg::SetPopupForCurrentUser(bool bPopup)
{
  m_pMEGuiCommands->SetPopupForCurrentUser(bPopup);
}

//Get the OEM configured Url for the "click here" link button
std::string MEStatusDlg::GetOEMUrl() const
{
  return m_pMEGuiCommands->GetOEMUrl();
}

//Get default OEM Url for the "click here" link button
std::string MEStatusDlg::GetDefaultOEMUrl() const
{
  return m_pMEGuiCommands->GetDefaultOEMUrl();
}

//Invoke specified web browser with specified url. 
//Returns true on success, false otherwise.
bool MEStatusDlg::InvokeBrowser(const std::string &browser, const std::string &url)
{
  if (Glib::find_program_in_path(browser) == "")
  {
    //browser was not found in path - invoke failed.
    return false;
  }

  system((browser + " " + url + " &").c_str());
  return true;
}
