/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#ifndef _IPC_AMT_STATUS_H
#define _IPC_AMT_STATUS_H

#ifdef _WIN32
#include <windows.h>
#endif

#include "IPCBuffer.h"


//*****************************************************************************
//
// Class:			IPCAMTStatus
// Description:	    IPCAMTStatus is an object that holds the status of AMT, and
//                  this status can be shared across processes. When creating the
//                  object, you must specify whether you are the server or client
//                  of this status - being the server means you create the status
//                  and only write to it, and being a client means you connect
//                  to an existing status and only read from it.
//                  After initialization, you should use isOk() and lastError()
//                  to check the error status of the object.
//*****************************************************************************
class IPCAMTStatus
{
public:
	//\En*a"ble*ment\, n. The act of enabling, or the state of being enabled; ability. --Bacon
	enum Enablement {STATUS_DISABLED, STATUS_ENABLED, STATUS_UNKNOWN};

	//default ctor: must use init() afterwards
	IPCAMTStatus();

	//ctor
	IPCAMTStatus(bool bServer);

	//initialize this object after it was created
	bool init(bool bServer);

	//release resources
	void close();

	//get error code of last error
	unsigned long lastError() const;

	//is this IPCAMTStatus object ok (no errors)
	bool isOk() const;

#ifndef _WIN32
	//enable/disable connecting to shm before each shared memory access
	void setShmReconnect(bool reconnect);
#endif

	//was this object properly written to shared memory, in case it's a server.
	//If it isn't, writeOk() returns true always.
	bool writeOk() const {return m_bWriteOk;}

	//set the enablement. returns false on failure.
	bool SetEnablement(Enablement status);

	//get the enablement. returns STATUS_UNKNOWN on failure.
	Enablement GetEnablement();

private:
	static const char* BUF_NAME;
	static const unsigned int BUF_SIZE;
	static const unsigned long TYPE_ENABLEMENT_ONLY;

	IPCBuffer m_IPCBuffer;
	bool m_bWriteOk;
	bool m_bServer;
#ifndef _WIN32
	bool m_shmReconnect;
#endif

	struct Status
	{
		struct Header {
			unsigned long dwType;
			unsigned long dwBodyLength;
		} header;

		union StatusBody
		{
			struct EnablementOnly
			{
				Enablement enablement;
			} enablementOnly;
		} body;
	};

	//set the enablement. returns false on failure.
	bool _SetEnablement(Enablement status);

	//get the enablement. returns STATUS_UNKNOWN on failure.
	Enablement _GetEnablement() const;

	bool writeStatus(const Status& status);
	bool readStatus(Status& status) const;
	
	bool shmInit();
};

#endif //_IPC_AMT_STATUS_H
