/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <QAction>
#include <QApplication>
#include <QCursor>
#include <QDebug>
#include <QFile>
#include <QHBoxLayout>
#include <QMenu>
#include <QMenuBar>
#include <QPushButton>
#include <QScrollArea>
#include <QSettings>
#include <QSortFilterProxyModel>
#include <QSplitter>
#include <QString>
#include <QStringList>
#include <QToolBar>
#include <QTreeView>
#include <QVBoxLayout>

#include <iostream>
#include <sstream>
#include <fstream>
#include <cstdio>
#include <stdio.h>
#include <stdlib.h>

#include "MvFileInfo.h"

#include "MvQObstatScatterToolBase.h"
#include "MvKeyProfile.h"

#include "MvQFileInfoLabel.h"
#include "MvQKeyManager.h"
#include "MvQKeyProfileModel.h"
//#include "MvQPlotWidget.h"
#include "MvQTreeView.h"

#include "mars.h"

MvQObstatScatterToolBase::MvQObstatScatterToolBase(QWidget *)
{
	//File info label        
	fileInfoLabel_=new MvQFileInfoLabel;

	//Set up grib message group box
	setupMessageBox();
	
	//Set up grib dump group box
	//setupDumpBox();
	setupPlotBox();

	//----------------------
	// Init
	//----------------------

	currentBlockNo_=-1;
	currentItemNo_=-1;

	//----------------------------
	// Signals and slots
	//----------------------------

	setupViewActions();
	setupSettingsActions();

	swapAxes_=false;

	//----------------------------
	// Signals and slots
	//----------------------------

	//Dump combo
	//connect(dumpCombo_, SIGNAL(currentIndexChanged(int)),
       //         dumpStacked_, SLOT(setCurrentIndex(int)));
	
	//Message list selection	
	connect(blockTree_,SIGNAL(clicked(QModelIndex)),
		this,SLOT(slotSelectBlock(QModelIndex)));

	connect(blockTree_,SIGNAL(activated(QModelIndex)),
		this,SLOT(slotSelectBlock(QModelIndex)));

	//Message list selection	
	connect(messageTree_,SIGNAL(clicked(QModelIndex)),
		this,SLOT(slotSelectMessage(QModelIndex)));

	connect(messageTree_,SIGNAL(activated(QModelIndex)),
		this,SLOT(slotSelectMessage(QModelIndex)));


	connect(swapPb_,SIGNAL(toggled(bool)),
		this,SLOT(slotSwapAxes(bool)));	

	/*connect(messageTree_->header(),SIGNAL(customContextMenuRequested(const QPoint &)),
                this, SLOT(slotMessageTreeContextMenu(const QPoint &)));

	connect(messageTree_->header(),SIGNAL(sectionMoved(int,int,int)),
                this, SLOT(slotMessageTreeColumnMoved(int,int,int)));*/
	
	char *mvbin=getenv("METVIEW_BIN");
	if (mvbin == 0)  
	{	
		qDebug() << "MvQObstatScatterToolBase::MvQObstatScatterToolBase() --> No METVIEW_BIN found!";
			//marslog(LOG_EXIT|LOG_EROR,"%s\n",sbuff.str().c_str());
			//setError(13,"Environment variable METVIEW_USER_DIRECTORY is not defined!");	
		return;	
	}

	magmlScript_=QString(mvbin);
	magmlScript_+="/ObstatScatter.magml";
}
 
MvQObstatScatterToolBase::~MvQObstatScatterToolBase()
{	
	if(messageModel_) delete messageModel_;
	//if(summaryModel_) delete summaryModel_;
	//if(stdDumpModel_) delete stdDumpModel_;
	//if(wmoDumpModel_) delete wmoDumpModel_;	
}

void MvQObstatScatterToolBase::setupMessageBox()
{
	QWidget *w;
	QLabel *label;
	QVBoxLayout *vb;

	//--------------------------------
	// Message tree (key profile)
	//--------------------------------

	messagePanel_=new QWidget;
	QVBoxLayout *messageLayout = new QVBoxLayout;
	messageLayout->setContentsMargins(0,0,0,0);
	messagePanel_->setLayout(messageLayout);
	//messageSplitter_->addWidget(messageWidget);	

	blockSplitter_=new QSplitter;
	blockSplitter_->setOrientation(Qt::Vertical);
	messageLayout->addWidget(blockSplitter_);
	
	w=new QWidget;
	vb=new QVBoxLayout;
	w->setLayout(vb);
	blockSplitter_->addWidget(w);

	//Label
	label=new QLabel(tr("Scatterplot blocks"));
	label->setFrameShape(QFrame::StyledPanel);
	label->setProperty("mvStyle","panel");
	vb->addWidget(label);

	//Message tree
	blockModel_= new MvQKeyProfileModel;
	blockSortModel_= new QSortFilterProxyModel;
        blockSortModel_->setSourceModel(blockModel_);
	blockSortModel_->setDynamicSortFilter(true);
	blockSortModel_->setFilterRole(Qt::UserRole);
	//messageSortModel_->setFilterRegExp(QRegExp("[1]"));
	blockSortModel_->setFilterFixedString("1");
	blockSortModel_->setFilterKeyColumn(0);


	blockTree_=new MvQTreeView;
	blockTree_->setObjectName("blockTree");
	blockTree_->setProperty("mvStyle",0);
	blockTree_->setSortingEnabled(true); 
	blockTree_->sortByColumn (0,Qt::AscendingOrder); 
	blockTree_->setAlternatingRowColors(true);
        blockTree_->setAllColumnsShowFocus(true);
	blockTree_->setModel(blockSortModel_);	
	//messageTree_->header()->setContextMenuPolicy(Qt::CustomContextMenu);
	blockTree_->setActvatedByKeyNavigation(true);
	blockTree_->setAcceptDrops(true);
	blockTree_->setDragDropMode(QAbstractItemView::DropOnly);
	blockTree_->setRootIsDecorated(false);

	vb->addWidget(blockTree_);	

	//--------------------------------
	// Message tree (key profile)
	//--------------------------------	
	
	w=new QWidget;
	vb=new QVBoxLayout;
	w->setLayout(vb);
	blockSplitter_->addWidget(w);	

	//Label
	label=new QLabel(tr("Scatterplot items"));
	label->setFrameShape(QFrame::StyledPanel);
	label->setProperty("mvStyle","panel");
	vb->addWidget(label);

	//Message tree
	messageModel_= new MvQKeyProfileModel;
	messageSortModel_= new QSortFilterProxyModel;
        messageSortModel_->setSourceModel(messageModel_);
	messageSortModel_->setDynamicSortFilter(true);
	messageSortModel_->setFilterRole(Qt::UserRole);
	//messageSortModel_->setFilterRegExp(QRegExp("[1]"));
	messageSortModel_->setFilterFixedString("1");
	messageSortModel_->setFilterKeyColumn(0);


	messageTree_=new MvQTreeView;
	messageTree_->setObjectName("messageTree");
	messageTree_->setProperty("mvStyle",0);
	messageTree_->setSortingEnabled(true); 
	messageTree_->sortByColumn (0,Qt::AscendingOrder); 
	messageTree_->setAlternatingRowColors(true);
        messageTree_->setAllColumnsShowFocus(true);
	messageTree_->setModel(messageSortModel_);	
	//messageTree_->header()->setContextMenuPolicy(Qt::CustomContextMenu);
	messageTree_->setActvatedByKeyNavigation(true);
	messageTree_->setAcceptDrops(true);
	messageTree_->setDragDropMode(QAbstractItemView::DropOnly);
	messageTree_->setRootIsDecorated(false);


	vb->addWidget(messageTree_);	
}

void MvQObstatScatterToolBase::setupPlotBox()
{
//	QWidget *w;
	QLabel *label;

	//--------------------------------
	// Message tree (key profile)
	//--------------------------------

	plotPanel_=new QWidget;
	QVBoxLayout *plotLayout = new QVBoxLayout;
	plotLayout->setContentsMargins(0,0,0,0);
	plotPanel_->setLayout(plotLayout);
	//messageSplitter_->addWidget(messageWidget);	

	//Label
	label=new QLabel(tr("Plot"));
	label->setFrameShape(QFrame::StyledPanel);
	label->setProperty("mvStyle","panel");
	plotLayout->addWidget(label);

	//Buttons
	QHBoxLayout *hb=new QHBoxLayout;
	plotLayout->addLayout(hb);

	swapPb_=new QPushButton(tr("Swap axes"));
	swapPb_->setCheckable(true);
	hb->addWidget(swapPb_);
	hb->addStretch(1);


	QScrollArea *area=new QScrollArea;
	
	//plot area
	plotWidget_= new QLabel;
	plotWidget_->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
     	plotWidget_->setScaledContents(true);

	//plotWidget_->setMinimumSize(800,700);
	//plotWidget_->setSize(800,600);
	//plotLayout->addWidget(plotWidget_);

	area->setWidget(plotWidget_);
	plotLayout->addWidget(area);
	
}

void MvQObstatScatterToolBase::setupViewActions()
{
	//
	QAction* actionFileInfo = new QAction(this);
        actionFileInfo->setObjectName(QString::fromUtf8("actionFileInfo"));
	actionFileInfo->setText(tr("File info"));
	actionFileInfo->setCheckable(true);
	actionFileInfo->setChecked(true);
	actionFileInfo->setToolTip(tr("Show/hide file info"));
	QIcon icon;
	icon.addPixmap(QPixmap(QString::fromUtf8(":/examinerBase/fileInfo.png")), QIcon::Normal, QIcon::Off);
 	actionFileInfo->setIcon(icon);	
	
	//Define routines
	connect(actionFileInfo,SIGNAL(triggered(bool)),
		fileInfoLabel_,SLOT(setVisible(bool)));


	MvQMainWindow::MenuType menuType=MvQMainWindow::ViewMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionFileInfo));	
}

void MvQObstatScatterToolBase::setupSettingsActions()
{
	//
	QAction* actionKeyManager = new QAction(this);
        actionKeyManager->setObjectName(QString::fromUtf8("actionKeyManager"));
	actionKeyManager->setText(tr("Key manager"));
	actionKeyManager->setToolTip(tr("Key profile manager"));
	QIcon icon;
	icon.addPixmap(QPixmap(QString::fromUtf8(":/examinerBase/keyManager.png")), QIcon::Normal, QIcon::Off);
 	actionKeyManager->setIcon(icon);
	actionKeyManager->setToolTip(tr("Key manager"));
  
	//Key profile management
	connect(actionKeyManager, SIGNAL(triggered()), 
		this, SLOT(slotManageKeyProfiles()));

	//Combo box
	//We will add items + signal/slots later in "init"!
	QHBoxLayout *hb=new  QHBoxLayout;

	//Combo box for key profile selection
	QLabel *keyLabel = new QLabel(tr("Key profile:"));
	keyCombo_ = new QComboBox;
	keyLabel->setBuddy(keyCombo_);

	hb->addWidget(keyLabel);
	hb->addWidget(keyCombo_);
	QWidget* wh=new QWidget;
	wh->setLayout(hb);

	MvQMainWindow::MenuType menuType=MvQMainWindow::SettingsMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionKeyManager));	
}


void MvQObstatScatterToolBase::init(MvQObstatScatterData *data)
{	
	data_=data;
	
	//----------------------------------
	// Initilaize the key profile list
	//---------------------------------
	
	MvKeyProfile *prof=new MvKeyProfile("Metview default");
	loadKeyProfile(prof);

	//disconnect(keyCombo_,SIGNAL(currentIndexChanged(int)),this,SLOT(loadKeyProfile(int)));
	
	//Set tree focus
	messageTree_->setFocus(Qt::OtherFocusReason);
	
	//Fileinfo label
	updateFileInfoLabel();
}


void MvQObstatScatterToolBase::loadKeyProfile(MvKeyProfile *prof)
{	
	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prevProfile = blockModel_->keyProfile();

	//Update grib metadata with the new key profile
	data_->loadKeyProfile(prof);

	//---------------------
	// Update message tree
	//---------------------
	
	blockModel_->setKeyProfile(prof);

	for(int i=0;i < blockModel_->columnCount()-1; i++)
	{ 
		blockTree_->resizeColumnToContents(i);
	}

	blockTree_->setCurrentIndex(blockSortModel_->index(0,0));
	slotSelectBlock(blockTree_->currentIndex());

	//Clear the previous provile
	if(prevProfile != 0 && prof  != prevProfile)
		prevProfile->clearKeyData();	

	/*
	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prevProfile = messageModel_->keyProfile();

	//Update grib metadata with the new key profile
	data_->loadKeyProfile(prof);

	//---------------------
	// Update message tree
	//---------------------
	
	messageModel_->setKeyProfile(prof);

	for(int i=0;i < messageModel_->columnCount()-1; i++)
	{ 
		messageTree_->resizeColumnToContents(i);
	}

	messageTree_->setCurrentIndex(messageSortModel_->index(0,0));
	slotSelectMessage(messageTree_->currentIndex());

	//Clear the previous provile
	if(prevProfile != 0 && prof  != prevProfile)
		prevProfile->clearKeyData();
	*/	
}

void MvQObstatScatterToolBase::slotSelectBlock(QModelIndex index)
{	
	int cnt=blockSortModel_->mapToSource(index).row(); 

	string fdump;

	qDebug() << "slotSelectBlock" << currentBlockNo_ << cnt ;

	if(currentBlockNo_ == cnt)
	{
		return;
	}

	currentBlockNo_=cnt;

	//Override cursor
	QApplication::setOverrideCursor(QCursor(Qt::BusyCursor));

	//Generate and read grib dumps
	//createPlot();

	emit blockSelected(currentBlockNo_);

	loadItems(currentBlockNo_);

	//Preview plot
	//createPlot();

	//Log
	//updateLogOutput();

	//Restore cursor
	QApplication::restoreOverrideCursor();
}


void MvQObstatScatterToolBase::slotSelectMessage(QModelIndex index)
{	
	int cnt=messageSortModel_->mapToSource(index).row(); 

	string fdump;

	if(currentItemNo_ == cnt)
	{
		return;
	}

	currentItemNo_=cnt;


	//Override cursor
	QApplication::setOverrideCursor(QCursor(Qt::BusyCursor));

	//Generate and read grib dumps
	createPlot();

	emit messageSelected(currentItemNo_);

	//Preview plot
	//createPlot();

	//Log
	//updateLogOutput();

	//Restore cursor
	QApplication::restoreOverrideCursor();
}

void MvQObstatScatterToolBase::loadItems(int blockNo)
{
	/*if(currentBlockNo_ == blockNo)
	{
		return;
	}
	currentBlockNo_=blockNo;*/

	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prevProfile = messageModel_->keyProfile();

	if(prevProfile)
		delete prevProfile;

	MvKeyProfile * prof=new MvKeyProfile("Items");

	//Update grib metadata with the new key profile
	data_->block(blockNo)->loadKeyProfile(prof);

	//---------------------
	// Update message tree
	//---------------------
	
	messageModel_->setKeyProfile(prof);

	for(int i=0;i < messageModel_->columnCount()-1; i++)
	{ 
		messageTree_->resizeColumnToContents(i);
	}

	messageTree_->setCurrentIndex(messageSortModel_->index(0,0));
	slotSelectMessage(messageTree_->currentIndex());

	//Clear the previous provile
	if(prevProfile != 0 && prof  != prevProfile)
		prevProfile->clearKeyData();

}

//==============================
// Header operations
//==============================


void MvQObstatScatterToolBase::slotMessageTreeContextMenu(const QPoint &position)
{
/*	QHeaderView *header=messageTree_->header();
	int section=header->logicalIndexAt(position); 
	
	if(section< 0 || section >= header->count())
		return;

	qDebug() << "context event in section " << section;

       	QList<QAction*> actions;

	//Filter
	QAction *actionFilter = new QAction(this);
        actionFilter->setObjectName(QString::fromUtf8("actionFilter"));
	actionFilter->setText(tr("Filter"));
        //actionRename->setShortcut(tr("Ctrl+L"));
	actions.append(actionFilter);

	QMenu *filterMenu = new QMenu;

	MvKeyProfile *prof=messageModel_->keyProfile();	
	MvKey* key=prof->at(section);
	if(key)
	{
		MvQKeyFilter filter=messageModel_->keyFilter();
		QString keyName(key->name().c_str());
		QStringList filterItems=filter[keyName];		

		//filterValueList_->clear();

		//QStringList filterItems=messageModel_->keyFilter()[keyName];

		for(map<string,int>::iterator it=key->counter().begin(); 
                    it != key->counter().end(); it++)
		{
			QString str(it->first.c_str());

			QAction *action=new QAction(str,this);
			action->setCheckable(true);

			if(filterItems.count() == 0 || filterItems.contains(str))
			{
				action->setChecked(true);
			}
			else
			{
				action->setChecked(false);
			}
			filterMenu->addAction(action);			
		}
	}
	actionFilter->setMenu(filterMenu);
	

 	//Rename
	QAction *actionRename = new QAction(this);
        actionRename->setObjectName(QString::fromUtf8("actionRename"));
	actionRename->setText(tr("Rename"));
        //actionRename->setShortcut(tr("Ctrl+L"));
	actions.append(actionRename);

	QAction *actionDelete = new QAction(this);
        actionDelete->setObjectName(QString::fromUtf8("actionDelete"));
	actionDelete->setText(tr("Delete"));
	actionDelete->setEnabled(false);
        //actionDelete->setShortcut(tr("Ctrl+D"));
	actions.append(actionDelete);
	
    	if(actions.count() > 0)
	{
        	QAction *action=QMenu::exec(actions, header->mapToGlobal(position));

		if(action == actionRename)
		{
			bool ok;
     			QString text = QInputDialog::getText(0, 
					tr("Rename header"),
                        		tr("New name:"), QLineEdit::Normal,
                                        "", &ok);
     			if (ok && !text.isEmpty())
         			renameMessageTreeHeader(section,text);				
		}
		else if(action == actionDelete)
		{
			deleteMessageTreeColumn(section);
		}
		//Filter
		else if(action)
		{
			action->text();	
			action->isChecked();

			MvQKeyFilter filter=messageModel_->keyFilter();
			QString keyName(key->name().c_str());
			QStringList filterItems=filter[keyName];

			if(action->isChecked() == false)
			{
				if(filterItems.isEmpty())
				{
					for(map<string,int>::iterator it=key->counter().begin(); 
                   				 it != key->counter().end(); it++)
					{
						QString str(it->first.c_str());
						if(str != action->text())
						{
							filterItems.push_back(str);
						}					
					}

				}
				else				
				{
					if(filterItems.contains(action->text()) == true)
					{
						int i=filterItems.indexOf(action->text());
						filterItems.removeAt(i);
					}
				}

				filter[keyName]=filterItems;
			}
			else
			{
				filterItems.push_back(action->text());
				if(filterItems.count() == key->counter().size())
				{
					filter.remove(keyName);
				}
				else
				{
					filter[keyName]=filterItems;
				}
			}
		
			messageModel_->setKeyFilter(filter);
		
			qDebug() << filter;
		}

	}
	
	foreach(QAction *action,actions)
	{
			delete action;
	}
	delete filterMenu;
	*/	
}

void MvQObstatScatterToolBase::renameMessageTreeHeader(int section,QString text)
{
/*	MvKeyProfile* prof = messageModel_->keyProfile();
	if(section >=0 && section < prof->size())
	{
		prof->at(section)->setShortName(text.toStdString());
		messageModel_->setKeyProfile(prof); 		
		keyManager_->saveProfiles();
	}*/
}	

void MvQObstatScatterToolBase::deleteMessageTreeColumn(int column)
{
/*	MvKeyProfile* prof = messageModel_->keyProfile();
	if(column >=0 && column < prof->size())
	{
		prof->deleteKey(column);
		loadKeyProfile(prof);
		keyManager_->saveProfiles();
	}
*/
}	

void MvQObstatScatterToolBase::slotMessageTreeColumnMoved(int logicalIndex, int oldVisualIndex, int newVisualIndex)
{
	/*if(logicalIndex != oldVisualIndex)
	{
		qDebug() << "MvQObstatScatterToolBase::slotMessageTreeColumnMoved> problem with column indices!"; 
	}
	MvKeyProfile* prof = messageModel_->keyProfile();
	if(oldVisualIndex >=0 && oldVisualIndex < prof->size() &&
	   newVisualIndex >=0 && newVisualIndex < prof->size()	&&
	   oldVisualIndex != newVisualIndex)
	{
		for(int i=0; i < messageTree_->header()->count(); i++)
			qDebug() << "header:" << messageTree_->header()->logicalIndex(i);

		prof->reposition(oldVisualIndex,newVisualIndex);
		messageModel_->setKeyProfile(prof); 		
		keyManager_->saveProfiles();
		//messageTree_->header()->reset();

		for(int i=0; i < messageTree_->header()->count(); i++)
			qDebug() << "header:" << messageTree_->header()->logicalIndex(i);
	}*/
}

//-----------------------------------------------------
// Drag and drop
//-----------------------------------------------------

void MvQObstatScatterToolBase::slotDropToMessageTree(QString data)
{
	/*qDebug() << "Drop from dump: " << data;

	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prof = messageModel_->keyProfile();
	prof->addKey(new MvKey(data.toStdString(),data.toStdString()));
	loadKeyProfile(prof);

	keyManager_->saveProfiles();*/
}


//-----------------------------------------------------
// Search
//-----------------------------------------------------

void MvQObstatScatterToolBase::slotManageKeyProfiles()
{
	/*MvQGribKeyDialog keyDialog;

	connect(&keyDialog,SIGNAL(dataChanged()),
		this,SLOT(slotUpdateKeyProfiles()));

	connect(&keyDialog,SIGNAL(logChanged()),
		this,SIGNAL(logOutputChanged()));

	keyDialog.init(keyManager_,grib_);
	keyDialog.exec();*/

/*
	if (!keyDialog_) 
	{
		keyDialog_ = new MvQGribKeyDialog;
		//connect(keyDialog_, SIGNAL(find(const QString &,QTextDocument::FindFlags)),
		//	this, SLOT(find(const QString &,QTextDocument::FindFlags)));

		keyDialog_->init(keyManager_,grib_);
		keyDialog_->show();
		keyDialog_->activateWindow();
		
		connect(keyDialog_,SIGNAL(dataChanged()),
			this,SLOT(slotUpdateKeyProfiles()));

		connect(keyDialog_,SIGNAL(logChanged()),
			this,SIGNAL(logOutputChanged()));
		

	}
	else if(!keyDialog_->isVisible()) 
	{
		keyDialog_->show();
	}

*/
}


void MvQObstatScatterToolBase::slotUpdateKeyProfiles()
{
	//Save the current profile name
	/*string currentProfName=keyCombo_->currentText().toStdString(); 

	//Clear the profile list
	keyCombo_->clear();

	int i=0;
	int index=-1;
	for (vector<MvKeyProfile*>::const_iterator it=keyManager_->begin(); it != keyManager_->end(); it++) 
	{
		keyCombo_->addItem(QString((*it)->name().c_str() ));
		if((*it)->name()== currentProfName)
		{
			index=i;
		}
		i++;
	}

	if(index != -1)
	{
		keyCombo_->setCurrentIndex(index);
		loadKeyProfile(keyManager_->at(index));
	}
	else
	{
		keyCombo_->setCurrentIndex(0);
		loadKeyProfile(keyManager_->at(0));
	}*/
}

void MvQObstatScatterToolBase::updateFileInfoLabel()
{
	fileInfoLabel_->setObstatScatterTextLabel(data_->fileName(),data_->itemNum());
}


void MvQObstatScatterToolBase::slotSwapAxes(bool status)
{
	if(swapAxes_!= status)
	{
		swapAxes_=status;
		createPlot();
	}
}

void MvQObstatScatterToolBase::createPlot()
{
	//Preview plot
	MvQScatterBlock *block=data_->block(currentBlockNo_);
	MvQScatterItem *item=block->items().at(currentItemNo_);

	//QString nc="/var/tmp/cgr/PERFORCE/metview_4/src/ObstatScatterTool/res.nc";
	
	QString ncFile=QString::fromStdString(marstmp());
	item->writeDataMatrixToNc(ncFile,swapAxes_);

	QVector<float> meanX,meanY;
	QStringList tmpStr;
	item->computeMeanForColumns(meanX,meanY);
	
	foreach(float v,meanX)
	{
		tmpStr << QString::number(v);
	}
	QString meanXStr=tmpStr.join("/");
	
	tmpStr.clear();
	foreach(float v,meanY)
	{
		tmpStr << QString::number(v);
	}
	QString meanYStr=tmpStr.join("/");


	QString magml="scatter.magml";
	
	//QString res="/var/tmp/cgr/PERFORCE/metview_4/src/ObstatScatterTool/res.png";
	//QString res1="/var/tmp/cgr/PERFORCE/metview_4/src/ObstatScatterTool/res.1.png";

	QString paramX, paramY;
	float  startX, endX, startY, endY;

	if(swapAxes_ == false)
	{		
		paramX=item->titleX();
		paramY=item->titleY();
		startX=item->startX();
		endX=item->endX();
		startY=item->startY();
		endY=item->endY();

	}
	else
	{
		paramX=item->titleY();
		paramY=item->titleX();
		startX=item->startY();
		endX=item->endY();
		startY=item->startX();
		endY=item->endX();
	}

	QString resFile=QString::fromStdString(marstmp());

	QString cmd="magmlx " +  magmlScript_ +
	" -dataFile=" + ncFile +
	" -blockTitle=\"" + block->title() + "\"" +
	" -startDate=" + block->startDate().toString("yyyyMMddhh") +
	" -endDate=" + block->endDate().toString("yyyyMMddhh") +
	" -area=" + block->areaNames().at(item->area()-1) +
	" -flag=" + QString::number(item->flag()) +
	" -x_param=\"" + paramX.simplified() + "\"" +
	" -y_param=\"" + paramY.simplified() + "\"" +
	" -start_x=" + QString::number(startX) +
	" -end_x=" + QString::number(endX) +
	" -start_y=" + QString::number(startY) +
	" -end_y=" + QString::number(endY) +
	" -mean_x_lst=" + meanXStr +
	" -mean_y_lst=" + meanYStr +
	" -x_min=" + QString::number(item->minX()) +
	" -x_max=" + QString::number(item->maxX()) +
	" -y_min=" + QString::number(item->minY()) +
	" -y_max=" + QString::number(item->maxY()) +
	" -source_num=" + QString::number(item->sourceDataNum()) +
	" -source_num=" + QString::number(item->sourceDataNum()) +
 	" -path=" + resFile;

	system(cmd.toStdString().c_str());

	qDebug() << cmd;

	//cout << sout.str() << endl;

	QString resFileWithSuffix=resFile + ".png";

	QPixmap pixmap(resFileWithSuffix,"PNG");
 	plotWidget_->setPixmap(pixmap);
	plotWidget_->adjustSize();

	//CLEAR

	return;

	QFile fRes(resFile);
	if(!fRes.remove())
	{
		qDebug() << "MvQObstatScatterToolBase::createPlot() ---> Res file" << resFile << "could not be deleted!";
	}

	QFile fResSuffix(resFileWithSuffix);
	if(!fResSuffix.remove())
	{
		qDebug() << "MvQObstatScatterToolBase::createPlot() ---> Res file" << resFileWithSuffix << "could not be deleted!";
	}

	QFile fNc(ncFile);
	if(!fNc.remove())
	{
		qDebug() << "MvQObstatScatterToolBase::createPlot() ---> Nc file" << ncFile << "could not be deleted!";
	}

}
