/*
 * (C) Copyright 1996- ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 * In applying this licence, ECMWF does not waive the privileges and immunities
 * granted to it by virtue of its status as an intergovernmental organisation nor
 * does it submit to any jurisdiction.
 */

/// @author Baudouin Raoult
/// @author Pedro Maciel
/// @date Apr 2015


#include "mir/action/interpolate/Gridded2RotatedLL.h"

#include <iostream>
#include <vector>
#include "eckit/exception/Exceptions.h"
#include "mir/config/LibMir.h"
#include "mir/param/MIRParametrisation.h"
#include "mir/repres/latlon/RotatedLL.h"


namespace mir {
namespace action {
namespace interpolate {


Gridded2RotatedLL::Gridded2RotatedLL(const param::MIRParametrisation& parametrisation) :
    Gridded2RotatedGrid(parametrisation) {

    std::vector<double> value;
    ASSERT(parametrisation_.userParametrisation().get("grid", value));

    ASSERT(value.size() == 2);
    increments_ = util::Increments(value[0], value[1]);

    PointLatLon ref(0, 0);
    if (parametrisation_.userParametrisation().get("area", value)) {
        ASSERT(value.size() == 4);
        bbox_ = util::BoundingBox(value[0], value[1], value[2], value[3]);
        ref = PointLatLon(bbox_.south(), bbox_.west());
    }

    repres::latlon::LatLon::globaliseBoundingBox(bbox_, increments_, ref);

    eckit::Log::debug<LibMir>()
            << "Gridded2RotatedLL: globalise:"
            << "\n\t" << increments_
            << "\n\t" << bbox_
            << "\n\t" "shifted in latitude? " << increments_.isLatitudeShifted(bbox_)
            << "\n\t" "shifted in longitude? " << increments_.isLongitudeShifted(bbox_)
            << std::endl;
}


Gridded2RotatedLL::~Gridded2RotatedLL() = default;


bool Gridded2RotatedLL::sameAs(const Action& other) const {
    auto o = dynamic_cast<const Gridded2RotatedLL*>(&other);
    return o && (increments_ == o->increments_) && (bbox_ == o->bbox_) && Gridded2RotatedGrid::sameAs(*o);
}


void Gridded2RotatedLL::print(std::ostream& out) const {
    out << "Gridded2RotatedLL["
           "increments=" << increments_ << ","
           "bbox=" << bbox_ << ","
           "rotation=" << rotation() << ",";
    Gridded2RotatedGrid::print(out);
    out << "]";
}


const repres::Representation *Gridded2RotatedLL::outputRepresentation() const {
    const PointLatLon ref(bbox_.south(), bbox_.west());
    return new repres::latlon::RotatedLL(increments_, rotation(), bbox_, ref);
}


const char* Gridded2RotatedLL::name() const {
    return "Gridded2RotatedLL";
}


namespace {
static ActionBuilder< Gridded2RotatedLL > grid2grid("interpolate.grid2rotated-regular-ll");
}


}  // namespace interpolate
}  // namespace action
}  // namespace mir

