/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include "proto.h"

extern GeneralSection *general_section;

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif				/* HAVE_ZLIB */

/*
create empty filebuf
*/
Filebuf::Filebuf()
{
	data = NULL;
	type = NULL;
	size = realsize = 0;
}

Filebuf::~Filebuf()
{
	if (data != NULL)
		xfree(data);
	if (type != NULL)
		xfree(type);
}


Filebuf *Filebuf::Dup()
{
	Filebuf *ret;

	ret = xnew Filebuf();

	ret->data = (size != 0) ? (char*)xmalloc(size) : NULL;
	memcpy(ret->data, data, size);
	ret->size = ret->realsize = size;

	return ret;
}

/*
reduce filebuf size to actual size of contents
*/
void Filebuf::Shorten()
{
	if (realsize > size) {
		realsize = size;
		data = (char*)xrealloc(data, size);
	}
}

/*
append data to a file buffer
*/
void Filebuf::Add(const char *data, int len)
{
	int x;

	x = size;

	Resize(x + len);

	memcpy(&this->data[x], (const char *)data, len);
}


/*
add formatted data to filebuf
*/
void Filebuf::Addf(const char *fmt, ...)
{
	int ret;
	char buf[8192];
	va_list valist;

	va_start(valist, fmt);
	ret = vsnprintf(buf, sizeof(buf), fmt, valist);
	va_end(valist);

	Add(buf, (ret > sizeof(buf) || ret == -1) ? sizeof(buf) : ret);
}

/*
resize a filebuf (without adding anything)
*/
void Filebuf::Resize(int newsize)
{
	if (realsize >= newsize && realsize <= newsize + FILEBUF_ALIGNMENT && data != NULL)
		size = newsize;
	else if (newsize != 0) {
		/* aligning the filebuf size makes it much more difficult to catch off-by-one errors */
#ifdef _DEBUG
		realsize = newsize;
#else
		realsize = ALIGN2(newsize, FILEBUF_ALIGNMENT);
#endif
		data = (char*)xrealloc(data, realsize);
		size = newsize;
	}
}

/*
write a filebuf to a file
*/
int Filebuf::Save(char *filename)
{
	int fd, ret;

	fd = open(filename, O_CREAT | O_WRONLY);
	if (fd == -1)
		return -1;

	ret = write(fd, data, size);

	close(fd);

	return ret;
}

/*
read a file into a filebuf
*/
int Filebuf::Read(char *filename)
{
	int x, ret = 0, fd;
	char buf[BLOCKSIZE];

	fd = open(filename, O_RDONLY);
	if (fd == -1)
		return -1;

	while ((x = read(fd, buf, BLOCKSIZE)) > 0) {
		Add(buf, x);

		ret += x;
	}

	close(fd);

	return ret;
}

/*
replace area of filebuf at specified offset and length with something else
*/
void Filebuf::Replace(int so, int eo, char *replace)
{
	int len1 = eo - so;
	int len2 = strlen(replace);

	if (len2 > len1)
		Resize(size + (len2 - len1));

	if (len1 != len2)
		memmove(&data[eo + (len2 - len1)], &data[eo], size - eo);

	if (len2 < len1)
		Resize(size + (len2 - len1));

	memcpy(&data[so], replace, len2);
}

#ifdef HAVE_ZLIB

int Filebuf::Compress(int gzip)
{
	int x, i, done = 0;
	char gzhdr[10] = { 0x1f, 0x8b, Z_DEFLATED, 0, 0, 0, 0, 0, 0, 0x03 };
	char gzbuffer[8192], *ptr;
	unsigned long gz_crc32, len;
	z_stream gz_stream;
	Filebuf *cfilebuf;

	/* it seems zlib makes an allocation with a size of 0, so using the xmalloc/xfree
	   wrappers is out of the question since they assert for that */
	gz_stream.zalloc = NULL;
	gz_stream.zfree = NULL;
	gz_stream.opaque = NULL;
	gz_stream.next_in = gz_stream.next_out = 0;
	gz_stream.avail_in = gz_stream.avail_out = 0;

	x = deflateInit2(&gz_stream, Z_BEST_COMPRESSION, Z_DEFLATED, -MAX_WBITS, 9, Z_DEFAULT_STRATEGY);
	if (x != Z_OK)
		return FALSE;

	cfilebuf = xnew Filebuf();
	if (gzip == TRUE)
		cfilebuf->Add(gzhdr, sizeof(gzhdr));

	gz_stream.next_out = (Bytef*)gzbuffer;
	gz_stream.avail_out = sizeof(gzbuffer);
	gz_stream.next_in = (Bytef*)data;
	gz_stream.avail_in = size;

	while (gz_stream.avail_in != 0) {
		if (gz_stream.avail_out == 0) {
			cfilebuf->Add(gzbuffer, sizeof(gzbuffer));

			gz_stream.next_out = (Bytef*)gzbuffer;
			gz_stream.avail_out = sizeof(gzbuffer);
		}

		x = deflate(&gz_stream, Z_NO_FLUSH);
		if (x == Z_STREAM_END)
			break;
		else if (x != Z_OK)
			goto error;
	}

	while (1) {
		len = sizeof(gzbuffer) - gz_stream.avail_out;
		if (len > 0) {
			cfilebuf->Add(gzbuffer, len);

			gz_stream.next_out = (Bytef*)gzbuffer;
			gz_stream.avail_out = sizeof(gzbuffer);
		}

		if (done)
			break;

		x = deflate(&gz_stream, Z_FINISH);
		if (len == 0 && x == Z_BUF_ERROR)
			x = Z_OK;

		done = (gz_stream.avail_out != 0 || x == Z_STREAM_END);

		if (x != Z_OK && x != Z_STREAM_END)
			break;
	}

	if (x != Z_STREAM_END)
		goto error;

	if (gzip == TRUE) {
		len = cfilebuf->size;
		cfilebuf->Resize(cfilebuf->size + 8);

		/* append the crc32 and size */
		gz_crc32 = crc32(0L, (Bytef*)data,  size);

		for (ptr = cfilebuf->data + len, i = 0; i < 4; i++) {
			*ptr++ = (gz_crc32 & 0xFF);
			gz_crc32 >>= 8;
		}

		x = gz_stream.total_in;
		for (i = 0; i < 4; i++) {
			*ptr++ = (x & 0xFF);
			x >>= 8;
		}
	}

	xfree(data);
	data = cfilebuf->data;
	size = cfilebuf->size;
	realsize = cfilebuf->realsize;

	cfilebuf->data = NULL;
	xdelete cfilebuf;

	deflateEnd(&gz_stream);

	return TRUE;

      error:

	zlib_error(x);

	xdelete cfilebuf;
	deflateEnd(&gz_stream);

	return FALSE;
}

int Filebuf::Uncompress(int gzip)
{
	int x, done = 0;
	char gzbuffer[8192];
	unsigned long len;
	z_stream gz_stream;
	Filebuf *cfilebuf;

	/* sanity check, we assume it's at least this big further down */
	if (size < 18 && gzip == TRUE)
		return FALSE;

	gz_stream.zalloc = NULL;
	gz_stream.zfree = NULL;
	gz_stream.opaque = NULL;
	gz_stream.next_in = gz_stream.next_out = 0;
	gz_stream.avail_in = gz_stream.avail_out = 0;

	x = inflateInit2(&gz_stream, -MAX_WBITS);
	if (x != Z_OK)
		return FALSE;

	cfilebuf = xnew Filebuf();

	gz_stream.next_out = (Bytef*)gzbuffer;
	gz_stream.avail_out = sizeof(gzbuffer);
	if (gzip == TRUE) {
		/* skip gzip header */
		gz_stream.next_in = (Bytef*)data + 10;
		gz_stream.avail_in = size - 18;
	} else {
		gz_stream.next_in = (Bytef*)data;
		gz_stream.avail_in = size;
	}

	while (gz_stream.avail_in != 0) {
		if (gz_stream.avail_out == 0) {
			cfilebuf->Add(gzbuffer, sizeof(gzbuffer));

			gz_stream.next_out = (Bytef*)gzbuffer;
			gz_stream.avail_out = sizeof(gzbuffer);
		}

		x = inflate(&gz_stream, Z_NO_FLUSH);
		if (x == Z_STREAM_END) 
			break;
		else if (x != Z_OK)
			goto error;
	}

	while (1) {
		len = sizeof(gzbuffer) - gz_stream.avail_out;
		if (len > 0) {
			cfilebuf->Add(gzbuffer, len);

			gz_stream.next_out = (Bytef*)gzbuffer;
			gz_stream.avail_out = sizeof(gzbuffer);
		}

		if (done)
			break;

		x = inflate(&gz_stream, Z_SYNC_FLUSH);
		if (len == 0 && x == Z_BUF_ERROR)
			x = Z_OK;

		done = (gz_stream.avail_out != 0 || x == Z_STREAM_END);

		if (x != Z_OK && x != Z_STREAM_END)
			break;
	}

	if (x != Z_STREAM_END)
		goto error;

	/* skip validating crc32 and size... I'm lazy :P */

	xfree(data);
	data = cfilebuf->data;
	size = cfilebuf->size;
	realsize = cfilebuf->realsize;

	cfilebuf->data = NULL;
	xdelete cfilebuf;

	inflateEnd(&gz_stream);

	return TRUE;

      error:

	zlib_error(x);

	xdelete cfilebuf;
	inflateEnd(&gz_stream);

	return FALSE;
}

void zlib_error(int x)
{
	switch (x) {
	case Z_DATA_ERROR:
		putlog(MMLOG_ERROR, "zlib data error");
		break;
	case Z_MEM_ERROR:
		putlog(MMLOG_ERROR, "zlib memory error");
		break;
	case Z_BUF_ERROR:
		putlog(MMLOG_ERROR, "zlib buffer error");
		break;
	case Z_STREAM_ERROR:
		putlog(MMLOG_ERROR, "zlib stream error");
		break;
	case Z_NEED_DICT:
		putlog(MMLOG_ERROR, "zlib dictionary error");
		break;
	default:
		putlog(MMLOG_ERROR, "zlib unknown error %d", x);
		break;
	}
}

#else

int Filebuf::Compress(int gzip)
{
	return FALSE;
}

int Filebuf::Uncompress(int gzip)
{
	return FALSE;
}

#endif				/* HAVE_ZLIB */

struct STACK_T *Filebuf::to_stack()
{
	char *lptr, *eolptr;
	struct STACK_T *ret = NULL, *start = NULL;

	lptr = data;
	while (lptr != NULL) {
		eolptr = (char*)memchr(lptr, '\n', size - (lptr - data));

		PUSHN(ret, lptr, (eolptr != NULL) ? eolptr - lptr : size - (lptr - data));
		if (start == NULL)
			start = ret;

		lptr = (eolptr != NULL && eolptr - data != size - 1) ? eolptr + 1 : NULL;
	}

	return start;
}

int Filebuf::Decode(CONNECTION * connection)
{
	int ret;

	if (s_strcasestr(connection->rheader->content_encoding, "gzip") != NULL) {
		ret = Uncompress(TRUE);
		if (ret == FALSE) {
			putlog(MMLOG_ERROR, "failed to decompress gzip encoded content");

			/* failed to decompress, just send as-is */
			return FALSE;
		} else
			putlog(MMLOG_DEBUG, "decompressed gzip encoded file");
	} else if (s_strcasestr(connection->rheader->content_encoding, "deflate") != NULL) {
		ret = Uncompress(FALSE);
		if (ret == FALSE) {
			putlog(MMLOG_ERROR, "failed to decompress deflate encoded content");

			return FALSE;
		} else
			putlog(MMLOG_DEBUG, "decompressed deflate encoded file");
	} else {
		/* unhandled content encoding, send as-is */
		putlog(MMLOG_WARN, "unhandled content encoding: %s", connection->rheader->content_encoding);

		return FALSE;
	}

	return TRUE;
}

int Filebuf::Encode(CONNECTION * connection)
{
	int ret;

	if (connection->flags & CONNECTION_ENCODED)
		return FALSE;

	if (connection->header->accept_encoding != NULL && general_section->compressout_get() == TRUE) {
		/* compress the content if the browser supports gzip encoding */
		if (s_strcasestr(connection->header->accept_encoding, "gzip") != NULL) {
			ret = Compress(TRUE);
			if (ret == TRUE) {
				putlog(MMLOG_DEBUG, "gzip encoded file");
				connection->rheader->content_encoding = xstrdup("gzip");

				return TRUE;
			} else {
				putlog(MMLOG_DEBUG, "failed to gzip encode file");

				return FALSE;
			}
		} else if (s_strcasestr(connection->header->accept_encoding, "deflate") != NULL) {
			ret = Compress(FALSE);
			if (ret == TRUE) {
				putlog(MMLOG_DEBUG, "deflate encoded file");
				connection->rheader->content_encoding = xstrdup("deflate");

				return TRUE;
			} else {
				putlog(MMLOG_DEBUG, "failed to deflate encode file");

				return FALSE;
			}
		}
	}

	return FALSE;
}
