/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include "proto.h"


extern TemplateSection *template_section;

FilterSection::FilterSection():
	Section("filter", RWLOCK),
     enabled (field_vec[0].int_value),
     policy  (field_vec[1].int_value),
     dtempl  (field_vec[2].string_value)
{
}

void FilterSection::update()
{
	allow_list.clear();
	deny_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		allow_list.push_back(Filter(*item));
	
	for (item = sub_vec[1].item_list.begin(); item != sub_vec[1].item_list.end(); item++) 
		deny_list.push_back(Filter(*item));
}

Filter::Filter(const Item& item):
	Encapsulator(item),
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     host     (item.field_vec[3].string_value),
     file     (item.field_vec[4].string_value),
     iprange  (item.field_vec[5].string_value),
     templ    (item.field_vec[6].string_value)
{
	he = (host != "") ? reg_compile(host.c_str(), REGFLAGS) : NULL;
	fe = (file != "") ? reg_compile(file.c_str(), REGFLAGS) : NULL;
}
 
Filter::Filter(const Filter& f):
	Encapsulator(f.item),
     enabled  (f.enabled),
     comment  (f.comment),
     profiles (f.profiles),
     host     (f.host),
     file     (f.file),
     iprange  (f.iprange),
     templ    (f.templ)
{
	he = (host != "") ? reg_compile(host.c_str(), REGFLAGS) : NULL;
	fe = (file != "") ? reg_compile(file.c_str(), REGFLAGS) : NULL;
}
     

/* Constructor for empty filter */    
Filter::Filter():
	enabled (TRUE)
{
	he = fe = NULL;
}

Filter::~Filter()
{
	if (he != NULL) {
		reg_free(he);
	}
	if (fe != NULL) {
		reg_free(fe);
	}
}

/*
determine whether or not the host and/or file is allowed to be retrieved
based on the rules given in the filter_list linked list
*/
const Filter* FilterSection::check(CONNECTION * connection)
{
	int action = FALSE, result = TRUE, ret, i;
	FilterList *list;
	const Filter *match = NULL;
	HOSTENT *hostent = NULL;

	if (this->enabled == FALSE || connection->bypass & FEATURE_FILTER)
		return NULL;

	for (i = 0; i < 2; i++) {
		if (i == 0) {
			if (this->policy == POLICY_ALLOW) {
				list = &this->deny_list;
				action = FALSE;
				result = TRUE;
			} else {
				list = &this->allow_list;
				action = TRUE;
				result = FALSE;
			}
		} else if (result == action) {
			if (this->policy == POLICY_ALLOW) {
				list = &this->allow_list;
				action = TRUE;
			} else {
				list = &this->deny_list;
				action = FALSE;
			}
		} else
			break;

		FilterList::const_iterator current;
		for (current = list->begin(); current != list->end(); current++) {
			if (current->enabled == FALSE)
				continue;

			if (!profile_find(connection->profiles, current->profiles))
				continue;

			if (current->he != NULL) {
				ret = reg_exec(current->he, connection->header->host);
				if (ret)
					continue;
			}

			if (current->fe != NULL) {
				ret = reg_exec(current->fe, connection->header->file);
				if (ret)
					continue;
			}

			if (current->iprange.empty() == FALSE) {
				char ip[16];

				if (hostent == NULL) hostent = net_dns(connection->header->host);
				/* this will never be NULL since we would have failed this request by now, 
				   but I like to play it safe. */
				if (hostent != NULL) {
					inet_ntop(AF_INET, hostent->addr, ip, sizeof(ip));
					ret = current->iprange.has(ip);

					if (!ret) continue;
				}
			}

			match = &*current;
			result = action;
			break;
		}
	}

	if (hostent != NULL) hostent_free(hostent);

	return (!result) ? (match != NULL) ? match : &this->empty : NULL;
}


/*
 * Check and block the request if it is not allowed to be retireved
 * and send a template page as a response.
 * return 1 if blocked, 0 if allowed.
 */
int FilterSection::check_and_block(CONNECTION * connection)
{
	read_lock();

	const Filter* match = check(connection);

	if (match != NULL) {
		putlog(MMLOG_FILTER, "blocked %s%s", connection->header->host, connection->header->file);
		
		template_section->send((match->templ != "") ? match->templ.c_str() : (this->dtempl != "") ? this->dtempl.c_str() : "blocked", connection, (connection->header->type == HTTP_CONNECT) ? 404 : 200);
		
		unlock();
		return 1;
	}
	
	unlock();
	return 0;
}

void FilterSection::check_show(CONNECTION * connection)
{
	read_lock();

	const Filter *fl;
	Filebuf *filebuf = xnew Filebuf();

	fl = check(connection);

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>URL filter matche for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">URL filter match for %s%s</td></tr>\n", connection->header->host, connection->header->file);

	if (fl == NULL)
		filebuf->Addf("<tr><td align=\"center\" colspan=\"2\">None</td></tr>\n");
	else {
		fl->item->display(filebuf, FALSE);
	}

	filebuf->Addf("</table></td></tr></table></body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;

	unlock();
}
