/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <errno.h>
#include "proto.h"

extern GLOBAL *global;
extern CookieSection *cookie_section;
extern CacheSection *cache_section;

#include <stdio.h>
#include <string.h>
#include "proto.h"


HeaderSection::HeaderSection():
     Section("header", RWLOCK),
     enabled (field_vec[0].int_value),
     policy  (field_vec[1].int_value)
{
}

void HeaderSection::update()
{
	allow_list.clear();
	deny_list.clear();
	insert_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		allow_list.push_back(Header(*item));
	
	for (item = sub_vec[1].item_list.begin(); item != sub_vec[1].item_list.end(); item++) 
		deny_list.push_back(Header(*item));

	for (item = sub_vec[2].item_list.begin(); item != sub_vec[2].item_list.end(); item++) 
		insert_list.push_back(Header(*item));
}


Header::Header(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     type     (item.field_vec[3].string_value),
     value    (item.field_vec[4].string_value),
     which    (item.field_vec[5].int_value)
{
	te = (type != "") ? reg_compile(type.c_str(), REGFLAGS) : NULL;
	ve = (value != "") ? reg_compile(value.c_str(), REGFLAGS) : NULL;
}

Header::Header(const Header& h):
     enabled  (h.enabled),
     comment  (h.comment),
     profiles (h.profiles),
     type     (h.type),
     value    (h.value),
     which    (h.which)
{
	te = (type != "") ? reg_compile(type.c_str(), REGFLAGS) : NULL;
	ve = (value != "") ? reg_compile(value.c_str(), REGFLAGS) : NULL;
}

Header::~Header()
{
	if (te != NULL)
		reg_free(te);
	if (ve != NULL)
		reg_free(ve);
}

HEADER *http_header_parse_response(char *s)
{
	int x;
	char *ptr, buf[256], version[10], code[10];
	HEADER *header;

	header = header_new();

	header->type = HTTP_RESP;

	if (strncasecmp(s, "HTTP/", 5)) {
		/* HTTP/0.9 response (no header) */
		header->version = HTTP_HTTP09;
		header->code = 200;

		putlog(MMLOG_DEBUG, "HTTP/0.9 header");
	} else {
		/* it would appear sscanf will read past what is required for the format
		   under some circumstances, and the buffer isn't always NULL terminated..
		   so this is just to be safe; it showed up a few times under valgrind */
		ptr = strchr(s, '\n');
		if (ptr == NULL || ptr - s >= sizeof(buf)) goto error;
		s_strncpy(buf, s, ptr - s + 1);
		x = sscanf(buf, "%10s %10s", version, code);
		if (x != 2)
			goto error;

		if (!strcasecmp(version, "HTTP/1.1"))
			header->version = HTTP_HTTP11;
		else
			header->version = HTTP_HTTP10;

		header->code = atoi(code);
		if (header->code <= 0)
			goto error;

		http_header_list_parse(header, s);
	}

	header->raw = (header->version != HTTP_HTTP09) ? xstrndup(s, header_length(s, ~0)) : xstrdup("HTTP/1.1 200 OK\r\n\r\n");

	return header;

      error:
	http_header_free(header);
	return NULL;
}

HEADER *http_header_parse_request(char *s)
{
	int x;
	char *ptr;
	char buf[2048], method[24], version[10], username[128], password[128];
	HEADER *header;

	/* parse first line */
	/* version can be left out, i.e. "GET /" */
	x = sscanf(s, "%24s %2048s %10s", method, buf, version);
	if (x < 2)
		return NULL;

	header = header_new();

	if (!strcasecmp(method, "CONNECT"))
		header->type = HTTP_CONNECT;
	else if (buf[0] == '/')
		header->type = HTTP_REQUEST;
	else
		header->type = HTTP_PROXY;

	if (x == 3 && !strcasecmp(version, "HTTP/1.1"))
		header->version = HTTP_HTTP11;
	else
		header->version = HTTP_HTTP10;


	if (header->type != HTTP_REQUEST) {
		/* proxy request */

		/* find : in http:// */
		if (header->type != HTTP_CONNECT) {
			ptr = strchr(buf, ':');
			if (ptr == NULL)
				goto error;

			*(ptr++) = '\0';

			header->proto = xstrdup(buf);
			string_tolower(header->proto);

			if (ptr[1] == '\0')
				goto error;

			memmove(buf, &ptr[2], strlen(&ptr[2]) + 1);
		} else header->proto = xstrdup("connect");

		/* find beginning of file */
		ptr = strchr(buf, '/');
		if (ptr != NULL) {
			header->file = url_path_fix(ptr);
			*ptr = '\0';
		}

		ptr = strchr(buf, '@');
		if (ptr != NULL) {
			x = sscanf(buf, "%128[^:]:%128[^@]", username, password);
			if (x == 2) {
				header->username = url_decode(username, ~0);
				header->password = url_decode(password, ~0);
			}

			memmove(buf, &ptr[1], strlen(&ptr[1]) + 1);
		}

		/* grab the port */
		ptr = strchr(buf, ':');
		if (ptr != NULL) {
			header->port = atoi(&ptr[1]);
			*ptr = '\0';
		} else {
			header->flags |= HEADER_DPORT;

			if (header->type == HTTP_CONNECT)
				header->port = HTTPS_PORT;
			else
				header->port = protocol_port(header->proto);
		}

		header->host = url_decode(buf, strlen(buf));
		string_tolower(header->host);

		snprintf(buf, sizeof(buf), "%s://%s:%d%s", header->proto, header->host, header->port, header->file);
		header->url = xstrdup(buf);
	} else {
		/* http request */

		header->proto = xstrdup("http");
		header->file = url_path_fix(buf);
		header->port = HTTP_PORT;

		// header->url is built later once we determine if this is a transparent proxy request
	}

	if (header->file == NULL)
		header->file = xstrdup("/");
	header->method = xstrdup(method);

	http_header_list_parse(header, s);

	header->raw = xstrndup(s, header_length(s, ~0));

	return header;

      error:
	http_header_free(header);
	return NULL;
}

void http_header_list_parse(HEADER * header, char *s)
{
	int send;
	char *ptr, *lptr, *eolptr, buf[4096];

	/* iterate through each header, grab useful header values and put them all into a linked list */
	lptr = strchr(s, '\n');
	while (lptr != NULL) {
		lptr++;

		eolptr = strchr(lptr, '\n');

		s_strncpy(buf, lptr, (eolptr - lptr + 1) < sizeof(buf) ? eolptr - lptr + 1 : sizeof(buf));

		STRIP_CRLF(buf);

		ptr = strchr(buf, ':');

		if (ptr != NULL && ptr[1] != '\0') {
			*ptr = '\0';
			ptr += (ptr[1] == ' ') ? 2 : 1;

			send = FALSE;

			if (!(header->flags & HEADER_CL) && !strcasecmp(buf, "Content-length")) {
				header->content_length = strtoul(ptr, NULL, 10);
				header->flags |= HEADER_CL;
			} else if (!header->chunked && !strcasecmp(buf, "Transfer-Encoding") && !strcasecmp(ptr, "chunked"))
				header->chunked = TRUE;
			else if (!header->content_type && !strcasecmp(buf, "Content-Type"))
				header->content_type = xstrdup(ptr);
			else if (!header->content_encoding && !strcasecmp(buf, "Content-Encoding"))
				header->content_encoding = xstrdup(ptr);
			else if (!header->accept_encoding && !strcasecmp(buf, "Accept-Encoding"))
				header->accept_encoding = xstrdup(ptr);
			else if (!header->location && !strcasecmp(buf, "Location"))
				header->location = xstrdup(ptr);
			else if (!header->host_header && !strcasecmp(buf, "Host"))
				header->host_header = xstrdup(ptr);
			else if (!header->authorization && !strcasecmp(buf, "Authorization"))
				header->authorization = xstrdup(ptr);
			else if (!header->proxy_authenticate && !strcasecmp(buf, "Proxy-Authenticate"))
				header->proxy_authenticate = xstrdup(ptr);
			else if (!header->proxy_authorization && !strcasecmp(buf, "Proxy-Authorization"))
				header->proxy_authorization = xstrdup(ptr);
			else if (!header->referer && !strcasecmp(buf, "Referer"))
				header->referer = xstrdup(ptr);
			else if (!header->cache_control && !strcasecmp(buf, "Cache-Control"))
				header->cache_control = xstrdup(ptr);
			else if (!header->expires && !strcasecmp(buf, "Expires"))
				header->expires = xstrdup(ptr);
			else if (!header->if_modified_since && !strcasecmp(buf, "If-Modified-Since"))
				header->if_modified_since = xstrdup(ptr);
			else if (!header->last_modified && !strcasecmp(buf, "Last-Modified"))
				header->last_modified = xstrdup(ptr);
			else if (!header->range && !strcasecmp(buf, "Range"))
				header->range = xstrdup(ptr);
			else if (!header->x_cache && !strcasecmp(buf, "X-Cache"))
				header->x_cache = xstrdup(ptr);
			else if (!header->via && !strcasecmp(buf, "Via"))
				header->via = xstrdup(ptr);
			/* these are additional hop-to-hop headers not needed by the proxy, but can't
			   be passed on to the website or proxy */
			else if (!strcasecmp(buf, "TE"));
			else if (!strcasecmp(buf, "Trailers"));
			else if (!strcasecmp(buf, "Upgrade"));
			else if (!strcasecmp(buf, "Keep-Alive"));
			else if (header->keepalive == -1 && !strcasecmp(buf, "Connection")) {
				/* some browsers send stuff after the keep-alive or close */
				if (!strncasecmp(ptr, "keep-alive", 10))
					header->keepalive = TRUE;
				else if (!strncasecmp(ptr, "close", 5))
					header->keepalive = FALSE;
			} else if (header->proxy_keepalive == -1 && !strcasecmp(buf, "Proxy-Connection")) {
				if (!strncasecmp(ptr, "keep-alive", 10))
					header->proxy_keepalive = TRUE;
				else if (!strncasecmp(ptr, "close", 5))
					header->proxy_keepalive = FALSE;
			} else send = TRUE;

			struct HTTP_HEADER hdr;
			hdr.type = buf;
			hdr.value = ptr;
			hdr.send = send;

			header->header.push_back(hdr);
		}

		if (eolptr != NULL && (eolptr[1] == '\r' || eolptr[1] == '\n'))
			break;
		lptr = eolptr;
	}
}


/*
free memory used by HEADER structure 
*/
void http_header_free(HEADER * header)
{
	if (!header)
		return;

	FREE_AND_NULL(header->raw);
	FREE_AND_NULL(header->url);
	FREE_AND_NULL(header->method);
	FREE_AND_NULL(header->proto);
	FREE_AND_NULL(header->host);
	FREE_AND_NULL(header->file);
	FREE_AND_NULL(header->username);
	FREE_AND_NULL(header->password);
	FREE_AND_NULL(header->content_type);
	FREE_AND_NULL(header->content_encoding);
	FREE_AND_NULL(header->accept_encoding);
	FREE_AND_NULL(header->referer);
	FREE_AND_NULL(header->location);
	FREE_AND_NULL(header->host_header);
	FREE_AND_NULL(header->authorization);
	FREE_AND_NULL(header->proxy_authenticate);
	FREE_AND_NULL(header->proxy_authorization);
	FREE_AND_NULL(header->cache_control);
	FREE_AND_NULL(header->expires);
	FREE_AND_NULL(header->if_modified_since);
	FREE_AND_NULL(header->last_modified);
	FREE_AND_NULL(header->warning);
	FREE_AND_NULL(header->age);
	FREE_AND_NULL(header->x_cache);
	FREE_AND_NULL(header->range);
	FREE_AND_NULL(header->xforwardedfor);
	FREE_AND_NULL(header->via);

	xdelete header;
}


/*
filter header list according to rules set in the XML config, return a new list
containing only headers for which no deny rule matched
*/
void HeaderSection::filter(CONNECTION * connection, int flags)
{
	int action = FALSE, result = TRUE, i, ret;
	HEADER *header;
	HeaderList* header_list;
	HeaderList::const_iterator current;
	HttpHeaderList::const_iterator http_header_list;

	putlog(MMLOG_HEADER, "header_filter flags=%s", flags==HEADER_CLIENT? "client" : "server");

	if (connection->bypass & FEATURE_HEADER)
		return;

	read_lock();

	if (this->enabled == FALSE) {
		unlock();

		return;
	}

	/* get the header to filter */
	header = flags==HEADER_CLIENT ? connection->header : connection->rheader;

	/* a flag is needed to indicate if the header was filtered because
	   this function returns NULL if the filtering results in an empty header,
	   and also when filtering is disabled... both cases are handled differently. */
	header->flags |= HEADER_FILTERED;

	for (http_header_list = header->header.begin(); http_header_list != header->header.end(); http_header_list++) {
		for (i = 0; i < 2; i++) {
			if (i == 0) {
				if (this->policy == POLICY_ALLOW) {
					header_list = &this->deny_list;
					action = FALSE;
					result = TRUE;
				} else {
					header_list = &this->allow_list;
					action = TRUE;
					result = FALSE;
				}
			} else if (result == action) {
				if (this->policy == POLICY_ALLOW) {
					header_list = &this->allow_list;
					action = TRUE;
				} else {
					header_list = &this->deny_list;
					action = FALSE;
				}
			} else
				break;

			for (current = header_list->begin(); current != header_list->end(); current++) {
				if (current->enabled == FALSE)
					continue;

				if (!(current->which & flags))
					continue;

				if (!profile_find(connection->profiles, current->profiles))
					continue;

				if (current->te != NULL) {
					ret = reg_exec(current->te, http_header_list->type.c_str());
					if (ret)
						continue;
				}

				if (current->ve != NULL) {
					ret = reg_exec(current->ve, http_header_list->value.c_str());
					if (ret)
						continue;
				}

				result = action;
				break;
			}
		}

		if (result == TRUE) {
			struct HTTP_HEADER hdr;
			hdr.type = http_header_list->type;
			hdr.value = http_header_list->value;
			hdr.send = http_header_list->send;

			header->header_filtered.push_back(hdr);
		} else
			putlog(MMLOG_HEADER, "removed %s: %s", http_header_list->type.c_str(), http_header_list->value.c_str());
	}


	for (current = insert_list.begin(); current != insert_list.end(); current++) {
		if (current->enabled == FALSE)
			continue;

		if (!(current->which & flags))
			continue;

		if (!profile_find(connection->profiles, current->profiles))
			continue;

		if (current->type != "" && current->value != "") {
			struct HTTP_HEADER hdr;
			hdr.type = current->type;
			hdr.value = current->value;
			hdr.send = TRUE;

			header->header_filtered.push_back(hdr);

			putlog(MMLOG_HEADER, "added %s: %s", current->type.c_str(), current->value.c_str());
		}
	}

	unlock();
}

/*
retrieve header from browser or server
*/
char *header_get(CONNECTION * connection, int flags, int timeout)
{
	int x, pos = 0, first = FALSE;
	char buf[4096];
	struct pollfd pfd[2];
	Socket *sock1, *sock2;
	time_t start = time(NULL), now;

	sock1 = (flags == SERVER) ? connection->server : connection->client;
	sock2 = (flags == SERVER) ? connection->client : connection->server;

	pfd[0].fd = sock1->fd;
	pfd[0].events = POLLIN;

	if (sock2 != NULL) {
		pfd[1].fd = sock2->fd;
		pfd[1].events = POLLIN;
	} else
		pfd[1].events = pfd[1].revents = 0;

	while (1) {
		now = time(NULL);

		if (now - start >= timeout)
			break;

		x = p_poll(pfd, (sock2 != NULL) ? 2 : 1, (sock1->inbuf_len != 0) ? 0 : (timeout - (now - start)) * 1000);
		if (x == -1) {
			if (errno == EINTR) 
				continue;
			else
				break;
		}

		if (sock1->inbuf_len != 0 || pfd[0].revents & POLLIN) {
			x = sock1->GetLine(&buf[pos], sizeof(buf) - pos - 3);
			if (x <= 0)
				break;

			if (buf[pos] == '\n' || (buf[pos] == '\r' && x >= 2 && buf[pos + 1] == '\n') || pos == sizeof(buf) - 3) {
				/* blank line or buffer size limit reached */

				/* ignore leading blanklines, this will resolve some issues
				   with non-compliant browsers */
				if (first == TRUE || pos == sizeof(buf) - 3) {
					buf[pos] = '\r';
					buf[pos + 1] = '\n';
					buf[pos + 2] = '\0';

					return xstrdup(buf);
				}
			} else {
				first = TRUE;
				pos += x;
			}
		} else if (pfd[0].revents & (POLLHUP | POLLERR))
			break;

		if (pfd[1].revents & POLLIN) {
			x = sock2->Read(NULL, -1);

			if (x <= 0) {
				/* the server may disconnect (or already be disconnected)
				   while reading client header */
				if (flags == CLIENT) {
					xdelete connection->server;
					connection->server = NULL;
					sock2 = NULL;
					pfd[1].events = pfd[1].revents = 0;
				} else
					break;
			}
		} else if (pfd[1].revents & (POLLHUP | POLLERR))
			break;
	}

	if (pos > 0) {
		buf[pos] = '\0';
		return xstrdup(buf);
	}

	return NULL;
}

void header_send(HEADER * header, CONNECTION * connection, int which, int flags)
{
	Filebuf *filebuf;
	Socket *sock;

	if (which == CLIENT && (connection->flags & CONNECTION_NOCLIENT)) return;

	sock = (which == SERVER) ? connection->server : connection->client;

	filebuf = xnew Filebuf();

	header_send_filebuf(filebuf, header, connection, which, flags);

	net_filebuf_send(filebuf, connection, which);

	sock->Flush();

	xdelete filebuf;
}


/*
send formatted HTTP header to browser or website without unwanted headers and blocked cookies
*/
void header_send_filebuf(Filebuf * filebuf, HEADER * header, CONNECTION * connection, int which, int flags)
{
	int send = TRUE;
	HeaderList header_list;
	HttpHeaderList::const_iterator http_header_list;
	char *ptr, *ptr2;

	if (header->type == HTTP_RESP)
		filebuf->Addf("HTTP/1.1 %d %s\r\n", header->code, code_to_error(header->code));
	else {
		if (flags == HEADER_DIRECT)
			filebuf->Addf("%s %s HTTP/1.1\r\n", header->method, header->file);
		else {
			/* this is going to another proxy */
			if (header->type == HTTP_CONNECT)
				filebuf->Addf("CONNECT %s:%d HTTP/1.1\r\n", header->host, header->port);
			else {
				if (header->username != NULL && header->password != NULL) {
					ptr = url_encode(header->username, ~0);
					ptr2 = url_encode(header->password, ~0);
					filebuf->Addf("%s %s://%s:%s@%s:%d%s HTTP/1.1\r\n", header->method, header->proto, ptr, ptr2, header->host, header->port, header->file);
					xfree(ptr);
					xfree(ptr2);
				} else
					filebuf->Addf("%s %s HTTP/1.1\r\n", header->method, header->url);
			}
		}
	}

	for (http_header_list = ((header->flags & HEADER_FILTERED) ? header->header_filtered.begin() : header->header.begin()); http_header_list != ((header->flags & HEADER_FILTERED) ? header->header_filtered.end() : header->header.end()); http_header_list++) {
		if (!strncasecmp(http_header_list->type.c_str(), "Set-Cookie", 10) || !strncasecmp(http_header_list->type.c_str(), "Cookie", 6)) {
			CookieHeader cookie_header(http_header_list->value);

			send = cookie_section->check((header->type == HTTP_RESP) ? COOKIE_IN : COOKIE_OUT, connection, cookie_header);
			if (send == FALSE) {
				if (header->type == HTTP_RESP)
					putlog(MMLOG_COOKIE, "blocked incoming from %s", header->host);
				else
					putlog(MMLOG_COOKIE, "blocked outgoing to %s", header->host);

				continue;
			} else if (!cookie_header.name_pairs.empty()) {
				filebuf->Addf("%s: %s\r\n", http_header_list->type.c_str(), cookie_header.Build().c_str());

				continue;
			}
		}

		if (http_header_list->send == TRUE) filebuf->Addf("%s: %s\r\n", http_header_list->type.c_str(), http_header_list->value.c_str());
	}

	/* content type could have been changed if the content was parsed with an external program,
	   therefore we need to send our own content-type header, not the one the site sent */
	if (header->content_type != NULL && strcmp(header->content_type, ""))
		filebuf->Addf("Content-Type: %s\r\n", header->content_type);

	if (header->flags & HEADER_CL)
		filebuf->Addf("Content-Length: %d\r\n", header->content_length);

	if (header->chunked)
		filebuf->Addf("Transfer-Encoding: chunked\r\n");


	switch (flags) {
	case HEADER_FORWARD:
		filebuf->Addf("Proxy-Connection: keep-alive\r\n");

		if (connection->proxy_auth != NULL)
			filebuf->Addf("Proxy-Authorization: %s\r\n", connection->proxy_auth);

		if (header->cache_control)
			filebuf->Addf("Cache-Control: %s\r\n", header->cache_control);
	case HEADER_DIRECT:
		if (header->authorization != NULL)
			filebuf->Addf("Authorization: %s\r\n", header->authorization);

		if (header->if_modified_since != NULL)
			filebuf->Addf("If-Modified-Since: %s\r\n", header->if_modified_since);

		if (header->referer != NULL)
			filebuf->Addf("Referer: %s\r\n", header->referer);

		if (header->range != NULL)
			filebuf->Addf("Range: %s\r\n", header->range);

		/* if gzip encoding is not supported, not sending an Accept-Encoding header at all
		   fixes some issues with buggy versions of PHP */
#ifdef HAVE_ZLIB
		if (header->accept_encoding != NULL)
			filebuf->Addf("Accept-Encoding: %s\r\n", header->accept_encoding);
#endif				/* HAVE_ZLIB */
		/* we need to pass our own Host: header, since the one the browser sent would be wrong
		   if we're doing a redirect 
		   note: some servers have problems interpreting the port field, omit it if it's HTTP_PORT */
		if (header->type != HTTP_CONNECT && header->port == protocol_port(header->proto))
			filebuf->Addf("Host: %s\r\n", header->host);
		else
			filebuf->Addf("Host: %s:%d\r\n", header->host, header->port);

		filebuf->Addf("Connection: keep-alive\r\n");

		if (header->xforwardedfor != NULL)
			filebuf->Addf("X-Forwarded-For: %s\r\n", header->xforwardedfor);

		if (header->via != NULL)
			filebuf->Addf("Via: %s\r\n", header->via);

		break;
	case HEADER_RESP:
		if (header->cache_control != NULL)
			filebuf->Addf("Cache-Control: %s\r\n", header->cache_control);

		if (header->expires != NULL)
			filebuf->Addf("Expires: %s\r\n", header->expires);

		if (header->last_modified != NULL)
			filebuf->Addf("Last-Modified: %s\r\n", header->last_modified);

		if (header->location != NULL)
			filebuf->Addf("Location: %s\r\n", header->location);

		if (header->proxy_authenticate != NULL)
			filebuf->Addf("Proxy-Authenticate: %s\r\n", header->proxy_authenticate);

		if (header->content_encoding != NULL)
			filebuf->Addf("Content-Encoding: %s\r\n", header->content_encoding);

		filebuf->Addf("Connection: %s\r\n", (connection->keepalive_client) ? "keep-alive" : "close");
		filebuf->Addf("Proxy-Connection: %s\r\n", (connection->keepalive_client) ? "keep-alive" : "close");

		if (header->x_cache != NULL)
			filebuf->Addf("X-Cache: %s\r\n", header->x_cache);

		if (header->age != NULL)
			filebuf->Addf("Age: %s\r\n", header->age);

		if (header->warning != NULL)
			filebuf->Addf("Warning: %s\r\n", header->warning);

		break;
	}

	filebuf->Addf("\r\n");
}

/*
return an emtpy HEADER * struct
*/
#ifdef _DEBUG_MEMORY
HEADER *header_new_(char* file, int line)
#else
HEADER *header_new()
#endif
{
	HEADER *header;

	header = xnew HEADER;

	header->raw = NULL;
	header->url = NULL;
	header->code = -1;
	header->method = NULL;
	header->proto = NULL;
	header->host = NULL;
	header->file = NULL;
	header->username = NULL;
	header->password = NULL;
	header->content_type = NULL;
	header->content_encoding = NULL;
	header->accept_encoding = NULL;
	header->referer = NULL;
	header->location = NULL;
	header->host_header = NULL;
	header->authorization = NULL;
	header->proxy_authenticate = NULL;
	header->proxy_authorization = NULL;
	header->cache_control = NULL;
	header->expires = NULL;
	header->if_modified_since = NULL;
	header->last_modified = NULL;
	header->warning = NULL;
	header->age = NULL;
	header->x_cache = NULL;
	header->range = NULL;
	header->xforwardedfor = NULL;
	header->via = NULL;

	header->flags = 0;
	header->keepalive = -1;
	header->proxy_keepalive = -1;
	header->null_body = FALSE;
	header->chunked = FALSE;

	return header;
}

HEADER *http_header_dup(HEADER * h)
{
	HEADER *header;

	header = header_new();

	header->raw = (h->raw != NULL) ? xstrdup(h->raw) : NULL;
	header->url = (h->url != NULL) ? xstrdup(h->url) : NULL;
	header->proto = (h->proto != NULL) ? xstrdup(h->proto) : NULL;
	header->host = (h->host != NULL) ? xstrdup(h->host) : NULL;
	header->file = (h->file != NULL) ? xstrdup(h->file) : NULL;
	header->username = (h->username != NULL) ? xstrdup(h->username) : NULL;
	header->password = (h->password != NULL) ? xstrdup(h->password) : NULL;
	header->method = (h->method != NULL) ? xstrdup(h->method) : NULL;
	header->version = h->version;
	header->code = h->code;
	header->type = h->type;
	header->port = h->port;
	header->header = h->header;
	header->header_filtered = h->header_filtered;
	header->flags = h->flags;
	header->content_length = h->content_length;
	header->chunked = h->chunked;
	header->keepalive = h->keepalive;
	header->proxy_keepalive = h->proxy_keepalive;
	header->content_type = (h->content_type != NULL) ? xstrdup(h->content_type) : NULL;
	header->content_encoding = (h->content_encoding != NULL) ? xstrdup(h->content_encoding) : NULL;
	header->accept_encoding = (h->accept_encoding != NULL) ? xstrdup(h->accept_encoding) : NULL;
	header->location = (h->location != NULL) ? xstrdup(h->location) : NULL;
	header->cache_control = (h->cache_control != NULL) ? xstrdup(h->cache_control) : NULL;
	header->host_header = (h->host_header != NULL) ? xstrdup(h->host_header) : NULL;
	header->authorization = (h->authorization != NULL) ? xstrdup(h->authorization) : NULL;
	header->range = (h->range != NULL) ? xstrdup(h->range) : NULL;
	header->proxy_authenticate = (h->proxy_authenticate != NULL) ? xstrdup(h->proxy_authenticate) : NULL;
	header->proxy_authorization = (h->proxy_authorization != NULL) ? xstrdup(h->proxy_authorization) : NULL;
	header->referer = (h->referer != NULL) ? xstrdup(h->referer) : NULL;
	header->expires = (h->expires != NULL) ? xstrdup(h->expires) : NULL;
	header->if_modified_since = (h->if_modified_since != NULL) ? xstrdup(h->if_modified_since) : NULL;
	header->last_modified = (h->last_modified != NULL) ? xstrdup(h->last_modified) : NULL;
	header->warning = (h->warning != NULL) ? xstrdup(h->warning) : NULL;
	header->age = (h->age != NULL) ? xstrdup(h->age) : NULL;
	header->x_cache = (h->x_cache != NULL) ? xstrdup(h->x_cache) : NULL;
	header->xforwardedfor = (h->xforwardedfor != NULL) ? xstrdup(h->xforwardedfor) : NULL;
	header->via = (h->via != NULL) ? xstrdup(h->via) : NULL;

	return header;
}

int header_length(char *buf, size_t len)
{
	int x = 0, nlcount = 0;

	for (; len && buf[x]; x++, len--) {
		if (buf[x] == '\n')
			nlcount++;
		else if (buf[x] != '\r')
			nlcount = 0;

		if (nlcount == 2)
			return x + 1;
	}

	return -1;
}

/*
parse options prefixed to a url, these can be stacked.. i.e. debug..bypass[r]..host.com
*/
struct url_command_t **url_command_parse(char *url)
{
	int count = 0;
	char *end, *ptr, *ptr2;
	struct url_command_t **ret = NULL;

	while ((end = strstr(url, ".."))) {
		ret = (url_command_t**)xrealloc(ret, (count + 2) * sizeof(struct url_command_t *));
		ret[count] = (url_command_t*)xmalloc(sizeof(struct url_command_t));

		ret[count + 1] = NULL;

		ptr = (char*)memchr(url, '[', end - url);
		ptr2 = NULL;

		if (ptr != NULL) {
			ptr2 = (char*)memchr(ptr, ']', end - url);
			if (ptr2 != NULL)
				ret[count]->options = xstrndup(&ptr[1], ptr2 - ptr - 1);
			else
				ret[count]->options = NULL;
		} else
			ret[count]->options = NULL;

		ret[count]->command = xstrndup(url, (ptr != NULL) ? ptr - url : end - url);

		ptr = (ptr2 != NULL) ? &ptr2[3] : &end[2];
		memmove(url, ptr, strlen(ptr) + 1);

		count++;
	}

	return ret;
}

char *url_command_create(struct url_command_t **url_command)
{
	char *ret;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	for (; *url_command; url_command++) {
		if ((*url_command)->options != NULL)
			filebuf->Addf("%s[%s]..", (*url_command)->command, (*url_command)->options);
		else
			filebuf->Addf("%s..", (*url_command)->command);
	}

	filebuf->Add("", 1);

	filebuf->Shorten();
	ret = filebuf->data;
	filebuf->data = NULL;

	xdelete filebuf;

	return ret;
}

void url_command_free(struct url_command_t **url_command)
{
	struct url_command_t **tmp = url_command;

	while (*tmp != NULL) {
		FREE_AND_NULL((*tmp)->command);
		FREE_AND_NULL((*tmp)->options);

		xfree(*tmp);

		tmp++;
	}

	xfree(url_command);
}

/* prefix URL command to a URL */
char *url_command_add(CONNECTION * connection, char *url)
{
	char *uc, *ptr, *ptr2, buf[4096];

	uc = url_command_create(connection->url_command);

	ptr = strstr(url, "://");
	if (ptr == NULL) {
		if (*url != '/') {
			/* relative path for file on same server */
			ptr2 = buf;

			ptr2 += snprintf(ptr2, sizeof(buf), "%s://%s%s:%d", connection->header->proto, uc, connection->header->host, connection->header->port);

			ptr = strrchr(connection->header->file, '/');

			s_strncpy(ptr2, connection->header->file, (sizeof(buf) - (ptr2 - buf) < ptr - connection->header->file + 1) ? sizeof(buf) - (ptr2 - buf) + 1 : ptr - connection->header->file + 2);
			ptr2 += strlen(ptr2);

			s_strncpy(ptr2, url, sizeof(buf) - (ptr2 - buf) + 1);
		} else {
			/* absolute path for file on same server */
			snprintf(buf, sizeof(buf), "%s://%s%s:%d%s", connection->header->proto, uc, connection->header->host, connection->header->port, url);
		}
	} else {
		ptr += 3;
		snprintf(buf, sizeof(buf), "%s://%s%s", connection->header->proto, uc, ptr);
	}

	xfree(uc);

	return xstrdup(buf);
}

struct url_command_t *url_command_find(struct url_command_t **url_command, char *command)
{
	if (url_command == NULL)
		return NULL;

	for (; *url_command; url_command++)
		if (!strcasecmp((*url_command)->command, command))
			return *url_command;

	return NULL;
}

void xcache_header_update(CONNECTION *connection, int flags) {
	char buf[1024];

	if (connection->rheader->x_cache != NULL)
		snprintf(buf, sizeof(buf), "%s, %s from Middleman", connection->rheader->x_cache, (flags == HIT) ? "HIT" : "MISS");
	else
		snprintf(buf, sizeof(buf), "%s from Middleman", (flags == HIT) ? "HIT" : "MISS"); 

	FREE_AND_STRDUP(connection->rheader->x_cache, buf);
}

void via_header_update(CONNECTION *connection) {
	char buf[1024];

	if (connection->header->via != NULL)
		snprintf(buf, sizeof(buf), "%s, %s:%d (Middleman %s)", connection->header->via, connection->interface, connection->port, VERSION);
	else
		snprintf(buf, sizeof(buf), "%s:%d (Middleman %s)", connection->interface, connection->port, VERSION);

	FREE_AND_STRDUP(connection->header->via, buf);
}

void conditional_header(HEADER *header, CACHEMAP *cachemap) {
	char buf[128];
	time_t t;
	struct tm tt;

	t = cache_section->atomic_read(&cachemap->mtime);
	localtime_r(&t, &tt);

	strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &tt);

	FREE_AND_NULL(header->if_modified_since);
	header->if_modified_since = xstrdup(buf);
}

int url_command_extract(CONNECTION * connection)
{
        char buf[4096], *ptr, *ptr2;
        struct url_command_t **url_command;
        HEADER *header;

	if (connection->url_command != NULL) {
		url_command_free(connection->url_command);
		connection->url_command = NULL;
	}

        if (connection->access & ACCESS_URLCOMMAND) {
                /* parse URL command */
                if (connection->header->type != HTTP_REQUEST) {
                        /* take URL command from beginning of Host */
                        connection->url_command = url_command_parse(connection->header->host);
			if (connection->url_command != NULL) {
				snprintf(buf, sizeof(buf), "%s://%s:%d%s", connection->header->proto, connection->header->host, connection->header->port, connection->header->file);
				FREE_AND_STRDUP(connection->header->url, buf);
			}
                } else if (connection->header->host_header != NULL)
                        /* take URL command from beginning of Host header (transparent proxy) */
                        connection->url_command = url_command_parse(connection->header->host_header);
                else {
                        ptr2 = strchr(&connection->header->file[1], '/');
                        if (ptr2 != NULL) {
                                /* take url command from beginning of file if this is an http request */
                                ptr = url_decode(&connection->header->file[1], ptr2 - connection->header->file - 1);
                                connection->url_command = url_command_parse(ptr);
                                xfree(ptr);

                                if (connection->url_command != NULL) {
                                        memmove(connection->header->file, ptr2, strlen(ptr2) + 1);

                                        connection->header->file = (char*)xrealloc(connection->header->file, strlen(connection->header->file) + 1);
                                }
                        }
                }

                if (connection->header->referer != NULL) {
                        /* get url command from Referer header sent by browser */
                        ptr2 = strstr(connection->header->referer, "://");
                        if (ptr2 != NULL)
                                ptr2 += 3;
                        else
                                ptr2 = connection->header->referer;

                        ptr = url_decode(ptr2, strlen(ptr2));

                        url_command = url_command_parse(ptr);

                        if (url_command != NULL) {
                                /* get rid of URL command from Referer string */
                                xfree(connection->header->referer);

                                /* this will be encoded incorrectly, I doubt many sites care though */
                                ptr2 = url_encode(ptr, strlen(ptr));
                                snprintf(buf, sizeof(buf), "%s://%s", connection->header->proto, ptr2);
                                xfree(ptr2);

                                connection->header->referer = xstrdup(buf);

                                /* this all needs to be done even if we've already got a url command from the URL
                                   to get rid of the URL command from the Referer URL */
                                if (connection->url_command == NULL) {
                                        if (!strcasecmp(connection->header->method, "GET") && (connection->header->host != NULL || connection->header->host_header != NULL)) {
                                                /* send redirect so URL command is used on new page as well */
                                                ptr2 = url_command_create(url_command);
                                                snprintf(buf, sizeof(buf), "%s://%s%s:%d%s", connection->header->proto, ptr2, (connection->header->host != NULL) ? connection->header->host : connection->header->host_header, connection->header->port, connection->header->file);
                                                xfree(ptr2);

                                                header = header_new();
                                                header->type = HTTP_RESP;
                                                header->code = 302;
						header->flags |= HEADER_CL;
                                                header->content_length = 0;
                                                header->location = xstrdup(buf);

                                                header_send(header, connection, CLIENT, HEADER_RESP);

                                                http_header_free(header);

                                                xfree(ptr);

                                                url_command_free(url_command);

                                                return FALSE;
                                        } else
                                                connection->url_command = url_command;
                                } else
                                        url_command_free(url_command);
                        }

                        xfree(ptr);
                }
        }

        return TRUE;
}

