/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <new>
#include "proto.h"

#ifdef USE_SYSLOG
#include <syslog.h>
#endif				/* USE_SYSLOG */

#ifdef _DEBUG
int loglevel = MMLOG_DEBUG * 2 - 1;
#else
int loglevel = MMLOG_DEBUG - 1;
#endif
int daemonize = TRUE;

#ifdef _DEBUG_MEMORY
int mstart = FALSE;
MINFO marray[2][MMAX];
pthread_rwlock_t mlock;
#endif

char configfile[256] = "";
char sectionfile[256] = "";

char logfile[256] = "";
char pidfile[256] = "";

char username[256] = "";
char group[256] = "";

char proxyhost[256] = "";

char masterhost[256] = "";
int masterport;

char listenip[16] = "";
int listenport;

char syncsections[256] = "";
char nosyncsections[256] = "";

int slave = FALSE;
int syncinterval = 60;

THREADLIST threads[MAXTHREADS];

TiXmlDocument *document = NULL;
pthread_mutex_t documentlock = PTHREAD_MUTEX_INITIALIZER;
time_t documentsynctime;

GLOBAL *global = NULL;
int gmtoffset;

SectionMap Section::map;

AccessSection *access_section;
FilterSection *filter_section;
PrefetchSection *prefetch_section;
CacheSection *cache_section;
GeneralSection *general_section;
DnsblSection *dnsbl_section;
NetworkSection *network_section;
HeaderSection *header_section;
TemplateSection *template_section;
CookieSection *cookie_section;
RewriteSection *rewrite_section;
ExternalSection *external_section;
FtpSection *ftp_section;
ProfileSection *profile_section;
MimeSection *mime_section;
RedirectSection *redirect_section;
KeywordSection *keyword_section;
ForwardSection *forward_section;
LimitSection *limit_section;
AntivirusSection *antivirus_section;

int main(int argc, char **argv)
{
	int fpid;
	int x;
	struct stat fileinfo;
	char *ptr;

	std::set_new_handler(out_of_memory);

	srand(time(NULL) + getpid());

	if (argc < 2) {
		show_help(argv);
		exit(EXIT_SUCCESS);
	}

	while ((x = getopt(argc, argv, "hfp:L:H:M:S:E:I:c:s:l:d:u:g:")) != EOF) {
		switch (x) {
		case 'c':
			s_strncpy(configfile, optarg, sizeof(configfile));
			break;
		case 's':
			s_strncpy(sectionfile, optarg, sizeof(sectionfile));
			break;
		case 'l':
			s_strncpy(logfile, optarg, sizeof(logfile));
			break;
		case 'p':
			s_strncpy(pidfile, optarg, sizeof(pidfile));
			break;
		case 'd':
			loglevel = atoi(optarg);
			break;
		case 'u':
			s_strncpy(username, optarg, sizeof(username));
			break;
		case 'g':
			s_strncpy(group, optarg, sizeof(group));
			break;
		case 'f':
			daemonize = FALSE;
			break;
		case 'M':
			ptr = strchr(optarg, ':');
			if (ptr != NULL && ptr - optarg < sizeof(masterhost)) {
				slave = TRUE;

				s_strncpy(masterhost, optarg, (ptr - optarg) + 1);
				masterport = atoi(ptr + 1);
			} else {
				show_help(argv);
				exit(EXIT_SUCCESS);
			}

			break;
		case 'L':
			ptr = strchr(optarg, ':');
			if (ptr != NULL && ptr - optarg < sizeof(listenip)) {
				s_strncpy(listenip, optarg, (ptr - optarg) + 1);
				listenport = atoi(ptr + 1);
			} else {
				show_help(argv);
				exit(EXIT_SUCCESS);
			}

			break;
		case 'S':
			s_strncpy(syncsections, optarg, sizeof(syncsections));
			break;
		case 'E':
			s_strncpy(nosyncsections, optarg, sizeof(nosyncsections));
			break;
		case 'I':
			syncinterval = atoi(optarg);
			break;
		case 'H':
			s_strncpy(proxyhost, optarg, sizeof(proxyhost));
			break;
		case 'h':
			show_help(argv);
			exit(EXIT_SUCCESS);
		}
	}

	while (--argc > 0)
		memset(argv[argc], 0, strlen(argv[argc]));

	if (*configfile) {
		if (stat(configfile, &fileinfo) == -1) {
			fprintf(stderr, "couldn't stat %s\n", configfile);
			exit(EXIT_FAILURE);
		}
	} else if (slave == FALSE) {
		fprintf(stderr, "config file option missing\n");
		exit(EXIT_FAILURE);
	}

	if (*sectionfile) {
		if (stat(sectionfile, &fileinfo) == -1) {
			fprintf(stderr, "couldn't stat %s\n", sectionfile);
			exit(EXIT_FAILURE);
		}
	} else {
		fprintf(stderr, "section file option missing\n");
		exit(EXIT_FAILURE);
	}

	/* Riadh: I am used to debug Middleman with gdb, so I don't daimonize
	   the process in debug mode */

	if (daemonize == TRUE) {
		fpid = fork();

		switch (fpid) {
		case -1:
			fprintf(stderr, "failed to fork daemon: first fork failed\n");
			exit(EXIT_FAILURE);
			break;
		case 0:
			chdir("/");
			setsid();
			umask(0);
			close(STDIN_FILENO);
			close(STDOUT_FILENO);
			close(STDERR_FILENO);

			fpid = fork();

			switch (fpid) {
			case -1:
				fprintf(stderr, "failed to fork daemon: second fork failed\n");
				exit(EXIT_FAILURE);
				break;
			case 0:
				umask(022);
				fix_gid();
				fix_uid();
				if (*pidfile) {
					x = pid_check(pidfile);

					switch (x) {
					case 0:
						break;
					case -1:
						exit(EXIT_FAILURE);
					default:
						exit(EXIT_FAILURE);
					}
				}
				umask(027);
				mainloop();
				break;
			}
		}
	} else {
		fix_gid();
		fix_uid();
		umask(027);
		mainloop();
	}

	exit(EXIT_SUCCESS);
}

void fix_uid()
{
	int x;
	struct passwd *pwd = NULL;

	if (strcmp(username, "")) {
		pwd = getpwnam(username);
		if (pwd == NULL) {
			fprintf(stderr, "getpwnam: unknown user\n");
			exit(EXIT_FAILURE);
		}

		x = setuid(pwd->pw_uid);
		if (x == -1) {
			perror("setuid");
			exit(EXIT_FAILURE);
		}
	} else {
		pwd = getpwuid(getuid());
		if (pwd == NULL)
			snprintf(username, sizeof(username), "%d", getuid());
		else
			s_strncpy(username, pwd->pw_name, sizeof(username));
	}

}

void fix_gid()
{
	int x;
	struct group *grp = NULL;

	if (strcmp(group, "")) {
		grp = getgrnam(group);
		if (grp == NULL) {
			fprintf(stderr, "getgrnam: unknown group\n");
			exit(EXIT_FAILURE);
		}

		x = setgid(grp->gr_gid);
		if (x == -1) {
			perror("setgid");
			exit(EXIT_FAILURE);
		}
	} else {
		grp = getgrgid(getgid());
		if (grp == NULL)
			snprintf(group, sizeof(group), "%d", getgid());
		else
			s_strncpy(group, grp->gr_name, sizeof(group));
	}
}

void show_help(char **argv)
{
	fprintf(stderr, "MiddleMan filtering proxy server v%s\n\n", VERSION);
	fprintf(stderr, "Usage: %s [options]\n\n", argv[0]);
	fprintf(stderr, " -c <file>      : location of config file (required)\n");
	fprintf(stderr, " -s <file>      : location of section file (required)\n");
#ifndef USE_SYSLOG
	fprintf(stderr, " -l <file>      : file to log actvity to\n");
#endif
	fprintf(stderr, " -H <hostname>  : proxy's hostname (overrides configuration file)\n");
	fprintf(stderr, " -L <ip:port>   : specify address and port to listen on\n");
	fprintf(stderr, " -M <host:port> : master host and port to syncronize configuration from\n");
	fprintf(stderr, " -S <sections>  : comma seperated list of sections to syncronize from master (default: all sections)\n");
	fprintf(stderr, " -E <sections>  : comma seperated list of sections to not syncronize from master (default: none)\n");
	fprintf(stderr, " -I <seconds>   : time interval between syncronization attempts to master\n");
	fprintf(stderr, " -p <file>      : PID file\n");
	fprintf(stderr, " -u <username>  : run as alternate user\n");
	fprintf(stderr, " -g <groupname> : run in alternate group\n");
	fprintf(stderr, " -d <level>     : set log level (default: %d)\n", loglevel);
	fprintf(stderr, " -f             : don't daemonize (log written to stdout)\n");
	fprintf(stderr, " -h             : help\n\n");
	fprintf(stderr, " Add any of the following to specify logging detail:\n");
	fprintf(stderr, " %7d = requests\n", MMLOG_REQUEST);
	fprintf(stderr, " %7d = network\n", MMLOG_NETWORK);
	fprintf(stderr, " %7d = url filtering\n", MMLOG_FILTER);
	fprintf(stderr, " %7d = header filtering\n", MMLOG_HEADER);
	fprintf(stderr, " %7d = mime filtering\n", MMLOG_MIME);
	fprintf(stderr, " %7d = cookie filtering\n", MMLOG_COOKIE);
	fprintf(stderr, " %7d = redirections\n", MMLOG_REDIRECT);
	fprintf(stderr, " %7d = templates\n", MMLOG_TEMPLATE);
	fprintf(stderr, " %7d = keyword filtering\n", MMLOG_KEYWORDS);
	fprintf(stderr, " %7d = rewriting\n", MMLOG_REWRITE);
	fprintf(stderr, " %7d = limits\n", MMLOG_LIMITS);
	fprintf(stderr, " %7d = caching\n", MMLOG_CACHE);
	fprintf(stderr, " %7d = prefetching\n", MMLOG_PREFETCH);
	fprintf(stderr, " %7d = ICP\n", MMLOG_ICP);
	fprintf(stderr, " %7d = forwarding\n", MMLOG_FORWARD);
	fprintf(stderr, " %7d = config syncronization\n", MMLOG_SYNC);
	fprintf(stderr, " %7d = antivirus\n", MMLOG_ANTIVIRUS);
	fprintf(stderr, " %7d = external parsers\n", MMLOG_EXTERNAL);
	fprintf(stderr, " %7d = DNS blacklist\n", MMLOG_DNSBL);
	fprintf(stderr, " %7d = security\n", MMLOG_SECURITY);
	fprintf(stderr, " %7d = warnings\n", MMLOG_WARN);
	fprintf(stderr, " %7d = errors\n", MMLOG_ERROR);
	fprintf(stderr, " %7d = debug\n", MMLOG_DEBUG);
}

/*
check if pidfile exists and whether or not the pid inside is active, otherwise
write current pid.
*/
int pid_check(char *pidfile)
{
	int i = 0, x;
	FILE *fptr;

	fptr = fopen(pidfile, "r");
	if (fptr == NULL)
		goto makepidfile;

	x = fscanf(fptr, "%d", &i);
	if (x == 0) {
		fclose(fptr);
		goto makepidfile;
	}

	x = kill(i, SIGCHLD);
	if (x == 0) {
		fclose(fptr);
		return i;
	}

      makepidfile:
	unlink(pidfile);
	fptr = fopen(pidfile, "w");
	if (fptr == NULL)
		return TRUE;

	fprintf(fptr, "%u\n", (unsigned int) getpid());

	fclose(fptr);
	return FALSE;
}

/*
things that only need to be done once at startup
*/
void config()
{
	int i, x;
	time_t t, curtime = time(NULL);
	struct tm tt;
	char buf[1024];

	gmtime_r(&curtime, &tt);
	t = mktime(&tt);

	/* tt.tm_isdst can also be -1 */
	gmtoffset = t - curtime - (tt.tm_isdst == 1) ? tt.tm_isdst : 0;

#ifdef HAVE_SSL
	SSL_library_init();
	SSL_load_error_strings();	// I don't think this is necessary
	SSLeay_add_ssl_algorithms();
	SSL_thread_setup();

	// XXX: a better random source is needed
	for (x = 0; x < sizeof(buf); x++)
		buf[x] = rand() % 255;

	RAND_seed(buf, sizeof(buf));
#endif

#ifdef USE_SYSLOG
	openlog("mman", LOG_PID, LOG_DAEMON);
#endif

	global = xnew GLOBAL;;

	pthread_mutex_init(&global->rusage_lock, NULL);
	memset(&global->rusage, 0, sizeof(global->rusage));

	global->logbuffer = (LOGBUFFER *) xmalloc(sizeof(LOGBUFFER));
	global->logbuffer->entries = 0;
	global->logbuffer->size = LOGBUFFERSIZE;
	global->logbuffer->head = global->logbuffer->tail = NULL;
	pthread_rwlock_init(&global->logbuffer->lock, NULL);

	access_section = xnew AccessSection;
	filter_section = xnew FilterSection;
	prefetch_section = xnew PrefetchSection;
	cache_section = xnew CacheSection;
	general_section = xnew GeneralSection;
	dnsbl_section = xnew DnsblSection;
	network_section = xnew NetworkSection;
	header_section = xnew HeaderSection;
	template_section = xnew TemplateSection;
	cookie_section = xnew CookieSection;
	rewrite_section = xnew RewriteSection;
	external_section = xnew ExternalSection;
	ftp_section = xnew FtpSection;
	profile_section = xnew ProfileSection;
	mime_section = xnew MimeSection;
	redirect_section = xnew RedirectSection;
	keyword_section = xnew KeywordSection;
	forward_section = xnew ForwardSection;
	limit_section = xnew LimitSection;
	antivirus_section = xnew AntivirusSection;

#ifdef _DEBUG
	pcre_free = free;
	pcre_malloc = malloc;
#else
	pcre_free = xfree;
	// grr... argument to malloc is int on some systems, and long on others
	(void *) pcre_malloc = (void *) xmalloc;
#endif

	global->dns_cache = hash_create(DNS_HASH_SIZE);
	pthread_mutex_init(&global->dns_cache_lock, NULL);

	pthread_mutex_init(&global->unlink_lock, NULL);
	pthread_cond_init(&global->unlink_cond, NULL);
	global->unlink_list = xnew UnlinkList;

	global->pool = pool_init();
	//global->timers = timers_init();

	global->stats.AddGroup("system", StatGroup::NORMAL);
	global->stats.AddCounter("system", "user time", stat_usertime, StatCounter::STRING);
	global->stats.AddCounter("system", "system time", stat_systime, StatCounter::STRING);
	global->stats.AddCounter("system", "memory resident", stat_resident, StatCounter::FILESIZE);
	global->stats.AddCounter("system", "memory size", stat_size, StatCounter::FILESIZE);
	global->stats.AddCounter("system", "memory shared", stat_share, StatCounter::FILESIZE);
	global->stats.AddCounter("system", "minor pagefaults", stat_minflt, StatCounter::UINT);
	global->stats.AddCounter("system", "major pagefaults", stat_majflt, StatCounter::UINT);

	global->stats.AddGroup("network", StatGroup::NORMAL);
	global->stats.AddCounter("network", "successful connections", StatCounter::UINT);
	global->stats.AddCounter("network", "failed connections", StatCounter::UINT);
	global->stats.AddCounter("network", "DNS failures", StatCounter::UINT);
	global->stats.AddCounter("network", "bytes in", StatCounter::FILESIZE);
	global->stats.AddCounter("network", "bytes out", StatCounter::FILESIZE);

	global->stats.AddGroup("DNS cache", StatGroup::RATIO);
	global->stats.AddCounter("DNS cache", "hit", StatCounter::UINT);
	global->stats.AddCounter("DNS cache", "miss", StatCounter::UINT);

	global->stats.AddGroup("cache", StatGroup::RATIO);
	global->stats.AddCounter("cache", "hit", StatCounter::UINT);
	global->stats.AddCounter("cache", "miss", StatCounter::UINT);

	global->stats.AddGroup("connection pool", StatGroup::RATIO);
	global->stats.AddCounter("connection pool", "hit", StatCounter::UINT);
	global->stats.AddCounter("connection pool", "miss", StatCounter::UINT);

	signal_setup();
	net_init();

	for (i = 0; i < MAXTHREADS; i++) {
		threads[i].flags = THREAD_UNUSED;
		pthread_mutex_init(&threads[i].lock, NULL);
		pthread_cond_init(&threads[i].cond, NULL);
		threads[i].arg = NULL;
	}

	if (configfile[0])
		config_load(3, configfile);

	if (slave) {
		config_sync();
		thread_create((void *) sync_thread, NULL);
	}

	if (listenip[0])
		net_listen(listenport, listenip);

	network_section->check();

	prefetch_section->init_threads();
	thread_create((void *) icp_thread, NULL);
	thread_create((void *) unlink_thread, NULL);

	putlog(MMLOG_DEBUG, "GMT offset is -%u seconds", gmtoffset);
}

int config_load(int overwrite, const string file)
{
	pthread_mutex_lock(&documentlock);

	xdelete document;
	document = xnew TiXmlDocument(file.c_str());
	bool loadOkay = document->LoadFile();

	ASSERT(loadOkay);

	SectionMap::const_iterator iter;
	for (iter = Section::map.begin(); iter != Section::map.end(); iter++) {
		iter->second->load(document, overwrite);
	}

	pthread_mutex_unlock(&documentlock);

	if (slave == FALSE) documentsynctime = time(NULL);

	return TRUE;
}

/* 
main event loop; accept connections, check access list, then create thread to
continue connection.
*/
void mainloop()
{
	int x;
	CONNECTION *connection;

	config();

#ifdef _DEBUG_MEMORY
	/* Start debugging dynamic memory allocation */
	pthread_rwlock_init(&mlock, NULL);
	pthread_rwlock_wrlock(&mlock);
	mstart = TRUE;
	pthread_rwlock_unlock(&mlock);
#endif

	while (1) {
		connection = net_accept(1);

		//timers_check(global->timers);
		limit_section->reset();

		if (connection != NULL) {
			if (access_section->check_and_setup(connection)) {
				x = process_new(connection);
				if (x != 0) {
					putlog(MMLOG_ERROR, "failed to create thread for %s", connection->ip);
					net_close(connection);
				}
			} else {
				net_close(connection);
			}
		}

		pool_clean(global->pool, 0);
	}
}

/*
create new thread
*/
int process_new(CONNECTION * connection)
{
	int thread = -1, i;

	for (i = 0; i < MAXTHREADS; i++) {
		pthread_mutex_lock(&threads[i].lock);
		if (threads[i].flags == THREAD_WAITING) {
			threads[i].flags = THREAD_IDLE;

			connection->thread = i;

			threads[i].arg = connection;
			pthread_cond_signal(&threads[i].cond);

			pthread_mutex_unlock(&threads[i].lock);

			return 0;
		}

		if (thread == -1 && threads[i].flags == THREAD_UNUSED)
			thread = i;

		pthread_mutex_unlock(&threads[i].lock);
	}

	if (thread == -1)
		return -1;
	else {
		threads[thread].flags = THREAD_IDLE;
		threads[thread].host = NULL;
		threads[thread].file = NULL;
		threads[thread].method = NULL;
		threads[thread].port = 0;
		threads[thread].requests = 0;

		threads[thread].ip = xstrdup(connection->ip);
	}

	connection->thread = thread;

	return thread_create((void *) process_entry, connection);
}

int thread_create(void *func, void *arg)
{
	int perr;
	pthread_t thread_id;
	pthread_attr_t thread_attr;

	pthread_attr_init(&thread_attr);
#ifndef _DEBUG
	//pthread_attr_setstacksize(&thread_attr, 1024 * 1024);
#endif
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);

	perr = pthread_create(&thread_id, &thread_attr, (void *(*)(void *)) func, arg);

	pthread_attr_destroy(&thread_attr);

	return perr;
}

/*
entry function for new threads
*/
void process_entry(CONNECTION * connection)
{
	int x, ret;
	char *headbuf = NULL, *ptr, buf[8192];
	Filebuf *filebuf;
	HEADER *header;
	HOSTENT *hostent;
	struct timespec at;

      newconnection:
	/* write log message here so the pid matches */
	putlog(MMLOG_NETWORK, "allowed connect from %s on port %d", connection->ip, connection->port);

	pthread_mutex_lock(&threads[connection->thread].lock);
	threads[connection->thread].pid = (unsigned int) getpid();
	threads[connection->thread].flags = THREAD_BUSY;
	FREE_AND_STRDUP(threads[connection->thread].ip, connection->ip);
	pthread_mutex_unlock(&threads[connection->thread].lock);

	connection->variables["IP"] = connection->ip;
	snprintf(buf, sizeof(buf), "%d", connection->port);
	connection->variables["PORT"] = buf;
	connection->variables["INTERFACE"] = connection->interface;


	while (1) {
		profiles_reset(connection);

		headbuf = header_get(connection, CLIENT, (connection->request) ? general_section->keeptimeout_get() : general_section->timeout_get());
		if (headbuf == NULL) {
			if (!connection->request) {
				putlog(MMLOG_WARN, "timeout waiting for header from %s", connection->ip);

				template_section->send("badrequest", connection, 400);
			}

			break;
		}

		/* this needs to be parsed twice; the first time to get the host/file, and the second
		   to let the rewrite rules take effect */
		connection->header = http_header_parse_request(headbuf);
		if (connection->header == NULL) {
			if (!connection->request) {
				putlog(MMLOG_WARN, "invalid header reveived from %s", connection->ip);

				template_section->send("badrequest", connection, 400);
			}

			xfree(headbuf);

			break;
		}

		/* this is done here to make it possible to have URL commands that
		   enable certain profile entries, which then may activate entries
		   that modify the header. */
		if (url_command_extract(connection) == FALSE) {
			http_header_free(connection->header);
			xfree(headbuf);
			break;
		}

		bypass_check(connection);
		profile_section->profiles_update(connection);
		http_header_free(connection->header);

		/* pass the client header through the rewrite rules before parsing */
		filebuf = xnew Filebuf();
		filebuf->data = headbuf;
		filebuf->size = strlen(headbuf) + 1;

		rewrite_section->rewrite_do(connection, filebuf, REWRITE_CLIENT, TRUE);

		headbuf = filebuf->data;
		filebuf->data = NULL;

		xdelete filebuf;

		/* now parse the (possibly) rewritten header */
		connection->header = http_header_parse_request(headbuf);
		xfree(headbuf);
		if (connection->header == NULL) {
			if (!connection->request) {
				putlog(MMLOG_WARN, "invalid header reveived from %s", connection->ip);

				template_section->send("badrequest", connection, 400);
			}

			break;
		}


		/* this is done again to strip URL command from request */
		if (url_command_extract(connection) == FALSE) {
			http_header_free(connection->header);
			break;
		}

		if (connection->header->flags & HEADER_CL) {
                        /* the browser wants to send something to the remote site after the header (POST, PUT, etc.) */
                        putlog(MMLOG_DEBUG, "post length: %u", connection->header->content_length);

                        /* this needs to be buffered.. otherwise we won't have any POST data
                           after authentication */
                        connection->postbody = http_transfer_filebuf(connection, CLIENT, ~0);
                        rewrite_section->rewrite_do(connection, connection->postbody, REWRITE_POST, TRUE);

                        connection->header->chunked = FALSE;
                        connection->header->content_length = connection->postbody->size;
                }

		if (connection->header->host != NULL) connection->variables["HTTP_HOST"] = connection->header->host;
		connection->variables["HTTP_FILE"] = connection->header->file;
		connection->variables["HTTP_METHOD"] = connection->header->method;
		snprintf(buf, sizeof(buf), "%d", connection->header->port);
		connection->variables["HTTP_PORT"] = buf;

		/* determine if the connection should be kept alive with the information gathered
		   so far, this is incase a template or web interface is used */
		if (connection->header->type == HTTP_CONNECT)
			connection->keepalive_client = FALSE;
		else if (connection->header->keepalive == FALSE || connection->header->proxy_keepalive == FALSE)
			connection->keepalive_client = FALSE;
		else if (connection->header->proxy_keepalive == TRUE || connection->header->keepalive == TRUE)
			connection->keepalive_client = TRUE;
		else if (connection->header->version == HTTP_HTTP11)
			connection->keepalive_client = TRUE;
		else
			connection->keepalive_client = FALSE;

		if (connection->authenticate == TRUE && connection->header->proxy_authorization == NULL) {
			header = header_new();
			header->type = HTTP_RESP;
			header->code = 407;
			header->content_length = 0;
			header->flags |= HEADER_CL;
			header->proxy_authenticate = xstrdup("Basic");

			header_send(header, connection, CLIENT, HEADER_RESP);

			http_header_free(header);

			goto skip;
		} else if (connection->header->proxy_authorization != NULL) {
			ptr = strchr(connection->header->proxy_authorization, ' ');
			if (ptr != NULL) {
				ptr++;

				/* buf is large enough to hold the maximum possible string
				   length since a header line can only be 4kb */
				ret = from64tobits(buf, ptr);
				buf[ret] = '\0';

				ptr = strchr(buf, ':');
				if (ptr != NULL) {
					*ptr++ = '\0';

					access_section->check_and_setup(connection, buf, ptr);
				}
			}

			if (connection->authenticate == TRUE) {
				header = header_new();
				header->type = HTTP_RESP;
				header->code = 407;
				header->content_length = 0;
				header->flags |= HEADER_CL;
				header->proxy_authenticate = xstrdup("Basic");

				header_send(header, connection, CLIENT, HEADER_RESP);

				http_header_free(header);

				goto skip;
			}
		}
		// ICAP: reqmod precache processing point

		if (url_command_find(connection->url_command, "profiles")) {
			show_profiles(connection);

			goto skip;
		}

		/* redirect the request if any matching rules found (redirect_do will fill in the host, file and port
		   members of the connection struct if any are found) */
		x = redirect_section->redirect_do(connection, REDIRECT_REQUEST);
		if (x == TRUE) {
			/* 302 redirect sent, no need to continue */
			goto skip;
		}

		if (url_command_find(connection->url_command, "https")) {
			/* if a port wasn't explicity given, change it to the HTTPS port */
			if (connection->header->flags & HEADER_DPORT)
				connection->header->port = HTTPS_PORT;

			/* this connection can't be forwarded to a regular proxy. */
			connection->flags |= CONNECTION_NOPROXY | CONNECTION_SSLCLIENT;
		}


		pthread_mutex_lock(&threads[connection->thread].lock);
		FREE_AND_STRDUP(threads[connection->thread].host, connection->header->host);
		FREE_AND_STRDUP(threads[connection->thread].file, connection->header->file);
		FREE_AND_STRDUP(threads[connection->thread].method, connection->header->method);
		threads[connection->thread].port = connection->header->port;
		pthread_mutex_unlock(&threads[connection->thread].lock);

		header_section->filter(connection, HEADER_CLIENT);

		if (connection->header->type != HTTP_CONNECT) {
			if ((connection->header->type != HTTP_REQUEST && !strcasecmp(connection->header->host, INTERFACEURL)) || (connection->header->type == HTTP_REQUEST && !strncasecmp(&connection->header->file[1], INTERFACEURL, strlen(INTERFACEURL)))) {
				/* request for web interface */
				putlog(MMLOG_REQUEST, "request for web interface from %s", connection->ip);

				interface_handle_request(connection);

				goto skip;
			}

			if ((connection->header->type == HTTP_REQUEST && !(connection->access & ACCESS_HTTP)) || (connection->header->type == HTTP_PROXY && !(connection->access & ACCESS_PROXY))) {
				template_section->send("noaccess", connection, 404);

				goto skip;
			}
		} else if (!(connection->access & ACCESS_CONNECT)) {
			template_section->send("noaccess", connection, 404);

			goto skip;
		} else {
			ret = general_section->cportrange_check(connection->header->port);

			if (ret == FALSE) {
				putlog(MMLOG_SECURITY, "denied CONNECT request to port %d", connection->header->port);

				template_section->send("noaccess", connection, 404);

				goto skip;
			}
		}

		if (connection->header->type == HTTP_REQUEST && connection->header->host == NULL) {
			/* this is a regular HTTP request */
			if ((connection->access & ACCESS_TRANSPARENT) && connection->header->host_header != NULL) {
				/* use Host: header if it's there */
				ptr = strchr(connection->header->host_header, ':');
				connection->header->host = xstrndup(connection->header->host_header, (ptr != NULL) ? ptr - connection->header->host_header : strlen(connection->header->host_header));
				if (ptr != NULL)
					connection->header->port = atoi(&ptr[1]);

				/* try redirecting again now that we've retrieved the host from the Host: header */
				x = redirect_section->redirect_do(connection, REDIRECT_REQUEST);
				if (x == TRUE) {
					/* 302 redirect sent */
					goto skip;
				}

				/* this feature causes a recursion where the proxy keeps
				   connecting to itself if an HTTP request is made to the proxy
				   which doesn't match a redirect rule and isn't a request for the web interface.
				   (the Host: header will have the proxies own hostname/port) 
				   There's no reliable way to detect this except to forbid connections
				   to websites on the same port as the proxy
				 */
				if (connection->header->port == connection->port) {
					template_section->send("nofile", connection, 404);

					goto skip;
				}

				snprintf(buf, sizeof(buf), "%s://%s:%d%s", connection->header->proto, connection->header->host, connection->header->port, connection->header->file);
				FREE_AND_STRDUP(connection->header->url, buf);

				connection->header->type = HTTP_PROXY;
			} else {
				/* not a request for web interface, no host header, and no matching redirect rule */
				template_section->send("nofile", connection, 404);
				goto skip;
			}
		}

		if (connection->header->host != NULL) connection->variables["HTTP_HOST"] = connection->header->host;
		connection->variables["HTTP_FILE"] = connection->header->file;
		connection->variables["HTTP_METHOD"] = connection->header->method;
		snprintf(buf, sizeof(buf), "%d", connection->header->port);
		connection->variables["HTTP_PORT"] = buf;

		struct url_command_t *uc;
		if (url_command_find(connection->url_command, "prefetch")) {
			URL *url = url_parse(connection->header->url);

			ASSERT(url != NULL);
			prefetch_section->queue_add(url, &connection->header->header, connection->transferlimit, 1);
			if ((uc = url_command_find(connection->url_command, "template")))
				template_section->send((uc->options != NULL) ? uc->options : "unknown", connection, 200);
			else
				template_section->send("unknown", connection, 404);

			goto skip;
		} else if ((uc = url_command_find(connection->url_command, "template"))) {
				template_section->send(uc->options, connection, 200);

				goto skip;
		}

		/* do the DNS lookup outside of any locks so it's cached when we do it in 
		   filter checking for IP address blocking. */
		hostent = net_dns(connection->header->host);
		if (hostent != NULL) 
			hostent_free(hostent);
		else {
			/* may as well save the trouble of going through the rest of the code. */
			template_section->send("nodns", connection, 404);
			goto skip;
		}

		if (url_command_find(connection->url_command, "filter")) {
			filter_section->check_show(connection);

			goto skip;
		}

		if (url_command_find(connection->url_command, "cache")) {
			cache_show(connection);

			goto skip;
		}

		if (filter_section->check_and_block(connection))
			goto skip;

		connection->transferred = 0;
		connection->transferlimit = ~0;

		if (limit_section->check_and_block(connection))
			goto skip;

		ret = dnsbl_section->check_and_block(connection);
		if (ret == FALSE) goto skip;

		if (url_command_find(connection->url_command, "offline"))
			connection->flags |= CONNECTION_OFFLINE;

		if (url_command_find(connection->url_command, "fresh") == NULL && cacheable(connection->header)) {
			connection->cachemap = cache_section->cache_open(connection, connection->header->url, NULL, (connection->flags & CONNECTION_OFFLINE) ? CACHE_OFFLINE : 0);

			if (connection->cachemap != NULL) {
				putlog(MMLOG_CACHE, "hit: %s", connection->cachemap->key);

				/* validate if the client is requesting an uncached copy */
				if (!(connection->flags & CONNECTION_OFFLINE)) {
					cache_section->check(connection->header);
					if (cache_section->atomic_read(&connection->cachemap->flags) & CACHE_VALIDATE) {
						// ICAP: reqmod postcache processing point

						connection->flags |= CONNECTION_VALIDATE;
						conditional_header(connection->header, connection->cachemap);
					}
				}
			} else {
				// ICAP: reqmod postcache processing point
			}

			/* this must be done here, since cache_open is sometimes called from cache_check as well */
			if (connection->cachemap != NULL) {
				global->stats.Increment("cache", "hit");
			} else {
				global->stats.Increment("cache", "miss");
			}
		}

		if (connection->flags & CONNECTION_OFFLINE) {
			if (connection->cachemap == NULL) {
				template_section->send("nocache", connection, 404);

				goto skip;
			}
		}

#ifdef HAVE_ZLIB
		if (general_section->compressin_get() == TRUE) {
			if (accepts_encoded_content(connection) == FALSE) {
				connection->flags |= CONNECTION_ENCODED;
				FREE_AND_STRDUP(connection->header->accept_encoding, ACCEPTED_ENCODINGS);
				putlog(MMLOG_DEBUG, "asking server for encoded content");
			}
		}
#endif				/* HAVE_ZLIB */

	      protocol_start:
		x = 0;

		if (connection->server == NULL && (connection->cachemap == NULL || (connection->flags & CONNECTION_VALIDATE))) {
			/* check if this request should be forwarded through another proxy */
			/* forward_do will fill in all the necessary members of the connection struct */
			connection->proxy_type = PROXY_DIRECT;
			connection->flags &= ~CONNECTION_LOGGEDIN;
			forward_section->forward_do(connection);

			x = protocol_start(connection);
		}

		if (x >= 0) {
			if (connection->header->type != HTTP_CONNECT) {
				if (url_command_find(connection->url_command, "proxytest")) {
					show_proxytest(connection);

					goto skip;
				}

				/* only process files retrieved through HTTP */
				if (!strcasecmp(connection->header->proto, "http"))
					connection->flags |= CONNECTION_PROCESS;

				if (!strcasecmp(connection->header->proto, "http") || connection->proxy_type == PROXY_NORMAL || (connection->cachemap != NULL && !(connection->flags & CONNECTION_VALIDATE)))
					x = protocol_http(connection);
				else if (!strcasecmp(connection->header->proto, "ftp"))
					x = protocol_ftp(connection);
				else {
					template_section->send("badprotocol", connection, 501);

					goto skip;
				}
			} else
				x = protocol_connect(connection);
		} else {
			if (connection->cachemap != NULL) {
				/* fall back on cache file if connection fails */
				connection->flags &= ~CONNECTION_VALIDATE;

				putlog(MMLOG_CACHE, "using unvalidated cache for failed connection");

				goto protocol_start;
			}

			template_section->send(error_to_template(x), connection, 503);

			putlog(MMLOG_HEADER, "error reading header from %s", connection->header->host);
		}

	      skip:

		if (connection->transferred > 0 && connection->transferlimit == 0)
			putlog(MMLOG_LIMITS, "transfer prematurely terminated due to transfer limit restrictions");

		limit_section->update(connection);

		connection->client->Flush();

		if (connection->flags & CONNECTION_FAILED) {
			connection->keepalive_server = FALSE;
			connection->keepalive_client = FALSE;

			if (connection->cachemap != NULL) {
				cache_section->invalidate(connection->cachemap);
				connection->cachemap = NULL;
			}
		}

		if (connection->cachemap != NULL) {
			cache_section->cache_close(connection->cachemap);
			connection->cachemap = NULL;
		}

		if (connection->url_command != NULL) {
			url_command_free(connection->url_command);
			connection->url_command = NULL;
		}

		if (connection->htmlstream != NULL) {
			htmlstream_free(connection->htmlstream);
			connection->htmlstream = NULL;
		}

		connection->request++;
		connection->flags = 0;

		if (connection->server != NULL) {
			if (connection->keepalive_server == TRUE)
				pool_add(global->pool, connection->server, connection->header->proto, (connection->proxy_type == PROXY_NORMAL && connection->proxy_host != NULL) ? connection->proxy_host : connection->header->host, (connection->proxy_type == PROXY_NORMAL) ? connection->proxy_port : connection->header->port, connection->header->username, connection->header->password);
			else
				xdelete connection->server;

			connection->server = NULL;
		}

		http_header_free(connection->header);

		putlog(MMLOG_DEBUG, "keepalive_client = %d", connection->keepalive_client);

		if (!connection->keepalive_client)
			break;

		pthread_mutex_lock(&threads[connection->thread].lock);
		FREE_AND_NULL(threads[connection->thread].host);
		FREE_AND_NULL(threads[connection->thread].file);
		FREE_AND_NULL(threads[connection->thread].method);
		threads[connection->thread].port = 0;
		threads[connection->thread].requests++;
		pthread_mutex_unlock(&threads[connection->thread].lock);

		connection->header = NULL;
		connection->rheader = NULL;

		FREE_AND_NULL(connection->proxy_host);
	}

	if (connection->server != NULL)
		xdelete connection->server;

	putlog(MMLOG_NETWORK, "%s disconnected after making %d requests", connection->ip, connection->request);

	x = connection->thread;
	net_close(connection);

	pthread_mutex_lock(&threads[x].lock);
	threads[x].flags = THREAD_WAITING;
	FREE_AND_NULL(threads[x].ip);
	FREE_AND_NULL(threads[x].host);
	FREE_AND_NULL(threads[x].file);
	FREE_AND_NULL(threads[x].method);
	threads[x].port = 0;
	threads[x].requests = 0;

	/* wait for another connection to handle */
	at.tv_sec = time(NULL) + THREAD_TIMEOUT;
	at.tv_nsec = 0;

	ret = pthread_cond_timedwait(&threads[x].cond, &threads[x].lock, &at);
	if (ret == ETIMEDOUT || threads[x].arg == NULL) {
		threads[x].flags = THREAD_UNUSED;
		pthread_mutex_unlock(&threads[x].lock);

		rusage_update();

		pthread_exit(0);
	}

	putlog(MMLOG_DEBUG, "reusing thread");
	connection = (CONNECTION *) threads[x].arg;
	threads[x].arg = NULL;
	pthread_mutex_unlock(&threads[x].lock);

	goto newconnection;
}

/*
save all config settings to a file
*/
int config_save(const string & filename)
{
	pthread_mutex_lock(&documentlock);

	config_xml_build();

	bool saveOkay = document->SaveFile(filename.c_str());

	pthread_mutex_unlock(&documentlock);

	return saveOkay;
}

void config_xml_build() {
	SectionMap::const_iterator iter;
	for (iter = Section::map.begin(); iter != Section::map.end(); iter++) {
		iter->second->save(document);
	}
}


void profiles_reset(CONNECTION * connection)
{
	connection->profiles = connection->oprofiles;
}

void bypass_check(CONNECTION * connection)
{
	int x;
	char *ptr;
	struct url_command_t *url_command;

	if ((connection->access & ACCESS_BYPASS) && connection->url_command != NULL) {
		url_command = url_command_find(connection->url_command, "bypass");
		if (url_command != NULL) {
			ptr = url_command->options;
			if (ptr != NULL) {
				x = TRUE;

				for (; *ptr; ptr++) {
					switch (*ptr) {
					case '+':
						x = TRUE;
						break;
					case '-':
						x = FALSE;
						break;
					case 'f':
						if (x == TRUE)
							connection->bypass |= FEATURE_FILTER;
						else
							connection->bypass &= ~FEATURE_FILTER;
						break;
					case 'h':
						if (x == TRUE)
							connection->bypass |= FEATURE_HEADER;
						else
							connection->bypass &= ~FEATURE_HEADER;
						break;
					case 'm':
						if (x == TRUE)
							connection->bypass |= FEATURE_MIME;
						else
							connection->bypass &= ~FEATURE_MIME;
						break;
					case 'r':
						if (x == TRUE)
							connection->bypass |= FEATURE_REDIRECT;
						else
							connection->bypass &= ~FEATURE_REDIRECT;
						break;
					case 'c':
						if (x == TRUE)
							connection->bypass |= FEATURE_COOKIES;
						else
							connection->bypass &= ~FEATURE_COOKIES;
						break;
					case 'w':
						if (x == TRUE)
							connection->bypass |= FEATURE_REWRITE;
						else
							connection->bypass &= ~FEATURE_REWRITE;
						break;
					case 'e':
						if (x == TRUE)
							connection->bypass |= FEATURE_EXTERNAL;
						else
							connection->bypass &= ~FEATURE_EXTERNAL;
						break;
					case 'p':
						if (x == TRUE)
							connection->bypass |= FEATURE_FORWARD;
						else
							connection->bypass &= ~FEATURE_FORWARD;
						break;
					case 'k':
						if (x == TRUE)
							connection->bypass |= FEATURE_KEYWORDS;
						else
							connection->bypass &= ~FEATURE_KEYWORDS;
						break;
					case 'd':
						if (x == TRUE)
							connection->bypass |= FEATURE_DNSBL;
						else
							connection->bypass &= ~FEATURE_DNSBL;
						break;
					case 'l':
						if (x == TRUE)
							connection->bypass |= FEATURE_LIMITS;
						else
							connection->bypass &= ~FEATURE_LIMITS;
						break;
					}
				}
			} else
				connection->bypass = ~0;
		}
	}
}

void rusage_update()
{
	struct rusage ru;

	getrusage(RUSAGE_SELF, &ru);

	pthread_mutex_lock(&global->rusage_lock);

	global->rusage.ru_stime.tv_sec += ru.ru_stime.tv_sec;
	global->rusage.ru_stime.tv_usec += ru.ru_stime.tv_usec;
	global->rusage.ru_stime.tv_sec += global->rusage.ru_stime.tv_usec / 1000000;
	global->rusage.ru_stime.tv_usec %= 1000000;

	global->rusage.ru_utime.tv_sec += ru.ru_utime.tv_sec;
	global->rusage.ru_utime.tv_usec += ru.ru_utime.tv_usec;
	global->rusage.ru_utime.tv_sec += global->rusage.ru_utime.tv_usec / 1000000;
	global->rusage.ru_utime.tv_usec %= 1000000;

	global->rusage.ru_minflt += ru.ru_minflt;
	global->rusage.ru_majflt += ru.ru_majflt;

	pthread_mutex_unlock(&global->rusage_lock);
}
