/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include "proto.h"

time_t last_dns_expire = 0;
extern char username[];
extern GLOBAL *global;
extern CacheSection cache_section;
SOCKLIST socklist[MAXLISTENPORTS];

/*
initialize the socket list
*/
void net_init()
{
	int i;

	for (i = 0; i < MAXLISTENPORTS; i++) {
		socklist[i].status = NET_UNUSED;
		socklist[i].ip = NULL;
		socklist[i].port = 0;
	}
}

CONNECTION *connection_new()
{
	CONNECTION *connection;

	connection = xnew CONNECTION;

	connection->flags = 0;
       	connection->client = NULL;
        connection->server = NULL;
        connection->transferred = 0;
        connection->transferlimit = ~0;
        connection->depth = 0;
        connection->port = 0;
        connection->ip = NULL;
        connection->bypass = 0;
        connection->header = NULL;
        connection->rheader = NULL;
        connection->keepalive_client = FALSE;
        connection->keepalive_server = FALSE;
        connection->request = 0;
        connection->proxy_host = NULL;
        connection->proxy_type = PROXY_DIRECT;
        connection->proxy_username = NULL;
        connection->proxy_password = NULL;
        connection->proxy_domain = NULL;
        connection->proxy_auth = NULL;
        connection->authenticate = FALSE;
        connection->cachemap = NULL;
        connection->htmlstream = NULL;
        connection->url_command = NULL;
	connection->postbody = NULL;

	return connection;
}

void connection_free(CONNECTION *connection) {
	connection->profiles.clear();
	connection->oprofiles.clear();

	FREE_AND_NULL(connection->ip);
	FREE_AND_NULL(connection->interface);
	FREE_AND_NULL(connection->proxy_host);
	FREE_AND_NULL(connection->proxy_username);
	FREE_AND_NULL(connection->proxy_password);
	FREE_AND_NULL(connection->proxy_domain);
	FREE_AND_NULL(connection->proxy_auth);
	xdelete connection->postbody;

	xdelete connection;
}

/*
open a listening socket on a specific port and ip
*/
int net_listen(int port, const char *ip)
{
	int i, x = -1, on = 1;
	struct sockaddr_in saddr;

	for (i = 0; i < MAXLISTENPORTS; i++) {
		if (socklist[i].status == NET_UNUSED) {
			x = i;
			break;
		}
	}

	if (x == -1)
		goto error;

	socklist[x].fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (socklist[x].fd == -1)
		goto error;

	setsockopt(socklist[x].fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	setsockopt(socklist[x].fd, IPPROTO_TCP, TCP_NODELAY, &on, sizeof(on));

	fcntl(socklist[x].fd, F_SETFL, FD_CLOEXEC);

	memset(&saddr, 0, sizeof(struct sockaddr_in));

	saddr.sin_family = AF_INET;
	saddr.sin_port = htons(port);

	if (ip == NULL)
		saddr.sin_addr.s_addr = INADDR_ANY;
	else
		inet_pton(AF_INET, ip, &saddr.sin_addr.s_addr);

	i = bind(socklist[x].fd, (struct sockaddr *) &saddr, sizeof(saddr));
	if (i == -1) {
		close(socklist[x].fd);
		goto error;
	}

	i = listen(socklist[x].fd, 25);
	if (i == -1) {
		close(socklist[x].fd);
		goto error;
	}

	if (ip != NULL)
		socklist[x].ip = xstrdup(ip);
	else
		socklist[x].ip = NULL;

	socklist[x].port = port;
	socklist[x].status = NET_LISTEN;

	putlog(MMLOG_NETWORK, "listening on %s:%d", (ip != NULL) ? ip : "0.0.0.0", port);

	return x;

	error:

	putlog(MMLOG_NETWORK, "failed to bind to %s:%d", (ip != NULL) ? ip : "0.0.0.0", port);

	return -1;
}

/*
poll on every open listening socket in socklist until timeout is reached (or forever if 
timeout is -1), accept then return a pointer to a CONNECTION struct
*/
CONNECTION *net_accept(int timeout)
{
	int i, j, newfd;
	char ip[16];
	struct sockaddr_in saddr;
	socklen_t socklen = sizeof(struct sockaddr_in);
	unsigned int sl = sizeof(struct sockaddr);
	CONNECTION *connection;
	struct pollfd pfd[MAXLISTENPORTS];

	for (i = 0; i < MAXLISTENPORTS; i++) {
		if (socklist[i].status == NET_LISTEN) {
			pfd[i].fd = socklist[i].fd;
			pfd[i].events = POLLIN;
		} else
			break;
	}

	j = p_poll(pfd, i, (timeout != -1) ? timeout * 1000 : -1);

	if (j > 0) {
		for (i = 0; i < MAXLISTENPORTS; i++) {
			if (pfd[i].revents & POLLIN) {
				newfd = accept(socklist[i].fd, (struct sockaddr *) &saddr, &sl);

				if (newfd == -1)
					return NULL;

				connection = connection_new();

				connection->client = xnew Socket(newfd);
				connection->port = socklist[i].port;

				inet_ntop(AF_INET, &saddr.sin_addr, ip, sizeof(ip));
				connection->ip = xstrdup(ip);
				
				getsockname(newfd, (struct sockaddr *)&saddr, &socklen);
				inet_ntop(AF_INET, &saddr.sin_addr, ip, sizeof(ip));
				connection->interface = xstrdup(ip);

				return connection;
			}
		}
	}

	return NULL;
}

/*
open a connection to somewhere and return file descriptor
*/
int net_connect(const char *host, int port, int timeout)
{
	int fd, ret, err = 0, on = TRUE;
	HOSTENT *hostent;
	struct sockaddr_in saddr;
	struct pollfd pfd;
	socklen_t optlen = sizeof(int);

	if (!host || port <= 0) {
		err = ERROR_UNKNOWN;
		goto error;
	}

	memset(&saddr, 0, sizeof(saddr));

	hostent = net_dns(host);
	if (hostent == NULL) {
		global->stats.Increment("network", "DNS failures");

		err = ERROR_DNS;
		goto error;
	}

	saddr.sin_addr = *(struct in_addr *) hostent->addr;

	hostent_free(hostent);

	saddr.sin_family = AF_INET;
	saddr.sin_port = htons(port);

	fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (fd == -1) {
		err = ERROR_CONNECT;
		goto error;
	}

	setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &on, sizeof(on));
	fcntl(fd, F_SETFD, FD_CLOEXEC);
	fcntl(fd, F_SETFL, O_NONBLOCK);

	ret = connect(fd, (struct sockaddr *) &saddr, sizeof(saddr));
	if (ret == -1) {
		if (errno != EINPROGRESS) {
			close(fd);

			err = ERROR_CONNECT;
			goto error;
		} else {
			pfd.fd = fd;
			pfd.events = POLLOUT;

			ret = p_poll(&pfd, 1, (timeout != -1) ? timeout * 1000 : -1);
			if (ret == 0) {
				close(fd);

				err = ERROR_CONNECT;
				goto error;
			} else {
				getsockopt(fd, SOL_SOCKET, SO_ERROR, &ret, &optlen);
				if (ret != 0) {
					close(fd);	

					err = ERROR_CONNECT;
					goto error;
				}
			}
		}
	}

	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) & ~O_NONBLOCK);

	global->stats.Increment("network", "successful connections");

	return fd;

	error:

	global->stats.Increment("network", "failed connections");

	return err;
}

/*
close a connection and free used memory from structure
*/
void net_close(CONNECTION * connection)
{
	if (!connection)
		return;

	xdelete connection->client;

	connection_free(connection);
}

/*
close all open listening sockets
*/
void net_listen_closeall()
{
	int i;

	for (i = 0; i < MAXLISTENPORTS; i++) {
		if (socklist[i].status == NET_LISTEN) {
			close(socklist[i].fd);
			if (socklist[i].ip != NULL)
				xfree(socklist[i].ip);
			socklist[i].status = NET_UNUSED;
		}
	}
}

NetworkSection::NetworkSection():
	Section("network", RWLOCK, true)
{
}

void NetworkSection::update()
{
	listen_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++) {

		listen_list.push_back(Listen(*item));
	}
}

/*
 * Constructor of Listen
 */
Listen::Listen(const Item& item):
	enabled (item.field_vec[0].int_value),
	comment (item.field_vec[1].string_value),
	port    (item.field_vec[2].int_value),
	ip      (item.field_vec[3].string_value)

{ 
	// nothing to do !
}


/*
look through network->listen_list linked list and bind to listening ports
*/
void NetworkSection::check()
{
	read_lock();

	ListenList::const_iterator listen;
	for (listen = listen_list.begin(); listen != listen_list.end(); listen++) {
		if (listen->enabled == TRUE && listen->port != -1)
			net_listen(listen->port, listen->ip.c_str());
	}

	unlock();
}

/*
relay anything from one socket to another, and vice versa
*/
int net_proxy(CONNECTION * connection, int len)
{
	int x, bytes = 0;
	char buf[BLOCKSIZE];
	struct pollfd pfd[2];
	Socket *sock1, *sock2;

	sock1 = connection->client;
	sock2 = connection->server;

	pfd[0].fd = sock1->fd;
	pfd[0].events = POLLIN;
	pfd[1].fd = sock2->fd;
	pfd[1].events = POLLIN;

	while (1) {
		x = p_poll(pfd, 2, (sock1->inbuf_len != 0 || sock2->inbuf_len != 0) ? 0 : -1);
		if (x == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}

		if (sock1->inbuf_len != 0 || pfd[0].revents & POLLIN) {
			x = sock1->Read(buf, (len != -1) ? (len - bytes < sizeof(buf)) ? len - bytes : sizeof(buf) : sizeof(buf));
			if (x > 0)
				sock2->Write(buf, x);
			else
				break;

			bytes += x;
		} else if (pfd[0].revents & (POLLHUP | POLLERR))
			break;

		if (sock1->inbuf_len != 0 || pfd[1].revents & POLLIN) {
			x = sock2->Read(buf, (len != -1) ? (len - bytes < sizeof(buf)) ? len - bytes : sizeof(buf) : sizeof(buf));
			if (x > 0)
				sock1->Write(buf, x);
			else
				break;

			bytes += x;
		} else if (pfd[1].revents & (POLLHUP | POLLERR))
			break;

		if (len != -1 && bytes == len)
			break;
	}

	return bytes;
}

/*
tranfer data one way between two sockets
*/
int net_transfer(CONNECTION * connection, int flags, int bytes)
{
	int x, sent = 0;
	char buf[BLOCKSIZE];
	struct pollfd pfd[2];
	Socket *sock1, *sock2;

	sock1 = (flags == SERVER) ? connection->server : connection->client;
	sock2 = (flags == SERVER) ? connection->client : connection->server;

	pfd[0].fd = sock1->fd;
	pfd[0].events = POLLIN;

	if (sock2 != NULL) {
		pfd[1].fd = sock2->fd;
		pfd[1].events = POLLIN;
	} else
		pfd[1].revents = 0;

	while (bytes == -1 || bytes > 0) {
		x = p_poll(pfd, (sock2 != NULL) ? 2 : 1, (sock1->inbuf_len != 0) ? 0 : -1);
		if (x == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}

		if (sock1->inbuf_len != 0 || pfd[0].revents & POLLIN) {
			x = sock1->Read(buf, (bytes != -1) ? (bytes < sizeof(buf)) ? bytes : sizeof(buf) : sizeof(buf));

			if (x > 0) {
				if (bytes != -1)
					bytes -= x;

				sent += x;

				if (sock2 != NULL) {
					x = sock2->Write(buf, x);
					if (x == -1)
						break;
				}
			} else
				break;
		} else if (pfd[0].revents & (POLLHUP | POLLERR))
			break;

		if (pfd[1].revents & POLLIN) {
			x = sock2->Read(NULL, -1);
			if (x <= 0)
				break;
		} else if (pfd[1].revents & (POLLHUP | POLLERR))
			break;
	}

	if (bytes != 0 && bytes != -1)
		connection->flags |= CONNECTION_FAILED;

	return sent;
}

int net_filebuf_read(Filebuf * filebuf, CONNECTION * connection, int flags, int bytes)
{
	int x;
	char buf[BLOCKSIZE];
	struct pollfd pfd[2];
	Socket *sock1, *sock2;

	sock1 = (flags == SERVER) ? connection->server : connection->client;
	sock2 = (flags == SERVER) ? connection->client : connection->server;

	pfd[0].fd = sock1->fd;
	pfd[0].events = POLLIN;

	if (sock2 != NULL) {
		pfd[1].fd = sock2->fd;
		pfd[1].events = POLLIN;
	} else
		pfd[1].revents = 0;

	while (bytes == -1 || bytes > 0) {
		x = p_poll(pfd, (sock2 != NULL) ? 2 : 1, (sock1->inbuf_len != 0) ? 0 : -1);
		if (x == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}

		if (sock1->inbuf_len != 0 || pfd[0].revents & POLLIN) {
			x = sock1->Read(buf, (bytes != -1) ? (bytes < sizeof(buf)) ? bytes : sizeof(buf) : sizeof(buf));
			if (x > 0) {
				if (filebuf != NULL)
					filebuf->Add(buf, x);

				if (bytes != -1)
					bytes -= x;
			} else
				return FALSE;
		} else if (pfd[0].revents & (POLLHUP | POLLERR))
			break;

		if (pfd[1].revents & POLLIN) {
			x = sock2->Read(NULL, -1);
			if (x <= 0)
				break;
		} else if (pfd[1].revents & (POLLHUP | POLLERR))
			break;
	}

	if (bytes != -1 && bytes != 0) {
		connection->flags |= CONNECTION_FAILED;

		return FALSE;
	}

	return TRUE;
}

int net_filebuf_send(Filebuf * filebuf, CONNECTION * connection, int flags)
{
	Socket *sock;

	if (flags == CLIENT && (connection->flags & CONNECTION_NOCLIENT)) return filebuf->size;

	sock = (flags == SERVER) ? connection->server : connection->client;

	return sock->WriteFilebuf(filebuf);
}

int net_cache_send(CACHEMAP * cachemap, CONNECTION * connection, int flags, unsigned int offset)
{
	Socket *sock;

	sock = (flags == SERVER) ? connection->server : connection->client;

	offset += cachemap->offset;
	if (offset > cachemap->size)
		offset = cachemap->size;

	return sock->Write(cachemap->data + offset, cachemap->size - offset);
}

/*
perform a dns lookup, using cached response from a previous lookup if possible
*/
HOSTENT *net_dns(const char *host)
{
	time_t t;
	char hst[128], buf[24];
	HOSTENT *hostent;
	struct HASH_LIST *hl;

	pthread_mutex_lock(&global->dns_cache_lock);

	t = time(NULL);

	s_strncpy(hst, host, sizeof(hst));
	string_tolower(hst);

	if (last_dns_expire == 0)
		last_dns_expire = t;

	if (t - last_dns_expire >= DNS_EXPIRE) {
		putlog(MMLOG_DEBUG, "dns cache expire: entry > %d seconds", DNS_EXPIRE);

		hash_expire(global->dns_cache, DNS_EXPIRE);
		last_dns_expire = t;
	}

	if (isip(host)) {
		hostent = (HOSTENT*)xmalloc(sizeof(HOSTENT));
		hostent->addr = (in_addr*)xmalloc(sizeof(struct in_addr));
		hostent->len = sizeof(struct in_addr);

		inet_aton(host, hostent->addr);
	} else {
		hl = hash_search(global->dns_cache, hst);
		if (hl == NULL) {
			pthread_mutex_unlock(&global->dns_cache_lock);

			hostent = p_gethostbyname(host);
			if (hostent == NULL)
				return NULL;

			pthread_mutex_lock(&global->dns_cache_lock);

			inet_ntop(AF_INET, hostent->addr, buf, sizeof(buf));

			hl = hash_insert(global->dns_cache, hst, xstrdup(buf));
			hl->age = t;

			global->stats.Increment("DNS cache", "miss");
			putlog(MMLOG_DEBUG, "dns cache insert: %s -> %s", hst, buf);
		} else {
			global->stats.Increment("DNS cache", "hit");
			putlog(MMLOG_DEBUG, "dns cache hit: %s -> %s", hst, hl->data);

			hostent = (HOSTENT*)xmalloc(sizeof(HOSTENT));
			hostent->addr = (in_addr*)xmalloc(sizeof(struct in_addr));
			hostent->len = sizeof(struct in_addr);

			inet_aton((char*)hl->data, hostent->addr);
		}
	}

	pthread_mutex_unlock(&global->dns_cache_lock);

	return hostent;
}

/*
connect through socks4 server
*/
int proxy_socks4(CONNECTION * connection)
{
	int ret, bytes = 0;
	char buf[64];
	HOSTENT *hostent;

	hostent = net_dns(connection->header->host);
	if (hostent == NULL)
		return ERROR_DNS;

	ret = snprintf(buf, sizeof(buf), "\004\001%c%c%c%c%c%c%s", (connection->header->port >> 8) % 256, connection->header->port % 256, '0', '0', '0', '0', username);
	memcpy(&buf[4], hostent->addr, hostent->len);

	hostent_free(hostent);

	connection->server->Write(buf, ret + 1);

	while (bytes < 8) {
		ret = connection->server->Read(buf + bytes, 8 - bytes);
		if (ret != -1)
			bytes += ret;
		else
			return ERROR_CONNECT;
	}

	switch (buf[1]) {
	case SOCKS_GRANTED:
		return TRUE;
		break;
	case SOCKS_FAILED:
		return ERROR_CONNECT;
		break;
	case SOCKS_BADIDENT:
	case SOCKS_NOIDENT:
		return ERROR_AUTH;
		break;
	default:
		return ERROR_UNKNOWN;
	}
}
