/*
    MiddleMan filtering proxy server
    Copyright (C) 2003  Riadh Elloumi
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include "proto.h"
#include "pages.h"


TemplateSection::TemplateSection():
     Section("templates", RWLOCK),
     path   (field_vec[0].string_value)
{
}

void TemplateSection::update()
{
	template_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		template_list.push_back(Template(*item));
	
}


Template::Template(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     name     (item.field_vec[3].string_value),
     file     (item.field_vec[4].string_value),
     mime     (item.field_vec[5].string_value),
     code     (item.field_vec[6].int_value),
     type     (item.field_vec[7].int_value),
     parsable (item.field_vec[8].int_value)
{
	/* Nothing to do ! */
}

/*
load a template from a file, and place in a buffer
*/
Filebuf *TemplateSection::get(CONNECTION * connection, const char *name, int *code)
{
	int i = 0, x = -1, parsable = FALSE;
	char buf[256];
	Filebuf *ret = NULL;

	read_lock();

	TemplateList::const_iterator templ;
	for (templ = template_list.begin(); templ != template_list.end(); templ++) {
		/* search through templates from config file first */

		if (!profile_find(connection->profiles, templ->profiles))
			continue;

		if (templ->name != "") {
			if (templ->enabled == TRUE && !strcasecmp(templ->name.c_str(), name)) {
				parsable = templ->parsable;
				break;
			}
		}
	}

	if (templ == template_list.end() || templ->file == "") {
		/* no matching template in config file, search through compiled-in templates */
		for (; PAGES[i].name; i++) {
			if (!strcasecmp(PAGES[i].name, name)) {
				x = i;
				parsable = PAGES[i].parsable;
				break;
			}
		}

		/* nope, nothing... oh well */
		if (x == -1)
			goto out;

		i = (PAGES[x].len != -1) ? PAGES[x].len : strlen(PAGES[x].content);
		ret = xnew Filebuf();

		/* give built-in templates a stylesheet */
		if (!strcasecmp(PAGES[x].mime, "text/html") && strcasecmp(name, "interface.css"))
			interface_stylesheet(connection, ret);

		ret->Add(PAGES[x].content, i);
		ret->type = xstrdup(PAGES[x].mime);
	} else if (templ->type == TEMPLATE_FILE) {
		if (templ->file[0] != '/' && this->path != "")
			snprintf(buf, sizeof(buf), "%s/%s", this->path.c_str(), templ->file.c_str());
		else
			snprintf(buf, sizeof(buf), "%s", templ->file.c_str());

		ret = xnew Filebuf();

		ret->Read(buf);

		if (templ->mime != "")
			ret->type = xstrdup(templ->mime.c_str());
	} else if (templ->type == TEMPLATE_EXECUTABLE) {
		ret = external_exec(connection, templ->file.c_str(), NULL, EXTERNAL_PIPE);
		if (templ->mime != "" && ret != NULL) {
			if (!strcasecmp(templ->mime.c_str(), "STDIN"))
				ret->type = external_getmime(ret);
			else
				ret->type = xstrdup(templ->mime.c_str());
		}
	}

	if (ret != NULL && parsable == TRUE)
		template_parse(connection, ret);

	if (templ != template_list.end() && ret != NULL && code != NULL)
		*code = templ->code;

      out:
	unlock();

	return ret;
}

void template_parse(CONNECTION * connection, Filebuf * filebuf)
{
	char *s;
	Filebuf *ret;

	ret = xnew Filebuf();

	s = filebuf->data;
	while (s != NULL && s < filebuf->data + filebuf->size) {
		if (*s == '$') {
			if (s[1] == '$') {
				ret->Add("$", 1);
				s += 2;
			} else {
				string variable = string(s, 64);
				variable = variable.substr(1, variable.find('$', 1) - 1);
				string value = connection->variables[variable];

				if (variable != "" && value != "") {
					ret->Addf("%s", value.c_str());
					s += variable.length() + 2;
				} else
					for (s++; s < filebuf->data + filebuf->size && *s != ' ' && *s != '$'; s++);
			}
		} else
			ret->Add(s++, 1);
	}

	if (filebuf->data != NULL)
		xfree(filebuf->data);

	filebuf->data = ret->data;
	filebuf->size = ret->size;
	filebuf->realsize = ret->realsize;

	ret->data = NULL;
	xdelete ret;
}

/*
send a template with http headers and proper mime type to web browser
*/
int TemplateSection::send(const char *templ, CONNECTION * connection, int error)
{
	int code = -1;
	Filebuf *filebuf = NULL;
	HEADER *header;

	if (connection->flags & CONNECTION_NOCLIENT) return FALSE;

	filebuf = this->get(connection, templ, &code);

	if (!filebuf)
		return FALSE;

	header = header_new();
	header->type = HTTP_RESP;
	header->code = (code != -1) ? code : error;
	header->content_length = filebuf->size;
	header->flags |= HEADER_CL;
	header->content_type = (filebuf->type != NULL) ? xstrdup(filebuf->type) : xstrdup("text/plain");

	header_send(header, connection, CLIENT, HEADER_RESP);
	http_header_free(header);

	/* don't actually send anything if this is a HEAD request */
	if (connection->header == NULL || strcasecmp(connection->header->method, "HEAD"))
		net_filebuf_send(filebuf, connection, CLIENT);

	putlog(MMLOG_TEMPLATE, "sent %s", templ);

	xdelete filebuf;

	return TRUE;
}

int TemplateSection::insert(CONNECTION *connection, Filebuf *filebuf, const char *templ)
{
	Filebuf *fb;

	fb = this->get(connection, templ, NULL);
	if (fb != NULL) {
		filebuf->Add(fb->data, fb->size);
		xdelete fb;

		return TRUE;
	}

	return FALSE;
}
