/*
 * Copyright (c) 2022 MiLaboratories Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.milaboratory.core.sequence;

import gnu.trove.set.hash.TLongHashSet;

import java.util.Objects;

public class ShortSequenceSet {
    private final TLongHashSet set = new TLongHashSet();

    public boolean add(NucleotideSequence seq) {
        return set.add(toLong(seq));
    }

    public boolean contains(NucleotideSequence seq) {
        return !seq.containsWildcards() && set.contains(toLong(seq));
    }

    public int size() {
        return set.size();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ShortSequenceSet)) return false;
        ShortSequenceSet that = (ShortSequenceSet) o;
        return set.equals(that.set);
    }

    @Override
    public int hashCode() {
        return Objects.hash(set);
    }

    public static long toLong(NucleotideSequence seq) {
        if (seq.size() > 29)
            throw new IllegalArgumentException("Only sequences shorter than 30 nucleotides are supported.");
        if (seq.containsWildcards())
            throw new IllegalArgumentException("Sequences containing wildcards are not supported.");
        long ret = 0;
        for (int i = 0; i < seq.size(); i++) {
            ret <<= 2;
            ret |= seq.codeAt(i);
        }
        ret |= ((long) seq.size()) << 58;
        return ret;
    }
}
