/*======================================================================*\
|*		Editor mined						*|
|*		keyboard input						*|
\*======================================================================*/

#include "mined.h"
#include "io.h"

#ifndef __TURBOC__
#include <sys/time.h>
#endif
#ifdef CURSES
#include <curses.h>
#endif


voidfunc keyproc = I;	/* function addressed by entered function key */
char keyshift = '0';	/* shift state of entered function key */

static
int keymap_delay = 900;		/* wait to detect mapped key sequence */

static
int escape_delay = 450;		/* wait to detect escape sequence */

static
int dont_suppress_keymap = 1;


/*======================================================================*\
|*		Handling function key and keyboard mapping tables	*|
\*======================================================================*/

#ifndef msdos

extern struct fkeyentry {
	char * fk;
	voidfunc fp;
} fkeymap [];

extern struct fkeyentry fkeymap_xterm [];
extern struct fkeyentry fkeymap_vt100 [];
extern struct fkeyentry fkeymap_hp [];
extern struct fkeyentry fkeymap_siemens [];

struct fkeyentry * fkeymap2 = fkeymap_xterm;

void
set_fkeymap (term)
  char * term;
{
  if (term == NIL_PTR) {
	return;
  } else if (* term == 'x') {
	fkeymap2 = fkeymap_xterm;
  } else if (* term == 'h') {
	fkeymap2 = fkeymap_hp;
  } else if (* term == 'v') {
	fkeymap2 = fkeymap_vt100;
  } else if (* term == 's') {
	fkeymap2 = fkeymap_siemens;
  }
}

#endif


#define stringtables
#ifdef stringtables
#define keymaptabletype char *
#else
#define keymaptabletype struct keymap *
struct keymap {
	char * fk;
	char * fp;
};
#endif

extern struct {
	keymaptabletype table;
	char * shortcut;
} keymappingtable [];
extern int keymappingtable_len;


char * keyboard_mapping = "--";
static char * last_keyboard_mapping = "--";
static char * next_keyboard_mapping = "";

static keymaptabletype keyboard_map = (keymaptabletype) NIL_PTR;
static keymaptabletype last_keyboard_map = (keymaptabletype) NIL_PTR;

/**
   Set keymap.
 */
static
void
set_keymap (new_keymap, current, next)
  keymaptabletype new_keymap;
  char * current;
  char * next;
{
  last_keyboard_map = keyboard_map;
  last_keyboard_mapping = keyboard_mapping;
  keyboard_map = new_keymap;
  keyboard_mapping = current;
  next_keyboard_mapping = next;
}

/**
   Exchange current and previous keyboard mapping; set previous active.
   With HOP, reset keyboard mapping to none.
 */
void
toggleKEYMAP ()
{
  if (allow_keymap) {
	if (hop_flag > 0) {
		set_keymap ((keymaptabletype) NIL_PTR, "--", keymappingtable [0].shortcut);
	} else {
		set_keymap (last_keyboard_map, last_keyboard_mapping, last_keyboard_mapping);
	}
	displayflags ();
  } else {
	error ("Keyboard mapping not active", NIL_PTR);
  }
}

/**
   Set keyboard mapping.
 */
static
FLAG
selectKEYMAP (script)
  char * script;
{
  int i;

  if (script == NIL_PTR || * script == '\0') {
	set_keymap ((keymaptabletype) NIL_PTR, "--", keymappingtable [0].shortcut);
	displayflags ();
	return True;
  } else {
	/* map alternative shortcuts */
	if (strisprefix ("el", script)) {
		script = "gr";
	} else if (strisprefix ("ru", script)) {
		script = "cy";
	}
	/* map shortcut to keymap */
	for (i = 0; i < keymappingtable_len; i ++) {
	    if (strisprefix (keymappingtable [i].shortcut, script)) {
		set_keymap (keymappingtable [i].table,
			    keymappingtable [i].shortcut,
			    i + 1 == keymappingtable_len ?
			      "" : keymappingtable [i + 1].shortcut);
		displayflags ();
		return True;
	    }
	}
	return False;
  }
}

/**
   Set keyboard mapping.
 */
void
setKEYMAP (script)
  char * script;
{
  if (script != NIL_PTR && * script == '-') {
	/* set keymap as secondary map */
	script ++;
	selectKEYMAP (script);
	selectKEYMAP (NIL_PTR);
  } else {
	selectKEYMAP (script);
  }
}

/**
   Cycle through keyboard mappings.
 */
void
cycleKEYMAP ()
{
  selectKEYMAP (next_keyboard_mapping);
}

/**
   Enable keyboard mapping interactively.
   With HOP, cycle through keyboard mappings.
 */
void
setupKEYMAP ()
{
  if (allow_keymap) {
	if (hop_flag > 0) {
		hop_flag = 0;
		selectKEYMAP (next_keyboard_mapping);
	} else {
		handleKeymapmenu (0);
	}
  } else {
	error ("Keyboard mapping not active", NIL_PTR);
  }
}

/**
   get_string_nokeymap prompts a string like get_string
   but doesn't apply international keyboard mapping
 */
int
get_string_nokeymap (prompt, inbuf, statfl, term_input)
  char * prompt;
  char * inbuf;
  FLAG statfl;
  char * term_input;
{
  int ret;

  dont_suppress_keymap = 0;
  ret = get_string (prompt, inbuf, statfl, term_input);
  dont_suppress_keymap = 1;

  return ret;
}


/*======================================================================*\
|*		Digits input						*|
\*======================================================================*/

/*
 * get_digits () reads in a number. In contrast to get_number, it does no
 * echoing, no messaging, and it does not require any digits at all.
 * The last character typed in is returned.
 * The resulting number is put into the integer the arguments points to.
 * It is for use within I/O routines (e.g. reading mouse escape sequences 
 * or cursor position report) and thus calls _readchar ().
 */
int
get_digits (result)
  int * result;
{
  register int index;
  register int count;

  index = _readchar_nokeymap ();
  if (index == quit_char) {
	quit = True;
  }
  * result = -1;
/* Convert input to a decimal number */
  count = 0;
  while (index >= '0' && index <= '9' && quit == False) {
	count *= 10;
	count += index - '0';
	* result = count;
	index = _readchar_nokeymap ();
	if (index == quit_char) {
		quit = True;
	}
  }

  if (quit == True) {
	return quit_char;
  }
  return index;
}


/*======================================================================*\
|*				Terminal input				*|
\*======================================================================*/

#define max_stamps 10
static long last_sec = 0;
static long last_msec = 0;
static int n_stamps = 0;
static int i_stamp = 0;
static long stamps [max_stamps];
static long total_deltas = 0;
long average_delta_readchar = 0;
long last_delta_readchar = 0;

static
void
timestamp_readchar ()
{
#ifndef __TURBOC__
  struct timeval now;
  long sec;
  long msec;

  gettimeofday (& now, 0);
  sec = now.tv_sec;
  msec = now.tv_usec / 1000;
  if (last_sec == 0) {
	last_delta_readchar = 1000;
  } else {
	last_delta_readchar = (sec - last_sec) * 1000 + msec - last_msec;
  }
  last_sec = sec;
  last_msec = msec;

  if (n_stamps < max_stamps) {
	n_stamps ++;
  } else {
	/* remove oldest stamp from statistics */
	total_deltas -= stamps [i_stamp];
  }
  stamps [i_stamp] = last_delta_readchar;
  /* add stamp to statistics */
  total_deltas += last_delta_readchar;
  i_stamp ++;
  if (i_stamp == max_stamps) {
	i_stamp = 0;
  }

  average_delta_readchar = total_deltas / n_stamps;
#endif
}

/*
 * Readchar () reads one character from the terminal.
 * There are problems due to interruption of the read operation by signals
 * (QUIT, WINCH). The waitingforinput flag is only a partial solution.
 * Unix doesn't provide sufficient facilities to handle these situations
 * neatly and properly. Moreover, different Unix versions yield different
 * surprising effects. However, the use of select () could still be
 * an improvement.
 */
static
int
readchar ()
{
  register character c;
#ifdef pc_charset
  int i;
#endif

#ifdef msdos
#define winchhere
#endif

#ifdef winchhere
  FLAG waiting;

  if (winchg == True && waitingforinput == False) {
	/* In the Unix version, this is now done in __readchar () */
	RDwin ();
  }
  /* must save waitingforinput flag since in the MSDOS version, 
     readchar can be called recursively */
  waiting = waitingforinput;
#endif
  waitingforinput = True;

  c = _readchar ();
  while (c == (character) FUNcmd && (* keyproc) == MOUSEescape 
	&& mouse_button == releasebutton && report_release == False)
  {
	c = _readchar ();
  }
  report_release = False;

#ifdef winchhere
  waitingforinput = waiting;
#else
  waitingforinput = False;
#endif

  /* the modification
       if (quit == True) c = quit_char;
     (now in __readchar) must not be placed after resetting the flag
       waitingforinput = False;
     .
     Otherwise a QUIT signal coming in just between these two would
     discard the last valid character just taken up. */

  timestamp_readchar ();

#ifdef pc_charset
  if (utf8_text == True && utf8_input == False && c >= 0x80) {
	for (i = 0; i < 128; i ++) {
		if (c == (character) pc_charmap [i]) {
			return (character) (i + 128);
		}
	}
  }
#endif

  return c;
}

/*
   readchar_nokeymap reads like readchar
   but doesn't apply international keyboard mapping
 */
static
int
readchar_nokeymap ()
{
  int res;
  int prev_dont_suppress_keymap = dont_suppress_keymap;

  dont_suppress_keymap = 0;
  res = readchar ();
  dont_suppress_keymap = prev_dont_suppress_keymap;

  return res;
}

/*
   readcharacter_allbuttons reads like readcharacter
   but always reports mouse button release
 */
unsigned long
readcharacter_allbuttons ()
{
  report_release = True;
  return readcharacter ();
}

/*
   readcharacter () reads one Unicode character, suppressing keyboard mapping
 */
unsigned long
readcharacter ()
{
  return readcharacter_mapping (False);
}

/*
   readcharacter_mapping () reads one Unicode character
   either with or without keyboard mapping
 */
unsigned long
readcharacter_mapping (map_keyboard)
  FLAG map_keyboard;
{
  return readcharacter_mapping_valid (map_keyboard, False);
}

/*
   readcharacter_mapping () reads one Unicode character
   either with or without keyboard mapping
 */
unsigned long
readcharacter_mapping_valid (map_keyboard, valid)
  FLAG map_keyboard;
  FLAG valid;
{
  int utfcount = 1;
  unsigned long unichar;
  int byte2;

  if (map_keyboard == True) {
	unichar = readchar ();
  } else {
	unichar = readchar_nokeymap ();
	if (unichar == (character) FUNcmd &&
		(     (* keyproc) == SNL
		/* || (* keyproc) == Popmark */
		)) {
		return '\r';
	}
  }

  if ((unichar & 0x80) == 0x00) {
	return unichar;
  } else if (cjk_term == True && multichar (unichar)) {
	if (cjk_encoding == 'C' && unichar == 0x8E) {
		return unichar << 24
			| (readchar_nokeymap () << 16)
			| (readchar_nokeymap () << 8)
			| readchar_nokeymap ();
	} else if (cjk_encoding == 'J' && unichar == 0x8F) {
		return unichar << 16
			| (readchar_nokeymap () << 8)
			| readchar_nokeymap ();
	} else {
		byte2 = readchar_nokeymap ();
		if (cjk_encoding == 'G' && '0' <= byte2 && byte2 <= '9') {
			return unichar << 24
				| (byte2 << 16)
				| (readchar_nokeymap () << 8)
				| readchar_nokeymap ();
		} else {
			return unichar << 8 | byte2;
		}
	}
  } else if (utf8_input == False) {
	return unichar;
  } else {
	if ((unichar & 0xE0) == 0xC0) {
		utfcount = 2;
		unichar = unichar & 0x1F;
	} else if ((unichar & 0xF0) == 0xE0) {
		utfcount = 3;
		unichar = unichar & 0x0F;
	} else if ((unichar & 0xF8) == 0xF0) {
		utfcount = 4;
		unichar = unichar & 0x07;
	} else if ((unichar & 0xFC) == 0xF8) {
		utfcount = 5;
		unichar = unichar & 0x03;
	} else if ((unichar & 0xFE) == 0xFC) {
		utfcount = 6;
		unichar = unichar & 0x01;
	} else /* illegal UTF-8 code */ {
		return unichar;
	}
	while (utfcount > 1) {
		unichar = (unichar << 6) | (readchar_nokeymap () & 0x3F);
		utfcount --;
	}
	if (cjk_text == True || visciimode ()) {
		unichar = cjk (unichar, True);
		if (unichar == quit_char && valid == True) {
			return readcharacter_mapping_valid (map_keyboard, valid);
		} else {
			return unichar;
		}
	}
	return unichar;
  }
}

/*-------------------------------------------------------------------------*/

/*
 * DIRECT...getxy () reads in the information part of a cursor escape sequence
 */
static
void
DIRECTxtermgetxy (code)
  char code;	/* 'M' for normal mouse report, 't'/'T' for tracking */
{
  character button;
  int xpos;
  int ypos;
#ifdef mouse_hilite_tracking
  char track [22];
#endif

  mouse_lastbutton = mouse_button;
  if (mouse_button == leftbutton || mouse_button == rightbutton || mouse_button == middlebutton) {
	mouse_prevbutton = mouse_button;
  }
  mouse_lastxpos = mouse_xpos;
  mouse_lastypos = mouse_ypos;

  if (code == 't') {
	mouse_button = releasebutton;
  } else if (code == 'T') {
	/* could be Siemens 97801/97808 page down ... !?! */
	button = _readchar_nokeymap ();
	button = _readchar_nokeymap ();
	button = _readchar_nokeymap ();
	button = _readchar_nokeymap ();
	mouse_button = releasebutton;
  } else {	/* code == 'M' */
	button = _readchar_nokeymap ();
	if (button == quit_char) {
		quit = True;
	}
	button -= 32;
	mouse_shift = button & 0x1C;
	if (button & 32) {
		mouse_button = movebutton;
	} else if (button & 64) {
		button &= 0x03;
		if (button == 0x00) {
			mouse_button = wheelup;
		} else {
			mouse_button = wheeldown;
		}
	} else {
		button &= 0x03;
		if (button == 0x00) {
			mouse_button = leftbutton;
		} else if (button == 0x01) {
			mouse_button = middlebutton;
		} else if (button == 0x02) {
			mouse_button = rightbutton;
		} else {
			mouse_button = releasebutton;
		}
	}
  }

  xpos = _readchar_nokeymap ();
  if (xpos == quit_char) {
	quit = True;
  }
  mouse_xpos = xpos - 33;
  ypos = _readchar_nokeymap ();
  if (ypos == quit_char) {
	quit = True;
  }
  mouse_ypos = ypos - 33 - MENU;

#ifdef mouse_hilite_tracking
  if (mouse_button == leftbutton) {
	/* handle hilite tracking */
	if (mouse_ypos < 0 || mouse_ypos > last_y 
	    || (mouse_xpos == XMAX && disp_scrollbar == True) 
	    || is_menu_open ())
	{
		/* abort hilite tracking */
		putstring ("\033[0;0;0;0;0T");
		flush ();
	} else {
		build_string (track, "\033[1;%d;%d;%d;%dT", 
				mouse_xpos + 1, mouse_ypos + 1 + MENU, 
				1 + MENU, last_y + 3);
		putstring (track);
		flush ();
	}
  }
#endif
}

static
void
DIRECTcrttoolgetxy ()
{
  character c;
  int xpos;
  int ypos;

  mouse_lastbutton = mouse_button;
  if (mouse_button == leftbutton || mouse_button == rightbutton || mouse_button == middlebutton) {
	mouse_prevbutton = mouse_button;
  }
  mouse_lastxpos = mouse_xpos;
  mouse_lastypos = mouse_ypos;

  c = get_digits (& ypos); /* c should be ';' */
  c = get_digits (& xpos);
  if (c == 'l') {
	mouse_button = leftbutton;
  } else if (c == 'm') {
	mouse_button = middlebutton;
  } else if (c == 'r') {
	mouse_button = rightbutton;
  } else {
	mouse_button = releasebutton;
  }

  mouse_ypos = ypos - 1 - MENU;
  mouse_xpos = xpos - 1;
}

/*
 * DIRECT... () reads in a direct cursor movement input sequence.
 * Then it either performs a direct function (move/copy/paste) or 
 * invokes the menu functions.
 * Is no longer called as this is already handled in _readchar ().
 */
void
DIRECTxterm ()
{
  DIRECTxtermgetxy ('M');
  MOUSEescape ();
}
void
TRACKxterm ()
{
  DIRECTxtermgetxy ('t');
  MOUSEescape ();
}
void
TRACKxtermT ()
{
  DIRECTxtermgetxy ('T');
  MOUSEescape ();
}

void
DIRECTcrttool ()
{
  DIRECTcrttoolgetxy ();
  MOUSEescape ();
}


#ifdef CURSES
#ifdef KEY_MOUSE

#ifdef __PDCURSES__

/*
 * CURSgetxy () reads in the information of a mouse event
 */
static
void
CURSgetxy ()
{
# define mouse_state Mouse_status.changes
# define mouse_x MOUSE_X_POS
# define mouse_y MOUSE_Y_POS

  int button = 0;

  mouse_lastbutton = mouse_button;
  if (mouse_button == leftbutton || mouse_button == rightbutton || mouse_button == middlebutton) {
	mouse_prevbutton = mouse_button;
  }
  mouse_lastxpos = mouse_xpos;
  mouse_lastypos = mouse_ypos;

  request_mouse_pos ();

  if (BUTTON_CHANGED (1)) {
	button = 1;
  } else if (BUTTON_CHANGED (2)) {
	button = 2;
  } else if (BUTTON_CHANGED (3)) {
	button = 3;
  }

  mouse_shift = (BUTTON_STATUS (button) & BUTTON_MODIFIER_MASK) == BUTTON_SHIFT;

  if ((BUTTON_STATUS (button) & BUTTON_ACTION_MASK) == BUTTON_PRESSED) {
	if (button == 1) {
		mouse_button = leftbutton;
	} else if (button == 2) {
		mouse_button = middlebutton;
	} else if (button == 3) {
		mouse_button = rightbutton;
	} else {
		mouse_button = releasebutton;
	}
  } else {
	mouse_button = releasebutton;
  }

  mouse_xpos = mouse_x;
  mouse_ypos = mouse_y - MENU;
}

# else

static
void
CURSgetxy ()
{
  MEVENT mouse_info;
# define mouse_state mouse_info.bstate
# define mouse_x mouse_info.x
# define mouse_y mouse_info.y

  mouse_lastbutton = mouse_button;
  if (mouse_button == leftbutton || mouse_button == rightbutton || mouse_button == middlebutton) {
	mouse_prevbutton = mouse_button;
  }
  mouse_lastxpos = mouse_xpos;
  mouse_lastypos = mouse_ypos;

  getmouse (& mouse_info);

  mouse_shift = mouse_state & BUTTON_SHIFT;
  if (mouse_state & BUTTON1_CLICKED) {
	mouse_button = leftbutton;
  } else if (mouse_state & BUTTON2_CLICKED) {
	mouse_button = middlebutton;
  } else if (mouse_state & BUTTON3_CLICKED) {
	mouse_button = rightbutton;
  } else {
	mouse_button = releasebutton;
  }

  mouse_xpos = mouse_x - 1;
  mouse_ypos = mouse_y - 1 - MENU;
}

#endif

#endif
#endif


/*-------------------------------------------------------------------------*/

#ifndef msdos

/* max. length of function key sequence to be detected
   (depending on the fkeymap table, approx. 7),
   or of keyboard mapping sequence plus potential mapping rest
*/
#define MAXCODELEN 33

/*
 * queue collects the keys of an Escape sequence typed in until the 
 * sequence can be detected or rejected.
 * If the queue is not empty, queue [0] contains the character next 
 * to be delivered by _readchar () (it's not a ring buffer).
 * The queue contents is always terminated by a '\0', so queue can also 
 * be taken as a character string.
 */
static character queue [MAXCODELEN + 1], * endp = queue;
static character ctrl_queue [MAXCODELEN + 1], * endcp = ctrl_queue;

#ifdef unused
static
int
q_empty ()
{
  return endp == queue ? 1 : 0;
}
#endif

static
int
q_notfull ()
{
  return endp - queue == MAXCODELEN ? 0 : 1;
}

static
void
q_clear ()
{
  endp = queue;
  endcp = ctrl_queue;
}

static
int
q_len ()
{
  return endp - queue;
}

static
void
q_put (c)
  character c;
/* queue must not be full prior to this call! */
{
  * endp = c;
  * ++ endp = '\0';
  if (c == '\0') {
	* endcp = '@';
  } else {
	* endcp = c;
  }
  * ++ endcp = '\0';
}

/*
static
character
q_unput ()
{
  character c;
  if (q_len () == 0) {
	return '\0';
  } else {
	endp --;
	endcp --;
	c = * endp;
	* endp = '\0';
	* endcp = '\0';
	return c;
  }
}
*/

static
character
q_get ()
{
  character c;
  register character * pd;
  register character * ps;

  c = * queue;
  pd = queue;
  ps = pd + 1;
  while (ps <= endp) {
	* pd ++ = * ps ++;
  }
  if (endp > queue) {
	endp --;
	endcp --;
  }
  return c;
}

/*
 * Look up key sequence in fkeymap table.
 * if matchmode == 0:
 * findkey (str) >=  0: escape sequence found
 *				(str == fkeymap [findkey (str)].fk)
 *			keyproc = fkeymap [i].fp (side effect)
 *		 == -1: escape sequence prefix recognised
 *				(str is prefix of some entry in fkeymap)
 *		 == -2: escape sequence not found
 *				(str is not contained in fkeymap)
 * * found == index of exact match if found
 * if matchmode == 1 (not used):
 * return -1 if any prefix detected (even if other exact match found)
 * if matchmode == 2 (not used):
 * don't return -1, only report exact or no match
 */
static
int
findkeyin (str, fkeymap, matchmode, found)
  char * str;
  struct fkeyentry * fkeymap;
  int matchmode;
  int * found;
{
  int lastmatch = 0;	/* last index with string matching prefix */
  register int i;
  int ret = -2;

  * found = -1;

  if (fkeymap [0].fk == NIL_PTR) {
	return ret;
  }

  i = lastmatch;
  do {
	if (strncmp (str, fkeymap [i].fk, strlen (str)) == 0) {
		/* str is prefix of current entry */
		lastmatch = i;
		if (strlen (str) == strlen (fkeymap [i].fk)) {
			/* str is equal to current entry */
			* found = i;
			keyproc = fkeymap [i].fp;
			if (matchmode == 1) {
				/* return index unless other prefix 
				   was or will be found */
				if (ret == -2) {
					ret = i;
				}
			} else {
				return i;
			}
		} else {
			/* str is partial prefix of current entry */
			if (matchmode == 1) {
				/* return -1 but continue to search 
				   for exact match */
				ret = -1;
			} else if (matchmode != 2) {
				return -1;
			} else {
			}
		}
	}
	++ i;
	if (fkeymap [i].fk == NIL_PTR) {
		i = 0;
		return ret;
	}
  } while (i != lastmatch);

  return ret;
}

/*
   Look up key sequence in keyboard mapping table.
   Similar to findkeyin but with modified parameter details.
 * if matchmode == 0 (not used):
 * mapkb (str) >=  0: key sequence str found
 *			* mapped set to mapped string
 *		 == -1: key sequence prefix recognised
 *				(str is prefix of some table entry)
 *		 == -2: key sequence not found
 *				(str is not contained in table)
 * * found == index of exact match if found
 * if matchmode == 1:
 * return -1 if any prefix detected (even if other exact match found)
 * if matchmode == 2:
 * don't return -1, only report exact or no match
 */
static
int
mapkb (str, kbmap, matchmode, found, mapped)
  char * str;
  keymaptabletype kbmap;
  int matchmode;
  char * * found;
  char * * mapped;
{
#ifdef stringtables

  int lastmatch = 0;	/* last index with string matching prefix */
  register int i;
  int ret = -2;
  int len = strlen (str);

  * found = NIL_PTR;

  if (* kbmap == '\0') {
	return ret;
  }

  i = lastmatch;
  do {
	if (strncmp (str, kbmap, len) == 0) {
		/* str is prefix of current entry */
		lastmatch = i;
		if (len == strlen (kbmap)) {
			/* str is equal to current entry */
			* found = kbmap;
			* mapped = kbmap + len + 1;
			if (matchmode == 1) {
				/* return index unless other prefix 
				   was or will be found */
				if (ret == -2) {
					ret = i;
				}
			} else {
				return i;
			}
		} else {
			/* str is partial prefix of current entry */
			if (matchmode == 1) {
				/* return -1 but continue to search 
				   for exact match */
				ret = -1;
			} else if (matchmode != 2) {
				return -1;
			} else {
			}
		}
	}
	++ i;
	kbmap += strlen (kbmap) + 1;
	kbmap += strlen (kbmap) + 1;
	if (* kbmap == '\0') {
		i = 0;
		return ret;
	}
  } while (i != lastmatch);

  return ret;

#else

  int lastmatch = 0;	/* last index with string matching prefix */
  register int i;
  int ret = -2;

  * found = NIL_PTR;

  if (kbmap [0].fk == NIL_PTR) {
	return ret;
  }

  i = lastmatch;
  do {
	if (strncmp (str, kbmap [i].fk, strlen (str)) == 0) {
		/* str is prefix of current entry */
		lastmatch = i;
		if (strlen (str) == strlen (kbmap [i].fk)) {
			/* str is equal to current entry */
			* found = kbmap [i].fk;
			* mapped = kbmap [i].fp;
			if (matchmode == 1) {
				/* return index unless other prefix 
				   was or will be found */
				if (ret == -2) {
					ret = i;
				}
			} else {
				return i;
			}
		} else {
			/* str is partial prefix of current entry */
			if (matchmode == 1) {
				/* return -1 but continue to search 
				   for exact match */
				ret = -1;
			} else if (matchmode != 2) {
				return -1;
			} else {
			}
		}
	}
	++ i;
	if (kbmap [i].fk == NIL_PTR) {
		i = 0;
		return ret;
	}
  } while (i != lastmatch);

  return ret;

#endif
}

static
int
findkey (str)
  char * str;
{
  int dummy;
  int res = findkeyin (str, fkeymap, 0, & dummy);
  if (res == -2) {
	return findkeyin (str, fkeymap2, 0, & dummy);
  } else {
	return res;
  }
}

#endif	/* ifndef msdos */


#ifdef CURSES
chtype key;

static
void
showfkey ()
{
  build_string (text_buffer, "Curses function key entered: %03X / %04o", key, key);
  status_msg (text_buffer);
}


/*
 * Read a character, ignoring uninterpreted key strokes (SHIFT etc.)
 * also handles resize events
 */
int
curs_readchar ()
{
  int keycode;
  keycode = __readchar ();
  while (keycode >= KEY_MIN) {
	switch (keycode) {
#ifdef KEY_RESIZE
	case KEY_RESIZE:
  printf ("curses resize key received \n"); sleep (1);
			keyproc = showfkey; return FUNcmd;
#endif
#ifdef KEY_SHIFT_L
	case KEY_SHIFT_L:
	case KEY_SHIFT_R:
	case KEY_CONTROL_L:
	case KEY_CONTROL_R:
	case KEY_ALT_L:
	case KEY_ALT_R:
			break;
#endif
	default:	return keycode;
	}
	keycode = __readchar ();
  }

  return keycode;
}

/*
 * Mapping KEY_ code (curses) -> mined function
 */
struct {
	chtype keycode;
	voidfunc func;
} cursfunc [] =
{
#ifdef PAD0
	{PAD0, PASTE},
#endif
	{KEY_F(1), HELP},
	{KEY_F(2), WT},
	{KEY_F(3), EDIT},
	{KEY_F(4), INSFILE},
	{KEY_F(5), insert_diaeresis},
	{KEY_F(6), insert_diaeresis},
	{KEY_F(7), SFW},
	{KEY_F(8), SFW},
	{KEY_F(9), RS},
	{KEY_F(10), FILEMENU},
	{KEY_F(20), QUICKMENU},
	{KEY_F(11), LOWCAP},
	{KEY_F(12), UML},
#ifdef __PDCURSES__
	{PADSLASH, I},
	{PADSTAR, I},
	{PADMINUS, I},
	{PADPLUS, I},
	{PADSTOP, CUT},
	{PADENTER, SNL},
	{ALT_F, FILEMENU},
	{ALT_E, EDITMENU},
	{ALT_S, SEARCHMENU},
	{ALT_X, EXTRAMENU},
	{ALT_P, PARAMENU},
#endif
	{KEY_IC, PASTE},
	{KEY_DC, DCC},
	{KEY_HOME, MARK},
	{KEY_END, COPY},
	{0x1BF, BLINE},
	{0x1C0, ELINE},
	{KEY_PPAGE, PU},
	{KEY_NPAGE, PD},

	{0}
};

/*
 * Look up mined function by curses key code
 * lookup_curskey (str) >=  0: keycode == cursfunc [lookup_curskey (str)].fk
 *			== -1: keycode is not contained in mapping table
 */
static
voidfunc
lookup_curskey (keycode)
  chtype keycode;
{
  register int i;

  i = 0;
  while (cursfunc [i].keycode != 0 && cursfunc [i].keycode != keycode) {
	i ++;
  }
  if (cursfunc [i].keycode == 0) {
	return showfkey;
  } else {
	return cursfunc [i].func;
  }
}

#endif


/*
 * Read a character from terminal, considering function keys and 
 * composing special character of an 8 bit character set.
 * _readchar () takes the following actions:
 *  -	function key sequences according to the table fkeymap are 
 *	transformed into a special controlling character which is 
 *	assigned the function FUNKEY. Also the intended editor function, 
 *	as taken from the table fkeymap, is saved in a variable for 
 *	use by FUNKEY.
 *  -	the prefix keys for diacritic and special characters are 
 *	combined with the following key to make up the character.
 */
int
_readchar_nokeymap ()
{
  int res;
  int prev_dont_suppress_keymap = dont_suppress_keymap;

  dont_suppress_keymap = 0;
  res = _readchar ();
  dont_suppress_keymap = prev_dont_suppress_keymap;

  return res;
}

#ifndef msdos
static int choose_char _((char *));

static character rest_queue [MAXCODELEN + 1];
static FLAG have_rest_queue = False;

static
void
putback_rest (rest)
  char * rest;
{
  char old_rest_queue [MAXCODELEN + 1];

	if (! streq (rest, " ")) {
		if (have_rest_queue == True) {
			/**
			   overdraft characters to be mapped are put back into 
			   rest_queue to be considered for subsequent mapping
				-> Hiragana nihongo
				-> Hiragana nyi
				-> test abcde
			 */
			strcpy (old_rest_queue, (char *) rest_queue);
		} else {
			old_rest_queue [0] = '\0';
		}
		strcpy ((char *) rest_queue, rest);
		strcat ((char *) rest_queue, old_rest_queue);
		have_rest_queue = True;
	}
}

#endif

character
read1byte ()
{
#ifndef msdos
  character ch;
  character * cr;
  if (have_rest_queue == True) {
	ch = * rest_queue;
	cr = rest_queue;
	while (* (cr + 1) != '\0') {
		* cr = * (cr + 1);
		cr ++;
	}
	* cr = '\0';
	if (* rest_queue == '\0') {
		have_rest_queue = False;
	}
	return ch;
  } else
#endif
  {
#ifdef CURSES
	return curs_readchar ();
#else
	return __readchar ();
#endif
  }
}

#ifndef msdos

static
int
bytereadyafter (msec)
  int msec;
{
  return (have_rest_queue == True) || inputreadyafter (input_fd, msec);
}

/*
 * Is a character available within a specified number of milliseconds ?
 */
int
char_ready_within (msec)
  int msec;
{
  return (q_len () > 0) || bytereadyafter (msec);
}

#else	/* ifndef msdos */

int
char_ready_within (msec)
  int msec;
{
  return inputreadyafter (input_fd, msec);
}

#endif

int
_readchar ()
{
  register character ch;
#ifndef msdos
  char second_esc_char;
#endif
#ifdef dosmouse
  int ich;
#endif
#ifndef msdos
  int res;
  char * found;
  char * mapped;
  char * prev_found = NIL_PTR;
#endif
  char * choice;
  int choice_index;
#ifndef msdos
  unsigned long unichar;
  int utflen;
  character cjkbytes [5];
#endif


#ifndef msdos
  if (q_len () > 0) {
	return q_get ();
  }
#endif

  keyshift = '0';

#ifdef CURSES

  key = read1byte ();
  while (key >= KEY_MIN) {
	switch (key) {
#ifdef KEY_A2
	case KEY_A2:
#endif
	case KEY_UP:	keyproc = MUP; return FUNcmd;
#ifdef KEY_C2
	case KEY_C2:
#endif
	case KEY_DOWN:	keyproc = MDN; return FUNcmd;
#ifdef KEY_B1
	case KEY_B1:
#endif
	case KEY_LEFT:	keyproc = MLF; return FUNcmd;
#ifdef KEY_B3
	case KEY_B3:
#endif
	case KEY_RIGHT:	keyproc = MRT; return FUNcmd;
	case KEY_B2:	keyproc = HOP; return FUNcmd;
	case KEY_A1:	keyproc = MARK; return FUNcmd;
	case KEY_A3:	keyproc = PU; return FUNcmd;
	case KEY_C1:	keyproc = COPY; return FUNcmd;
	case KEY_C3:	keyproc = PD; return FUNcmd;
#ifdef KEY_MOUSE
	case KEY_MOUSE:
			CURSgetxy ();
			keyproc = MOUSEescape;
			return FUNcmd;
#endif

	default:
		keyproc = lookup_curskey (key); return FUNcmd;
	}
	/* loop should not occur, handled in curs_readchar */
	key = read1byte ();
  }
  ch = key;

#else	/* ifdef CURSES */

#ifdef dosmouse
  ich = __readchar ();
  if (ich < 0) {
	DIRECTdosmousegetxy ();
	keyproc = MOUSEescape;
	return FUNcmd /* index of FUNKEY */;
  } else {
	ch = ich;
  }
#else
  ch = read1byte ();
#endif

#endif	/* else CURSES */


#ifdef msdos
  if (ch == '\000') {
# ifdef dosmouse
	ch = getch ();
# else
	ch = read1byte ();
# endif
	keyproc = pc_key_map [ch];
# ifdef DEBUG
	if ((voidfunc) keyproc == (voidfunc) I) {
		return ch;
	}
	/* (voidfunc) is an identity cast here. It seems to be required 
	   for the sake of the apparently totally rotten microvax compiler */
# endif
	return FUNcmd /* index of FUNKEY */;
  }
#else
  if (ch == '\000') {
	keyproc = MARK;
	return FUNcmd;
  }
#endif


#ifndef msdos

    else if (ch == '\033') {
	q_put (ch);
	second_esc_char = ch;
	while  ((res = findkey (ctrl_queue)) == -1 /* prefix of table entry */
		&& q_notfull ()
		&& bytereadyafter (escape_delay)
	       )
	{
		ch = read1byte ();

		/* handle generic shift state of ANSI sequences: */
		if (ch == ';' && second_esc_char == '[') {
			keyshift = read1byte () - 1;
			ch = read1byte ();
		} else if (ch >= '0' && ch <= '9' && second_esc_char == 'O') {
			keyshift = ch - 1;
			ch = read1byte ();
		}

		q_put (ch);
		if (second_esc_char == '\033') {
			second_esc_char = ch;
		}
	}

	if (quit == True) {
		return '\0';
	} else if (res < 0) { /* key pattern not detected in fkeymap table */
		return q_get () /* just deliver the typed characters */;
	} else {
		q_clear ();

		if (keyproc == DIRECTxterm) {
			DIRECTxtermgetxy ('M');
			keyproc = MOUSEescape;
		} else if (keyproc == TRACKxterm) {
			DIRECTxtermgetxy ('t');
			keyproc = MOUSEescape;
		} else if (keyproc == TRACKxtermT) {
			DIRECTxtermgetxy ('T');
			keyproc = MOUSEescape;
		} else if (keyproc == DIRECTcrttool) {
			DIRECTcrttoolgetxy ();
			keyproc = MOUSEescape;
		}

		return FUNcmd /* index of FUNKEY */;
	}
  }
#define dont_debug_kbmap
    else if (allow_keymap && dont_suppress_keymap
		&& keyboard_map != (keymaptabletype) NIL_PTR) {
	q_put (ch);
	while  ((res = mapkb (ctrl_queue, keyboard_map, 1, & found, & mapped)) == -1
		 /* prefix of some table entry */
		&& q_notfull ()
		&& bytereadyafter (keymap_delay)
	       )
	{
		q_put (read1byte ());
		/* exact match found? note for later consideration */
		if (found != NIL_PTR) {
			prev_found = found;
		}
	}
	/* possible results:
		res >= 0: match found, no further prefix match
		res == -1, prev_found != NIL_PTR: prefix with previous match
		res == -1: prefix only, no further key typed
		res == -2: no match with previous match
	*/

#ifdef debug_kbmap
	printf ("mapping %s, res %d, found %s -> ", ctrl_queue, res, found);
#endif
	if (res < 0 && prev_found != NIL_PTR) {
		res = 1;
		if (found == NIL_PTR) {
#ifdef debug_kbmap
	printf ("rest %s, ", ctrl_queue + strlen (prev_found));
#endif
			putback_rest (ctrl_queue + strlen (prev_found));
		}
	} else if (res == -1) {
		res = mapkb (ctrl_queue, keyboard_map, 2, & found, & mapped);
#ifdef debug_kbmap
	printf ("find again %s, found %s, ", ctrl_queue, found);
#endif
	}

#ifdef debug_kbmap
	printf ("res %d\n", res);
#endif
	if (res >= 0) { /* key pattern detected in key mapping table */
		q_clear ();

		/* check if mapping defines multiple choices */
		while (* mapped == ' ') {
			mapped ++;
		}
		choice = strchr (mapped, ' ');
		while (choice != NIL_PTR && * choice == ' ') {
			choice ++;
		}
		if (choice != NIL_PTR && * choice != '\0') {
			choice_index = choose_char (mapped);
#ifdef debug_kbmap
	printf (" choose_char (%s) -> %d\n", mapped, choice_index);
#endif
			if (choice_index < 0) {
				mapped = NIL_PTR;
				/* mapping cancelled */
			} else {
				while (choice_index > 0
					&& mapped != NIL_PTR
					&& * mapped != '\0')
				{
					mapped = strchr (mapped, ' ');
					if (mapped != NIL_PTR) {
						while (* mapped == ' ') {
							mapped ++;
						}
					}
					choice_index --;
				}
			}
		}

		if (mapped != NIL_PTR) {
		    if (cjk_text == True && utf8_input == False) {
			utf8_info (mapped, & utflen, & unichar);
#ifdef debug_kbmap
	printf (" mapped %04X -> %04X\n", unichar, cjk (unichar, False));
#endif
			unichar = cjk (unichar, True);
			if (unichar != quit_char) {
				(void) cjkencode (unichar, cjkbytes);
				mapped = (char *) cjkbytes;
				while (* mapped != '\0') {
					q_put (* mapped);
					mapped ++;
				}
			}
		    } else {
			while (* mapped != '\0' && * mapped != ' ') {
				q_put (* mapped);
				mapped ++;
			}
		    }
		}
#ifdef debug_kbmap
	printf (" queued %s\n", ctrl_queue);
#endif
	}

	if (q_len () > 0) {
		return q_get () /* deliver the first mapped byte */;
	} else {
		ring_bell ();
		flush ();	/* clear key mapping menu */
		return _readchar ();
	}
  }

#endif	/* #ifndef msdos */

    else {
	return ch;
  }
}

#ifndef msdos

static
void
dummyfunc ()
{
}

#define dont_debug_choose_char

static
int
choose_char (choices)
  char * choices;
{
  int choice_count = 0;
  char * choicepoi = choices;
  int choice_col = x;
  int choice_line = y + 1;

  char * thischoiceline;
  char * choicelines;
  menuitemtype * choicemenu;

  int li, ci, lastcol, lastrow;
  char * choicelinepoi;

  int selected;

  int in_status_line = current_cursor_y == YMAX;

	/* count choices */
	do {
		choice_count ++;
		choicepoi = strchr (choicepoi, ' ');
		while (choicepoi != NIL_PTR && * choicepoi == ' ') {
			choicepoi ++;
		}
	} while (choicepoi != NIL_PTR && * choicepoi != '\0');
#ifdef debug_choose_char
	printf ("choices <%s>\n", choices);
	printf ("choice_count %d\n", choice_count);
#endif

	lastrow = (choice_count - 1) / 10;
	choicelines = alloc (strlen (choices) + 1 + choice_count * 2);
	choicemenu = (menuitemtype *) alloc ((lastrow + 1) * sizeof (menuitemtype));
	if (choicelines == NIL_PTR || choicemenu == (menuitemtype *) NIL_PTR) {
		if (choicelines != NIL_PTR) {
			free_space (choicelines);
		}
		ring_bell ();
		flush ();
		return -1;
	}

	/* adjust menu position if too far down */
	if (choice_line + lastrow + 3 > YMAX) {
		choice_line = y - lastrow - 3;
		if (choice_line < 0) {
			choice_line = 0;
		}
	}

	/* adjust menu position if on status line */
	if (in_status_line) {
		choice_line = YMAX - lastrow - 3;
		if (choice_line < 0) {
			choice_line = 0;
		}
		choice_col = lpos;
	}

	/* construct menu items */
	choicepoi = choices;
	while (* choicepoi == ' ') {
		choicepoi ++;
	}
	thischoiceline = choicelines;
	for (li = 0; li <= lastrow; li ++) {
		/* construct menu line */
		choicelinepoi = thischoiceline;
		if (li == lastrow) {
			lastcol = (choice_count - 1) % 10;
		} else {
			lastcol = 9;
		}
		for (ci = 0; ci <= lastcol; ci ++) {
			* choicelinepoi ++ = (ci + 1) % 10 + '0';
			* choicelinepoi ++ = ':';
			while (* choicepoi != ' ' && * choicepoi != '\0') {
				* choicelinepoi ++ = * choicepoi ++;
			}
			while (* choicepoi == ' ') {
				choicepoi ++;
			}
			if (ci < lastcol) {
				* choicelinepoi ++ = ' ';
			}
		}
		* choicelinepoi ++ = '\0';
		/* add menu line to menu */
#ifdef debug_choose_char
	printf ("choiceline <%s>\n", thischoiceline);
#endif
		choicemenu [li].itemname = thischoiceline;
		choicemenu [li].itemfu = dummyfunc;
		choicemenu [li].hopitemname = "";

		thischoiceline = choicelinepoi;
	}
#ifdef debug_choose_char
	printf ("choicelines alloc %d length %d\n",
		strlen (choices) + 1 + choice_count * 2 + lastrow + 1,
		thischoiceline - choicelines);
#endif

	selected = popup_menu (choicemenu, lastrow + 1, 
				choice_col, choice_line, 
				"Choose character", "1234567890");
	if (in_status_line) {
		redraw_prompt ();
	}
#ifdef debug_choose_char
	printf ("selected %d\n", selected);
#endif

	free_space (choicemenu);
	free_space (choicelines);

	if (selected < choice_count) {
		return selected;
	} else {
		return -1;
	}
}

#endif


/*======================================================================*\
|*				End					*|
\*======================================================================*/
