/*======================================================================*\
|*		Editor mined						*|
|*		part 2							*|
\*======================================================================*/

#include "mined.h"


/*======================================================================*\
|*			Forward declarations				*|
\*======================================================================*/

static void file_insert ();
static void yank_text ();
static int forward_scroll ();
static int reverse_scroll ();
static int legal ();
static FLAG checkmark ();


/*======================================================================*\
|*		Global variables and collective buffer handling		*|
\*======================================================================*/

FLAG yank_status = NOT_VALID;	/* Status of yank_file */
FLAG html_status = NOT_VALID;	/* Status of html_file */
FLAG redraw_pending = False;	/* was a redraw suppressed in find_y ? */
int buffer_open_flag = 0;	/* Counter flag for the collective buffer */
static int yank_buf_no = 0;	/* Buffer # for trials and multiple buffers */
static int max_yank_buf_no = 0;	/* Max Buffer # used */

static LINE * pasted_start_line = NIL_LINE;
static char * pasted_start_textp = NIL_PTR;
static LINE * pasted_end_line = NIL_LINE;
static char * pasted_end_textp = NIL_PTR;

static
void
set_buffer_open (appending)
  FLAG appending;
{
#ifdef debug_ring_buffer
  printf ("set_buffer_open %d\n", buffer_open_flag);
#endif
  if (buffer_open_flag == 0 && (appending == False || yank_buf_no == 0)) {
	yank_buf_no ++;
	if (yank_buf_no > max_yank_buf_no) {
		max_yank_buf_no = yank_buf_no;
	}
	yank_status = NOT_VALID;
#ifdef debug_ring_buffer
	flags_changed = True;
#endif
  }
  buffer_open_flag = 2;
}

static
void
close_buffer ()
{
#ifdef debug_ring_buffer
  if (buffer_open_flag > 0) {
	flags_changed = True;
  }
#endif
  buffer_open_flag = 0;
}

static
void
revert_yank_buf ()
{
  yank_buf_no --;
  if (yank_buf_no <= 0) {
	yank_buf_no = max_yank_buf_no;
  }
}

static
void
delete_text_buf (start_line, start_textp, end_line, end_textp)
  register LINE * start_line;
  LINE * end_line;
  char * start_textp;
  char * end_textp;
{
  if (emacs_buffer == True) {
	set_buffer_open (False);
	yank_text (yankfile (WRITE, True), & yank_status, 
			start_line, start_textp, end_line, end_textp, 
			DELETE, True);
  } else {
	(void) delete_text (start_line, start_textp, end_line, end_textp);
  }
}


/*======================================================================*\
|*			Definitions specific for mined2.c		*|
\*======================================================================*/

#ifdef unix
#define linkyank
#endif

#ifdef __pcgcc__
#undef linkyank
#endif
#ifdef __CYGWIN__
#define linkyank
#endif

/*
 * viewonlyerr () outputs an error message with a beep
 */
void
viewonlyerr ()
{
  ring_bell ();
  error ("View only mode", NIL_PTR);
}

/*
 * restrictederr () outputs an error message with a beep
 */
void
restrictederr ()
{
  ring_bell ();
  error ("Restricted mode", NIL_PTR);
}


/*======================================================================*\
|*			Move commands					*|
\*======================================================================*/

/*
 * Move one line up.
 */
void
MUP ()
{
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	MPPARA ();
	return;
  }

  if (hop_flag > 0) {
	HIGH ();
  } else if (y == 0) {		/* Top line of screen. Scroll one line */
	if (reverse_scroll (True) != ERRORS) {
		move_y (y);
	}
  } else {			/* Move to previous line */
	move_y (y - 1);
  }
}

/*
 * Move one line down.
 */
void
MDN ()
{
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	MNPARA ();
	return;
  }

  if (hop_flag > 0) {
	LOW ();
  } else if (y == last_y) {	/* Last line of screen. Scroll one line */
	if (bot_line->next == tail && bot_line->text [0] != '\n') {
		return;
	} else {
		(void) forward_scroll (True);
		move_y (y);
	}
  } else {			/* Move to next line */
	move_y (y + 1);
  }
}

/*
 * Move left one position.
 */
void
MLF ()
{
  if ((keyshift & ctrlshift_mask) == ctrlshift_mask) {
	keyshift = '0';
	BLINE ();
	return;
  }
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	ctrl_MLF ();
	return;
  }
  if (keyshift & shift_mask) {
	keyshift = '0';
	MPW ();
	return;
  }

  if (hop_flag > 0) {
	BLINE ();
  } else if (x == 0 && get_shift (cur_line->shift_count) == 0) {
	/* Begin of line */
	if (cur_line->prev != header) {
		MUP ();				/* Move one line up */
		move_to (LINE_END, y);
	}
  } else {
	move_to (x - 1, y);
  }
}

/*
 * Move right one position.
 */
void
MRT ()
{
  if ((keyshift & ctrlshift_mask) == ctrlshift_mask) {
	keyshift = '0';
	ELINE ();
	return;
  }
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	ctrl_MRT ();
	return;
  }
  if (keyshift & shift_mask) {
	keyshift = '0';
	MNW ();
	return;
  }

  if (hop_flag > 0) {
	ELINE ();
  } else if (* cur_text == '\n') {
	if (cur_line->next != tail) {		/* Last char of file */
		MDN ();				/* Move one line down */
		move_to (LINE_START, y);
	}
  } else {
	move_to (x + 1, y);
  }
}

/*
 * Move left one Unicode character (may enter into combined character).
 */
void
ctrl_MLF ()
{
  char * curpoi;

  if (hop_flag > 0) {
	BLINE ();
  } else if (cur_text == cur_line->text) {/* Begin of line */
	if (cur_line->prev != header) {
		MUP ();				/* Move one line up */
		move_to (LINE_END, y);
	}
  } else {
	curpoi = cur_text;
	precede_char (& curpoi, cur_line->text);
	move_address (curpoi, y);
  }
}

/*
 * Move right one Unicode character (may enter into combined character).
 */
void
ctrl_MRT ()
{
  char * curpoi;

  if (hop_flag > 0) {
	ELINE ();
  } else if (* cur_text == '\n') {
	if (cur_line->next != tail) {		/* Last char of file */
		MDN ();				/* Move one line down */
		move_to (LINE_START, y);
	}
  } else {
	curpoi = cur_text;
	advance_char (& curpoi);
	move_address (curpoi, y);
  }
}

/*
 * Move to top of screen
 */
void
HIGH ()
{
  move_y (0);
}

/*
 * Move to bottom of screen
 */
void
LOW ()
{
  move_y (last_y);
}

/*
 * Move to begin of line.
 */
void
BLINE ()
{
  move_to (LINE_START, y);
}

/*
 * Move to end of line.
 */
void
ELINE ()
{
  move_to (LINE_END, y);
}

/*
 * GOTO () prompts for a linenumber and moves to that line.
 */
void
goline (number)
  int number;
{
  LINE * line;
  if (number <= 0 || (line = proceed (header->next, number - 1)) == tail) {
	error ("Invalid line number: ", num_out ((long) number));
  } else {
	Pushmark ();
	clear_status ();
	move_y (find_y (line));
  }
}

void
goproz (number)
  int number;
{
  goline ((long) (total_lines - 1) * number / 100 + 1);
}

void
GOTO ()
{
  unsigned long c;
  char end;
  int number;

  if (MENU) {
	hop_flag = 1;
	displayflags ();
	set_cursor_xy ();
	flush ();
	hop_flag = 0;
  }
  if (! char_ready_within (500)) {
	status_msg ("HOP: type command (fortified) or number...");
  }
  if (quit == True) {
	return;
  }
  c = readcharacter ();
  if (quit == True) {
	return;
  }
  if ('0' <= c && c <= '9') {
	if (lines_per_page > 0) {
		end = get_number ("...number, then go line / %(of lines) / p(age / m(ark / g(o marker / f(ile #", c, & number);
	} else {
		end = get_number ("...number, then go line / %(of lines) / m(ark / g(o marker / f(ile #", c, & number);
	}
	if (end == '%') {
		goproz (number);
	} else if (end == 'm' || end == 'M' || end == ',') {
		MARKn (number);
	} else if (end == '\'' || end == '.' || end == 'g' || end == 'G') {
		GOMAn (number);
	} else if (end == 'f' || end == 'F' || end == '#') {
		edit_nth_file (number);
	} else if (lines_per_page > 0 && (end == 'p' || end == 'P') && number > 0) {
		goline (number * lines_per_page - lines_per_page + 1);
	} else if (end != ERRORS) {
		goline (number);
	}
	return;
  } else {
	clear_status ();
	hop_flag = 1;
	invoke_key_function (c);
	return;
  }
}

/*
 * Scroll forward one page or to eof, whatever comes first. (Bot_line becomes
 * top_line of display.) Try to leave the cursor on the same line. If this is
 * not possible, leave cursor on the line halfway the page.
 */
void
PD ()
{
  register int i;
  int new_y;

  if (keyshift & ctrl_mask) {
	keyshift = '0';
	SD ();
	return;
  }

  if (hop_flag > 0) {
	hop_flag = 0;
	EFILE ();
	return;
  }

  for (i = 0; i < SCREENMAX; i ++) {
	if (forward_scroll (page_scroll) == ERRORS) {
		break;			/* EOF reached */
	}
  }

  if (y - i < 0) {			/* Line no longer on screen */
	new_y = (page_stay == True) ? 0 : SCREENMAX >> 1;
  } else {
	new_y = y - i;
  }

  if (page_scroll == False) {
	display (0, top_line, last_y, new_y);
  }

  move_y (new_y);
}

/*
 * Scroll backwards one page or to top of file, whatever comes first.
 * (Top_line becomes bot_line of display).
 * The very bottom line (YMAX) is always blank.
 * Try to leave the cursor on the same line.
 * If this is not possible, leave cursor on the line halfway the page.
 */
void
PU ()
{
  register int i;
  int new_y;

  if (keyshift & ctrl_mask) {
	keyshift = '0';
	SU ();
	return;
  }

  if (hop_flag > 0) {
	hop_flag = 0;
	BFILE ();
	return;
  }

  for (i = 0; i < SCREENMAX; i ++) {
	if (reverse_scroll (page_scroll) == ERRORS) {
		/* should also flag reverse_scroll that clearing of 
		   bottom line is not desired */
		break;			/* Top of file reached */
	}
  }

  if (y + i > SCREENMAX) {		/* line no longer on screen */
	new_y = (page_stay == True) ? last_y : SCREENMAX >> 1;
  } else {
	new_y = y + i;
  }

  if (can_scroll_reverse == True && page_scroll == True) {
	set_cursor (0, YMAX);	/* Erase very bottom line */
	clear_lastline ();
  } else {
	display (0, top_line, last_y, new_y);
  }

  move_y (new_y);
}

/*
 * Go to top of file, scrolling if possible, else redrawing screen.
 */
void
BFILE ()
{
  Pushmark ();

  if (proceed (top_line, - SCREENMAX) == header) {
	PU ();				/* It fits. Let PU do it */
  } else {
	reset (header->next, 0);	/* Reset top_line, etc. */
	RD_y (0);			/* Display full page */
  }
  move_to (LINE_START, 0);
}

/*
 * Go to last position of text, scrolling if possible, else redrawing screen
 */
void
EFILE ()
{
  Pushmark ();

  if (proceed (bot_line, SCREENMAX) == tail) {
	PD ();			/* It fits. Let PD do it */
  } else {
	reset (proceed (tail->prev, - SCREENMAX), SCREENMAX);
	RD_y (last_y);		/* Display full page */
  }
  move_to (LINE_END, last_y);
}

/*
 * Scroll one line up. Leave the cursor on the same line (if possible).
 */
void
SU ()
{
  register int i;

  if (hop_flag > 0) {
	hop_flag = 0;
	for (i = 0; i < (SCREENMAX >> 1); i ++) {
		if (i > 0 && disp_scrollbar == True) {
			(void) display_scrollbar (True);
		}
		SU ();
	}
	return;
  }

  if (reverse_scroll (True) != ERRORS) {	/* else we are at top of file */
	move_y ((y == SCREENMAX) ? SCREENMAX : y + 1);
  }
}

/*
 * Scroll one line down. Leave the cursor on the same line (if possible).
 */
void
SD ()
{
  register int i;

  if (hop_flag > 0) {
	hop_flag = 0;
	for (i = 0; i < (SCREENMAX >> 1); i ++) {
		if (i > 0 && disp_scrollbar == True) {
			(void) display_scrollbar (True);
		}
		SD ();
	}
	return;
  }

  if (forward_scroll (True) != ERRORS) {
	move_y ((y == 0) ? 0 : y - 1);
  }
}

/*
 * Perform a forward scroll. It returns ERRORS if we're at the last line of
 * the file.
 */
static
int
forward_scroll (update)
  FLAG update;
{
  if (bot_line->next == tail) {		/* Last line of file. No dice */
	return ERRORS;
  }
  top_line = top_line->next;
  bot_line = bot_line->next;
  cur_line = cur_line->next;
  line_number ++;

/* Perform the scroll on screen */
  if (update == True) {
	clean_menus ();
	scroll_forward ();
	scrollbar_scroll_up (0);
	set_cursor (0, SCREENMAX);
	line_print (SCREENMAX, bot_line);
  }

  return FINE;
}

/*
 * Perform a backwards scroll. It returns ERRORS if we're at the first line 
 * of the file. It updates the display completely if update is True. 
 * Otherwise it leaves that to the caller (page up function).
 */
static
int
reverse_scroll (update)
  FLAG update;
{
  if (top_line->prev == header) {
	/* Top of file. Can't scroll */
	return ERRORS;
  }

  if (last_y != SCREENMAX) {	/* Reset last_y if necessary */
	last_y ++;
  } else {
	bot_line = bot_line->prev;	/* Else adjust bot_line */
  }
  top_line = top_line->prev;
  cur_line = cur_line->prev;
  line_number --;

/* Perform the scroll on screen */
  if (update == True) {
    if (can_scroll_reverse == True) {
	clean_menus ();
	set_cursor (0, 0);
	scroll_reverse ();
	scrollbar_scroll_down (0);
	set_cursor (0, YMAX);	/* Erase very bottom line */
	clear_lastline ();
	set_cursor (0, 0);
	line_print (0, top_line);
    } else {
	display (0, top_line, last_y, y);
    }
  }

  return FINE;
}


/*----------------------------------------------------------------------*\
	Contents-dependent moves
\*----------------------------------------------------------------------*/

/*
 * A word was previously defined as a number of non-blank characters
 * separated by tabs, spaces or linefeeds.
 * By consulting idfchar (), sequences of real letters only or digits
 * or underlines are recognized as words.
 */

/*
 * BSEN () and ESEN () look for the beginning or end of the current sentence.
 */
void
BSEN ()
{
  search_for ("[;.]", REVERSE);
}

void
ESEN ()
{
  search_for ("[;.]", FORWARD);
}

/*
 * MNPARA () and MPPARA () look for end or beginning of current paragraph.
 */
void
MNPARA ()
{
  do {
	if (cur_line->next == tail) {
		ELINE ();
		break;
	}
	if (JUSmode == 0 && cur_line->text [strlen (cur_line->text) - 2] != ' ') {
		MDN ();
		BLINE ();
		break;
	}
	MDN ();
	if (JUSmode == 1 && * (cur_line->text) == '\n') {
		break;
	}
  } while (True);
}

void
MPPARA ()
{
  if (JUSmode == 0 && cur_text == cur_line->text) {
	/* prevent sticking if already at paragraph beginning */
	MUP ();
  }
  do {
	if (cur_line->prev == header) {
		BLINE ();
		break;
	}
	if (JUSmode == 0 && cur_line->prev->text [strlen (cur_line->prev->text) - 2] != ' ') {
		BLINE ();
		break;
	}
	MUP ();
	if (JUSmode == 1 && * (cur_line->text) == '\n') {
		break;
	}
  } while (True);
}

void
search_tag (poi)
  char * poi;
{
  char pat [maxLINE_LEN];
  FLAG direction = FORWARD;
  char * patpoi = & pat [3];

  strcpy (pat, "</*");
  poi ++;
  if (* poi == '/') {
	direction = REVERSE;
	poi ++;
  }
  while (! white_space (* poi) && * poi != '\n' && * poi != '>' && * poi != '\0') {
	* patpoi = * poi;
	patpoi ++;
	poi ++;
  }
  * patpoi = '\0';

  search_corresponding (pat, direction, "/");
}

/*
 * SCORR () looks for a corresponding bracket ([<->] etc.)
 */
void
SCORR ()
{
  char * poi;
  unsigned int cv = charvalue (cur_text);

  if (hop_flag > 0) {
	hop_flag = 0;
	search_wrong_enc ();
	return;
  }

  switch (cv) {
    case '(':	search_corresponding ("[()]", FORWARD, ")"); return;
    case ')':	search_corresponding ("[()]", REVERSE, "("); return;
    case '[':	search_corresponding ("[\\[\\]]", FORWARD, "]"); return;
    case ']':	search_corresponding ("[\\[\\]]", REVERSE, "["); return;
    case '{':	search_corresponding ("[{}]", FORWARD, "}"); return;
    case '}':	search_corresponding ("[{}]", REVERSE, "{"); return;
    case (character) '': /* « LEFT-POINTING DOUBLE ANGLE QUOTATION MARK */
		if (utf8_text == True) {
			search_corresponding ("[«»]", FORWARD, "»"); return;
		} else {
			search_corresponding ("[]", FORWARD, ""); return;
		}
    case (character) '': /* » RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK */
		if (utf8_text == True) {
			search_corresponding ("[«»]", REVERSE, "«"); return;
		} else {
			search_corresponding ("[]", REVERSE, ""); return;
		}
    case 0x2039:	/* ‹ SINGLE LEFT-POINTING ANGLE QUOTATION MARK */
		search_corresponding ("[‹›]", FORWARD, "›"); return;
    case 0x203A:	/* › SINGLE RIGHT-POINTING ANGLE QUOTATION MARK */
		search_corresponding ("[‹›]", REVERSE, "‹"); return;
    case 0x2045:	/* ⁅ LEFT SQUARE BRACKET WITH QUILL */
		search_corresponding ("[⁅⁆]", FORWARD, "⁆"); return;
    case 0x2046:	/* ⁆ RIGHT SQUARE BRACKET WITH QUILL */
		search_corresponding ("[⁅⁆]", REVERSE, "⁅"); return;
    case 0x207D:	/* ⁽ SUPERSCRIPT LEFT PARENTHESIS */
		search_corresponding ("[⁽⁾]", FORWARD, "⁾"); return;
    case 0x207E:	/* ⁾ SUPERSCRIPT RIGHT PARENTHESIS */
		search_corresponding ("[⁽⁾]", REVERSE, "⁽"); return;
    case 0x208D:	/* ₍ SUBSCRIPT LEFT PARENTHESIS */
		search_corresponding ("[₍₎]", FORWARD, "₎"); return;
    case 0x208E:	/* ₎ SUBSCRIPT RIGHT PARENTHESIS */
		search_corresponding ("[₍₎]", REVERSE, "₍"); return;
    default:	if (dim_HTML == True) {
			poi = cur_text;
			if (* poi == '>') {
				MLF ();
			}
			while (poi != cur_line->text && * poi != '<') {
				precede_char (& poi, cur_line->text);
			}
			if (* poi == '<') {
				search_tag (poi);
			} else {
				error ("No bracket or tag to match", NIL_PTR); return;
			}
		} else {
			if (* cur_text == '<') {
				search_corresponding ("[<>]", FORWARD, ">"); return;
			} else if (* cur_text == '>') {
				search_corresponding ("[<>]", REVERSE, "<");
			} else {
				error ("No bracket to match", NIL_PTR); return;
			}
		}
  }
}

/*
   get_idf extracts the identifier at the current position (text) 
   into idf_buf.
   start points to the beginning of the current line.
 */
int
get_idf (idf_buf, text, start)
  char * idf_buf;
  char * text;
  char * start;
{
  char * idf_buf_poi = idf_buf;
  char * idf_poi;
  char * copy_poi;

  if (! idfchar (text)) {
	error ("No identifier", NIL_PTR);
	return ERRORS;
  } else {
	idf_poi = text;
	while (idfchar (idf_poi) && idf_poi != start) {
		precede_char (& idf_poi, start);
	}
	if (! idfchar (idf_poi)) {
		advance_char (& idf_poi);
	}
	while (idfchar (idf_poi)) {
		copy_poi = idf_poi;
		advance_char (& idf_poi);
		while (copy_poi != idf_poi) {
			* idf_buf_poi ++ = * copy_poi ++;
		}
	}
	* idf_buf_poi = '\0';
	return FINE;
  }
}

/*
 * SIDF () searches for the identifier at the current position
 */
void
SIDF (method)
  FLAG method;
{
  char idf_buf [MAX_CHARS];	/* identifier to search for */

  int ret = get_idf (idf_buf, cur_text, cur_line->text);
  if (ret == ERRORS) {
	return;
  }

  search_expr (idf_buf, method);
}

/*
 * MPW () moves to the start of the previous word. A word is defined as a
 * number of non-blank characters separated by tabs spaces or linefeeds.
 */
void
move_previous_word (remove)
  FLAG remove;
{
  register char * begin_line;
  char * textp;
  char start_char = * cur_text;
  char * start_pos = cur_text;
  FLAG idfsearch;

  if (remove == DELETE && viewonly == True) {
	viewonlyerr ();
	return;
  }

/* First check if we're at the beginning of line. */
  if (cur_text == cur_line->text) {
	if (cur_line->prev == header) {
		return;
	}
	start_char = '\0';
  }

  MLF ();

  begin_line = cur_line->text;
  textp = cur_text;

/* Check if we're in the middle of a word. */
  if (! alpha (* textp) || ! alpha (start_char)) {
	while (textp != begin_line 
		&& (white_space (* textp) || * textp == '\n'))
	{
		precede_char (& textp, begin_line);
	}
  }

/* Now we're at the end of previous word. Skip non-blanks until a blank comes */
  if (wordnonblank == True) {
	while (textp != begin_line && alpha (* textp)) {
		precede_char (& textp, begin_line);
	}
  } else {
	if (idfchar (textp)) {
		idfsearch = True;
		while (textp != begin_line && idfchar (textp)) {
			precede_char (& textp, begin_line);
		}
	} else {
		idfsearch = False;
		while (textp != begin_line 
			&& alpha (* textp) && ! idfchar (textp))
		{
			precede_char (& textp, begin_line);
		}
	}
  }

/* Go to the next char if we're not at the beginning of the line */
/* At the beginning of the line, check whether to stay or to go to the word */
  if (textp != begin_line && * textp != '\n') {
	advance_char (& textp);
  } else if (textp == begin_line && * textp != '\n' &&
	     ((wordnonblank == True) ? * textp == ' '
		:   ((idfsearch == True) ? ! idfchar (textp)
			: (! alpha (* textp) || idfchar (textp)))))
  {
	advance_char (& textp);
	if (white_space (* textp) || textp == start_pos) {
		/* no word there or not moved, so go back */
		precede_char (& textp, begin_line);
	}
  }

/* Find the x-coordinate of this address, and move to it */
  move_address (textp, y);
  if (remove == DELETE) {
	(void) delete_text (cur_line, textp, cur_line, start_pos);
  }
}

void
MPW ()
{
  if (hop_flag > 0) {
	BSEN ();
  } else {
	move_previous_word (NO_DELETE);
  }
}

/*
 * MNW () moves to the start of the next word. A word is defined as a number of
 * non-blank characters separated by tabs spaces or linefeeds. Always keep in
 * mind that the pointer shouldn't pass the '\n'.
 */
void
move_next_word (remove)
  FLAG remove;
{
  char * textp = cur_text;

  if (remove == DELETE && viewonly == True) {
	viewonlyerr ();
	return;
  }

/* Move to the end of the current word. */
  if (wordnonblank == True) {
	if (* textp != '\n') {
		advance_char (& textp);
	}
	while (alpha (* textp)) {
		advance_char (& textp);
	}
  } else {
	if (idfchar (textp)) {
		while (* textp != '\n' && idfchar (textp)) {
			advance_char (& textp);
		}
	} else {
		while (alpha (* textp) && ! idfchar (textp)) {
			advance_char (& textp);
		}
	}
  }

/* Skip all white spaces */
  while (* textp != '\n' && white_space (* textp)) {
	textp ++;
  }
/* If we're deleting, delete the text in between */
  if (remove == DELETE) {
	delete_text_buf (cur_line, cur_text, cur_line, textp);
	return;
  }

/* If we're at end of line, move to the beginning of (first word on) the next line */
  if (* textp == '\n' && cur_line->next != tail) {
	MDN ();
	move_to (LINE_START, y);
	textp = cur_text;
/*
	while (* textp != '\n' && white_space (* textp)) {
		textp ++;
	}
*/
  }
  move_address (textp, y);
}

void
MNW ()
{
  if (hop_flag > 0) {
	ESEN ();
  } else {
	move_next_word (NO_DELETE);
  }
}

/*
 * find_y () checks if the matched line is on the current page. If it is, it
 * returns the new y coordinate, else it displays the correct page with the
 * matched line in the middle (unless redrawflag is set to False) 
 * and returns the new y value.
 */
int
find_y_RD (match_line, redrawflag)
  LINE * match_line;
  FLAG redrawflag;
{
  register LINE * line;
  register int count = 0;

/* Check if match_line is on the same page as currently displayed. */
  for (line = top_line; line != match_line && line != bot_line->next;
						      line = line->next) {
	count ++;
  }
  if (line != bot_line->next) {
	return count;
  }

/* Display new page, with match_line in center. */
  if ((line = proceed (match_line, - (SCREENMAX >> 1))) == header) {
  /* Can't display in the middle. Make first line of file top_line */
	count = 0;
	for (line = header->next; line != match_line; line = line->next) {
		count ++;
	}
	line = header->next;
  } else {	/* New page is displayed. Set cursor to middle of page */
	count = SCREENMAX >> 1;
  }

/* Reset pointers and redraw the screen */
  reset (line, 0);
  if (redrawflag == True) {
	RD_y (count);
	redraw_pending = False;
  } else {
	redraw_pending = True;
  }

  return count;
}

int
find_y (match_line)
  LINE * match_line;
{
  return find_y_RD (match_line, True);
}

int
find_y_w_o_RD (match_line)
  LINE * match_line;
{
  return find_y_RD (match_line, False);
}


/*======================================================================*\
|*			Modify commands					*|
\*======================================================================*/

/*
 * DCC deletes the character under the cursor. If this character is a '\n' the
 * current line is joined with the next one.
 * If this character is the only character of the line, the current line will
 * be deleted.
 * DCC0 deletes without justification.
 */
static
void
delete_char (with_combinings)
  FLAG with_combinings;
{
  char * after_char;
  unsigned long unichar;
  int utflen;

  if (* cur_text == '\n') {
	if (cur_line->next == tail) {
		if (cur_line->return_type != lineend_NONE) {
			set_modified ();
			cur_line->return_type = lineend_NONE;
			set_cursor_xy ();
			put_line (y, cur_line, x, True, False);
			status_msg ("Trailing line-end deleted");
		}
	} else {
		(void) delete_text (cur_line, cur_text, cur_line->next, cur_line->next->text);
	}
  } else {
	after_char = cur_text;

	if (utf8_text == True && combining_mode == True) {
		utf8_info (after_char, & utflen, & unichar);
		advance_char (& after_char);
		if (with_combinings == True && ! iscombining (unichar)) {
		/* delete combining accents together with base character */
			utf8_info (after_char, & utflen, & unichar);
			while (iscombining (unichar)) {
				advance_char (& after_char);
				utf8_info (after_char, & utflen, & unichar);
			}
		}
	} else {
		advance_char (& after_char);
	}

	(void) delete_text (cur_line, cur_text, cur_line, after_char);
  }
}

void
DCC ()
{
  delete_char (True);
}

void
DCC0 ()
{
  DCC ();
}

/*
   DPC0 deletes the character on the left side of the cursor.
   If the cursor is at the beginning of the line, the line end 
   is deleted, merging the two lines.
   With hop flag, delete left part of line from current point.
 */
void
DPC0 ()
{
  char * delete_pos;

  if (x == 0 && cur_line->prev == header) {
	/* Top of file */
	return;
  }

  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if (hop_flag > 0) {
	hop_flag = 0;
	if (emulation == 'e') {	/* emacs mode */
		DPW ();
	} else if (cur_text != cur_line->text) {
	  delete_pos = cur_text;
	  BLINE ();
	  (void) delete_text (cur_line, cur_line->text, cur_line, delete_pos);
	}
  } else if (keyshift & shift_mask) {
	keyshift = '0';
	CUT ();
  } else {
	MLF ();			/* Move one left */
	DCC ();			/* Delete character under cursor */
  }
}

/*
   DPC normally deletes the character left just as DPC0 does.
   However, unless the hop flag is set, it first checks if there is 
   anything but white space on the current line left of the current position.
   If there is only white space, it tries to perform a "backtab" function, 
   reverting the indentation to the previous amount above in the text 
   (unless if it's in the line immediately above the current line).
 */
void
DPC ()
{
  char * cp;
  int column;
  int previous_col;
  LINE * lp;

  if (hop_flag > 0) {
	DPC0 ();
  } else {
	cp = cur_line->text;
	column = 0;
	while (* cp != '\0' && cp != cur_text && white_space (* cp)) {
		advance_char_scr (& cp, & column, cur_line->text);
	}
	if (cp == cur_text) {
		/* only white space left of current position */
		previous_col = column;
		lp = cur_line->prev;
		while (previous_col >= column && lp != header) {
			/* count white space on line lp */
			cp = lp->text;
			previous_col = 0;
			while (* cp != '\0' && previous_col < column && white_space (* cp)) {
				advance_char_scr (& cp, & previous_col, lp->text);
			}
			if (* cp == '\n' || * cp == '\0') {
				/* don't count space lines */
				previous_col = column;
			}

			lp = lp->prev;
		}

		/* if less indented previous line was found, 
		   and this was not on the line immediately 
		   preceeding the current line,
		   perform the back TAB function */
		if (previous_col < column && cur_line->prev != lp->next) {
			while (column > previous_col) {
				DPC0 ();
				column = 0;
				cp = cur_line->text;
				while (* cp != '\0' && cp != cur_text) {
					advance_char_scr (& cp, & column, cur_line->text);
				}
			}
			while (column < previous_col) {
				S (' ');
				column ++;
			}
		} else {
			DPC0 ();
		}
	} else {
		DPC0 ();
	}
  }
}

/*
 * DLINE delete the whole current line.
 */
void
DLINE ()
{
  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if (hop_flag > 0) {
    hop_flag = 0;
    if (* cur_text != '\n') {
	delete_text_buf (cur_line, cur_text, cur_line, cur_text + length_of (cur_text) - 1);
    }
  } else {
    BLINE ();
    if (* cur_text != '\n') {
	DLN ();
    }
    DCC ();
  }
}

/*
 * DLN deletes all characters until the end of the line. If the current
 * character is a '\n', then delete that char.
 */
void
DLN ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	DLINE ();
  } else if (* cur_text == '\n') {
/*	DCC ();	*/
	if (cur_line->next != tail) {
		delete_text_buf (cur_line, cur_text, cur_line->next, cur_line->next->text);
	}
  } else {
	delete_text_buf (cur_line, cur_text, cur_line, cur_text + length_of (cur_text) - 1);
  }
}

/*
 * DNW () deletes the next word (as defined in MNW ())
 */
void
DNW ()
{
  if (* cur_text == '\n') {
	DCC ();
  } else {
	move_next_word (DELETE);
  }
}

/*
 * DPW () deletes the previous word (as defined in MPW ())
 */
void
DPW ()
{
  if (cur_text == cur_line->text) {
	DPC0 ();
  } else {
	move_previous_word (DELETE);
  }
}

void
enterNUL ()
{
  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  S ('\n');				/* Insert a new line */
  MUP ();				/* Move one line up */
  move_to (LINE_END, y);		/* Move to end of this line */
  cur_line->return_type = lineend_NUL;
  put_line (y, cur_line, x, True, False);
  MRT ();				/* move behind inserted NUL */
}


/*
 * Functions to insert character at current location.
 * S0 inserts without justification.
 */

static unsigned long previous_unichar = 0;

/*
 * enter a character or a byte of a multi-byte character 
 * (with intermediate storage)
 */
static
void
S1 (newchar, JUSlvl, utf8_transform)
  register character newchar;
  int JUSlvl;
  FLAG utf8_transform;
{
  static character buffer [7];
  static character * utfpoi = buffer;
  static int utfcount = 1;
  static int cjkremaining = 0;
  static character firstbyte = '\0';
  static unsigned long unichar;

  int width = 1;

  if (newchar == '\0') {
	if (firstbyte != '\0') {
		firstbyte = '\0';
		ring_bell ();
	} else {
		enterNUL ();
	}
	return;
  }

  if (utf8_text == True) {
	if (utf8_transform == True) {
	/* UTF-8 input for UTF-8 text */
		if (newchar < 0x80) {
			unichar = newchar;
			* utfpoi = newchar;
			utfpoi ++;
			* utfpoi = '\0';
			utfpoi = buffer;
		} else if ((newchar & 0xC0) == 0x80) {
		/* UTF-8 sequence byte */
			unichar = (unichar << 6) | (newchar & 0x3F);
			* utfpoi = newchar;
			utfpoi ++;
			utfcount --;
			if (utfcount == 0) {
				* utfpoi = '\0';
				utfpoi = buffer;
				width = uniscrwidth (unichar, cur_text, cur_line->text);
			} else {
				return;
			}
		} else { /* first UTF-8 byte */
			utfpoi = buffer;
			* utfpoi = newchar;

			if ((newchar & 0xE0) == 0xC0) {
				utfcount = 2;
				unichar = newchar & 0x1F;
			} else if ((newchar & 0xF0) == 0xE0) {
				utfcount = 3;
				unichar = newchar & 0x0F;
			} else if ((newchar & 0xF8) == 0xF0) {
				utfcount = 4;
				unichar = newchar & 0x07;
			} else if ((newchar & 0xFC) == 0xF8) {
				utfcount = 5;
				unichar = newchar & 0x03;
			} else if ((newchar & 0xFE) == 0xFC) {
				utfcount = 6;
				unichar = newchar & 0x01;
			} else /* ignore illegal UTF-8 code */
				return;

			utfpoi ++;
			utfcount --;
			return;
		}
	} else {
	/* 8-bit input for UTF-8 text */
		unichar = newchar;
		if (newchar < 0x80) {
			buffer [0] = newchar;
			buffer [1] = '\0';
		} else {
			buffer [0] = (newchar >> 6) | 0xC0;
			buffer [1] = (newchar & 0x3F) | 0x80;
			buffer [2] = '\0';
		}
	}
  } else if (cjk_text == True) {
	/* 8/16-bit (CJK) input for CJK text */
	if (cjkremaining > 0) {
		* utfpoi ++ = newchar;
		* utfpoi = '\0';
		cjkremaining --;
		if (cjkremaining > 0) {
			return;
		}
	} else if (firstbyte != '\0') {
		buffer [0] = firstbyte;
		buffer [1] = newchar;
		buffer [2] = '\0';
		cjkremaining = CJK_len (buffer) - 2;
		if (cjkremaining > 0) {
			firstbyte = '\0';
			utfpoi = & buffer [2];
			return;
		}
	} else if (multichar (newchar)) {
		firstbyte = newchar;
		return;
	} else {
		buffer [0] = newchar;
		buffer [1] = '\0';
	}
	width = col_count (buffer);
	firstbyte = '\0';
  } else if (utf8_transform == True) {
	/* UTF-8 input for 8-bit text */
	buffer [1] = '\0';
	if (newchar < 0x80) {
		buffer [0] = newchar;
	} else if ((newchar & 0xC0) == 0x80) {
	/* UTF-8 sequence byte */
		unichar = (unichar << 6) | (newchar & 0x3F);
		utfcount --;
		if (utfcount == 0) {
			if ((unichar & 0xFF) == unichar) {
				buffer [0] = unichar & 0xFF;
			} else {
				buffer [0] = '';
			}
		} else {
			return;
		}
	} else { /* first UTF-8 byte */
		if ((newchar & 0xE0) == 0xC0) {
			utfcount = 2;
			unichar = newchar & 0x1F;
		} else if ((newchar & 0xF0) == 0xE0) {
			utfcount = 3;
			unichar = newchar & 0x0F;
		} else if ((newchar & 0xF8) == 0xF0) {
			utfcount = 4;
			unichar = newchar & 0x07;
		} else if ((newchar & 0xFC) == 0xF8) {
			utfcount = 5;
			unichar = newchar & 0x03;
		} else if ((newchar & 0xFE) == 0xFC) {
			utfcount = 6;
			unichar = newchar & 0x01;
		} else { /* ignore illegal UTF-8 code */
			return;
		}
		utfcount --;
		return;
	}
  } else {
	/* 8-bit input for 8-bit text */
	buffer [0] = newchar;
	buffer [1] = '\0';
  }

/* right-to-left support */
  if (poormansbidi == True && utf8_text == True && is_right_to_left (previous_unichar)) {
	if (newchar == '\n') {
		ELINE ();
	} else if (iscombining (unichar) && * cur_text != '\n') {
		MRT ();
	}
  }


/* Insert the new character */
  if (insert (cur_line, cur_text, buffer) == ERRORS) {
	return;
  }

/* Fix screen */
  if (newchar == '\n') {
	set_cursor (0, y);
	if (y == SCREENMAX) {		/* Can't use display () */
		line_print (y, cur_line);
		(void) forward_scroll (True);
		move_to (0, y);
	} else {
		reset (top_line, y);	/* Reset pointers */
		if (can_add_line == True) {
			clean_menus ();
			add_line (y + 1);
			scrollbar_scroll_down (y + 1);
			clear_status ();
			display (y, cur_line, 1, y + 1);
		} else {
			display (y, cur_line, last_y - y, y + 1);
		}
		move_to (0, y + 1);
	}
  } else if (x + width == XBREAK) { /* If line must be shifted, just call move_to */
	move_to (x + width, y);
  } else {			/* else display rest of line */
/*	also redraw previous char if it was incomplete ...
	just always redraw it!
	if (width != 0) {
		put_line (y, cur_line, x, False, False);
		move_to (x + width, y);
	} else
*/
	{
		/* combining char added to left char */
		char * newpoi;
		int newx = x + width;
		move_to (newx, y);
		newpoi = cur_text;
		move_to (x - 1, y);
		put_line (y, cur_line, x, False, False);
		move_address (newpoi, y);
	}
  }

/* right-to-left support */
  if (poormansbidi == True && utf8_text == True) {
	if (unichar == ' ' || unichar == '\t') {
		if (is_right_to_left (previous_unichar)) {
			move_to (x - 1, y);
		}
	} else if (unichar != '\n') {
		if (iscombining (unichar)) {
			if (is_right_to_left (previous_unichar)) {
				move_to (x - 1, y);
			}
		} else {
			if (is_right_to_left (unichar)) {
				move_to (x - 1, y);
			}
			previous_unichar = unichar;
		}
	}
  }

  if (JUSlvl > 0) {
	JUSandreturn ();
  }
}

/*
 * Insert newline with auto indentation.
 */
static
void
SNLindent ()
{
  char * coptext;

  S ('\n');
  coptext = cur_line->prev->text;

  while (* coptext == ' ' || * coptext == '\t') {
	S (* coptext);
	coptext ++;
  }
}

/*
 * Insert new line at current location.
 */
void
SNL ()
{
  if (keyshift & alt_mask) {
	keyshift = '0';
	Popmark ();
	return;
  }

  if (utf8_lineends == False && (keyshift & ctrl_mask)) {
	keyshift = '0';
	/* new line within same paragraph */
	if (JUSmode == 0) {
		if (* (cur_text -1) != ' ') {
			S (' ');
		}
	}
  }

  if (autoindent == False 
      || last_delta_readchar < 10 || average_delta_readchar < 10) {
	S ('\n');
  } else {
	SNLindent ();
  }
}

void
Spair (l, r)
  character l;
  character r;
{
  S1 (l, JUSlevel, utf8_input);
  SNLindent ();
  S1 (r, JUSlevel, utf8_input);

  MUP ();
  ELINE ();
}

static character lastchar = '\0';

static
void
Suni1 (newchar)
  register character newchar;
{
  S1 (newchar, JUSlevel, False);
}

static
void
Sutf8 (newchar)
  unsigned long newchar;
{
  if (newchar < 0x80) {
	S1 ((character) newchar, JUSlevel, True);
  } else if (newchar < 0x800) {
	S1 ((character) (0xC0 | (newchar >> 6)), JUSlevel, True);
	S1 ((character) (0x80 | (newchar & 0x3F)), JUSlevel, True);
  } else if (newchar < 0x10000) {
	S1 ((character) (0xE0 | (newchar >> 12)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 6) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | (newchar & 0x3F)), JUSlevel, True);
  } else if (newchar < 0x200000) {
	S1 ((character) (0xF0 | (newchar >> 18)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 12) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 6) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | (newchar & 0x3F)), JUSlevel, True);
  } else if (newchar < 0x4000000) {
	S1 ((character) (0xF8 | (newchar >> 24)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 18) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 12) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 6) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | (newchar & 0x3F)), JUSlevel, True);
  } else if (newchar < 0x80000000) {
	S1 ((character) (0xFC | (newchar >> 30)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 24) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 18) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 12) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | ((newchar >> 6) & 0x3F)), JUSlevel, True);
	S1 ((character) (0x80 | (newchar & 0x3F)), JUSlevel, True);
  } else {
	error ("Invalid Unicode value", NIL_PTR);
  }
}

void
S0 (newchar)
  register character newchar;
{
  S1 (newchar, 0, utf8_input);
}

void
S (newchar)
  register character newchar;
{
  if (hop_flag > 0) {
	lastchar = newchar;
	if (newchar == '\n') {
		S1 (newchar, JUSlevel, utf8_input);
		return;
	}
	hop_flag = 0;
	flags_changed = True;
	if (newchar == '(') {
		Spair ('(', ')');
		return;
	} else if (newchar == '[') {
		Spair ('[', ']');
		return;
	} else if (newchar == '{') {
		Spair ('{', '}');
		return;
	} else if (newchar == '<') {
		Spair ('<', '>');
		return;
	} else if (newchar == '/') {
		if (* cur_text != '\n') {
			SNLindent ();
			MUP ();
		}
		S1 ('/', JUSlevel, utf8_input);
		S1 ('*', JUSlevel, utf8_input);
		S1 ('*', JUSlevel, utf8_input);
		SNLindent ();
		S1 (' ', JUSlevel, utf8_input);
		SNLindent ();
		S1 ('*', JUSlevel, utf8_input);
		S1 ('/', JUSlevel, utf8_input);
		MUP ();
		S1 (' ', JUSlevel, utf8_input);
		S1 (' ', JUSlevel, utf8_input);
		return;
	}
  } else if (utf8_text == True && quote_type != 0 && newchar == '-' 
		&& lastchar == '-'
		&& cur_text != cur_line->text && * (cur_text - 1) == '-')
  {
	lastchar = ' ';
	DPC ();
	if (cur_text != cur_line->text && * (cur_text - 1) == ' ') {
		Sutf8 (0x2013);	/* – EN DASH */
		S1 (' ', JUSlevel, utf8_input);
	} else {
		Sutf8 (0x2014);	/* — EM DASH */
	}
	return;
  }

  lastchar = newchar;
  S1 (newchar, JUSlevel, utf8_input);
}

void
Scharacter (code)
  unsigned long code;
{
  character cjkbytes [5];
  character * cp;

  if (code < 0x80) {
	S (code);	/* go through HOP handling */
  } else if (utf8_text == True) {
	Sutf8 (code);
  } else if (cjk_text == True) {
	(void) cjkencode (code, cjkbytes);
	if (* cjkbytes != '\0') {
		cp = cjkbytes;
		while (* cp != '\0') {
			S1 (* cp ++, JUSlevel, False);
		}
	} else {
		ring_bell ();
		error ("Invalid CJK character code", NIL_PTR);
	}
  } else if (code < 0x100) {
	Suni1 (code & 0xFF);
  } else {
	ring_bell ();
  }
}


/* Smart Quotes */

typedef enum {leftdouble, rightdouble, leftsingle, rightsingle} quoteposition;

static
unsigned long
quote_mark_value (pos)
	quoteposition pos;
{
	unsigned long unichar;
	int utflen;
	char * q = quote_mark (quote_type, pos);
	utf8_info (q, & utflen, & unichar);
	return unichar;
}

int quote_type = 0;
static FLAG quote_open [2] = {False, False};

static
void
reset_quote_state ()
{
	quote_open [False] = False;
	quote_open [True] = False;
}

void
quote_type_up ()
{
  if (quote_type < count_quote_types () - 1) {
	quote_type ++;
  } else {
	quote_type = 0;
  }
  reset_quote_state ();
}

void
quote_type_down ()
{
  if (quote_type > 0) {
	quote_type --;
  } else {
	quote_type = count_quote_types () - 1;
  }
  reset_quote_state ();
}

void
set_quote_type (qt)
  int qt;
{
  if (qt >= 0 && qt < count_quote_types ()) {
	quote_type = qt;
  }
  reset_quote_state ();
}

void
set_quote_style (q)
  char * q;
{
  set_quote_type (lookup_quotes (q));
}

static
void
Squote (doublequote)
  FLAG doublequote;
{
  unsigned long prevchar;
  FLAG insert_left;

  if (utf8_text == True && smart_quotes == True) {
	prevchar = precedingchar (cur_text, cur_line->text);
	if (quote_open [doublequote] == True) {
		insert_left = False;
		quote_open [doublequote] = False;
	} else if (prevchar == 0x0A) {
		insert_left = True;
		quote_open [doublequote] = UNSURE;
	} else if (iswide (prevchar)) {
		if (quote_open [doublequote] == UNSURE
		 || quote_open [doublequote] == OPEN) {
			insert_left = False;
			quote_open [doublequote] = False;
		} else {
			insert_left = True;
			quote_open [doublequote] = True;
		}
	} else {
		insert_left = prevchar == (unsigned char) '('
			|| prevchar == (unsigned char) '['
			|| prevchar == (unsigned char) '{'
			|| prevchar == (unsigned char) ' '
			|| prevchar == (unsigned char) '	'
			|| prevchar == (unsigned char) '\n'
			|| prevchar == quote_mark_value (leftdouble)
			|| prevchar == quote_mark_value (leftsingle)
			;
		if (insert_left == True) {
			quote_open [doublequote] = OPEN;
		} else {
			quote_open [doublequote] = False;
		}
	}

	if (insert_left == True) {
		/* insert left quote */
		if (doublequote == True) {
			Sutf8 (quote_mark_value (leftdouble));
		} else {
			Sutf8 (quote_mark_value (leftsingle));
		}
	} else {
		/* insert right quote */
		if (doublequote == True) {
			Sutf8 (quote_mark_value (rightdouble));
		} else {
			Sutf8 (quote_mark_value (rightsingle));
		}
	}
  } else {
	if (doublequote == True) {
		S ('"');
	} else {
		S ('\'');
	}
  }
}

void
Sdoublequote ()
{
  Squote (True);
}

void
Ssinglequote ()
{
  if (hop_flag > 0 && utf8_text == True) {
	/* enter apostrophe */
	Sutf8 (0x2019);
  } else {
	Squote (False);
  }
}

/*
 * Replace current character with its hex/octal/decimal representation.
 */
static
character
hexdig (c)
  character c;
{
  if (c < 10) {
	return c + '0';
  } else {
	return c - 10 + 'A';
  }
}

static
void
insertcode (c, radix)
  character c;
  int radix;
{
  int radix2;

  if (radix == 8) {
	S (hexdig ((c >> 6) & 007));
	S (hexdig ((c >> 3) & 007));
	S (hexdig ((c) & 007));
  } else if (radix == 16) {
	S (hexdig ((c >> 4) & 017));
	S (hexdig ((c) & 017));
  } else {	/* assume radix = 10 or, at least, three digits suffice */
	radix2 = radix * radix;
	S (hexdig (c / radix2));
	S (hexdig ((c % radix2) / radix));
	S (hexdig (c % radix));
  }
}

static
void
insertvalue (v, radix)
  unsigned long v;
  int radix;
{
  char buffer [12];
  char * bufpoi = & buffer [11];
  * bufpoi = '\0';
  while (v > 0) {
	bufpoi --;
	* bufpoi = hexdig (v % radix);
	v = v / radix;
  }
  while (* bufpoi != '\0') {
	S (* bufpoi ++);
  }
}

static
void
changetocode (radix, wholevalue)
  int radix;
  FLAG wholevalue;
{
  character c = * cur_text;
  int utfcount = 1;
  unsigned long unichar;
#ifdef endlessloop
  int utflen;
#endif
  char buffer [7];
  char * textpoi;
  char * utfpoi;

  if (c == '\n') {
#ifdef msdos
	insertcode ('\r', radix);
#endif
	insertcode ('\n', radix);
  } else if (utf8_text == True) {
	if ((c & 0x80) == 0x00) {
		utfcount = 1;
		unichar = c;
	} else if ((c & 0xE0) == 0xC0) {
		utfcount = 2;
		unichar = c & 0x1F;
	} else if ((c & 0xF0) == 0xE0) {
		utfcount = 3;
		unichar = c & 0x0F;
	} else if ((c & 0xF8) == 0xF0) {
		utfcount = 4;
		unichar = c & 0x07;
	} else if ((c & 0xFC) == 0xF8) {
		utfcount = 5;
		unichar = c & 0x03;
	} else if ((c & 0xFE) == 0xFC) {
		utfcount = 6;
		unichar = c & 0x01;
	} else /* illegal UTF-8 code */ {
		if (wholevalue == False) {
		/*	DCC ();	*/
			insertcode (c, radix);
		}
		error ("Invalid UTF-8 sequence", NIL_PTR);
		return;
	}
	utfcount --;
	utfpoi = buffer;
	* utfpoi ++ = c;
	textpoi = cur_text;
	textpoi ++;
	while (utfcount > 0 && (* textpoi & 0xC0) == 0x80) {
		c = * textpoi ++;
		* utfpoi ++ = c;
		unichar = (unichar << 6) | (c & 0x3F);
		utfcount --;
	}
	* utfpoi = '\0';
/*	delete_char (False);	*/
	if (wholevalue == True) {
		if (radix == 16) {
			if (unichar > 0xFFFF) {
				insertcode (unichar >> 24, radix);
				insertcode (unichar >> 16, radix);
			}
			insertcode (unichar >> 8, radix);
			insertcode (unichar, radix);
		} else {
			insertvalue (unichar, radix);
		}
	} else {
		utfpoi = buffer;
		while (* utfpoi != '\0') {
			insertcode (* utfpoi ++, radix);
		}
	}

#ifdef endlessloop
	utf8_info (cur_text, & utflen, & unichar);
	if (iscombining (unichar)) {
		changetocode (radix, wholevalue);
	}
#endif

	if (utfcount > 0) {
		error ("Invalid UTF-8 sequence", NIL_PTR);
	}
  } else if (cjk_text == True && multichar (c)) {
	if (wholevalue == True) {
		insertvalue (charvalue (cur_text), radix);
	} else {
		(void) cjkencode (charvalue (cur_text), buffer);
	/*	DCC ();	*/
		textpoi = buffer;
		while (* textpoi != '\0') {
			insertcode (* textpoi ++, radix);
		}
	}
  } else {
/*	DCC ();	*/
	insertcode (c, radix);
  }
}

static
void
changefromcode (format)
  char * format;
{
  unsigned long code;

  if (sscanf (cur_text, format, & code) > 0) {
	Scharacter (code);
  } else {
	ring_bell ();
	hop_flag = 0;
	MRT ();
  }
}

void
changeuni ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	changefromcode ("%lx");
  } else {
	changetocode (16, True);
  }
}

void
changehex ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	if (utf8_text == False && cjk_text == False) {
		changefromcode ("%2lx");
	} else {
		changefromcode ("%lx");
	}
  } else {
	changetocode (16, False);
  }
}

void
changeoct ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	changefromcode ("%lo");
  } else {
	changetocode (8, True);
  }
}

void
changedec ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	changefromcode ("%lu");
  } else {
	changetocode (10, True);
  }
}


static
struct scriptentry {
	unsigned long first, last;
	char * scriptname;
} scripttable [] =
{
#include "scriptbl.h"
};

/*
   Determine script of Unicode character according to script range table.
 */
char *
script (ucs)
  unsigned long ucs;
{
  int min = 0;
  int max = sizeof (scripttable) / sizeof (struct scriptentry) - 1;
  int mid;

  /* binary search in table */
  while (max >= min) {
    mid = (min + max) / 2;
    if (scripttable [mid].last < ucs) {
      min = mid + 1;
    } else if (scripttable [mid].first > ucs) {
      max = mid - 1;
    } else if (scripttable [mid].first <= ucs && scripttable [mid].last >= ucs) {
      return scripttable [mid].scriptname;
    }
  }

  return "";
}


/*
 * display_code shows UTF-8 code sequence and Unicode value on the status line
 */
char hexbuf [20];
char * hexbufpoi = hexbuf;

static
void
appendcode (c)
  character c;
{
  * hexbufpoi ++ = hexdig ((c >> 4) & 0x0F);
  * hexbufpoi ++ = hexdig ((c) & 0x0F);
}

void
display_the_code ()
{
  character c = * cur_text;
  int utfcount = 1;
  unsigned long unichar;
  int utfcount2;
  unsigned long unichar2;
  char * textpoi;
  FLAG invalid = False;
  char * scriptmsg;
  char * scriptsep;
  char * widemsg;
  char * combinedmsg;

  int len;
  unsigned long cjkchar;
  int charlen;
  character cjkbytes [5];

  hexbufpoi = hexbuf;

  if (c == '\n') {
	switch (cur_line->return_type) {
	    case lineend_NONE:	textpoi = "No lineend (split line)";
				break;
	    case lineend_NUL:	appendcode ('\0');
				textpoi = "Nul character";
				break;
	    case lineend_LF:	appendcode ('\n');
				textpoi = "Line end";
				break;
	    case lineend_CRLF:	appendcode ('\r');
				appendcode ('\n');
				textpoi = "DOS line end";
				break;
	    case lineend_CR:	appendcode ('\r');
				textpoi = "Mac line end";
				break;
	    case lineend_LS:	/* Unicode line separator 2028:   */
				appendcode ('\342');
				appendcode ('\200');
				appendcode ('\250');
				textpoi = "Unicode line separator, code 2028";
				break;
	    case lineend_PS:	/* Unicode paragraph separator 2029:   */
				appendcode ('\342');
				appendcode ('\200');
				appendcode ('\251');
				textpoi = "Unicode paragraph separator, code 2029";
				break;
	    default:		appendcode ('\n');
				textpoi = "Unknown line end";
				break;
	}
	* hexbufpoi = '\0';
	build_string (text_buffer, "%s: %s", textpoi, hexbuf);
	status_msg (text_buffer);
	return;
  } else if (utf8_text == True) {
	if ((c & 0x80) == 0x00) {
		utfcount = 1;
		unichar = c;
	} else if ((c & 0xE0) == 0xC0) {
		utfcount = 2;
		unichar = c & 0x1F;
	} else if ((c & 0xF0) == 0xE0) {
		utfcount = 3;
		unichar = c & 0x0F;
	} else if ((c & 0xF8) == 0xF0) {
		utfcount = 4;
		unichar = c & 0x07;
	} else if ((c & 0xFC) == 0xF8) {
		utfcount = 5;
		unichar = c & 0x03;
	} else if ((c & 0xFE) == 0xFC) {
		utfcount = 6;
		unichar = c & 0x01;
	} else /* illegal UTF-8 code */ {
		appendcode (c);
		invalid = True;
	}
	textpoi = cur_text;
	if (invalid == False) {
		utfcount --;
		appendcode (c);
		textpoi ++;
		while (utfcount > 0 && (* textpoi & 0xC0) == 0x80) {
			c = * textpoi ++;
			appendcode (c);
			unichar = (unichar << 6) | (c & 0x3F);
			utfcount --;
		}
		if (utfcount > 0) {
			invalid = True;
		}
	} else {
		textpoi ++;
	}

	utf8_info (textpoi, & utfcount2, & unichar2);
	if (iscombining (unichar2)) {
		combinedmsg = " - combined ...";
	} else if (iscombined (unichar2, textpoi, cur_line->text)) {
		combinedmsg = " - joined ...";
	} else {
		combinedmsg = "";
	}

	* hexbufpoi = '\0';
	if (invalid == False) {
	    scriptmsg = script (unichar);
	    if (* scriptmsg == '\0') {
		scriptsep = scriptmsg;
	    } else {
		scriptsep = " ";
	    }

	    if (iswide (unichar)) {
		if (iscombining (unichar)) {
			widemsg = "wide combining ";
		} else {
			widemsg = "wide ";
		}
	    } else if (iscombining (unichar)) {
		widemsg = "combining ";
	    } else if (iscombined (unichar, cur_text, cur_line->text)) {
		widemsg = "joining ";
	    } else if ((unichar & 0x7FFFFC00) == 0xD800) {
		widemsg = "single high surrogate ";
	    } else if ((unichar & 0x7FFFFC00) == 0xDC00) {
		widemsg = "single low surrogate ";
	    } else if ((unichar & 0xFFFE) == 0xFFFE) {
		widemsg = "reserved ";
	    } else {
		widemsg = "";
	    }

	    if (unichar > 0xFFFF) {
		build_string (text_buffer, 
			"UTF-8 sequence: %s, %s%s%sUCS-4: %08lX%s", 
			hexbuf, scriptmsg, scriptsep, widemsg, unichar, combinedmsg);
	    } else {
		build_string (text_buffer, 
			"UTF-8 sequence: %s, %s%s%sUCS-2 (Unicode): %04lX%s", 
			hexbuf, scriptmsg, scriptsep, widemsg, unichar, combinedmsg);
	    }
	} else {
	    build_string (text_buffer, "Invalid UTF-8 sequence: %s%s", hexbuf, combinedmsg);
	}
	status_msg (text_buffer);
	return;
  } else if (cjk_text == True && multichar (c)) {
	len = CJK_len (cur_text);
	cjkchar = charvalue (cur_text);
	charlen = cjkencode (cjkchar, cjkbytes);

	textpoi = cur_text;
	while (len > 0 && * textpoi != '\0' && * textpoi != '\n') {
		appendcode (* textpoi);
		textpoi ++;
		len --;
		charlen --;
	}
	* hexbufpoi = '\0';
	if (len != 0 || charlen != 0) {
		build_string (text_buffer, "Incomplete CJK character code: %s", hexbuf);
	} else {
		build_string (text_buffer, "CJK character code: %s", hexbuf);
	}
	status_msg (text_buffer);
	return;
  } else {
	appendcode (c);
  }
  * hexbufpoi = '\0';
  build_string (text_buffer, "Character code: %s", hexbuf);
  status_msg (text_buffer);
}

void
display_code ()
{
  if (hop_flag > 0) {
	hop_flag = 0;
	if (always_disp_code == True) {
		always_disp_code = False;
	} else {
		always_disp_code = True;
	}
  } else {
	display_the_code ();
  }
}

/*
   Search for wrong encoded character.
   Searches for UTF-8 character in Latin-1 text or vice versa.
 */
void
search_wrong_enc ()
{
  LINE * prev_line;
  char * prev_text;
  LINE * lpoi;
  char * cpoi;
  int utfcount;
  unsigned long unichar;
  FLAG isutf;

  if (cjk_text == True || visciimode ()) {
	error ("Not supported in this encoding", NIL_PTR);
	return;
  }

  prev_line = cur_line;
  prev_text = cur_text;
  lpoi = cur_line;
  cpoi = cur_text;
  while (True) {
	/* advance character pointer */
	if (* cpoi == '\n') {
		lpoi = lpoi->next;
		if (lpoi == tail) {
			lpoi = header->next;
			status_msg ("Search wrapped around end of file");
		}
		cpoi = lpoi->text;
	} else {
		advance_char (& cpoi);
	}

	/* check if wrong encoded character */
	if ((* cpoi & 0x80) != 0) {
		if ((* cpoi & 0xC0) == 0xC0) {
			utf8_info (cpoi, & utfcount, & unichar);
			isutf = UTF_len (* cpoi) == utfcount;
		} else {
			isutf = False;
		}
		if (isutf != utf8_text) {
			break;
		}
	}

	/* check search wrap-around */
	if (lpoi == prev_line && cpoi == prev_text) {
		status_msg ("No more wrong encoding found");
		return;
	}
  }
  move_address (cpoi, find_y (lpoi));
}

/*
   Replace current diacritic transcription with real character.
   Prefer national diacritic according to parameter:
   g (German):	ae -> ä/, oe -> ö/, ue -> ü/
   d (Danish)	ae -> æ/, oe -> ø/
   f (French)	oe -> œ/oe ligature (UTF-8 only)
 */
void
UML (lang)
  char lang;
{
  unsigned long uc;
  unsigned long c1;
  unsigned long c2;
  char * cpoi = cur_text;

  int utfcount;
  unsigned long unichar;

  if (cjk_text == True) {
	return;
  }

  c1 = charvalue (cpoi);
  advance_char (& cpoi);
  c2 = charvalue (cpoi);

  /* language-specific conversion preferences */
  if (lang == 'd') {
	if (c1 == 'a' && c2 == 'e') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'A' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'o' && c2 == 'e') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'O' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Suni1 (''); return;
	}
  } else if (lang == 'f') {
	if (c1 == 'o' && c2 == 'e') {
		DCC (); DCC (); Sutf8 (0x0153); return;
	} else if (c1 == 'O' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Sutf8 (0x0152); return;
	}
  } else if (lang == 'g') {
	if (c1 == 'a' && c2 == 'e') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'o' && c2 == 'e') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'u' && c2 == 'e') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'A' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'O' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 'U' && (c2 & ~0x20) == 'E') {
		DCC (); DCC (); Suni1 (''); return;
	} else if (c1 == 's' && c2 == 's') {
		DCC (); DCC (); Suni1 (''); return;
	}
  }

  /* check mnemonic / diacritic conversion */
  uc = compose (c1, c2);

  if (uc != 0) {
	/* apply mnemonic conversion */
		DCC (); DCC (); Sutf8 (uc);
  } else if (utf8_text == True && ((character) * cur_text) >= 0x80) {

	/* Latin-1 -> UTF-8 conversion */
	c1 = (character) * cur_text;
	if (c1 >= 0xC0) {
		utf8_info (cur_text, & utfcount, & unichar);
		if (UTF_len (c1) == utfcount) {
			ring_bell ();
			error ("Already a UTF-8 character", NIL_PTR);
			return;
		}
	}
	if (delete_text (cur_line, cur_text, cur_line, cur_text + 1) == FINE) {
		Sutf8 (c1);
	}
  } else if (utf8_text == False && cjk_text == False && (c1 & 0xC0) == 0xC0) {

	/* UTF-8 -> Latin-1 conversion */
	utf8_info (cur_text, & utfcount, & unichar);
	if (UTF_len (c1) != utfcount) {
		ring_bell ();
		error ("Not a UTF-8 character", NIL_PTR);
		return;
	}
	if (visciimode ()) {
		ring_bell ();
		error ("Cannot map Unicode character", NIL_PTR);
		return;
	}
	if (unichar < 0x100) {
		if (delete_text (cur_line, cur_text, cur_line, cur_text + utfcount) == FINE) {
			Scharacter (unichar);
		}
	} else {
		ring_bell ();
		error ("Cannot encode Unicode character", NIL_PTR);
	}
  } else {
	ring_bell ();
	error ("Invalid character mnemonic", NIL_PTR);
  }
}

void
delete_basechar ()
{
  char * after_char = cur_text;
  int text_offset = cur_text - cur_line->text;

  advance_char (& after_char);
  (void) delete_text (cur_line, cur_text, cur_line, after_char);
  /* fieser Trick: */
  move_address (cur_line->text + text_offset, y);
}

struct {
	unsigned long base;
	short toupper, tolower;
} caseconv_table [] =
{
#include "casetabl.h"
};

#define caseconv_table_size	(sizeof (caseconv_table) / sizeof (* caseconv_table))

typedef struct {unsigned short u1, u2, u3;} uniseq;
#define U_cond_Final_Sigma		1
#define U_cond_After_Soft_Dotted	2
#define U_cond_More_Above		4
#define U_cond_Not_Before_Dot		8
#define U_cond_tr			16
#define U_cond_lt			32
#define U_cond_az			64
struct {
	unsigned long base;
	uniseq lower, title, upper;
	short condition;
} caseconv_special [] =
{
#include "casespec.h"
};

#define caseconv_special_size	(sizeof (caseconv_special) / sizeof (* caseconv_special))

int
lookup_caseconv (basechar)
  unsigned long basechar;
{
	int low = 0;
	int high = caseconv_table_size - 1;
	int i;

	while (low <= high) {
		i = (low + high) / 2;
		if (caseconv_table [i].base == basechar) {
			return i;
		} else if (caseconv_table [i].base >= basechar) {
			high = i - 1;
		} else {
			low = i + 1;
		}
	}
	/* notify "not found" */
	return -1;
}

int
lookup_caseconv_special (basechar)
  unsigned long basechar;
{
	int low = 0;
	int high = caseconv_special_size - 1;
	int i;

	while (low <= high) {
		i = (low + high) / 2;
		if (caseconv_special [i].base == basechar) {
			return i;
		} else if (caseconv_special [i].base >= basechar) {
			high = i - 1;
		} else {
			low = i + 1;
		}
	}
	/* notify "not found" */
	return -1;
}

/**
   Convert lower and upper case letters
   dir == 0: toggle
   dir == 1: convert to upper
   dir == -1: convert to lower
 */
static
void
lowcap (dir)
  int dir;
{
  /* for UTF-8 mode */
  int utfcount;
  unsigned long unichar;
  /* for single-byte mode */
  character c1;
  int prev_x;
  int tabix;
  unsigned long convchar;
  short condition;
  char * after_char;
  unsigned long unichar2;

  if (* cur_text == '\n') {
	MRT ();
  } else if (utf8_text == True) {
    do {
	tabix = 0;
	utf8_info (cur_text, & utfcount, & unichar);
	if (unichar == (character) '')
	{
		prev_x = x;
		delete_basechar ();
		Sutf8 ('S'); Sutf8 ('S');
		if (x <= prev_x + 1) {	/* may occur with combining chars */
			move_to (prev_x + 2, y);
		}
	}
#define dont_optimise_Latin1
#ifdef optimise_Latin1
	else if (dir >= 0 &&
		 ((unichar >= 'a' && unichar <= 'z' && unichar != 'i')
		  || (unichar >= (character) '' && unichar <= (character) '' && unichar != 247)
		 )
		)
	{
		prev_x = x;
		delete_basechar ();
		Sutf8 (unichar - 0x20);
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	}
	else if (dir <= 0 &&
		 ((unichar >= 'A' && unichar <= 'Z' && unichar != 'I')
		  || (unichar >= (character) '' && unichar <= (character) '' && unichar != 215)
		 )
		)
	{
		prev_x = x;
		delete_basechar ();
		Sutf8 (unichar + 0x20);
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	}
#endif
	else if (dir >= 0 && unichar >= 0x3041 && unichar <= 0x3096)
	{	/* Hiragana -> Katakana */
		prev_x = x;
		delete_basechar ();
		Sutf8 (unichar + 0x60);
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	}
	else if (dir <= 0 && unichar >= 0x30A1 && unichar <= 0x30F6)
	{	/* Katakana -> Hiragana */
		prev_x = x;
		delete_basechar ();
		Sutf8 (unichar - 0x60);
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	}
	else if ((tabix = lookup_caseconv_special (unichar)) >= 0)
	{
	    condition = caseconv_special [tabix].condition;
	    if (condition == U_cond_Final_Sigma) {
		after_char = cur_text;
		advance_char (& after_char);
		utf8_info (after_char, & utfcount, & unichar2);
		while (iscombining (unichar2)) {
			advance_char (& after_char);
			utf8_info (after_char, & utfcount, & unichar2);
		}
		if (unichar2 < (unsigned long) 'A'
		|| (unichar2 > (unsigned long) 'Z' && unichar2 < (unsigned long) 'a')
		|| (unichar2 > (unsigned long) 'z' && unichar2 < (unsigned long) (character) '')
		) {	/* final position detected */
			condition = 0;
		}
	    } else if (condition == U_cond_tr && Turkish == True) {
		condition = 0;
	    }

	    if (condition == 0) {
		prev_x = x;
		if (caseconv_special [tabix].base == caseconv_special [tabix].lower.u1)
		{
		    if (dir >= 0) {
			delete_basechar ();
			/* convert to upper (or title ...) */
			Sutf8 (caseconv_special [tabix].upper.u1);
			if (caseconv_special [tabix].upper.u2 != 0) {
				Sutf8 (caseconv_special [tabix].upper.u2);
				if (caseconv_special [tabix].upper.u3 != 0) {
					Sutf8 (caseconv_special [tabix].upper.u3);
				}
			}
		    } else {
			move_to (x + 1, y);
		    }
		} else {
		    if (dir <= 0) {
			delete_basechar ();
			/* convert to lower */
			Sutf8 (caseconv_special [tabix].lower.u1);
			if (caseconv_special [tabix].lower.u2 != 0) {
				Sutf8 (caseconv_special [tabix].lower.u2);
				if (caseconv_special [tabix].lower.u3 != 0) {
					Sutf8 (caseconv_special [tabix].lower.u3);
				}
			}
		    } else {
			move_to (x + 1, y);
		    }
		}
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	    } else {
		/* notifiy to try further */
		tabix = -1;
	    }
	}
	if (tabix == -1 && (tabix = lookup_caseconv (unichar)) >= 0)
	{
		convchar = unichar;
		if (caseconv_table [tabix].toupper != 0) {
		    if (dir >= 0) {
			convchar = unichar + caseconv_table [tabix].toupper;
		    }
		} else {
		    if (dir <= 0) {
			convchar = unichar + caseconv_table [tabix].tolower;
		    }
		}
		prev_x = x;
		delete_basechar ();
		Sutf8 (convchar);
		if (x == prev_x) {	/* may occur with combining chars */
			move_to (prev_x + 1, y);
		}
	}
	else if (tabix == -1)
	{
		move_to (x + 1, y);
	}
    } while (hop_flag > 0 && idfchar (cur_text));
  } else {
    do {
	c1 = * cur_text;
	if (dir >= 0 &&
	    ((c1 >= 'a' && c1 <= 'z')
	      || (cjk_text == False && c1 >= (character) '' && c1 <= (character) '' && c1 != (character) 247)
	    )
	   )
	{
		DCC ();
		Suni1 ((character) ((unsigned int) c1 - 0x20));
	}
	else if (dir <= 0 &&
		 ((c1 >= 'A' && c1 <= 'Z')
		  || (cjk_text == False && c1 >= (character) '' && c1 <= (character) '' && c1 != (character) 215)
		 )
		) {
		DCC ();
		Suni1 ((character) ((unsigned int) c1 + 0x20));
	}
	else if (dir >= 0 && cjk_text == False && c1 == (character) '') {
		DCC ();
		Suni1 ('S'); Suni1 ('S');
	}
	else move_to (x + 1, y);
    } while (hop_flag > 0 && idfchar (cur_text));
  }
}

/**
   Toggle lower and upper case letters
 */
void
LOWCAP ()
{
  int text_offset;

  if (keyshift & alt_mask) {
	if (keyshift & shift_mask) {
		text_offset = cur_text - cur_line->text;
		UML (' ');
		move_address (cur_line->text + text_offset, y);
	}
	search_wrong_enc ();
	return;
  } else if ((keyshift & ctrlshift_mask) == ctrlshift_mask) {
	hop_flag = 1;
	changeuni ();
	return;
  } else if (keyshift & shift_mask) {
	keyshift = '0';
	hop_flag = 1;
  } else if (keyshift & ctrl_mask) {
	UML (' ');
	return;
  }
  lowcap (0);
}

/**
   Convert to lower case letters
 */
void
LOWER ()
{
  lowcap (-1);
}

/**
   Convert to upper case letters
 */
void
UPPER ()
{
  lowcap (+1);
}

/**
   Convert single character to upper case letter
 */
void
CAPWORD ()
{
  hop_flag = 0;
  lowcap (+1);
  MLF ();
  MNW ();
}

/*
 * insert_character inserts composed character
 */
static
void
insert_character (composed)
  unsigned long composed;
{
  if (composed == 0) {
	ring_bell ();
	error ("Unknown character mnemonic", NIL_PTR);
  } else if (composed == quit_char) {
	ring_bell ();
	error ("Invalid character mnemonic", NIL_PTR);
  } else if (utf8_text == True) {
	Sutf8 (composed);
  } else if (cjk_text == True) {
	Scharacter (composed);
  } else if (composed < 0x100) {
	Suni1 (composed);
  } else {
	ring_bell ();
	error ("Invalid character code", NIL_PTR);
  }
}

/*
 * insert_accent combines and inserts accented character
 */
static
void
insert_accent (name, routine)
  char * name;
  charfunc routine;
{
  unsigned long composed;

  build_string (text_buffer, "Enter character to compose with %s...", name);
  status_msg (text_buffer);
  composed = (* routine) (readcharacter ());
  clear_status ();

  insert_character (composed);
}

/*
   insert next char with defined accent prefix (invoked by function key)
	standard function key assignment:
		F5	F6
		"	´
	shift	~	`
	ctrl	°	^
 */
void
insert_diaeresis ()
{
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	insert_accent ("ring", angstrom);
  } else if (keyshift & shift_mask) {
	keyshift = '0';
	insert_accent ("tilde", tilde);
  } else {
	insert_accent ("diaeresis (umlaut)", diaeresis);
  }
}

/*
   insert next char with defined accent prefix (invoked by function key)
	standard function key assignment:
		F5	F6
		"	´
	shift	~	`
	ctrl	°	^
 */
void
insert_acute ()
{
  if (keyshift & ctrl_mask) {
	keyshift = '0';
	insert_accent ("circumflex", circumflex);
  } else if (keyshift & shift_mask) {
	keyshift = '0';
	insert_accent ("grave", grave);
  } else {
	insert_accent ("acute (d'aigu)", acute);
  }
}

/*
   insert next char with defined accent prefix (invoked by function key)
 */
void
insert_tilde ()
{
	insert_accent ("tilde", tilde);
}

/*
   insert next char with defined accent prefix (invoked by function key)
 */
void
insert_grave ()
{
	insert_accent ("grave", grave);
}

/*
   insert next char with defined accent prefix (invoked by function key)
 */
void
insert_angstrom ()
{
	insert_accent ("ring", angstrom);
}

/*
   insert next char with defined accent prefix (invoked by function key)
 */
void
insert_circumflex ()
{
	insert_accent ("circumflex", circumflex);
}

unsigned long
max_cjk_value ()
{
  switch (cjk_encoding) {
	case 'G': return 0xFFFFFFFF;
	case 'C': return 0x8EFFFFFF;
	case 'J': return 0x8FFFFF;
	default: return 0xFFFF;
  }
}

/*
 * CTRLINS inserts a control-char or encoded or mnemonic character 
 */
void
CTRLINS ()
{
  unsigned long ctrl;
  unsigned long unichar;
  char mnemonic [maxLINE_LEN];

  status_msg ("Enter control char / # hex/octal/decimal / compose char / <blank> mnemonic ...");
  ctrl = readcharacter ();

  if (
#ifdef pc_charset
      ! (utf8_text == True && utf8_input == False) ? ctrl == DOS_ring :
#endif
      ctrl == UNI_ring
     )
  {
	insert_accent ("ring", angstrom); return;
  } else if (
#ifdef pc_charset
      ! (utf8_text == True && utf8_input == False) ? ctrl == DOS_acute :
#endif
      ctrl == UNI_acute
     )
  {
	insert_accent ("acute (d'aigu)", acute); return;
  } else switch (ctrl) {
	case '"':	{insert_accent ("diaeresis (umlaut)", diaeresis); return;}
	case '\'':	{insert_accent ("acute (d'aigu)", acute); return;}
	case '`':	{insert_accent ("grave", grave); return;}
	case '^':	{insert_accent ("circumflex", circumflex); return;}
	case '~':	{insert_accent ("tilde", tilde); return;}
  }

  if (ctrl == FUNcmd && keyproc != I) {
	clear_status ();
	if (keyproc == MLF) {
		ctrl_MLF ();
		return;
	} else if (keyproc == MRT) {
		ctrl_MRT ();
		return;
	} else if (keyproc == DCC) {
		delete_char (False);
		return;
	}
	ring_bell ();
	return;
  }

  if (utf8_text == True) {
    if (ctrl == '#') {
	if (get_hex_number (& unichar, 0x7FFFFFFF) != ERRORS) {
		Sutf8 (unichar);
	}
	return;
    } else if (ctrl == ' ') {
	if (get_string_nokeymap ("Enter character mnemonic: ", mnemonic, False, " ") 
		== ERRORS) {
		return;
	}
	unichar = compose_mnemo (mnemonic);
	clear_status ();
	insert_character (unichar);
	return;
    } else if (
	(ctrl > ' ' && ctrl <= 'Z') || (ctrl > '^' && ctrl != 0x7F)
	) {
	build_string (text_buffer, "Enter second composing character...", ctrl);
	status_msg (text_buffer);
	unichar = compose (ctrl, readcharacter ());
	clear_status ();
	insert_character (unichar);
	return;
    }
  } else if (ctrl == '#') {
	if (cjk_text == True) {
		if (get_hex_number (& unichar, max_cjk_value ()) != ERRORS) {
			Scharacter (unichar);
		}
	} else {
		if (get_hex_number (& unichar, 0xFF) != ERRORS) {
			Suni1 (unichar);
		}
	}
	return;
  } else if ((ctrl > ' ' && ctrl <= 'Z') || (ctrl > '^' && ctrl != 0x7F)) {
	build_string (text_buffer, "Enter second composing character...", ctrl);
	status_msg (text_buffer);
	unichar = compose (ctrl, readcharacter ());
	clear_status ();
	insert_character (unichar);
	return;
  }

  clear_status ();
  if (ctrl == '\177') {
	S ('\177');
	return;
  }
  ctrl = ctrl & '\237';
  S (ctrl);
}

/*
 * LIB insert a line at the current position and moves back to the end of
 * the previous line. It keeps the line end type of the current line.
 */
void
LIB ()
{
  lineend_type return_type;

  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if (hop_flag > 0) {
	return_type = lineend_NONE;
	hop_flag = 0;
  } else if (cur_line->return_type == lineend_PS) {
	return_type = lineend_LS;
  } else {
	return_type = cur_line->return_type;
  }

  S ('\n');				/* Insert the line */
  MUP ();				/* Move one line up */
  move_to (LINE_END, y);		/* Move to end of this line */
  if (cur_line->return_type != return_type) {
	cur_line->return_type = return_type;
	put_line (y, cur_line, x, True, False);
  }
}


/*======================================================================*\
|*			Yank commands					*|
\*======================================================================*/

/* default marker */
static LINE * mark_line = NIL_LINE;		/* For marking position. */
static char * mark_text = NIL_PTR;

/* explicit markers */
#define maxmarkers 10
static LINE * mark_n_line [maxmarkers] = {
	NIL_LINE, NIL_LINE, NIL_LINE, NIL_LINE, NIL_LINE, 
	NIL_LINE, NIL_LINE, NIL_LINE, NIL_LINE, NIL_LINE};
static char * mark_n_text [maxmarkers] = {
	NIL_PTR, NIL_PTR, NIL_PTR, NIL_PTR, NIL_PTR, 
	NIL_PTR, NIL_PTR, NIL_PTR, NIL_PTR, NIL_PTR};

/* implicit marker stack */
#define markstacklen 10
static struct {
	LINE * line;
	char * text;
	char * file;
	int lineno;
	int col;
} mark_stack [markstacklen] = {
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
	{NIL_LINE, NIL_PTR, NIL_PTR, -1, -1},
};
static int mark_stack_poi = 0;
static int mark_stack_top = 0;
static int mark_stack_begin = 0;
static int mark_stack_count = 0;

#define dont_debug_mark_stack

#ifdef debug_mark_stack
#define printf_mark_stack(s)	printf ("%s - mark_stack %d [%d..%d] @ %d\n", s, mark_stack_count, mark_stack_begin, mark_stack_top, mark_stack_poi)
#define printf_debug_mark_stack(s)	printf (s)
#else
#define printf_mark_stack(s)	
#define printf_debug_mark_stack(s)	
#endif


/**
   Insert the buffer at the current location.
   PASTE () moves the cursor behind the inserted text.
   PASTEstay () moves the cursor in front of the inserted text.
 */
static
void
paste_buffer (old_pos)
  FLAG old_pos;
{
  register int fd;		/* File descriptor for buffer */

  if (keyshift & alt_mask) {
	keyshift = '0';
	YANKRING ();
	return;
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	COPY ();
	return;
  }

  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if (hop_flag > 0) {
	if ((fd = open (yankie_file, O_RDONLY | O_BINARY, 0)) < 0) {
		error ("No inter window buffer present", NIL_PTR);
		return;
	}
  } else {
	if ((fd = yankfile (READ, False)) == ERRORS) {
		error ("Buffer is empty", NIL_PTR);
		return;
	}
	if (append_flag == True) {
		close_buffer ();
	}
  }
  /* Insert the buffer */
  file_insert (fd, old_pos);
}

void
PASTE ()
{
  paste_buffer (paste_stay_left);
}

void
PASTEstay ()
{
  paste_buffer (True);
}

void
YANKRING ()
{
  if (cur_line == pasted_end_line && cur_text == pasted_end_textp 
   && checkmark (pasted_start_line, pasted_start_textp) == SMALLER)
  {
	move_address (pasted_start_textp, find_y (pasted_start_line));
	if (delete_text (pasted_start_line, pasted_start_textp, pasted_end_line, pasted_end_textp)
		== ERRORS) {
		sleep (2) /* give time to read allocation error msg */;
	} else {
		/* for some mysterious reason, 
		   this is needed to fix the display: */
		clear_status ();

		revert_yank_buf ();
		PASTE ();
	}
  } else {
	error ("Cannot undo previous paste", NIL_PTR);
  }
}

/*
 * paste_HTML () inserts the HTML embedding buffer at the current location.
 */
void
paste_HTML ()
{
  int fd;		/* File descriptor for buffer */

  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if ((fd = open (html_file, O_RDONLY | O_BINARY, 0)) < 0) {
	error ("HTML paste buffer vanished", NIL_PTR);
	return;
  }
  file_insert (fd, True);
}

/*
 * INSFILE () prompts for a filename and inserts the file at the current
 * location in the file.
 */
void
INSFILE ()
{
  register int fd;		/* File descriptor of file */
  char name [maxLINE_LEN];	/* Buffer for file name */

  if (keyshift & shift_mask) {
	keyshift = '0';
	WB ();
	return;
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	YANKRING ();
	return;
  }


  if (restricted == True) {
	restrictederr ();
	return;
  }

  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

/* Get the file name */
  if (get_file ("Get and insert file:", name) != FINE) {
	return;
  }
  clear_status ();

  status_line ("Inserting ", name);
  if ((fd = open (name, O_RDONLY | O_BINARY, 0)) < 0) {
	error ("Cannot open file: " /*, name */, serror ());
  } else {	/* Insert the file */
	file_insert (fd, True);	/* leave cursor at begin of insertion */
  }
}

/*
 * File_insert () inserts the contents of an opened file (as given by
 * filedescriptor fd) at the current location.
 * After the insertion, if old_pos is True, the cursor remains at the
 * start of the inserted text, if old_pos is False, it is placed to
 * its end. If old_pos is False, this works erroneously if the last line
 * inserted contains a TAB character!!
 */
static
void
file_insert (fd, old_pos)
  int fd;
  FLAG old_pos;
{
  char line_buffer [MAX_CHARS];		/* Buffer for next line */
  register LINE * line = cur_line;
  register int line_count = total_lines;	/* Nr of lines inserted */
  LINE * page = cur_line;
  int ret;
  int len;
  lineend_type return_type;

  reset_get_line ();

/* Get the first piece of text (might be ended with a '\n') from fd */
  ret = get_line (fd, line_buffer, & len);
  if (ret == ERRORS) {
	/* empty file */
	return;
  }

/* Adjust line end type if present */
  if (ret == SPLIT_LINE) {
	return_type = lineend_NONE;
  } else if (ret == NUL_LINE) {
	return_type = lineend_NUL;
  } else if (ret != NO_LINE) {
	return_type = extract_lineend_type (line_buffer, len);
  } else {
	return_type = cur_line->return_type;
  }

/* Insert this text at the current location */
  len = cur_text - cur_line->text;
  if (insert (line, cur_text, line_buffer) == ERRORS) {
	pasted_end_line = NIL_LINE;
	return;
  }
  cur_line->return_type = return_type;

  pasted_end_line = line;
  pasted_end_textp = line->text + len + length_of (line_buffer);


/* Repeat getting lines (and inserting lines) until EOF is reached */
  while (line != NIL_LINE
	 && (ret = get_line (fd, line_buffer, & len)) != ERRORS && ret != NO_LINE)
  {
	if (ret == SPLIT_LINE) {
		return_type = lineend_NONE;
	} else if (ret == NUL_LINE) {
		return_type = lineend_NUL;
	} else {
		return_type = extract_lineend_type (line_buffer, len);
	}
	line = line_insert (line, line_buffer, len, return_type);
  }

/* Calculate nr of lines added */
  line_count = total_lines - line_count;

  if (line == NIL_LINE) {
	pasted_end_line = NIL_LINE;
	/* show memory allocation error msg */
	sleep (2);
  } else if (ret == NO_LINE) {	/* Last line read not ended by a '\n' */
	line = line->next;
	if (insert (line, line->text, line_buffer) == ERRORS) {
		pasted_end_line = NIL_LINE;
		/* give time to read error msg */
		sleep (2);
	} else {
		pasted_end_line = line;
		pasted_end_textp = line->text + length_of (line_buffer);
	}
  } else if (line_count > 0) {
	pasted_end_line = line->next;
	pasted_end_textp = line->next->text;
  }

  (void) close (fd);

/* If illegal lines were input, report */
  show_get_l_errors ();

/* Fix the screen */
  if (line_count == 0) {		/* Only one line changed */
	set_cursor (0, y);
	line_print (y, line);

	move_to (x, y);
	pasted_start_line = cur_line;
	pasted_start_textp = cur_text;
	if (old_pos == False) {
		move_to (x + col_count (line_buffer), y);
	}
  } else {				/* Several lines changed */
	reset (top_line, y);	/* Reset pointers */
	while (page != line && page != bot_line->next) {
		page = page->next;
	}
	if (page != bot_line->next || old_pos == True) {
		display (y, cur_line, SCREENMAX - y, y);
		/* screen display style parameter (last) may be inaccurate */
	}

	move_to (x, y);
	pasted_start_line = cur_line;
	pasted_start_textp = cur_text;
	if (old_pos == False) {
		if (ret == NO_LINE) {
			move_to (col_count (line_buffer), find_y (line));
		} else {
			move_to (0, find_y (line->next));
		}
	}
  }

/* If nr of added line >= REPORT, print the count */
  if (line_count >= REPORT) {
	status_line (num_out ((long) line_count), " lines added");
  }
}

/*
 * WB () writes the buffer (yank_file) into another file, which
 * is prompted for.
 */
void
WB ()
{
  register int new_fd;		/* Filedescriptor to copy file */
  int yank_fd;			/* Filedescriptor to buffer */
  register int cnt;		/* Count check for read/write */
  int ret = FINE;		/* Error check for write */
  char wfile_name [maxLINE_LEN];	/* Output file name */
  char * msg_doing; char * msg_done;

  if (restricted == True) {
	restrictederr ();
	return;
  }

/* Checkout the buffer */
  if ((yank_fd = yankfile (READ, False)) == ERRORS) {
	error ("Buffer is empty", NIL_PTR);
	return;
  }

/* Get file name */
  if (get_file ((hop_flag > 0) ? "Append buffer to file:"
			       : "Write buffer to file:", wfile_name) != FINE)
  {
	return;
  }

/* Create the new file or open previous file for appending */
  if (hop_flag > 0) {
    status_line ("Opening ", wfile_name);
    if ((new_fd = open (wfile_name, O_WRONLY | O_CREAT | O_APPEND | O_BINARY, fprot)) < 0) {
	error ("Cannot append to file: ", serror ());
	return;
    }
    msg_doing = "Appending "; msg_done = "Appended";
  } else {
    if (checkoverwrite (wfile_name) != True) {
	return;
    } else {
	status_line ("Opening ", wfile_name);
	if ((new_fd = open (wfile_name, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, fprot)) < 0) {
		error ("Cannot create file: ", serror ());
		return;
	}
    }
    msg_doing = "Writing "; msg_done = "Wrote";
  }

  status_line (msg_doing, wfile_name);

/* Copy buffer into file */
  while ((cnt = read (yank_fd, text_buffer, sizeof (text_buffer))) > 0) {
	if (write (new_fd, text_buffer, (unsigned int) cnt) != cnt) {
		bad_write (new_fd);
		ret = ERRORS;
		break;
	}
  }

/* Clean up open files and status_line */
  (void) close (yank_fd);
  if (close (new_fd) < 0) {
	error ("Write buffer to file failed: ", serror ());
	ret = ERRORS;
  }

  if (ret != ERRORS) {			/* Bad write */
	file_status (msg_done, bytes_saved, wfile_name, lines_saved, 
			False, True, False, False, chars_saved);
  }
}

/*
 * MARK sets mark_line / mark_text to the current line / current text pointer.
 */
void
MARK ()
{
  if (hop_flag > 0) {
	GOMA ();
  } else {
	mark_line = cur_line;
	mark_text = cur_text;
	status_msg ("Mark set");
  }
}

/*
 * GOMA moves to the marked position
 */
void
GOMA ()
{
  if (checkmark (mark_line, mark_text) == NOT_VALID) {
	error ("Mark not set", NIL_PTR);
  } else {
	Pushmark ();

	move_address (mark_text, find_y (mark_line));
  }
}

/*
 * MARKn sets mark n to the current line / current text pointer.
 * Markn sets it silently.
 */
void
MARKn (n)
  int n;
{
  if (hop_flag > 0) {
	GOMAn (n);
  } else {
	if (n < 0 || n >= maxmarkers) {
		error ("Marker # out of range", NIL_PTR);
		return;
	}
	mark_n_line [n] = cur_line;
	mark_n_text [n] = cur_text;
	status_msg ("Mark set");
  }
}

void
Markn (n)
  int n;
{
	if (n == -1) {	/* initial mark */
		mark_line = cur_line;
		mark_text = cur_text;
	} else if (n < 0 || n >= maxmarkers) {
		error ("Marker # out of range", NIL_PTR);
		return;
	} else {
		mark_n_line [n] = cur_line;
		mark_n_text [n] = cur_text;
	}
}

/*
 * GOMAn moves to the marked position n
 */
void
GOMAn (n)
  int n;
{
  Pushmark ();

  if (n < 0 || n >= maxmarkers) {
	error ("Marker # out of range", NIL_PTR);
	return;
  }

  if (checkmark (mark_n_line [n], mark_n_text [n]) == NOT_VALID) {
	error ("Mark not set", NIL_PTR);
  } else {
	move_address (mark_n_text [n], find_y (mark_n_line [n]));
  }
}


static
char *
copied_file_name ()
{
  int i;
  char * dup;
  char * filei;

  /* check if file name already in stack */
  for (i = 0; i < markstacklen; i ++) {
	filei = mark_stack [i].file;
	if (filei != NIL_PTR && streq (filei, file_name)) {
		return filei;
	}
  }

  /* make a new copy of file name string */
  dup = alloc (strlen (file_name) + 1);
  if (dup != NIL_PTR) {
	strcpy (dup, file_name);
  }
  return dup;
}

/*
   Pushmark pushes the current position to the mark stack.
 */
void
Pushmark ()
{
	int cur_col = get_cur_col ();

	mark_stack [mark_stack_top].line = cur_line;
	mark_stack [mark_stack_top].text = cur_text;
	mark_stack [mark_stack_top].file = copied_file_name ();
	mark_stack [mark_stack_top].lineno = line_number;
	mark_stack [mark_stack_top].col = cur_col;

	mark_stack_top = (mark_stack_top + 1) % markstacklen;
	if (mark_stack_top == mark_stack_begin) {
		mark_stack_begin = (mark_stack_begin + 1) % markstacklen;
	} else {
		mark_stack_count ++;
	}
	mark_stack_poi = mark_stack_top;
	printf_mark_stack ("Push");
}

/*
   Popmark pops the current position from the mark stack.
 */
void
Popmark ()
{
	FLAG switch_files;

	if (hop_flag > 0) {
		/* climb up stack towards top */
		printf_mark_stack ("HOP Pop");
		if (mark_stack_count == 0
		    || (mark_stack_poi % markstacklen) == mark_stack_top
		    || ((mark_stack_poi + 1) % markstacklen) == mark_stack_top
		   )
		{
			printf_debug_mark_stack ("HOP Pop no more\n");
			error ("No more stacked positions", NIL_PTR);
			return;
		}
		mark_stack_poi = (mark_stack_poi + 1) % markstacklen;
		printf_mark_stack ("...");
	} else {
		/* climb down stack towards bottom */
		if (mark_stack_poi == mark_stack_begin) {
			printf_debug_mark_stack ("Pop no more\n");
			error ("No more stacked positions", NIL_PTR);
			return;
		}
		if (mark_stack_poi == mark_stack_top) {
			/* at top, push current position first */
			printf_mark_stack ("Pop Push");
			Pushmark ();
			mark_stack_poi --;
		}
		mark_stack_poi --;
		if (mark_stack_poi < 0) {
			mark_stack_poi = markstacklen - 1;
		}
	}

	if (mark_stack [mark_stack_poi].file == NIL_PTR) {
		printf_debug_mark_stack ("not valid\n");
		error ("Stacked position not valid", NIL_PTR);
		return;
	}
	switch_files = ! streq (mark_stack [mark_stack_poi].file, file_name);
	if (switch_files ||
		checkmark (mark_stack [mark_stack_poi].line, 
			mark_stack [mark_stack_poi].text) == NOT_VALID)
	{
		int mark_lineno;
		int mark_col;
		LINE * open_line;
		int cur_column;
		char * cpoi;

		if (switch_files) {
			if (save_text_load_file (mark_stack [mark_stack_poi].file) == ERRORS) {
				return;
			}
		}

		mark_lineno = mark_stack [mark_stack_poi].lineno - 1;
		mark_col = mark_stack [mark_stack_poi].col;
		open_line = proceed (header->next, mark_lineno);
		if (open_line == tail) {
			EFILE ();
			error ("Stacked position not present anymore", NIL_PTR);
		} else {
			cur_column = 0;
			move_to (0, find_y (open_line));
			cpoi = cur_line->text;
			while (* cpoi != '\n' && cur_column < mark_col) {
				advance_char_scr (& cpoi, & cur_column, cur_line->text);
			}
			move_address (cpoi, y);
		}
	} else {
		move_address (mark_stack [mark_stack_poi].text, 
				find_y (mark_stack [mark_stack_poi].line));
	}
}


#ifndef linkyank
/*
 * copy one file to the other
 */
void
copyfile (yank_file, yankie_file)
  char * yank_file;
  char * yankie_file;
{
  int yank_fd;		/* Filedescriptor to yank buffer */
  int new_fd;		/* Filedescriptor to yankie file */
  register int cnt;	/* Count check for read/write */

/* Checkout the buffer */
  if ((yank_fd = open (yank_file, O_RDONLY | O_BINARY, 0)) < 0) {
	return;
  }

/* Create yankie file */
  if ((new_fd = open (yankie_file, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, fprot)) < 0) {
	return;
  }

/* Copy buffer into file */
  while ((cnt = read (yank_fd, text_buffer, sizeof (text_buffer))) > 0) {
	if (write (new_fd, text_buffer, (unsigned int) cnt) != cnt) {
		break;
	}
  }

/* Clean up open files and status_line */
  (void) close (yank_fd);
  (void) close (new_fd);
}
#endif

/*
 * Yankie () provides a reference to the last saved buffer to be read
 * by other mined invocations.
 */
static
void
yankie ()
{
#ifdef linkyank
  delete_file (yankie_file);
  (void) link (yank_file, yankie_file);
#else
  (void) copyfile (yank_file, yankie_file);
#endif
}

/*
 * yank_block is an interface to the actual yank.
 * It calls checkmark () to check if the marked position is still valid.
 * If it is, yank_text is called.
 */
static
void
yank_block (remove, append)
  FLAG remove;	/* == DELETE if text should be deleted */
  FLAG append;	/* == True if text should only be appended to yank buffer */
{
  switch (checkmark (mark_line, mark_text)) {
	case NOT_VALID :
		error ("Mark not set", NIL_PTR);
		return;
	case SMALLER :
		set_buffer_open (append);
		yank_text (yankfile (WRITE, append), & yank_status, 
				mark_line, mark_text, cur_line, cur_text, 
				remove, append);
		yankie ();
		break;
	case BIGGER :
		set_buffer_open (append);
		yank_text (yankfile (WRITE, append), & yank_status, 
				cur_line, cur_text, mark_line, mark_text, 
				remove, append);
		yankie ();
		break;
	case SAME :		/* Ignore stupid behaviour */
		status_msg ("Nothing to save");
		break;
	default :
		error ("Internal mark error", NIL_PTR);
		return;
  }
}

void
yank_HTML (remove)
  FLAG remove;	/* == DELETE if text should be deleted */
{
  switch (checkmark (mark_line, mark_text)) {
	case NOT_VALID :
		error ("Mark not set", NIL_PTR);
		return;
	case SMALLER :
		yank_text (htmlfile (WRITE, False), & html_status, 
				mark_line, mark_text, cur_line, cur_text, 
				remove, False);
		break;
	case BIGGER :
		yank_text (htmlfile (WRITE, False), & html_status, 
				cur_line, cur_text, mark_line, mark_text, 
				remove, False);
		break;
	case SAME :		/* Ignore stupid behaviour */
		status_msg ("Nothing to save");
		break;
	default :
		error ("Internal mark error", NIL_PTR);
		return;
  }
}

/*
 * COPY () puts the text between the marked position and the current
 * in the buffer.
 */
void
COPY ()
{
  if (append_flag == True) {
	yank_block (NO_DELETE, True);
  } else if (hop_flag > 0) {
	yank_block (NO_DELETE, True);
  } else {
	yank_block (NO_DELETE, False);
  }
}

/*
 * CUT () is essentially the same as COPY (), but the text is deleted.
 */
void
CUT ()
{
  if (viewonly == True) {
	viewonlyerr ();
	return;
  }

  if (append_flag == True) {
	yank_block (DELETE, True);
  } else if (hop_flag > 0) {
	yank_block (DELETE, True);
  } else {
	yank_block (DELETE, False);
  }
}

/*
 * Check_mark () checks if mark_line and mark_text are still valid pointers.
 * If they are it returns
 * SMALLER if the marked position is before the current,
 * BIGGER if it isn't or SAME if somebody didn't get the point.
 * NOT_VALID is returned when mark_line and/or mark_text are no longer valid.
 * Legal () checks if mark_text is valid on the mark_line.
 */
static
FLAG
checkmark (mark_line, mark_text)
  register LINE * mark_line;
  register char * mark_text;
{
  register LINE * line;
  FLAG cur_seen = False;

/* Special case: check is mark_line and cur_line are the same. */
  if (mark_line == cur_line) {
	if (mark_text == cur_text) {
		/* Even same place */
		return SAME;
	}
	if (legal (mark_line, mark_text) == ERRORS) {
		/* mark_text out of range */
		return NOT_VALID;
	}
	if (mark_text < cur_text) {
		return SMALLER;
	} else {
		return BIGGER;
	}
  }

/* Start looking for mark_line in the line structure */
  for (line = header->next; line != tail; line = line->next) {
	if (line == cur_line) {
		cur_seen = True;
	} else if (line == mark_line) {
		break;
	}
  }

/* If we found mark_line (line != tail) check for legality of mark_text */
  if (line == tail || legal (mark_line, mark_text) == ERRORS) {
	return NOT_VALID;
  }

/* cur_seen is True if cur_line is before mark_line */
  if (cur_seen == True) {
	return BIGGER;
  } else {
	return SMALLER;
  }
}

/*
 * Legal () checks if mark_text is still a valid pointer.
 */
static
int
legal (mark_line, mark_text)
  register LINE * mark_line;
  register char * mark_text;
{
  register char * textp = mark_line->text;

/* Locate mark_text on mark_line */
  while (textp != mark_text && * textp != '\0') {
	textp ++;
  }
  return (* textp == '\0') ? ERRORS : FINE;
}

/*
 * Yank puts all the text between start_position and end_position into
 * the buffer.
 * The caller must check that the arguments to yank_text () are valid (e.g. in
 * the right order).
 */
static
void
yank_text (fd, buf_status, 
	start_line, start_textp, end_line, end_textp, 
	remove, append)
  int fd;
  FLAG * buf_status;
  LINE * start_line;
  LINE * end_line;
  char * start_textp;
  char * end_textp;
  FLAG remove;	/* == DELETE if text should be deleted */
  FLAG append;	/* == True if text should only be appended to yank buffer */
{
  register LINE * line = start_line;
  register char * textp = start_textp;
  long chars_written = 0L;	/* chars written to buffer this time */
  long bytes_written = 0L;	/* bytes written to buffer this time */
  int lines_written = 0;	/* lines written to buffer this time */
  int return_len;


/* Create file to hold buffer */
  if (fd == ERRORS) {
	return;
  }

  if (append == True) {
	status_msg ("Appending text ...");
  } else {
	status_msg ("Saving text ...");
	chars_saved = 0L;
	bytes_saved = 0L;
	lines_saved = 0;
  }

/* Keep writing chars until the end_location is reached. */
  chars_written = char_count (textp) - 1;
  while (textp != end_textp) {
	if (* textp == '\n') {
		/* handle different line ends */
		return_len = write_lineend (fd, line->return_type);
		if (return_len == ERRORS) {
			(void) close (fd);
			return;
		}

		lines_written ++;
		if (line->return_type != lineend_NONE) {
			chars_written ++;
		}
		bytes_written += return_len;

		/* move to the next line */
		line = line->next;
		textp = line->text;

		chars_written += char_count (textp) - 1;
	} else {
		if (writechar (fd, * textp) == ERRORS) {
			(void) close (fd);
			return;
		}

		bytes_written ++;

		/* move to the next byte */
		textp ++;
	}
  }

  chars_written -= char_count (end_textp) - 1;

/* Flush the I/O buffer and close file */
  if (flush_buffer (fd) == ERRORS) {
	(void) close (fd);
	return;
  }
  if (close (fd) < 0) {
	error ("Write to buffer failed: ", serror ());
	return;
  }
  * buf_status = VALID;


  /*
   * Check if the text should be deleted as well. In case it should,
   * the following hack is used to save a lot of code.
   * First move back to the start_position (this might be the current
   * location) and then delete the text.
   * This might look a bit confusing to the user the first time.
   * Delete () will fix the screen.
   */
  if (remove == DELETE) {
	move_to (find_x (start_line, start_textp), find_y (start_line));
	if (delete_text (start_line, start_textp, end_line, end_textp)
		== ERRORS) {
		sleep (2) /* give time to read allocation error msg */;
	}
	mark_line = cur_line;
	mark_text = cur_text;
  }

  bytes_saved += bytes_written;
  chars_saved += chars_written;
  lines_saved += lines_written;

  build_string (text_buffer, "%s %d lines to paste buffer %s(chars/bytes: %ld/%ld) - Paste with %s/Insert", 
	(remove == DELETE) ? "Moved" : "Copied", lines_written, 
	(append == True) ? "(appended) " : "", 
	chars_written, bytes_written,
	emulation == 'e' ? "^Y" : emulation == 'w' ? "^K^C" : "^P"
	);
  status_msg (text_buffer);
}


/*
 * scratchfile/yankfile () tries to create a unique file in a temporary directory.
 * It tries several different filenames until one can be created
 * or MAXTRIALS attempts have been made.
 * After MAXTRIALS times, an error message is given and ERRORS is returned.
 */

#define MAXTRIALS 99

static
void
set_yank_file_name (buf_name, which, no)
  char * buf_name;
  char * which;
  int no;
{
#ifdef msdos
  build_string (buf_name, "%s.%s%d", yankie_file, which, no);
#else
  build_string (buf_name, "%s.%s%d-%d", yankie_file, which, getpid (), no);
#endif
}

/*
 * Delete yank file if there is one.
 */
void
delete_yank_files ()
{
/*  if (yank_status == VALID) {
	delete_file (yank_file);
  }
*/
  while (max_yank_buf_no > 0) {
	set_yank_file_name (yank_file, "", max_yank_buf_no);
	delete_file (yank_file);
	max_yank_buf_no --;
  }

  if (html_status == VALID) {
	delete_file (html_file);
  }
}

static
int
scratchfile (mode, append, buf_name, which, buf_status)
  FLAG mode;	/* Can be READ or WRITE permission */
  FLAG append;	/* == True if text should only be appended to yank buffer */
  char * buf_name;
  char * which;
  FLAG * buf_status;
{
  int fd = 0;			/* Filedescriptor to buffer */

  set_yank_file_name (buf_name, which, yank_buf_no);

/* If * buf_status == NOT_VALID, scratchfile is called for the first time */
  if (* buf_status == NOT_VALID && mode == WRITE) { /* Create new file */
	/* Generate file name. */
	/*set_yank_file_name (buf_name, which, yank_buf_no);*/
	/* Check file existence */
	if (access (buf_name, 0 /* F_OK */) == 0
	    || (
/*		fd = creat (buf_name, bufprot)		*/
		fd = open (buf_name, O_CREAT | O_TRUNC | O_WRONLY | O_BINARY, bufprot)
		) < 0)
	{
		if (++ yank_buf_no >= MAXTRIALS) {
		    build_string (text_buffer, "Unable to create scratchfile %s: ", buf_name);
		    if (fd == 0) {
			error (text_buffer, "File exists");
		    } else {
			error (text_buffer, serror ());
		    }
		    return ERRORS;
		} else {	/* try again */
		    return scratchfile (mode, append, buf_name, which, buf_status);
		}
	}
  }
  else if (* buf_status == NOT_VALID && mode == READ) {
	return ERRORS;
  }
  else /* * buf_status == VALID */
	if (  (mode == READ && (fd = open (buf_name, O_RDONLY | O_BINARY, 0)) < 0)
	   || (mode == WRITE &&
		(fd = open (buf_name, O_WRONLY | O_CREAT
				| ((append == True) ? O_APPEND : O_TRUNC)
				| O_BINARY
				, bufprot)) < 0))
  {
	* buf_status = NOT_VALID;
	return ERRORS;
  }

  clear_buffer ();
  return fd;
}

int
yankfile (mode, append)
  FLAG mode;	/* Can be READ or WRITE permission */
  FLAG append;	/* == True if text should only be appended to yank buffer */
{
  return scratchfile (mode, append, yank_file, "", & yank_status);
}

int
htmlfile (mode, append)
  FLAG mode;	/* Can be READ or WRITE permission */
  FLAG append;	/* == True if text should only be appended to yank buffer */
{
  return scratchfile (mode, append, html_file, "h", & html_status);
}


/*======================================================================*\
|*			Configurable functions				*|
\*======================================================================*/

/**
   Homekey () is invoked by a Home keypad key, function configurable
 */
void
HOMEkey ()
{
  if (keyshift & shift_mask) {
	keyshift = '0';
	MARK ();
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	BLINE ();
  } else {
	if (mined_keypad == True) {
		MARK ();
	} else {
		BLINE ();
	}
  }
}

void
smallHOMEkey ()
{
  if (keyshift & shift_mask) {
	keyshift = '0';
	MARK ();
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	BLINE ();
  } else {
	if (mined_keypad == True) {
		BLINE ();
	} else {
		MARK ();
	}
  }
}

/**
   ENDkey () is invoked by an End keypad key, function configurable
 */
void
ENDkey ()
{
  if (keyshift & shift_mask) {
	keyshift = '0';
	COPY ();
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	ELINE ();
  } else {
	if (mined_keypad == True) {
		COPY ();
	} else {
		ELINE ();
	}
  }
}

void
smallENDkey ()
{
  if (keyshift & shift_mask) {
	keyshift = '0';
	COPY ();
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	ELINE ();
  } else {
	if (mined_keypad == True) {
		ELINE ();
	} else {
		COPY ();
	}
  }
}

/**
   DELkey () is invoked by a Delete key, function configurable
 */
void
DELkey ()
{
  if (keyshift & shift_mask) {
	keyshift = '0';
	CUT ();
  } else if (keyshift & ctrl_mask) {
	keyshift = '0';
	DCC ();
  } else if (mined_del_is_cut == True) {
	CUT ();
  } else {
	DCC ();
  }
}


/*======================================================================*\
|*				End					*|
\*======================================================================*/
