/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <dirent.h>

#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#else
#ifdef HAVE_SYS_SHM_H
#include <sys/shm.h>
#endif
#ifdef HAVE_SYS_IPC_H
#include <sys/ipc.h>
#endif
#endif

#ifdef WIN32
#include <process.h>
#endif

#include <limits.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_cache.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_agent.h"
#include "udm_xmalloc.h"
#include "udm_stopwords.h"
#include "udm_proto.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_conf.h"
#include "udm_doc.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_vars.h"
#include "udm_log.h"
#include "udm_mkind.h"
#include "udm_utils.h"
#include "udm_store.h"
#include "udm_hash.h"
#include "udm_sqldbms.h"

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

/* This should be last include */
#ifdef DMALLOC
#include "dmalloc.h"
#endif

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

#define DEBUG_SEARCH 1

/* uncomment this to enable MODE_ALL realisation via search limits */
#define MODE_ALL_VIA_LIMITS
/*
static int open_host(char *hostname,int port, int timeout) {
	int net;
	struct hostent *host;
	struct sockaddr_in sa_in;

	bzero((void*)&sa_in, sizeof(sa_in));

	if (port){
		sa_in.sin_port= htons((u_short)port);
	}else{
		return(UDM_NET_ERROR);
	}

	if ((sa_in.sin_addr.s_addr=inet_addr(hostname)) != INADDR_NONE){
		sa_in.sin_family=AF_INET;
	}else{
		host=gethostbyname(hostname);
		if (host){
			sa_in.sin_family=host->h_addrtype;
			memcpy(&sa_in.sin_addr, host->h_addr, (size_t)host->h_length);
		}else{
			return(UDM_NET_CANT_RESOLVE);
		}
	}
	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect(net, (struct sockaddr *)&sa_in, sizeof (sa_in)))
		return(UDM_NET_CANT_CONNECT);

	return(net);
}
*/
typedef struct{
	urlid_t url_id;
	uint4   wrd_id;
	uint4   coord;
} T_URL_WRD_CRD;

typedef struct{
	uint4	wrd_id;
	off_t	pos;
	size_t	len;
} T_DAT_IND;

static int cmp_url_id(const UDM_URL_CRD *t1, const UDM_URL_CRD *t2){
	if(t1->url_id<t2->url_id) return(-1);
	if(t1->url_id>t2->url_id) return(1);
	{
		uint4 n1=UDM_WRDPOS(t1->coord);
		uint4 n2=UDM_WRDPOS(t2->coord);
		if(n1<n2)return(-1);
		if(n1>n2)return(1);
	}
	return(0);
}

static int cmp_ind(const T_DAT_IND *t1, const T_DAT_IND *t2){
	if(t1->wrd_id<t2->wrd_id) return(-1);
	if(t1->wrd_id>t2->wrd_id) return(1);
	return(0);
}

static int cmp_urlid_t(const urlid_t *u1, const urlid_t *u2){
	if(*u1<*u2) return(-1);
	if(*u1>*u2) return(1);
	return(0);
}
         
/***********************************/

#define LOGDIR	"raw"
#define TREEDIR	"tree"
#define SPLDIR  "splitter"

/******** Convert category string into 32 bit number *************/
void UdmDecodeHex8Str(const char *hex_str, uint4 *hi, uint4 *lo, uint4 *fhi, uint4 *flo){
  char str[33],str_hi[17],str_lo[17], *s = str;

        strncpy(str, hex_str, 13);
	str[12] = '\0';
	strcat(str,"000000000000");
	while(*s == '0') *s++ = ' ';
	strncpy(str_hi,&str[0],6); str_hi[6]=0;
	strncpy(str_lo,&str[6],6); str_lo[6]=0;
	
	*hi = (uint4)strtoul(str_hi, (char **)NULL, 36);
	*lo = (uint4)strtoul(str_lo, (char **)NULL, 36);

	if ((fhi != NULL) && (flo != NULL)) {
	  strncpy(str, hex_str, 13);
	  str[12] = '\0';
	  strcat(str,"ZZZZZZZZZZZZ");
	  strncpy(str_hi, &str[0], 6); str_hi[6] = 0;
	  strncpy(str_lo, &str[6], 6); str_lo[6] = 0;
	
	  *fhi = strtoul(str_hi, (char **)NULL, 36);
	  *flo = strtoul(str_lo, (char **)NULL, 36);

	}
}

/*************************** Sort functions **************************/
/* Function to sort LOGWORD list in (wrd_id,url_id,coord,time_stamp) order */
static int cmplog(const UDM_LOGWORD *s1,const UDM_LOGWORD *s2){
	if(s1->wrd_id<s2->wrd_id)return(-1);
	if(s1->wrd_id>s2->wrd_id)return(1);

	if(s1->url_id<s2->url_id)return(-1);
	if(s1->url_id>s2->url_id)return(1);

	if(s1->coord<s2->coord)return(-1);
	if(s1->coord>s2->coord)return(1);

	if(s2->stamp<s1->stamp)return(-1);
	if(s2->stamp>s1->stamp)return(1);

	return(0);
}

/**
   Function to sort LOGDEL list in URL_ID order 
*/
int UdmCmpurldellog(const void *s1,const void *s2) {
	unsigned int n1,n2;
	
	n1=((const UDM_LOGDEL*)s1)->url_id;
	n2=((const UDM_LOGDEL*)s2)->url_id;
	if(n1==n2){
		n1=((const UDM_LOGDEL*)s2)->stamp;
		n2=((const UDM_LOGDEL*)s1)->stamp;
	}
	if(n1<n2)return(-1);
	if(n1>n2)return(1);
	return(0);
}



static int UdmLogdInit(UDM_ENV *Env, UDM_DB *db, const char* var_dir, size_t i, int shared);

int UdmOpenCache(UDM_AGENT *A, int shared) {
	char		*host,*prt;
	int		port=UDM_LOGD_PORT;
	UDM_DB		*db;
	const char	*vardir=UdmVarListFindStr(&A->Conf->Vars,"VarDir",UDM_VAR_DIR);
	size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
	  if(db->DBMode != UDM_DBMODE_CACHE) continue;
	
	  if(db->cached_sd == 0) {
		if(UDM_OK != UdmLogdInit(A->Conf, db, vardir, i, shared)){
/*			strcpy(A->Conf->errstr,db->errstr);
			A->Conf->errcode=db->errcode;*/
			return UDM_ERROR;
		}
	  }
	}
	return(UDM_OK);
}

static int UdmLogdCloseLogs(UDM_AGENT *Agent);

int UdmCloseCache(UDM_AGENT *A, int shared) {
  UDM_DB *db;
  size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
  int res = UdmLogdCloseLogs(A);

  for (i = dbfrom; i < dbto; i++) {
    db = &A->Conf->dbl.db[i];
	if(db->DBMode!=UDM_DBMODE_CACHE) continue;
	if(db->logd_fd > 0) {
	  closesocket(db->logd_fd);
	  res =  UDM_OK;
	} else res = UdmLogdClose(A, db, shared);
	if (res != UDM_OK) break;
  }
  return res;
}

static int PresentInDelLog(UDM_LOGDEL *buf, int buf_count, size_t *start, const urlid_t url_id) {
	register int m,l,r;
	
	if (buf == NULL || buf_count == 0) return 0;
	if(start)l=*start;
	else l=0;
	r=buf_count;
	while(l<r){
		m=(l+r)/2;
		if(buf[m].url_id<url_id) l=m+1;
		else r=m;
	}
	if(start)*start=r;
	if(r==buf_count) return(0);
	else
		if(buf[r].url_id==url_id) return(buf[r].stamp);
		else return(0);
}

/***** Write a marker that the old content of url_id should be deleted ****/
int UdmDeleteURLFromCache(UDM_AGENT * Indexer, urlid_t url_id, UDM_DB *db){
	int sent;
	ssize_t recvt;
	UDM_LOGD_CMD cmd;
	char reply;

	cmd.stamp=time(NULL);
	cmd.url_id=url_id;
	cmd.cmd = UDM_LOGD_CMD_WORD;
	cmd.nwords=0;

	if(db->cached_sd){
		sent = UdmSend(db->cached_sd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr,"%s [%d] Can't write to cached: %s", __FILE__, __LINE__, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		}
		while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		  if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached [%d] %d, %s", __LINE__, recvt, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		  }
		  UDMSLEEP(1);
		}
		if (reply != 'O') {
			sprintf(db->errstr, "Can't incorrect reply from cached %d", __LINE__);
			db->errcode = 1;
		  return UDM_ERROR;
		}
	}else{
		if(UdmLogdStoreDoc(Indexer, cmd, NULL, db)) {
			return(UDM_ERROR);
		}
	}
	return(UDM_OK);
}

int UdmStoreWordsCache(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	size_t		sent;
	ssize_t         recvt;
	size_t		i;
	UDM_LOGD_CMD	cmd;
	UDM_LOGD_WRD	*wrd;
	char		reply;
	urlid_t		url_id = (urlid_t)UdmVarListFindInt(&Doc->Sections, "ID", 0);
	
	/* Mark that old content should be deleted    */
	/* Note that this should be done every time   */
	/* even if previous status=0, i.e. first time */
	/* indexing                                   */
	
	if ( (cmd.nwords = Doc->Words.nwords) == 0) {
	  return (UDM_OK);
	}

	cmd.stamp=time(NULL);
	cmd.url_id=url_id;
	cmd.cmd = UDM_LOGD_CMD_WORD;
	
	wrd=(UDM_LOGD_WRD*)malloc(cmd.nwords*sizeof(UDM_LOGD_WRD));
	for(i=0;i<Doc->Words.nwords;i++){
		wrd[i].coord=Doc->Words.Word[i].coord;
		wrd[i].wrd_id = UdmStrHash32(Doc->Words.Word[i].word);
	}
	if(db->cached_sd){
		sent = UdmSend(db->cached_sd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr, "%s [%d] Can't write to cached: %s", __FILE__, __LINE__, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		}
		while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		  if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached [%d] %d, %s", __LINE__, recvt, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		  }
		  UDMSLEEP(1);
		}
		if (reply != 'O') {
			sprintf(db->errstr, "Incorrect reply received from cached %d", __LINE__);
			db->errcode = 1;
		  return UDM_ERROR;
		}
		for (i = 0; i < cmd.nwords; i++) {
		  
		  sent = UdmSend(db->cached_sd, &wrd[i], sizeof(UDM_LOGD_WRD), 0);
		  if(sent != sizeof(UDM_LOGD_WRD)){
			sprintf(db->errstr, "[%d] Can't write to cached: %s", __LINE__, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		  }
/*		  while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		    if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached %d", __LINE__);
			db->errcode = 1;
			return(UDM_ERROR);
		    }
		    UDMSLEEP(1);
		  }
		  if (reply != 'O') {
			sprintf(db->errstr, "Incorrect reply received from cached %d", __LINE__);
			db->errcode = 1;
			return UDM_ERROR;
		  }*/
		}
		if (cmd.nwords) {
		  while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		    if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached %d", __LINE__);
			db->errcode = 1;
			return(UDM_ERROR);
		    }
		    UDMSLEEP(1);
		  }
		  if (reply != 'O') {
			sprintf(db->errstr, "Incorrect reply received from cached %d", __LINE__);
			db->errcode = 1;
			return UDM_ERROR;
		  }
		}
	}else{
		if(UdmLogdStoreDoc(Indexer, cmd, wrd, db)) {
			return(UDM_ERROR);
		}
	}
	UDM_FREE(wrd);
	return(UDM_OK);
}

/***************** split one log into cache *****************/

/* 
This function removes duplicate
(wrd_id,url_id) pairs using timestamps.
Most new pairs will remane.
*/

__C_LINK int __UDMCALL UdmRemoveDelLogDups(UDM_LOGDEL *words, int n){
	int i,j;
	
	j=0;
	for(i=1;i<n;i++){
		if(words[j].url_id!=words[i].url_id){
			j++;
			if(i!=j)words[j]=words[i];
		}
	}
	return(j+1);
}

/**< 
This function removes non-fresh  
words from UDM_LOGWORD array using timestamps
from delete log. Only those (word_id,url_id)
pairs which are newer than (url_id) from 
delete log will remane in "words" array 
*/

static size_t RemoveOldWords(UDM_LOGWORD *words, size_t n,UDM_LOGDEL *del, int del_count){
	size_t i,j=0;
	int u;


  if (del_count == 0) return n;

  while ((j < n) && (PresentInDelLog(del, del_count, NULL, words[j].url_id) <= words[j].stamp)) {
    i = j;
    while((j < n) && (words[j].url_id == words[i].url_id) && (words[j].stamp >= words[i].stamp)) j++;
  }

  for (i = j + 1; (i < n) && (words[i].url_id == words[j].url_id); i++);
  
  for (; i < n; i++) {
    
    u = (PresentInDelLog(del, del_count, NULL, words[i].url_id) <= words[i].stamp);
    if (u) words[j++] = words[i];
    while ((i + 1 < n) && (words[i].url_id == words[i+1].url_id) && (words[i + 1].stamp >= words[i].stamp)) {
      i++;
      if (u) words[j++] = words[i];
    }

  }
  return j;


/* Old realisation	
	for(i=1;i<n;i++){
		if((words[j].wrd_id!=words[i].wrd_id)
		   ||(words[j].url_id!=words[i].url_id)
		   ||(words[j].coord!=words[i].coord)
		){
			if(PresentInDelLog(del,del_count,NULL,words[i].url_id)<=words[i].stamp){
				j++;
				if(i!=j){
					words[j]=words[i];
				}
			}
		}
	}*/
	/* Check first word */
/*	if(j==0){
		if(PresentInDelLog(del,del_count,NULL,words[0].url_id)>words[0].stamp)
			return(0);
	}
	return (j < n) ? (j + 1) : n;
*/
}


/**< 
This function removes non-fresh records form UDM_URL_CRD array using timestamps from delete log.
*/

static size_t RemoveOldURLCRD(UDM_URL_CRD *Coords, size_t n, UDM_LOGDEL *del, int del_count) {
  size_t i, j = 0;
  int u;

  if (del_count == 0) return n;

  while ((j < n) && !PresentInDelLog(del, del_count, NULL, Coords[j].url_id )) {
    i = j;
    while((j < n) && Coords[j].url_id == Coords[i].url_id) j++;
  }

  for (i = j + 1; (i < n) && (Coords[i].url_id == Coords[j].url_id); i++);
  
  for (; i < n; i++) {
    
    u = (!PresentInDelLog(del, del_count, NULL, Coords[i].url_id));
    if (u) Coords[j++] = Coords[i];
    while (i + 1 < n && Coords[i].url_id == Coords[i+1].url_id) {
      i++;
      if (u) Coords[j++] = Coords[i];
    }

  }
  return j;
}


int UdmClearCacheTree(UDM_ENV * Conf){
	int		i;
	char		fname[1024];
	const char	*vardir=UdmVarListFindStr(&Conf->Vars,"VarDir",UDM_VAR_DIR);
	
	for(i = 0; i < UDM_MAX_LOG; i++){
		udm_snprintf(fname,sizeof(fname),"%s%s%s%c%03X.dat",vardir,UDMSLASHSTR,TREEDIR,UDMSLASH,i);
		unlink(fname);
		udm_snprintf(fname,sizeof(fname),"%s%s%s%c%03X.ind",vardir,UDMSLASHSTR,TREEDIR,UDMSLASH,i);
		unlink(fname);
	}
	return(0);
}

typedef struct{
	uint4 wrd_id;
	uint4 pos;
	uint4 len;
	int   done;
} T_WRD_POS_LEN;

static int PresentInNew(T_WRD_POS_LEN *keys, size_t nkeys, uint4 wrd_id){
	size_t l,r,m;
	
	l=0;
	r=nkeys;
	while(l<r){
		m=(l+r)/2;
		if(keys[m].wrd_id<wrd_id) l=m+1;
		else r=m;
	}
	if(r==nkeys) return(-1);
	else
		if(keys[r].wrd_id==wrd_id) return(r);
		else return(-1);
}

__C_LINK int __UDMCALL UdmSplitCacheLog(UDM_ENV * Env,int log,UDM_LOGDEL *del_buf,int del_count){
	size_t bytes;
	char log_name[1024];
	char old_dat_name[1024];
	char new_dat_name[1024];
	char old_ind_name[1024];
	char new_ind_name[1024];
	struct stat sb;
	UDM_LOGWORD *log_buf = NULL;
	const char	*vardir=UdmVarListFindStr(&Env->Vars,"VarDir",UDM_VAR_DIR);

	int log_fd=0;
	int old_dat_fd=0, new_dat_fd=0;
	int old_ind_fd=0, new_ind_fd=0;
	
	size_t	old_ind_count=0;
	T_DAT_IND *old_ind=NULL;
	
	size_t new_ind_count=0;
	size_t new_ind_mcount=500;
	T_DAT_IND *new_ind=(T_DAT_IND*)malloc(new_ind_mcount*sizeof(T_DAT_IND));

printf("Log %03X ",log); fflush(stdout);

	udm_snprintf(old_dat_name, sizeof(old_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log, ".dat");
	old_dat_fd=open(old_dat_name,O_RDONLY|UDM_BINARY);
	udm_snprintf(old_ind_name, sizeof(old_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log, ".ind");
	old_ind_fd=open(old_ind_name,O_RDONLY|UDM_BINARY);
	
	if((old_dat_fd>0)&&(old_ind_fd>0)){
		fstat(old_ind_fd, &sb);
#if 0
		ind=(T_DAT_IND*)mmap(0,sb.st_size,PROT_READ,MAP_SHARED,ind_fd,0);
#endif
		old_ind = (T_DAT_IND*)malloc((size_t)sb.st_size + 1);
		if (sb.st_size > 0) read(old_ind_fd, old_ind, (size_t)sb.st_size);
		close(old_ind_fd);
		old_ind_count=sb.st_size/sizeof(T_DAT_IND);
	}

	/* Open log file */
	udm_snprintf(log_name, sizeof(log_name), "%s%c%s%c%03X.log", vardir, UDMSLASH, SPLDIR, UDMSLASH, log);
	if((log_fd = open(log_name, O_RDWR|UDM_BINARY)) < 0){
	  if (errno == ENOENT) {
printf("No new... "); fflush(stdout);
	  } else {
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': (%d) %s", log_name, errno, strerror(errno));
/*		goto err1;*/
	  }
	}
	udm_snprintf(new_dat_name, sizeof(new_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR,UDMSLASH, log, ".dat.new");
	if((new_dat_fd=open(new_dat_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': %s", new_dat_name, strerror(errno));
		goto err1;
	}
	udm_snprintf(new_ind_name, sizeof(new_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log, ".ind.new");
	if((new_ind_fd=open(new_ind_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': %s\n",new_ind_name,strerror(errno));
		goto err1;
	}

	{
		size_t mkeys=500;
		size_t nkeys=0;
		T_WRD_POS_LEN *keys=(T_WRD_POS_LEN*)malloc(mkeys*sizeof(T_WRD_POS_LEN));
		size_t i,prev=0;
		size_t n = 0;
		size_t k;
		size_t pos=0;
		

		/* Allocate buffer for log */
		if (log_fd > 0) { 
printf("reading... "); fflush(stdout);
                  UdmWriteLock(log_fd); 
		  fstat(log_fd, &sb);
		  log_buf = (sb.st_size > 0) ? (UDM_LOGWORD*)malloc((size_t)sb.st_size) : NULL;
		  if (log_buf != NULL) {
		    bytes=read(log_fd,log_buf,(size_t)sb.st_size);
		    ftruncate(log_fd, (off_t)0);

		    n=bytes/sizeof(UDM_LOGWORD);

printf("sorting... "); fflush(stdout);
                    UdmSort(log_buf,n,sizeof(UDM_LOGWORD),(qsort_cmp)cmplog);
		    n=RemoveOldWords(log_buf,n,del_buf,del_count);
		  }
		  UdmUnLock(log_fd);
		  close(log_fd);
		}

printf("writing... "); fflush(stdout);

		for(i=1;i<n+1;i++){
			/* Next word? */
			if( (i==n) || (log_buf[prev].wrd_id != log_buf[i].wrd_id)){
				if(mkeys==nkeys){
					mkeys+=500;
					keys=(T_WRD_POS_LEN*)realloc(keys,mkeys*sizeof(T_WRD_POS_LEN));
				}
				keys[nkeys].wrd_id=log_buf[prev].wrd_id;
				keys[nkeys].pos=prev;
				keys[nkeys].len=i-prev;
				keys[nkeys].done=0;
				nkeys++;
			
				prev=i;
			}
		}
		for(k=0;k<old_ind_count;k++){
			uint4		wrd_id=old_ind[k].wrd_id;
			UDM_URL_CRD	*new_data=NULL;
			UDM_URL_CRD	*old_data=NULL;
			size_t		old_count;
			size_t		old_added=0; 
			size_t		new_count=0;
			size_t		new_len;
			int		wrd_ind;
			size_t 		o;
			size_t 		start = 0;

			old_data=(UDM_URL_CRD*)malloc(old_ind[k].len);
			if(old_ind[k].pos!=lseek(old_dat_fd,old_ind[k].pos,SEEK_SET)){
				UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't seek '%s': %s\n", old_dat_name, strerror(errno));
				goto err1;
			}
			read(old_dat_fd,old_data,old_ind[k].len);
			new_data=(UDM_URL_CRD*)malloc(old_ind[k].len);

			old_count=old_ind[k].len/sizeof(UDM_URL_CRD);

			for(o=0;o<old_count;o++){
				if(!PresentInDelLog(del_buf,del_count,&start,old_data[o].url_id)){
					new_data[old_added]=old_data[o];
					old_added++;
				}
			}
#if 0
printf("del_count=%d\told_count=%d\told_added=%d\n",del_count,old_count,old_added);
#endif
			if( (wrd_ind=PresentInNew(keys, nkeys, wrd_id))>=0 ){
				new_count=old_added+keys[wrd_ind].len;
				new_len=new_count*sizeof(UDM_URL_CRD);
				new_data=(UDM_URL_CRD*)realloc(new_data,new_len);
#if 0
//printf("wrd_ind=%d old_added=%d keys[wrd_ind].pos=%d keys[wrd_ind].len=%d\n",wrd_ind,old_added,keys[wrd_ind].pos,keys[wrd_ind].len);
#endif
				if(new_data){
					size_t j;
					for(j=old_added;j<new_count;j++){
						new_data[j].url_id=log_buf[keys[wrd_ind].pos+j-old_added].url_id;
						new_data[j].coord=log_buf[keys[wrd_ind].pos+j-old_added].coord;
					}
				}else{
					goto err1;
				}
				keys[wrd_ind].done=1;
			}else{
				new_count=old_added;
			}
			if(new_count){
				UdmSort(new_data,new_count,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				new_len=new_count*sizeof(UDM_URL_CRD);
				write(new_dat_fd,new_data,new_len);

				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=new_len;
				pos+=new_len;
				new_ind_count++;
			}
			UDM_FREE(old_data);
			UDM_FREE(new_data);
		}
		if(old_ind)UDM_FREE(old_ind);
		if(old_dat_fd)close(old_dat_fd);
		for(i=0;i<nkeys;i++){
			if(!keys[i].done){
				UDM_URL_CRD       *new_data=NULL;
				size_t		j;
				
				new_data=(UDM_URL_CRD*)malloc(keys[i].len*sizeof(*new_data));
				for(j=0;j<keys[i].len;j++){
					new_data[j].url_id=log_buf[keys[i].pos+j].url_id;
					new_data[j].coord=log_buf[keys[i].pos+j].coord;
				}
				UdmSort(new_data,keys[i].len,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				write(new_dat_fd,new_data,keys[i].len*sizeof(UDM_URL_CRD));
				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=keys[i].wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=keys[i].len*sizeof(UDM_URL_CRD);
				pos+=keys[i].len*sizeof(UDM_URL_CRD);
				new_ind_count++;

				UDM_FREE(new_data);
			}
		}
		UDM_FREE(keys);
	}
	UDM_FREE(log_buf);
	
	close(new_dat_fd);

printf("creating index (%d)... ", new_ind_count); fflush(stdout);

        if (new_ind != NULL && new_ind_count > 0) {
	  UdmSort(new_ind,new_ind_count,sizeof(T_DAT_IND),(qsort_cmp)cmp_ind);	
	  write(new_ind_fd,new_ind,new_ind_count*sizeof(T_DAT_IND));
	}
	UDM_FREE(new_ind);
	close(new_ind_fd);

#ifdef WIN32
	remove(old_dat_name);
	remove(old_ind_name);
#endif
	
	rename(new_dat_name,old_dat_name);
	rename(new_ind_name,old_ind_name);

printf("Done\n"); fflush(stdout);
	
	return(0);

err1:
	if(log_fd)close(log_fd);
	if(old_ind_fd)close(old_ind_fd);
	if(new_ind_fd)close(new_ind_fd);
	if(old_dat_fd)close(old_dat_fd);
	if(new_dat_fd)close(new_dat_fd);

	return(-1);
}

/****************** Search stuff ************************************/
/*
static int cmp_hex8_ind(const UDM_UINT8_POS_LEN *c1, const UDM_UINT8_POS_LEN *c2){
	uint4 n1=c1->hi;
	uint4 n2=c2->hi;

	if(n1==n2){
		n1=c1->lo; 
		n2=c2->lo;
	}
	if(n1<n2) return(-1);
	if(n1>n2) return(1);
	return(0);
}
*/

static int cmp_hex4_ind(const UDM_UINT4_POS_LEN *c1, const UDM_UINT4_POS_LEN *c2){
	if(c1->val<c2->val) return(-1);
	if(c1->val>c2->val) return(1);
	return(0);
}

int __UDMCALL UdmAddSearchLimit(UDM_AGENT *Agent, int type, const char *file_name, const char *val){
	uint4 hi, lo, f_hi, f_lo;
	
	if(Agent->nlimits == MAX_SEARCH_LIMIT - 1) return(1);
	
	Agent->limits[Agent->nlimits].type = type;
	strcpy(Agent->limits[Agent->nlimits].file_name, file_name);
	switch(type){
		case 0: UdmDecodeHex8Str(val, &hi, &lo, &f_hi, &f_lo); break;
		case 1: f_hi = hi = 0; f_lo = lo = 0; break;
		case 2: hi=atoi(val); lo=0; f_hi = hi; f_lo = lo; break;
		case 3: hi=UdmStrHash32(val); lo = 0; f_hi = hi; f_lo = 0; break;
	}	
	Agent->limits[Agent->nlimits].hi = hi;
	Agent->limits[Agent->nlimits].lo = lo;
	Agent->limits[Agent->nlimits].f_hi = f_hi;
	Agent->limits[Agent->nlimits].f_lo = f_lo;
	
	Agent->nlimits++;

	UdmLog(Agent, UDM_LOG_DEBUG, "val: %s  %x %x   %x %x", val, hi, lo, f_hi, f_lo);
	
	return(0);
}

static urlid_t* LoadNestedLimit(UDM_AGENT *Agent, size_t lnum, size_t *size) {
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT8_POS_LEN *ind=NULL;
	struct	stat sb;
	size_t	num;
	urlid_t	*data;
	size_t	start = -1, stop = -1, len;
	uint4   hi = Agent->limits[lnum].hi, lo = Agent->limits[lnum].lo, 
	  f_hi = Agent->limits[lnum].f_hi, f_lo = Agent->limits[lnum].f_lo;
	const char *name = Agent->limits[lnum].file_name;
	const char	*vardir=UdmVarListFindStr(&Agent->Conf->Vars,"VarDir",UDM_VAR_DIR);
	
	UdmLog(Agent, UDM_LOG_DEBUG, "%08x %08x - %08x %08x", hi, lo, f_hi, f_lo);
	if(hi==0&&lo==0)return(NULL);

	udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.ind", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT8_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);
	num=sb.st_size/sizeof(UDM_UINT8_POS_LEN);		

	{
		size_t l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if (ind[m].hi < hi) l = m + 1;
			else
			if((ind[m].hi == hi) && (ind[m].lo < lo)) l=m+1;
			else r = m;
		}
		if(r == num) goto err1;
		start = r;
		if (ind[r].hi > f_hi || (ind[r].hi == f_hi && ind[r].lo > f_lo ) ) start--;
			UdmLog(Agent, UDM_LOG_DEBUG, "start:%d   r: %d  .hi: %08x  .lo: %08x", start, r, ind[r].hi, ind[r].lo);
	}	
	if (start != -1) {
		size_t l = start, r = num, m;

		while(l < r){
			m = (l + r) / 2;
			UdmLog(Agent, UDM_LOG_DEBUG, "m: %d  .hi: %08x  .lo: %08x", m, ind[m].hi, ind[m].lo);
			if(ind[m].hi < f_hi) l = m + 1;
			else
			if((ind[m].hi == f_hi) && (ind[m].lo < f_lo)) l = m + 1;
			else r = m;
		}
		if (r == num) stop = num - 1;
		else stop = r;
		if (ind[stop].hi > f_hi || (ind[stop].hi == f_hi && ind[stop].lo > f_lo)) stop--;
	}	

	UdmLog(Agent, UDM_LOG_DEBUG, "num: %d  start: %d [%08x %08x]   stop: %d [%08x %08x]", num, start, ind[start].hi, ind[start].lo,
	       stop, ind[stop].hi, ind[stop].lo);

	if (start != -1) {
	  udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.dat", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	  if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	  }
	  if(ind[start].pos!=lseek(dat_fd,ind[start].pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	  }
	  len=ind[stop].pos+ind[stop].len-ind[start].pos;
	  UdmLog(Agent, UDM_LOG_DEBUG, "len: %d", len);
	  data = (urlid_t*)malloc(len);
	  if(len != (size_t)read(dat_fd,data,len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	  }
	  UdmSort(data, len / sizeof(*data), sizeof(urlid_t), (qsort_cmp)cmp_urlid_t);
	} else {
	  len = 0;
	  data = (urlid_t*)malloc(len + 1);
	}

	close(dat_fd);
	UDM_FREE(ind);

	*size = len / sizeof(*data);
	return(data);

err1:
	if(ind)UDM_FREE(ind);
	return(NULL);
}

static int* LoadLinearLimit(UDM_AGENT *Agent,const char *name,uint4 val,size_t *size){
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT4_POS_LEN key,*found,*ind=NULL;
	struct	stat sb;
	size_t	num;
	urlid_t	*data;
	const char	*vardir=UdmVarListFindStr(&Agent->Conf->Vars,"VarDir",UDM_VAR_DIR);
	
	udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.ind", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT4_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);

	num=sb.st_size/sizeof(UDM_UINT4_POS_LEN);
	key.val=val;
	if(!(found=bsearch(&key,ind,num,sizeof(UDM_UINT4_POS_LEN),(qsort_cmp)cmp_hex4_ind))){
		goto err1;
	}
	udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.dat", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	if(found->pos!=lseek(dat_fd,found->pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	}
	data = (urlid_t*)malloc(found->len);
	if(found->len != (size_t)read(dat_fd,data,found->len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(dat_fd);

	*size = found->len / sizeof(*data);

	UDM_FREE(ind);

	return(data);

err1:
	if(ind)UDM_FREE(ind);
	return(NULL);
}

static int* LoadTimeLimit(UDM_AGENT *Agent,const char *name,uint4 from,uint4 to,size_t *size){
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT4_POS_LEN *found1,*found2,*ind=NULL;
	struct	stat sb;
	size_t	num,len;
	urlid_t	*data;
	const char *dt = UdmVarListFindStr(&Agent->Conf->Vars, "dt", "");
	uint4 dp = 1;
	struct tm tm;
	const char	*vardir=UdmVarListFindStr(&Agent->Conf->Vars,"VarDir",UDM_VAR_DIR);


	bzero((void*)&tm, sizeof(struct tm));
	if (!strcasecmp(dt, "back")) {
	  dp = Udm_dp2time_t(UdmVarListFindStr(&Agent->Conf->Vars, "dp", "")) / 3600;
	  to = time(NULL) / 3600;
	  from = to - dp;
	} else if (!strcasecmp(dt, "er")) {
	  tm.tm_mday = UdmVarListFindInt(&Agent->Conf->Vars, "dd", 1);
	  tm.tm_mon = UdmVarListFindInt(&Agent->Conf->Vars, "dm", 1);
	  tm.tm_year = UdmVarListFindInt(&Agent->Conf->Vars, "dy", 1970) - 1900;
	  if (UdmVarListFindInt(&Agent->Conf->Vars, "dx", 1) == -1) {
	    from = 0;
	    to = mktime(&tm) / 3600;
	  } else {
	    from = mktime(&tm) / 3600;
	    to = INT_MAX;
	  }
	} else if (!strcasecmp(dt, "range")) {
	  sscanf(UdmVarListFindStr(&Agent->Conf->Vars, "db", "01/01/1970"), "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
	  tm.tm_year -= 1900;
	  from = mktime(&tm) / 3600;
	  sscanf(UdmVarListFindStr(&Agent->Conf->Vars, "de", "01/01/1970"), "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
	  tm.tm_year -= 1900;
	  to = mktime(&tm) / 3600;
	} else return NULL;
	
	if(((from==0)&&(to==0))||(from>to)||(dp==0)) return(NULL);

	udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.ind", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT4_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);

	num=sb.st_size/sizeof(UDM_UINT4_POS_LEN);

	if(!from){
		found1=&ind[0];
	}else{
		uint4 l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if(ind[m].val<from) l=m+1;
			else r=m;
		}
		if(r==num) found1=NULL;
		found1=&ind[r];
	}	
	if(!to){
		found2=&ind[num-1];
	}else{
		uint4 l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if(ind[m].val<to) l=m+1;
			else r=m;
		}
		if(r==num) found2=&ind[num-1];
		else if(ind[r].val==to) found2=&ind[r];
		else if(l>0) found2=&ind[l-1];
		else found2=NULL;
	}	
	if(!found1||!found2)return(NULL);

	udm_snprintf(fname, sizeof(fname), "%s%c%s%c%s.dat", vardir, UDMSLASH, TREEDIR, UDMSLASH, name);
	if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	if(found1->pos!=lseek(dat_fd,found1->pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	}
	len=found2->pos+found2->len-found1->pos;
	data = (urlid_t*)malloc(len);
	if(len != (size_t)read(dat_fd,data,len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(dat_fd);
	
	UDM_FREE(ind);
	
	UdmSort(data, len / sizeof(*data), sizeof(*data), (qsort_cmp)cmp_urlid_t);
	*size = len / sizeof(*data);
	return(data);

err1:
	if(ind)UDM_FREE(ind);
	return(NULL);
}

static int PresentInLimit(urlid_t *buf, size_t buf_count, size_t *start, const urlid_t url_id) {
	register size_t m;
	register size_t l;
	register size_t r;

	if(start)l=*start;
	else l=0;
	r=buf_count;
	while(l<r){
		m=(l+r)/2;
/*		fprintf(stderr, "PIL: %d ? %d\n", url_id, buf[m]);*/
		if(buf[m] == url_id) {
		  *start = m;
		  return 1;
		}
		if(buf[m]<url_id) l=m+1;
		else r=m;
	}
/*	fprintf(stderr, "r: %d  buf_count: %d  buf[r]: %d  m: %d  buf[m]: %d\n", r, buf_count, buf[r], m, buf[m]);*/
	if (start) *start = r;
	if(r==buf_count) return(0);
	else
		if(buf[r]==url_id) return(1);
		else return(0);
}


static int cmpdata(UDM_URLDATA *d1, UDM_URLDATA *d2) {
  if (d1->url_id < d2->url_id) return -1;
  if (d1->url_id > d2->url_id) return 1;
  return 0;
}

static int UdmLoadURLData(UDM_AGENT *A, UDM_RESULT *R) {
        size_t i, count, nrec = 0, first = 0;
	int filenum, fd = -1, prevfilenum = - 1;
	char fname[2048];
	UDM_URLDATA *Dat, *D = NULL, K, *F;
	UDM_URL_CRD *Crd;
	struct stat sb;
	const char	*vardir=UdmVarListFindStr(&A->Conf->Vars,"VarDir",UDM_VAR_DIR);

	UdmLog(A, UDM_LOG_DEBUG, "VarDir: %s", vardir);
	count = R->CoordList.ncoords;
	if (count == 0) return UDM_OK;
	Dat = R->CoordList.Data = (UDM_URLDATA*)realloc(R->CoordList.Data, count * sizeof(UDM_URLDATA));
	Crd = R->CoordList.Coords;
	for (i = 0; i < count; i++) {
	  filenum = (Crd[i].url_id >> 16) & 0xFFF;
	  if (filenum != prevfilenum) {
	    if (fd > 0) close(fd);
	    udm_snprintf(fname, sizeof(fname), "%s%c%s%curl%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, filenum, ".dat");
	    fd = open(fname, O_RDONLY|UDM_BINARY, 0644);
	    prevfilenum = filenum;
	    nrec = 0;
	    UdmLog(A, UDM_LOG_DEBUG, "Open %s %s", fname, (fd > 0) ? "OK" : "FAIL");
	    if (fd > 0) {
	      UdmReadLock(fd); 
	      fstat(fd, &sb);
	      D = (UDM_URLDATA*)realloc(D, (size_t)sb.st_size);
	      read(fd, D, (size_t)sb.st_size);
	      nrec = sb.st_size / sizeof(UDM_URLDATA);
	      first = 0;
	      UdmUnLock(fd);
	      UdmLog(A, UDM_LOG_DEBUG, "%d records readed", nrec);
	    }
	  }
	  K.url_id = Crd[i].url_id;
	  if ((nrec > 0) && (F = (UDM_URLDATA*)bsearch(&K, &D[first], nrec - first, sizeof(UDM_URLDATA),(qsort_cmp) cmpdata)) != NULL) {
	    Dat[i] = *F;
	    first = (F - D) / sizeof(UDM_URLDATA);
/*	    UdmLog(A, UDM_LOG_DEBUG, "rec_id %d found", Crd[i].url_id);*/
	  } else {
	    Dat[i].url_id = K.url_id;
	    Dat[i].site_id = 0;
	    Dat[i].pop_rank = 0.0;
/*	    UdmLog(A, UDM_LOG_DEBUG, "rec_id %d not found", Crd[i].url_id);*/
	  }
	}
	UDM_FREE(D);
	if (fd > 0) close(fd);
	return UDM_OK;
}


typedef struct {
	int last;
	int cur;
	int num;
} UDM_MERG;
                 
typedef struct {
	UDM_URL_CRD *plast;
	UDM_URL_CRD *pcur;
	int num;
	int count;
} UDM_PMERG;

#define MAXMERGE 256 /* <= one byte */

int UdmFindWordsCache(UDM_AGENT * Indexer, UDM_RESULT *Res, UDM_DB *db){
	size_t i, nmerg = 0, mmerg = MAXMERGE;
	UDM_URLCRDLIST MCoordList;
	UDM_MERG *merg;
	UDM_PMERG *pmerg = NULL;
	int ind_fd=0,dat_fd=0;
	int wf[256];
	char dname[1024]="";
	struct stat sb;
	int dd, del_count = 0;
	UDM_LOGDEL *del_buf=NULL;
	
	urlid_t *lim_data[MAX_SEARCH_LIMIT];
	size_t lim_size[MAX_SEARCH_LIMIT];
	size_t lim_start[MAX_SEARCH_LIMIT];
	uint4 nlims=0;
	
	int search_mode = UdmSearchMode(UdmVarListFindStr(&Indexer->Conf->Vars, "m", "all"));
	const char	*vardir=UdmVarListFindStr(&Indexer->Conf->Vars,"VarDir",UDM_VAR_DIR);

#ifdef DEBUG_SEARCH
	unsigned long ticks;
	unsigned long total_ticks=UdmStartTimer();
	
	UdmLog(Indexer, UDM_LOG_DEBUG, "Start UdmFindCache()");
#endif

	if ((merg = (UDM_MERG*)malloc(mmerg * sizeof(UDM_MERG))) == NULL) return UDM_ERROR;
	
	UdmWeightFactorsInit(UdmVarListFindStr(&Indexer->Conf->Vars,"wf",""),wf);
	bzero((void*)&MCoordList, sizeof(MCoordList));
	bzero((void*)lim_start, MAX_SEARCH_LIMIT*sizeof(uint4));


#ifdef DEBUG_SEARCH
	ticks = UdmStartTimer();
#endif
	/* Open del log file */
	udm_snprintf(dname, sizeof(dname), "%s%c%s%cdel.log", vardir, UDMSLASH, SPLDIR, UDMSLASH);
	if((dd = open(dname, O_RDONLY | UDM_BINARY)) < 0) {
	  UdmLog(Indexer, UDM_LOG_ERROR, "Can't open del log '%s': %s", dname, strerror(errno));
	  UDM_FREE(merg);
	  return UDM_ERROR;
	}

	/* Allocate del buffer */
	fstat(dd, &sb);
	if (sb.st_size != 0) {
	  del_buf = (UDM_LOGDEL*)malloc((size_t)sb.st_size);
	  del_count = read(dd,del_buf, (size_t)sb.st_size) / sizeof(UDM_LOGDEL);
	}
	close(dd);

	/* Remove duplicates URLs in DEL log     */
	/* Keep only oldest records for each URL */
	if (del_count > 0) {
	  UdmSort(del_buf, (size_t)del_count, sizeof(UDM_LOGDEL), UdmCmpurldellog);
	  del_count = UdmRemoveDelLogDups(del_buf, del_count);
	}

	
#ifdef DEBUG_SEARCH
	ticks = UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Del log loaded (%d)... %.4fs", del_count, (float)ticks / 1000);
#endif


#ifdef DEBUG_SEARCH
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Reading limits (%d)... ", Indexer->nlimits);
	ticks=UdmStartTimer();
#endif
	for(i = 0; i < Indexer->nlimits; i++){
		switch(Indexer->limits[i].type){
			case UDM_LIMTYPE_NESTED:
			        if((lim_data[nlims] = LoadNestedLimit(Indexer, i, &lim_size[nlims]))) nlims++;
				break;
			case UDM_LIMTYPE_TIME:
			        if((lim_data[nlims] = LoadTimeLimit(Indexer,Indexer->limits[i].file_name,
								Indexer->limits[i].hi,
								Indexer->limits[i].lo,
								&lim_size[nlims]))) nlims++;
				break;
			case UDM_LIMTYPE_LINEAR_INT:
			case UDM_LIMTYPE_LINEAR_CRC:
                                 if((lim_data[nlims] = LoadLinearLimit(Indexer,Indexer->limits[i].file_name,
								Indexer->limits[i].hi,
								&lim_size[nlims]))) nlims++;
				break;
		}
	}
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\t\t\tDone (%.2f)", (float)ticks / 1000);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Reading .wrd files (%d words)... ", Res->WWList.nwords);
	ticks=UdmStartTimer();
#endif
	for(i=0;i<Res->WWList.nwords;i++){
		size_t num;
		char ind_name[1024];
		char dat_name[1024];
		T_DAT_IND *ind=NULL;
		T_DAT_IND key, *found;
		
		if (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP) continue;

		udm_snprintf(ind_name, sizeof(ind_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH,
			 (Res->WWList.Word[i].crcword) & (UDM_MAX_LOG - 1), ".ind");
		if((ind_fd=open(ind_name,O_RDONLY|UDM_BINARY))<0){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't open '%s': %s", ind_name, strerror(errno));
			continue;
		}
		fstat(ind_fd, &sb);
		ind=(T_DAT_IND*)malloc((size_t)sb.st_size);
		if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't read '%s': %s", ind_name, strerror(errno));
			close(ind_fd); ind_fd=0;
			goto cont;
		}
		close(ind_fd); ind_fd=0;

		num=sb.st_size/sizeof(T_DAT_IND);		
		key.wrd_id=Res->WWList.Word[i].crcword;
		if(!(found=bsearch(&key,ind,num,sizeof(T_DAT_IND),(qsort_cmp)cmp_ind))){
			goto cont;
		}
		udm_snprintf(dat_name, sizeof(dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH,
			 ((uint4)Res->WWList.Word[i].crcword) & (UDM_MAX_LOG - 1), ".dat");
		if((dat_fd=open(dat_name,O_RDONLY|UDM_BINARY))<0){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't open '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		if(found->pos!=lseek(dat_fd,found->pos,SEEK_SET)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't seek '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		Res->CoordList.Coords=(UDM_URL_CRD*)realloc(Res->CoordList.Coords,
			Res->CoordList.ncoords*sizeof(UDM_URL_CRD)+found->len);
		
		if(found->len != (size_t)read(dat_fd,Res->CoordList.Coords+Res->CoordList.ncoords,found->len)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't read '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		close(dat_fd); dat_fd=0;
		
		num = RemoveOldURLCRD(Res->CoordList.Coords, found->len / sizeof(UDM_URL_CRD), del_buf, del_count);
		Res->WWList.Word[i].count = num;
		UdmLog(Indexer, UDM_LOG_DEBUG, "Word[%d]: %s, %08x cnt: %d", 
		       i, Res->WWList.Word[i].word, Res->WWList.Word[i].crcword & (UDM_MAX_LOG - 1), num);

		if(num){
		        if (nmerg >= mmerg) {
			  mmerg += MAXMERGE;
			  if ((merg = (UDM_MERG*)realloc(merg, mmerg * sizeof(UDM_MERG))) == NULL) {
			    UDM_FREE(del_buf);
			    return UDM_ERROR;
			  }
			}
			merg[nmerg].cur=Res->CoordList.ncoords;
			merg[nmerg].last=Res->CoordList.ncoords+num-1;
			merg[nmerg].num=i;
			nmerg++;
		}

		Res->CoordList.ncoords+=num;
cont:
		UDM_FREE(ind);
	}
	
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\tDone (%.2f)", (float)ticks / 1000);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Merging (%d groups, %d urls)... ", nmerg, Res->CoordList.ncoords);
	ticks=UdmStartTimer();
#endif

	if (search_mode == UDM_MODE_ALL) {

#ifdef MODE_ALL_VIA_LIMITS

	  size_t n_total = 0, min_total = 0, icurrent = 0, corder = 0, nid, z;
	  UDM_URL_CRD *p;

	  for (i = 0; (i < Res->WWList.nwords) && (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP); i++);
	  corder = Res->WWList.Word[i].order;
	  for( ; (i < Res->WWList.nwords) && (Res->WWList.Word[i].order == corder); i++) {
	    min_total += Res->WWList.Word[i].count;
	  }
	  for(i = 0; (i < Res->WWList.nwords); i++) {
	    if (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP) continue;
	    if(Res->WWList.Word[i].order == corder) {
	      n_total += Res->WWList.Word[i].count;
	    } else {
	      if (min_total > n_total) {
		min_total = n_total;
	      }
	      corder = Res->WWList.Word[i].order;
	      n_total = Res->WWList.Word[i].count;
	    }
	  }
	  if (min_total > n_total) {
	    min_total = n_total;
	  }
	  if (min_total == 0) {
	    Res->CoordList.ncoords = 0;
	    UDM_FREE(Res->CoordList.Coords);
	    Res->total_found = 0;
	    UDM_FREE(merg);
	    UDM_FREE(del_buf);
	    return UDM_OK;
	  }

	  corder = Res->WWList.Word[merg[0].num].order;
	  n_total = 0;
	  p = Res->CoordList.Coords + merg[0].cur;
	  for(i = 0; (i < nmerg); i++) {
	    if (Res->WWList.Word[merg[i].num].order == corder) {
	      n_total += Res->WWList.Word[merg[i].num].count;
	    } else {
		  p = Res->CoordList.Coords + merg[icurrent].cur;
		  lim_data[nlims] = (uint4*)malloc( n_total * sizeof(uint4));
		  nid = 0;
		  for (z = 0; z < n_total; z++) {
		    lim_data[nlims][nid] = p->url_id;
		    if ((nid == 0) || (lim_data[nlims][nid] != lim_data[nlims][nid-1])) {
		      nid++;
		    }
		    p++;
		  }
		  UdmSort(lim_data[nlims], (size_t)nid, sizeof(uint4), (qsort_cmp)cmp_urlid_t);
		  lim_size[nlims] = nid;
		  lim_start[nlims] = 0;
		  nlims++;
		
	      icurrent = i;
	      corder = Res->WWList.Word[merg[i].num].order;
	      n_total = Res->WWList.Word[merg[i].num].count;
	    }
	  }
	    p = Res->CoordList.Coords + merg[icurrent].cur;

	    lim_data[nlims] = (uint4*)malloc(n_total * sizeof(uint4));
	    nid = 0;
	    for (z = 0; z < n_total; z++) {
	      lim_data[nlims][nid] = p->url_id;
	      if ((nid == 0) || (lim_data[nlims][nid] != lim_data[nlims][nid-1])) {
		nid++;
	      }
	      p++;
	    }
	    UdmSort(lim_data[nlims], (size_t)nid, sizeof(uint4), (qsort_cmp)cmp_urlid_t);
	    lim_size[nlims] = nid;
	    lim_start[nlims] = 0;
	    nlims++;

#else /* MODE_ALL_VIA_MODE_BOOL */
	    size_t z;
	    UdmVarListReplaceStr(&Indexer->Conf->Vars, "m", "bool"); 
	    for(z = 0; z < Res->WWList.nuniq; z++) {
	      if (z) {
		Res->items[Res->nitems].cmd = UDM_STACK_AND;
		Res->items[Res->nitems].arg = 0;
		Res->nitems++;
		if (Res->nitems >= Res->mitems) {
		  Res->mitems += UDM_MAXSTACK;
		  Res->items = (UDM_STACK_ITEM*)realloc(Res->items, Res->mitems * sizeof(UDM_STACK_ITEM));
		}
	      }
	      Res->items[Res->nitems].cmd=UDM_STACK_WORD;
	      Res->items[Res->nitems].arg = z;
	      Res->nitems++;
	      if (Res->nitems >= Res->mitems) {
		Res->mitems += UDM_MAXSTACK;
		Res->items = (UDM_STACK_ITEM*)realloc(Res->items, Res->mitems * sizeof(UDM_STACK_ITEM));
	      }
	    }
#endif

	}

	if(Res->CoordList.ncoords){
		/* Allocate memory for merged words */
		MCoordList.Coords=(UDM_URL_CRD*)malloc(Res->CoordList.ncoords*sizeof(UDM_URL_CRD));
		if ((pmerg = (UDM_PMERG*)malloc(nmerg * sizeof(UDM_PMERG))) == NULL) { 
		  UDM_FREE(merg); 
		  UDM_FREE(del_buf);
		  return UDM_ERROR; 
		}

		for(i=0;i<nmerg;i++){
			pmerg[i].pcur  = Res->CoordList.Coords+merg[i].cur;
			pmerg[i].plast = Res->CoordList.Coords+merg[i].last;
			pmerg[i].num   = merg[i].num; /* Res->WWList.Word[merg[i].num].order; */
			pmerg[i].count = 0;
		}
	}
	while(nmerg && Res->CoordList.ncoords){
		UDM_PMERG *p;

		if(nmerg==1){
			/*
			if(nlims==0){
				size_t num;

				num=pmerg[0].plast-pmerg[0].pcur;
				memcpy(&wrd1[nwrd1],pmerg[0].pcur,num*sizeof(*wrd1));
				nwrd1+=num;
			}else
			*/
			for(p=&pmerg[0];(p->pcur)<=(p->plast);(p->pcur)++){
				size_t l;
				uint4 weight;
				
				weight=wf[UDM_WRDSEC(p->pcur->coord)];

				if(weight){
					for(l=0;l<nlims;l++){
						if(!PresentInLimit(lim_data[l],lim_size[l],&lim_start[l],p->pcur->url_id)){
							weight=0;
							break;
						}
					}
				}
				if(weight){
					UDM_URL_CRD *Crd = MCoordList.Coords + MCoordList.ncoords;
					*Crd=*(p->pcur);
					Crd[0].coord &= 0xFFFFFF00;
					Crd[0].coord += (p->num & 0xFF);
					MCoordList.ncoords++;
					p->count++;
				}
			}
			break;
		}else{
			size_t k;
			urlid_t min_url_id = (pmerg[0].pcur)->url_id;
			uint4 min_pos=UDM_WRDPOS((pmerg[0].pcur)->coord);
			uint4 min_num=0;
			size_t l;
			uint4 weight;

			for(k=1;k<nmerg;k++){ 
				/* note ">=" to compare pos in the same url */
				if( (min_url_id>(pmerg[k].pcur)->url_id) ||
				    ((min_url_id==(pmerg[k].pcur)->url_id)&&
				     (min_pos>UDM_WRDPOS((pmerg[k].pcur)->coord))) ){
					min_url_id=(pmerg[k].pcur)->url_id;
					min_pos=UDM_WRDPOS((pmerg[k].pcur)->coord);
					min_num=k;
				}
			}
			p=&pmerg[min_num];
			weight=wf[UDM_WRDSEC(p->pcur->coord)];
			if(weight){
				for(l=0;l<nlims;l++){
					if(!PresentInLimit(lim_data[l],lim_size[l],&lim_start[l],p->pcur->url_id)){
						weight=0;
						break;
					}
				}
			}
			if(weight){
				UDM_URL_CRD *Crd = MCoordList.Coords + MCoordList.ncoords;
				Crd[0]=*(p->pcur);
				Crd[0].coord &= 0xFFFFFF00;
				Crd[0].coord += (p->num & 0xFF);
				MCoordList.ncoords++;
				p->count++;
			}
			
			if(p->pcur==p->plast){
				nmerg--;
				if(min_num<nmerg){
					memmove(p,p+1,sizeof(UDM_PMERG)*(nmerg-min_num));
				}
			}else{
				(p->pcur)++;
			}
		}
	}
	UDM_FREE(Res->CoordList.Coords);
	Res->CoordList = MCoordList;
	
	for(i=0;i<nlims;i++) UDM_FREE(lim_data[i]);

#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\tDone (%.2f) Merged ncoords1=%d", (float)ticks / 1000, MCoordList.ncoords);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Grouping by url_id... ");
	ticks=UdmStartTimer();
#endif

	UdmGroupByURL(Indexer,Res);
	
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\t\tDone (%.2f)", (float)ticks / 1000);
#endif

	UdmLog(Indexer, UDM_LOG_DEBUG, "Start load url data %d docs", Res->CoordList.ncoords);
	ticks = UdmStartTimer();
	UdmLoadURLData(Indexer, Res);
	ticks = UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "Stop load url data:\t%.2f", (float)ticks / 1000);


	if ((!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "GroupBySite", "no"), "yes")) 
	    && (UdmVarListFindInt(&Indexer->Conf->Vars, "site", 0) == 0)) {

#ifdef DEBUG_SEARCH
	  ticks=UdmStartTimer() - ticks;
	  UdmLog(Indexer, UDM_LOG_DEBUG, "    Grouping by site_id... ");
#endif

	  UdmSortSearchWordsBySite(&Res->CoordList, Res->CoordList.ncoords);
	  UdmGroupBySite(Indexer, Res);

#ifdef DEBUG_SEARCH
	  ticks=UdmStartTimer() - ticks;
	  UdmLog(Indexer, UDM_LOG_DEBUG, "\t\tDone (%.2f)", (float)ticks / 1000);
#endif

	}

#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Sort by relevancy,pop_rank... ");
#endif

	UdmSortSearchWordsByPattern(Res, &Res->CoordList, Res->CoordList.ncoords, UdmVarListFindStr(&Indexer->Conf->Vars, "s", "RP"));

#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\t\tDone (%.2f)", (float)ticks / 1000);
#endif

	

	Res->total_found = Res->CoordList.ncoords;
#ifdef DEBUG_SEARCH
	total_ticks=UdmStartTimer() - total_ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "Stop UdmFindWordsCache() - %.2f\n", (float)total_ticks / 1000);
#endif


	UDM_FREE(merg);
	UDM_FREE(pmerg);
	UDM_FREE(del_buf);
	return UDM_OK;
/*	
	if(ind_fd)close(ind_fd);
	if(dat_fd)close(dat_fd);
	return(-1);
*/
}


#ifdef UDM_OPEN_ALL_DEL_LOGS
static int Udm_del_fd[UDM_MAX_LOG];
#endif

/* Function to sort LOGDEL list in URL_ID order */
static int cmpurldellog(const void *s1,const void *s2){
	unsigned int n1,n2;
	
	n1=((const UDM_LOGDEL*)s1)->url_id;
	n2=((const UDM_LOGDEL*)s2)->url_id;
	if(n1==n2){
		n1=((const UDM_LOGDEL*)s2)->stamp;
		n2=((const UDM_LOGDEL*)s1)->stamp;
	}
	if(n1<n2)return(-1);
	if(n1>n2)return(1);
	return(0);
}

static int UdmLogdSaveBuf(UDM_AGENT *Indexer, UDM_ENV * Env, size_t log_num){
	UDM_DB *db;
	UDM_LOGDEL *del_buf = NULL;
	int del_count;
	UDM_LOGD *logd;
	char old_dat_name[1024];
	char new_dat_name[1024];
	char old_ind_name[1024];
	char new_ind_name[1024];
	struct stat sb;
	UDM_LOGWORD *log_buf = NULL;
	int old_dat_fd = 0, new_dat_fd = 0;
	int old_ind_fd = 0, new_ind_fd = 0;
	size_t	old_ind_count = 0;
	T_DAT_IND *old_ind = NULL;
	size_t new_ind_count = 0;
	size_t new_ind_mcount = 500;
	T_DAT_IND *new_ind;
	const char	*vardir;
	size_t z, dbfrom = 0, dbto =  Env->dbl.nitems;

	UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
	vardir = UdmVarListFindStr(&Env->Vars,"VarDir",UDM_VAR_DIR);
	UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);

	for (z = dbfrom; z < dbto; z++) {
	  db = &Env->dbl.db[z];
	  if (db->DBMode != UDM_DBMODE_CACHE) continue;
	  logd = db->LOGD;

	  if (Env->logs_only) {
	    char		fname[1024];
	    int           fd;
	    size_t        nbytes = logd->wrd_buf[log_num].nrec * sizeof(UDM_LOGWORD);

	    if (nbytes > 0) {
	      udm_snprintf(fname, sizeof(fname), "%s%03X.log", db->log_dir, log_num);
	
	      if((fd=open(fname,open_flags,open_perm))!=-1){
		nbytes = logd->wrd_buf[log_num].nrec * sizeof(UDM_LOGWORD);
		UdmWriteLock(fd);
		if(nbytes != (size_t)write(fd, logd->wrd_buf[log_num].data,nbytes)){
			sprintf(db->errstr,"Can't write %d nbytes to '%s': %s\n", nbytes, fname, strerror(errno));
			db->errcode=1;
			UdmUnLock(fd);
			close(fd);
			return UDM_ERROR;
		}
		UdmUnLock(fd);
		close(fd);
		logd->wrd_buf[log_num].nrec = 0;
	      }else{
		sprintf(db->errstr,"Can't open '%s': %s\n",fname,strerror(errno));
		db->errcode=1;
		return UDM_ERROR;
	      }
	    }
	    return UDM_OK;
	  }



	  new_ind = (T_DAT_IND*)malloc(new_ind_mcount*sizeof(T_DAT_IND));

	  /* Open del log file */
	  del_buf = logd->wrd_buf[log_num].del_buf;
	  del_count = logd->wrd_buf[log_num].ndel;
	
	  /* Remove duplicates URLs in DEL log     */
	  /* Keep only oldest records for each URL */
	  UdmSort(del_buf, (size_t)del_count, sizeof(UDM_LOGDEL), cmpurldellog);
	  del_count = UdmRemoveDelLogDups(del_buf, del_count);

	  UdmLog(Indexer, UDM_LOG_INFO, "Cache %03X update", log_num);

	  udm_snprintf(old_dat_name, sizeof(old_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log_num, ".dat");
	  old_dat_fd = open(old_dat_name, O_RDONLY | UDM_BINARY);
	  udm_snprintf(old_ind_name, sizeof(old_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log_num, ".ind");
	  old_ind_fd = open(old_ind_name, O_RDONLY | UDM_BINARY);
	
	  if((old_dat_fd > 0) && (old_ind_fd > 0)){
		fstat(old_ind_fd, &sb);
#if 0
		//ind=(T_DAT_IND*)mmap(0,sb.st_size,PROT_READ,MAP_SHARED,ind_fd,0);
#endif
		old_ind = (T_DAT_IND*)malloc((size_t)sb.st_size + 1);
		if (sb.st_size > 0) read(old_ind_fd, old_ind, (size_t)sb.st_size);
		close(old_ind_fd);
		old_ind_count = sb.st_size / sizeof(T_DAT_IND);
	  }

	  udm_snprintf(new_dat_name, sizeof(new_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR,UDMSLASH, log_num, ".dat.new");
	  if((new_dat_fd=open(new_dat_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog(Indexer, UDM_LOG_ERROR, "Can't open '%s': %s", new_dat_name, strerror(errno));
		goto err1;
	  }
	  udm_snprintf(new_ind_name, sizeof(new_dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, log_num, ".ind.new");
	  if((new_ind_fd=open(new_ind_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog(Indexer, UDM_LOG_ERROR, "Can't open '%s': %s", new_ind_name, strerror(errno));
		goto err1;
	  }

	  {
		size_t mkeys=500;
		size_t nkeys=0;
		T_WRD_POS_LEN *keys=(T_WRD_POS_LEN*)malloc(mkeys*sizeof(T_WRD_POS_LEN));
		size_t i,prev=0;
		size_t n;
		size_t k;
		size_t pos=0;
		

		/* Allocate buffer for log */
		log_buf = logd->wrd_buf[log_num].data;
		n = logd->wrd_buf[log_num].nrec++;
		UdmSort(log_buf, n, sizeof(UDM_LOGWORD), (qsort_cmp)cmplog);
		n=RemoveOldWords(log_buf,n,del_buf,del_count);


/*printf("writing... "); fflush(stdout);*/

		for(i=1;i<n+1;i++){
			/* Next word? */
			if( (i==n) || (log_buf[prev].wrd_id != log_buf[i].wrd_id)){
				if(mkeys==nkeys){
					mkeys+=500;
					keys=(T_WRD_POS_LEN*)realloc(keys,mkeys*sizeof(T_WRD_POS_LEN));
				}
				keys[nkeys].wrd_id=log_buf[prev].wrd_id;
				keys[nkeys].pos=prev;
				keys[nkeys].len=i-prev;
				keys[nkeys].done=0;
				nkeys++;
			
				prev=i;
			}
		}
		for(k=0;k<old_ind_count;k++){
			uint4		wrd_id=old_ind[k].wrd_id;
			UDM_URL_CRD	*new_data=NULL;
			UDM_URL_CRD	*old_data=NULL;
			size_t		old_count;
			size_t		old_added=0; 
			size_t		new_count=0;
			size_t		new_len;
			int		wrd_ind;
			size_t 		o;
			size_t 		start = 0;

			old_data=(UDM_URL_CRD*)malloc(old_ind[k].len);
			if(old_ind[k].pos!=lseek(old_dat_fd,old_ind[k].pos,SEEK_SET)){
				UdmLog(Indexer, UDM_LOG_ERROR, "Can't seek '%s': %s\n", old_dat_name, strerror(errno));
				goto err1;
			}
			read(old_dat_fd,old_data,old_ind[k].len);
			new_data=(UDM_URL_CRD*)malloc(old_ind[k].len);

			old_count=old_ind[k].len/sizeof(UDM_URL_CRD);

			for(o=0;o<old_count;o++){
				if(!PresentInDelLog(del_buf,del_count,&start,old_data[o].url_id)){
					new_data[old_added]=old_data[o];
					old_added++;
				}
			}
#if 0
printf("del_count=%d\told_count=%d\told_added=%d\n",del_count,old_count,old_added);
#endif

			if( (wrd_ind=PresentInNew(keys, nkeys, wrd_id))>=0 ){
				new_count=old_added+keys[wrd_ind].len;
				new_len=new_count*sizeof(UDM_URL_CRD);
				new_data=(UDM_URL_CRD*)realloc(new_data,new_len);
#if 0
//printf("wrd_ind=%d old_added=%d keys[wrd_ind].pos=%d keys[wrd_ind].len=%d\n",wrd_ind,old_added,keys[wrd_ind].pos,keys[wrd_ind].len);
#endif
				if(new_data){
					size_t j;
					for(j=old_added;j<new_count;j++){
						new_data[j].url_id=log_buf[keys[wrd_ind].pos+j-old_added].url_id;
						new_data[j].coord=log_buf[keys[wrd_ind].pos+j-old_added].coord;
					}
				}else{
					goto err1;
				}
				keys[wrd_ind].done=1;
			}else{
				new_count=old_added;
			}
			if(new_count){
				UdmSort(new_data,new_count,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				new_len=new_count*sizeof(UDM_URL_CRD);
				write(new_dat_fd,new_data,new_len);

				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=new_len;
				pos+=new_len;
				new_ind_count++;
			}
			UDM_FREE(old_data);
			UDM_FREE(new_data);
		}
		if(old_ind)UDM_FREE(old_ind);
		if(old_dat_fd)close(old_dat_fd);
		for(i=0;i<nkeys;i++){
			if(!keys[i].done){
				UDM_URL_CRD       *new_data=NULL;
				size_t		j;
				
				new_data=(UDM_URL_CRD*)malloc(keys[i].len*sizeof(*new_data));
				for(j=0;j<keys[i].len;j++){
					new_data[j].url_id=log_buf[keys[i].pos+j].url_id;
					new_data[j].coord=log_buf[keys[i].pos+j].coord;
				}
				UdmSort(new_data,keys[i].len,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				write(new_dat_fd,new_data,keys[i].len*sizeof(UDM_URL_CRD));
				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=keys[i].wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=keys[i].len*sizeof(UDM_URL_CRD);
				pos+=keys[i].len*sizeof(UDM_URL_CRD);
				new_ind_count++;

				UDM_FREE(new_data);
			}
		}
		UDM_FREE(keys);
	  }
	  logd->wrd_buf[log_num].nrec = 0;
	  logd->wrd_buf[log_num].ndel = 0;
	
	  close(new_dat_fd);

/*printf("creating index... "); fflush(stdout);*/

	  UdmSort(new_ind, new_ind_count, sizeof(T_DAT_IND), (qsort_cmp)cmp_ind);	
	  write(new_ind_fd, new_ind, new_ind_count * sizeof(T_DAT_IND));
	  UDM_FREE(new_ind);
	  close(new_ind_fd);

#ifdef WIN32
	  remove(old_dat_name);
	  remove(old_ind_name);
#endif
	
	  rename(new_dat_name, old_dat_name);
	  rename(new_ind_name, old_ind_name);

/*	  UdmLog(Indexer, UDM_LOG_EXTRA, "\tDone cache %03X", log_num);*/

/*printf("Done\n"); fflush(stdout);*/
	}

	return UDM_OK;

err1:
	if(old_ind_fd) close(old_ind_fd);
	if(new_ind_fd) close(new_ind_fd);
	if(old_dat_fd) close(old_dat_fd);
	if(new_dat_fd) close(new_dat_fd);

	sprintf(db->errstr, "SaveBuf error");
	db->errcode = 1;
	return UDM_ERROR;
}

int UdmLogdSaveAllBufs(UDM_AGENT *Agent) {
  UDM_DB *db;
  UDM_ENV *Env = Agent->Conf;
	size_t i;
	size_t j, dbfrom = 0, dbto =  Env->dbl.nitems;
	int res = UDM_OK;

	UDM_GETLOCK(Agent, UDM_LOCK_CONF);
	dbto =  Env->dbl.nitems;
	UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);
	
	for (j = dbfrom; j < dbto; j++) {
	  UDM_GETLOCK(Agent, UDM_LOCK_CONF);
	  db = &Env->dbl.db[j];
	  UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);
	  UDM_GETLOCK(Agent, UDM_LOCK_DB);
	  if (db->LOGD == NULL) { 
	    UDM_RELEASELOCK(Agent, UDM_LOCK_DB);
	    continue;
	  }
	  for(i = 0; i < UDM_MAX_LOG; i++){
		if(db->LOGD->wrd_buf[i].nrec || db->LOGD->wrd_buf[i].ndel){
			res = UdmLogdSaveBuf(Agent, Env, i);
		}
		if (res != UDM_OK) break;
	  }
	  UDM_RELEASELOCK(Agent, UDM_LOCK_DB);
	  UDMSLEEP(0);
	  if (res != UDM_OK) break;
	}
	return res;
}

static int UdmLogdCloseLogs(UDM_AGENT *Agent) {
  UDM_ENV *Env = Agent->Conf;
  size_t j, dbfrom = 0, dbto =  Env->dbl.nitems;
  UDM_DB *db;
  for (j = dbfrom; j < dbto; j++) {
    db = &Env->dbl.db[j];
    if (Env->logs_only) {
	if(db->del_fd){
		close(db->del_fd);
		db->del_fd = 0;
	}
    }
  }
  return  UdmLogdSaveAllBufs(Agent);
}

static int UdmLogdInit(UDM_ENV *Env, UDM_DB *db, const char* var_dir, size_t i, int shared) {
        char shm_name[1024];
	int fd;

        udm_snprintf(db->log_dir, 1024, "%s%s%s%s", var_dir, UDMSLASHSTR, SPLDIR, UDMSLASHSTR);
	db->errstr[0] = 0;

	if (shared) {
	  
#ifdef WIN32
	  return UDM_ERROR;
#else

	    sprintf(shm_name, "%s%sLOGD.%d", var_dir, UDMSLASHSTR, i);
#ifdef HAVE_SYS_MMAN_H
	    if ((fd = shm_open(shm_name, O_RDWR | O_CREAT, (mode_t)0644)) < 0) {
	      sprintf(shm_name, "%sLOGD.%d", UDMSLASHSTR, i);
	      if ((fd = shm_open(shm_name, O_RDWR | O_CREAT, (mode_t)0644)) < 0) {
		sprintf(Env->errstr, "shm_open (%s): %d: %s", shm_name, errno, strerror(errno));
		return UDM_ERROR;
	      }
	    }
	    fprintf(stderr, "Open LOGD shared: %s\n", shm_name);
	    ftruncate(fd, (off_t)sizeof(UDM_LOGD));
	    if ((db->LOGD = mmap( NULL, sizeof(UDM_LOGD), PROT_READ | PROT_WRITE, MAP_SHARED, fd, (off_t)0)) == NULL) {
	      sprintf(Env->errstr, "mmap: %d: %s", errno, strerror(errno));
	      return UDM_ERROR;
	    }
	    close(fd);
#elif HAVE_SYS_SHM_H
	    if ((fd = shmget(ftok(shm_name, 0), sizeof(UDM_LOGD), IPC_CREAT | SHM_R | SHM_W | (SHM_R>>3) | (SHM_R>>6) )) < 0) {
	      sprintf(Env->errstr, "shmget (%s): %d: %s", shm_name, errno, strerror(errno));
	      return UDM_ERROR;
	    }
	    if ((db->LOGD = (UDM_LOGD*)shmat( fd, NULL, 0)) == -1) {
	      sprintf(Env->errstr, "mmap: %d: %s", errno, strerror(errno));
	      return UDM_ERROR;
	    }
#else
#error No shared memory support found
#endif

#endif

	} else {
	  if ((db->LOGD = (UDM_LOGD*)malloc(sizeof(UDM_LOGD))) == NULL) {
		strcpy(Env->errstr, "Out of memory");
		return UDM_ERROR;
	  }
	}

	for(i=0;i<UDM_MAX_LOG;i++) {
	    db->LOGD->wrd_buf[i].nrec = 0;
	    db->LOGD->wrd_buf[i].ndel = 0;
	}
	
	if (Env->logs_only) {
	    char	del_log_name[1024];

	    udm_snprintf(del_log_name,sizeof(del_log_name),"%s%s", db->log_dir, "del.log");
	    if((db->del_fd = open(del_log_name,O_RDWR | O_CREAT | UDM_BINARY, 0644))==-1){
		sprintf(db->errstr,"Can't open '%s': %s\n",del_log_name,strerror(errno));
		db->errcode=1;
		return UDM_ERROR;
	    }
	    lseek(db->del_fd, (off_t)0, SEEK_END);
	}
	return UDM_OK;
}

int UdmLogdClose(UDM_AGENT *Agent, UDM_DB *db, int shared) {

	if (!shared) UDM_FREE(db->LOGD);
	return UDM_OK;
}

static int URLDataWrite(UDM_ENV *Conf, UDM_DB *db) {
        size_t i;
	int filenum, fd = -1, prevfilenum = - 1;
	UDM_URLDATALIST	DatL;
	char fname[2048];
	const char	*vardir=UdmVarListFindStr(&Conf->Vars,"VarDir",UDM_VAR_DIR);

	UdmURLData(Conf, &DatL, db);

	for(i = 0; i <= 0xFFF; i++) {
		udm_snprintf(fname, sizeof(fname), "%s%c%s%curl%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, i, ".dat");
		unlink(fname);
	}

	for (i = 0; i < DatL.nitems; i++) {
	  filenum = (DatL.Item[i].url_id >> 16) & 0xFFF;
	  if (filenum != prevfilenum) {
	    if (fd > 0) close(fd);
	    udm_snprintf(fname, sizeof(fname), "%s%c%s%curl%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, filenum, ".dat");
	    fd = open(fname, O_WRONLY|O_APPEND|O_CREAT|UDM_BINARY, 0644);
	    prevfilenum = filenum;
	  }
	  if (fd > 0) {
	    UdmWriteLock(fd);
	    write(fd, &DatL.Item[i], sizeof(UDM_URLDATA));
	    UdmUnLock(fd);
	  }
	}
	if (fd > 0) close(fd);

	UDM_FREE(DatL.Item);
	return UDM_OK;
}


int UdmURLDataWrite(UDM_AGENT *Indexer, UDM_DB *db) {
	size_t sent;
	ssize_t recvt;
	UDM_LOGD_CMD cmd;
	char reply;

	if (db->DBMode != UDM_DBMODE_CACHE) return UDM_OK;

	cmd.stamp = time(NULL);
	cmd.url_id = 0;
	cmd.cmd = UDM_LOGD_CMD_DATA;
	cmd.nwords = 0;

	  if(db->cached_sd){
		sent = UdmSend(db->cached_sd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr,"[%d] Can't write to cached: %s", __LINE__, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		}
		while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		  if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached [%d], %d, %s", __LINE__, recvt, strerror(errno));
			db->errcode = 1;
			return(UDM_ERROR);
		  }
		  UDMSLEEP(1);
		}
		if (reply != 'O') {
			sprintf(db->errstr, "Can't incorrect reply from cached %d", __LINE__);
			db->errcode = 1;
		  return UDM_ERROR;
		}
	  }else{
	        UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
		if(URLDataWrite(Indexer->Conf, db)){
		  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
			return UDM_ERROR;
		}
		UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
		if (UdmCacheMakeIndexes(Indexer, db)) {
		  return UDM_ERROR;
		}
	  }
	return UDM_OK;
}


int UdmLogdStoreDoc(UDM_AGENT *Agent, UDM_LOGD_CMD cmd, UDM_LOGD_WRD *wrd, UDM_DB *db) {
	UDM_LOGDEL logdel;
	UDM_LOGD *logd;
	UDM_ENV *Env = Agent->Conf;
	register size_t i;

	if (db->DBMode != UDM_DBMODE_CACHE) return UDM_OK;
	logd = db->LOGD;

	  if (cmd.cmd == UDM_LOGD_CMD_DATA) {
	    URLDataWrite(Env, db);
	    return UDM_OK;
	  }


	  logdel.stamp = cmd.stamp;
	  logdel.url_id = cmd.url_id;

	  if (Env->logs_only) {
	    UdmWriteLock(db->del_fd);
	    if(sizeof(UDM_LOGDEL)!=write(db->del_fd,&logdel,sizeof(UDM_LOGDEL))){
		sprintf(db->errstr,"Can't write to del.log: %s\n",strerror(errno));
		db->errcode=1;
		UdmUnLock(db->del_fd);
		return UDM_ERROR;
	    }
	    UdmUnLock(db->del_fd);
	  } else {
	    int nrec;
	      
	    for (i = 0; i < UDM_MAX_LOG; i++) {
	      nrec = logd->wrd_buf[i].ndel;
	      if (nrec == UDM_MAX_DEL || ( (nrec > UDM_MAX_DEL - UDM_INF_DEL) && (nrec + (rand() % UDM_INF_DEL) > UDM_MAX_DEL)) ) {
		UdmLog(Agent, UDM_LOG_DEBUG, "num: %03x\t: nrec:%d ndel:%d", 
			       i, logd->wrd_buf[i].nrec, logd->wrd_buf[i].ndel);
		if(UDM_OK != UdmLogdSaveBuf(Agent, Env, i)) {
		  return UDM_ERROR;
		}
		nrec = 0;
	      }
	      logd->wrd_buf[i].del_buf[nrec] = logdel;
	      logd->wrd_buf[i].ndel++;
	    }
	  }

	  if(cmd.nwords == 0) return UDM_OK;

	  for(i = 0; i < cmd.nwords; i++){
		size_t	num = wrd[i].wrd_id & (UDM_MAX_LOG - 1); /* wrd[i].wrd_id >> 20; */
		int	nrec;

		nrec = logd->wrd_buf[num].nrec;
		logd->wrd_buf[num].data[nrec].stamp = cmd.stamp;
		logd->wrd_buf[num].data[nrec].url_id = cmd.url_id;
		logd->wrd_buf[num].data[nrec].wrd_id = wrd[i].wrd_id;
		logd->wrd_buf[num].data[nrec].coord = wrd[i].coord;
		logd->wrd_buf[num].nrec++;
		if(logd->wrd_buf[num].nrec == UDM_MAX_WRD){
		        UdmLog(Agent, UDM_LOG_DEBUG, "num: %03x\t: nrec:%d ndel:%d", 
				       num, logd->wrd_buf[num].nrec, logd->wrd_buf[num].ndel);
			if (UDM_OK != UdmLogdSaveBuf(Agent, Env, num)) {
			  return UDM_ERROR;
			}
		}
	  }
	return UDM_OK;
}

static int UdmLogdCachedCheck(UDM_AGENT *A, UDM_DB *db) {
#ifdef HAVE_SQL
  size_t b, i, ncrd;
  char dat_name[1024];
  int dat_fd, res;
  urlid_t prevrec = (urlid_t)-1;
  const char *vardir;
  struct stat sb;
  UDM_URL_CRD *crd = NULL;
  UDM_SQLRES	SQLRes;
  urlid_t *todel = (int*)malloc(128 * sizeof(urlid_t));
  size_t ndel = 0, mdel = 128;
  UDM_LOGD_CMD	cmd;

  if (todel == NULL) return UDM_ERROR;

  UDM_GETLOCK(A, UDM_LOCK_CONF);
  vardir = UdmVarListFindStr(&A->Conf->Vars, "VarDir", UDM_VAR_DIR);
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  for (b = 0; b < UDM_MAX_LOG; b++) {
    udm_snprintf(dat_name, sizeof(dat_name), "%s%c%s%c%03X%s", vardir, UDMSLASH, TREEDIR, UDMSLASH, b, ".dat");
    dat_fd = open(dat_name, O_RDONLY | UDM_BINARY);
    if (dat_fd <= 0) continue;
    UdmReadLock(dat_fd);
    fstat(dat_fd, &sb);
    ncrd = sb.st_size / sizeof(*crd);
    if ((crd = (UDM_URL_CRD*) realloc(crd, sizeof(*crd) * ncrd)) == NULL) return UDM_ERROR;
    read(dat_fd, crd, ncrd * sizeof(*crd));
    UdmUnLock(dat_fd);
    close(dat_fd);
    UdmSort(crd, ncrd, sizeof(*crd), (qsort_cmp)cmp_url_id);

    for (i = 0; i < ncrd; i++) {
      if (prevrec == crd[i].url_id) continue;
      udm_snprintf(dat_name, sizeof(dat_name), "SELECT rec_id FROM url WHERE rec_id=%d", crd[i].url_id);
      UDM_GETLOCK(A, UDM_LOCK_DB);
      res = UdmSQLQuery(db, &SQLRes, dat_name);
      UDM_RELEASELOCK(A, UDM_LOCK_DB);
      if(UDM_OK != res) return res;
      if (UdmSQLNumRows(&SQLRes) == 0) {
	if (ndel >= mdel) {
	  mdel += 128;
	  todel = (urlid_t*)realloc(todel, mdel * sizeof(urlid_t));
	  if (todel == NULL) return UDM_ERROR;
	}
	todel[ndel++] = crd[i].url_id;
      }
      UdmSQLFree(&SQLRes);
      prevrec = crd[i].url_id;
    }
    for (i = 0; i < ndel; i++) {
        UdmLog(A, UDM_LOG_EXTRA, "Cache %03X: deleting url_id: %d", b, todel[i]);
	cmd.stamp=time(NULL);
	cmd.url_id = todel[i];
	cmd.cmd = UDM_LOGD_CMD_WORD;
	cmd.nwords=0;
	if(UdmLogdStoreDoc(A, cmd, NULL, db)) {
	  return UDM_ERROR;
	}
    }
    UdmLog(A, UDM_LOG_INFO, "Cache %03X, %d lost records deleted", b, ndel);
    ndel = 0;
  }
  UDM_FREE(crd);
  UDM_FREE(todel);
#endif
  return UDM_OK;
}

int UdmCachedCheck(UDM_AGENT *A) {
	size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
	UDM_DB	*db;
	int sent;
	ssize_t recvt;
	char reply;
	UDM_LOGD_CMD cmd;

	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	for (i = dbfrom; i < dbto; i++) {
	  UDM_GETLOCK(A, UDM_LOCK_CONF);
	  db = &A->Conf->dbl.db[i];
	  UDM_RELEASELOCK(A, UDM_LOCK_CONF);
	  UDM_GETLOCK(A, UDM_LOCK_DB);
	  if (db->DBMode != UDM_DBMODE_CACHE) {
	    UDM_RELEASELOCK(A, UDM_LOCK_DB);
	    continue;
	  }

	  cmd.nwords = 0;
	  cmd.stamp = time(NULL);
	  cmd.url_id = 0;
	  cmd.cmd = UDM_LOGD_CMD_CHECK;
	  if(db->cached_sd) {
		sent = UdmSend(db->cached_sd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr, "[%d] Can't write to cached: %s", __LINE__, strerror(errno));
			db->errcode = 1;
			UDM_RELEASELOCK(A, UDM_LOCK_DB);
			return(UDM_ERROR);
		}
		while ((recvt = UdmRecvall(db->cached_sd, &reply, sizeof(char))) != 1) {
		  if (recvt < 0) {
			sprintf(db->errstr, "Can't receive from cached [%d]: %d %s", __LINE__, recvt, strerror(errno));
			db->errcode = 1;
			UDM_RELEASELOCK(A, UDM_LOCK_DB);
			return(UDM_ERROR);
		  }
		  UDMSLEEP(1);
		}
		if (reply != 'O') {
			sprintf(db->errstr, "Incorrect reply received from cached %d", __LINE__);
			db->errcode = 1;
			UDM_RELEASELOCK(A, UDM_LOCK_DB);
			return UDM_ERROR;
		}
		UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  } else {
	    UDM_RELEASELOCK(A, UDM_LOCK_DB);
	    UdmLogdCachedCheck(A, db);
	  }
	}
	return UDM_OK;
}


/* cache:// */

static int UdmCacheLinksAdd(UDM_AGENT *A, urlid_t ot, urlid_t k) {
  UDM_STORE_PARAM P;
  UDM_CACHE_LINK  L;
  long NewItemPos;

  bzero(&P, sizeof(P));
  P.subdir = "links";
  P.BASE_SIG = UDM_LINKS_SIG;
  P.basename = "";
  P.indname = "ot";
  P.rec_id = ot;
  
  if (UdmOpenBase(A, &P, UDM_WRITE_LOCK) != UDM_OK) {
    return UDM_ERROR;
  }

  if (P.Item.rec_id == ot) {
    return UdmCloseBase(&P);
  } else {
    if (P.mishash && P.Item.rec_id != 0) {
      if (fseek(P.Ifd, 0, SEEK_END)) {
	return UDM_ERROR;
      }
      P.Item.next = NewItemPos = ftell(P.Ifd);
      if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
	return UDM_ERROR;
      }
      if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	return UDM_ERROR;
      }
      P.CurrentItemPos = NewItemPos;
    }
    P.Item.rec_id = ot;
    P.Item.next = 0;
    if (fseek(P.Sfd, 0, SEEK_END)) {
      return UDM_ERROR;
    }
    P.Item.offset = ftell(P.Sfd);
  }
  L.ot = ot;
  L.k = k;
  L.weight = 0.0;
  if (fwrite(&L, sizeof(L), 1, P.Sfd) != 1) {
    return UDM_ERROR;
  }
  if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
    return UDM_ERROR;
  }
  if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
    UdmLog(A, UDM_LOG_ERROR, "Can't write index for file %s", P.Ifilename);
  }

  return  UdmCloseBase(&P);
}

static int UdmCacheLinksDelOt(UDM_AGENT *A, urlid_t ot) {
  UDM_STORE_PARAM P;
  long NextItemPos;

  bzero(&P, sizeof(P));
  P.subdir = "links";
  P.BASE_SIG = UDM_LINKS_SIG;
  P.basename = "";
  P.indname = "ot";
  P.rec_id = ot;

  if (UdmOpenBase(A, &P, UDM_WRITE_LOCK) != UDM_OK) {
    return UDM_ERROR;
  }

  if (P.Item.rec_id == ot) {
    NextItemPos = P.Item.next;
    while(NextItemPos != 0) {
      if (fseek(P.Ifd, NextItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fread(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	UdmLog(A, UDM_LOG_ERROR, "Can't read hash chain for file %s", P.Ifilename);
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      P.CurrentItemPos = NextItemPos;
      NextItemPos = P.Item.next;
      P.Item.next = P.CurrentItemPos;
      if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	UdmLog(A, UDM_LOG_ERROR, "Can't write hash chain for file %s", P.Ifilename);
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
    }
    P.Item.rec_id = 0;
    P.Item.offset = 0;
    P.Item.next = 0;
    if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
      UdmCloseBase(&P);
      return UDM_ERROR;
    }
    if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
      UdmLog(A, UDM_LOG_ERROR, "Can't write hash chain for file %s", P.Ifilename);
      UdmCloseBase(&P);
      return UDM_ERROR;
    }
  }
  return  UdmCloseBase(&P);
}


static int UdmAddURLCache(UDM_AGENT *A, UDM_DOCUMENT *Doc, UDM_DB *db) {
  UDM_STORE_PARAM P;
  UDM_CACHE_URL   REC;
  const char	  *url = UdmVarListFindStr(&Doc->Sections,"URL","");
  int		  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars,"UseCRC32URLId","no"),"yes");
  size_t          size_to_store, i;
  long            NewItemPos;
  udmhash32_t     url_id = UdmVarListFindInt(&Doc->Sections, "URL_ID", 0);

  REC.seed = url_id & 0xFF;
	
  if(use_crc32_url_id){
    REC.rec_id = url_id;
  } else {
    char  fname[1024];
    int   fd;
    udm_snprintf(fname, 1024, "%s%s", UdmVarListFindStr(&A->Conf->Vars, "VarDir", UDM_VAR_DIR), "url.seq");
    if((fd = open(fname, O_RDWR | O_CREAT | UDM_BINARY, 0644)) == -1) {
      UdmLog(A, UDM_LOG_ERROR, "Can't open file %s ", fname);
      return UDM_ERROR;
    }
    UdmWriteLock(db->del_fd);
    if (read(fd, &REC.rec_id, sizeof(REC.rec_id)) != sizeof(REC.rec_id)) {
      UdmLog(A, UDM_LOG_ERROR, "Can't read file %s, assuming start at 1", fname);
      REC.rec_id = 1;
    }
    REC.rec_id++;
    lseek(fd, (off_t)0, SEEK_SET);
    if (write(fd, &REC.rec_id, sizeof(REC.rec_id)) != sizeof(REC.rec_id)) {
      UdmLog(A, UDM_LOG_ERROR, "Can't write file %s ", fname);
      return UDM_ERROR;
    }
    UdmUnLock(fd);
    close(fd);
  }

  REC.status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
  REC.docsize = UdmVarListFindInt(&Doc->Sections, "Content-Length", 0);
  REC.next_index_time = UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Next-Index-Time", ""));
  REC.last_mod_time = UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified", ""));
  REC.referrer = UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
  REC.hops = UdmVarListFindInt(&Doc->Sections, "Hops", 0);
  REC.crc32 = UdmVarListFindInt(&Doc->Sections,"crc32",0);
  REC.bad_since_time = time(NULL);
  REC.site_id = UdmVarListFindInt(&Doc->Sections,"Site_id",0);
  REC.server_id = UdmVarListFindInt(&Doc->Sections,"Server_id",0);
  REC.pop_rank = 0.0;
  REC.infoseek = size_to_store = sizeof(UDM_CACHE_URL) + strlen(url) + 1;

  for(i = 0; i < Doc->Sections.nvars; i++){
    UDM_VAR *Sec = &Doc->Sections.Var[i];
    if(Sec->curlen && Sec->val && Sec->name && Sec->section){
      size_to_store += strlen(Sec->name) + Sec->curlen + 2;
    }
  }
  size_to_store++; /* plus one 0 for urlinfo end mark */

  bzero(&P, sizeof(P));
  P.subdir = "url";
  P.BASE_SIG = UDM_URL_SIG;
  P.basename = "";
  P.indname = "rec_id";
  P.rec_id = REC.rec_id;

  if (UdmOpenBase(A, &P, UDM_WRITE_LOCK) != UDM_OK) {
    return UDM_ERROR;
  }
  if (P.Item.rec_id == REC.rec_id) {
    if (P.Item.size < size_to_store) {
      if (fseek(P.Sfd, 0, SEEK_END)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      P.Item.offset = ftell(P.Sfd);
    } else {
      if (fseek(P.Sfd, P.Item.offset, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
    }
  } else { /* new rec_id added */
    if (P.mishash && P.Item.rec_id != 0) {
      if (fseek(P.Ifd, 0, SEEK_END)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      P.Item.next = NewItemPos = ftell(P.Ifd);
      if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      P.CurrentItemPos = NewItemPos;
    }
    P.Item.rec_id = REC.rec_id;
    P.Item.next = 0;
    if (fseek(P.Sfd, 0, SEEK_END)) {
      UdmCloseBase(&P);
      return UDM_ERROR;
    }
    P.Item.offset = ftell(P.Sfd);
  }

  P.Item.size = size_to_store;

  if (fwrite(&REC, sizeof(REC), 1, P.Sfd) != 1) {
    UdmCloseBase(&P);
    return UDM_ERROR;
  }
  if (fwrite(url, strlen(url) + 1, 1, P.Sfd) != 1) {
    UdmCloseBase(&P);
    return UDM_ERROR;
  }
  for(i = 0; i < Doc->Sections.nvars; i++){
    UDM_VAR *Sec = &Doc->Sections.Var[i];
    if(Sec->curlen && Sec->val && Sec->name && Sec->section){
      if (fwrite(Sec->name, strlen(Sec->name) + 1, 1, P.Sfd) != 1) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fwrite(Sec->val, strlen(Sec->val) + 1, 1, P.Sfd) != 1) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
    }
  }
  {
    char b = '\0';
    if (fwrite(&b, 1, 1, P.Sfd) != 1)  {
      UdmCloseBase(&P);
      return UDM_ERROR;
    }
  }

  if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
    UdmCloseBase(&P);
    return UDM_ERROR;
  }
  if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
    UdmLog(A, UDM_LOG_ERROR, "Can't write index for file %s", P.Ifilename);
    UdmCloseBase(&P);
    return UDM_ERROR;
  }

  if (UdmCacheLinksAdd(A, (urlid_t)UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0), REC.rec_id) != UDM_OK) {
    return UDM_ERROR;
  }

  return UdmCloseBase(&P);
}

static int UdmDeleteURLCache(UDM_AGENT *A, UDM_DOCUMENT *Doc, UDM_DB *db) {
  long   NextItemPos;
  udmhash32_t url_id = (udmhash32_t)UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int    res;
  UDM_STORE_PARAM P;

  bzero(&P, sizeof(P));
  P.subdir = "url";
  P.BASE_SIG = UDM_URL_SIG;
  P.basename = "";
  P.indname = "rec_id";
  P.rec_id = url_id;

  if (UdmOpenBase(A, &P, UDM_WRITE_LOCK) != UDM_OK) {
    return UDM_ERROR;
  }

  if (P.Item.rec_id == url_id) {
    NextItemPos = P.Item.next;
    while(NextItemPos != 0) {
      if (fseek(P.Ifd, NextItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fread(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	UdmLog(A, UDM_LOG_ERROR, "Can't read hash chain for file %s", P.Ifilename);
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
      P.CurrentItemPos = NextItemPos;
      NextItemPos = P.Item.next;
      P.Item.next = P.CurrentItemPos;
      if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
	UdmLog(A, UDM_LOG_ERROR, "Can't write hash chain for file %s", P.Ifilename);
	UdmCloseBase(&P);
	return UDM_ERROR;
      }
    }
    P.Item.rec_id = 0;
    P.Item.offset = 0;
    P.Item.next = 0;
    if (fseek(P.Ifd, P.CurrentItemPos, SEEK_SET)) {
	UdmCloseBase(&P);
      return UDM_ERROR;
    }
    if (fwrite(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) != 1) {
      UdmLog(A, UDM_LOG_ERROR, "Can't write hash chain for file %s", P.Ifilename);
      UdmCloseBase(&P);
      return UDM_ERROR;
    }
  }

  if ((res = UdmCloseBase(&P)) != UDM_OK) return UDM_ERROR;

  UdmStoreDeleteDoc(A, Doc);

  UdmCacheLinksDelOt(A, url_id);

/* to be implemented 

        UdmCacheLinksDelK(A, url_id);

	if (UDM_OK != (rc=UdmDeleteBadHrefs(Indexer,Doc, db))) return rc;

	sprintf(qbuf,"UPDATE url SET referrer = 0 WHERE referrer=%d", url_id);
*/
  return UDM_OK;
}

int __UDMCALL UdmURLActionCache(UDM_AGENT *A, UDM_DOCUMENT *D, int cmd, UDM_DB *db) {
	int res;

	switch(cmd){
		case UDM_URL_ACTION_ADD:
			res = UdmAddURLCache(A, D, db);
			break;
			
		case UDM_URL_ACTION_DELETE:
			res = UdmDeleteURLCache(A, D, db);
			break;
/*			
		case UDM_URL_ACTION_SUPDATE:
			res=UdmUpdateUrlCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_LUPDATE:
			res=UdmLongUpdateURLCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_INSWORDS:
			res=UdmStoreWordsCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_INSCWORDS:
			res = UDM_OK; /UdmStoreCrossWords(A,D,db);/
			break;
			
		case UDM_URL_ACTION_DELWORDS:
			res=UdmDeleteWordFromURLCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_DELCWORDS:
			res = UDM_OK; /UdmDeleteCrossWordFromURL(A,D,db);/
			break;
			
		case UDM_URL_ACTION_UPDCLONE:
			res=UdmUpdateCloneCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_REGCHILD:
			res=UdmRegisterChildCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDBYURL:
			res=UdmFindURLCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDBYMSG:
			res=UdmFindMessageCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDORIG:
			res=UdmFindOriginCache(A,D,db);
			break;
			
		case UDM_URL_ACTION_EXPIRE:
			res=UdmMarkForReindexCache(A,db);
			break;
			
		case UDM_URL_ACTION_REFERERS:
			res=UdmGetReferersCache(A,db);
			break;
		
		case UDM_URL_ACTION_DOCCOUNT:
			res=UdmGetDocCountCache(A,db);
			break;
		
	        case UDM_URL_ACTION_WRITEDATA:
			if (db->DBMode == UDM_DBMODE_CACHE) {
			  return UdmURLDataWrite(A);
			}
		        return UDM_OK;

		case UDM_URL_ACTION_LINKS_DELETE:
			res = UdmDeleteLinksCache(A, D, db);
			break;
*/			
		default:
  		        UdmLog(A, UDM_LOG_ERROR, "Unsupported URL Action SQL");
			res=UDM_ERROR;
	}

	return res;
}

int UdmStatActionCache(UDM_AGENT *A, UDM_STATLIST *S, UDM_DB *db) { /* FIXME: need to implemet */
  return UDM_OK;
}
