/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_IO_H   /* for Win */
#include <io.h>
#endif
#ifdef HAVE_PROCESS_H
#include <process.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef   HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#endif
#ifdef HAVE_NETINET_IP_H
#include <netinet/ip.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


#include "udm_common.h"
#include "udm_cache.h"
#include "udm_utils.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_services.h"
#include "udm_log.h"
#include "udm_env.h"
#include "udm_conf.h"
#include "udm_agent.h"
#include "udm_mkind.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_signals.h"

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

#define DEBUG_LOGD 1
/*#define DEBUG_LOGD_CMD 1*/

#define SPLDIR  "splitter"

#define STATE_CMD 1
#define STATE_WRD 2
#define UDM_CACHED_MAXCLNTS 128

/************************ type defenitions ************************/ 
typedef struct {
	int  fd;	/* Socket descriptor    */
	UDM_LOGD_CMD cmd;
        UDM_LOGD_WRD *wrd;
} UDM_LOGD_CL;

#ifdef DEBUG_LOGD_CMD
static void printcmd(UDM_LOGD_CMD *c){
	fprintf(stderr, "cmd=%d nwords=%d stamp=%d url_id=%d\n", c->cmd, c->nwords, (int)c->stamp, c->url_id);
}
#endif

static void init_client(UDM_LOGD_CL * client) {
        client->fd = 0;
	client->wrd = NULL;
}

static void deinit_client(UDM_LOGD_CL * client) {
        UDM_FREE(client->wrd);
}

/********************* SIG Handlers ****************/

static void sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
#ifdef WIN32
#else
	UdmSignal(SIGPIPE,sighandler);
	UdmSignal(SIGCHLD, sighandler);
	UdmSignal(SIGALRM, sighandler);
#endif
	UdmSignal(SIGHUP,sighandler);
	UdmSignal(SIGINT,sighandler);
	UdmSignal(SIGTERM,sighandler);
}

static void sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif

	switch(sign){
		case SIGPIPE:
		  have_sigpipe = 1;
		  break;
		case SIGHUP:
		  have_sighup = 1;
		  break;
	        case SIGCHLD:
		  while (waitpid(-1, &status, WNOHANG) > 0);
		  break;
		case SIGINT:
		  have_sigint = 1;
		  break;
		case SIGTERM:
		  have_sigterm = 1;
		  break;
	        case SIGALRM:
		  _exit(0);
		  break;
		default: ;
	}
}

/*************************************************************/

static char pid_name[1024];
static char time_pid[1024];

static char *Logd_time_pid_info(void) {
	time_t t=time(NULL);
	struct tm *tim=localtime(&t);
	
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

static void RotateDelLog(UDM_ENV *Conf) {
  UDM_DB *db;
  UDM_LOGD *logd;
  char del_log_name[1024];
  int split_fd, nbytes;
  size_t i, dbfrom = 0, dbto =  Conf->dbl.nitems;

  for (i = dbfrom; i < dbto; i++) {
    db = &Conf->dbl.db[i];
    logd = db->LOGD;

    udm_snprintf(del_log_name, 1024, "%s%s", db->log_dir, "del-split.log");

    if((split_fd = open(del_log_name, O_WRONLY | O_CREAT | UDM_BINARY, 0600)) == -1) {
      sprintf(db->errstr, "Can't open '%s' for writing: %s\n", del_log_name, strerror(errno));
      UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s %s", Logd_time_pid_info(), db->errstr);
      return;
    }

    UdmWriteLock(db->del_fd);
  
    lseek(db->del_fd, (off_t)0, SEEK_SET);
    while((nbytes = read(db->del_fd, del_log_name, 1024)) > 0) {
      write(split_fd, del_log_name, (size_t)nbytes);
    }
    close(split_fd);
    lseek(db->del_fd, (off_t)0, SEEK_SET);
    ftruncate(db->del_fd, (off_t)0);
  
    UdmUnLock(db->del_fd);
  }
}

static void FlushAllBufs(UDM_AGENT *Agent) {
  UDM_DB *db;
  size_t i, dbfrom = 0, dbto;

  UDM_GETLOCK(Agent, UDM_LOCK_CONF);
  dbto =  Agent->Conf->dbl.nitems;
  UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);

  UdmLog(Agent, UDM_LOG_INFO, "%s Flushing all buffers... ", Logd_time_pid_info());
  if (Agent->Conf->logs_only) RotateDelLog(Agent->Conf);

  if(UDM_OK != UdmLogdSaveAllBufs(Agent)) {
    for (i = dbfrom; i < dbto; i++) {
      UDM_GETLOCK(Agent, UDM_LOCK_DB);
      db = &Agent->Conf->dbl.db[i];
      if (db->errcode) UdmLog(Agent, UDM_LOG_ERROR, "%s Error: %s", Logd_time_pid_info(), db->errstr);
      UDM_RELEASELOCK(Agent, UDM_LOCK_DB);
    }
    UdmLog(Agent, UDM_LOG_ERROR,"%s Shutdown", Logd_time_pid_info());
  }
  UdmLog(Agent, UDM_LOG_INFO, "Done");

}

static int client_action(UDM_AGENT *Agent, UDM_LOGD_CL * client){

        size_t i;

#ifdef DEBUG_LOGD_CMD
	printcmd(&client->cmd);
#endif

	if (client->cmd.nwords > 0) {

	  client->wrd = (UDM_LOGD_WRD*)realloc(client->wrd, client->cmd.nwords * sizeof(UDM_LOGD_WRD));
	  if (client->wrd == NULL) {
	    if (UdmSend(client->fd, "F", 1, 0) != 1) return UDM_ERROR;
	    return UDM_ERROR;
	  }
	  if (UdmSend(client->fd, "O", 1, 0) != 1) return UDM_ERROR;
	  for (i = 0; i < client->cmd.nwords; i++) {
	    if (UdmRecvall(client->fd, &client->wrd[i], sizeof(UDM_LOGD_WRD)) != sizeof(UDM_LOGD_WRD)) return UDM_ERROR;
/*	    if (UdmSend(client->fd, "O", 1, 0) != 1) return UDM_ERROR;*/
	  }
	}
	if (client->cmd.cmd == UDM_LOGD_CMD_CHECK) {
	  UdmCachedCheck(Agent);
	} else
	if (client->cmd.cmd == UDM_LOGD_CMD_DATA) {
	  UdmURLAction(Agent, NULL, UDM_URL_ACTION_WRITEDATA);
	} else {
	  register size_t dbnum;
	  register int rc;
	  UDM_DB *db;
	  UDM_GETLOCK(Agent, UDM_LOCK_CONF);
	  dbnum = client->cmd.url_id % Agent->Conf->dbl.nitems;
	  db = &Agent->Conf->dbl.db[dbnum];
	  UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);
	  UDM_GETLOCK(Agent, UDM_LOCK_DB);
	  rc = UdmLogdStoreDoc(Agent, client->cmd, client->wrd, db);
	  UDM_RELEASELOCK(Agent, UDM_LOCK_DB);
	  if(rc != UDM_OK) return rc;
	}

	if (UdmSend(client->fd, "O", 1, 0) != 1) return UDM_ERROR;

	return UDM_OK;
}

static void usage(void){

	fprintf(stderr,
"\n\
cached from %s-%s-%s\n\
http://www.mnogosearch.org/ (C) 1998-2002, LavTech Corp.\n\
\n\
Usage: cached [OPTIONS] [configfile]\n\
\n\
Options are:\n\
  -w /path      choose alternative working /var directory\n\
  -p xxx        listen port xxx\n\
  -v n          verbose level, 0-5\n\
  -l            write logs only mode\n\
  -s k          sleep k sec. at start-up\n\
  -h,-?         print this help page and exit\n\
\n\
Please post bug reports and suggestions to http://www.mnogosearch.org/bugs/\n",
	PACKAGE,VERSION,UDM_DBTYPE);

	return;
}

static void exitproc(void){
#ifdef WIN32
	WSACleanup();
#endif
}

typedef struct {
  UDM_AGENT *Agent;
  int fd;
} UDM_CACHED_CFG;

static int next_thread = 20;

#ifdef  WIN32
DWORD WINAPI thread_child(void *arg){
#else
static void * thread_child(void *arg){
#endif

  UDM_CACHED_CFG *C = (UDM_CACHED_CFG*)arg;
  int lbytes;
  UDM_LOGD_CL cl;
  UDM_AGENT *Indexer = C->Agent;

  pthread_detach(pthread_self());

  init_client(&cl);
  cl.fd = C->fd;

  UdmLog(Indexer, UDM_LOG_INFO, "%s Client thread started", Logd_time_pid_info());


  while (1) {
#ifdef DEBUG_LOGD
    UdmLog(Indexer, UDM_LOG_DEBUG, "%s try read cmd (%d bytes)", Logd_time_pid_info(), sizeof(UDM_LOGD_CMD));
#endif
    lbytes = UdmRecvall(cl.fd, &cl.cmd, sizeof(UDM_LOGD_CMD));

    if (lbytes == sizeof(UDM_LOGD_CMD)) {
      if (client_action(Indexer, &cl) != UDM_OK) {
	UdmLog(Indexer, UDM_LOG_INFO, "%s Client action failed", Logd_time_pid_info());
	deinit_client(&cl);
	init_client(&cl);
      }
    } else {
      UdmLog(Indexer, UDM_LOG_INFO, "%s Client left", Logd_time_pid_info());
      close (cl.fd);
      deinit_client(&cl);
      init_client(&cl);
      break;
    }
  }
  UdmAgentFree(Indexer);
  return NULL;
}

int main(int argc,char **argv){
	struct	sockaddr_in server_addr;
	struct	sockaddr_in his_addr;
	struct	in_addr bind_address;
	const char	*var_dir = NULL;
	int	on = 1;
	int	ctl_sock, sel;
	struct timeval tval;
	int	pid_fd,ch,port=UDM_LOGD_PORT;
	char	pidbuf[1024];
	UDM_LOGD_CL cl;
	UDM_ENV * Conf = NULL;
	UDM_AGENT *Agent;
	int log2stderr = 1;
	const char * config_name = UDM_CONF_DIR "/cached.conf";
	fd_set	mask;
	
	init_client(&cl);
		
	UdmInit(); /* Initialize library */
	
	UdmInitMutexes();
	Conf = UdmEnvInit(NULL);
	UdmSetLockProc(Conf, UdmLockProc);
		
	while ((ch = getopt(argc, argv, "hlv:w:p:s:?")) != -1){
		switch (ch) {
			case 'p':
				port=atoi(optarg);
				break;
		        case 'v': UdmSetLogLevel(NULL, atoi(optarg)); break;
			case 'w':
				var_dir = optarg;
				break;
		        case 'l':
			        Conf->logs_only = 1;
				break;
		        case 's':
			        sleep((unsigned int)atoi(optarg));
				break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;
	
	if(argc > 1){
		usage();
		goto err1;
	} else if (argc == 1) {
	        config_name = argv[0];
	}
	
	Agent=UdmAgentInit(NULL, Conf, 0);
	if(UDM_OK != UdmEnvLoad(Agent, config_name, 0)){
		fprintf(stderr, "%s\n", UdmEnvErrMsg(Conf));
		UdmEnvFree(Conf);
		return(1);
	}
		
	UdmOpenLog("cached", Conf, log2stderr);

	if (var_dir == NULL) var_dir = UdmVarListFindStr(&Conf->Vars, "VarDir", UDM_VAR_DIR);

	/* Check that another instance isn't running and create PID file */
	udm_snprintf(pid_name, 1024, "%s%s%s", var_dir, UDMSLASHSTR, "cached.pid");
	pid_fd=open(pid_name, O_CREAT|O_EXCL|O_WRONLY, 0644);
	if(pid_fd<0){
		UdmLog(Agent, UDM_LOG_ERROR, "%s Can't create '%s': %s", Logd_time_pid_info(), pid_name, strerror(errno));
		if(errno==EEXIST){
			UdmLog(Agent, UDM_LOG_ERROR, 
				       "It seems that another cached is already running!\nRemove '%s' if it is not true.", pid_name);
		}
		goto err2;
	}
	sprintf(pidbuf,"%d\n",(int)getpid());
	write(pid_fd,&pidbuf,strlen(pidbuf));
	close(pid_fd);
	
#ifdef WIN32
	{
		WSADATA wsaData;

		if(WSAStartup(0x101,&wsaData)!=0){
			UdmLog(Agent, UDM_LOG_ERROR, "%s WSAStartup() error %d", Logd_time_pid_info(&LOGD), WSAGetLastError);
			goto err1;
		}
	}
#endif
	/* Initialize variables */

	UdmLog(Agent, UDM_LOG_INFO, "Starting in %s mode", (Conf->logs_only) ? "old" : "new");
	
	if(UDM_OK != UdmOpenCache(Agent, 0)) {
		UdmLog(Agent, UDM_LOG_ERROR, "Error: %s", Conf->errstr);
		goto err1;
	}

	atexit(&exitproc);
	init_signals();
	
	UdmLog(Agent, UDM_LOG_INFO, "%s Started on port %d. Accepting %d connections", Logd_time_pid_info(), port, FD_SETSIZE);
	
	ctl_sock = socket(AF_INET, SOCK_STREAM, 0);
	if (ctl_sock < 0) {
		UdmLog(Agent, UDM_LOG_ERROR , "%s socket() error %d", Logd_time_pid_info(), errno);
		goto err1;
	}
	if (setsockopt(ctl_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
		UdmLog(Agent, UDM_LOG_ERROR, "%s setsockopt() error %d", Logd_time_pid_info(), errno);
		goto err1;
	}
	
	
	/* Prepare to start TCP server */
	
	bind_address.s_addr 	= htonl(INADDR_ANY);
	server_addr.sin_family	= AF_INET;
	server_addr.sin_addr	= bind_address;
	server_addr.sin_port	= ntohs(port);
	
	if (bind(ctl_sock, (struct sockaddr *)&server_addr, sizeof(server_addr))) {
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s bind() error %d %s", Logd_time_pid_info(), errno, strerror(errno));
		goto err1;
	}
	if (listen(ctl_sock, UDM_CACHED_MAXCLNTS) < 0) {
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s listen() error %d %s", Logd_time_pid_info(), errno, strerror(errno));
		goto err1;
	}

	
	while (1) {
#ifdef __irix__
 	        int addrlen;
#else
		socklen_t addrlen;
#endif
		int pid;
		
		if (have_sighup) {
		  UdmLog_noagent(Conf, UDM_LOG_INFO, "%s SIGHUP arrived. Flushing buffers and making indexes.", Logd_time_pid_info());
		  FlushAllBufs(Agent);
		  UdmURLAction(Agent,  NULL, UDM_URL_ACTION_WRITEDATA);
		  /*UdmURLDataWrite(Agent);*/
		  have_sighup = 0;
		}
		if(have_sigpipe){
			UdmLog(Agent, UDM_LOG_ERROR, "%s FATAL: SIGPIPE arrived. Broken pipe !", Logd_time_pid_info());
			have_sigpipe = 0;
			break;
		}
		if(have_sigint){
			UdmLog(Agent, UDM_LOG_INFO, "%s SIGINT arrived. Shutdown.", Logd_time_pid_info());
			FlushAllBufs(Agent);
			UdmURLAction(Agent, NULL, UDM_URL_ACTION_WRITEDATA);
			have_sigint = 0;
			break;
		}
		if(have_sigterm){
			UdmLog(Agent, UDM_LOG_INFO, "SIGTERM arrived. Shutdown.");
			FlushAllBufs(Agent);
			UdmURLAction(Agent, NULL, UDM_URL_ACTION_WRITEDATA);
			have_sigterm = 0;
			break;
		}
		
		
		tval.tv_sec = 300;
		tval.tv_usec = 0;
		FD_ZERO(&mask);
		FD_SET(ctl_sock, &mask);

		sel = select(FD_SETSIZE, &mask, 0, 0, &tval);

		if(sel==0){
			/* Time limit expired */
 			continue;
		}
		if(sel<1){
			/* FIXME: add errno checking */
			continue;
		}


		if (FD_ISSET(ctl_sock, &mask)) {
		  UDM_CACHED_CFG C;

		  bzero((void*)&his_addr, addrlen = sizeof(his_addr));
		  cl.fd = accept(ctl_sock, (struct sockaddr *)&his_addr, &addrlen);
		  if (cl.fd <= 0) {
		    continue;
		  }

#define Client inet_ntoa(his_addr.sin_addr)
		  UdmLog(Agent, UDM_LOG_INFO, "[%s] Connected", Client);

		  {
		    UDM_AGENT *Indexer;
		  
		    UDM_GETLOCK(Agent, UDM_LOCK_THREAD);
		    UDM_GETLOCK(Agent, UDM_LOCK_CONF);
		    Indexer = UdmAgentInit(NULL, Agent->Conf, next_thread++);
		    if (next_thread == 50) next_thread = 20;
		    UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);
		    UDM_RELEASELOCK(Agent, UDM_LOCK_THREAD);

#ifdef HAVE_PTHREAD  
#ifdef WIN32
		    CreateThread(NULL, 0, &thread_child, NULL, 0, NULL); /* FIXME: */
#else
		    {
		      pthread_attr_t attr;
		      size_t stksize=1024*256;
		      pthread_t tid;
		  
		      C.Agent = Indexer;
		      C.fd = cl.fd;

		      pthread_attr_init(&attr);
		      pthread_attr_setstacksize(&attr, stksize);
		      pthread_create(&tid, &attr, &thread_child, &C);
		      pthread_attr_destroy(&attr);
		    }
#endif
#else
		    C.Agent = Indexer;
		    C.fd = cl.fd;
		    thread_child(&C);
#endif
		    UDMSLEEP(0);
		  }
		}
		
	}
err0:
	UdmCloseCache(Agent, 0);
	UdmAgentFree(Agent);
	UdmEnvFree(Conf);
	UdmDestroyMutexes();
	deinit_client(&cl);
	unlink(pid_name);
	exit(0);
err1:
	unlink(pid_name);
err2:
	UdmCloseCache(Agent, 0);
	UdmAgentFree(Agent);
	UdmEnvFree(Conf);
	UdmDestroyMutexes();
	deinit_client(&cl);
	exit(1);

}
