/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>

#include "udm_common.h"
#include "udm_spell.h"
#include "udm_proto.h"
#include "udm_url.h"
#include "udm_parser.h"
#include "udm_conf.h"
#include "udm_log.h"
#include "udm_hrefs.h"
#include "udm_robots.h"
#include "udm_utils.h"
#include "udm_host.h"
#include "udm_server.h"
#include "udm_alias.h"
#include "udm_search_tl.h"
#include "udm_env.h"
#include "udm_match.h"
#include "udm_stopwords.h"
#include "udm_guesser.h"
#include "udm_unicode.h"
#include "udm_synonym.h"
#include "udm_vars.h"
#include "udm_db.h"
#include "udm_agent.h"
#include "udm_db_int.h"
#include "udm_chinese.h"

static int EnvLoad(UDM_CFG *Cfg,const char *cname);

/****************************  Load Configuration **********************/

int UdmSearchMode(const char * mode){
	if(!mode)return(UDM_MODE_ALL);
	if(!strcmp(mode,"all"))return(UDM_MODE_ALL);
	if(!strcmp(mode,"any"))return(UDM_MODE_ANY);
	if(!strcmp(mode,"bool"))return(UDM_MODE_BOOL);
	if(!strcmp(mode,"phrase"))return(UDM_MODE_PHRASE);
	return(UDM_MODE_ALL);
}


int UdmMatchMode(const char * mode){
	if(!mode)return(UDM_MATCH_FULL);
	if(!strcmp(mode,"wrd"))return(UDM_MATCH_FULL);
	if(!strcmp(mode,"full"))return(UDM_MATCH_FULL);
	if(!strcmp(mode,"beg"))return(UDM_MATCH_BEGIN);
	if(!strcmp(mode,"end"))return(UDM_MATCH_END);
	if(!strcmp(mode,"sub"))return(UDM_MATCH_SUBSTR);
	return(UDM_MATCH_FULL);
}

__C_LINK const char * __UDMCALL UdmFollowStr(int method) {
	switch(method){
		case UDM_FOLLOW_NO:		return "Page";
		case UDM_FOLLOW_PATH:		return "Path";
		case UDM_FOLLOW_SITE:		return "Site";
		case UDM_FOLLOW_WORLD:		return "World";
	}
	return "<Unknown follow type>";
}


int UdmFollowType(const char * follow){
	if(!follow)return UDM_FOLLOW_UNKNOWN;
	if(!strcasecmp(follow,"no"))return(UDM_FOLLOW_NO);
	if(!strcasecmp(follow,"page"))return(UDM_FOLLOW_NO);
	if(!strcasecmp(follow,"yes"))return(UDM_FOLLOW_PATH);
	if(!strcasecmp(follow,"path"))return(UDM_FOLLOW_PATH);
	if(!strcasecmp(follow,"site"))return(UDM_FOLLOW_SITE);
	if(!strcasecmp(follow,"world"))return(UDM_FOLLOW_WORLD);
	return(UDM_FOLLOW_UNKNOWN);
}

const char *UdmMethodStr(int method){
	switch(method){
		case UDM_METHOD_DISALLOW:	return "Disallow";
		case UDM_METHOD_GET:		return "Allow";
		case UDM_METHOD_CHECKMP3ONLY:	return "CheckMP3Only";
		case UDM_METHOD_CHECKMP3:	return "CheckMP3";
		case UDM_METHOD_HEAD:		return "CheckOnly";
		case UDM_METHOD_HREFONLY:	return "HrefOnly";
		case UDM_METHOD_VISITLATER:	return "Skip";
	}
	return "<Unknown method>";
}

int UdmMethod(const char *s){
	if(!strcasecmp(s,"Disallow"))		return UDM_METHOD_DISALLOW;
	if(!strcasecmp(s,"Allow"))		return UDM_METHOD_GET;
	if(!strcasecmp(s,"CheckMP3Only"))	return UDM_METHOD_CHECKMP3ONLY;
	if(!strcasecmp(s,"CheckMP3"))		return UDM_METHOD_CHECKMP3;
	if(!strcasecmp(s,"CheckOnly"))		return UDM_METHOD_HEAD;
	if(!strcasecmp(s,"HrefOnly"))		return UDM_METHOD_HREFONLY;
	if(!strcasecmp(s,"Skip"))		return UDM_METHOD_VISITLATER;
	return UDM_METHOD_UNKNOWN;
}



void UdmWeightFactorsInit(const char *wf, int *res){
	size_t len;
	int sn;
	
	for(sn=0;sn<256;sn++){
		res[sn]=1;
	}
	
	len=strlen(wf);
	if((len>0)&&(len<256)){
		const char *sec;
			
		for(sec=wf+len-1;sec>=wf;sec--){
			res[len-(sec-wf)]=UdmHex2Int(*sec);
		}
	}
}

static size_t rel_etc_name(UDM_ENV *Env,char *res,size_t maxlen,const char *name){
	size_t		n;
	const char	*dir=UdmVarListFindStr(&Env->Vars,"ConfDir",UDM_CONF_DIR);
	if(name[0]=='/')n = udm_snprintf(res,maxlen,name);
	else		n = udm_snprintf(res,maxlen,"%s%s%s",dir,UDMSLASHSTR,name);
	res[maxlen]='\0';
	return n;
}

static size_t rel_var_name(UDM_ENV *Env,char *res,size_t maxlen,const char *name){
	size_t		n;
	const char	*dir=UdmVarListFindStr(&Env->Vars,"VarDir",UDM_VAR_DIR);
	if(name[0]=='/')n = udm_snprintf(res,maxlen,name);
	else		n = udm_snprintf(res,maxlen,"%s%s%s",dir,UDMSLASHSTR,name);
	res[maxlen]='\0';
	return n;
}

size_t UdmGetArgs(char *str, char **av, size_t max){
	size_t	ac=0;
	char	*lt;
	char	*tok;
	
	bzero((void*)av, max * sizeof(*av));
	tok=UdmGetStrToken(str,&lt);
	
	while (tok && (ac<max)){
		av[ac]=tok;
		ac++;
		tok=UdmGetStrToken(NULL,&lt);
	}
	return ac;
}


static int add_srv(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	UDM_AGENT *Indexer = C->Indexer;
	size_t	i;
	int	has_alias=0;
	
	if(!(C->flags & UDM_FLAG_ADD_SERV))
		return UDM_OK;
	
	C->Srv->command = 'S';
	C->Srv->ordre = ++C->ordre;
	C->Srv->Match.nomatch=0;
	C->Srv->Match.case_sense = 1;
	UdmVarListReplaceStr(&C->Srv->Vars, "Method", "Allow");

	if(!strcasecmp(av[0],"Server")){
		C->Srv->Match.match_type=UDM_MATCH_BEGIN;
	}else
	if(!strcasecmp(av[0],"Subnet")){
		C->Srv->Match.match_type=UDM_MATCH_SUBNET;
		Conf->Servers.have_subnets=1;
	}else{
		C->Srv->Match.match_type=UDM_MATCH_WILD;
	}
	
	UdmVarListReplaceInt(&C->Srv->Vars,"Follow",UDM_FOLLOW_PATH);
	
	for(i=1; i<ac; i++){
		int	o;
		
		if(UDM_FOLLOW_UNKNOWN!=(o=UdmFollowType(av[i])))UdmVarListReplaceInt(&C->Srv->Vars,"Follow",o);
		else
		if(UDM_METHOD_UNKNOWN!=(o=UdmMethod(av[i])))UdmVarListReplaceStr(&C->Srv->Vars,"Method",av[i]);
		else
		if(!strcasecmp(av[i],"nocase"))C->Srv->Match.case_sense=0;
		else
		if(!strcasecmp(av[i],"case"))C->Srv->Match.case_sense=1;
		else
		if(!strcasecmp(av[i],"match"))C->Srv->Match.nomatch=0;
		else
		if(!strcasecmp(av[i],"nomatch"))C->Srv->Match.nomatch=1;
		else
		if(!strcasecmp(av[i],"string"))C->Srv->Match.match_type=UDM_MATCH_WILD;
		else
		if(!strcasecmp(av[i],"regex"))C->Srv->Match.match_type=UDM_MATCH_REGEX;
		else{
			if(!C->Srv->Match.pattern)
				C->Srv->Match.pattern = (char*)strdup(av[i]);
			else
			if(!has_alias){
				has_alias=1;
				UdmVarListReplaceStr(&C->Srv->Vars,"Alias",av[i]);
			}else{
				sprintf(Conf->errstr,"too many argiments: '%s'",av[i]);
				return UDM_ERROR;
			}
		}
	}
	if(UDM_OK != UdmServerAdd(Indexer, C->Srv)) {
		char * s_err;
		s_err = (char*)strdup(Conf->errstr);
		sprintf(Conf->errstr,"%s",s_err);
		UDM_FREE(s_err);
		UDM_FREE(C->Srv->Match.pattern);
		return UDM_ERROR;
	}
	if((C->Srv->Match.match_type==UDM_MATCH_BEGIN)&&(C->Srv->Match.pattern[0])&&(C->flags&UDM_FLAG_ADD_SERVURL)) {
		UDM_HREF Href;
		
		bzero((void*)&Href, sizeof(Href));
		Href.url=C->Srv->Match.pattern;
		Href.method=UDM_METHOD_GET;
		Href.site_id = C->Srv->site_id;
		Href.server_id = C->Srv->site_id;
		UdmHrefListAdd(&Indexer->Hrefs, &Href);
	}
	UDM_FREE(C->Srv->Match.pattern);
	UdmVarListDel(&C->Srv->Vars,"AuthBasic");
	UdmVarListDel(&C->Srv->Vars,"Alias");
	return UDM_OK;
}

static int add_alias(void *Cfg, size_t ac,char **av){
	UDM_CFG		*C=(UDM_CFG*)Cfg;
	UDM_ENV		*Conf = C->Indexer->Conf;
	UDM_MATCH	Alias;
	size_t		i;
	
	UdmMatchInit(&Alias);
	Alias.match_type=UDM_MATCH_BEGIN;
	
	for(i=1; i<ac; i++){
		if(!strcasecmp(av[i],"regex"))
			Alias.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"regexp"))
			Alias.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"case"))
			Alias.case_sense=1;
		else
		if(!strcasecmp(av[i],"nocase"))
			Alias.case_sense=0;
		else
		if(!Alias.pattern){
			Alias.pattern=av[i];
		}else{
			char		err[120]="";
			UDM_MATCHLIST	*L=NULL;
			
			Alias.arg=av[i];
			
			if(!strcasecmp(av[0],"Alias"))L=&Conf->Aliases;
			if(!strcasecmp(av[0],"ReverseAlias"))L=&Conf->ReverseAliases;
			
			if(UDM_OK!=UdmMatchListAdd(L,&Alias,err,sizeof(err))){
				udm_snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"%s",err);
				return UDM_ERROR;
			}
		}
	}
	if(!Alias.arg){
		udm_snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"too few arguments");
		return UDM_ERROR;
	}
	return UDM_OK;
}


static int add_filter(void *Cfg, size_t ac,char **av){
	UDM_CFG		*C=(UDM_CFG*)Cfg;
	UDM_ENV		*Conf=C->Indexer->Conf;
	UDM_MATCH	M;
	size_t		i;
	
	UdmMatchInit(&M);
	M.match_type=UDM_MATCH_WILD;
	M.case_sense=1;
	
	C->ordre++;
	for(i=1; i<ac ; i++){
		if(!strcasecmp(av[i],"case"))M.case_sense=1;
		else
		if(!strcasecmp(av[i],"nocase"))M.case_sense=0;
		else
		if(!strcasecmp(av[i],"regex"))M.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"regexp"))M.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"string"))M.match_type=UDM_MATCH_WILD;
		else
		if(!strcasecmp(av[i],"nomatch"))M.nomatch=1;
		else
		if(!strcasecmp(av[i],"match"))M.nomatch=0;
		else{
			char		err[120]="";
			
			M.arg=av[0];
			M.pattern=av[i];
			
			if(UDM_OK!=UdmMatchListAdd(&Conf->Filters,&M,err,sizeof(err))){
				udm_snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"%s",err);
				return UDM_ERROR;
			}
		}
	}
	return UDM_OK;
}

static int add_type(void *Cfg, size_t ac,char **av){
	UDM_CFG		*C=(UDM_CFG*)Cfg;
	UDM_ENV		*Conf=C->Indexer->Conf;
	UDM_MATCH	M;
	size_t		i;
	int		rc=UDM_OK;
	char err[128];
	
	UdmMatchInit(&M);
	M.match_type=UDM_MATCH_WILD;
	
	for (i=1; i<ac; i++){
		if(!strcasecmp(av[i],"regex"))M.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"regexp"))M.match_type=UDM_MATCH_REGEX;
		else
		if(!strcasecmp(av[i],"string"))M.match_type=UDM_MATCH_WILD;
		else
		if(!strcasecmp(av[i],"case"))M.case_sense=1;
		else
		if(!strcasecmp(av[i],"nocase"))M.case_sense=0;
		else
		if(!M.arg)
			M.arg=av[i];
		else{
			M.pattern=av[i];
			if(UDM_OK != (rc=UdmMatchListAdd(&Conf->MimeTypes,&M,err,sizeof(err)))){
				udm_snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"%s",err);
				return rc;
			}
		}
	}
	return rc;
}

static int add_parser(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	UDM_PARSER P;
	P.from_mime=av[1];
	P.to_mime=av[2];
	P.cmd = UDM_NULL2EMPTY(av[3]);
	UdmParserAdd(&Conf->Parsers,&P);
	return UDM_OK;
}


static int add_section(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	UDM_VAR S;
	bzero((void*)&S, sizeof(S));
	S.name=av[1];
	S.section=atoi(av[2]);
	S.maxlen=av[3]?atoi(av[3]):0;
	UdmVarListReplace(&Conf->Sections,&S);
	return UDM_OK;
}


static int do_include(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	if(C->level<5){
		int	rc;
		char	fname[1024];
		rel_etc_name(C->Indexer->Conf, fname, sizeof(fname)-1, av[1]);
		C->level++;
		rc=EnvLoad(C,fname);
		C->level--;
		return rc;
	}else{
		sprintf(C->Indexer->Conf->errstr,"too big (%d) level in included files",C->level);
		return UDM_ERROR;
	}
	return UDM_OK;
}

static int add_affix(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	
	if(C->flags&UDM_FLAG_SPELL){
		char	fname[1024];
		rel_etc_name(Conf,fname,sizeof(fname)-1,av[3]);
		if(UdmImportAffixes(Conf,av[1],av[2],fname)){
			sprintf(Conf->errstr,"Can't load affix :%s",fname);
			return UDM_ERROR;
		}
	}
	return UDM_OK;
}

static int add_spell(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	
	if(C->flags&UDM_FLAG_SPELL){
		char	fname[1024];
		rel_etc_name(Conf,fname,sizeof(fname)-1,av[3]);
		if(UdmImportDictionary(Conf,av[1],av[2],fname,0,"")){
		        sprintf(Conf->errstr,"Can't load dictionary :%s",fname);
			return UDM_ERROR;
		}
	}
	return UDM_OK;
}

static int add_stoplist(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res;
	char	fname[1024];
	rel_etc_name(Conf,fname,sizeof(fname)-1,av[1]);
	res=UdmStopListLoad(Conf,fname);
	return res;
}

static int add_langmap(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res=UDM_OK;
	if(C->flags&UDM_FLAG_LOAD_LANGMAP){
		char	fname[1024];
		rel_etc_name(Conf,fname,sizeof(fname)-1,av[1]);
		res=UdmLoadLangMapFile(Conf,fname);
	}
	return res;
}

static int add_synonym(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res=UDM_OK;
	if(C->flags&UDM_FLAG_SPELL){
		char	fname[1024];
		rel_etc_name(Conf,fname,sizeof(fname)-1,av[1]);
		res=UdmSynonymListLoad(Conf,fname);
	}
	return res;
}

static int add_chinese(void *Cfg, size_t ac,char **av){
#ifdef HAVE_CHARSET_gb2312
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	if(C->flags & UDM_FLAG_ADD_SERV){
		return UdmChineseListLoad(&Conf->Chi, (av[1] != NULL) ? av[1] : UDM_CONF_DIR "/mandarin.freq" );
	}
#endif
	return UDM_OK;
}
static int add_thai(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	if(C->flags & UDM_FLAG_ADD_SERV){
		return UdmChineseListLoad(&Conf->Thai, (av[1] != NULL) ? av[1] : UDM_CONF_DIR "/thai.freq" );
	}
	return UDM_OK;
}
				

static int add_url(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	UDM_AGENT *Indexer = C->Indexer;
	
	if(C->flags&UDM_FLAG_ADD_SERV){
		char		*al = NULL;
		UDM_SERVER	*Srv;
		UDM_HREF	Href;
		if((Srv = UdmServerFind(Conf, &Conf->Servers, av[1], &al))) {
			UdmHrefInit(&Href);
			Href.url=av[1];
			Href.method=UDM_METHOD_GET;
			UdmHrefListAdd(&Indexer->Hrefs, &Href);
		}
		UDM_FREE(al);
	}
	return UDM_OK;
}
		

static int add_srv_table(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int		res = UDM_OK;
	UDM_AGENT	Agent;
	UDM_DBLIST      dbl;
	UDM_DB		*db;
	
	UdmDBListInit(&dbl);
	UdmDBListAdd(&dbl, av[1], UDM_OPEN_MODE_READ);
	db = &dbl.db[0];
	UdmAgentInit(&Agent,Conf,0);
	Agent.flags = C->flags;

#ifdef HAVE_SQL
	res = UdmSrvActionSQL(&Agent, &Conf->Servers, UDM_SRV_ACTION_TABLE, db);
#endif
	if(res != UDM_OK){
	  strcpy(Conf->errstr,db->errstr);
	}

	UdmAgentFree(&Agent);
	UdmDBListFree(&dbl);
	return res;
}


static int add_limit(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	char * sc;
	char * nm;
	
	if((sc = strchr(av[1],':'))){
		*sc++='\0';
		nm=(char*)malloc(strlen(av[1])+8);
		sprintf(nm,"Limit-%s",av[1]);
		UdmVarListReplaceStr(&Conf->Vars, nm, sc);
		UDM_FREE(nm);
	}
	return UDM_OK;
}

static int flush_srv_table(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res=UDM_OK;
	if(C->flags&UDM_FLAG_ADD_SERV){
		UDM_AGENT A;
		A.Conf = Conf;
		res=UdmSrvAction(&A, &Conf->Servers, UDM_SRV_ACTION_FLUSH);
	}
	return res;
}
		
static int env_rpl_charset(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf = C->Indexer->Conf;
	UDM_CHARSET	*cs;
	if(!(cs=UdmGetCharSet(av[1]))){
		sprintf(Conf->errstr,"charset '%s' is not supported",av[1]);
		return UDM_ERROR;
	}
	if(!strcasecmp(av[0],"LocalCharset")){
		Conf->lcs=cs;
		UdmVarListReplaceStr(&Conf->Vars,av[0],av[1]);
	}else
	if(!strcasecmp(av[0],"BrowserCharset")){
		Conf->bcs=cs;
		UdmVarListReplaceStr(&Conf->Vars,av[0],av[1]);
	}
	return UDM_OK;
}


static int srv_rpl_charset(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf = C->Indexer->Conf;
	UDM_CHARSET	*cs;
	if(!(cs=UdmGetCharSet(av[1]))){
		sprintf(Conf->errstr,"charset '%s' is not supported",av[1]);
		return UDM_ERROR;
	}
	UdmVarListReplaceStr(&C->Srv->Vars,av[0],av[1]);
	return UDM_OK;
}

static int srv_rpl_mirror(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	if(!strcasecmp(av[0],"MirrorRoot") || !strcasecmp(av[0],"MirrorHeadersRoot")){
		char	fname[1024];
		rel_var_name(C->Indexer->Conf, fname, sizeof(fname)-1, av[1]);
		UdmVarListReplaceStr(&C->Srv->Vars,av[0],fname);
	}else
	if(!strcasecmp(av[0],"MirrorPeriod")){
		int	tm=Udm_dp2time_t(av[1]);
		UdmVarListReplaceInt(&C->Srv->Vars,"MirrorPeriod",tm);
	}
	return UDM_OK;
}
		

static int srv_rpl_auth(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	char	name[128];
	udm_snprintf(name,sizeof(name)-1,"%s",av[0]);
	name[sizeof(name)-1]='\0';
	if(av[1]){
		size_t	len=strlen(av[1]);
		char	*auth=(char*)malloc(BASE64_LEN(strlen(av[1])));
		udm_base64_encode(av[1],auth,len);
		UdmVarListReplaceStr(&C->Srv->Vars,name,auth);
		UDM_FREE(auth);
	}else{
		UdmVarListReplaceStr(&C->Srv->Vars,name,"");
	}
	return UDM_OK;
}

static char *str_store (char *dest, char *src) {
	size_t d_size = (dest ? strlen(dest) : 0);
	size_t s_size = strlen(src) + 1;
	char *d = realloc(dest, d_size + s_size);

	if (d) memcpy(d + d_size, src, s_size);
	else UDM_FREE(dest);
	return(d);
}

static char *ParseEnvVar (UDM_ENV *Conf, const char *str) {
	char *p1 = (char *)str;
	char *p2 = (char *)str;
	char *p3;
	char *s;
	char *o = NULL;

	if (! str) return(NULL);
	while ((p1 = strchr(p1, '$'))) {
		if (p1[1] != '(') {
			p1++;
			continue;
		}
		*p1 = 0;
		o = str_store(o, p2);
		*p1 = '$';
		if ((p3 = strchr(p1 + 2, ')'))) {
			*p3 = 0;
			s = (char *)UdmVarListFindStr(&Conf->Vars, p1 + 2, NULL);
			if (s) o = str_store(o, s);
			*p3 = ')';
			p2 = p1 = p3 + 1;
		} else {
			UDM_FREE(o);
			return(NULL);
		}
	}
	o = str_store(o, p2);
	return(o);
}

static int env_rpl_env_var (void *Cfg, size_t ac, char **av) {
	UDM_ENV *Conf = ((UDM_CFG *)Cfg)->Indexer->Conf;
	char *p = getenv(av[1]);
	if (! p) {
		sprintf(Conf->errstr, "ImportEnv '%s': no such variable.", av[1]);
		return UDM_ERROR;
	}
	UdmVarListReplaceStr(&Conf->Vars, av[1], p);
	return UDM_OK;
}


static int env_rpl_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	if(!strcasecmp(av[0],"DBAddr")){
		if(UDM_OK != UdmDBListAdd(&Conf->dbl, av[1] ? av[1] : "", UDM_OPEN_MODE_WRITE)){
			sprintf(Conf->errstr, "Invalid DBAddr: '%s'",av[1]?av[1]:"");
			return UDM_ERROR;
		}
	}
	UdmVarListReplaceStr(&Conf->Vars,av[0],av[1]);
	return UDM_OK;
}

static int srv_rpl_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UdmVarListReplaceStr(&C->Srv->Vars,av[0],av[1]);
	return UDM_OK;
}

static int srv_rpl_hdr(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	char	*nam=NULL;
	char	*val=NULL;
	char	name[128];
	
	switch(ac){
		case 3:	nam=av[1];val=av[2];break;
		case 2:
			if((val=strchr(av[1],':'))){
				*val++='\0';
				val=UdmTrim(val," \t");
				nam=av[1];
			}
			break;
	}
	if(nam){
		udm_snprintf(name,sizeof(name),"Request.%s",nam);
		name[sizeof(name)-1]='\0';
		UdmVarListReplaceStr(&C->Srv->Vars,name,val);
	}
	return UDM_OK;
}

static int srv_rpl_bool_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	int res=!strcasecmp(av[1],"yes");
	UdmVarListReplaceInt(&C->Srv->Vars,av[0],res);
	return UDM_OK;
}

static int env_rpl_bool_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int res=!strcasecmp(av[1],"yes");
	if(!strcasecmp(av[0],"LogsOnly"))Conf->logs_only=res;
	return UDM_OK;
}

static int env_rpl_num_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res=atoi(av[1]);
	if(!strcasecmp(av[0],"IspellCorrectFactor"))Conf->WordParam.correct_factor=res;
	if(!strcasecmp(av[0],"IspellIncorrectFactor"))Conf->WordParam.incorrect_factor=res;
	if(!strcasecmp(av[0],"NumberFactor"))Conf->WordParam.number_factor=res;
	if(!strcasecmp(av[0],"AlnumFactor"))Conf->WordParam.alnum_factor=res;
	if(!strcasecmp(av[0],"MinWordLength"))Conf->WordParam.min_word_len=res;
	if(!strcasecmp(av[0],"MaxWordLength"))Conf->WordParam.max_word_len=res;
	return UDM_OK;
}		

static int srv_rpl_num_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	int	res=atoi(av[1]);
	UdmVarListReplaceInt(&C->Srv->Vars,av[0],res);
	if (strcasecmp(av[0], "MaxHops") == 0) C->Srv->MaxHops = (uint4) res;
	return UDM_OK;
}

static int srv_rpl_time_var(void *Cfg, size_t ac,char **av){
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	int	res=Udm_dp2time_t(av[1]);
	if(res==-1){
		sprintf(Conf->errstr,"bad time interval: %s",av[1]);
		res=UDM_ERROR;
	}
	UdmVarListReplaceInt(&C->Srv->Vars,av[0],res);
	return UDM_OK;
}

static int srv_rpl_category(void *Cfg, size_t ac, char **av) {
	UDM_CFG	*C=(UDM_CFG*)Cfg;
	UDM_ENV	*Conf=C->Indexer->Conf;
	unsigned int cid = UdmGetCategoryId(Conf, av[1]);
	char buf[64];
	udm_snprintf(buf, 64, "%u", cid);
	UdmVarListReplaceStr(&C->Srv->Vars, av[0], buf);
	return UDM_OK;
}

typedef struct conf_cmd_st {
	const char	*name;
	size_t		argmin;
	size_t		argmax;
	int		(*action)(void *a,size_t n,char **av);
} UDM_CONFCMD;


static UDM_CONFCMD commands[] = 
{
	{"Include",		1,1,	do_include},

	{"ImportEnv",		1,1,	env_rpl_env_var},
	{"DBAddr",		1,1,	env_rpl_var},
	{"URLDAddr",		1,1,	env_rpl_var},
	{"Listen",		1,1,	env_rpl_var},
	{"UseRemoteContentType",1,1,	env_rpl_var},
	{"UseCRC32URLId",	1,1,	env_rpl_var},
	{"NewsExtensions",	1,1,	env_rpl_var},
	{"CrossWords",		1,1,	env_rpl_var},
	{"SyslogFacility",	1,1,	env_rpl_var},
	{"AliasProg",		1,1,	env_rpl_var},
	{"ForceIISCharset1251",	1,1,	env_rpl_var},
	{"GroupBySite",		1,1,	env_rpl_var},
	{"Cache",		1,1,	env_rpl_var},
	{"PopRankSkipSameSite",	1,1,	env_rpl_var},
	{"PopRankFeedBack",	1,1,	env_rpl_var},
	{"PopRankUseTracking",	1,1,	env_rpl_var},
	{"PopRankUseShowCnt",	1,1,	env_rpl_var},
	{"PopRankShowCntRatio",	1,1,	env_rpl_var},
	{"PopRankShowCntWeight",1,1,	env_rpl_var},
	{"VarDir",		1,1,	env_rpl_var},
	{"DocMemCacheSize",	1,1,	env_rpl_var},
	{"IspellUsePrefixes",	1,1,	env_rpl_var},
	{"URLSelectCacheSize",	1,1,	env_rpl_var},
	{"MaxDocSize",		1,1,	env_rpl_var},
	{"StoredFiles",		1,1,	env_rpl_var},
	{"OptimizeInterval",	1,1,	env_rpl_var},
	{"OptimizeRatio",	1,1,	env_rpl_var},
	{"ParserTimeOut",	1,1,	env_rpl_var},
	{"NumSections",		1,1,	env_rpl_var},
	{"DateFormat",		1,1,	env_rpl_var},
	{"GuesserUseMeta",	1,1,	env_rpl_var},
	{"ResultsLimit",        1,1,    env_rpl_var},
	{"IspellCorrectFactor",	1,1,	env_rpl_num_var},
	{"IspellInCorrectFactor",1,1,	env_rpl_num_var},
	{"NumberFactor",	1,1,	env_rpl_num_var},
	{"AlnumFactor",		1,1,	env_rpl_num_var},
	{"MinWordLength",	1,1,	env_rpl_num_var},
	{"MaxWordLength",	1,1,	env_rpl_num_var},
	{"LogsOnly",		1,1,	env_rpl_bool_var},
	{"LocalCharset",	1,1,	env_rpl_charset},
	{"BrowserCharset",	1,1,	env_rpl_charset},
	
	{"HTDBAddr",		1,1,	srv_rpl_var},
	{"HTDBList",		1,1,	srv_rpl_var},
	{"HTDBDoc",		1,1,	srv_rpl_var},
	{"HTDBLimit",           1,1,    srv_rpl_var},
	{"DefaultLang",		1,1,	srv_rpl_var},
	{"Category",		1,1,	srv_rpl_category},
	{"Tag",			1,1,	srv_rpl_var},
	{"Proxy",		1,1,	srv_rpl_var},
	{"VaryLang",            1,1,    srv_rpl_var},
	{"MaxNetErrors",	1,1,	srv_rpl_num_var},
	{"MaxHops",		1,1,	srv_rpl_num_var},
	{"ServerWeight",	1,1,	srv_rpl_num_var},
	{"Robots",		1,1,	srv_rpl_bool_var},
	{"DetectClones",	1,1,	srv_rpl_bool_var},
	{"Index",		1,1,	srv_rpl_bool_var},
	{"NetErrorDelayTime",	1,1,	srv_rpl_time_var},
	{"ReadTimeOut",		1,1,	srv_rpl_time_var},
	{"DocTimeOut",		1,1,	srv_rpl_time_var},
	{"Period",		1,1,	srv_rpl_time_var},
	{"HoldBadHrefs",	1,1,	srv_rpl_time_var},
	{"HTTPHeader",		1,2,	srv_rpl_hdr},
	{"ProxyAuthBasic",	1,1,	srv_rpl_auth},
	{"AuthBasic",		1,1,	srv_rpl_auth},
	{"MirrorRoot",		1,1,	srv_rpl_mirror},
	{"MirrorHeadersRoot",	1,1,	srv_rpl_mirror},
	{"MirrorPeriod",	1,1,	srv_rpl_mirror},
	{"RemoteCharset",	1,1,	srv_rpl_charset},
	
	{"Disallow",		1,100,	add_filter},
	{"Allow",		1,100,	add_filter},
	{"CheckMP3Only",	1,100,	add_filter},
	{"CheckMP3",		1,100,	add_filter},
	{"CheckOnly",		1,100,	add_filter},
	{"HrefOnly",		1,100,	add_filter},
	
	{"Server",		1,100,	add_srv},
	{"Realm",		1,100,	add_srv},
	{"Subnet",		1,100,	add_srv},
	{"URL",			1,1,	add_url},
	
	{"Alias",		1,100,	add_alias},
	{"ReverseAlias",	1,100,	add_alias},
	
	{"AddType",		1,100,	add_type},
	{"Mime",		2,3,	add_parser},
	{"Section",		2,3,	add_section},
	{"Affix",		3,3,	add_affix},
	{"Spell",		3,3,	add_spell},
	{"StopwordFile",	1,1,	add_stoplist},
	{"LangMapFile",		1,1,	add_langmap},
	{"LangMapUpdate",	1,1,	env_rpl_var},
	{"Synonym",		1,1,	add_synonym},
	{"LoadChineseList",	0,1,	add_chinese},
	{"LoadThaiList",        0,1,    add_thai},
	{"Limit",		1,1,	add_limit},
	{"ServerTable",		1,1,	add_srv_table},
	{"FlushServerTable",	0,0,	flush_srv_table},
	
	{NULL,0,0,0}
};

__C_LINK int __UDMCALL UdmEnvAddLine(UDM_CFG *C,char *str){
	UDM_ENV		*Conf=C->Indexer->Conf;
	UDM_CONFCMD	*Cmd;
	char		*av[255];
	size_t		ac;
	int             res = UDM_OK;
	
	ac = UdmGetArgs(str, av, 255);
	
	for (Cmd=commands ; Cmd->name ; Cmd++){
		if(!strcasecmp(Cmd->name,av[0])){
			int argc=ac;
			size_t i;
			char *p;
			
			argc--;
			if(ac<Cmd->argmin+1){
				sprintf(Conf->errstr,"too few (%d) arguments for command '%s'",argc,Cmd->name);
				return UDM_ERROR;
			}
			
			if(ac>Cmd->argmax+1){
				sprintf(Conf->errstr,"too many (%d) arguments for command '%s'",argc,Cmd->name);
				return UDM_ERROR;
			}
			
			for (i = 1; i < ac; i++) {
				if (! av[i]) continue;
				p = ParseEnvVar(Conf, av[i]);
				if (! p) {
					sprintf(Conf->errstr, "An error occured while parsing '%s'", av[i]);
					return UDM_ERROR;
				}
				av[i] = p;
			}

			if(Cmd->action){
				res = Cmd->action((void*)C,ac,av);
			}
			
			for (i = 1; i < ac; i++) UDM_FREE(av[i]);

			if(Cmd->action){
				return res;
			}
		}
	}
	sprintf(Conf->errstr,"Unknown command: %s",av[0]);
	return UDM_ERROR;
}


static int EnvLoad(UDM_CFG *Cfg,const char *cname){
	char	*str0 = NULL;	/* Unsafe copy - will be used in strtok	*/
	char	str1[1024]="";	/* To concatenate lines			*/
	FILE	*config;	/* File struct */
	int	rc=UDM_OK;
	size_t	line = 0, str0len = 0, str1len, str0size = 4096;
	
	if ((str0 = (char*)malloc(str0size)) == NULL) {
		sprintf(Cfg->Indexer->Conf->errstr, "Can't alloc %d bytes at '%s': %d", str0size, __FILE__, __LINE__);
		return UDM_ERROR;
	}
	str0[0]=0;
	
	/* Open config file */
	if(!(config=fopen(cname,"r"))){
		sprintf(Cfg->Indexer->Conf->errstr, "Can't open config file '%s': %s", cname, strerror(errno));
		UDM_FREE(str0);
		return UDM_ERROR;
	}

	/*  Read lines and parse */
	while(fgets(str1,sizeof(str1),config)){
		char	*end;
		
		line++;
		
		if(str1[0]=='#')continue;
		for (end = str1 + (str1len = strlen(str1)) - 1 ; (end>=str1) && (*end=='\r'||*end=='\n'||*end==' ') ; *end--='\0');
		if(!str1[0])continue;
		
		if(*end=='\\'){
			*end=0;
			if (str0len + str1len >= str0size) {
			  str0size += 4096 + str1len;
			  if ((str0 = (char*)realloc(str0, str0size)) == NULL) {
			    sprintf(Cfg->Indexer->Conf->errstr, "Can't realloc %d bytes at '%s': %d", str0size, __FILE__, __LINE__);
			    return UDM_ERROR;
			  }
			}
			strcat(str0,str1);
			str0len += str1len;
			continue;
		}
		strcat(str0,str1);
		str0len += str1len;
		
		if(UDM_OK != (rc=UdmEnvAddLine(Cfg,str0))){
			char	err[2048];
			strcpy(err,Cfg->Indexer->Conf->errstr);
			sprintf(Cfg->Indexer->Conf->errstr,"%s:%d: %s",cname,line,err);
			break;
		}
		
		str0[0]=0;
		str0len = 0;
	}
	UDM_FREE(str0);
	fclose(config);
	return rc;
}


__C_LINK  int __UDMCALL UdmEnvLoad(UDM_AGENT *Indexer, const char *cname, int lflags) {
	UDM_CFG		Cfg;
	UDM_SERVER	Srv;
	int		rc=UDM_OK;
	const char	*dbaddr=NULL;
	
	UdmServerInit(&Srv);
	bzero((void*)&Cfg, sizeof(Cfg));
	Cfg.Indexer = Indexer;
	Indexer->Conf->Cfg_Srv = Cfg.Srv = &Srv;
	Cfg.flags=lflags;
	Cfg.level=0;
	
	/* Set DBAddr if for example passed from environment */
	if((dbaddr=UdmVarListFindStr(&Indexer->Conf->Vars,"DBAddr",NULL))){
		if(UDM_OK != UdmDBListAdd(&Indexer->Conf->dbl, dbaddr, UDM_OPEN_MODE_WRITE)){
			sprintf(Indexer->Conf->errstr, "Invalid DBAddr: '%s'", dbaddr);
			rc=UDM_ERROR;
			goto freeex;
		}
	}
	
	if(UDM_OK == (rc=EnvLoad(&Cfg,cname))){
		/* Sort ispell dictionay if it has been loaded */
		if(Indexer->Conf->Spells.nspell) {
			UdmSortDictionary(&Indexer->Conf->Spells);
			UdmSortAffixes(&Indexer->Conf->Affixes, &Indexer->Conf->Spells);
		}
		
		/* Sort synonyms */
		UdmSynonymListSort(&Indexer->Conf->Synonyms);
		
		UdmVarListInsStr(&Indexer->Conf->Vars, "Request.User-Agent", UDM_USER_AGENT "/" VERSION);
	}

freeex:
	UdmServerFree(&Srv);
	return rc;
}
