/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_ARPA_NAMESER_H
#include <arpa/nameser.h>
#endif
#ifdef HAVE_RESOLV_H
#include <resolv.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <assert.h>


#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_utils.h"
#include "udm_url.h"
#include "udm_sdp.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_searchtool.h"
#include "udm_result.h"
#include "udm_log.h"

#include "udm_proto.h"
#include "udm_host.h"
#include "udm_hash.h"
#include "udm_doc.h"
#include "udm_cache.h"
#include "udm_services.h"
#include "udm_xmalloc.h"
#include "udm_worddb.h"
#include "udm_searchcache.h"

#define UDM_THREADINFO(A,s,m)	if(A->Conf->ThreadInfo)A->Conf->ThreadInfo(A,s,m)


void *UdmDBInit(void *vdb){
	UDM_DB *db=vdb;
	size_t	nbytes=sizeof(UDM_DB);
	
	if(!db){
		db=(UDM_DB*)malloc(nbytes);
		bzero((void*)db, nbytes);
		db->freeme=1;
	}else{
		bzero((void*)db, nbytes);
	}
	db->numtables=32;

	UdmURLInit(&db->addr);
	
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
	db->hDbc=SQL_NULL_HDBC;
	db->hEnv=SQL_NULL_HENV;
	db->hstmt=SQL_NULL_HSTMT;
#endif
#if (HAVE_IBASE)
	db->DBH=NULL;
#endif
#if (HAVE_ORACLE8)
	db->par = UdmXmalloc(sizeof(struct param_struct));
#endif
	return db;
}


void UdmDBFree(void *vdb){
	UDM_DB	*db=vdb;
	
	UdmURLFree(&db->addr);
	UDM_FREE(db->DBADDR);
	UDM_FREE(db->DBName);
	UDM_FREE(db->DBUser);
	UDM_FREE(db->DBPass);
	UDM_FREE(db->DBSock);
	UDM_FREE(db->where);
	UDM_FREE(db->from);

	if (db->cached_sd) {
	  shutdown(db->cached_sd, 2);
	  closesocket(db->cached_sd);
	}
	if (db->stored_sd) { 
	  const char *hello = "B\0";
	  UdmSend(db->stored_sd, hello, 1, 0);
	  shutdown(db->stored_sd, 2);
	  closesocket(db->stored_sd);
	}
	if (db->searchd) UdmSearchdClose(db);
	
	if(!db->connected)goto ret;
	
#if HAVE_MYSQL
	if(db->DBDriver==UDM_DB_MYSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_PGSQL
	if(db->DBDriver==UDM_DB_PGSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_MSQL
	if(db->DBDriver==UDM_DB_MSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_IBASE
	if(db->DBDriver==UDM_DB_IBASE){
		UdmSQLClose(db);
		goto ret;
	}
#endif

#if HAVE_ORACLE8
	if(db->DBDriver==UDM_DB_ORACLE8){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_ORACLE7
	if(db->DBDriver==UDM_DB_ORACLE7){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_CTLIB
	if(db->DBDriver==UDM_DB_MSSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
	UdmSQLClose(db);
	goto ret;
#endif
ret:
	if(db->freeme)UDM_FREE(vdb);
	return;
}



__C_LINK int __UDMCALL UdmLimit8(UDM_ENV *Conf, UDM_UINT8URLIDLIST *L,const char *field,int type, void *vdb){
	UDM_DB	*db=vdb;
	int	rc=UDM_OK;
#ifdef HAVE_SQL
	rc=UdmLimit8SQL(Conf, L, field, type, db);
#endif
	strcpy(Conf->errstr, db->errstr);
	return rc;
}

__C_LINK int __UDMCALL UdmLimit4(UDM_ENV *Conf, UDM_UINT4URLIDLIST *L,const char *field, int type, void *vdb){
	UDM_DB	*db=vdb;
	int	rc=UDM_OK;
	
#ifdef HAVE_SQL
	rc=UdmLimit4SQL(Conf, L, field, type, db);
#endif
	strcpy(Conf->errstr, db->errstr);
	return rc;
}


int UdmURLData(UDM_ENV *Conf, UDM_URLDATALIST *L, UDM_DB *db) {
	int	res=UDM_OK;
	
	L->nitems=0;
	
#ifdef HAVE_SQL
	res=UdmURLDataSQL(Conf, L, db);
#endif
	return res;
}


__C_LINK int __UDMCALL UdmClearDatabase(UDM_AGENT *A){
	int	res=UDM_ERROR;
	UDM_DB	*db;
	size_t i, dbto =  A->Conf->dbl.nitems;

	for (i = 0; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
#ifdef HAVE_SQL
	  res = UdmClearDBSQL(A, db);
	  UDM_FREE(db->where);          /* clear db->where for next parameters */
#endif
	  if (res != UDM_OK) break;
	}
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
	}
	return res;
}



static int DocUpdate(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc) {
	int		result=UDM_OK;
	const char	*c;
	int		status=UdmVarListFindInt(&Doc->Sections,"Status",0);
	urlid_t		origin_id = 0;
	urlid_t		url_id = (urlid_t)UdmVarListFindInt(&Doc->Sections, "ID", 0);
	time_t		next_index_time;
	char		dbuf[64];
	int             use_crosswords, use_newsext; 

	UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
	use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"CrossWords","no"),"yes");
	use_newsext    = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"NewsExtensions","no"),"yes");
	UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
	
	/* First of all check that URL must be delated */
	
	if(Doc->method==UDM_METHOD_DISALLOW){
		UdmLog(Indexer,UDM_LOG_ERROR,"Deleting %s", UdmVarListFindStr(&Doc->Sections, "URL", ""));
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELETE);
		return result;
	}

	next_index_time=time(NULL)+Doc->Spider.period;
	UdmTime_t2HttpStr(next_index_time,dbuf);
	UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
	
	switch(status){
	
	case 0: /* No HTTP code */
		if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
		UdmLog(Indexer,UDM_LOG_ERROR,"No HTTP response status");
		next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
		UdmTime_t2HttpStr(next_index_time,dbuf);
		UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
	
	case UDM_HTTP_STATUS_OK:				/* 200 */
	case UDM_HTTP_STATUS_PARTIAL_OK:			/* 206 */
		if(!UdmVarListFind(&Doc->Sections,"Content-Type")){
			UdmLog(Indexer,UDM_LOG_ERROR,"No Content-type header");
			next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
			UdmTime_t2HttpStr(next_index_time,dbuf);
			UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
			UdmVarListReplaceInt(&Doc->Sections,"Status",UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR);
			if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
			result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
			return result;
		} else {
				if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors = 0;
		}
		break;
	
	case UDM_HTTP_STATUS_MULTIPLE_CHOICES:			/* 300 */
	case UDM_HTTP_STATUS_MOVED_PARMANENTLY:			/* 301 */
	case UDM_HTTP_STATUS_MOVED_TEMPORARILY:			/* 302 */
	case UDM_HTTP_STATUS_SEE_OTHER:				/* 303 */
	case UDM_HTTP_STATUS_NOT_MODIFIED:			/* 304 */
		/* FIXME: check that status is changed and remove words if necessary */
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
		break;
	
	case UDM_HTTP_STATUS_USE_PROXY:				/* 305 */
	case UDM_HTTP_STATUS_BAD_REQUEST:			/* 400 */
	case UDM_HTTP_STATUS_UNAUTHORIZED:			/* 401 */
	case UDM_HTTP_STATUS_PAYMENT_REQUIRED:			/* 402 */
	case UDM_HTTP_STATUS_FORBIDDEN:				/* 403 */
	case UDM_HTTP_STATUS_NOT_FOUND:				/* 404 */
	case UDM_HTTP_STATUS_METHOD_NOT_ALLOWED:		/* 405 */
	case UDM_HTTP_STATUS_NOT_ACCEPTABLE:			/* 406 */
	case UDM_HTTP_STATUS_PROXY_AUTHORIZATION_REQUIRED:	/* 407 */
	case UDM_HTTP_STATUS_REQUEST_TIMEOUT:			/* 408 */
	case UDM_HTTP_STATUS_CONFLICT:				/* 409 */
	case UDM_HTTP_STATUS_GONE:				/* 410 */
	case UDM_HTTP_STATUS_LENGTH_REQUIRED:			/* 411 */
	case UDM_HTTP_STATUS_PRECONDITION_FAILED:		/* 412 */
	case UDM_HTTP_STATUS_REQUEST_ENTITY_TOO_LARGE:		/* 413 */
	case UDM_HTTP_STATUS_REQUEST_URI_TOO_LONG:		/* 414 */	
	case UDM_HTTP_STATUS_UNSUPPORTED_MEDIA_TYPE:		/* 415 */
	case UDM_HTTP_STATUS_NOT_IMPLEMENTED:			/* 501 */
	case UDM_HTTP_STATUS_BAD_GATEWAY:			/* 502 */
	case UDM_HTTP_STATUS_NOT_SUPPORTED:			/* 505 */
	
		/*
		  FIXME: remove words from database
		  Check last reffering time to remove when
		  there are no links to this document anymore
		  UdmLog(Indexer,UDM_LOG_EXTRA,"Deleting URL");
		*/
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
	
	case UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR:		/* 500 */
	case UDM_HTTP_STATUS_SERVICE_UNAVAILABLE:		/* 503 */
	case UDM_HTTP_STATUS_GATEWAY_TIMEOUT:			/* 504 */
	
		/* Keep words in database                */
		/* We'll retry later, maybe host is down */
		if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
		next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
		UdmTime_t2HttpStr(next_index_time,dbuf);
		UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
	
	default: /* Unknown status, retry later */
		UdmLog(Indexer,UDM_LOG_WARN,"HTTP %d We don't yet know how to handle it, skipped",status);
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
	}
	
	
	if(Doc->method==UDM_METHOD_GET && Doc->Spider.use_clones){
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_FINDORIG);
		if(result!=UDM_OK)return result;
		origin_id = (urlid_t)UdmVarListFindInt(&Doc->Sections,"Origin-ID",0);
	}
	
	
	/* Check clones */
	if((origin_id)&&(origin_id!=url_id)){
	        if (UdmNeedLog(UDM_LOG_EXTRA))
		  UdmLog(Indexer, UDM_LOG_EXTRA, "Duplicate Document %s with #%d", 
			 UdmVarListFindStr(&Doc->Sections, "URL", ""), origin_id);
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELWORDS);
		if(use_crosswords){
			if(result == UDM_OK) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELCWORDS);
		}
		if(result == UDM_OK) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_UPDCLONE);
		return result;
	}
	
	/* Check that document wasn't modified since last indexing */
	if( (UdmVarListFindInt(&Doc->Sections,"crc32", 0) != 0) 
	    &&  (UdmVarListFindInt(&Doc->Sections,"crc32old",0)==UdmVarListFindInt(&Doc->Sections,"crc32",0)) 
	    &&  (!(Indexer->flags&UDM_FLAG_REINDEX))) {
		result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
		return result;
	}
	
	/* Compose site_id string and calc it's Hash32 */
/*	sprintf(site_id_str,"%s://%s/",Doc->CurURL.schema,Doc->CurURL.hostinfo);
	UdmVarListReplaceInt(&Doc->Sections, "Site-ID", UdmStrHash32(site_id_str)); for what this need ? */
	
	/* Copy default languages, if not given by server and not guessed */
	if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL))){
		if((c=UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
			UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
	}
	
	/* For NEWS extension: get rec_id from my */
	/* parent out of db (if I have one...)    */
	if(use_newsext){
		UDM_VAR		*Sec;
		const char	*parent=NULL;
		int		parent_id=0;
		
		if((Sec=UdmVarListFind(&Doc->Sections,"Header.References")) && Sec->val){
			/* References contains all message IDs of my */
			/* predecessors, space separated             */
			/* my direct parent is the last in the list  */
			if((parent = strrchr(Sec->val,' '))){
				/* parent now points to the space */
				/* character skip it              */
				++parent;
			}else{
				/* there is only one entry in */
				/* references, so this is my parent */
				parent=Sec->val;
			}	
		}
		
		/* get parent from database */
		if(parent && strlen(parent) && strchr(parent,'@')){
			UDM_DOCUMENT Msg;
			
			UdmDocInit(&Msg);
			UdmVarListReplaceStr(&Msg.Sections,"Header.Message-ID",parent);
			result = UdmURLAction(Indexer, &Msg, UDM_URL_ACTION_FINDBYMSG);
			parent_id = UdmVarListFindInt(&Msg.Sections,"ID",0);
			UdmVarListReplaceInt(&Doc->Sections,"Header.Parent-ID",parent_id);
			UdmDocFree(&Msg);
		}
		
		/* Now register me with my parent  */
		if(parent_id) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_REGCHILD);
		
		if(result!=UDM_OK)return result;
	}
	
	/* Now store words and crosswords */
	if(UDM_OK != (result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_INSWORDS)))
		return result;
	
	if(use_crosswords)
		if(UDM_OK != (result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_INSCWORDS)))
			return result;
	
	result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_LUPDATE);
	
	return result;
}


static int UdmDocUpdate(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc){
	size_t		maxsize;
	size_t		sec;
	int		flush=0;
	int		rc=UDM_OK;
	UDM_RESULT	*I = &Indexer->Indexed;

	UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
	maxsize = UdmVarListFindInt(&Indexer->Conf->Vars,"DocMemCacheSize",0) * 1024 * 1024;
	UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
	if(Doc){
		/* Add document into cache */
		I->Doc=(UDM_DOCUMENT*)realloc(I->Doc,(I->num_rows+1)*sizeof(UDM_DOCUMENT));
		I->Doc[I->num_rows]=Doc[0];
		I->num_rows++;
		
		I->memused+=sizeof(UDM_DOCUMENT);
		/* Aproximation for Words memory usage  */
		I->memused+=Doc->Words.nwords*(sizeof(UDM_WORD)+10);
		/* Aproximation for CrossWords memory usage */
		I->memused+=Doc->CrossWords.ncrosswords*(sizeof(UDM_CROSSWORD)+10+20);
		/* Aproximation for Sections memory usage */
		for(sec=0;sec<Doc->Sections.nvars;sec++){
			I->memused+=sizeof(UDM_VAR);
			I->memused+=Doc->Sections.Var[sec].maxlen*9;
		}
	}
	
	UdmLog(Indexer,UDM_LOG_EXTRA,"DocCacheSize: %d/%d",I->memused,maxsize);
	if(I->memused>=maxsize)flush=1;
	if(I->num_rows>=1024)flush=1;
	if(!Doc)flush=1;
	
	if(flush){
		size_t	docnum;
		
		if (Indexer->Indexed.num_rows > 1)
			UdmLog(Indexer, UDM_LOG_EXTRA, "Flush %d document(s)", Indexer->Indexed.num_rows);
		
		for (docnum=0;docnum<I->num_rows;docnum++){
			/* Flush all hrefs from cache in-memory    */
			/* cache into database. Note, this must    */
			/* be done before call of  StoreCrossWords */
			/* because we need to know all new URL IDs */
			
			UDM_THREADINFO(Indexer,"Updating",UdmVarListFindStr(&I->Doc[docnum].Sections,"URL",""));
			if(UDM_OK != (rc = DocUpdate(Indexer, &I->Doc[docnum])))
				return rc;
		}
/* this should be only if we use blob mode, otherwise built-in wan't work
		if(UDM_OK != (rc = UdmResAction(Indexer, &Indexer->Indexed, UDM_RES_ACTION_INSWORDS)))
			return rc;*/
		UdmResultFree(&Indexer->Indexed);
	}
	return rc;
}

__C_LINK int __UDMCALL UdmURLAction(UDM_AGENT *A, UDM_DOCUMENT *D, int cmd) {
	int res=UDM_ERROR, execflag = 0;
	size_t i, dbfrom = 0, dbto;
	UDM_DB	*db;

	if(cmd == UDM_URL_ACTION_FLUSH) {
	  return UdmDocUpdate(A, D);
	}

#ifdef USE_TRACE
	fprintf(A->TR, "[%d] URLAction: %d\n", A->handle, cmd);
#endif
	
	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	if (D != NULL) {
	  udmhash32_t	url_id=UdmVarListFindInt(&D->Sections,"URL_ID", 0);
	  dbfrom = dbto = ((url_id) ? url_id : UdmStrHash32(UdmVarListFindStr(&D->Sections, "URL", ""))) % A->Conf->dbl.nitems;
	  dbto++;
	}
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];

	  UDM_GETLOCK(A, UDM_LOCK_DB);
	  switch(db->DBDriver) {
	  case UDM_DB_SEARCHD:
	    res = UdmSearchdURLAction(A, D, cmd, db);
	    execflag = 1;
	    break;
	  case UDM_DB_CACHE:
	    res = UdmURLActionCache(A, D, cmd, db);
	    execflag = 1;
	    break;
#ifdef HAVE_SQL
	  default:
		res=UdmURLActionSQL(A,D,cmd,db);
		if (cmd == UDM_URL_ACTION_EXPIRE) UDM_FREE(db->where);  /* clear db->where for next parameters */
		execflag = 1;
		break;
#endif
	  }
	  if (res != UDM_OK && execflag) {
	    UdmLog (A, UDM_LOG_ERROR, db->errstr);
	  }
	  UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  if (res != UDM_OK) break;
	}
	if ((res != UDM_OK) && !execflag) {
	  UdmLog(A, UDM_LOG_ERROR, "no supported DBAddr specified");
	}
	return res;
}


__C_LINK int __UDMCALL UdmTargets(UDM_AGENT *A) {
	int	res=UDM_ERROR;
	UDM_DB	*db;
	size_t i, dbfrom = 0, dbto;

	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	UdmResultFree(&A->Conf->Targets);
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
	  UDM_GETLOCK(A, UDM_LOCK_DB);
#ifdef HAVE_SQL
	  res = UdmTargetsSQL(A, db);
#endif
	  if(res != UDM_OK){
		UdmLog(A, UDM_LOG_ERROR, db->errstr);
	  }
	  UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  if (res != UDM_OK) break;
	}
	return res;
}

__C_LINK int __UDMCALL UdmResAction(UDM_AGENT *A, UDM_RESULT *R, int cmd){
	int	res=UDM_ERROR;
	UDM_DB	*db;
	size_t i, dbfrom = 0, dbto;
	
	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
	  UDM_GETLOCK(A, UDM_LOCK_DB);
#ifdef HAVE_SQL
	  res = UdmResActionSQL(A, R, cmd, db, i);
#endif
	  if(res != UDM_OK){
		UdmLog(A, UDM_LOG_ERROR, db->errstr);
	  }
	  UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  if (res != UDM_OK) break;
	}
	return res;
}

__C_LINK int __UDMCALL UdmCatAction(UDM_AGENT *A, UDM_CATEGORY *C, int cmd){
	UDM_DB	*db;
	int	res=UDM_ERROR;
	size_t i, dbfrom = 0, dbto;

	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
	  UDM_GETLOCK(A, UDM_LOCK_DB);
	  switch(db->DBDriver) {
	  case UDM_DB_SEARCHD:
	        res = UdmSearchdCatAction(A, C, cmd, db);
		break;
#ifdef HAVE_SQL
	  default:
		res=UdmCatActionSQL(A,C,cmd,db);
#endif
	  }
	  if(res != UDM_OK){
		UdmLog(A, UDM_LOG_ERROR, db->errstr);
	  }
	  UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  if (res != UDM_OK) break;
	}
	return res;
}

__C_LINK int __UDMCALL UdmSrvAction(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd) {
	UDM_DB	*db;
	int	res=UDM_ERROR;
	size_t i, dbfrom = 0, dbto;
	
	UDM_GETLOCK(A, UDM_LOCK_CONF);
	dbto =  A->Conf->dbl.nitems;
	UDM_RELEASELOCK(A, UDM_LOCK_CONF);

	strcpy(A->Conf->errstr, "No appropriate storage support compiled");
	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];

	  UDM_GETLOCK(A, UDM_LOCK_DB); 
#ifdef HAVE_SQL
	  res = UdmSrvActionSQL(A, S, cmd, db);
#endif
	  if(res != UDM_OK){
		UdmLog(A, UDM_LOG_ERROR, db->errstr);
	  }
	  UDM_RELEASELOCK(A, UDM_LOCK_DB);
	  if (res != UDM_OK) break;
	}
	return res;
}

static const int search_cache_size=1000;

int UdmFindWords(UDM_AGENT *A, UDM_RESULT *Res) {
	const char      *cache_mode  = UdmVarListFindStr(&A->Conf->Vars, "Cache", "no");
	UDM_DB		*db;
	size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
	int res = UDM_OK;
	size_t		nwrd=0;
	size_t		nwrdX[256];
	size_t          *PerSite[256], *persite, *curpersite;
	size_t          ResultsLimit = UdmVarListFindUnsigned(&A->Conf->Vars, "ResultsLimit", 0);
	UDM_URL_CRD	*wrdX[256];
	UDM_URL_CRD	*wrd = NULL, *curwrd = NULL;
	UDM_URLDATA     *udtX[256];
	UDM_URLDATA     *udt = NULL, *curudt = NULL;

	if( strcasecmp(cache_mode, "yes") || UdmSearchCacheFind(A, Res) ){
		/* If not found in search cache       */
		/* Let's do actual search now:        */
		/* Get groupped by url_id words array */
	
	  for (i = dbfrom; i < dbto; i++) {
	    db = &A->Conf->dbl.db[i];
	    UdmLog(A, UDM_LOG_DEBUG, "UdmFind for %s", db->DBADDR);
	    Res->CoordList.Coords = NULL;
	    Res->CoordList.Data = NULL;
	    Res->CoordList.ncoords = 0;
	    Res->total_found = 0;
	    switch(db->DBDriver){
		case UDM_DB_SEARCHD:
		  res = UdmFindWordsSearchd(A, Res, db);
			break;
#ifdef HAVE_SQL
		default:
		  if (db->DBMode == UDM_DBMODE_BLOB) {
		    res = UdmFindWordsBlob(A, Res, db);
		  } else if (db->DBMode == UDM_DBMODE_CACHE) {
		    res = UdmFindWordsCache(A, Res, db);
		  } else {
		    res = UdmFindWordsSQL(A, Res, db);
		  }
			break;
#endif
	    }
	    wrdX[i] = Res->CoordList.Coords;
	    udtX[i] = Res->CoordList.Data;
	    nwrdX[i] = Res->total_found;
	    nwrd += Res->total_found;
	    if ((PerSite[i] = Res->PerSite) == NULL) {
	      PerSite[i] = (nwrdX[i]) ? (size_t*)UdmXmalloc(sizeof(size_t) * nwrdX[i]) : NULL;
	    }
	  }

	  if (nwrd > 0) {
	    curwrd=wrd=(UDM_URL_CRD*)malloc(sizeof(*wrd)*nwrd);
	    curudt=udt=(UDM_URLDATA*)malloc(sizeof(*udt)*nwrd);
	    Res->PerSite = persite = curpersite = (size_t*)malloc(sizeof(size_t) * nwrd);
	    for (i = dbfrom; i < dbto; i++) {
		if(wrdX[i]){
/*			size_t j;*/
			/* Set machine number */
/*			for(j=0;j<nwrdX[i];j++){
				wrdX[i][j].coord = (wrdX[i][j].coord << 8) + (i & 255);
			}*/

			memcpy(curwrd,wrdX[i],sizeof(*curwrd)*nwrdX[i]);
			curwrd+=nwrdX[i];
			UDM_FREE(wrdX[i]);
			memcpy(curpersite, PerSite[i], sizeof(size_t) * nwrdX[i]);
			curpersite += nwrdX[i];
			UDM_FREE(PerSite[i]);
			if (udtX[i] != NULL) {
			  memcpy(curudt, udtX[i], sizeof(*curudt) * nwrdX[i]);
			} else {
			  bzero(curudt, sizeof(*curudt) * nwrdX[i]);
			}
			curudt += nwrdX[i];
			UDM_FREE(udtX[i]);
		}
	    }
/*	    if (Res->offset) UdmSortSearchWordsByWeight(wrd,nwrd);*/
	  }

	  Res->total_found = Res->CoordList.ncoords = nwrd;
	  Res->CoordList.Coords = wrd;
	  Res->CoordList.Data = udt;
	  Res->num_rows=Res->CoordList.ncoords;
	
	  if (dbto - dbfrom > 1) {
	    int		use_site_id = ((!strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "GroupBySite", "no"), "yes"))
				       && (UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0));
	    if (use_site_id) {
	      UdmSortSearchWordsBySite(&Res->CoordList, Res->CoordList.ncoords);
	      UdmGroupBySite(A, Res);
	    }
	    UdmSortSearchWordsByPattern(Res, &Res->CoordList, Res->CoordList.ncoords, UdmVarListFindStr(&A->Conf->Vars, "s", "RP"));
	    Res->total_found = Res->CoordList.ncoords;
	    
	  }

	  if (ResultsLimit > 0 && ResultsLimit < Res->total_found) {
	    Res->total_found = Res->CoordList.ncoords = ResultsLimit;
	  }

	  if((!strcasecmp(cache_mode,"yes"))&&(search_cache_size>-1)){
			fflush(stdout);
			fflush(stderr);
			UdmSearchCacheStore(A, Res);
	  }
	}

  return res;
}

UDM_RESULT * __UDMCALL UdmFind(UDM_AGENT *A) {
	UDM_DB		*db;
	UDM_RESULT	*Res;
	int		res=UDM_OK;
	unsigned long	ticks=UdmStartTimer(), ticks_;
	size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
	int		page_number = UdmVarListFindInt(&A->Conf->Vars, "np", 0);
	int		page_size   = UdmVarListFindInt(&A->Conf->Vars, "ps", 20);
	char		str[128];
	
	UdmLog(A,UDM_LOG_DEBUG,"Start UdmFind");
	
	/* Allocate result */
	Res=UdmResultInit(NULL);
	UdmPrepare(A, Res);				/* Prepare query    */
	
	UdmVarListAddStr(&A->Conf->Vars, "orig_m", UdmVarListFindStr(&A->Conf->Vars, "m", "all"));

	UdmFindWords(A, Res);

	UdmVarListReplaceStr(&A->Conf->Vars, "m", UdmVarListFindStr(&A->Conf->Vars, "orig_m", "all"));
	UdmVarListDel(&A->Conf->Vars, "orig_m");

	Res->first = page_number * page_size;	
	if(Res->first >= Res->total_found) Res->first = (Res->total_found)? (Res->total_found - 1) : 0;
	if((Res->first+page_size)>Res->total_found){
		Res->num_rows=Res->total_found-Res->first;
	}else{
		Res->num_rows=page_size;
	}
	Res->last=Res->first+Res->num_rows-1;

	/* Allocate an array for documents information */
	if (Res->num_rows > 0) Res->Doc = (UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT) * (Res->num_rows));
	
	/* Copy url_id and coord to result */
	for(i=0;i<Res->num_rows;i++){
		uint4	score = Res->CoordList.Coords[i + Res->first /* * Res->offset*/].coord;
		UdmDocInit(&Res->Doc[i]);
		UdmVarListReplaceInt(&Res->Doc[i].Sections, "ID", Res->CoordList.Coords[i + Res->first /* * Res->offset*/].url_id);
		udm_snprintf(str, 128, "%.3f%%", ((double)(score /*>> 8*/)) / 1000);
		UdmVarListReplaceStr(&Res->Doc[i].Sections, "Score", str);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(i + Res->first + 1));
		UdmVarListReplaceInt(&Res->Doc[i].Sections, "sdnum", (int)(score & 0xFF));
	}
	
	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
	  switch(db->DBDriver){
		case UDM_DB_SEARCHD:
		  res = UdmResAddDocInfoSearchd(A, db, Res, i);
		  break;
#ifdef HAVE_SQL
		default:
		  res = UdmResAddDocInfoSQL(A, db, Res, i);
			break;
#endif
	  }
	}

	ticks_=UdmStartTimer();
	UdmLog(A, UDM_LOG_DEBUG, "Start Clones");
	
	if(!strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "DetectClones", "yes"), "yes")) {
		size_t num=Res->num_rows;
		for(i=0;i<num;i++){
			UDM_RESULT *Cl = UdmCloneList(A, &Res->Doc[i]);
			if(Cl){
				Res->Doc=(UDM_DOCUMENT*)realloc(Res->Doc,sizeof(UDM_DOCUMENT)*(Res->num_rows+Cl->num_rows));
				memcpy(&Res->Doc[Res->num_rows],Cl->Doc,sizeof(UDM_DOCUMENT)*Cl->num_rows);
				Res->num_rows+=Cl->num_rows;
				UDM_FREE(Cl->Doc);
				UdmResultFree(Cl);
			}
		}
	}
	ticks_ = UdmStartTimer() - ticks_;
	UdmLog(A, UDM_LOG_DEBUG, "Stop  Clones: %.2f", (float)ticks_/1000);

	/* first and last begins from 0, make it begin from 1 */
	Res->first++;
	Res->last++;
	for(i=0;i<Res->num_rows;i++){
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(Res->first+i));
	}
/******************************/

	UdmConvert(A->Conf, Res, A->Conf->lcs, A->Conf->bcs);
	Res->work_time=ticks=UdmStartTimer()-ticks;
	UdmLog(A,UDM_LOG_DEBUG,"Done  UdmFind %.2f",(float)ticks/1000);

	UdmTrack(A, Res);
	
	if(res!=UDM_OK){
		UdmResultFree(Res);
		Res=NULL;
	}
	return Res;
}


static int UdmStr2DBMode(const char * str1){
	int m = -1;
	if(!strncasecmp(str1,"multi-crc",9))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!strncasecmp(str1,"crc-multi",9))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!strncasecmp(str1,"single",6))m=UDM_DBMODE_SINGLE;
	else
	if(!strncasecmp(str1,"crc",3))m=UDM_DBMODE_SINGLE_CRC;
	else
	if(!strncasecmp(str1,"multi",5))m=UDM_DBMODE_MULTI;
	else
	if(!strncasecmp(str1,"word2url",8))m=UDM_DBMODE_WORD2URL;
	else
	if(!strncasecmp(str1,"cache",5))m=UDM_DBMODE_CACHE;
	else
	if(!strncasecmp(str1,"blob",4))m=UDM_DBMODE_BLOB;
	return(m);
}

int UdmDBSetAddr(UDM_DB *db, const char *dbaddr, int mode){
	char	*s, *cport;
	char *stored_host = NULL, *cached_host = NULL;
	struct hostent *hp;
	struct sockaddr_in server_addr;
	int nport;
	
	if (!dbaddr) return UDM_ERROR;

	if(UdmURLParse(&db->addr,dbaddr))return UDM_ERROR;
	UDM_FREE(db->DBADDR);
	UDM_FREE(db->DBName);
	UDM_FREE(db->DBUser);
	UDM_FREE(db->DBPass);
	UDM_FREE(db->DBSock);
	UDM_FREE(db->where);
	UDM_FREE(db->from);

	db->open_mode=mode;
	db->DBADDR = (char*)strdup(dbaddr);

	if (db->addr.schema == NULL) return UDM_ERROR;
	if(!strcasecmp(db->addr.schema,"cached")){
		db->DBType = UDM_DB_CACHED;
	}
	else if(!strcasecmp(db->addr.schema,"cache")){
		db->DBType = UDM_DB_CACHE;
	}
	else if(!strcasecmp(db->addr.schema,"searchd")){
		db->DBType=UDM_DB_SEARCHD;
		if (UDM_OK != UdmSearchdConnect(db)) return UDM_ERROR;
	}
#if (HAVE_MSQL||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"msql")){
		db->DBType=UDM_DB_MSQL;
		db->DBSQL_LIMIT=1;
	}
#endif
#if (HAVE_SOLID||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"solid")){
		db->DBType=UDM_DB_SOLID;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_ORACLE7||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"oracle7")){
		db->DBType=UDM_DB_ORACLE7;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
		db->DBSQL_SUBSELECT = 1;
	}
#endif
#if (HAVE_ORACLE8||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"oracle8")){
		db->DBType=UDM_DB_ORACLE8;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
		db->DBSQL_SUBSELECT = 1;
	}
	else if(!strcasecmp(db->addr.schema,"oracle")){
		db->DBType=UDM_DB_ORACLE8;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
		db->DBSQL_SUBSELECT = 1;
	}
#endif
#if (HAVE_CTLIB||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"mssql")){
		db->DBType=UDM_DB_MSSQL;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
	}
#endif
#if (HAVE_MYSQL||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"mysql")){
		db->DBType=UDM_DB_MYSQL;
		db->DBSQL_IN=1;
		db->DBSQL_LIMIT=1;
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_PGSQL||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"pgsql")){
		db->DBType=UDM_DB_PGSQL;
		db->DBSQL_IN=1;
		db->DBSQL_LIMIT=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_SELECT_FROM_DELETE=0;
		db->DBSQL_SUBSELECT = 1;
	}
#endif
#if (HAVE_IBASE||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"ibase")){
		db->DBType=UDM_DB_IBASE;
		
		/* 
		while indexing large sites and using the SQL in statement 
		interbase will fail when the items in the in IN statements
		are more then 1500. We'd better have to fix code to avoid 
		big INs instead of hidding DBSQL_IN.
		*/
		
		/* db->DBSQL_IN=1; */ 
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_SQLITE)
	else if (!strcasecmp(db->addr.schema,"sqlite")){
		db->DBType=UDM_DB_SQLITE;
		db->DBSQL_IN=1;
		db->DBSQL_LIMIT=1;
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_SAPDB||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"sapdb")){
		db->DBType=UDM_DB_SAPDB;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_DB2||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"db2")){
		db->DBType=UDM_DB_DB2;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}
#endif
#if (HAVE_DB_ACCESS||HAVE_ODBC)
	else if(!strcasecmp(db->addr.schema,"access")){
		db->DBType=UDM_DB_ACCESS;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_SUBSELECT = 1;
	}
#endif
	else {
		return UDM_ERROR;
	}
	
	db->DBDriver=db->DBType;
	
	if((s = strchr(UDM_NULL2EMPTY(db->addr.filename), '?'))) {
		char * tok, *lt;
		
		*s++='\0';
		tok = udm_strtok_r(s,"&",&lt);
		while(tok){
			char * val;
			
			if((val=strchr(tok,'='))){
				*val++='\0';
				if(!strcasecmp(tok,"socket")&&val[0]){
					UDM_FREE(db->DBSock);
					db->DBSock = (char*)strdup(val);
				}else
				if(!strcasecmp(tok,"numtables")&&val[0]){
					db->numtables=atoi(val);
					if(!db->numtables)
						db->numtables=1;
				}else
				if(!strcasecmp(tok,"dbmode")&&val[0]){
				  if ((db->DBMode=UdmStr2DBMode(val)) < 0) {
				    return UDM_ERROR;
				  }
				}else
				if(!strcasecmp(tok, "stored") && val[0]){
				  stored_host = (char*)strdup(val);
				}else
				if(!strcasecmp(tok, "cached") && val[0]){
				  cached_host = (char*)strdup(val);
				}
			}
			if (!strcasecmp(tok, "trackquery")) {
			  db->TrackQuery = 1;
			}
			tok = udm_strtok_r(NULL,"&",&lt);
		}
	}
	
	if(db->DBType==UDM_DB_IBASE || db->DBType==UDM_DB_SQLITE){
		/* Ibase is a special case        */
		/* It's database name consists of */
		/* full path and file name        */ 
		db->DBName = (char*)strdup(UDM_NULL2EMPTY(db->addr.path));
	}else{
		db->DBName = (char*)strdup(UDM_NULL2EMPTY(db->addr.path));
		sscanf(UDM_NULL2EMPTY(db->addr.path), "/%[^/]s", db->DBName);
	}
	if((s=strchr(UDM_NULL2EMPTY(db->addr.auth),':'))){
		*s=0;
		db->DBUser = (char*)strdup(db->addr.auth);
		db->DBPass = (char*)strdup(s+1);
		*s=':';
	}else{
		db->DBUser = (char*)strdup(UDM_NULL2EMPTY(db->addr.auth));
	}

	if (stored_host) {
	  if((cport = strchr(stored_host, ':'))) {
	    *cport = '\0';
	    nport = atoi(cport + 1);
	  } else {
	    nport = UDM_STORED_PORT;
	  }

	  if ((hp = gethostbyname(stored_host)) == 0 ) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "StoreOpen ERR gethostbyname: %s", hstrerror(h_errno));
	    return UDM_ERROR;
	  }
	  bzero((void*)&server_addr, sizeof(server_addr));
	  memmove(&server_addr.sin_addr, hp->h_addr, (size_t)hp->h_length);
	  server_addr.sin_family = hp->h_addrtype;
	  server_addr.sin_port = htons((u_short)nport);
	  if((db->stored_sd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "StoreOpen ERR socket: %s", strerror(errno));
	    return UDM_ERROR;
	  }
  
	  if(connect(db->stored_sd, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "StoreOpen ERR connect to %s: %s", inet_ntoa(server_addr.sin_addr), strerror(errno));
	    return UDM_ERROR;
	  }
	}

	if (cached_host) {
	  struct timeval tval;

	  if((cport = strchr(cached_host, ':')) !=NULL ) {
	    *cport = '\0';
	    nport = atoi(cport + 1);
	  } else {
	    nport = UDM_LOGD_PORT;
	  }

	  if ((hp = gethostbyname(cached_host ? cached_host : db->addr.hostname)) == 0 ) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "CachedOpen ERR gethostbyname: %s", hstrerror(h_errno));
	    return UDM_ERROR;
	  }
	  bzero((void*)&server_addr, sizeof(server_addr));
	  memmove(&server_addr.sin_addr, hp->h_addr, (size_t)hp->h_length);
	  server_addr.sin_family = hp->h_addrtype;
	  server_addr.sin_port = htons((u_short)nport);
	  if((db->cached_sd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "CachedOpen ERR socket: %s", strerror(errno));
	    return UDM_ERROR;
	  }
  
	  tval.tv_sec = 300;
	  tval.tv_usec = 0;
#if !defined(sgi) && !defined(__sgi) && !defined(__irix__)
	  if (setsockopt(db->cached_sd, SOL_SOCKET, SO_SNDTIMEO, (char *)&tval, sizeof(tval)) != 0){
	        fprintf(stderr, "%s [%d] setsockopt error: %d (%s)\n", __FILE__, __LINE__, errno, strerror(errno));
		return UDM_ERROR;
	  }
#endif
	  if(connect(db->cached_sd, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
	    UDM_FREE(stored_host); UDM_FREE(cached_host);
	    fprintf(stderr, "CachedOpen ERR connect to %s: %s", inet_ntoa(server_addr.sin_addr), strerror(errno));
	    return UDM_ERROR;
	  }
	}

	UDM_FREE(stored_host); UDM_FREE(cached_host);
	return UDM_OK;
}


__C_LINK int __UDMCALL UdmStatAction(UDM_AGENT *A, UDM_STATLIST *S){
	UDM_DB	*db;
	int	res=UDM_ERROR;
	size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
	
	bzero((void*)S, sizeof(S[0]));

	for (i = dbfrom; i < dbto; i++) {
	  db = &A->Conf->dbl.db[i];
/*	  if(db->DBDriver == UDM_DB_CACHE)
		res = UdmStatActionCache(A, S, db); */ /* FIXME: usr this after it implementation */
#ifdef HAVE_SQL
	  res = UdmStatActionSQL(A, S, db);
#endif
	  if (res != UDM_OK) break;
	}
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
	}
	return res;
}

unsigned int UdmGetCategoryId(UDM_ENV *Conf, char *category) {
	UDM_DB	*db;
	unsigned int rc = 0;
	size_t i, dbfrom = 0, dbto =  Conf->dbl.nitems;

	for (i = dbfrom; i < dbto; i++) {
	  db = &Conf->dbl.db[i];
#ifdef HAVE_SQL
	  rc = UdmGetCategoryIdSQL(Conf, category, db);
	  if (rc != 0) return rc;
#endif
	}
	return rc;
}


int UdmTrack(UDM_AGENT * query, UDM_RESULT *Res) {
  int rc = UDM_OK;
  size_t i, dbfrom = 0, dbto =  query->Conf->dbl.nitems; 
  UDM_DB *db;

	for (i = dbfrom; i < dbto; i++) {
	  db = &query->Conf->dbl.db[i];
	  if(db->TrackQuery) {
#ifdef HAVE_SQL
	    rc = UdmTrackSQL(query, Res, db);
#endif
	  }
	}
	  return rc;
}

UDM_RESULT * UdmCloneList(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc) {
	size_t i, dbfrom = 0, dbto =  Indexer->Conf->dbl.nitems;
	UDM_DB		*db;
	UDM_RESULT	*Res;
	int		rc = UDM_OK;

	Res = UdmResultInit(NULL);

	  for (i = dbfrom; i < dbto; i++) {
	    db = &Indexer->Conf->dbl.db[i];
	    switch(db->DBDriver) {
	    case UDM_DB_SEARCHD:
		  rc = UdmCloneListSearchd(Indexer, Doc, Res, db);
			break;
#ifdef HAVE_SQL
	    default:
	      rc = UdmCloneListSQL(Indexer, Doc, Res, db);
	      break;
#endif
	    }
	    if (rc != UDM_OK) break;
	  }
	  return NULL;
}


/********************************************************/


UDM_DBLIST * UdmDBListInit(UDM_DBLIST * List){
	bzero((void*)List, sizeof(*List));
	return(List);
}

size_t UdmDBListAdd(UDM_DBLIST *List, const char * addr, int mode){
	UDM_DB	*db;
	int res;
	db=List->db=(UDM_DB*)realloc(List->db,(List->nitems+1)*sizeof(UDM_DB));
	db+=List->nitems;
	UdmDBInit(db);
	res = UdmDBSetAddr(db, addr, mode);
	if (res == UDM_OK) List->nitems++;
	return res;
}

void UdmDBListFree(UDM_DBLIST *List){
	size_t	i;
	UDM_DB	*db=List->db;
	
	for(i = 0; i < List->nitems; i++){
		UdmDBFree(&db[i]);
	}
	UDM_FREE(List->db);
	UdmDBListInit(List);
}
