/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <udm_config.h>

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_ZLIB
#include <zlib.h>

#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <math.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_env.h"
#include "udm_agent.h"
#include "udm_conf.h"
#include "udm_log.h"
#include "udm_services.h"
#include "udm_store.h"
#include "udm_vars.h"
#include "udm_xmalloc.h"
#include "udm_vars.h"
#include "udm_parsehtml.h"
#include "udm_searchtool.h"
#include "udm_utils.h"
#include "udm_crc32.h"
#include "udm_cache.h"
#include "udm_signals.h"

#define UDM_STORED_MAXCLNTS 128

static char pidname[1024];
static char time_pid[100];
static UDM_ENV * Conf = NULL;
static UDM_AGENT * Agent = NULL;
static int verb = UDM_LOG_EXTRA;


static void exitproc(void){
	unlink(pidname);
}

static char * time_pid_info(void){
	struct tm * tim;
	time_t t;
	t=time(NULL);
	tim=localtime(&t);
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

/********************* SIG Handlers ****************/

static void sighandler(int sign);
static void Optimize_sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
#ifdef WIN32
#else
	UdmSignal(SIGPIPE, sighandler);
	UdmSignal(SIGCHLD, sighandler);
	UdmSignal(SIGALRM, sighandler);
#endif
	UdmSignal(SIGHUP, sighandler);
	UdmSignal(SIGINT, sighandler);
	UdmSignal(SIGTERM, sighandler);
}
static void Optimize_init_signals(void){
	/* Set up signals handler*/
#ifdef WIN32
#else
	UdmSignal(SIGPIPE, Optimize_sighandler);
	UdmSignal(SIGCHLD, Optimize_sighandler);
	UdmSignal(SIGALRM, Optimize_sighandler);
#endif
	UdmSignal(SIGHUP, Optimize_sighandler);
	UdmSignal(SIGINT, Optimize_sighandler);
	UdmSignal(SIGTERM, Optimize_sighandler);
}

static void sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif

	switch(sign){
		case SIGPIPE:
		  have_sigpipe = 1;
		  break;
		case SIGHUP:
		  have_sighup = 1;
		  break;
	        case SIGCHLD:
		  while (waitpid(-1, &status, WNOHANG) > 0);
		  break;
		case SIGINT:
		  have_sigint=1;
		  break;
		case SIGTERM:
		  have_sigterm=1;
		  break;
	        case SIGALRM:
		  _exit(0);
		  break;
		default: ;
	}
}

static void Optimize_sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif

	switch(sign){
		case SIGPIPE:
		  have_sigpipe = 1;
		  break;
		case SIGHUP:
		  have_sighup = 1;
		  break;
	        case SIGCHLD:
		  wait(&status);
		  break;
		case SIGINT:
		  have_sigint = 1;
		  break;
		case SIGTERM:
		  have_sigterm = 1;
		  break;
	        case SIGALRM:
		  have_sigalrm = 1;
		  break;
		default: ;
	}
}



static int cmpsi(const void *s1, const void *s2) {
  const UDM_SORTSTOREITEM *d1 = (const UDM_SORTSTOREITEM*)s1;
  const UDM_SORTSTOREITEM *d2 = (const UDM_SORTSTOREITEM*)s2;
  if ((d1->Item.offset) < (d2->Item.offset)) return -1;
  if ((d1->Item.offset) > (d2->Item.offset)) return 1;
  return 0;
}


size_t  DocSize = 0, filenamelen;
Byte *Doc = NULL, *CDoc = NULL;
int ns, StoredFiles, OptimizeInterval, OptimizeRatio;
UDM_STORE_PARAM P;


#define ABORT(x)    UDM_FREE(Doc); \
                    UDM_FREE(CDoc); \
                    UdmCloseBase(&P); \
		    if (x) exit (0); else continue;



static void Optimize(UDM_ENV *C) {
  unsigned int current_base = 0;

	Optimize_init_signals();
	alarm((unsigned int) OptimizeInterval); /* try optimize one base every OptimizeInterval sec. */
	while(1) {
	  sleep(1);
		if(have_sigint){
			UdmLog(Agent, verb, "Optimize: SIGINT arrived");
			have_sigint = 0;
			break;
		}
		if(have_sigterm){
			UdmLog(Agent, verb, "Optimize: SIGTERM arrived");
			have_sigterm = 0;
			break;
		}
		if(have_sigpipe){
			UdmLog(Agent,verb,"Optimize: SIGPIPE arrived. Broken pipe !");
			have_sigpipe = 0;
			break;
		}
		if (have_sigalrm) {
		        long ActualSize = 0, SSize, diff, gain, i, j, nitems = 0, pos, posold;
			size_t nread;
			double dr = 0;
			char buffer[BUFSIZ];

			bzero(&P, sizeof(P));
			P.subdir = "store";
			P.BASE_SIG = UDM_STORE_SIG;
			P.basename = "";
			P.indname = "";
			P.rec_id = current_base;
			if (UdmOpenBase(Agent, &P, UDM_WRITE_LOCK) != UDM_OK) {
			  UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
			  closesocket(ns);
			  ABORT(1);
			}
			fseek(P.Ifd, UDM_SIG_LEN, SEEK_SET);
			while(fread(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) == 1) {
			  if (P.Item.rec_id != 0) {
			    ActualSize += (long)P.Item.size;
			    nitems++;
			  }
			}
			fseek(P.Sfd, 0, SEEK_END);
			SSize = ftell(P.Sfd) - UDM_SIG_LEN;
			dr = (nitems) ? fabs(100.0 * (SSize - ActualSize) / SSize) : 0.0,
			UdmLog(Agent,verb,"Optimize: base 0x%X, %ld files stored, defragmentation: %5.2f%%, Data size: %ld File size: %ld", current_base,
			       nitems, dr,
			       ActualSize, SSize
			       );
			if (dr > (float)OptimizeRatio) {
			  UDM_SORTSTOREITEM *si = (UDM_SORTSTOREITEM*)malloc(nitems * sizeof(UDM_SORTSTOREITEM));
			  fseek(P.Ifd, UDM_SIG_LEN, SEEK_SET);
			  i = 0;
			  pos = UDM_SIG_LEN;
			  while(fread(&P.Item, sizeof(UDM_STOREITEM), 1, P.Ifd) == 1) {
			    if(P.Item.rec_id != 0) {
			      si[i].offset = pos;
			      memcpy(&si[i].Item, &P.Item, sizeof(UDM_STOREITEM));
			      i++;
			    }
			    pos = ftell(P.Ifd);
			  }

			  UdmSort((void*)si, (size_t)nitems, sizeof(UDM_SORTSTOREITEM), cmpsi);

			  gain = 0;
			  pos = UDM_SIG_LEN;
			  posold = (nitems) ? si[0].Item.offset : UDM_SIG_LEN;
			  if ((diff = (posold - pos)) > 0) {
			    for(
				fseek(P.Sfd, posold, SEEK_SET);
				(nread = fread(buffer, 1, BUFSIZ, P.Sfd)) > 0;
				fseek(P.Sfd, posold, SEEK_SET)
				) {
			      fseek(P.Sfd, pos, SEEK_SET);
			      fwrite(buffer, 1, nread, P.Sfd);
			      posold += (long)nread;
			      pos += (long)nread;
			    }
			    fseek(P.Sfd, 0, SEEK_END);
			    SSize = ftell(P.Sfd);
			    ftruncate(fileno(P.Sfd), (off_t)(SSize - diff));
			    for (j = 0; j < nitems; j++) si[j].Item.offset -= diff;
			    gain += diff;
			  }
			    
			  for (i = 0; i < nitems - 1; i++) {
			    pos = si[i].Item.offset + si[i].Item.size;
			    posold = si[i + 1].Item.offset;
			    if ((diff = posold - pos) > 0) {
			      for(
				  fseek(P.Sfd, posold, SEEK_SET);
				  (nread = fread(buffer, 1, BUFSIZ, P.Sfd)) > 0;
				  fseek(P.Sfd, posold, SEEK_SET)
				  ) {
				fseek(P.Sfd, pos, SEEK_SET);
				fwrite(buffer, 1, nread, P.Sfd);
				posold += (long)nread;
				pos += (long)nread;
			      }
			      fseek(P.Sfd, 0, SEEK_END);
			      SSize = ftell(P.Sfd);
			      ftruncate(fileno(P.Sfd), (off_t)(SSize - diff));
			      for (j = i + 1; j < nitems; j++) si[j].Item.offset -= diff;
			      gain += diff;
			    }
			  }
			  fseek(P.Sfd, 0, SEEK_END);
			  pos = (nitems) ? si[nitems - 1].Item.offset + si[nitems - 1].Item.size : UDM_SIG_LEN;
			  posold = ftell(P.Sfd);
			  if (posold > pos) {
			    ftruncate(fileno(P.Sfd), (off_t)(pos));
			    gain += (posold - pos);
			  }

			  if (gain > 0) {
			    for (i = 0; i < nitems; i++) {
			      fseek(P.Ifd, si[i].offset, SEEK_SET);
			      fwrite(&(si[i].Item), sizeof(UDM_STOREITEM), 1, P.Ifd);
			    }
			    UdmLog(Agent,verb,"Optimize: base 0x%X cleaned, %ld bytes fried", current_base, gain);
			  }
			    
			  UDM_FREE(si);
			}

			UdmCloseBase(&P);
			current_base++;
			current_base %= UdmVarListFindInt(&C->Vars, "StoredFiles", 0x10000);
			have_sigalrm = 0;
			alarm((unsigned int) OptimizeInterval);
		}
	}
}

static void usage(void){

	fprintf(stderr,
"\n\
stored from %s-%s-%s\n\
http://www.mnogosearch.org/ (C) 1998-2002, LavTech Corp.\n\
\n\
Usage: stored [OPTIONS] [configfile]\n\
\n\
Options are:\n\
  -w /path      choose alternative working /var directory\n\
  -p xxx        listen port xxx\n\
  -v n          verbose level, 0-5\n\
  -h,-?         print this help page and exit\n\
\n\
Please post bug reports and suggestions to http://www.mnogosearch.org/bugs/\n",
	PACKAGE,VERSION,UDM_DBTYPE);

	return;
}

#endif

int main(int argc, char **argv) {

#ifdef HAVE_ZLIB

        fd_set mask;
	struct sockaddr_in server_addr, client_addr;
	const char *config_name= UDM_CONF_DIR "/stored.conf";
	int pid, pid_fd, nport, s, on = 1, ch;
	char pidbuf[1024];
	char buf[1024];
	char var_dir[1024] = UDM_VAR_DIR UDMSLASHSTR;
	int log2stderr=1;
	const char *lstn;

	/* Check that another instance isn't running */
	/* and create PID file.                      */

	sprintf(pidname,"%s/%s",UDM_VAR_DIR,"stored.pid");
	pid_fd = open(pidname,O_CREAT|O_EXCL|O_WRONLY,0644);
	if(pid_fd < 0){
		fprintf(stderr,"%s Can't create '%s': %s\n", time_pid_info(), pidname, strerror(errno));
		if(errno == EEXIST){
			fprintf(stderr,"It seems that another stored is already running!\n");
			fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
 		}
		exit(1);
	}

	sprintf(pidbuf,"%d\n",(int)getpid());
	write(pid_fd,&pidbuf,strlen(pidbuf));

	atexit(&exitproc);
	init_signals();
	UdmInit(); /* Initialize library */

	Conf = UdmEnvInit(NULL);

	nport = UDM_STORED_PORT;

	while ((ch = getopt(argc, argv, "hv:w:p:?")) != -1){
		switch (ch) {
			case 'p':
				nport = atoi(optarg);
				break;
		        case 'v': UdmSetLogLevel(NULL, atoi(optarg)); break;
			case 'w':
				strcpy(var_dir,optarg);
				break;
		        case 'l':
			        Conf->logs_only = 1;
				break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;
	
	if(argc > 1){
		usage();
		UdmEnvFree(Conf);
		unlink(pidname);
		exit(1);
	} else if (argc == 1) {
	        config_name = argv[0];
	}
	
	Agent = UdmAgentInit(NULL, Conf, 0);

	if (UDM_OK != UdmEnvLoad(Agent, config_name, 0)) {
		fprintf(stderr,"%s\n",UdmEnvErrMsg(Conf));
		UdmEnvFree(Conf);
		unlink(pidname);
		exit(1);
	}

	StoredFiles = UdmVarListFindInt(&Conf->Vars, "StoredFiles", 0x10000);
	OptimizeInterval = UdmVarListFindInt(&Conf->Vars, "OptimizeInterval", 600);
	OptimizeRatio = UdmVarListFindInt(&Conf->Vars, "OptimizeRatio", 5);

	UdmOpenLog("stored",Conf,log2stderr);

	UdmLog(Agent, UDM_LOG_ERROR, "stored started on port %d using config file %s", nport, config_name);
	UdmLog(Agent, verb, "StoredFiles     : %d (0x%x)", StoredFiles, StoredFiles);
	UdmLog(Agent, verb, "OptimizeInterval: %d sec.", OptimizeInterval);
	UdmLog(Agent, verb, "OptimizeRatio   : %d %%", OptimizeRatio);

	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	  UdmLog(Agent, UDM_LOG_ERROR, "%s socket() error %d",time_pid_info(),errno);
	  UdmAgentFree(Agent);
	  UdmEnvFree(Conf);
	  unlink(pidname);
	  exit(1);
	}
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
		UdmLog(Agent, UDM_LOG_ERROR, "%s setsockopt() error %d",time_pid_info(),errno);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		unlink(pidname);
		exit(1);
	}

	bzero((void*)&server_addr, sizeof(server_addr));
	server_addr.sin_family = AF_INET;
	if((lstn = UdmVarListFindStr(&Agent->Conf->Vars, "Listen", NULL))) {
	  char * cport;
			
	  if((cport=strchr(lstn,':'))){
	    UdmLog(Agent,verb,"Listening '%s'",lstn);
	    *cport='\0';
	    server_addr.sin_addr.s_addr = inet_addr(lstn);
	    nport=atoi(cport+1);
	  }else{
	    nport=atoi(lstn);
	    server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	    UdmLog(Agent,verb,"Listening port %d",nport);
	  }
	}else{
	  UdmLog(Agent,verb,"Listening port %d",nport);
	  server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	}
	server_addr.sin_port = htons((u_short)nport);
	

	if (bind(s, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
		UdmLog(Agent, UDM_LOG_ERROR, "%s bind() error %d %s", time_pid_info(), errno, strerror(errno));
		unlink(pidname);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		exit(1);
	}

	if (listen(s, UDM_STORED_MAXCLNTS) == -1) {
		UdmLog(Agent, UDM_LOG_ERROR, "%s listen() error %d %s", time_pid_info(), errno, strerror(errno));
		unlink(pidname);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		exit(1);
	}
	
	FD_ZERO(&mask);
	FD_SET(s, &mask);


	if ((pid = fork() ) == -1) {
	  UdmLog(Agent, UDM_LOG_ERROR, "%s fork() error %d %s", time_pid_info(), errno, strerror(errno));
	  unlink(pidname);
	  exit(1);
	}
	if (pid == 0) { /* child process */
	  Optimize(Conf);
	  UdmAgentFree(Agent);
	  UdmEnvFree(Conf);
	  exit(0);
	}
	UdmLog(Agent,verb,"Optimization child started.");



	while(1) {
#ifdef __irix__
 	        int addrlen;
#else
		socklen_t addrlen;
#endif
		int sel;
		struct timeval tval;
		fd_set msk;
	  	
		tval.tv_sec = 300;
		tval.tv_usec = 0;
		msk = mask;
		sel = select(UDM_STORED_MAXCLNTS, &msk, 0, 0, &tval);
		
		if(have_sighup){
			UdmLog(Agent,verb,"SIGHUP arrived. Reloading config.");
			have_sighup=0;
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			Conf = UdmEnvInit(NULL);
			
			if(UDM_OK!=UdmEnvLoad(Conf,config_name,0)) {
				UdmLog(Agent, UDM_LOG_ERROR, "%s", UdmEnvErrMsg(Conf));
				UdmAgentFree(Agent);
				UdmEnvFree(Conf);
				unlink(pidname);
				exit(1);
			}
			Agent = UdmAgentInit(NULL, Conf, 0);
		}
		if(have_sigint){
			UdmLog(Agent,verb,"SIGINT arrived");
			have_sigint=0;
			break;
		}
		if(have_sigterm){
			UdmLog(Agent,verb,"SIGTERM arrived");
			have_sigterm=0;
			break;
		}
		if(have_sigpipe){
			UdmLog(Agent,verb,"SIGPIPE arrived. Broken pipe !");
			have_sigpipe=0;
			break;
		}
		
		if(sel==0)continue;
		if(sel==-1){
			switch(errno){
				case EINTR:	/* Child */
					break;
			        default:
					UdmLog(Agent,verb,"FIXME select error %d %s",errno,strerror(errno));
			}
			continue;
		}
		
		bzero((void*)&client_addr, addrlen = sizeof(client_addr));
	
	  if ((ns = accept(s, (struct sockaddr *) &client_addr, &addrlen)) == -1) {
	    UdmLog(Agent, UDM_LOG_ERROR, "%s accept() error %d %s", time_pid_info(), errno, strerror(errno));
	    unlink(pidname);
	    exit(1);
	  }

#define Client inet_ntoa(client_addr.sin_addr)

	  UdmLog(Agent, verb, "[%s] Accept", Client);
    
	  if ((pid = fork() ) == -1) {
	    UdmLog(Agent, UDM_LOG_ERROR, "%s fork() error %d %s", time_pid_info(), errno, strerror(errno));
	    unlink(pidname);
	    exit(1);
	  }

	  if (FD_ISSET(s, &msk)) {

	    if (pid == 0) { /* child process */

	      close(s);

	      alarm(3600); /* 60 min. - maximum time of child execution */
	      init_signals();

	      while (1) {
      
		if(have_sigpipe){
			UdmLog(Agent,verb,"Optimize: SIGPIPE arrived. Broken pipe !");
			have_sigpipe = 0;
			break;
		}
		if (recv(ns, buf, 1, 0)) {
		  if (strncmp(buf, "S", 1) == 0) { /* Store document */

		    if (UdmStoreSave(Agent, ns, Client) == UDM_OK) {
		      UdmSend (ns, "O", 1, 0); 
		    } else {
		      UdmSend (ns, "E", 1, 0); 
		    }

		  } else if(strncmp(buf, "G", 1) == 0) { /* retrieve document */

		    if (UdmStoreGet(Agent, ns, Client) != UDM_OK) {
		      UdmLog(Agent, UDM_LOG_ERROR, "Retrieve error");
		    }

		  } else if(strncmp(buf, "E", 1) == 0) {  /* retrieve by chuncks for excerption */

		    if (UdmStoreGetByChunks(Agent, ns, Client) != UDM_OK) {
		      UdmLog(Agent, UDM_LOG_ERROR, "Retrieve by chunks error");
		    }

		  } else if(strncmp(buf, "F", 1) == 0) {  /* check documents presence */
		    
		    if (UdmStoreFind(Agent, ns, Client) != UDM_OK) {
		      UdmLog(Agent, UDM_LOG_ERROR, "Find error");
		    }

		  } else if (strncmp(buf, "D", 1) == 0) { /* delete document */

		    if (UdmStoreDelete(Agent, ns, Client) != UDM_OK) {
		      UdmLog(Agent, UDM_LOG_ERROR, "Delete error");
		    }

		  } else if (strncmp(buf, "C", 1) == 0) { /* check-up stored database */

		    if (UdmStoredCheck(Agent, ns, Client) != UDM_OK) {
		      UdmLog(Agent, UDM_LOG_ERROR, "Stored check-up error");
		    }

		  } else if(strncmp(buf, "B", 1) == 0) { /* close connect command */
		    UdmLog(Agent, UDM_LOG_EXTRA, "[%s] Bye received.", Client);
		    close(ns);
		    exit(0);
		  }else {
		    UdmLog(Agent, UDM_LOG_ERROR, "%s <hello> error %d %s", time_pid_info(), errno,buf /*strerror(errno)*/);
		    continue;
		  }
		  alarm(3600); /* 60 min. - maximum time of child execution reset */
		} else {
		  UdmLog(Agent, UDM_LOG_DEBUG, "%s client silent.", time_pid_info());
		  sleep(1);
		}

	      }
	      close(ns);
	      exit(0);
	    }
	  }
	  /* parent process */
	  close(ns);
	}

	unlink(pidname);
	UdmAgentFree(Agent);
	UdmEnvFree(Conf);

#else
	fprintf(stderr, "zlib support required. Please rebuild with --with-zlib option\n");

#endif
	return 0;
}
