/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#ifdef WIN32
#include "stdafx.h"
#endif

#include "udm_config.h"

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#include <errno.h>

#ifdef HAVE_IO_H
#include <io.h>  /* for Win */
#endif

#ifdef HAVE_DIRECT_H
#include <direct.h> /* for Win */
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_xmalloc.h"
#include "udm_signals.h"

int tz_offset=0;	/* EXT */

/*** str functions implementation which may be missing on some OS ***/

#ifndef HAVE_BZERO
__C_LINK void __UDMCALL bzero(void *b, size_t len) {
	memset(b,0,len);
}
#endif

#ifndef HAVE_STRCASECMP
__C_LINK int __UDMCALL strcasecmp(const char *s1, const char *s2) {
	register const unsigned char
			*us1 = (const unsigned char *)s1,
			*us2 = (const unsigned char *)s2;

	while (tolower(*us1) == tolower(*us2++))
		if (*us1++ == '\0')
			return (0);
	return (tolower(*us1) - tolower(*--us2));
}
#endif

#ifndef HAVE_STRNCASECMP
int strncasecmp(const char *s1, const char *s2, size_t n)
{
	if (n != 0) {
		register const unsigned char
				*us1 = (const unsigned char *)s1,
				*us2 = (const unsigned char *)s2;

		do {
			if (tolower(*us1) != tolower(*us2++))
				return (tolower(*us1) - tolower(*--us2));
			if (*us1++ == '\0')
				break;
		} while (--n != 0);
	}
	return (0);
}
#endif

#ifndef HAVE_STRNDUP
char * strndup(const char * str,size_t len){
	char * res;
	res=(char*)malloc(len+1);
	strncpy(res,str,len);
	res[len]='\0';
	return res;
}
#endif

#ifndef HAVE_STRCASESTR
char * strcasestr(register const char *s, register const char *find) {
        register char c, sc;
        register size_t len;

        if ((c = *find++) != 0) {
                c = tolower((unsigned char)c);
                len = strlen(find);
                do {
                        do {
                                if ((sc = *s++) == 0)
                                        return (NULL);
                        } while ((char)tolower((unsigned char)sc) != c);
                } while (strncasecmp(s, find, len) != 0);
                s--;
        }
        return ((char *)s);
}
#endif

/**************** RFC 1522 ******************************************/
static char base64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

int UdmHex2Int(int h){
	if((h>='0')&&(h<='9'))return(h-'0');
	if((h>='A')&&(h<='F'))return(h-'A'+10);
	if((h>='a')&&(h<='f'))return(h-'a'+10);
	return(0);
}

char * udm_rfc1522_decode(char * dst, const char *src){
	const char *s;
	char  *d;

	s=src;
	d=dst;
	*dst=0;

	while(*s){
		char *e;

		if((e=strstr(s,"=?"))){
			char *schema;
			char *data;

			if(e>s){
				/* Copy last plain text buffer */
				strncpy(d,s,(size_t)(e-s));
				d+=(e-s);
				*d=0;
			}
			e+=2;
			if(!(schema=strchr(e,'?'))){
				/* Error */
				break;
			}
			schema++;
			data=schema+2;

			if(!(e=strstr(data,"?="))){
				/* This is an error */
				break;
			}
			switch(*schema){
				case 'Q':
				case 'q':
					while(data<e){
						char c;
						if(*data=='='){
							c=(char)(UdmHex2Int(data[1])*16+UdmHex2Int(data[2]));
							data+=3;
						}else{
							c=data[0];
							data++;
						}
						/* Copy one char to dst */
						*d=c;d++;*d=0;
					}
					break;
				case 'B':
				case 'b':
					while(data<e){
						char *p;
						int x0,x1,x2,x3,res;

						p=strchr(base64,data[0]);x0=p?p-base64:0;
						p=strchr(base64,data[1]);x1=p?p-base64:0;
						p=strchr(base64,data[2]);x2=p?p-base64:0;
						p=strchr(base64,data[3]);x3=p?p-base64:0;

						res=x3+x2*64+x1*64*64+x0*64*64*64;

						p=(char*)(&res);

						if(p[2])*d=p[2];d++;*d=0;
						if(p[1])*d=p[1];d++;*d=0;
						if(p[0])*d=p[0];d++;*d=0;

						data+=4;
					}
					break;
				default:
					/* Error */
					schema=NULL;
					break;
			}
			if(schema==NULL){
				/* Error */
				break;
			}
			s=e+2;
		}else{
			/* Copy plain text tail */
			strcpy(d,s);
			break;
		}
	}
	return(dst);
}


/************* Base 64 *********************************************/
/* BASE64 encoding converts  3x8 bits into 4x6 bits                */

__C_LINK void __UDMCALL udm_base64_encode (const char *s, char *store, size_t length){
	size_t i;
	unsigned char *p = (unsigned char *)store;

	for (i = 0; i < length; i += 3){
		*p++ = base64[s[0] >> 2];
		*p++ = base64[((s[0] & 3) << 4) + (s[1] >> 4)];
		*p++ = base64[((s[1] & 0xf) << 2) + (s[2] >> 6)];
		*p++ = base64[s[2] & 0x3f];
		s += 3;
	}
	/*Pad the result*/
	if (i == length + 1) *(p - 1) = '=';
	else if (i == length + 2) *(p - 1) = *(p - 2) = '=';
	*p = '\0';
}


__C_LINK size_t __UDMCALL udm_base64_decode(char * dst, const char * src, size_t len){
	int count=0;
	int b[4];
	char * dst0=dst;
	
	for( ; (*src)&&(len>3); src++){
		char * p=strchr(base64,*src);
		
		b[count]=p?p-base64:0;
		if(++count==4){
			int res;
			res=b[3]+b[2]*64+b[1]*64*64+b[0]*64*64*64;
			*dst++=(res>>16)&0xFF;
			*dst++=(res>>8)&0xFF;
			*dst++= res&0xFF;
			count=0;
			len-=3;
		}
	}
	*dst='\0';
	return dst-dst0;
}


unsigned long UdmStartTimer(void){
#ifdef WIN32
	return clock();
#else
#undef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC (sysconf(_SC_CLK_TCK))
	struct tms tms_tmp;
	return (float)times(&tms_tmp)*1000/CLOCKS_PER_SEC;
#endif
}

char * UdmTrim(char *p, const char *delim){
int len;
	len = strlen(p);
	while ((len > 0) && strchr(delim, p[len - 1] )) {
		p[len - 1] = '\0';
		len--;
	}
	while((*p)&&(strchr(delim,*p)))p++;
	return(p);
}

char * UdmRTrim(char* p, const char *delim){
int len;
	len = strlen(p);
	while ((len > 0) && strchr(delim, p[len - 1] )) {
		p[len - 1] = '\0';
		len--;
	}
	return(p);
}


/* strtok_r clone */
char * udm_strtok_r(char *s, const char *delim, char **last) {
    const char *spanp;
    int c, sc;
    char *tok;

    if (s == NULL && (s = *last) == NULL)
	return NULL;

cont:
    c = *s++;
    for (spanp = delim; (sc = *spanp++) != 0; )
    {
	if (c == sc)
	{
	    goto cont;
	}
    }

    if (c == 0)		/* no non-delimiter characters */
    {
	*last = NULL;
	return NULL;
    }
    tok = s - 1;

    for (;;)
    {
	c = *s++;
	spanp = delim;
	do
	{
	    if ((sc = *spanp++) == c)
	    {
		if (c == 0)
		{
		    s = NULL;
		}
		else
		{
		    char *w = s - 1;
		    *w = '\0';
		}
		*last = s;
		return tok;
	    }
	}
	while (sc != 0);
    }
}

/* This function parses string tokens      */
/* It understands: text 'text' "text"      */
/* I.e. Words, tokens enclosed in ' and "  */
/* Behavior is the same with strtok_r()    */
char * UdmGetStrToken(char * s,char ** last){
	char * tbeg,lch;
	if (s == NULL && (s = *last) == NULL)
		return NULL;

	/* Find the beginning of token */
	for(;(*s)&&(strchr(" \r\n\t",*s));s++);

	if(!*s)return(NULL);

	lch=*s;
	if((lch=='\'')||(lch=='"'))s++;
	else	lch=' ';
	tbeg=s;

	while(1){
		switch(*s){
			case '\0': *last=NULL;break;
/*			case '\\': memmove(s,s+1,strlen(s+1)+1);break;*/
			case '"':
			case '\'':
				if(lch==*s){
					*s='\0';
					*last=s+1;	
				}
				break;
			case ' ':
			case '\r':
			case '\n':
			case '\t':
				if(lch==' '){
					*s='\0';
					*last=s+1;
				}
				break;
			default:;
		}
		if(*s)s++;
		else break;
	}
	return(tbeg);
}



char * UdmUnescapeCGIQuery(char *d,char *s){
int hi,lo=0;
char *dd;
	if((d==NULL)||(s==NULL))return(0);
	dd=d;
	while(*s){
		if(*s=='%'){
			if(strchr("0123456789",*(++s))) hi=*s-'0';
			else hi=*s-'A'+10;
			if(strchr("0123456789",*(++s))) lo=*s-'0';
			else lo=*s-'A'+10;
			*d=hi*16+lo;
		}else
		if(*s=='+'){
			*d=' ';
		}else{
			*d=*s;
		}
		s++; d++;
	}
	*d=0;return(dd);
}

char * __UDMCALL UdmEscapeURL(char *d,const char *s){
	char *dd;
	if((d==NULL)||(s==NULL))return(0);
	dd=d;
	while(*s){
		if((*s & 0x80) || strchr("%&<>+[](){}/?#'\"\\;,",*s)) {
			sprintf(d,"%%%X",(int)*s);
			d+=2;
		}else
		if(*s==' '){
			*d='+';
		}else{
			*d=*s;
		}
		s++;d++;
	}
	*d=0;
	return(dd);
}

char * UdmEscapeURI(char *d,const char *s){
	char *dd;
	if((d==NULL)||(s==NULL))return(0);
	dd=d;
	while(*s){
		if(strchr(" ",*s)){
			sprintf(d,"%%%X",(int)*s);
			d+=2;
		}else{
			*d=*s;
		}
		s++;d++;
	}
	*d=0;
	return(dd);
}


/* Oh,no! We have to remove parent level like /parent/../index.html from path
   Let's do it recursively! */
char * UdmRemove2Dot(char *path){
        char *ptr;
	char *tail;
    
        if(!(ptr=strstr(path,"../"))) return path;
	if(ptr==path) return path; /* How could it be? */
        tail=ptr+2;
	ptr--;
        *ptr=0;
	if(!(ptr=strrchr(path,'/'))) *path=0; else *ptr=0;
        path=strcat(path,tail);
	return UdmRemove2Dot(path);
}

/* Function to convert date returned by web-server to time_t
 *
 * Earlier we used strptime, but it seems to be completely broken
 * on Solaris 2.6. Thanks to Maciek Uhlig <muhlig@us.edu.pl> for pointing
 * out this and bugfix proposal (to use Apache's ap_parseHTTPdate).
 *
 * 10 July 2000 kir.
 */

#define BAD_DATE 0

/*****
 *  BEGIN: Below is taken from Apache's util_date.c
 */

/* ====================================================================
 * Copyright (c) 1996-1999 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/********
 * BEGIN
 * This is taken from ap_ctype.h
 * --kir.
 */

#include <ctype.h>

#ifdef __cplusplus
extern "C" {
#endif

/* These macros allow correct support of 8-bit characters on systems which
 * support 8-bit characters.  Pretty dumb how the cast is required, but
 * that's legacy libc for ya.  These new macros do not support EOF like
 * the standard macros do.  Tough.
 */
#define ap_isalnum(c) (isalnum(((unsigned char)(c))))
#define ap_isalpha(c) (isalpha(((unsigned char)(c))))
#define ap_iscntrl(c) (iscntrl(((unsigned char)(c))))
#define ap_isdigit(c) (isdigit(((unsigned char)(c))))
#define ap_isgraph(c) (isgraph(((unsigned char)(c))))
#define ap_islower(c) (islower(((unsigned char)(c))))
#define ap_isprint(c) (isprint(((unsigned char)(c))))
#define ap_ispunct(c) (ispunct(((unsigned char)(c))))
#define ap_isspace(c) (isspace(((unsigned char)(c))))
#define ap_isupper(c) (isupper(((unsigned char)(c))))
#define ap_isxdigit(c) (isxdigit(((unsigned char)(c))))
#define ap_tolower(c) (tolower(((unsigned char)(c))))
#define ap_toupper(c) (toupper(((unsigned char)(c))))

#ifdef __cplusplus
}
#endif

/*******
 * END of taken from ap_ctype.h
 */

/*
 * Compare a string to a mask
 * Mask characters (arbitrary maximum is 256 characters, just in case):
 *   @ - uppercase letter
 *   $ - lowercase letter
 *   & - hex digit
 *   # - digit
 *   ~ - digit or space
 *   * - swallow remaining characters 
 *  <x> - exact match for any other character
 */
static int ap_checkmask(const char *data, const char *mask)
{
    int i;
    char d;

    for (i = 0; i < 256; i++) {
	d = data[i];
	switch (mask[i]) {
	case '\0':
	    return (d == '\0');

	case '*':
	    return 1;

	case '@':
	    if (!ap_isupper(d))
		return 0;
	    break;
	case '$':
	    if (!ap_islower(d))
		return 0;
	    break;
	case '#':
	    if (!ap_isdigit(d))
		return 0;
	    break;
	case '&':
	    if (!ap_isxdigit(d))
		return 0;
	    break;
	case '~':
	    if ((d != ' ') && !ap_isdigit(d))
		return 0;
	    break;
	default:
	    if (mask[i] != d)
		return 0;
	    break;
	}
    }
    return 0;			/* We only get here if mask is corrupted (exceeds 256) */
}

/*
 * tm2sec converts a GMT tm structure into the number of seconds since
 * 1st January 1970 UT.  Note that we ignore tm_wday, tm_yday, and tm_dst.
 * 
 * The return value is always a valid time_t value -- (time_t)0 is returned
 * if the input date is outside that capable of being represented by time(),
 * i.e., before Thu, 01 Jan 1970 00:00:00 for all systems and 
 * beyond 2038 for 32bit systems.
 *
 * This routine is intended to be very fast, much faster than mktime().
 */
static time_t ap_tm2sec(const struct tm * t)
{
    int year;
    time_t days;
    static const int dayoffset[12] =
    {306, 337, 0, 31, 61, 92, 122, 153, 184, 214, 245, 275};

    year = t->tm_year;

    if (year < 70 || ((sizeof(time_t) <= 4) && (year >= 138)))
	return BAD_DATE;

    /* shift new year to 1st March in order to make leap year calc easy */

    if (t->tm_mon < 2)
	year--;

    /* Find number of days since 1st March 1900 (in the Gregorian calendar). */

    days = year * 365 + year / 4 - year / 100 + (year / 100 + 3) / 4;
    days += dayoffset[t->tm_mon] + t->tm_mday - 1;
    days -= 25508;		/* 1 jan 1970 is 25508 days since 1 mar 1900 */

    days = ((days * 24 + t->tm_hour) * 60 + t->tm_min) * 60 + t->tm_sec;

    if (days < 0)
	return BAD_DATE;	/* must have overflowed */
    else
	return days;		/* must be a valid time */
}

/*
 * Parses an HTTP date in one of three standard forms:
 *
 *     Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822, updated by RFC 1123
 *     Sunday, 06-Nov-94 08:49:37 GMT ; RFC 850, obsoleted by RFC 1036
 *     Sun Nov  6 08:49:37 1994       ; ANSI C's asctime() format
 *
 * and returns the time_t number of seconds since 1 Jan 1970 GMT, or
 * 0 if this would be out of range or if the date is invalid.
 *
 * The restricted HTTP syntax is
 * 
 *     HTTP-date    = rfc1123-date | rfc850-date | asctime-date
 *
 *     rfc1123-date = wkday "," SP date1 SP time SP "GMT"
 *     rfc850-date  = weekday "," SP date2 SP time SP "GMT"
 *     asctime-date = wkday SP date3 SP time SP 4DIGIT
 *
 *     date1        = 2DIGIT SP month SP 4DIGIT
 *                    ; day month year (e.g., 02 Jun 1982)
 *     date2        = 2DIGIT "-" month "-" 2DIGIT
 *                    ; day-month-year (e.g., 02-Jun-82)
 *     date3        = month SP ( 2DIGIT | ( SP 1DIGIT ))
 *                    ; month day (e.g., Jun  2)
 *
 *     time         = 2DIGIT ":" 2DIGIT ":" 2DIGIT
 *                    ; 00:00:00 - 23:59:59
 *
 *     wkday        = "Mon" | "Tue" | "Wed"
 *                  | "Thu" | "Fri" | "Sat" | "Sun"
 *
 *     weekday      = "Monday" | "Tuesday" | "Wednesday"
 *                  | "Thursday" | "Friday" | "Saturday" | "Sunday"
 *
 *     month        = "Jan" | "Feb" | "Mar" | "Apr"
 *                  | "May" | "Jun" | "Jul" | "Aug"
 *                  | "Sep" | "Oct" | "Nov" | "Dec"
 *
 * However, for the sake of robustness (and Netscapeness), we ignore the
 * weekday and anything after the time field (including the timezone).
 *
 * This routine is intended to be very fast; 10x faster than using sscanf.
 *
 * Originally from Andrew Daviel <andrew@vancouver-webpages.com>, 29 Jul 96
 * but many changes since then.
 *
 */
time_t UdmHttpDate2Time_t(const char *date){
    struct tm ds;
    int mint, mon;
    const char *monstr, *timstr;
    static const int months[12] =
    {
	('J' << 16) | ('a' << 8) | 'n', ('F' << 16) | ('e' << 8) | 'b',
	('M' << 16) | ('a' << 8) | 'r', ('A' << 16) | ('p' << 8) | 'r',
	('M' << 16) | ('a' << 8) | 'y', ('J' << 16) | ('u' << 8) | 'n',
	('J' << 16) | ('u' << 8) | 'l', ('A' << 16) | ('u' << 8) | 'g',
	('S' << 16) | ('e' << 8) | 'p', ('O' << 16) | ('c' << 8) | 't',
	('N' << 16) | ('o' << 8) | 'v', ('D' << 16) | ('e' << 8) | 'c'};

    if (!date)
	return BAD_DATE;

    while (*date && ap_isspace(*date))	/* Find first non-whitespace char */
	++date;

    if (*date == '\0')
	return BAD_DATE;

    if ((date = strchr(date, ' ')) == NULL)	/* Find space after weekday */
	return BAD_DATE;

    ++date;			/* Now pointing to first char after space, which should be */
    /* start of the actual date information for all 3 formats. */

    if (ap_checkmask(date, "## @$$ #### ##:##:## *")) {	/* RFC 1123 format */
	ds.tm_year = ((date[7] - '0') * 10 + (date[8] - '0') - 19) * 100;
	if (ds.tm_year < 0)
	    return BAD_DATE;

	ds.tm_year += ((date[9] - '0') * 10) + (date[10] - '0');

	ds.tm_mday = ((date[0] - '0') * 10) + (date[1] - '0');

	monstr = date + 3;
	timstr = date + 12;
    }
    else if (ap_checkmask(date, "##-@$$-## ##:##:## *")) {		/* RFC 850 format  */
	ds.tm_year = ((date[7] - '0') * 10) + (date[8] - '0');
	if (ds.tm_year < 70)
	    ds.tm_year += 100;

	ds.tm_mday = ((date[0] - '0') * 10) + (date[1] - '0');

	monstr = date + 3;
	timstr = date + 10;
    }
    else if (ap_checkmask(date, "@$$ ~# ##:##:## ####*")) {	/* asctime format  */
	ds.tm_year = ((date[16] - '0') * 10 + (date[17] - '0') - 19) * 100;
	if (ds.tm_year < 0)
	    return BAD_DATE;

	ds.tm_year += ((date[18] - '0') * 10) + (date[19] - '0');

	if (date[4] == ' ')
	    ds.tm_mday = 0;
	else
	    ds.tm_mday = (date[4] - '0') * 10;

	ds.tm_mday += (date[5] - '0');

	monstr = date;
	timstr = date + 7;
    }
    else
	return BAD_DATE;

    if (ds.tm_mday <= 0 || ds.tm_mday > 31)
	return BAD_DATE;

    ds.tm_hour = ((timstr[0] - '0') * 10) + (timstr[1] - '0');
    ds.tm_min = ((timstr[3] - '0') * 10) + (timstr[4] - '0');
    ds.tm_sec = ((timstr[6] - '0') * 10) + (timstr[7] - '0');

    if ((ds.tm_hour > 23) || (ds.tm_min > 59) || (ds.tm_sec > 61))
	return BAD_DATE;

    mint = (monstr[0] << 16) | (monstr[1] << 8) | monstr[2];
    for (mon = 0; mon < 12; mon++)
	if (mint == months[mon])
	    break;
    if (mon == 12)
	return BAD_DATE;

    if ((ds.tm_mday == 31) && (mon == 3 || mon == 5 || mon == 8 || mon == 10))
	return BAD_DATE;

    /* February gets special check for leapyear */

    if ((mon == 1) &&
	((ds.tm_mday > 29)
	 || ((ds.tm_mday == 29)
	     && ((ds.tm_year & 3)
		 || (((ds.tm_year % 100) == 0)
		     && (((ds.tm_year % 400) != 100)))))))
	return BAD_DATE;

    ds.tm_mon = mon;

    /* printf("parseHTTPdate: %s is %lu\n", date, ap_tm2sec(&ds)); */

    return ap_tm2sec(&ds);
}

/********
 * END: Above is taken from Apache's util_date.c
 */

time_t UdmFTPDate2Time_t(char *date){
	struct tm ds;

	if (!(ap_checkmask(date+4, "##############*")))
		return BAD_DATE;

/*        strptime(date+6, "%y%m%d%H%M%S", &tm_s); */

	ds.tm_year = (((date[4] - '0') * 10) + (date[5] - '0') - 19)*100;
	ds.tm_year += ((date[6] - '0') * 10) + (date[7] - '0');

	ds.tm_mon = ((date[8] - '0') * 10) + (date[9] - '0') - 1;
	ds.tm_mday = ((date[10] - '0') * 10) + (date[11] - '0');

	ds.tm_hour = ((date[12] - '0') * 10) + (date[13] - '0');
	ds.tm_min = ((date[14] - '0') * 10) + (date[15] - '0');
	ds.tm_sec = ((date[16] - '0') * 10) + (date[17] - '0');

	/* printf("parseFTPdate: %s is %lu\n", date, ap_tm2sec(&ds)); */

	return ap_tm2sec(&ds);
}

time_t Udm_dp2time_t(const char * time_str){
	time_t t=0;
	long i;
	char *s;
	const char *ts;

	/* flag telling us that time_str is exactly <num> without any char
	 * so we think it's seconds
	 * flag==0 means not defined yet
	 * flag==1 means ordinary format (XXXmYYYs)
	 * flag==2 means seconds only
	 */
	int flag=0;
	
	ts = time_str;
	do{
		i=strtol(ts, &s, 10);
		if (s==ts){ /* falied to find a number */
			/* FIXME: report error */
			return -1;
		}
		
		/* ignore spaces */
		while (isspace(*s))
			s++;
	    
		switch(*s){
			case 's': /* seconds */
				t+=i; flag=1; break;
			case 'M': /* minutes */
				t+=i*60; flag=1; break;
			case 'h': /* hours */
				t+=i*60*60; flag=1; break;
			case 'd': /* days */
				t+=i*60*60*24; flag=1; break;
			case 'm': /* months */
				/* I assume month is 30 days */
				t+=i*60*60*24*30; flag=1; break;
			case 'y': /* years */
				t+=i*60*60*24*365; flag=1; break;
			case '\0': /* end of string */
				if (flag==1)
					return -1;
				else{
					t=i;
					flag=2;
					return t;
				}    
			default:
				/* FIXME: report error here! */
				return -1;
		}
		/* go to next char */
		ts=++s;
	/* is it end? */
	} while (*s);
	
	return t;
}

/* * * * * * * * * * * * *
 * DEBUG CRAP...
 */

/*
#define DEBUG_DP2TIME
*/

#ifdef DEBUG_DP2TIME
int main(int argc, char **argv){

	char *dp;
	
	if (argc>1)
		dp=argv[1];
	else{
		printf("Usage: %s time_string\n", argv[0]);
		return 1;
	}
	
	printf("str='%s'\ttime=%li\n", dp, Udm_dp2time_t(dp));
	return 0;
}
#endif /* DEBUG_DP2TIME */


void UdmTime_t2HttpStr(time_t t, char *str){
    struct tm *tim;
/*    if (t==0)
	*str='\0';
    else*/ {
	/* FIXME: I suspect that gmtime IS NOT REENTRANT */

	tim=gmtime(&t);

#ifdef WIN32
	if (!strftime(str, UDM_MAXTIMESTRLEN, "%a, %d %b %Y %H:%M:%S GMT", tim))
#else
	if (!strftime(str, UDM_MAXTIMESTRLEN, "%a, %d %b %Y %H:%M:%S %Z", tim))
#endif
		*str='\0';
	/* /FIXME end of BAD code */
    }
}

/* Find out tz_offset for the functions above
 */
int UdmInitTZ(void){
        time_t tt;
	struct tm *tms;

	tms=localtime(&tt);
	/* localtime sets the external variable timezone */
	tz_offset=(int)timezone;
	return 0;
#if 0

#ifdef HAVE_TM_GMTOFF
	time_t clock;
	struct tm *tim;
        
	clock=time(NULL);
	tim=localtime(&clock);
        tz_offset=tim->tm_gmtoff;
	return 0;
#else                                                                           
	struct tm *t;
	int days, hours, minutes;

	tt= time(NULL);
        gmt = *gmtime(&tt);
        t = localtime(&tt);
        days = t->tm_yday - gmt.tm_yday;
        hours = ((days < -1 ? 24 : 1 < days ? -24 : days * 24)
                + t->tm_hour - gmt.tm_hour);
        minutes=hours * 60 + t->tm_min - gmt.tm_min;
        tz_offset=minutes*60;
	return 0;
#endif /* HAVE_TM_GMTOFF */

#endif /* if 0 */

/* ONE MORE VARIANT - how many do we have?
    struct timezone tz;

    gettimeofday(NULL, &tz);
    tz_offset=tz.tz_minuteswest*60;
    return 0;
*/
}


__C_LINK int __UDMCALL UdmInit(){
	UdmInitTZ();
	return(0);
}


#ifndef HAVE_VSNPRINTF

#ifndef HAVE_STRNLEN
static int strnlen(const char *s, int max){
	const char *e;
	e=s;
	if (max>=0) {
	    while((*e)&&((e-s)<max))e++;
	} else {
	    while(*e)e++;
	}	
	return(e-s);
}
#endif /* ifndef HAVE_STRNLEN */


/* we use this so that we can do without the ctype library */
#define is_digit(c)	((c) >= '0' && (c) <= '9')
#define ZEROPAD	1		/* pad with zero */
#define SIGN	2		/* unsigned/signed long */
#define PLUS	4		/* show plus */
#define SPACE	8		/* space if plus */
#define LEFT	16		/* left justified */
#define SPECIAL	32		/* 0x */
#define LARGE	64		/* use 'ABCDEF' instead of 'abcdef' */

#define do_div(n,base) ({ \
int __res; \
__res = ((unsigned long) n) % (unsigned) base; \
n = ((unsigned long) n) / (unsigned) base; \
__res; })

static int skip_atoi(const char **s){
	int i=0;

	while (is_digit(**s))
		i = i*10 + *((*s)++) - '0';
	return i;
}

static int number(long num, int base, int size, int precision ,int type)
{
	int buflen=0;
	char c,sign,tmp[66];
	const char *digits="0123456789abcdefghijklmnopqrstuvwxyz";
	int i;

	if (type & LARGE)
		digits = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	if (type & LEFT)
		type &= ~ZEROPAD;
	if (base < 2 || base > 36)
		return 0;
	c = (type & ZEROPAD) ? '0' : ' ';
	sign = 0;
	if (type & SIGN) {
		if (num < 0) {
			sign = '-';
			num = -num;
			size--;
		} else if (type & PLUS) {
			sign = '+';
			size--;
		} else if (type & SPACE) {
			sign = ' ';
			size--;
		}
	}
	if (type & SPECIAL) {
		if (base == 16)
			size -= 2;
		else if (base == 8)
			size--;
	}
	i = 0;
	if (num == 0)
		tmp[i++]='0';
	else while (num != 0){
		tmp[i++] = digits[(unsigned long) num % (unsigned) base];
                (unsigned long) num /= (unsigned) base;
             }
	if (i > precision)
		precision = i;
	size -= precision;
	if (!(type&(ZEROPAD+LEFT)))
		while(size-->0)
			buflen++;
	if (sign)
		buflen++;
	if (type & SPECIAL) {
		if (base==8)
			buflen++;
		else if (base==16) {
			buflen++;
			buflen++;
		}
	}
	if (!(type & LEFT))
		while (size-- > 0)
			buflen++;
	while (i < precision--)
		buflen++;
	while (i-- > 0)
		buflen++;
	while (size-- > 0)
		buflen++;
	return buflen;
}

static size_t udm_vsnprintf_len(const char *fmt, va_list args)
{
	size_t buflen=0;
	int len;
	unsigned long num;
	int i, base;
	const char *s;
	int ichar;
	double ifloat;

	int flags;		/* flags to number() */

	int field_width;	/* width of output field */
	int precision;		/* min. # of digits for integers; max
				   number of chars for from string */
	int qualifier;		/* 'h', 'l', or 'L' for integer fields */

	for (; *fmt ; ++fmt) {
		if (*fmt != '%') {
			buflen++;
			continue;
		}
			
		/* process flags */
		flags = 0;
		repeat:
			++fmt;		/* this also skips first '%' */
			switch (*fmt) {
				case '-': flags |= LEFT; goto repeat;
				case '+': flags |= PLUS; goto repeat;
				case ' ': flags |= SPACE; goto repeat;
				case '#': flags |= SPECIAL; goto repeat;
				case '0': flags |= ZEROPAD; goto repeat;
				}
		
		/* get field width */
		field_width = -1;
		if (is_digit(*fmt))
			field_width = skip_atoi(&fmt);
		else if (*fmt == '*') {
			++fmt;
			/* it's the next argument */
			field_width = va_arg(args, int);
			if (field_width < 0) {
				field_width = -field_width;
				flags |= LEFT;
			}
		}

		/* get the precision */
		precision = -1;
		if (*fmt == '.') {
			++fmt;	
			if (is_digit(*fmt))
				precision = skip_atoi(&fmt);
			else if (*fmt == '*') {
				++fmt;
				/* it's the next argument */
				precision = va_arg(args, int);
			}
			if (precision < 0)
				precision = 0;
		}

		/* get the conversion qualifier */
		qualifier = -1;
		if (*fmt == 'h' || *fmt == 'l' || *fmt == 'L') {
			qualifier = *fmt;
			++fmt;
		}

		/* default base */
		base = 10;

		switch (*fmt) {
		case 'f':
                        if (precision < 0)
                                precision = 6;
                        buflen += precision;
                        if ((flags & SPECIAL) && (precision == 0))
                                buflen++;
                        if (flags & (PLUS || SPACE))
                                buflen++;
                        ifloat = va_arg(args, double);
                        while ((ifloat >= 1.) || (field_width-- > 0)) {
                                ifloat /= base;
                                buflen++;
                        };
 
                        continue;
		case 'c':
			if (!(flags & LEFT))
				while (--field_width > 0)
					buflen++;
			ichar = va_arg(args, int);
			buflen+= sizeof((unsigned char) ichar);
			while (--field_width > 0)
				buflen++;
			continue;

		case 's':
			s = va_arg(args, char *);
			if (!s)
				s = "<NULL>";

			len = strnlen(s, precision);

			if (!(flags & LEFT))
				while (len < field_width--)
					buflen++;
			for (i = 0; i < len; ++i){
				buflen++;
				s++;
			}
			while (len < field_width--)
				buflen++;
			continue;

		case 'p':
			if (field_width == -1) {
				field_width = 2*sizeof(void *);
				flags |= ZEROPAD;
			}
			buflen += number(
				(long) va_arg(args, void *), 16,
				field_width, precision, flags);
			continue;


		case '%':
			buflen++;
			continue;

		/* integer number formats - set up the flags and "break" */
		case 'o':
			base = 8;
			break;

		case 'X':
			flags |= LARGE;
		case 'x':
			base = 16;
			break;

		case 'd':
		case 'i':
			flags |= SIGN;
		case 'u':
			break;

		default:
			buflen++;
			if (*fmt)
				buflen++;
			else
				--fmt;
			continue;
		}
		if (qualifier == 'l')
			num = va_arg(args, unsigned long);
		else if (qualifier == 'h') {
			num = (unsigned short) va_arg(args, int);
			if (flags & SIGN)
				num = (short) num;
		} else if (flags & SIGN)
			num = va_arg(args, int);
		else
			num = va_arg(args, unsigned int);
		buflen += number((long)num, base, field_width, precision, flags);
	}
	return buflen;
}

int vsnprintf(char *str, size_t size, const char  *fmt,  va_list ap){
	if (udm_vsnprintf_len(fmt, ap)<size){
                return(vsprintf(str, fmt, ap));
        }else{
                strncpy(str, "Oops! Buffer overflow in vsnprintf()", size);
                return(-1);
        }
}
#endif /* ifndef HAVE_VSNPRINTF */



/* This snprintf function has been written
 * by Murray Jensen <Murray.Jensen@cmst.csiro.au>
 * So if it works for you - praise him, 
 * and if it doesn't - blame him...  
 *   --kir.
 */

__C_LINK int __UDMCALL udm_snprintf(char *buf, size_t len, const char *fmt, ...){
#ifndef HAVE_VSNPRINTF
       char *tmpbuf;
       size_t need;
#endif /* ifndef HAVE_VSNPRINTF */
       va_list ap;
       int ret;

       va_start(ap, fmt);

#ifdef HAVE_VSNPRINTF
       ret = vsnprintf(buf, len, fmt, ap);
       buf[len-1] = '\0';
#else

       need = udm_vsnprintf_len(fmt, ap);

       if ((tmpbuf = (char *) malloc(need + 1)) == NULL)
               strncpy(buf, "Oops! Out of memory in snprintf", len-1);
       else {
               vsprintf(tmpbuf, fmt, ap);
               strncpy(buf, tmpbuf, len-1);
               UDM_FREE(tmpbuf);
       }

       ret = strlen(buf);
#endif /* HAVE_VSNPRINTF */

       va_end(ap);

       return ret;
}



#ifndef HAVE_HSTRERROR
const char *h_errlist[] = {
    "Error 0",
    "Unknown host",
    "Host name lookup failure",
    "Unknown server error",
    "No address associated with name"
};
#endif




#ifdef WIN32
int UdmBuild(char *path, int omode) {
	struct stat sb;
	int last, retval, ret_code;
	char *p;

	p = path;
	retval = 0;
	if (p[0] == UDMSLASH)		/* Skip leading slashes. */
		++p;
	for (last = 0; !last ; ++p) {
		if (p[0] == '\0')
			last = 1;
		else
		if (p[0] != UDMSLASH)
			continue;
		*p = '\0';
		if (p[1] == '\0')
			last = 1;

		if(path[strlen(path)-1]==':'){
			char buf[5*1024];

			sprintf(buf,"%s",path);
			strcat(buf, "\\");
			ret_code=stat(buf, &sb);
		}else{
			ret_code=stat(path, &sb);
		}

		if (ret_code) {
			if (errno != ENOENT || mkdir(path) < 0 ){
				/* warn("%s", path); */
				retval = 1;
				break;
			}
		}
		else if ((sb.st_mode & S_IFMT) != S_IFDIR) {
			if (last)
				errno = EEXIST;
			else
				errno = ENOTDIR;
			/* warn("%s", path); */
			retval = 1;
			break;
		}
		if (!last)
			*p = UDMSLASH;
	}
	return (retval);
}

#else

int UdmBuild(char *path, int omode) {
	struct stat sb;
	mode_t numask, oumask;
	int first, last, retval;
	char *p;

	p = path;
	oumask = 0;
	retval = 0;
	if (p[0] == UDMSLASH)		/* Skip leading slashes. */
		++p;
	for (first = 1, last = 0; !last ; ++p) {
		if (p[0] == '\0')
			last = 1;
		else if (p[0] != UDMSLASH)
			continue;
		*p = '\0';
		if (p[1] == '\0')
			last = 1;
		if (first) {
			/*
			 * POSIX 1003.2:
			 * For each dir operand that does not name an existing
			 * directory, effects equivalent to those cased by the
			 * following command shall occcur:
			 *
			 * mkdir -p -m $(umask -S),u+wx $(dirname dir) &&
			 *    mkdir [-m mode] dir
			 *
			 * We change the user's umask and then restore it,
			 * instead of doing chmod's.
			 */
			oumask = umask(0);
			numask = oumask & ~(S_IWUSR | S_IXUSR);
			(void)umask(numask);
			first = 0;
		}
		if (last)
			(void)umask(oumask);
		if (stat(path, &sb)) {
			if (errno != ENOENT ||
			    mkdir(path, last ? omode :
				  S_IRWXU | S_IRWXG | S_IRWXO) < 0
				 ){
				/* warn("%s", path); */
				retval = 1;
				break;
			}
		}
		else if ((sb.st_mode & S_IFMT) != S_IFDIR) {
			if (last)
				errno = EEXIST;
			else
				errno = ENOTDIR;
			/* warn("%s", path); */
			retval = 1;
			break;
		}
		if (!last)
			*p = UDMSLASH;
	}
	if (!first && !last)
		(void)umask(oumask);
	return (retval);
}

#endif


char * UdmBuildParamStr(char * dst,size_t len,const char * src,char ** argv,size_t argc){
	const char * s;
	char * d;
	size_t argn;
	size_t curlen;

	*dst='\0';
	s=src;
	d=dst;
	curlen=0;
	
	while(*s){
		if(*s=='$'){
			argn=atoi(s+1);
			if((argn<=argc)&&(argn>0)){
				size_t arglen;
				arglen=strlen(argv[argn-1]);
				if(arglen+curlen+1<len){
					strcpy(d,argv[argn-1]);
					d+=strlen(d);
					curlen+=arglen;
				}else{
					break;
				}
			}
			s++;
			while(*s>='0'&&*s<='9')
				s++;
		}else
#ifdef WIN32
#else
		if(*s=='\\'){
			s++;
			if(*s){
				if(curlen+2<len){
					*d=*s;
					s++;
					d++;
					*d='\0';
					curlen++;
				}else{
					break;
				}
			}
		}else
#endif
		{
			if(curlen+2<len){
				*d=*s;
				s++;
				d++;
				*d='\0';
				curlen++;
			}else{
				break;
			}
		}
	}
	return(dst);
}

/************************************************/

int UdmSetEnv(const char * name,const char * value){
#ifdef HAVE_SETENV
	return(setenv(name,value,1));
#else
#ifdef HAVE_PUTENV
	int res;
	char * s;
	s=(char*)malloc(strlen(name)+strlen(value)+3);
	sprintf(s,"%s=%s",name,value);
	res=putenv(s);
	UDM_FREE(s);
	return res;
#else
	return 0;
#endif
#endif

}

void UdmUnsetEnv(const char * name){
#ifdef HAVE_UNSETENV
	unsetenv(name);
#else
	UdmSetEnv(name,"");
#endif
}

/****************************************************/

char * UdmStrRemoveChars(char * str, const char * sep){
	char * s, *e;
	int has_sep=0;

	e=s=str;  
	while(*s){
		if(strchr(sep,*s)){
			if(!has_sep){
				e=s;
				has_sep=1;
			}
		}else{
			if(has_sep){
				memmove(e,s,strlen(s)+1);
				s=e;
				has_sep=0;
			}
		}
		s++;
	}
	/* End spaces */
	if(has_sep)*e='\0';
	
	return(str);
}


char * UdmStrRemoveDoubleChars(char * str, const char * sep){
	char * s, *e;
	int has_sep=0;
	
	/* Initial spaces */
	for(s=str;(*s)&&(strchr(sep,*s));s++);
	if(s!=str)memmove(str,s,strlen(s)+1);
	e=s=str;
	
	/* Middle spaces */
	while(*s){
		if(strchr(sep,*s)){
			if(!has_sep){
				e=s;
				has_sep=1;
			}
		}else{
			if(has_sep){
				*e=' ';
				memmove(e+1,s,strlen(s)+1);
				s=e+1;
				has_sep=0;
			}
		}
		s++;
	}
	
	/* End spaces */
	if(has_sep)*e='\0';
	
	return(str);
}


void UdmUniRemoveDoubleSpaces(int * ustr){
	int * u, *e, addspace=0;
				
	for(u=e=ustr;*u;){
		switch(*u){
			case ' ':
			case '\t':
			case '\r':
			case '\n':
			case 0xA0: /* nbsp */
				addspace=1;
				u++;
				break;
			default:
				if(addspace){
					if(e>ustr){
						*e=' ';
						e++;
					}
					addspace=0;
				}
				*e=*u;
				e++;
				u++;
				break;
		}
	}
	*e=0;
}

void UdmUniPrint(int * ustr){
	int * lt;
	for(lt=ustr;*lt;lt++){
		fprintf(stderr,"%04X ",*lt);
	}
	fprintf(stderr,"\n");
}



ssize_t UdmRecvall(int s, void *buf, size_t len){

#ifdef MSG_WAITALL
	if (len == 0) return 0;
	return recv(s, buf, len, MSG_WAITALL);
#else

	ssize_t received = 0, r=0;
	char *b = buf;
	time_t start = time(NULL);
	if (len == 0) return received;
	while ( ((size_t)received < len) && ((r = recv(s, b+received, len-received,0)) >= 0)){
		received+=r;
		if (have_sigint || have_sigterm || have_sigpipe) break;
		if (time(NULL) - start > 300) break;
	}
	return (r < 0) ? r : received;
#endif
}

/* To bust performance you may increase this value, but be sure, that kernel can handle it */
#define UDM_BLKLEN 8196

ssize_t UdmSend(int s, const void *msg, size_t len, int flags) {
  ssize_t o = 0;
  size_t olen, slen;
  const char *p = msg;

  while(len) {
    slen = (len < UDM_BLKLEN) ? len : UDM_BLKLEN;
    olen = send(s, p, slen, flags);
    if (olen == (size_t)-1) return olen;
    len -= olen;
    p += olen;
    o += olen;
  }
  return o;
}

/***********/

#ifndef WIN32

void (*UdmSignal(int signo, void (*handler)(int)))(int) {
  struct sigaction act, oact;

  act.sa_handler = handler;
  sigemptyset(&act.sa_mask);
  act.sa_flags  = 0;
  if (signo == SIGCHLD) act.sa_flags |= SA_NOCLDSTOP;
  act.sa_flags |= SA_RESTART;
  if (sigaction(signo, &act, &oact) < 0) {
    return(SIG_ERR);
  }
  return(oact.sa_handler);
}

/***********/

static struct flock* file_lock(struct flock *ret, int type, int whence) {
    ret->l_type = type ;
    ret->l_start = 0 ;
    ret->l_whence = whence ;
    ret->l_len = 0 ;
    ret->l_pid = getpid() ;
    return ret ;
}

#endif

void UdmWriteLock(int fd) {  /* an exclusive lock on an entire file */

#ifndef WIN32

    static struct flock ret ;
    fcntl(fd, F_SETLKW, file_lock(&ret, F_WRLCK, SEEK_SET));

#else
	// Windows routine
	
	DWORD filelen_high;
	DWORD filelen_low;
	OVERLAPPED overlap_str;
	int lock_res;

	overlap_str.Offset = 0;
	overlap_str.OffsetHigh = 0;
	overlap_str.hEvent = 0;

	filelen_low = GetFileSize((HANDLE)fd, &filelen_high);
	lock_res = LockFileEx((HANDLE)fd, LOCKFILE_EXCLUSIVE_LOCK, 0, filelen_low, filelen_high, &overlap_str);
	assert(lock_res);	

#endif

}

void UdmUnLock(int fd) { /* ulock an entire file */

#ifndef WIN32
    
	static struct flock ret ;
    fcntl(fd, F_SETLKW, file_lock(&ret, F_UNLCK, SEEK_SET));

#else
	// Windows routine

	DWORD filelen_high;
	DWORD filelen_low;
	int lock_res;

	filelen_low = GetFileSize((HANDLE)fd, &filelen_high);
	lock_res = UnlockFile((HANDLE)fd, 0, 0, filelen_low, filelen_high);
	assert(lock_res);	 

#endif

}

void UdmReadLock(int fd) {   /* a shared lock on an entire file */

#ifndef WIN32
    
    static struct flock ret ;
    fcntl(fd, F_SETLKW, file_lock(&ret, F_RDLCK, SEEK_SET));

#else
	// Windows routine
	
	DWORD filelen_high;
	DWORD filelen_low;
	OVERLAPPED overlap_str;
	int lock_res;

	overlap_str.Offset = 0;
	overlap_str.OffsetHigh = 0;
	overlap_str.hEvent = 0;

	filelen_low = GetFileSize((HANDLE)fd, &filelen_high);
	lock_res = LockFileEx((HANDLE)fd, 0, 0, filelen_low, filelen_high, &overlap_str);
	assert(lock_res);

#endif

}

void UdmReadLockFILE(FILE *f) {   /* a shared lock on an entire file */

#ifndef WIN32
    
	static struct flock ret ;
    fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_RDLCK, SEEK_SET));

#else
	// Windows routine
	int file_desc = fileno(f);
	UdmReadLock(file_desc);

#endif

}

void __UDMCALL UdmWriteLockFILE(FILE *f) {  /* an exclusive lock on an entire file */

#ifndef WIN32

    static struct flock ret ;
    fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_WRLCK, SEEK_SET));

#else
	// Windows routine

	int file_desc = fileno(f);
	UdmWriteLock(file_desc);

#endif

}

void __UDMCALL UdmUnLockFILE(FILE *f) { /* ulock an entire file */

#ifndef WIN32

    static struct flock ret ;
    fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_UNLCK, SEEK_SET));

#else
	// Windows routine

	int file_desc = fileno(f);
	UdmUnLock(file_desc);

#endif

}
