/* Copyright (C) 2000-2009 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <string.h>
#include "udm_common.h"
#include "udm_utils.h"
#include "udm_searchtool.h"

void
UdmSectionListPrint(UDM_SECTIONLIST *SectionList)
{
  UDM_SECTIONLIST *L= SectionList;
  size_t section;
  fprintf(stderr, "ncoords=%d nsec=%d\n",
          SectionList->ncoords, SectionList->nsections);
  for (section= 0; section < L->nsections; section++)
  {
    UDM_SECTION *S= &L->Section[section];
    size_t coord;
    /*
    fprintf(stderr,
            "       [%d:%d] ncoords=%d seclen=%d secno=%d min=%d max=%d offs=%d\n",
            S->url_id, S->secno, S->ncoords, S->seclen, S->secno,
            S->minpos, S->maxpos,
            S->Coord - SectionList->Coord);
    */
    if (!S->Coord)
      continue;
    for (coord= 0; coord < S->ncoords; coord++)
    {
      UDM_COORD2 *C= &S->Coord[coord];
      fprintf(stderr, "[%d]secno=%d pos=%d seclen=%d num=%d order=%d ncoords=%d min=%d max=%d\n",
                       S->url_id, S->secno, C->pos, S->seclen, S->wordnum, S->order,
                       S->ncoords, S->minpos, S->maxpos);
    }
  }
}


int
UdmSectionListAlloc(UDM_SECTIONLIST *List, size_t ncoords, size_t nsections)
{
  bzero((void*)List, sizeof(*List));
  if (ncoords)
  {
    List->Coord= (UDM_COORD2*) UdmMalloc(ncoords * sizeof(UDM_COORD2));
    List->Section= (UDM_SECTION*) UdmMalloc(nsections * sizeof(UDM_SECTION));
    List->mcoords= ncoords;
    List->msections= nsections;
    UDM_ASSERT(nsections > 0);
  }
  return UDM_OK;
}


void
UdmSectionListFree(UDM_SECTIONLIST *List)
{
  UDM_FREE(List->Coord);
  UDM_FREE(List->Section);
}


int
UdmSectionListListAdd(UDM_SECTIONLISTLIST *List, UDM_SECTIONLIST *Item)
{
  if (List->nitems >= List->mitems)
  {
    List->mitems+= 256;
    List->Item= (UDM_SECTIONLIST*) UdmRealloc(List->Item, List->mitems * sizeof(*Item));
  }
  List->Item[List->nitems]= *Item;
  List->nitems++;
  return UDM_OK;
}


void
UdmSectionListListInit(UDM_SECTIONLISTLIST *List)
{
  bzero((void*) List, sizeof(*List));
}


void
UdmSectionListListFree(UDM_SECTIONLISTLIST *List)
{
  size_t i;
  for (i= 0; i < List->nitems; i++)
    UdmSectionListFree(&List->Item[i]);
  UDM_FREE(List->Item);
}


/*
  Compare two UDM_SECTIONLISTs by nsections,
  return the list with more sections first.
*/
static int
cmp_section_list_by_nsections_rev(UDM_SECTIONLIST *s1, UDM_SECTIONLIST *s2)
{
  if (s1->nsections > s2->nsections) return -1;
  if (s1->nsections < s2->nsections) return  1;
  return 0;
}

/*
  Merge sorted lists
*/
int
UdmSectionListListMergeSorted(UDM_SECTIONLISTLIST *SrcList,
                              UDM_SECTIONLIST *Dst, int opt)
{
  UDM_SECTION **p, **e, *Section;
  size_t list, nlists, nsections;
  int rc= UDM_OK;
  urlid_t *id;
  int newver= 1;
  UDM_SECTIONLIST Short;
  
  bzero((void*) Dst, sizeof(*Dst));
  bzero((void*) &Short, sizeof(Short));
  if (!SrcList->nitems)
    return rc;
  
  if (!(p= UdmMalloc(SrcList->nitems * sizeof(UDM_SECTIONLIST*) * 2)))
    return UDM_ERROR;
  e= p + SrcList->nitems;
  if (!(id= (urlid_t*) UdmMalloc(SrcList->nitems * sizeof(urlid_t))))
    return UDM_ERROR;

  {
    const char *env= getenv ("NEWVER");
    if (env)
      newver= atoi(env);
  }
  if (newver)
  {
    UdmSort((void*) SrcList->Item, SrcList->nitems, sizeof(UDM_SECTIONLIST),
            (udm_qsort_cmp) cmp_section_list_by_nsections_rev);
  }


  /*
    Calc number of non-empty lists
    and total number of sections
  */
  nsections= nlists= 0;
  for (list= 0; list < SrcList->nitems; list++) 
  {
    size_t nsec= SrcList->Item[list].nsections;
    if (nsec)
    {
      /*fprintf(stderr, "[%d] nsec=%d\n", list, nsec); */
      
      if (opt && newver && list > 2 && nsec < SrcList->Item[0].nsections / 10)
      {
        UDM_SECTIONLISTLIST Tmp= *SrcList;
        Tmp.Item+= list;
        Tmp.nitems-= list;
        /*UdmSectionListListMergeUsingSort(&Tmp, &Short);*/
        /* fprintf(stderr, "cut\n");*/
        UdmSectionListListMergeSorted(&Tmp, &Short, 0);
        nsections+= Short.nsections;
        p[nlists]= Short.Section;
        e[nlists]= p[nlists] + Short.nsections;
        id[nlists]= p[nlists]->url_id;
        nlists++;
        /*
        fprintf(stderr, "[%d] nsec=%d ptr=%p MERGED\n",
                list, Short.nsections, Short.Section);
        */
        break;
      }
      nsections+= nsec;
      p[nlists]= SrcList->Item[list].Section;
      e[nlists]= p[nlists] + nsec;
      id[nlists]= p[nlists]->url_id;
      nlists++;
    }
  }
  if (!nlists)
    goto ret;

  if (!(Section= (UDM_SECTION*)UdmMalloc(nsections * sizeof(UDM_SECTION))))
  {
    rc= UDM_ERROR;
    goto ret;
  }
  Dst->Section= Section;
  Dst->msections= nsections;
  
  if (nlists == 1)
    goto one;

  for ( ; ; )
  {
    size_t i, min= 0;
    urlid_t p_min_url_id= id[0];

    /* Find quickly min between the first and the second lists */
    if (id[1] > p_min_url_id)
    {
      /* do nothing */
    }
    else if (id[1] < p_min_url_id)
    {
      min= 1;
      p_min_url_id= id[1];
    }
    else if (p[1]->secno < p[0]->secno)
    {
      min= 1;
    }

    /* Check the other lists */
    for (i= 2; i < nlists; i++)
    {
      if (id[i] > p_min_url_id)
        continue;
      
      if (id[i] < p_min_url_id)
      {
        min= i;
        p_min_url_id= id[i];
        continue;
      }
      
      if (p[i]->secno < p[min]->secno)
        min= i;
    }
    
    *Section++= *p[min]++;
    
    if (p[min] == e[min])
    {
      nlists--;
      p[min]= p[nlists];
      e[min]= e[nlists];
      id[min]= id[nlists];
      if (nlists == 1)
        break;
    }
    else
      id[min]= p[min]->url_id;
  }

one:
  memcpy(Section, *p, (*e - *p) * sizeof(UDM_SECTION));
  Dst->nsections= nsections;
  
ret:
  UDM_FREE(Short.Section);
  UdmFree(p);
  UdmFree(id);
  return rc;
}


/*
  Compare two UDM_SECTIONs by url_id then secno
*/
/*
static int
cmp_section_by_urlid_then_secno(UDM_SECTION *s1, UDM_SECTION *s2)
{
  if (s1->url_id < s2->url_id) return -1;
  if (s1->url_id > s2->url_id) return  1;
  if (s1->secno < s2->secno) return -1;
  if (s1->secno > s2->secno) return 1;
  return 0;
}


static int
UdmSectionListListMergeUsingSort(UDM_SECTIONLISTLIST *SrcList, UDM_SECTIONLIST *Dst)
{
  size_t i, nsections;
  UDM_SECTION *p;
  
  for (i= 0, nsections=0 ; i < SrcList->nitems; i++)
    nsections+= SrcList->Item[i].nsections;
  fprintf(stderr, "skip, remain items=%d sections=%d\n",
          SrcList->nitems, nsections);
  if (!nsections)
    return UDM_OK;

  if (!(p= (UDM_SECTION*) UdmMalloc(nsections * sizeof(UDM_SECTION))))
    return UDM_ERROR;

  Dst->nsections= nsections;
  Dst->Section= p;
  
  for (i= 0; i < SrcList->nitems; i++)
  {
    UDM_SECTIONLIST *S= &SrcList->Item[i];
    size_t nbytes= S->nsections * sizeof(UDM_SECTION);
    memcpy(p, S->Section, nbytes);
    p+= S->nsections;
  }
  UdmSort((void*) Dst->Section, Dst->nsections, sizeof(UDM_SECTION),
          (udm_qsort_cmp) cmp_section_by_urlid_then_secno);
  return UDM_OK;
}
*/
