/* Copyright (C) 2000-2009 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include <math.h>

#ifdef WIN32
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_unicode.h"
#include "udm_unidata.h"
#include "udm_url.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_hash.h"
#include "udm_xmalloc.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_searchcache.h"
#include "udm_server.h"
#include "udm_stopwords.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_vars.h"
#include "udm_agent.h"
#include "udm_store.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_match.h"
#include "udm_indexer.h"
#include "udm_textlist.h"
#include "udm_parsehtml.h"



/************** some forward declarations ********************/
static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db);
static int UdmDeleteBadHrefs(UDM_AGENT *Indexer, 
                             UDM_DOCUMENT *Doc,
                             UDM_DB *db,
                             urlid_t url_id);
static int UdmDeleteLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db);

static int UdmDeleteWordFromURL(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db);

static int UdmURLDataListGroupBySiteUsingSort(UDM_AGENT *A,
                                              UDM_URLDATALIST *R,
                                              UDM_DB *db);

static int UdmURLDataListGroupBySiteUsingHash(UDM_AGENT *A,
                                              UDM_URLDATALIST *DataList,
                                              const char *rec_id_str,
                                              size_t rec_id_len,
                                              const char *site_id_str,
                                              size_t site_id_len);



/*********************** helper functions **********************/
/*
  TODO:
  reuse the same function from
  urlidlist.c, sql.c, dbmode-blob.c, dbmode-multi, dbmode-rawblob, score.c
*/
static int
cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}


const char*
UdmSQLBuildWhereCondition(UDM_ENV * Conf,UDM_DB *db)
{
  size_t  i;
  char    *urlstr;
  char    *tagstr;
  char    *statusstr;
  char    *catstr;
  char    *langstr;
  char    *typestr;
  char    *fromstr;
  char    *serverstr;
  char    *sitestr;
  char    *timestr;
  char    *urlinfostr;
  int     fromserver = 1, fromurlinfo_lang = 1, fromurlinfo_type = 1, fromurlinfo = 1;
  int     dt = UDM_DT_UNKNOWN, dx = 1, dm = 0, dy = 1970, dd = 1;
  time_t  dp = 0, DB = 0, DE = time(NULL), dstmp= 0;
  struct tm tm;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR ue;
  
  if(db->where)return(db->where);
  
  bzero((void*)&tm, sizeof(struct tm));
  UdmDSTRInit(&ue, 64);
  urlstr = (char*)UdmStrdup("");
  tagstr = (char*)UdmStrdup("");
  statusstr = (char*)UdmStrdup("");
  catstr = (char*)UdmStrdup("");
  langstr = (char*)UdmStrdup("");
  typestr = (char*)UdmStrdup("");
  fromstr = (char*)UdmStrdup("");
  serverstr = (char*)UdmStrdup("");
  sitestr = (char*)UdmStrdup("");
  timestr = (char*)UdmStrdup("");
  urlinfostr = (char*)UdmStrdup("");
  
  
  for(i=0;i<Conf->Vars.nvars;i++)
  {
    const char *var=Conf->Vars.Var[i].name?Conf->Vars.Var[i].name:"";
    const char *val=Conf->Vars.Var[i].val?Conf->Vars.Var[i].val:"";
    int intval=atoi(val);
    int longval= atol(val);
    
    if(!val[0])continue;
    
    if(!strcmp(var,"tag") || !strcmp(var,"t"))
    {
      tagstr=(char*)UdmRealloc(tagstr,strlen(tagstr)+strlen(val)+50);
      if(tagstr[0])strcpy(UDM_STREND(tagstr)-1," OR ");
      else  strcat(tagstr,"(");

      sprintf(UDM_STREND(tagstr),"s.tag LIKE '%s')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 12);
        sprintf(UDM_STREND(fromstr), ", server s");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id");
      }
    }
    
    if(!strcmp(var,"status"))
    {
      const char *range= strchr(val, '-');
      statusstr=(char*)UdmRealloc(statusstr,strlen(statusstr)+strlen(val)+50);
      
      if(db->DBSQL_IN && !range)
      {
        if(statusstr[0])sprintf(UDM_STREND(statusstr)-1,",%d)",intval);
        else  sprintf(statusstr," url.status IN (%d)",intval);
      }
      else
      {
        int first, second;
        if(statusstr[0])strcpy(UDM_STREND(statusstr)-1," OR ");
        else  strcat(statusstr,"(");
        if (range && 2 == sscanf(val, "%d-%d", &first, &second))
        {
          sprintf(UDM_STREND(statusstr),
                  "url.status>=%d AND url.status<=%d)", first, second);
        }
        else
          sprintf(UDM_STREND(statusstr),"url.status=%d)",intval);
      }
    }

    if(!strcmp(var,"ul"))
    {
      UDM_URL  URL;
      const char *first = "%";

      UdmURLInit(&URL);
      UdmURLParse(&URL,val);
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);

      if((URL.schema != NULL) && (URL.hostinfo != NULL))
      {
        first = "";
      }
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s%s%%')", first, val);
      UdmURLFree(&URL);
    }
    
    if(!strcmp(var,"ue"))
    {
      UDM_URL URL;
      const char *first;
      UdmURLInit(&URL);
      UdmURLParse(&URL, val);
      first= ((URL.schema != NULL) && (URL.hostinfo != NULL)) ? "" : "%";
      UdmURLFree(&URL);
      if (ue.size_data)
      {
        ue.size_data--; /* Remove right parenthesis */
        UdmDSTRAppend(&ue, " AND ", 5);
      }
      else
      {
        UdmDSTRAppend(&ue, "(", 1);
      }
      UdmDSTRAppendf(&ue, "url.url NOT LIKE '%s%s%%')", first, val);
    }
    
    if(!strcmp(var,"u"))
    {
      urlstr=(char*)UdmRealloc(urlstr,strlen(urlstr)+strlen(val)+50);
      if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
      else  strcat(urlstr,"(");
      sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')", val);
    }
    
    if(!strcmp(var,"lang") || !strcmp(var,"g"))
    {
      langstr=(char*)UdmRealloc(langstr,strlen(langstr)+strlen(val)+50);
      if(langstr[0])strcpy(UDM_STREND(langstr)-1," OR ");
      else  strcat(langstr,"(");
      sprintf(UDM_STREND(langstr),"il.sval LIKE '%s')",val);
      if (fromurlinfo_lang)
      {
        fromurlinfo_lang = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo il");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND il.url_id=url.rec_id AND il.sname='Content-Language'");
      }
    }

    if(!strncmp(var, "sl.", 3))
    {
      urlinfostr=(char*)UdmRealloc(urlinfostr, strlen(urlinfostr)+strlen(var)+strlen(val)+50);
      if(urlinfostr[0])strcpy(UDM_STREND(urlinfostr)-1," AND ");
      else  strcat(urlinfostr,"(");
      sprintf(UDM_STREND(urlinfostr),"isl%d.sname='%s' AND isl%d.sval LIKE '%s')",fromurlinfo,var+3,fromurlinfo,val);

      fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
      sprintf(UDM_STREND(fromstr), ", urlinfo isl%d", fromurlinfo);
      serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
      sprintf(UDM_STREND(serverstr), " AND isl%d.url_id=url.rec_id", fromurlinfo);
      fromurlinfo++;
    }
    
    if(!strcmp(var,"cat") && val[0])
    {
      catstr=(char*)UdmRealloc(catstr,strlen(catstr)+strlen(val)+50);
      if(catstr[0])strcpy(UDM_STREND(catstr)-1," OR ");
      else  strcat(catstr,"(");
      sprintf(UDM_STREND(catstr),"c.path LIKE '%s%%')",val);
      if (fromserver)
      {
        fromserver = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 32);
        sprintf(UDM_STREND(fromstr), ", server s, categories c");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND s.rec_id=url.server_id AND s.category=c.rec_id");
      }
    }
    if(!strcmp(var,"type") || !strcmp(var, "typ"))
    {
      /* 
         "type" is a reserved word in ASP,
         so "typ" is added as a workaround
      */
      typestr = (char*)UdmRealloc(typestr, strlen(typestr) + strlen(val) + 50);
      if(typestr[0]) strcpy(UDM_STREND(typestr) - 1, " OR ");
      else  strcat(typestr,"(");

      sprintf(UDM_STREND(typestr),"it.sval LIKE '%s')",val);
      if (fromurlinfo_type)
      {
        fromurlinfo_type = 0;
        fromstr = (char*)UdmRealloc(fromstr, strlen(fromstr) + 16);
        sprintf(UDM_STREND(fromstr), ", urlinfo it");
        serverstr = (char*)UdmRealloc(serverstr, strlen(serverstr) + 64);
        sprintf(UDM_STREND(serverstr), " AND it.url_id=url.rec_id AND it.sname='Content-Type'");
      }
    }
    if(!strcmp(var,"site") && intval != 0)
    {
      sitestr=(char*)UdmRealloc(sitestr, strlen(sitestr) + strlen(val) + 50);
      
      if(db->DBSQL_IN)
      {
        if (sitestr[0]) sprintf(UDM_STREND(sitestr) - 1, ",%s%i%s)", qu, intval, qu);
        else  sprintf(sitestr, " url.site_id IN (%s%i%s)", qu, intval, qu);
      }else{
        if (sitestr[0]) strcpy(UDM_STREND(sitestr) - 1, " OR ");
        else  strcat(sitestr, "(");
        sprintf(UDM_STREND(sitestr), "url.site_id=%s%d%s)", qu, intval, qu);
      }
    }
    if (!strcmp(var, "dt"))
    {
      if(!strcasecmp(val, "back")) dt = UDM_DT_BACK;
      else if (!strcasecmp(val, "er")) dt = UDM_DT_ER;
      else if (!strcasecmp(val, "range")) dt = UDM_DT_RANGE;
    }
    if (!strcmp(var, "dx"))
    {
      if (intval == 1 || intval == -1) dx = intval;
      else dx = 1;
    }
    if (!strcmp(var, "dm"))
    {
      dm = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dy"))
    {
      dy = (intval) ? intval : 1970;
    }
    if (!strcmp(var, "dd"))
    {
      dd = (intval) ? intval : 1;
    }
    if (!strcmp(var, "dstmp"))
    {
      dstmp= longval ? longval : 0;
    }
    if (!strcmp(var, "dp"))
    {
      dp = Udm_dp2time_t(val);
    }
    if (!strcmp(var, "db"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DB = mktime(&tm);
    }
    if (!strcmp(var, "de"))
    {
      sscanf(val, "%d/%d/%d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year);
      tm.tm_year -= 1900; tm.tm_mon--;
      DE= mktime(&tm) + 86400; /* Including the given date */
    }
  }
  
  switch(dt)
  {
    case UDM_DT_BACK:
      timestr= (char*)UdmRealloc(timestr, 128);
      if (dp)
        sprintf(timestr, "url.last_mod_time >= %li",
                (long int) time(NULL) - dp);
      break;
    case UDM_DT_ER:
      timestr= (char*)UdmRealloc(timestr, 128);
      tm.tm_mday= dd; tm.tm_mon= dm, tm.tm_year= dy - 1900;
      sprintf(timestr, "url.last_mod_time %s %li",
              (dx == 1) ? ">=" : "<=", (long int) (dstmp ? dstmp : mktime(&tm)));
      break;
    case UDM_DT_RANGE:
      timestr= (char*)UdmRealloc(timestr, 128);
      sprintf(timestr, "url.last_mod_time >= %li AND url.last_mod_time <= %li",
              (long int) DB, (long int) DE);
      break;
    case UDM_DT_UNKNOWN:
    default:
      break;
  }


  if(!urlstr[0] && !tagstr[0] && !statusstr[0] && !catstr[0] && !langstr[0] &&
     !typestr[0] && !serverstr[0] && !fromstr[0] && !sitestr[0] && !timestr[0] &&
     !urlinfostr[0] && !ue.size_data)
  {
    db->where = (char*)UdmStrdup("");
    db->from = (char*)UdmStrdup("");
    goto ret;
  }
  i= strlen(urlstr) + strlen(tagstr) + strlen(statusstr) + strlen(catstr) + 
     strlen(langstr) + strlen(typestr) + strlen(serverstr) + strlen(sitestr) +
     strlen(timestr) + strlen(urlinfostr) + ue.size_data;
  db->where=(char*)UdmMalloc(i+100);
  db->where[0] = '\0';
  UDM_FREE(db->from);
  db->from = (char*)UdmStrdup(fromstr);
  
  if(urlstr[0])
  {
    strcat(db->where,urlstr);
  }
  if (ue.size_data)
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,ue.data);
  }
  if(tagstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,tagstr);
  }
  if(statusstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,statusstr);
  }
  if(catstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,catstr);
  }
  if(langstr[0])
  {
    if(db->where[0])strcat(db->where," AND ");
    strcat(db->where,langstr);
  }
  if(typestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, typestr);
  }
  if(sitestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, sitestr);
  }
  if(serverstr[0])
  {
    if(!db->where[0]) strcat(db->where, " 1=1 ");
    strcat(db->where, serverstr);
  }
  if(timestr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, timestr);
  }
  if (urlinfostr[0])
  {
    if(db->where[0]) strcat(db->where, " AND ");
    strcat(db->where, urlinfostr);
  }
ret:
  UDM_FREE(urlstr);
  UDM_FREE(tagstr);
  UDM_FREE(statusstr);
  UDM_FREE(catstr);
  UDM_FREE(langstr);
  UDM_FREE(typestr);
  UDM_FREE(fromstr);
  UDM_FREE(serverstr);
  UDM_FREE(sitestr);
  UDM_FREE(timestr);
  UDM_FREE(urlinfostr);
  UdmDSTRFree(&ue);
  return db->where;
}


static int
UdmVarListSQLEscape(UDM_VARLIST *dst, UDM_VARLIST *src, UDM_DB *db)
{
  size_t i, nbytes= 0;
  char *tmp= NULL;
  for (i= 0; i < src->nvars; i++)
  {
    size_t len= src->Var[i].curlen;
    if (nbytes < len * 2 + 1)
    {
      nbytes= len * 2 + 1;
      tmp= (char*) UdmRealloc(tmp, nbytes);
    }
    UdmSQLEscStr(db, tmp, src->Var[i].val, len);  /* doc Section */
    UdmVarListAddStr(dst, src->Var[i].name, tmp);
  }
  UdmFree(tmp);
  return UDM_OK;
}



/******************** Popularity *****************************************/

static int
UdmPopRankCalculate(UDM_AGENT *A, UDM_DB *db)
{
  UDM_SQLRES  SQLres, Res, POPres;
  char    qbuf[1024];
  int             rc = UDM_ERROR, u;
  size_t    i, nrows;
  int    skip_same_site = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankSkipSameSite","no"),"yes");
  int    feed_back = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankFeedBack", "no"), "yes");
  int    use_tracking = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseTracking", "no"), "yes");
  int    use_showcnt = !strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  double          ratio = UdmVarListFindDouble(&A->Conf->Vars, "PopRankShowCntWeight", 0.01);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *typespec= (db->DBType == UDM_DB_PGSQL) ? "::float" : "";

  if (feed_back || use_tracking)
  {
    if (use_tracking) UdmLog(A, UDM_LOG_EXTRA, "Will calculate servers weights by tracking");
    if (feed_back) UdmLog(A, UDM_LOG_EXTRA, "Will calculate feed back servers weights");

    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, "SELECT rec_id FROM server WHERE command='S'")))
      goto Calc_unlock;

    nrows = UdmSQLNumRows(&Res);
    for (i = 0; i < nrows; i++)
    {

      if (use_tracking)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM qinfo WHERE name='site' AND value='%s'", UdmSQLValue(&Res, i, 0) );
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
        u = (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)) == 0);
      }
      if (feed_back && (u || !use_tracking))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(pop_rank) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
      }
      if (*UdmSQLValue(&SQLres, 0, 0))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE server SET weight=%s WHERE rec_id=%s%s%s", UdmSQLValue(&SQLres, 0, 0), 
         qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLQuery(db, NULL, qbuf);
      }
      UdmSQLFree(&SQLres);
    }
    UdmSQLFree(&Res);
    UdmSQLQuery(db, NULL, "UPDATE server SET weight=1 WHERE weight=0 AND command='S'");
  }


  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, "SELECT rec_id, url, weight FROM server WHERE command='S'")))
    goto Calc_unlock;
  
  nrows = UdmSQLNumRows(&SQLres);

  for (i = 0; i < nrows; i++)
  {
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT COUNT(*) FROM url WHERE site_id=%s%s%s", qu, UdmSQLValue(&SQLres, i, 0), qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf)))
    goto Calc_unlock;
    UdmLog(A, UDM_LOG_EXTRA, "Site_id: %s URL: %s Weight: %s URL count: %s",
           UdmSQLValue(&SQLres, i, 0),
           UdmSQLValue(&SQLres, i, 1),
           UdmSQLValue(&SQLres, i, 2),
           UdmSQLValue(&Res, 0, 0)); 
    if (atoi(UdmSQLValue(&Res, 0, 0)) > 0)
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                  "UPDATE server SET pop_weight=(%s/%s%s) WHERE rec_id=%s%s%s",
                  UdmSQLValue(&SQLres, i, 2),
                  UdmSQLValue(&Res, 0, 0),
                  typespec,
                  qu, UdmSQLValue(&SQLres, i, 0), qu);
      UdmSQLQuery(db, NULL, qbuf);
    }
    UdmSQLFree(&Res);

  }
  UdmSQLFree(&SQLres);


  UdmLog(A, UDM_LOG_EXTRA, "update links and pages weights");
  if (skip_same_site)  UdmLog(A, UDM_LOG_EXTRA, "Will skip links from same site");
  if (use_showcnt)  UdmLog(A, UDM_LOG_EXTRA, "Will add show count");

        udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id, site_id  FROM url ORDER BY rec_id");

  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) goto Calc_unlock;
  nrows = UdmSQLNumRows(&Res);
  for (i = 0; i < nrows; i++)
  {

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT count(*) FROM links WHERE ot=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (*UdmSQLValue(&SQLres, 0, 0)) 
    {
      if (UDM_ATOI(UdmSQLValue(&SQLres, 0, 0)))
      {
        udm_snprintf(qbuf, sizeof(qbuf), "SELECT pop_weight FROM server WHERE rec_id=%s%s%s", qu, UdmSQLValue(&Res, i, 1), qu);
        if(UDM_OK != (rc = UdmSQLQuery(db, &POPres, qbuf))) goto Calc_unlock;
        if (UdmSQLNumRows(&POPres) != 1)
        { 
          UdmSQLFree(&POPres);
          UdmSQLFree(&SQLres);
          continue;
        }

        udm_snprintf(qbuf, sizeof(qbuf),
                     "UPDATE links SET weight = (%s/%d.0) WHERE ot=%s%s%s",
                     UdmSQLValue(&POPres, 0, 0),
                     atoi(UdmSQLValue(&SQLres, 0, 0)),
                     qu, UdmSQLValue(&Res, i, 0), qu);
        UdmSQLFree(&POPres);
        UdmSQLQuery(db, NULL, qbuf);
      }
    }
    UdmSQLFree(&SQLres);

    if (skip_same_site)
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links l, url uo, url uk WHERE uo.rec_id=l.ot AND uk.rec_id=l.k AND uo.site_id <> uk.site_id AND l.k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    else
    {
      udm_snprintf(qbuf, sizeof(qbuf), "SELECT SUM(weight) FROM links WHERE k=%s%s%s", qu, UdmSQLValue(&Res, i, 0), qu);
    }
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf))) goto Calc_unlock;
    if (UdmSQLValue(&SQLres,0,0) && *UdmSQLValue(&SQLres, 0, 0))
    {
      if (use_showcnt)
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s + (shows * %f) WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), ratio, qu, UdmSQLValue(&Res, i, 0), qu );
      }
      else
      {
        udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=%s WHERE rec_id=%s%s%s", 
        UdmSQLValue(&SQLres, 0, 0), qu, UdmSQLValue(&Res, i, 0), qu );
      }
    } else {
      udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET pop_rank=(shows * %f) WHERE rec_id=%s%s%s", 
                   ratio, qu, UdmSQLValue(&Res, i, 0), qu );
    }
    UdmSQLQuery(db, NULL, qbuf);
    UdmSQLFree(&SQLres);
  }
  UdmSQLFree(&Res);

  rc = UDM_OK;

Calc_unlock:
  UdmLog(A, UDM_LOG_EXTRA, "Popularity rank done.");
  return rc;
}


/************* Servers ******************************************/
static int
UdmLoadServerTable(UDM_AGENT * Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  size_t    rows, i, j, jrows;
  UDM_SQLRES  SQLRes, SRes;
  UDM_HREF  Href;
  char    qbuf[1024];
  const char *filename= UdmVarListFindStr(&db->Vars, "filename", NULL);
  const char *name = (filename && filename[0]) ? filename : "server";
  const char *infoname = UdmVarListFindStr(&db->Vars, "srvinfo", "srvinfo");
  int   rc= UDM_OK;
  const char  *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT rec_id,url,tag,category,command,weight,ordre \
FROM %s WHERE enabled=1 AND parent=%s0%s ORDER BY ordre", name, qu, qu);
  
  if (UDM_OK != (rc=UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;
  
  bzero((void*)&Href, sizeof(Href));
  
  rows= UdmSQLNumRows(&SQLRes);
  for(i= 0; i < rows; i++)
  {
    const char  *val;
    UDM_SERVER  *Server = Indexer->Conf->Cfg_Srv;
    
    Server->site_id    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
    val= UdmSQLValue(&SQLRes, i, 1);
    Server->Match.pattern= UdmStrdup(val ? val : "");
    Server->ordre    = UDM_ATOI(UdmSQLValue(&SQLRes, i, 6));
    Server->command    = *UdmSQLValue(&SQLRes, i, 4);
    Server->weight    = UDM_ATOF(UdmSQLValue(&SQLRes, i, 5));
    
    if((val=UdmSQLValue(&SQLRes,i,2)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Tag", val);
    
    if((val=UdmSQLValue(&SQLRes,i,3)) && val[0])
      UdmVarListReplaceStr(&Server->Vars, "Category", val);
    

    sprintf(qbuf,"SELECT sname,sval FROM %s WHERE srv_id=%s%i%s", infoname, qu, Server->site_id, qu);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SRes, qbuf)))
      return rc;
    jrows= UdmSQLNumRows(&SRes);
    for(j= 0; j < jrows; j++)
    {
      const char *sname = UdmSQLValue(&SRes, j, 0);
      const char *sval = UdmSQLValue(&SRes, j, 1);
      UdmVarListReplaceStr(&Server->Vars, sname, sval);
    }
    UdmSQLFree(&SRes);

    Server->Match.match_type  = UdmVarListFindInt(&Server->Vars, "match_type", UDM_MATCH_BEGIN);
    Server->Match.case_sense= UdmVarListFindInt(&Server->Vars, "case_sense", UDM_CASE_INSENSITIVE);
    Server->Match.nomatch  = UdmVarListFindInt(&Server->Vars, "nomatch", 0);
    Server->Match.arg  = (char *)UdmStrdup(UdmVarListFindStr(&Server->Vars, "Arg", "Disallow"));

    if (Server->command == 'S')
    {
      UdmServerAdd(Indexer, Server, 0);
      if((Server->Match.match_type==UDM_MATCH_BEGIN) &&
         (Indexer->flags & UDM_FLAG_ADD_SERVURL))
      {
        Href.url = Server->Match.pattern;
        Href.method=UDM_METHOD_GET;
        Href.site_id = Server->site_id;
        Href.server_id = Server->site_id;
        Href.hops= (uint4) UdmVarListFindInt(&Server->Vars, "StartHops", 0);
        UdmHrefListAdd(&Indexer->Conf->Hrefs, &Href);
      }
    }
    else
    {
      char errstr[128];
      rc= UdmMatchListAdd(Indexer, &Indexer->Conf->Filters,
                          &Server->Match, errstr, sizeof(errstr), Server->ordre);
      if (rc != UDM_OK)
      {
        udm_snprintf(db->errstr, sizeof(db->errstr),
                    "Error while loading ServerTable '%s' at row %d: %s",
                    name, i, errstr);
        break;
      }
    }
    UDM_FREE(Server->Match.pattern);
  }
  UdmSQLFree(&SQLRes);
  return rc;
}


static int
UdmServerTableFlush(UDM_DB *db)
{
  int rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char str[128];
  
  udm_snprintf(str, sizeof(str),  "UPDATE server SET enabled=0 WHERE parent=%s0%s", qu, qu);
  rc = UdmSQLQuery(db, NULL, str);
  return rc;
}


static int
UdmServerTableAdd(UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  int    res = UDM_OK, done = 1;
  const char  *alias=UdmVarListFindStr(&S->Server->Vars,"Alias",NULL);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  size_t    i, len= 0;
  char    *buf, *arg;
  udmhash32_t     rec_id = UdmStrHash32(S->Server->Match.pattern);
  UDM_VARLIST  *Vars= &S->Server->Vars;
  
  for (i=0; i < Vars->nvars; i++)
    len= udm_max(len, Vars->Var[i].curlen);
  
  len+= S->Server->Match.pattern ? strlen(S->Server->Match.pattern) : 0;
  len+= alias ? strlen(alias) : 0;
  len+= 2048;
  
  buf = (char*)UdmMalloc(len);
  arg = (char*)UdmMalloc(len);
  if (buf == NULL || arg == NULL)
  {
    UDM_FREE(buf);
    UDM_FREE(arg);
    strcpy(db->errstr, "Out of memory");
    db->errcode = 1;
    return UDM_ERROR;
  }
  
  while(done)
  {
    udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%d%s", qu, rec_id, qu);
    if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
      goto ex;
  
    if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
    {
      rec_id++;
    } else done = 0;
    UdmSQLFree(&SQLRes);
  }

  UdmVarListReplaceInt(&S->Server->Vars, "match_type",  S->Server->Match.match_type);
  UdmVarListReplaceInt(&S->Server->Vars, "case_sense",  S->Server->Match.case_sense);
  UdmVarListReplaceInt(&S->Server->Vars, "nomatch",  S->Server->Match.nomatch);
  if (S->Server->command == 'F' && S->Server->Match.arg != NULL)
  {
    UdmVarListReplaceStr(&S->Server->Vars, "Arg", S->Server->Match.arg);
  } 
  
  udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, \
command, parent, ordre, weight, url, pop_weight \
) VALUES (%s%d%s, 1, '%s', %s, '%c', %s%d%s, %d, %f, '%s', 0\
)",
         qu, rec_id, qu,
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         UdmSQLEscStr(db, arg,  /* Server pattern */
                      UDM_NULL2EMPTY(S->Server->Match.pattern),
                      strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)))
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  udm_snprintf(buf, len, "\
UPDATE server SET enabled=1, tag='%s', category=%s, \
command='%c', parent=%s%i%s, ordre=%d, weight=%f \
WHERE rec_id=%s%d%s",
         UdmVarListFindStr(&S->Server->Vars, "Tag", ""),
         UdmVarListFindStr(&S->Server->Vars, "Category", "0"),
         S->Server->command,
         qu, S->Server->parent, qu,
         S->Server->ordre,
         S->Server->weight,
         qu, rec_id, qu
     );
  
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;
  
  S->Server->site_id = rec_id;

  sprintf(buf, "DELETE FROM srvinfo WHERE srv_id=%s%i%s", qu, S->Server->site_id, qu);
  if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf))) goto ex;

  for(i = 0; i < S->Server->Vars.nvars; i++)
  {
    UDM_VAR *Sec = &S->Server->Vars.Var[i];
    if(Sec->val && Sec->name && 
       strcasecmp(Sec->name, "Category") &&
       strcasecmp(Sec->name, "Tag"))
    {
      arg = UdmSQLEscStr(db, arg, Sec->val,strlen(Sec->val)); /* srvinfo */
      udm_snprintf(buf, len, "INSERT INTO srvinfo (srv_id,sname,sval) VALUES (%s%i%s,'%s','%s')",
        qu, S->Server->site_id, qu, Sec->name, arg);
      if(UDM_OK != (res = UdmSQLQuery(db, NULL, buf)))break;
    }
  }

ex:
  UDM_FREE(buf);
  UDM_FREE(arg);
  return res;
}


static int
UdmServerTableGetId(UDM_AGENT *Indexer, UDM_SERVERLIST *S, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  size_t len = ((S->Server->Match.pattern)?strlen(S->Server->Match.pattern):0) + 1024;
  char *buf = (char*)UdmMalloc(len);
  char *arg = (char*)UdmMalloc(len);
  int res, id = 0, i;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  

  if (buf == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }
  if (arg == NULL)
  {
    UDM_FREE(buf);
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
    return UDM_ERROR;
  }

  for (i = 0; i < UDM_SERVERID_CACHE_SIZE; i++)
  {
    if (Indexer->ServerIdCacheCommand[i] == S->Server->command)
      if (!strcmp(Indexer->ServerIdCache[i], S->Server->Match.pattern))
      {
        S->Server->site_id = id = Indexer->ServerIdCacheId[i];
        break;
      }
  }

  if (id == 0)
  {
    udmhash32_t     rec_id;
    int done = 1;
  
    udm_snprintf(buf, len, "SELECT rec_id FROM server WHERE command='%c' AND url='%s'",
           S->Server->command,
           UDM_NULL2EMPTY(S->Server->Match.pattern)
     );
    res = UdmSQLQuery(db, &SQLRes, buf);
    if ((res == UDM_OK) && UdmSQLNumRows(&SQLRes))
    {
      id = S->Server->site_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
      UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
      Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
      Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
      Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
      Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return UDM_OK;
    }
    UdmSQLFree(&SQLRes);
    rec_id = UdmStrHash32(S->Server->Match.pattern);
    while(done) 
    {
      udm_snprintf(buf, len, "SELECT rec_id, url FROM server WHERE rec_id=%s%i%s", qu, rec_id, qu);
      if (UDM_OK != (res = UdmSQLQuery(db, &SQLRes, buf)))
        return res;
      
      if (UdmSQLNumRows(&SQLRes) && (strcasecmp(S->Server->Match.pattern,UdmSQLValue(&SQLRes, 0, 1)) != 0))
      {
        rec_id++;
      } else done = 0;
      UdmSQLFree(&SQLRes);
    }
    udm_snprintf(buf, len, "SELECT enabled,tag,category,ordre FROM server WHERE rec_id=%s%i%s", qu, S->Server->parent, qu);
    res = UdmSQLQuery(db, &SQLRes, buf);
    if (res != UDM_OK)
    {
      UDM_FREE(buf); UDM_FREE(arg);
      UdmSQLFree(&SQLRes);
      return res;
    }

    udm_snprintf(buf, len, "\
INSERT INTO server (rec_id, enabled, tag, category, command, parent, ordre, weight, url) \
VALUES (%s%d%s, %d, '%s', %s, '%c', %s%d%s, %d, %f, '%s')\
",
           qu, rec_id, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0)),
           UdmSQLValue(&SQLRes, 0, 1),
           UdmSQLValue(&SQLRes, 0, 2),
           S->Server->command,
           qu, S->Server->parent, qu,
           UDM_ATOI(UdmSQLValue(&SQLRes, 0, 3)),
           S->Server->weight,
           UdmSQLEscStr(db, arg,  /* Server pattern */
                        UDM_NULL2EMPTY(S->Server->Match.pattern),
                        strlen(UDM_NULL2EMPTY(S->Server->Match.pattern)))
     );
    res = UdmSQLQuery(db, NULL, buf);
    UdmSQLFree(&SQLRes);

    S->Server->site_id = id = rec_id;
    UDM_FREE(Indexer->ServerIdCache[Indexer->pServerIdCache]);
    Indexer->ServerIdCache[Indexer->pServerIdCache] = (char*)UdmStrdup(S->Server->Match.pattern);
    Indexer->ServerIdCacheCommand[Indexer->pServerIdCache] = S->Server->command;
    Indexer->ServerIdCacheId[Indexer->pServerIdCache] = id;
    Indexer->pServerIdCache = (Indexer->pServerIdCache + 1) % UDM_SERVERID_CACHE_SIZE;
  }
  UDM_FREE(buf); UDM_FREE(arg);
  return UDM_OK;
}


int
UdmSrvActionSQL(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,UDM_DB *db)
{
  switch(cmd){
    case UDM_SRV_ACTION_TABLE:
      return UdmLoadServerTable(A,S,db);
    case UDM_SRV_ACTION_FLUSH:
      return UdmServerTableFlush(db);
    case UDM_SRV_ACTION_ADD:
      return UdmServerTableAdd(S, db);
    case UDM_SRV_ACTION_ID:
      return UdmServerTableGetId(A, S, db);
    case UDM_SRV_ACTION_POPRANK:
      return UdmPopRankCalculate(A, db);
    default:
      UdmLog(A, UDM_LOG_ERROR, "Unsupported Srv Action SQL");
      return UDM_ERROR;
  }
}


/********** Searching for URL_ID by various parameters ****************/

int
UdmFindURL(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  UDM_SQLRES  SQLRes;
  const char  *url=UdmVarListFindStr(&Doc->Sections,"URL","");
  udmhash32_t  id = 0;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  if(use_crc32_url_id)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    id = UdmStrHash32(url);
  }else{
    const char *o;
    char *qbuf, *e_url;
    size_t i, l, url_length= strlen(url);
    
    /* Escape URL string */
    if ((e_url = (char*)UdmMalloc(l = (8 * url_length + 1))) == NULL ||
        (qbuf = (char*)UdmMalloc( l + 100 )) == NULL)
    {
      UDM_FREE(e_url);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
    }
    UdmSQLEscStr(db,e_url,url,url_length);


    for(i = 0; i < UDM_FINDURL_CACHE_SIZE; i++)
    {
      if (Indexer->UdmFindURLCache[i])
        if (!strcmp(e_url, Indexer->UdmFindURLCache[i]))
        {
          id = Indexer->UdmFindURLCacheId[i];
          break;
        }
    }

    if (id == 0)
    {
      udm_snprintf(qbuf, l + 100, "SELECT rec_id FROM url WHERE url='%s'",e_url);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
      {
        UDM_FREE(e_url);
        UDM_FREE(qbuf);
        return rc;
      }
      for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
      {
        if((o=UdmSQLValue(&SQLRes,i,0)))
        {
          id=atoi(o);
          break;
        }
      }
      UdmSQLFree(&SQLRes);
      UDM_FREE(Indexer->UdmFindURLCache[Indexer->pURLCache]);
      Indexer->UdmFindURLCache[Indexer->pURLCache] = (char*)UdmStrdup(e_url);
      Indexer->UdmFindURLCacheId[Indexer->pURLCache] = id;
      Indexer->pURLCache = (Indexer->pURLCache + 1) % UDM_FINDURL_CACHE_SIZE;
    }
    UDM_FREE(e_url);
    UDM_FREE(qbuf);
  }
  UdmVarListReplaceInt(&Doc->Sections, "ID", id);
  return  rc;
}


static int
UdmFindMessage(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  size_t     i, len;
  char     *qbuf;
  char     *eid;
  UDM_SQLRES  SQLRes;
  const char  *message_id=UdmVarListFindStr(&Doc->Sections,"Header.Message-ID",NULL);
  int    rc;
  
  if(!message_id)
    return UDM_OK;
  
  len = strlen(message_id);
  eid = (char*)UdmMalloc(4 * len + 1);
  if (eid == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 128);
  if (qbuf == NULL)
  { 
    UDM_FREE(eid);
    return UDM_ERROR;
  }

  /* Escape URL string */
  UdmSQLEscStr(db, eid, message_id, len); /* Message ID */
  
  udm_snprintf(qbuf, 4 * len + 128, 
     "SELECT rec_id FROM url u, urlinfo i WHERE u.rec_id=i.url_id AND i.sname='Message-ID' AND i.sval='%s'", eid);
  rc = UdmSQLQuery(db,&SQLRes,qbuf);
  UDM_FREE(qbuf);
  UDM_FREE(eid);
  if (UDM_OK != rc)
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char * o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
    {
      UdmVarListReplaceInt(&Doc->Sections,"ID", UDM_ATOI(o));
      break;
    }
  }
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


/********************* Limits ********************/


static int
cmp_data_urls(UDM_URLDATA *d1, UDM_URLDATA *d2)
{
  if (d1->url_id > d2->url_id) return 1;
  if (d1->url_id < d2->url_id) return -1;
  return 0;
}


int
UdmLoadSlowLimit(UDM_DB *db, UDM_URLID_LIST *list, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  int exclude= list->exclude;
  bzero((void*) list, sizeof(UDM_URLID_LIST));
  list->exclude= exclude;
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(list->nurls= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  if (!(list->urls= UdmMalloc(sizeof(urlid_t) * list->nurls)))
  {
    rc= UDM_ERROR;
    list->nurls= 0;
    goto ret;
  }
  for (i= 0; i < list->nurls; i++)
  {
    list->urls[i]= atoi(UdmSQLValue(&SQLRes, i, 0));
  }

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}


static int
UdmLoadSlowLimitWithSort(UDM_DB *db, UDM_URLID_LIST *list, const char *q)
{
  int rc= UdmLoadSlowLimit(db, list, q);
  if (rc == UDM_OK && list->nurls > 1)
  {
    UdmSort(list->urls, list->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  }
  return rc;
}


static int
UdmSlowLimitLoadForConv(UDM_AGENT *A,
                        UDM_DB *db,
                        UDM_URLID_LIST *fl_urls,
                        const char *fl)
{
  int rc= UDM_OK; 
  unsigned long ticks= UdmStartTimer();
  char name[64];
  const char *q;

  bzero((void*) fl_urls, sizeof(*fl_urls));

  UdmLog(A, UDM_LOG_INFO, "Loading fast limit '%s'", fl);
  if ((fl_urls->exclude= (fl[0] == '-')))
    fl++;

  udm_snprintf(name, sizeof(name), "Limit.%s", fl);
  if (!(q= UdmVarListFindStr(&A->Conf->Vars, name, NULL)))
  {
    UdmLog(A, UDM_LOG_ERROR, "Limit '%s' not specified", fl);
    return UDM_ERROR;
  }

  if (UDM_OK != (rc= UdmLoadSlowLimitWithSort(db, fl_urls, q)))
    return rc;
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Limit '%s' loaded%s, %d records, %.2f sec",
         fl, fl_urls->exclude ? " type=excluding" : "", fl_urls->nurls,
         (float) ticks/1000);
  return rc;
}


/*
  Applying fast limit to UDLCRDLIST.
  Used in:
  - UdmFindAlwaysFoundWord()
  - UdmFindMultiWord()
  - UdmFindWordSingle()
  - UdmFindCrossWord()
  - UdmMultiAddCoords()
*/
int
UdmApplyFastLimit(UDM_URLCRDLIST *Coord, UDM_URLID_LIST *urls)
{
  UDM_URL_CRD *dst= Coord->Coords;
  UDM_URL_CRD *src= Coord->Coords;
  UDM_URL_CRD *srcend= Coord->Coords + Coord->ncoords;
  
  if (urls->exclude)
  {
    for ( ; src < srcend; src++)
    {
      if (!UdmBSearch(&src->url_id,
                      urls->urls, urls->nurls, sizeof(urlid_t),
                      (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  else
  {
    for ( ; src < srcend; src++)
    {
      if (UdmBSearch(&src->url_id,
                     urls->urls, urls->nurls, sizeof(urlid_t),
                     (udm_qsort_cmp)cmpaurls))
      {
        *dst= *src;
        dst++;
      }
    }
  }
  Coord->ncoords= dst- Coord->Coords;
  return UDM_OK;
}

/******************** Orders ********************************/

/*
  Apply a sorted UserOrder to UDM_URLDATALIST
*/
static int
UdmApplyFastOrderToURLDataList(UDM_URLDATALIST *Data,
                               UDM_URL_INT4_LIST *Order)
{
  UDM_URLDATA *d= Data->Item;
  UDM_URLDATA *de= Data->Item + Data->nitems;

  if (!Order->nitems)
    return UDM_OK;
  
  for ( ; d < de; d++)
  {
    UDM_URL_INT4 *found;
    if ((found= (UDM_URL_INT4*) UdmBSearch(&d->url_id,
                                           Order->Item,
                                           Order->nitems,
                                           sizeof(UDM_URL_INT4),
                                           (udm_qsort_cmp)cmpaurls)))
    {
      char buf[64];
      sprintf(buf, "%08X", found->param);
      d->section= UdmStrdup(buf);
    }
    else
    {
      d->section= UdmStrdup("00000001");
    }
  }
  return UDM_OK;
}


static int
UdmFastOrderLoadAndApplyToURLDataList(UDM_AGENT *Agent,
                                      UDM_DB *db,
                                      UDM_URLDATALIST *Data,
                                      const char *name,
                                      size_t *norder)
{
  UDM_URL_INT4_LIST Order;
  int rc;
  
  if ((UDM_OK != (rc= UdmBlobLoadFastOrder(db, &Order, name))) ||
      !Order.nitems)
    goto ret;

  rc= UdmApplyFastOrderToURLDataList(Data, &Order);
  
ret:
  *norder= Order.nitems;
  UDM_FREE(Order.Item);
  return rc;
}



/******************** URLData *******************************/

#define UDM_URLDATA_URL	        1
#define UDM_URLDATA_SITE        2
#define UDM_URLDATA_POP         4
#define UDM_URLDATA_LM          8 
#define UDM_URLDATA_SU         16
#define UDM_URLDATA_CACHE      32


static int
UdmLoadURLDataFromURLForConv(UDM_AGENT *A,
                             UDM_DB *db,
                             UDM_URLDATALIST *C,
                             UDM_URLID_LIST *fl_urls)
{
  int rc;
  unsigned long ticks= UdmStartTimer();
  char qbuf[4*1024];
  const char *url= (db->from && db->from[0]) ? "url." : "";
  size_t nbytes, i, j;
  UDM_SQLRES  SQLres;

  bzero((void*)C, sizeof(*C));

  UdmLog(A, UDM_LOG_INFO, "Loading URL list");
  
  udm_snprintf(qbuf, sizeof(qbuf),
              "SELECT %srec_id, site_id, pop_rank, last_mod_time"
              " FROM url%s%s%s",
              url, db->from, db->where[0] ? " WHERE " : "", db->where);

  if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
    goto fin;
  
  C->nitems= UdmSQLNumRows(&SQLres);
  nbytes= C->nitems * sizeof(UDM_URLDATA);
  C->Item= (UDM_URLDATA*) UdmMalloc(nbytes);

  for (i= 0, j= 0; i < C->nitems; i++)
  {
    urlid_t url_id= UDM_ATOI(UdmSQLValue(&SQLres, i, 0));
    if (fl_urls->nurls)
    {
      void *found= UdmBSearch(&url_id, fl_urls->urls, fl_urls->nurls,
                              sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
      if (found && fl_urls->exclude)
        continue;
      if (!found && !fl_urls->exclude)
        continue;
    }
    C->Item[j].url_id= url_id;
    C->Item[j].score= 0;
    C->Item[j].per_site= 0;
    C->Item[j].site_id= UDM_ATOI(UdmSQLValue(&SQLres, i, 1));
    C->Item[j].pop_rank= UDM_ATOF(UdmSQLValue(&SQLres, i, 2));
    C->Item[j].last_mod_time= UDM_ATOI(UdmSQLValue(&SQLres, i, 3));
    C->Item[j].url= NULL;
    C->Item[j].section= NULL;
    j++;
  }
  C->nitems= j;
  UdmSQLFree(&SQLres);

  if (C->nitems)
    UdmSort(C->Item, C->nitems, sizeof(UDM_URLDATA), (udm_qsort_cmp) cmp_data_urls);

fin:
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG,
         "URL list loaded: %d urls, %.2f sec", C->nitems, (float)ticks/1000);
  return rc;
}


static int
UdmLoadURLDataFromURLAndSlowLimitForConv(UDM_AGENT *A,
                                         UDM_DB *db,
                                         UDM_URLDATALIST *C)
{
  int rc= UDM_OK;
  const char *fl= UdmVarListFindStr(&A->Conf->Vars, "fl", NULL);
  UDM_URLID_LIST fl_urls;

  bzero((void*)&fl_urls, sizeof(fl_urls));
  bzero((void*)C, sizeof(*C));
  
  if (fl && (UDM_OK != (rc= UdmSlowLimitLoadForConv(A, db, &fl_urls, fl))))
    return rc;

  rc= UdmLoadURLDataFromURLForConv(A, db, C, &fl_urls);
  
  UDM_FREE(fl_urls.urls);

  return rc;
}


/*
  Load the section with the given name from the table "urlinfo",
  for sorting by section: "s=S&su=section".
*/
static int
UdmLoadURLDataFromURLInfoUsingIN(UDM_AGENT *A,
                                 UDM_URLDATALIST *DataList,
                                 UDM_DB *db,
                                 size_t num_best_rows,
                                 const char *esu)
{
  int rc= UDM_OK;
  size_t offs;
  char qbuf[4*1024];

  for (offs= 0; offs < num_best_rows; offs+= 256)
  {  
    size_t nrows, s, i;
    int notfirst= 0;
    UDM_SQLRES SQLres;
    char *end= qbuf + sprintf(qbuf, "SELECT url_id, sval"
                              " FROM urlinfo"
                              " WHERE sname='%s' AND url_id IN (", esu);
    
    for (i= 0; (i < 256) && (offs + i < DataList->nitems); i++)
    {
      end+= sprintf(end, "%s%i", (notfirst) ? "," : "",
                    DataList->Item[offs + i].url_id);
      notfirst= 1;
    }
    end+= sprintf(end, ") ORDER BY url_id");
  
    if (UDM_OK != (rc= UdmSQLQuery(db, &SQLres, qbuf)))
      goto fin;
  
    nrows= UdmSQLNumRows(&SQLres);

    for(i= 0, s= i + offs; i < nrows; s++)
    {
      if (s == DataList->nitems)
        break;
      if (DataList->Item[s].url_id != (urlid_t) UDM_ATOI(UdmSQLValue(&SQLres, i, 0)))
      {
        DataList->Item[s].section= UdmStrdup("");
      }
      else
      {
        DataList->Item[s].section= UdmStrdup(UdmSQLValue(&SQLres, i, 1));
        i++;
      }
    }
    UdmSQLFree(&SQLres);
  }

fin:
  return rc;
}


/*
  Load URL data from "url" for sorting by:
  site_id
  pop_rank
  last_mod_time
  url
  section
*/
static int
UdmLoadURLDataFromURLUsingIN(UDM_AGENT *A,
                             UDM_URLDATALIST *DataList,
                             UDM_DB *db,
                             size_t num_best_rows,
                             int flag)
{
  int need_url= (flag & UDM_URLDATA_URL);
  int rc= UDM_OK;
  char qbuf[4*1024];
  UDM_SQLRES SQLres;
  UDM_PSTR row[5];
  const char *su= UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  char *esu=su ? UdmSQLEscStrSimple(db, NULL, su, strlen(su)) : NULL; /* User sort name */
  size_t j;
  const char *hi_priority= db->DBType == UDM_DB_MYSQL ? "HIGH_PRIORITY " : " ";

  for (j= 0; j < num_best_rows; j+= 256)
  {
    size_t i;
    int notfirst = 0;
    udm_snprintf(qbuf, sizeof(qbuf),
            "SELECT %srec_id, site_id, pop_rank, last_mod_time%s"
            " FROM url"
            " WHERE rec_id IN (",
            hi_priority,
            need_url ? ",url" : "");
    for (i= 0; (i < 256) && (j + i < num_best_rows); i++)
    {
      sprintf(UDM_STREND(qbuf), "%s%i", (notfirst) ? "," : "", DataList->Item[j + i].url_id);
      notfirst= 1;
    }
    sprintf(UDM_STREND(qbuf), ") ORDER BY rec_id");
    if (UDM_OK != (rc= UdmSQLExecDirect(db, &SQLres, qbuf)))
      goto fin;
    for (i= 0; db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK; i++)
    {
      UDM_URLDATA *D= &DataList->Item[i + j];
      if (D->url_id != (urlid_t) UDM_ATOI(row[0].val))
      {
        UdmLog(A, UDM_LOG_ERROR, "Dat url_id (%d) != SQL url_id (%d)", 
               D->url_id, UDM_ATOI(row[0].val));
      }
      D->site_id= UDM_ATOI(row[1].val);
      D->pop_rank= UDM_ATOF(row[2].val);
      D->last_mod_time= UDM_ATOI(row[3].val);
      D->url= need_url ? UdmStrdup(row[4].val) : NULL;
      D->section= NULL;
    }
    UdmSQLFree(&SQLres);

  }

  if ((flag & UDM_URLDATA_SU) && su && su[0])
  {
    if (UDM_OK != (rc= UdmLoadURLDataFromURLInfoUsingIN(A, DataList, db,
                                                        num_best_rows, esu)))
      goto fin;
  }

fin:
  UDM_FREE(esu);
  return rc;
}


static int
UdmLoadURLDataFromURLUsingLoop(UDM_AGENT *A,
                               UDM_URLDATALIST *DataList,
                               UDM_DB *db,
                               size_t num_best_rows, int flag)
{
  int rc= UDM_OK;
  char qbuf[256];
  size_t i;
  int need_url= (flag & UDM_URLDATA_URL);
  const char *hi_priority= db->DBType == UDM_DB_MYSQL ? "HIGH_PRIORITY" : "";

  for (i = 0; i < num_best_rows; i++)
  {
    UDM_SQLRES SQLres;
    UDM_URLDATA *D= &DataList->Item[i];
    udm_snprintf(qbuf, sizeof(qbuf),
                 "SELECT %s site_id, pop_rank, last_mod_time%s"
                 " FROM url WHERE rec_id=%i",
                 hi_priority,
                 need_url ? ",url" : "",
                 DataList->Item[i].url_id);
    if (UDM_OK != (rc = UdmSQLQuery(db, &SQLres, qbuf)))
      goto fin;
    if(UdmSQLNumRows(&SQLres))
    {
      D->url_id= DataList->Item[i].url_id;
      D->site_id= UDM_ATOI(UdmSQLValue(&SQLres, 0, 0));
      D->pop_rank= UDM_ATOF(UdmSQLValue(&SQLres, 0, 1));
      D->last_mod_time= UDM_ATOI(UdmSQLValue(&SQLres, 0, 2));
      D->url= need_url ? UdmStrdup(UdmSQLValue(&SQLres, 0, 3)) : NULL;
      D->section= NULL;
    }
    UdmSQLFree(&SQLres);
  }

fin:
  return rc;
}


static int
UdmLoadURLDataFromURL(UDM_AGENT *A,
                      UDM_URLDATALIST *DataList,
                      UDM_DB *db,
                      size_t num_best_rows, int flag)
{
  int rc= UDM_OK;
  int group_by_site= (flag & UDM_URLDATA_SITE);
  int use_urlbasicinfo= UdmVarListFindBool(&A->Conf->Vars, "LoadURLBasicInfo", 1);

  /* Check that DataList is not empty and is sorted by url_id */
  UDM_ASSERT(num_best_rows && (DataList->Item[0].url_id <= DataList->Item[num_best_rows-1].url_id));
  
  if (!use_urlbasicinfo)
  {
    UdmLog(A,UDM_LOG_DEBUG,"Not using basic URL data from url");
    rc= UdmURLDataListClearParams(DataList, num_best_rows);
  }
  else if (db->DBSQL_IN)
  {
    UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from url");
    rc= UdmLoadURLDataFromURLUsingIN(A, DataList, db, num_best_rows, flag);
  }
  else
  {
    UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from url, not using IN");
    rc= UdmLoadURLDataFromURLUsingLoop(A, DataList, db, num_best_rows, flag);
  }

  if (rc == UDM_OK && group_by_site)
    rc= UdmURLDataListGroupBySiteUsingSort(A, DataList, db);
  return rc;
}


static int
UdmLoadURLDataFromBdict(UDM_AGENT *A,
                        UDM_URLDATALIST *DataList,
                        UDM_DB *db,
                        size_t num_best_rows, int flags)
{
  int rc;
  char qbuf[4*1024];
  UDM_SQLRES SQLres;
  const char *rec_id_str= NULL;
  const char *site_id_str= NULL;
  const char *last_mod_time_str= NULL;
  const char *pop_rank_str= NULL;
  const char *table = UdmBlobGetRTable(db);
  size_t rec_id_len= 0, site_id_len= 0;
  UDM_DSTR buf, rec_id_buf, site_id_buf, pop_rank_buf, last_mod_time_buf;
  UDM_PSTR row[2];
  size_t last_mod_time_len= 0, pop_rank_len= 0;
  size_t nrecs_expected= 1, nrecs_loaded= 0;
  unsigned long ticks= UdmStartTimer();
  int need_pop_rank= (flags & UDM_URLDATA_POP);
  int need_last_mod_time= (flags & UDM_URLDATA_LM);
  int group_by_site= (flags & UDM_URLDATA_SITE);
  int need_site_id= group_by_site;
  
  UdmDSTRInit(&buf, 64);
  UdmDSTRAppendSTR(&buf, "'#rec_id'");
  if (need_pop_rank)
  {
    UdmDSTRAppendSTR(&buf, ",'#pop_rank'");
    nrecs_expected++;
  }
  if (need_last_mod_time)
  {
    UdmDSTRAppendSTR(&buf, ",'#last_mod_time'");
    nrecs_expected++;
  }
  if (need_site_id)
  {
    UdmDSTRAppendSTR(&buf, ",'#site_id'");
    nrecs_expected++;
  }

  /* Check that DataList is not empty and is sorted by url_id */
  UDM_ASSERT(num_best_rows && (DataList->Item[0].url_id <= DataList->Item[num_best_rows-1].url_id));
  UdmLog(A,UDM_LOG_DEBUG,"Trying to load URL data from bdict");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT word, intag FROM %s WHERE word IN (%s)", table, buf.data);
  UdmDSTRFree(&buf);

  
  if (UDM_OK != (rc= UdmSQLExecDirect(db, &SQLres, qbuf)))
  {
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't run a query on bdict");
    return(rc);
  }

  UdmDSTRInit(&rec_id_buf, 4096);
  UdmDSTRInit(&site_id_buf, 4096);
  UdmDSTRInit(&pop_rank_buf, 4096);
  UdmDSTRInit(&last_mod_time_buf, 4096);

  while (db->sql->SQLFetchRow(db, &SQLres, row) == UDM_OK)
  {
    if (!strcmp(row[0].val, "#rec_id"))
    {
      rec_id_len= row[1].len;
      rec_id_str= UdmBlobModeInflateOrAlloc(A, &rec_id_buf, "#rec_id",
                                            row[1].val, &rec_id_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#site_id"))
    {
      site_id_len= row[1].len;
      site_id_str= UdmBlobModeInflateOrAlloc(A, &site_id_buf, "#site_id",
                                             row[1].val, &site_id_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#last_mod_time"))
    {
      last_mod_time_len= row[1].len;
      last_mod_time_str= UdmBlobModeInflateOrAlloc(A, &last_mod_time_buf,
                                                  "#last_mod_time",
                                                  row[1].val, &last_mod_time_len);
      nrecs_loaded++;
    }
    else if (!strcmp(row[0].val, "#pop_rank"))
    {
      pop_rank_len= row[1].len;
      pop_rank_str= UdmBlobModeInflateOrAlloc(A, &pop_rank_buf, "#pop_rank",
                                             row[1].val, &pop_rank_len);
      nrecs_loaded++;
    }
  }
  
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Fetch from bdict done: %.2f", (float)ticks/1000);

  if (rec_id_str && rec_id_len &&
      (site_id_str || ! need_site_id) &&
      (last_mod_time_str || ! need_last_mod_time) &&
      (pop_rank_str || ! need_pop_rank))
  {
    size_t nrows, j;

    nrows= rec_id_len / 4;

    ticks= UdmStartTimer();
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data %d rows\n", nrows);
    if (need_pop_rank || need_last_mod_time)
    {
      size_t i;
      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        if (rec_id == DataList->Item[j].url_id)
        {
          UDM_URLDATA *D= &DataList->Item[j];
          if (need_site_id)
          D->site_id= udm_get_int4(site_id_str);
          if (need_pop_rank)
            D->pop_rank= *((const double *)pop_rank_str);
          if (need_last_mod_time)
            D->last_mod_time= udm_get_int4(last_mod_time_str);
          j++;
          if (j == DataList->nitems) break;
        }
        rec_id_str+= 4;
        site_id_str+= 4;
        pop_rank_str+= 8;
        last_mod_time_str+= 4;
      }
    }
    else if (group_by_site && need_site_id)
    {
      if (UDM_OK != UdmURLDataListGroupBySiteUsingHash(A, DataList,
                                                      rec_id_str, rec_id_len,
                                                      site_id_str, site_id_len))
      {
        j= DataList->nitems;
        group_by_site= 1;
      }
      else
      {
        j= DataList->nitems;
        group_by_site= 0;
      }
    }
    else
    {
      UDM_URLDATA *Data= DataList->Item;
      size_t i, skip= 0, ncoords= DataList->nitems;

      for (j = 0, i = 0; i < nrows; i++)
      {
        urlid_t rec_id= udm_get_int4(rec_id_str);
        while (rec_id > Data[j].url_id && j < ncoords)
        {
          skip++;
          j++;
        }

        if (rec_id == Data[j].url_id)
        {
          if (need_site_id)
            Data[j].site_id= udm_get_int4(site_id_str + i*4);
          j++;
          if (j == ncoords) break;
        }
        rec_id_str+= 4;
      }
      if (j < ncoords)
      {
        skip+= (ncoords - j);
        UdmLog(A, UDM_LOG_DEBUG,
               "Warning: %d out of %d coords didn't have URL data", skip, DataList->nitems);
        UdmLog(A, UDM_LOG_DEBUG, "GroupBySite may work incorrectly");
        j= DataList->nitems;
      }
    }

/*
site_fin:
*/
    if (j == DataList->nitems)
    {
      if (group_by_site)
        rc= UdmURLDataListGroupBySiteUsingSort(A, DataList, db);
      else
        rc= UDM_OK;
      goto ret;
    }

    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking URL Data done: %.02f", (float)ticks/1000);
    UdmLog(A,UDM_LOG_DEBUG,"Expected to load %d URLs, loaded %d", DataList->nitems, j);
    UdmLog(A,UDM_LOG_DEBUG,"Couldn't load URL data from bdict");
  }
  else
  {
    UdmLog(A,UDM_LOG_DEBUG,"There is no URL data in bdict");
  }
  rc= UdmLoadURLDataFromURL(A, DataList, db, num_best_rows, flags);
  
ret:
  UdmSQLFree(&SQLres);
  UdmDSTRFree(&rec_id_buf);
  UdmDSTRFree(&site_id_buf);
  UdmDSTRFree(&pop_rank_buf);
  UdmDSTRFree(&last_mod_time_buf);
  return rc;
}

/****************************** User score **************************/ 

static int
UdmUserScoreListLoad(UDM_AGENT *A, UDM_DB *db,
                     UDM_URL_INT4_LIST *List, const char *q)
{
  size_t i;
  int rc;
  UDM_SQLRES SQLRes;
  bzero((void*) List, sizeof(UDM_URL_INT4_LIST));

  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, q)))
    goto ret;

  if (!(List->nitems= UdmSQLNumRows(&SQLRes)))
    goto sqlfree;

  UdmLog(A, UDM_LOG_DEBUG,
         "UserScore query returned %d columns, %d rows",
         UdmSQLNumCols(&SQLRes), List->nitems);
  
  if (2 != UdmSQLNumCols(&SQLRes))
  {
    udm_snprintf(db->errstr, sizeof(db->errstr),
                 "User Score query must return 2 columns, returned %d columns",
                 UdmSQLNumCols(&SQLRes));
    rc= UDM_ERROR;
    db->errcode= 1;
    goto sqlfree;
  }

  if (!(List->Item= UdmMalloc(sizeof(UDM_URL_INT4) * List->nitems)))
  {
    rc= UDM_ERROR;
    List->nitems= 0;
    goto sqlfree;
  }
  for (i= 0; i < List->nitems; i++)
  {
    List->Item[i].url_id= atoi(UdmSQLValue(&SQLRes, i, 0));
    List->Item[i].param= atoi(UdmSQLValue(&SQLRes, i, 1));
  }
  UdmSort(List->Item, List->nitems, sizeof(UDM_URL_INT4), (udm_qsort_cmp)cmpaurls);

sqlfree:
  UdmSQLFree(&SQLRes);
ret:
  return rc;
}


static int
UdmUserScoreListLoadAndApplyToURLScoreList(UDM_AGENT *Agent,
                                           UDM_URLSCORELIST *List,
                                           UDM_DB *db)
{
  char name[128];
  const char *us, *query;
  UDM_URL_INT4_LIST UserScoreList;
  int rc;
  int UserScoreFactor= UdmVarListFindInt(&Agent->Conf->Vars, "UserScoreFactor", 0);
  
  if (!UserScoreFactor ||
      !(us= UdmVarListFindStr(&Agent->Conf->Vars, "us", NULL)))
    return UDM_OK;
  udm_snprintf(name, sizeof(name), "Score.%s", us);
  if (!(query= UdmVarListFindStr(&Agent->Conf->Vars, name, NULL)))
    return UDM_OK;
  
  if ((UDM_OK != (rc= UdmUserScoreListLoad(Agent, db,
                                           &UserScoreList, query))) ||
      !UserScoreList.nitems)
    goto ret;

  rc= UdmUserScoreListApplyToURLScoreList(List, &UserScoreList, UserScoreFactor);
  
ret:
  UDM_FREE(UserScoreList.Item);
  return rc;
}


static int
UdmUserSiteScoreListLoadAndApplyToURLDataList(UDM_AGENT *Agent,
                                              UDM_URLDATALIST *List,
                                              UDM_DB *db)
{
  char name[128];
  const char *us, *query;
  UDM_URL_INT4_LIST UserScoreList;
  int rc;
  int UserScoreFactor= UdmVarListFindInt(&Agent->Conf->Vars, "UserScoreFactor", 0);

  if (!UserScoreFactor ||
      !(us= UdmVarListFindStr(&Agent->Conf->Vars, "ss", NULL)))
    return UDM_OK;
  udm_snprintf(name, sizeof(name), "SiteScore.%s", us);
  if (!(query= UdmVarListFindStr(&Agent->Conf->Vars, name, NULL)))
    return UDM_OK;
  
  if ((UDM_OK != (rc= UdmUserScoreListLoad(Agent, db,
                                           &UserScoreList, query))) ||
      !UserScoreList.nitems)
    goto ret;

  rc= UdmUserScoreListApplyToURLDataList(List, &UserScoreList, UserScoreFactor);

ret:
  UDM_FREE(UserScoreList.Item);
  return rc;
}



/*********************** Creating fast index ******************/

/*
  Write limits with COMMIT and timestamp
*/
int
UdmBlobWriteLimits(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  int rc;
  if (UDM_OK != (rc= UdmSQLBegin(db)) ||
      UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)) ||
      UDM_OK != (rc= UdmSQLCommit(db)))
    return rc;
  return UDM_OK;
}


int UdmConvert2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  int rc;
  UDM_URLDATALIST List;
  
  UdmSQLBuildWhereCondition(Indexer->Conf, db);
  
  if (UDM_OK != (rc= UdmLoadURLDataFromURLAndSlowLimitForConv(Indexer, db, &List)))
    return rc;
  
  if (db->DBMode == UDM_DBMODE_MULTI)
    rc= UdmMulti2BlobSQL(Indexer, db, &List);
  else if (db->DBMode == UDM_DBMODE_SINGLE)
    rc= UdmSingle2BlobSQL(Indexer, db, &List);
  else if (db->DBMode == UDM_DBMODE_BLOB)
    rc= UdmBlob2BlobSQL(Indexer, db, &List);
  
  UdmFree(List.Item);
  return rc;
}


static int
UdmStoreWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  switch(db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      return(UdmStoreWordsBlob(Indexer, Doc, db));
    case UDM_DBMODE_MULTI:
      return(UdmStoreWordsMulti(Indexer, Doc, db));
    case UDM_DBMODE_SINGLE:
    default:
      return(StoreWordsSingle(Indexer, Doc, db));
  }
}



/********************* Inserting URLs and Links ***************************/

static int
InsertLink(UDM_AGENT *A, UDM_DB *db, urlid_t from, urlid_t to)
{
  char qbuf[128];
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  if (from == to) /* Don't collect self links */
    return UDM_OK;
  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO links (ot,k,weight) VALUES (%s%i%s,%s%d%s,0.0)",
               qu, from, qu, qu, to, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}


static int
UdmAddURL(UDM_AGENT *Indexer,UDM_DOCUMENT * Doc,UDM_DB *db)
{
  char    *e_url, *qbuf;
  const char  *url;
  int    url_seed;
  int    use_crc32_url_id;
  int    usehtdburlid;
  int    rc=UDM_OK;
  size_t          len;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char *sql_export= UdmVarListFindStr(&Doc->Sections, "SQLExportHref", NULL);
  urlid_t rec_id = 0;

  if (sql_export)
  {
    char *part, *lt, *sql_export_copy= UdmStrdup(sql_export);
    UDM_DSTR d;
    UDM_VARLIST Vars;
    UdmVarListInit(&Vars);
    UdmDSTRInit(&d,256);
    
    UdmVarListSQLEscape(&Vars, &Doc->Sections, db);
    for (part= udm_strtok_r(sql_export_copy, ";", &lt) ;
         part ;
         part= udm_strtok_r(NULL, ";", &lt))
    {
      UdmDSTRParse(&d, part, &Vars);
      if(UDM_OK!= (rc= UdmSQLQuery(db, NULL, d.data)))
        return rc;
      UdmDSTRReset(&d);
    }
    UdmVarListFree(&Vars);
    UdmDSTRFree(&d);
    UdmFree(sql_export_copy);
  }
  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");
  usehtdburlid = UdmVarListFindInt(&Indexer->Conf->Vars, "UseHTDBURLId", 0);

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL)
  { 
    UDM_FREE(e_url);
    return UDM_ERROR;
  }
  
  url_seed = UdmStrHash32(url) & 0xFF;
  
  /* Escape URL string */
  UdmSQLEscStr(db, e_url, url, len);
  
  if(use_crc32_url_id || usehtdburlid)
  {
    /* Auto generation of rec_id */
    /* using CRC32 algorythm     */
    if (use_crc32_url_id) rec_id = UdmStrHash32(url);
    else rec_id = UdmVarListFindInt(&Doc->Sections, "HTDB_URL_ID", 0);
    
    udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (rec_id,url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES (%s%i%s,'%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
           qu, rec_id, qu,
           e_url,
           qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
           UdmVarListFindInt(&Doc->Sections,"Hops",0),
           (int)time(NULL),
           url_seed, (int)time(NULL),
           qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
           UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                              UdmVarListFindStr(&Doc->Sections, "Date", "")))
       );
  }else{
    /* Use dabatase generated rec_id */
    /* It depends on used DBType     */
    switch(db->DBType)
    {
    case UDM_DB_SOLID:
    case UDM_DB_ORACLE8:
    case UDM_DB_SAPDB:
      /* FIXME: Dirty hack for stupid too smart databases 
       Change this for config parameter checking */
/*      if (strlen(e_url)>UDM_URLSIZE)e_url[UDM_URLSIZE]=0;*/
      /* Use sequence next_url_id.nextval */
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,next_url_id.nextval,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MIMER:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,NEXT_VALUE OF rec_id_GEN,0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;    
    case UDM_DB_IBASE:
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id) VALUES ('%s',%i,%d,GEN_ID(rec_id_GEN,1),0,%d,0,%d,%d,%i,%i)",
        e_url,
        UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
        UdmVarListFindInt(&Doc->Sections,"Hops",0),
        (int)time(NULL),
         url_seed, (int)time(NULL),
         UdmVarListFindInt(&Doc->Sections, "Site_id", 0),
         UdmVarListFindInt(&Doc->Sections, "Server_id", 0)
         );
      break;
    case UDM_DB_MYSQL:
      /* MySQL generates itself */
    default:  
      udm_snprintf(qbuf, 4 * len + 512, "INSERT INTO url (url,referrer,hops,crc32,next_index_time,status,seed,bad_since_time,site_id,server_id,docsize,last_mod_time,shows,pop_rank) VALUES ('%s',%s%i%s,%d,0,%d,0,%d,%d,%s%i%s,%s%i%s,%s%i%s,%li,0,0.0)",
             e_url,
             qu, UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0), qu,
             UdmVarListFindInt(&Doc->Sections,"Hops",0),
             (int)time(NULL),
             url_seed, (int)time(NULL),
             qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Server_id", 0), qu,
             qu, UdmVarListFindInt(&Doc->Sections, "Content-Length", 0), qu,
             UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections, "Last-Modified",
             UdmVarListFindStr(&Doc->Sections, "Date", "")))
         );
    }
  }

  /* Exec INSERT now */
  if(UDM_OK!=(rc=UdmSQLQuery(db, NULL, qbuf)))
    goto ex;

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return rc;
}


static int
UdmAddLink(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db)
{
  char    *e_url, *qbuf;
  UDM_SQLRES  SQLRes;
  const char  *url;
  int    use_crc32_url_id;
  int    rc=UDM_OK;
  size_t          len;
  urlid_t rec_id = 0;

  url = UdmVarListFindStr(&Doc->Sections,"URL","");
  use_crc32_url_id = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars, "UseCRC32URLId", "no"), "yes");

  len = strlen(url);
  e_url = (char*)UdmMalloc(4 * len + 1);
  if (e_url == NULL) return UDM_ERROR;
  qbuf = (char*)UdmMalloc(4 * len + 512);
  if (qbuf == NULL) 
  { 
    UDM_FREE(e_url); 
    return UDM_ERROR;
  }
  
  if (use_crc32_url_id)
  {
    rec_id = UdmStrHash32(url);
  }
  else
  {
    /* Escape URL string */
    UdmSQLEscStr(db, e_url, url, len);
  
    udm_snprintf(qbuf, 4 * len + 512, "SELECT rec_id FROM url WHERE url='%s'", e_url);
    if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) 
      goto ex;
    if (UdmSQLNumRows(&SQLRes))
    {
      rec_id = UDM_ATOI(UdmSQLValue(&SQLRes, 0, 0));
    }
    UdmSQLFree(&SQLRes);
  }

  if (rec_id)
  {
    urlid_t from= UdmVarListFindInt(&Doc->Sections, "Referrer-ID", 0);
    UdmVarListReplaceInt(&Doc->Sections, "ID", rec_id);
    if(UDM_OK != (rc = InsertLink(Indexer, db, from, rec_id)))
      goto ex;
  }
  else
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "URL not found: %s", e_url);
  }

ex:

  UDM_FREE(qbuf);
  UDM_FREE(e_url);
  return UDM_OK;
}


/******************* Cached Copy *********************/

static void
SQLResToDoc(UDM_ENV *Conf, UDM_DOCUMENT *D, UDM_SQLRES *sqlres, size_t i)
{
  time_t    last_mod_time;
  char    dbuf[128];
  const char  *format = UdmVarListFindStr(&Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");
  double          pr;
  
  UdmVarListReplaceStr(&D->Sections,"URL",UdmSQLValue(sqlres,i,1));
  UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(sqlres,i,1)));
  last_mod_time=atol(UdmSQLValue(sqlres,i,2));
  UdmVarListReplaceInt(&D->Sections, "Last-Modified-Timestamp", (int) last_mod_time);
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Last-Modified",dbuf);
  UdmVarListReplaceStr(&D->Sections,"Content-Length",UdmSQLValue(sqlres,i,3));
  pr= atof(UdmSQLValue(sqlres,i,3)) / 1024;
  sprintf(dbuf, "%.2f", pr);
  UdmVarListReplaceStr(&D->Sections,"Content-Length-K",dbuf);  
  last_mod_time=atol(UdmSQLValue(sqlres,i,4));
  if (strftime(dbuf, 128, format, localtime(&last_mod_time)) == 0)
  {
    UdmTime_t2HttpStr(last_mod_time, dbuf);
  }
  UdmVarListReplaceStr(&D->Sections,"Next-Index-Time",dbuf);
  UdmVarListReplaceInt(&D->Sections, "Referrer-ID", UDM_ATOI(UdmSQLValue(sqlres,i,5)));
  UdmVarListReplaceInt(&D->Sections,"crc32",atoi(UdmSQLValue(sqlres,i,6)));
  UdmVarListReplaceStr(&D->Sections, "Site_id", UdmSQLValue(sqlres, i, 7));

#if BAR_COMMA_PERIOD_ORACLE_PROBLEM
  {
	char *num= UdmSQLValue(sqlres, i, 8);
	char *comma= strchr(num, ',');
	if (comma)
	  *comma= '.';
  }
#endif

  pr = atof(UdmSQLValue(sqlres, i, 8));
  snprintf(dbuf, 128, "%.5f", pr);
  UdmVarListReplaceStr(&D->Sections, "Pop_Rank", dbuf);
}


static int
UdmGetCachedCopy(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
#ifdef HAVE_ZLIB
  UDM_SQLRES SQLRes;
  char buf[1024];
  const char *sname;
  const char *sval;
  int _;
  int i;
  int CachedCopy_found= 0;

  UdmFindURL(Indexer, Doc, db);
  udm_snprintf(buf, sizeof(buf), "SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);
  if (! UdmSQLNumRows(&SQLRes)) return(UDM_ERROR);
  SQLResToDoc(Indexer->Conf, Doc, &SQLRes, 0);
  UdmSQLFree(&SQLRes);
  udm_snprintf(buf, sizeof(buf), "SELECT sname, sval FROM urlinfo WHERE url_id=%d", UDM_ATOI(UdmVarListFindStr(&Doc->Sections, "ID", "0")));
  _ = UdmSQLQuery(db, &SQLRes, buf);
  if (_ != UDM_OK) return(_);

  for (i = 0; i < UdmSQLNumRows(&SQLRes); i++)
  {
    sname = UdmSQLValue(&SQLRes, i, 0);
    sval = UdmSQLValue(&SQLRes, i, 1);
    if (! sname) continue;
    if (! sval) sval = "";

    if (! strcmp(sname, "CachedCopy")) 
    {
      size_t l;

      if (Doc->Buf.content) continue;

      UdmVarListReplaceStr(&Doc->Sections, "CachedCopyBase64", sval);
      l = strlen(sval);
      Doc->Buf.buf = UdmMalloc(UDM_MAXDOCSIZE);
      if (UDM_OK != UdmCachedCopyUnpack(Doc, sval, l))
        return UDM_ERROR;
      CachedCopy_found= 1;
    }
    else
    {
      UdmVarListReplaceStr(&Doc->Sections, sname, sval);
    }
  }
  
  UdmSQLFree(&SQLRes);
  
  if (!CachedCopy_found)
  {
    const char *url= UdmVarListFindStr(&Doc->Sections, "url", NULL);
    UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
    UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
    UdmGetURLSimple(Indexer, Doc, url);
    UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  }
  
  return(UDM_OK);
#else
  return(UDM_ERROR);
#endif
}


/********************** Reindexing "indexer -a" *************************/

static int
UdmMarkForReindex(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[1024];
  const char  *where;
  UDM_SQLRES   SQLRes;
  size_t          i, j;
  int             rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  UDM_DSTR buf;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = UdmSQLBuildWhereCondition(Indexer->Conf, db);
  
  if (db->flags & UDM_SQL_HAVE_SUBSELECT &&
      db->DBType != UDM_DB_MYSQL)
  {
    udm_snprintf(qbuf,sizeof(qbuf),"UPDATE url SET next_index_time=%d WHERE rec_id IN (SELECT url.rec_id FROM url%s %s %s)",
       (int)time(NULL), db->from, (where[0]) ? "WHERE" : "", where);
    return UdmSQLQuery(db,NULL,qbuf);
  }

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s %s %s", db->from, (where[0]) ? "WHERE" : "", where);
  if(UDM_OK != (rc = UdmSQLQuery(db, &SQLRes, qbuf))) return rc;

  UdmDSTRInit(&buf, 4096);
  if (db->DBSQL_IN) 
  {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i += 512) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id IN (", (int)time(NULL));
      for (j = 0; (j < 512) && (i + j < UdmSQLNumRows(&SQLRes)); j++) 
      {
        UdmDSTRAppendf(&buf, "%s%s%s%s", (j) ? "," : "", qu, UdmSQLValue(&SQLRes, i + j, 0), qu);
      }
      UdmDSTRAppendf(&buf, ")");
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  } else {
    for (i = 0; i < UdmSQLNumRows(&SQLRes); i++) 
    {
      UdmDSTRReset(&buf);
      UdmDSTRAppendf(&buf, "UPDATE url SET next_index_time=%d WHERE rec_id=%s", (int)time(NULL),  UdmSQLValue(&SQLRes, i, 0));
      if(UDM_OK != (rc = UdmSQLQuery(db, NULL, buf.data))) 
      {
        UdmSQLFree(&SQLRes);
	UdmDSTRFree(&buf);
        return rc;
      }
    }
  }
  UdmDSTRFree(&buf);
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


/************** Child - for new extensions ****************/

static int
UdmRegisterChild(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[1024];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections,"ID",0);
  urlid_t  parent_id = UdmVarListFindInt(&Doc->Sections,"Parent-ID",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  udm_snprintf(qbuf,sizeof(qbuf),"insert into links (ot,k,weight) values(%s%i%s,%s%i%s,0.0)", qu, parent_id, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}


/*********************** Update URL ***********************/

static int
UdmUpdateUrl(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char qbuf[256];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  status=UdmVarListFindInt(&Doc->Sections,"Status",0);
  int  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  int  next_index_time=UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time",""));
  int  res;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (prevStatus != status && status > 300 && status != 304)
    sprintf(qbuf, "UPDATE url SET status=%d,next_index_time=%d,bad_since_time=%d,site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, (int)time(NULL), qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
           qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);
  else
    sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d, site_id=%s%i%s,server_id=%s%i%s WHERE rec_id=%s%i%s",
            status, next_index_time, qu, UdmVarListFindInt(&Doc->Sections, "Site_id", 0), qu,
            qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu, qu, url_id, qu);

  if(UDM_OK!=(res=UdmSQLQuery(db,NULL,qbuf)))return res;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  return UdmDeleteBadHrefs(Indexer,Doc,db,url_id);
}

static int
UdmUpdateUrlWithLangAndCharset(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  *qbuf;
  int  rc;
  const char  *charset;
  UDM_VAR    *var;
  int    status, prevStatus;
  urlid_t         url_id;
  size_t    i, len = 0;
  char    qsmall[64];
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int IndexTime= UdmVarListFindInt(&Indexer->Conf->Vars, "IndexTime", 0);
  
  status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
  prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  
  if((var=UdmVarListFind(&Doc->Sections,"Content-Language")))
  {
    if (var->val == NULL)
    {
      var->val = (char*)UdmStrdup(UdmVarListFindStr(&Doc->Sections, "DefaultLang", "en"));
    }
    len=strlen(var->val);
    for(i = 0; i < len; i++)
    {
      var->val[i] = tolower(var->val[i]);
    }
  }
  
  charset = UdmVarListFindStr(&Doc->Sections, "Charset", 
            UdmVarListFindStr(&Doc->Sections, "RemoteCharset", "iso-8859-1"));
  charset = UdmCharsetCanonicalName(charset);
  UdmVarListReplaceStr(&Doc->Sections, "Charset", charset);
  
  if (prevStatus != status && status > 300 && status != 304)
    udm_snprintf(qsmall, 64, ", bad_since_time=%d", (int)time(NULL));
  else qsmall[0] = '\0';

  if (IndexTime)
  {
    if (! prevStatus) udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", time(NULL));
  }
  else
  {
    const char *lmsrc= UdmVarListFindStr(&Doc->Sections, "User.Date",
                       UdmVarListFindStr(&Doc->Sections, "Last-Modified",
                       UdmVarListFindStr(&Doc->Sections, "Date", "")));
    udm_snprintf(UDM_STREND(qsmall), 64, ",last_mod_time=%li", UdmHttpDate2Time_t(lmsrc));
  }
  qbuf=(char*)UdmMalloc(1024);
  
  
  udm_snprintf(qbuf, 1023, "\
UPDATE url SET \
status=%d,\
next_index_time=%li,\
docsize=%d,\
crc32=%d%s, site_id=%s%i%s, server_id=%s%i%s \
WHERE rec_id=%s%i%s",
  status,
  UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
  UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
  UdmVarListFindInt(&Doc->Sections,"crc32",0),
  qsmall,
  qu, UdmVarListFindInt(&Doc->Sections,"Site_id",0), qu,
  qu, UdmVarListFindInt(&Doc->Sections, "Server_id",0), qu,
  qu, url_id, qu);
  
  rc=UdmSQLQuery(db,NULL,qbuf);
  UDM_FREE(qbuf);
  return rc;
}


static int
UdmDocInsertSectionsUsingBind(UDM_DB *db, UDM_DOCUMENT *Doc)
{
  int rc= UDM_OK;
  size_t i;
  char qbuf[256];
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);

  UDM_ASSERT(db->sql->SQLPrepare);

  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO urlinfo (url_id,sname,sval) "
               "VALUES(%s, %s, %s)",
                UdmSQLParamPlaceHolder(db, 1),
                UdmSQLParamPlaceHolder(db, 2),
                UdmSQLParamPlaceHolder(db, 3));

  if (UDM_OK != (rc= UdmSQLPrepare(db, qbuf)))
    return rc;
  
  for(i= 0; i< Doc->Sections.nvars; i++)
  {
    UDM_VAR *Sec=&Doc->Sections.Var[i];
    if (Sec->val && Sec->name &&
        ((Sec->curlen && Sec->maxlen) || (!strcmp(Sec->name, "Z"))))
    {
      int bindtype= UdmSQLLongVarCharBindType(db);

      if (UDM_OK != (rc= UdmSQLBindParameter(db, 1,
                                             &url_id, (int) sizeof(url_id),
                                             UDM_SQLTYPE_INT)) ||
          UDM_OK != (rc= UdmSQLBindParameter(db, 2,
                                             Sec->name, (int) strlen(Sec->name),
                                             UDM_SQLTYPE_VARCHAR)) ||
          UDM_OK != (rc= UdmSQLBindParameter(db, 3,
                                             Sec->val, (int) strlen(Sec->val),
                                             bindtype)) ||
          UDM_OK != (rc= UdmSQLExecute(db)))
        return rc;
    }
  }

  return UdmSQLStmtFree(db);
}


static int
UdmLongUpdateURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int    rc=UDM_OK;
  size_t    i;
  char    qsmall[128];
  size_t    len=0;
  urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *c;
  int use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);
  int use_tnx=  (db->DBType != UDM_DB_MYSQL)    && /* OK */
                (db->DBType != UDM_DB_VIRT)     && /* TODO: check */
                (db->DBType != UDM_DB_ACCESS)   && /* TODO: check */
                (db->DBType != UDM_DB_DB2)      && /* TODO: check */
                (db->DBType != UDM_DB_CACHE)    && /* TODO: check */
                (db->DBType != UDM_DB_SYBASE);     /* TODO: check */

  if (use_tnx && UDM_OK != (rc= UdmSQLBegin(db)))
    return rc;
  
  /* Now store words and crosswords */
  if(UDM_OK != (rc= UdmStoreWords(Indexer, Doc, db)))
    return rc;
  
  if(use_crosswords)
    if(UDM_OK != (rc= UdmStoreCrossWords(Indexer, Doc, db)))
      return rc;
  

  /* Copy default languages, if not given by server and not guessed */
  if (!(c= UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL)))
  {
    if ((c= UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
      UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
  }
  

  if(UDM_OK != (rc= UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db)))
    return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if (UDM_OK != (rc= UdmDeleteBadHrefs(Indexer,Doc,db,url_id)))
    return rc;
  
  sprintf(qsmall,"DELETE FROM urlinfo WHERE url_id=%i", url_id);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qsmall)))
    return rc;

/* No need delete from links here, it has been done before */
  
  if (!Doc->Sections.nvars)
    goto commit;

  for(len= 0, i= 0; i < Doc->Sections.nvars; i++)
  {
    size_t l= Doc->Sections.Var[i].curlen +
              (Doc->Sections.Var[i].name ?
               strlen(Doc->Sections.Var[i].name) : 0);
    if (len < l)
      len= l;
  }
  if (!len)
    goto commit;


  if (db->flags & UDM_SQL_HAVE_BIND)
  {
    rc= UdmDocInsertSectionsUsingBind(db, Doc);
  }
  else
  {
    const char  *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
    size_t esc_multiply = (db->DBType == UDM_DB_PGSQL) ? 4 : 2;
    char *qbuf= (char*) UdmMalloc(esc_multiply * len + 128);
    char *arg= (char*) UdmMalloc(esc_multiply * len + 128);
    const char *E= (db->DBDriver == UDM_DB_PGSQL && db->version >= 80101) ? "E" : "";

    for(i= 0; i< Doc->Sections.nvars; i++)
    {
      UDM_VAR *Sec=&Doc->Sections.Var[i];
      if (Sec->val && Sec->name &&
          ((Sec->curlen && Sec->maxlen) || (!strcmp(Sec->name, "Z"))))
      {
        arg= UdmSQLEscStr(db,arg,Sec->val,strlen(Sec->val)); /* urlinfo value */
        sprintf(qbuf,"INSERT INTO urlinfo (url_id,sname,sval) VALUES (%s%i%s,'%s',%s'%s')",
                qu, url_id, qu, Sec->name, E, arg);
        if(UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
          break;
      }
    }
    UDM_FREE(qbuf);
    UDM_FREE(arg);
  }

commit:

  if(use_tnx && rc == UDM_OK)
    rc= UdmSQLCommit(db);

  return rc;
}


static int
UdmUpdateClone(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db)
{
  int rc;
  int use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer, Doc, db)))
    return rc;
  if(use_crosswords)
  {
    if (UDM_OK != (rc= UdmDeleteCrossWordFromURL(Indexer, Doc, db)))
      return rc;
  }
  rc= UdmUpdateUrlWithLangAndCharset(Indexer, Doc, db);
  return rc;
}



/************************ Clones stuff ***************************/
static int
UdmFindOrigin(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  size_t    i=0;
  char    qbuf[256]="";
  UDM_SQLRES  SQLRes;
  urlid_t    origin_id = 0;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  int    rc;
  
  if (scrc32==0)return UDM_OK;
  
  if (db->DBSQL_IN)
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status IN (200,304,206)",scrc32);
  else
    sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206)",scrc32);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
    return rc;
  
  for(i=0;i<UdmSQLNumRows(&SQLRes);i++)
  {
    const char *o;
    if((o=UdmSQLValue(&SQLRes,i,0)))
      if((!origin_id) || (origin_id > UDM_ATOI(o)))
        origin_id = UDM_ATOI(o);
  }
  UdmSQLFree(&SQLRes);
  UdmVarListReplaceInt(&Doc->Sections, "Origin-ID", origin_id);
  return(UDM_OK);
}


int
UdmCloneListSQL(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_RESULT *Res, UDM_DB *db)
{
  size_t    i, nr, nadd;
  char    qbuf[256];
  UDM_SQLRES  SQLres;
  int    scrc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
  urlid_t    origin_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int    rc;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char  *format = UdmVarListFindStr(&Indexer->Conf->Vars, "DateFormat", "%a, %d %b %Y, %X %Z");

  if (Res->num_rows > 4) return UDM_OK;
  
  if (!scrc32)
    return UDM_OK;
  
  sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206) AND rec_id<>%s%i%s", scrc32, qu, origin_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return UDM_OK;
  
  nr = UdmSQLNumRows(&SQLres);
  if( nr == 0)
  {
    UdmSQLFree(&SQLres);
    return UDM_OK;
  }
  nadd = 5 - Res->num_rows;
  if(nr < nadd) nadd = nr;
  
  Res->Doc = (UDM_DOCUMENT*)UdmRealloc(Res->Doc, (Res->num_rows + nadd) * sizeof(UDM_DOCUMENT));
  
  for(i = 0; i < nadd; i++)
  {
    time_t    last_mod_time;
    char    buf[128];
    UDM_DOCUMENT  *D = &Res->Doc[Res->num_rows + i];
    
    UdmDocInit(D);
    UdmVarListAddInt(&D->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLres,i,0)));
    UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(&SQLres,i,1));
    UdmVarListReplaceInt(&D->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLres,i,1)));
    last_mod_time=atol(UdmSQLValue(&SQLres,i,2));
    if (strftime(buf, 128, format, localtime(&last_mod_time)) == 0)
    {
      UdmTime_t2HttpStr(last_mod_time, buf);
    }
    UdmVarListAddStr(&D->Sections,"Last-Modified",buf);
    UdmVarListAddInt(&D->Sections,"Content-Length",atoi(UdmSQLValue(&SQLres,i,3)));
    UdmVarListAddInt(&D->Sections,"crc32",scrc32);
    UdmVarListAddInt(&D->Sections, "Origin-ID", origin_id);
  }
  Res->num_rows += nadd;
  UdmSQLFree(&SQLres);
  return UDM_OK;
}


/************** Get Target to be indexed ***********************/

void
UdmSQLTopClause(UDM_DB *db, size_t top_num,
                char *topstr, size_t topstr_size,
                char *rownumstr, size_t rownumstr_size,
                char *limitstr, size_t limitstr_size)
{
  *topstr= *rownumstr= *limitstr= '\0';
  if(db->flags & UDM_SQL_HAVE_LIMIT)
  {
    udm_snprintf(limitstr, limitstr_size, " LIMIT %d", top_num);
  }
  else if (db->flags & UDM_SQL_HAVE_TOP)
  {
    udm_snprintf(topstr, topstr_size, " TOP %d ", top_num);
  }
  else if (db->flags & UDM_SQL_HAVE_FIRST_SKIP)
  {
    udm_snprintf(topstr, topstr_size, " FIRST %d ", top_num);
  }
  else if (db->DBType == UDM_DB_ORACLE8)
  {
#if HAVE_ORACLE8
    if(db->DBDriver==UDM_DB_ORACLE8)
    {
      udm_snprintf(rownumstr, rownumstr_size, " AND ROWNUM<=%d", top_num); 
    }
#endif
    if(!rownumstr[0])
      udm_snprintf(rownumstr, rownumstr_size, " AND ROWNUM<=%d", top_num); 
  }
}


int
UdmTargetsSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  char    sortstr[128]= "";
  char    updstr[64]="";
  char    rownumstr[64]="";
  char    limitstr[64]="";
  char    topstr[64]="";
  char    selectstr[]= "url.url,url.rec_id,docsize,status,hops,crc32,last_mod_time,seed";
  size_t    i = 0, j, start, nrows, qbuflen;
  size_t    url_num;
  UDM_SQLRES   SQLRes;
  char    smallbuf[128];
  int    rc=UDM_OK;
  const char  *where;
  char    *qbuf=NULL;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  int skip_lock= UdmVarListFindBool(&Indexer->Conf->Vars, "URLSelectSkipLock", 0);

  url_num= UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_SELECT_CACHE);
  if (Indexer->Conf->url_number < url_num)
    url_num= Indexer->Conf->url_number;
  where= UdmSQLBuildWhereCondition(Indexer->Conf, db);
  qbuflen= 1024 + 4 * strlen(where);
  
  if ((qbuf = (char*)UdmMalloc(qbuflen + 2)) == NULL)
  {
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      return UDM_ERROR;
  }

  if ((Indexer->flags & (UDM_FLAG_SORT_HOPS | UDM_FLAG_SORT_EXPIRED)) ||
      !(Indexer->flags & UDM_FLAG_DONTSORT_SEED))
  {    
    sprintf(sortstr, " ORDER BY %s%s%s", 
      (Indexer->flags & UDM_FLAG_SORT_HOPS) ? "hops" : "",
      (Indexer->flags & UDM_FLAG_DONTSORT_SEED) ? "" : ((Indexer->flags & UDM_FLAG_SORT_HOPS) ? ",seed" : "seed"),
      (Indexer->flags & UDM_FLAG_SORT_EXPIRED) ? 
      ( ((Indexer->flags & UDM_FLAG_SORT_HOPS) || !(Indexer->flags & UDM_FLAG_DONTSORT_SEED)  ) ? 
        ",next_index_time" : "next_index_time") : "");
  }

  UdmSQLTopClause(db, url_num,
                  topstr, sizeof(topstr),
                  rownumstr, sizeof(rownumstr),
                  limitstr, sizeof(limitstr));

  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(qbuf, qbuflen,
                     "INSERT INTO udm_url_tmp "
                     "SELECT url.rec_id FROM url%s "
                     "WHERE next_index_time<=%d %s%s%s%s",
                    db->from,
                    (int)time(NULL), where[0] ? "AND " : "",  where,
                    sortstr, limitstr);
        if (UDM_OK != (rc= UdmSQLDropTableIfExists(db, "udm_url_tmp")) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TEMPORARY TABLE udm_url_tmp (rec_id int not null)")) ||
            (!skip_lock &&
             UDM_OK != (rc= UdmSQLQuery(db,NULL,"LOCK TABLES udm_url_tmp WRITE, url WRITE, urlinfo AS it WRITE, urlinfo AS il WRITE, server AS s WRITE, categories AS c WRITE"))) ||
            UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
          return rc;
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
        sprintf(updstr, " FOR UPDATE ");
/*        rc=UdmSQLQuery(db,NULL,"LOCK url");*/
        break;
      case UDM_DB_ORACLE8:
        sprintf(updstr, " FOR UPDATE ");
        break;
      case UDM_DB_SAPDB:
        sprintf(updstr, " WITH LOCK ");
        break;
      default:
        break;
    }
    if(rc!=UDM_OK)return rc;
  }
  
  db->res_limit=url_num;
  if (db->DBType == UDM_DB_MYSQL)
    udm_snprintf(qbuf, qbuflen, "SELECT %s FROM url, udm_url_tmp "
                                "WHERE url.rec_id=udm_url_tmp.rec_id", selectstr);
  else
    udm_snprintf(qbuf, qbuflen, "SELECT %s%s "
                                "FROM url%s "
                                "WHERE next_index_time<=%d %s%s%s"
                                "%s%s%s",
                 topstr, selectstr, db->from,
                 (int)time(NULL), where[0] ? "AND " : "",  where, rownumstr,
                 sortstr, updstr, limitstr);
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLRes, qbuf)))
    goto commit;
  
  if(!(nrows = UdmSQLNumRows(&SQLRes)))
  {
    UdmSQLFree(&SQLRes);
    goto commit;
  }

  start = Indexer->Conf->Targets.num_rows;
  Indexer->Conf->Targets.num_rows += nrows;
  
  Indexer->Conf->Targets.Doc = 
    (UDM_DOCUMENT*)UdmRealloc(Indexer->Conf->Targets.Doc, sizeof(UDM_DOCUMENT)*(Indexer->Conf->Targets.num_rows + 1));
  if (Indexer->Conf->Targets.Doc == NULL)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory at realloc %s[%d]", __FILE__, __LINE__);
    rc= UDM_ERROR;
    goto commit;
  }
  
  for(i = 0; i < nrows; i++)
  {
    char    buf[64]="";
    time_t    last_mod_time;
    UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
    
    UdmDocInit(Doc);
    UdmVarListAddStr(&Doc->Sections,"URL",UdmSQLValue(&SQLRes,i,0));
    UdmVarListReplaceInt(&Doc->Sections, "URL_ID", UdmStrHash32(UdmSQLValue(&SQLRes,i,0)));
    UdmVarListAddInt(&Doc->Sections, "ID", UDM_ATOI(UdmSQLValue(&SQLRes,i,1)));
    UdmVarListAddInt(&Doc->Sections,"Content-Length",atoi(UdmSQLValue(&SQLRes,i,2)));
    UdmVarListAddInt(&Doc->Sections,"Status",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"PrevStatus",atoi(UdmSQLValue(&SQLRes,i,3)));
    UdmVarListAddInt(&Doc->Sections,"Hops",atoi(UdmSQLValue(&SQLRes,i,4)));
    UdmVarListAddInt(&Doc->Sections,"crc32",atoi(UdmSQLValue(&SQLRes,i,5)));
    last_mod_time = (time_t) atol(UdmSQLValue(&SQLRes,i,6));
    UdmTime_t2HttpStr(last_mod_time, buf);
    if (last_mod_time != 0 && strlen(buf) > 0)
    {
      UdmVarListReplaceStr(&Doc->Sections,"Last-Modified",buf);
    }
  }
  UdmSQLFree(&SQLRes);
  
  
  if (db->DBSQL_IN)
  {
    char  *urlin=NULL;
    
    if ( (qbuf = (char*)UdmRealloc(qbuf, qbuflen = qbuflen + 35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc= UDM_ERROR;
      goto commit;
    }
    
    if ( (urlin = (char*)UdmMalloc(35 * URL_SELECT_CACHE)) == NULL)
    {
      UDM_FREE(qbuf);
      UdmLog(Indexer, UDM_LOG_ERROR, "Out of memory");
      rc = UDM_ERROR;
      goto commit;
    }
    urlin[0]=0;
    
    for(i = 0; i < nrows; i+= URL_SELECT_CACHE)
    {

      urlin[0] = 0;

      for (j = 0; (j < URL_SELECT_CACHE) && (i + j < nrows) ; j++)
      {

      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i + j];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      if(urlin[0])strcat(urlin,",");
      sprintf(urlin+strlen(urlin), "%s%i%s", qu, url_id, qu);
      }
      udm_snprintf(qbuf, qbuflen, "UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",
             (int)(time(NULL) + URL_LOCK_TIME), urlin);
      if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf)))
        goto commit;
    }
    UDM_FREE(urlin);
  }
  else
  {
    for(i = 0; i < nrows; i++)
    {
      UDM_DOCUMENT  *Doc = &Indexer->Conf->Targets.Doc[start + i];
      urlid_t    url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
      
      udm_snprintf(smallbuf, 128, "UPDATE url SET next_index_time=%d WHERE rec_id=%i",
             (int)(time(NULL) + URL_LOCK_TIME), url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,smallbuf)))
        goto commit;
    }
  }


commit:

  if (rc != UDM_OK)
  {
    UdmLog(Indexer, UDM_LOG_ERROR, "UdmTargetsSQL: DB error: %s", db->errstr);
  }
  if(1)
  {
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        if (!skip_lock)
          rc= UdmSQLQuery(db,NULL,"UNLOCK TABLES");
        break;
      case UDM_DB_PGSQL:
        rc=UdmSQLQuery(db,NULL,"END WORK");
        break;
      default:
        break;
    }
  }
  UDM_FREE(qbuf);
  return rc;
}



/******************* Truncate database ********************/

static int
UdmTruncateURL(UDM_AGENT *Indexer,UDM_DB *db)
{
  int  rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "url");
  if(rc!=UDM_OK)return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "links");
  if(rc != UDM_OK) return rc;
  
  rc= UdmSQLTableTruncateOrDelete(db, "urlinfo");
  return rc;
}


static int
UdmTruncateDict(UDM_AGENT *Indexer,UDM_DB *db)
{
  switch(db->DBMode)
  {
  case UDM_DBMODE_MULTI:
    return UdmTruncateDictMulti(Indexer, db);
  
  case UDM_DBMODE_BLOB:
    return UdmTruncateDictBlob(Indexer, db);

  default:
    return UdmTruncateDictSingle(Indexer, db);
  }
  return UDM_ERROR; /* Make compiler happy */
}


static int
UdmTruncateDB(UDM_AGENT *Indexer,UDM_DB *db)
{
  int rc, use_crosswords;
  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);

  if(use_crosswords)
  {
    if ((UDM_OK != (rc= UdmTruncateCrossDict(Indexer,db))))
      return rc;
  }
  if((UDM_OK != (rc= UdmTruncateDict(Indexer,db))) ||
     (UDM_OK != (rc= UdmTruncateURL(Indexer,db))))
    return rc ;
  return rc;
}


/******************* Clear database with condition ********/

static int
UdmDeleteWordsAndLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  int  rc;
  int use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);
  if (use_crosswords)
    if (UDM_OK!= (rc= UdmDeleteCrossWordFromURL(Indexer,Doc,db)))
      return rc;

  if (UDM_OK != (rc= UdmDeleteWordFromURL(Indexer,Doc,db)))
    return rc;

  if (UDM_OK != (rc= UdmDeleteLinks(Indexer, Doc, db)))
    return rc;

  /* Set status, bad_since_time, etc */
  if (UDM_OK != (rc= UdmUpdateUrl(Indexer, Doc, db)))
    return rc;

  return rc;
}


static int
UdmDeleteWordFromURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  int rc;
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);

  if (!UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    return UDM_OK;

  switch (db->DBMode)
  {
    case UDM_DBMODE_BLOB:
      rc= UdmDeleteWordsFromURLBlob(Indexer, db, url_id);
      break;
    case UDM_DBMODE_MULTI:
      rc= UdmDeleteWordsFromURLMulti(Indexer, db, url_id);
      break;
    case UDM_DBMODE_SINGLE:
      rc= UdmDeleteWordsFromURLSingle(Indexer, db, url_id);
      break;
    default:
      rc= UDM_ERROR;
  }
  return rc;
}


static int
UdmDeleteBadHrefs(UDM_AGENT *Indexer,
                  UDM_DOCUMENT *Doc,
                  UDM_DB *db,
                  urlid_t url_id)
{
  UDM_DOCUMENT  rDoc;
  UDM_SQLRES  SQLRes;
  char    q[256];
  size_t    i;
  size_t    nrows;
  int    rc=UDM_OK;
  int    hold_period= UdmVarListFindInt(&Doc->Sections,"HoldBadHrefs",0);
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  
  if (hold_period <= 0)
    return UDM_OK;
  
  udm_snprintf(q, sizeof(q), "SELECT rec_id FROM url WHERE status > 300 AND status<>304 AND referrer=%s%i%s AND bad_since_time<%d",
    qu, url_id, qu, qu, qu, (int)time(NULL) - hold_period);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,q)))return rc;
  
  nrows = UdmSQLNumRows(&SQLRes);
  
  UdmDocInit(&rDoc);
  for(i = 0; i < nrows ; i++)
  {
    UdmVarListReplaceStr(&rDoc.Sections,"ID", UdmSQLValue(&SQLRes,i,0));
    if(UDM_OK!=(rc=UdmDeleteURL(Indexer, &rDoc, db)))
      break;
  }
  UdmDocFree(&rDoc);
  UdmSQLFree(&SQLRes);
  return rc;
}


static int
UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
  char  qbuf[128];
  int  rc;
  urlid_t  url_id  =UdmVarListFindInt(&Doc->Sections, "ID", 0);
  int  use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  if(use_crosswords)
    if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,Doc,db)))return(rc);
  
  if(UDM_OK!=(rc=UdmDeleteWordFromURL(Indexer,Doc,db)))return(rc);
  
  sprintf(qbuf,"DELETE FROM url WHERE rec_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  sprintf(qbuf,"DELETE FROM links WHERE k=%s%i%s", qu, url_id, qu);
  if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
  
  /* remove all old broken hrefs from this document to avoid broken link collecting */
  if (UDM_OK != (rc=UdmDeleteBadHrefs(Indexer,Doc, db, url_id))) return rc;

  sprintf(qbuf,"UPDATE url SET referrer=%s0%s WHERE referrer=%s%i%s", qu, qu, qu, url_id, qu);
  return UdmSQLQuery(db,NULL,qbuf);
}


static int
UdmDeleteLinks(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char  qbuf[128];
  urlid_t  url_id = UdmVarListFindInt(&Doc->Sections, "ID", 0);
  const char *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";

  sprintf(qbuf,"DELETE FROM links WHERE ot=%s%i%s", qu, url_id, qu);
  return UdmSQLQuery(db, NULL, qbuf);
}


static int
UdmClearDBUsingIN(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLID_LIST *list)
{
  UDM_DSTR qbuf, urlin;
  int rc= UDM_OK; /* if list if empty */
  size_t part;
  size_t url_num = UdmVarListFindInt(&Indexer->Conf->Vars, "URLSelectCacheSize", URL_DELETE_CACHE);
  
  UdmDSTRInit(&qbuf, 4096);
  UdmDSTRInit(&urlin, 4096);
  
  for (part= 0; part < list->nurls; part+= url_num)
  {
    size_t offs;
    urlid_t *item= &list->urls[part];
    UdmDSTRReset(&urlin);
    for(offs= 0; (offs < url_num) && ((part + offs) < list->nurls); offs++)
    {
      if(offs) UdmDSTRAppend(&urlin,",", 1);
      UdmDSTRAppendf(&urlin, "%d", item[offs]);
    }
    
    if(UDM_OK != (rc= UdmSQLBegin(db)))
      goto ret;
    
    switch(db->DBMode)
    {
      case UDM_DBMODE_BLOB:
        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,
                       "DELETE FROM bdicti WHERE state=1 AND url_id IN (%s)",
                       urlin.data);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
          goto ret;

        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,
                       "UPDATE bdicti SET state=0 WHERE state=2 AND url_id IN (%s)",
                       urlin.data);
        if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
          goto ret;
        break;
      
      case UDM_DBMODE_MULTI:
        {
          size_t dictnum;
          for (dictnum= 0; dictnum <= MULTI_DICTS; dictnum++)
          {
            UdmDSTRReset(&qbuf);
            UdmDSTRAppendf(&qbuf,"DELETE FROM dict%02X WHERE url_id in (%s)",
                           dictnum, urlin.data);
            if(UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf.data)))
              goto ret;
          }
        }
        break;
        
      default:
        UdmDSTRReset(&qbuf);
        UdmDSTRAppendf(&qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin.data);
        if (UDM_OK != (rc=UdmSQLQuery(db,NULL,qbuf.data)))
          goto ret;
        break;
    }
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM urlinfo WHERE url_id in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE ot in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;
    
    UdmDSTRReset(&qbuf);
    UdmDSTRAppendf(&qbuf,"DELETE FROM links WHERE k in (%s)",urlin.data);
    if (UDM_OK != (rc= UdmSQLQuery(db,NULL,qbuf.data)))
      goto ret;

    if(UDM_OK != (rc= UdmSQLCommit(db)))
      goto ret;
  }
  
ret:
  UdmDSTRFree(&qbuf);
  UdmDSTRFree(&urlin);
  return rc;
}


static int
UdmClearDBUsingLoop(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLID_LIST *list)
{
  int rc= UDM_OK;
  size_t i;
  UDM_DOCUMENT Doc;
  bzero((void*)&Doc, sizeof(Doc));

  for(i=0; i < list->nurls; i++)
  {
    UdmVarListReplaceInt(&Doc.Sections, "ID", list->urls[i]);
    if(UDM_OK != (rc= UdmDeleteURL(Indexer, &Doc, db)))
      break;
  }
  UdmDocFree(&Doc);
  return rc;
}


int
UdmClearDBSQL(UDM_AGENT *Indexer,UDM_DB *db)
{
  int rc, use_crosswords;
  const char *where, *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  char ClearDBHook[128];

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  where = UdmSQLBuildWhereCondition(Indexer->Conf, db);
  use_crosswords= UdmVarListFindBool(&Indexer->Conf->Vars, "CrossWords", 0);
  udm_snprintf(ClearDBHook, sizeof(ClearDBHook),
               UdmVarListFindStr(&Indexer->Conf->Vars, "SQLClearDBHook", ""));
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  if (ClearDBHook[0] && (UDM_OK != (rc= UdmSQLQuery(db, NULL, ClearDBHook))))
    return rc;
  
  if(!where[0])
  {
    return UdmTruncateDB(Indexer, db);
  }
  else
  {
    UDM_URLID_LIST urllist;
    UDM_DSTR qbuf;
    UdmDSTRInit(&qbuf, 4096);
    
    bzero((void*) &urllist, sizeof(urllist));
    UdmDSTRAppendf(&qbuf,"SELECT url.rec_id, url.url FROM url%s WHERE url.rec_id<>%s0%s AND %s", 
                   db->from, qu, qu,  where);
    
    if (UDM_OK != (rc= UdmLoadSlowLimitWithSort(db, &urllist, qbuf.data)))
      goto fin;
    
    rc= db->DBSQL_IN ? UdmClearDBUsingIN(Indexer, db, &urllist) :
                       UdmClearDBUsingLoop(Indexer, db, &urllist);

fin:
    UdmFree(urllist.urls);
    UdmDSTRFree(&qbuf);
  }
  return rc;
}


/******************** Hrefs ****************************/

#define MAXHSIZE	1023*4	/* TUNE */

int
UdmStoreHrefsSQL(UDM_AGENT *Indexer)
{
  size_t i;
  int rc= UDM_OK, tnx_started= 0;
  UDM_DOCUMENT  Doc;
  UDM_HREFLIST *Hrefs;
  UDM_DB *db= Indexer->Conf->dbl.nitems == 1 ?
              &Indexer->Conf->dbl.db[0] : (UDM_DB*) NULL;

  if (db && !(db->flags & UDM_SQL_HAVE_GOOD_COMMIT))
    db= NULL;
  
  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
 
  UdmDocInit(&Doc);

  Hrefs= &Indexer->Conf->Hrefs;
  
  for (i= 0; i < Hrefs->nhrefs; i++)
  {  
    UDM_HREF *H = &Hrefs->Href[i];
    if (!H->stored)
    {
      if (!tnx_started && db && UDM_OK != UdmSQLBegin(db))
        goto ret;
      tnx_started= 1;
      UdmVarListAddLst(&Doc.Sections, &H->Vars, NULL, "*");
      UdmVarListReplaceInt(&Doc.Sections, "Referrer-ID", H->referrer);
      UdmVarListReplaceUnsigned(&Doc.Sections,"Hops", H->hops);
      UdmVarListReplaceStr(&Doc.Sections,"URL",H->url?H->url:"");
      UdmVarListReplaceInt(&Doc.Sections, "URL_ID", UdmStrHash32(H->url ? H->url : ""));
      UdmVarListReplaceInt(&Doc.Sections,"Site_id", H->site_id);
      UdmVarListReplaceInt(&Doc.Sections,"Server_id", H->server_id);
      UdmVarListReplaceInt(&Doc.Sections, "HTDB_URL_ID", H->rec_id);
      if (i >= Hrefs->dhrefs)
      {
        if(UDM_OK != (rc= UdmURLActionNoLock(Indexer, &Doc, UDM_URL_ACTION_ADD)))
          goto ret;
      }
      if (H->collect_links)
      {
        if(UDM_OK != (rc= UdmURLActionNoLock(Indexer, &Doc, UDM_URL_ACTION_ADD_LINK)))
          goto ret;
      }
      
      UdmVarListFree(&Doc.Sections);
      H->stored= 1;
    }
  }

  if (tnx_started && db && UDM_OK != UdmSQLCommit(db))
    goto ret;

ret:

  UdmDocFree(&Doc);
  
  /*
    Remember last stored URL num
    Note that it will became 0
    after next sort in AddUrl   
  */
  Hrefs->dhrefs = Hrefs->nhrefs;
  
  /*
    We should not free URL list with onw database
    to avoid double indexing of the same document
    So, do it if compiled with SQL only          
  */
  
  /* FIXME: this is incorrect with both SQL and built-in compiled */
  if(Hrefs->nhrefs > MAXHSIZE)
    UdmHrefListFree(&Indexer->Conf->Hrefs);

  return rc;
}


/********************* Categories ************************************/
static int
UdmCatList(UDM_AGENT * Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i, rows;
  char    qbuf[1024];
  UDM_SQLRES  SQLres;
  int    rc;
  
  if(db->DBType==UDM_DB_SAPDB)
  {
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }else{
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
  }
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
  
  if( (rows = UdmSQLNumRows(&SQLres)) )
  {
    size_t nbytes;
    
    nbytes = sizeof(UDM_CATITEM) * (rows + Cat->ncategories);
    Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,nbytes);
    for(i=0;i<rows;i++)
    {
      UDM_CATITEM *r = &Cat->Category[Cat->ncategories];
      r[i].rec_id=atoi(UdmSQLValue(&SQLres,i,0));
      strcpy(r[i].path,UdmSQLValue(&SQLres,i,1));
      strcpy(r[i].link,UdmSQLValue(&SQLres,i,2));
      strcpy(r[i].name,UdmSQLValue(&SQLres,i,3));
    }
    Cat->ncategories+=rows;
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}


static int
UdmCatPath(UDM_AGENT *Indexer,UDM_CATEGORY *Cat,UDM_DB *db)
{
  size_t    i,l;
  char    qbuf[1024];
  int    rc;
  char            *head = NULL;
  
  l=(strlen(Cat->addr)/2)+1;
  Cat->Category=(UDM_CATITEM*)UdmRealloc(Cat->Category,sizeof(UDM_CATITEM)*(l+Cat->ncategories));
  head = (char *)UdmMalloc(2 * l + 1);

  if (head != NULL)
  {
    UDM_CATITEM  *r = &Cat->Category[Cat->ncategories];

    for(i = 0; i < l; i++)
    {
      UDM_SQLRES  SQLres;

      strncpy(head,Cat->addr,i*2);head[i*2]=0;

      if(db->DBType==UDM_DB_SAPDB)
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path='%s'",head);
      }
      else
      {
        udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
      }
    
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
    
      if(UdmSQLNumRows(&SQLres))
      {
        r[i].rec_id=atoi(UdmSQLValue(&SQLres,0,0));
        strcpy(r[i].path,UdmSQLValue(&SQLres,0,1));
        strcpy(r[i].link,UdmSQLValue(&SQLres,0,2));
        strcpy(r[i].name,UdmSQLValue(&SQLres,0,3));
        Cat->ncategories++;
      }
      UdmSQLFree(&SQLres);
    }
    UDM_FREE(head);
  }
  return UDM_OK;
}


unsigned int
UdmGetCategoryIdSQL(UDM_ENV *Conf, char *category, UDM_DB *db)
{
  UDM_SQLRES Res;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, 128, "SELECT rec_id FROM categories WHERE path LIKE '%s'", category);
  if(UDM_OK != (rc = UdmSQLQuery(db, &Res, qbuf))) return rc;
  if ( UdmSQLNumRows(&Res) > 0)
  {
    sscanf(UdmSQLValue(&Res, 0, 0), "%u", &rc);
  }
  UdmSQLFree(&Res);
  return rc;
}


/*******************  GroupBySite  *********************************/


static size_t
test_key(const void *ptr)
{
  size_t key= (size_t) ((const UDM_URLDATA*)ptr)->site_id;
  return key;
}


static int
test_join(UDM_URLDATA *dst, UDM_URLDATA *src)
{
  dst->per_site+= src->per_site;
  if (dst->score > src->score)
  {
    return UDM_OK;
  }
  else if (dst->score == src->score)
  {
    if (dst->pop_rank > src->pop_rank)
    {
      return UDM_OK;
    }
    else if (dst->pop_rank == src->pop_rank)
    {
      if (dst->url_id < src->url_id)
        return UDM_OK;
    }
  }
  dst->url_id=        src->url_id;
  dst->score=         src->score;
  dst->last_mod_time= src->last_mod_time;
  dst->pop_rank=      src->pop_rank;
  dst->url=           src->url;
  dst->section=       src->section;
  return UDM_OK;
}


static int
UdmURLDataListGroupBySiteUsingSort(UDM_AGENT *A,
                                   UDM_URLDATALIST *R,
                                   UDM_DB *db)
{
  unsigned long ticks;
  UDM_URLDATA *Dat, *End;

  /* Initialize per_site */
  for (Dat= R->Item, End= R->Item + R->nitems;
       Dat < End; Dat++)
    Dat->per_site= 1;

  UdmLog(A,UDM_LOG_DEBUG,"Start sort by site_id %d words", R->nitems);
  ticks=UdmStartTimer();
  UdmURLDataSortBySite(R);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop sort by site_id:\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start group by site_id %d docs", R->nitems);
  ticks=UdmStartTimer();
  UdmURLDataGroupBySite(R);
  ticks=UdmStartTimer() - ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by site_id:\t%.2f", (float)ticks/1000);

  if (UDM_OK != UdmUserSiteScoreListLoadAndApplyToURLDataList(A, R, db))
    return UDM_ERROR;

  return UDM_OK;
}


static int
UdmURLDataListGroupBySiteUsingHash(UDM_AGENT *A,
                                  UDM_URLDATALIST *DataList,
                                  const char *rec_id_str, size_t rec_id_len,
                                  const char *site_id_str, size_t site_id_len)
{
  UDM_HASH Hash;
  UDM_URLDATA *HashData, *Item= DataList->Item, D;
  size_t hcoords= UdmHashSize(DataList->nitems);
  size_t nbytes= hcoords * sizeof(UDM_URLDATA);
  size_t j, i, nbad, ncoords= DataList->nitems;
  size_t rec_id_count= rec_id_len / 4;
  unsigned long ticks1;
  bzero(&D, sizeof(D));
  D.per_site= 1;
  HashData= (UDM_URLDATA*) UdmMalloc(nbytes);
  bzero((void*)HashData, nbytes);

  UdmHashInit(&Hash, HashData, hcoords, sizeof(UDM_URLDATA),
              test_key, (udm_hash_join) test_join);

  for (nbad= 0, j = 0, i = 0; i < rec_id_count && j < ncoords; )
  {
    D.url_id= udm_get_int4(rec_id_str + i * 4);

    /*fprintf(stderr, "%d/%d j=%d\n", D.url_id, Item[j].url_id, j);*/
 
    if (D.url_id == Item[j].url_id)
    {
      D.site_id= udm_get_int4(site_id_str + i * 4);
      D.score= Item[j].score;
      UdmHashPut(&Hash, &D);
      j++;
      i++;
    }
    else if (D.url_id > Item[j].url_id)
    {
      nbad++;
      if (nbad < 4)
      {
        UdmLog(A, UDM_LOG_DEBUG, "URL_ID=%d found in word index but not in '#rec_id' record", D.url_id);
      }
      
      /*
        url_id found in results but missing in "#rec_id"
        This is possible when one runs "indexer -Erewriteurl"
        after deleting some urls;
      */
      j++;
    }
    else
      i++;
  }


  if (j < ncoords)
  {
    UdmLog(A, UDM_LOG_ERROR, "'#rec_id' ended unexpectedly: no data for rec_id=%d", Item[j].url_id);
    nbad+= ncoords - j;
  }

  if (nbad >= 4)
    UdmLog(A, UDM_LOG_DEBUG, "GroupBySite may have worked incorrectly. Total URL_IDs not found in '#rec_id': %d", nbad);

  ticks1= UdmStartTimer();
  j= DataList->nitems= UdmURLDataCompact(DataList->Item, HashData, hcoords);
  ticks1= UdmStartTimer() - ticks1;
  UdmLog(A, UDM_LOG_DEBUG, "HashCompact %d to %d done: %.2f",
         hcoords, DataList->nitems, (float) ticks1 / 1000);
  UdmFree(HashData);
  return UDM_OK;
}


/******************* Search *************************/
static int
cmp_score_urlid(UDM_URL_SCORE *s1, UDM_URL_SCORE *s2)
{
  if (s1->url_id > s2->url_id) return(1);
  if (s1->url_id < s2->url_id) return(-1);
  return 0;
}
      

static void
UdmScoreListToURLData(UDM_URLDATA *D, UDM_URL_SCORE *C, size_t num)
{
  for ( ; num > 0; num--, D++, C++)
  {
    D->url_id= C->url_id;
    D->score= C->score;
  }
}


#ifdef HAVE_DEBUG
static void
UdmURLScoreListPrint(UDM_URLSCORELIST *List)
{
  size_t i;
  for (i= 0; i < List->nitems; i++)
  {
    UDM_URL_SCORE *Item= &List->Item[i];
    fprintf(stderr, "%d:%d\n", Item->url_id, Item->score);
  }
}
#endif

static int 
UdmSortAndGroupByURL(UDM_AGENT *A,
                     UDM_RESULT *Res,
                     UDM_SECTIONLIST *SectionList,
                     UDM_DB *db, size_t num_best_rows)
{
  UDM_URLSCORELIST ScoreList;
  UDM_URLDATALIST DataList;
  unsigned long  ticks=UdmStartTimer();
  const char *pattern= UdmVarListFindStr(&A->Conf->Vars, "s", "R");
  size_t nbytes;
  int flags= 0, rc= UDM_OK;
  const char *p;
  int RelevancyFactor= UdmVarListFindInt(&A->Conf->Vars, "RelevancyFactor", 255);
  int DateFactor= UdmVarListFindInt(&A->Conf->Vars, "DateFactor", 0);
  const char *su= UdmVarListFindStr(&A->Conf->Vars, "su", NULL);
  int group_by_site= UdmVarListFindBool(&A->Conf->Vars, "GroupBySite", 0) 
                     && UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0 ?
                     UDM_URLDATA_SITE : 0;
  size_t BdictThreshold= (size_t) UdmVarListFindInt(&A->Conf->Vars,
                                                    "URLDataThreshold", 0);
  size_t MaxResults= (size_t) UdmVarListFindInt(&db->Vars, "MaxResults", 0);
  int use_qcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  urlid_t debug_url_id= UdmVarListFindInt(&A->Conf->Vars, "DebugURLId", 0);

  flags|= group_by_site ? UDM_URLDATA_SITE : 0;
  flags|= DateFactor ? UDM_URLDATA_LM : 0;
  flags|= use_qcache ? UDM_URLDATA_CACHE : 0;
  
  for (p = pattern; *p; p++)
  {
    if (*p == 'U' || *p == 'u') flags|= UDM_URLDATA_URL;
    if (*p == 'P' || *p == 'p') flags|= UDM_URLDATA_POP;
    if (*p == 'D' || *p == 'd') flags|= UDM_URLDATA_LM;
    if (*p == 'S' || *p == 's') flags|= (su && su[0]) ? UDM_URLDATA_SU : 0;
  }
  
  ticks=UdmStartTimer();
  bzero((void*) &ScoreList, sizeof(ScoreList));

  UdmLog(A,UDM_LOG_DEBUG,"Start group by url_id %d sections", SectionList->nsections);
  UdmGroupByURL2(A, db, Res, SectionList, &ScoreList);

  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop group by url_id, %d docs found:\t%.2f",
         ScoreList.nitems, (float)ticks/1000);

#ifdef HAVE_DEBUG
  if (UdmVarListFindBool(&A->Conf->Vars, "DebugGroupByURL", 0))
  {
    UdmURLScoreListPrint(&ScoreList);
  }
#endif

  UdmApplyCachedQueryLimit(A, &ScoreList, db);
  if (ScoreList.nitems == 0)
    return UDM_OK;
  
  if (UDM_OK != (rc=  UdmUserScoreListLoadAndApplyToURLScoreList(A, &ScoreList, db)))
    return rc;

  UdmLog(A,UDM_LOG_DEBUG,"Start load url data %d docs", ScoreList.nitems);
  ticks=UdmStartTimer();

  nbytes= UdmHashSize(ScoreList.nitems) * sizeof(UDM_URLDATA);
  DataList.Item = (UDM_URLDATA*)UdmMalloc(nbytes);
  bzero((void*) DataList.Item, nbytes);
  DataList.nitems= ScoreList.nitems;

  /* Use full sort in case if DebugURLId is specified */
  if (debug_url_id)
    num_best_rows= ScoreList.nitems;
  
  if (num_best_rows > ScoreList.nitems)
    num_best_rows= ScoreList.nitems;
  
  if (num_best_rows < 256 && !flags)
  {
    udm_timer_t ticks1;
    
    Res->total_found= ScoreList.nitems;
    UdmLog(A,UDM_LOG_DEBUG,"Start SORT by score %d docs", ScoreList.nitems);
    ticks1=UdmStartTimer();
    if (ScoreList.nitems > 1000)
    {
      /*
      size_t nitems= ScoreList.nitems;
      UdmURLScoreListSortByScore(&ScoreList);
      ScoreList.nitems= 1000;
      UdmURLScoreListSortByScoreThenURL(&ScoreList);
      ScoreList.nitems= nitems;
      */
      UdmURLScoreListSortByScoreThenURLTop(&ScoreList, 1000);
    }
    else
    {
      UdmURLScoreListSortByScoreThenURL(&ScoreList);
    }
    UdmSort((void*) ScoreList.Item, num_best_rows,
            sizeof(UDM_URL_SCORE), (udm_qsort_cmp) cmp_score_urlid);
    UdmScoreListToURLData(DataList.Item, ScoreList.Item, num_best_rows);
    ticks1=UdmStartTimer()-ticks1;
    UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by score:\t%.2f",(float)ticks1/1000);
    /*
    Don't really need this:
    - sorting is done only by score
    - other parameters will be loaded in UdmResAddDocInfo()
    {
      UdmLog(A,UDM_LOG_DEBUG,"Start load url data quick %d docs", num_best_rows);
      ticks1=UdmStartTimer();
      rc= UdmLoadURLDataFromURL(A, &DataList, db, num_best_rows, flags);
      ticks1=UdmStartTimer()-ticks1;
      UdmLog(A,UDM_LOG_DEBUG,"Stop load url data quick:\t\t%.2f",(float)ticks1/1000);
    }
    */
    DataList.nitems= num_best_rows;
    goto date_factor;
  }

  UdmScoreListToURLData(DataList.Item, ScoreList.Item, DataList.nitems);
  
  if (flags & UDM_URLDATA_SU)
  {
    size_t norder;
    udm_timer_t ticks1= UdmStartTimer();
    UdmLog(A, UDM_LOG_DEBUG, "Trying to load fast section order '%s'", su);
    rc= UdmFastOrderLoadAndApplyToURLDataList(A, db, &DataList, su, &norder);
    UdmLog(A, UDM_LOG_DEBUG, "Loading fast order '%s' done, %d docs found, %.2f sec",
           su, norder, UdmStopTimer(&ticks1));
    if (norder)
      flags^= UDM_URLDATA_SU;
  }
  
  if (db->DBMode == UDM_DBMODE_BLOB &&
      !(flags & UDM_URLDATA_URL)    &&
      !(flags & UDM_URLDATA_SU)     &&
        BdictThreshold < ScoreList.nitems)
  {
    rc= UdmLoadURLDataFromBdict(A, &DataList, db, DataList.nitems, flags);
  }
  else
  {
    rc= UdmLoadURLDataFromURL(A, &DataList, db, DataList.nitems, flags);
  }

  Res->total_found= DataList.nitems;


date_factor:
  
  if (rc != UDM_OK)
    return rc;

  /* TODO: check whether limit by site works fine */
  if (!RelevancyFactor || DateFactor)
    UdmURLDataListApplyRelevancyFactors(A, &DataList, RelevancyFactor, DateFactor);

  ticks= UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop load url data:\t\t%.2f",(float)ticks/1000);
  
  UdmLog(A,UDM_LOG_DEBUG,"Start SORT by PATTERN %d docs", DataList.nitems);
  ticks=UdmStartTimer();
  UdmURLDataSortByPattern(&DataList, pattern);
  ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Stop SORT by PATTERN:\t%.2f",(float)ticks/1000);

  Res->URLData= DataList;
  UdmFree(ScoreList.Item);

  if (debug_url_id)
  {
    size_t i;
    for (i= 0; i < DataList.nitems; i++)
    {
      if (DataList.Item[i].url_id == debug_url_id)
      {
        char tmp[256];
        const char *s= UdmVarListFindStr(&A->Conf->Vars, "DebugScore", "");
        udm_snprintf(tmp, sizeof(tmp), "%s rank=%d", s, i + 1);
        UdmVarListReplaceStr(&A->Conf->Vars, "DebugScore", tmp);
        break;
      }
    }
  }
  
  if (MaxResults && MaxResults < Res->total_found)
  {
    UdmLog(A, UDM_LOG_DEBUG, "Applying MaxResults=%d, total_found=%d\n",
           (int) MaxResults, (int) Res->total_found);
    Res->total_found= MaxResults;
    if (Res->URLData.nitems > MaxResults)
      Res->URLData.nitems= MaxResults;
  }
  return UDM_OK;
}


static int /* WHERE limit */
LoadURL(UDM_DB *db, const char *where, UDM_URLID_LIST *buf)
{
  int rc;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  urlid_t *tmp;
  size_t i;

  if (!*where)
    return UDM_OK;

  /* TODO: reuse LoadSlowLimitWithSort() here */
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT url.rec_id FROM url%s WHERE %s", db->from, where);
  if  (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;

  if (!(nrows= UdmSQLNumRows(&SQLRes)))
  {
    buf->empty= 1;
    UdmSQLFree(&SQLRes);
    return(UDM_OK);
  }

  tmp= UdmMalloc(sizeof(urlid_t) * nrows);
  buf->urls= UdmMalloc(sizeof(urlid_t) * nrows);
  if (!tmp || !buf->urls)
  {
    UDM_FREE(buf->urls);
    UDM_FREE(tmp);
    goto ex;
  }
  
  for (i= 0; i < nrows; i++)
  {
    tmp[i]= (urlid_t) UDM_ATOI(UdmSQLValue(&SQLRes, i, 0));
  }
  UdmSort(tmp, nrows, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  
  /* Remove duplicates */
  for (i= 0; i < nrows; )
  {
    while (++i < nrows && tmp[i] == tmp[i - 1]);
    buf->urls[buf->nurls++] = tmp[i - 1];
  }
  UDM_FREE(tmp);
  if ((tmp= UdmRealloc(buf->urls, sizeof(urlid_t) * buf->nurls)))
    buf->urls = tmp;

ex:
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


int
UdmBuildCmpArgSQL(UDM_DB *db, int match, const char *word,
                  char *cmparg, size_t maxlen)
{
  char escwrd[1000];
  UdmSQLEscStr(db, escwrd, word, strlen(word)); /* Search word */
  switch(match)
  {  
    case UDM_MATCH_BEGIN:
      udm_snprintf(cmparg, maxlen, " LIKE '%s%%'", escwrd);
      break;
    case UDM_MATCH_END:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s'", escwrd);
      break;
    case UDM_MATCH_SUBSTR:
      udm_snprintf(cmparg, maxlen, " LIKE '%%%s%%'", escwrd);
      break;
    case UDM_MATCH_NUMERIC_LT:
      udm_snprintf(cmparg, maxlen, " < %d", atoi(escwrd));
      break;
    case UDM_MATCH_NUMERIC_GT:
      udm_snprintf(cmparg, maxlen, " > %d", atoi(escwrd));
      break;
    case UDM_MATCH_FULL:
    default:
      udm_snprintf(cmparg, maxlen, "='%s'", escwrd);
      break;
  }
  return(UDM_OK);
}


static int
UdmFindOneWordSQL (UDM_FINDWORD_ARGS *args)
{
  char cmparg[256];
  UdmBuildCmpArgSQL(args->db,
                    args->Word.match, args->Word.word,
                    cmparg, sizeof(cmparg));
  args->cmparg= cmparg;
  switch (args->db->DBMode) {
    case UDM_DBMODE_SINGLE: return(UdmFindWordSingle(args));
    case UDM_DBMODE_MULTI:  return(UdmFindWordMulti(args));
    case UDM_DBMODE_BLOB:
     return args->live_updates ? UdmFindWordBlobLiveUpdates(args) :
                                 UdmFindWordBlob(args);
    case UDM_DBMODE_RAWBLOB:return(UdmFindWordRawBlob(args, UDM_RAWBLOB_SEARCH));
  }
  return(UDM_ERROR);
}


static int
UdmSectionListListToURLCRDList(UDM_URLCRDLIST *List,
                               UDM_SECTIONLISTLIST *SrcList)
{
  size_t list, ncoords= 0, nsections= 0;
  UDM_URL_CRD *Crd;
  for (list=0; list < SrcList->nitems; list++)
  {
    /*UdmSectionListPrint(&SrcList->Item[list]);*/
    ncoords+= SrcList->Item[list].ncoords;
    nsections+= SrcList->Item[list].nsections;
  }
  bzero((void*) List, sizeof(*List));
  if (!(List->Coords= (UDM_URL_CRD*) UdmMalloc(ncoords * sizeof(UDM_URL_CRD))))
    return UDM_ERROR;
  for (Crd= List->Coords, list=0; list < SrcList->nitems; list++)
  {
    size_t sec;
    UDM_SECTIONLIST *SectionList= &SrcList->Item[list];
    for (sec=0; sec < SectionList->nsections; sec++)
    {
      UDM_SECTION *Section= &SectionList->Section[sec];
      size_t coord;
      for (coord= 0; coord < Section->ncoords; coord++)
      {
        UDM_COORD2 *Coord2= &Section->Coord[coord];
        Crd->url_id= Section->url_id;
        Crd->seclen= Section->seclen;
        Crd->pos= Coord2->pos;
        Crd->num= Section->wordnum;
        Crd->secno= Section->secno;
        Crd++;
      }
    }
  }
  UDM_ASSERT(ncoords == Crd - List->Coords);
  List->ncoords= ncoords;
  /*UdmCoordListPrint(List);*/

  return UDM_OK;
}


static int
UdmMultiWordMergeCoords(UDM_URLCRDLIST *Phrase, 
                        UDM_SECTIONLISTLIST *SrcList,
                        size_t orig_wordnum,
                        size_t nparts)
{
  UdmSectionListListToURLCRDList(Phrase, SrcList);
  UdmURLCRDListSortByURLThenSecnoThenPos(Phrase);
  UdmURLCRDListMergeMultiWord(Phrase, orig_wordnum, nparts);
  /*UdmCoordListPrint(Phrase);*/
  return UDM_OK;
}


static int
UdmFindMultiWordSQL (UDM_FINDWORD_ARGS *args)
{
  char *lt, *tmp_word, *tok;
  int rc= UDM_OK;
  UDM_SECTIONLISTLIST OriginalSectionListList;
  size_t orig_wordnum;
  size_t nparts= 0;
  const char *w;
  char *orig_word= args->Word.word;
  unsigned long ticks= UdmStartTimer();
  char delim[]= " \r\t_-./";

  /* Check if the word really multi-part */
  for (w= args->Word.word; ; w++)
  {
    if (!*w)
      return UdmFindOneWordSQL(args); /* No delimiters found */
    
    if (strchr(delim, *w)) /* Delimiter found */
      break;
  }
  
  if (!(tmp_word= UdmStrdup(args->Word.word)))
    return(UDM_ERROR);

  UdmLog(args->Agent, UDM_LOG_DEBUG,
         "Start searching for multiword '%s'", args->Word.word);
  OriginalSectionListList= args->SectionListList;
  UdmSectionListListInit(&args->SectionListList);
  orig_wordnum= args->Word.order;
  
  for (tok= udm_strtok_r(tmp_word, delim, &lt) ; tok ;
       tok= udm_strtok_r(NULL, delim, &lt))
  {
    unsigned long ticks1= UdmStartTimer();
    args->Word.word= tok;
    UdmLog(args->Agent, UDM_LOG_DEBUG,
           "Searching for subword '%s'", args->Word.word);
    rc= UdmFindOneWordSQL(args);
    ticks1= UdmStartTimer() -  ticks1;
    UdmLog(args->Agent, UDM_LOG_DEBUG,
           "Stop searching for subword '%s' %d coords found: %.2f",
           args->Word.word, args->Word.count, (float) ticks1 / 1000);
    /* If the next word wasn't found - no need to search for others. */
    if (rc != UDM_OK || !args->Word.count)
      goto ret;
    nparts++;
    args->Word.order++;
  }
  
  /* All parts returned results. Check phrase */
  {
    UDM_URLCRDLIST Phrase;
    UDM_SECTIONLIST SectionList;
    bzero((void*) &Phrase, sizeof(Phrase));
    UdmMultiWordMergeCoords(&Phrase, &args->SectionListList, orig_wordnum, nparts);
    if (args->urls.nurls)
      UdmApplyFastLimit(&Phrase, &args->urls);
    if (Phrase.ncoords)
    {
      UdmURLCRDListToSectionList(args, &SectionList, &Phrase);
      UdmSectionListListAdd(&OriginalSectionListList, &SectionList);
    }
    UDM_FREE(Phrase.Coords);
    args->Word.count= Phrase.ncoords;
  }
  
ret:
  UdmFree(tmp_word);
  UdmSectionListListFree(&args->SectionListList);
  args->SectionListList= OriginalSectionListList;
  args->Word.word= orig_word;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG,
         "Stop searching for multiword '%s'", args->Word.word);
  return rc;
}


static int
UdmFindAlwaysFoundWordSQL(UDM_FINDWORD_ARGS *args)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[1024 * 4];
  size_t nrows;
  size_t i;
  UDM_URLCRDLIST CoordList;

  bzero((void*) &CoordList, sizeof(CoordList));

  if (*args->where)
     udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url%s WHERE %s",
                  args->db->from, args->where);
  else
  {
    if (args->urls.nurls)
    {
      /*
        A fast limit is loaded.
        No needs to do "SELECT FROM url".
        Populate CoordList from the fast limit instead.
      */
      for (i= 0; i < args->urls.nurls; i++)
      {
        if (UDM_OK != (rc= UdmAddOneCoord(&CoordList, args->urls.urls[i],
                                          0x00010100,
                                          (args->Word.order & 0xFF))))
          return UDM_ERROR;
      }
      UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);
      return UDM_OK;
    }
    udm_snprintf(qbuf, sizeof(qbuf), "SELECT url.rec_id FROM url");
  }

  if ((rc= UdmSQLQuery(args->db, &SQLRes, qbuf)) != UDM_OK)
    return(rc);
  /* Note that rc is implicitly set to UDM_OK at this point. */
  if (! (nrows= UdmSQLNumRows(&SQLRes)))
    goto err;

  for (i = 0; i < nrows; i++)
    if (UDM_OK != (rc= UdmAddOneCoord(&CoordList,
                              (urlid_t)UDM_ATOI(UdmSQLValue(&SQLRes, i, 0)),
                              0x00010100,
                              (args->Word.order & 0xFF))))
      break;

  if (args->urls.nurls)
    UdmApplyFastLimit(&CoordList, &args->urls);
  if (CoordList.ncoords)
    UdmURLCRDListListAddWithSort2(args, &args->CoordListList, &CoordList);

err:
  UdmSQLFree(&SQLRes);
  return(rc);
}


static int
UdmCheckIndex(UDM_AGENT *query, UDM_DB *db)
{
  int tm, rc;
  if (UDM_OK != (rc= UdmBlobReadTimestamp(query, db, &tm, 0)))
    return rc;
  if (tm)
    return UDM_OK;
#ifdef WIN32
  sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'Create fast index'.");
#else
  sprintf(query->Conf->errstr, "Inverted word index not found. Probably you forgot to run 'indexer -Eblob'.");
#endif
  return UDM_ERROR;
}


static int
UdmMergeWords(UDM_FINDWORD_ARGS *args, UDM_SECTIONLIST *SectionList)
{
  unsigned long ticks= UdmStartTimer();
  UDM_AGENT *A= args->Agent;
  size_t nresults;

  UdmLog(A, UDM_LOG_DEBUG, "Start merging %d lists", args->SectionListList.nitems);
  UdmSectionListListMergeSorted(&args->SectionListList, SectionList, 1);
  ticks= UdmStartTimer() - ticks;
  UdmLog(A, UDM_LOG_DEBUG, "Merged %d lists %d sections: %.2f",
         args->SectionListList.nitems, SectionList->nsections, (float) ticks / 1000);
  nresults= SectionList->nsections;

  if (!nresults && args->db->DBMode == UDM_DBMODE_BLOB)
    return UdmCheckIndex(A, args->db);
  return UDM_OK;
}


static void
UdmSearchParamInit(UDM_FINDWORD_ARGS *args,
                   UDM_AGENT *A,
                   UDM_RESULT *Res,
                   UDM_DB *db)
{

  UDM_LOCK_CHECK_OWNER(A, UDM_LOCK_CONF);

  bzero((void*) args, sizeof(*args));
  UdmWideWordListInit(&args->CollationMatches);
  UdmURLCRDListListInit(&args->CoordListList);

  args->Agent= A;
  args->db= db;
  args->WWList= &Res->WWList;
  args->Word.match= UdmMatchMode(UdmVarListFindStr(&A->Conf->Vars, "wm", "wrd"));
  args->where= UdmSQLBuildWhereCondition(A->Conf, db);
  args->save_section_size= UdmVarListFindBool(&A->Conf->Vars, "SaveSectionSize", 1);
  args->live_updates= UdmVarListFindBool(&db->Vars, "LiveUpdates", 0);
}


static void
UdmSearchParamFree(UDM_FINDWORD_ARGS *args)
{
  UDM_FREE(args->urls.urls);
  UDM_FREE(args->live_update_active_urls.urls);
  UDM_FREE(args->live_update_deleted_urls.urls);
  UdmWideWordListFree(&args->CollationMatches);
  UdmURLCRDListListFree(&args->CoordListList);
  UdmSectionListListFree(&args->SectionListList);
}


int
UdmFindWordsSQL(UDM_AGENT *query, UDM_RESULT *Res, UDM_DB *db,
                size_t num_best_rows)
{
  size_t    wordnum;
  char   wf[256];
  const char  *always_found_word, *fl;
  int    use_crosswords, use_qcache;
  unsigned long   ticks=UdmStartTimer();
  int    rc= UDM_OK;
  UDM_FINDWORD_ARGS args;
  UDM_URLID_LIST fl_urls;
  UDM_SECTIONLIST SectionList;


  bzero((void*) &fl_urls, sizeof(fl_urls));
  bzero((void*) &SectionList, sizeof(SectionList));

  UDM_GETLOCK(query, UDM_LOCK_CONF);
  {
    UdmSearchParamInit(&args, query, Res, db);

    always_found_word= UdmVarListFindStr(&query->Conf->Vars, "AlwaysFoundWord", NULL);
    fl= UdmVarListFindStr(&query->Conf->Vars, "fl", UdmVarListFindStr(&db->Vars, "fl", ""));

    UdmWeightFactorsInit2(wf, &query->Conf->Vars, &db->Vars, "wf");
    args.wf= wf;

    use_crosswords= UdmVarListFindBool(&query->Conf->Vars, "CrossWords", 0);
    use_qcache= UdmVarListFindBool(&db->Vars, "qcache", 0);
  }
  UDM_RELEASELOCK(query, UDM_LOCK_CONF);

  if ((db->DBMode == UDM_DBMODE_BLOB && args.where) || fl[0])
  {
    UdmLog(query,UDM_LOG_DEBUG, "Start loading limits");
    ticks= UdmStartTimer();
    if (*args.where)
    {
      LoadURL(db, args.where, &args.urls);
      UdmLog(query, UDM_LOG_DEBUG, "WHERE limit loaded. %d URLs found", args.urls.nurls);
    }
    if (!args.urls.empty && fl[0])
    {
      char name[64];
      const char *q;
      if ((fl_urls.exclude= (fl[0] == '-')))
        fl++;
      udm_snprintf(name, sizeof(name), "Limit.%s", fl);
      if (UDM_OK != (rc= ((q= UdmVarListFindStr(&query->Conf->Vars, name, NULL)) ?
                         UdmLoadSlowLimitWithSort(db, &fl_urls, q) :
                         UdmBlobLoadFastURLLimit(db, fl, &fl_urls))))
        goto ret;
      UdmLog(query,UDM_LOG_DEBUG, "Limit '%s' loaded%s%s %d URLs",
             fl, fl_urls.exclude ? " type=excluding" : "",
             q ? " source=slow":"", fl_urls.nurls);

      UdmURLIdListMerge(&args.urls, &fl_urls);
    }
    ticks= UdmStartTimer()-ticks;
    UdmLog(query,UDM_LOG_DEBUG,"Stop loading limits, %d URLs found: %.2f",
           args.urls.nurls, (float)ticks/1000);
    if (args.urls.empty)
      goto ret;
  }

  if (db->DBMode == UDM_DBMODE_BLOB && args.live_updates)
  {
    udm_timer_t ticks1= UdmStartTimer();
    UdmLog(query, UDM_LOG_DEBUG, "Start loading LiveUpdate url_id list");
    if (UDM_OK != (rc= UdmLoadSlowLimitWithSort(db, &args.live_update_deleted_urls,
                                        "SELECT url_id FROM bdicti WHERE state=0")))
      goto ret;
    UdmLog(query, UDM_LOG_DEBUG,
           "Stop loading LiveUpdate url_id list: %.02f, %d updated docs found",
           UdmStopTimer(&ticks1),
           args.live_update_deleted_urls.nurls);
    args.live_update_deleted_urls.exclude= 1;
    UdmURLIdListCopy(&args.live_update_active_urls, &args.urls);
    UdmURLIdListMerge(&args.urls, &args.live_update_deleted_urls);
  }


  /* Now find each word */
  for(wordnum=0; wordnum < Res->WWList.nwords; wordnum++)
  {
    UDM_WIDEWORD  *W=&Res->WWList.Word[wordnum];

    if (W->origin == UDM_WORD_ORIGIN_STOP) continue;

    args.Word.order= wordnum;
    args.Word.count= 0;
    args.Word.word= W->word;
    args.Word.match= W->match;
    args.Word.secno= W->secno;

    ticks=UdmStartTimer();
    UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",Res->WWList.Word[wordnum].word);

    /*
       For now SYNONYMs only are treated as a possible multi-word
       origin. Probably it will be changed in future, so we will
       use this feature for phrase search.
     */
    if (always_found_word && !strcmp(W->word, always_found_word))
      rc= UdmFindAlwaysFoundWordSQL(&args);
    else if (W->origin == UDM_WORD_ORIGIN_SYNONYM || W->phrwidth > 0)
      rc= UdmFindMultiWordSQL(&args);
    else
      rc= UdmFindOneWordSQL(&args);

    if (rc != UDM_OK)
      goto ret;

    /*
      If CollationMatches is not empty, then we should skip
      updating word statistics here - it will be updated in
      the loop after UdmSortAndGroupByURL().
     */
    if (!args.CollationMatches.nwords)
      Res->WWList.Word[wordnum].count+= args.Word.count;

    ticks= UdmStartTimer() - ticks;
    UdmLog(query, UDM_LOG_DEBUG, "Stop search for '%s'\t%.2f found %u coords",
           Res->WWList.Word[wordnum].word, (float)ticks / 1000, args.Word.count);

    if (use_crosswords)
    {
      UdmLog(query,UDM_LOG_DEBUG,"Start search for crossword '%s'", W->word);
      args.Word.count= 0;
      if (UDM_OK != (rc= UdmFindCrossWord(&args)))
        goto ret;
      Res->WWList.Word[wordnum].count+= args.Word.count;
      
      ticks= UdmStartTimer()-ticks;
      UdmLog(query, UDM_LOG_DEBUG,
            "Stop search for crossword '%s'\t%.2f %d found",
             args.Word.word, (float)ticks/1000, args.Word.count);
    }
  }

  if (UDM_OK != (rc= UdmMergeWords(&args, &SectionList)))
    goto ret;
  
  if (UDM_OK != (rc= UdmSortAndGroupByURL(query, Res,
                                          &SectionList, db, num_best_rows)))
    goto ret;

  /* 
     We cannot add collation matches before
     UdmSortAndGroupByURL - to use optimized groupping
     functions when WWList->nwords==1
  */
  if (args.CollationMatches.nwords)
  {
    size_t i;
    for (i= 0; i < args.CollationMatches.nwords; i++)
    {
      UdmWideWordListAdd(&Res->WWList, &args.CollationMatches.Word[i]);
    }
  }


ret:
  UDM_FREE(fl_urls.urls);
  UdmSectionListFree(&SectionList);
  UdmSearchParamFree(&args);
  return rc;
}


/****************** Track ***********************************/

int UdmTrackSQL(UDM_AGENT * query, UDM_RESULT *Res, UDM_DB *db)
{
  char    *qbuf;
  char    *text_escaped;
  const char  *words=UdmVarListFindStr(&query->Conf->Vars,"q",""); /* "q-lc" was here */
  const char      *IP = UdmVarListFindStr(&query->Conf->Vars, "IP", "");
  size_t          i, escaped_len, qbuf_len;
  int             res, qtime, rec_id;
  const char      *qu = (db->DBType == UDM_DB_PGSQL) ? "'" : "";
  const char      *value= (db->DBType == UDM_DB_IBASE ||
                           db->DBType == UDM_DB_MIMER ||
                           db->DBType == UDM_DB_DB2   ||
                           db->DBType == UDM_DB_ORACLE8) ? "sval" : "value";
  
  if (*words == '\0') return UDM_OK; /* do not store empty queries */

  escaped_len = 4 * strlen(words);
  qbuf_len = escaped_len + 4096;

  if ((qbuf = (char*)UdmMalloc(qbuf_len)) == NULL) return UDM_ERROR;
  if ((text_escaped = (char*)UdmMalloc(escaped_len)) == NULL)
  { 
    UDM_FREE(qbuf);
    return UDM_ERROR;
  }
  
  /* Escape text to track it  */
  UdmSQLEscStr(db, text_escaped, words, strlen(words)); /* query for tracking */
  
  if (db->DBType == UDM_DB_IBASE ||
      db->DBType == UDM_DB_MIMER ||
      db->DBType == UDM_DB_ORACLE8)
  {
    const char *next;
    switch (db->DBType)
    {
      case UDM_DB_IBASE: next= "SELECT GEN_ID(qtrack_GEN,1) FROM rdb$database"; break;
      case UDM_DB_MIMER: next= "SELECT NEXT_VALUE OF qtrack_GEN FROM system.onerow"; break;
      case UDM_DB_ORACLE8: next= "SELECT qtrack_seq.nextval FROM dual"; break;
    }
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, next)))
      goto UdmTrack_exit;
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (rec_id,ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "(%d,'%s','%s',%d,%d,%d)",
                 rec_id, IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
    if (UDM_OK != (res = UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
  }
  else
  {
    udm_snprintf(qbuf, qbuf_len - 1,
                 "INSERT INTO qtrack (ip,qwords,qtime,wtime,found) "
                 "VALUES "
                 "('%s','%s',%d,%d,%d)",
                 IP, text_escaped, qtime= (int)time(NULL),
                 Res->work_time, Res->total_found);
  
    if (UDM_OK != (res= UdmSQLQuery(db, NULL, qbuf)))
      goto UdmTrack_exit;
    
    if (db->DBType == UDM_DB_MYSQL)
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT last_insert_id()");
    else
      udm_snprintf(qbuf, qbuf_len - 1, "SELECT rec_id FROM qtrack WHERE ip='%s' AND qtime=%d", IP, qtime);
    if (UDM_OK != (res= UdmSQLQueryOneRowInt(db, &rec_id, qbuf)))
      goto UdmTrack_exit;
  }
  
  for (i = 0; i < query->Conf->Vars.nvars; i++)
  {
    UDM_VAR *Var = &query->Conf->Vars.Var[i];
    if (strncasecmp(Var->name, "query.",6)==0 && strcasecmp(Var->name, "query.q") && strcasecmp(Var->name, "query.BrowserCharset")
       && strcasecmp(Var->name, "query.IP")  && Var->val != NULL && *Var->val != '\0') 
    {
      udm_snprintf(qbuf, qbuf_len,
                   "INSERT INTO qinfo (q_id,name,%s) "
                   "VALUES "
                   "(%s%i%s,'%s','%s')", 
      value, qu, rec_id, qu, &Var->name[6], Var->val);
      res= UdmSQLQuery(db, NULL, qbuf);
      if (res != UDM_OK) goto UdmTrack_exit;
    }
  }
UdmTrack_exit:
  UDM_FREE(text_escaped);
  UDM_FREE(qbuf);
  return res;
}


/********************* Adding URLInfo to Res *********************/

static int UpdateShows(UDM_DB *db, urlid_t url_id)
{
  char qbuf[64];
  udm_snprintf(qbuf, sizeof(qbuf), "UPDATE url SET shows = shows + 1 WHERE rec_id = %s%i%s",
               (db->DBType == UDM_DB_PGSQL) ? "'" : "",
               url_id,
               (db->DBType == UDM_DB_PGSQL) ? "'" : "");
  return UdmSQLQuery(db, NULL, qbuf);
}

static void SQLResToSection(UDM_SQLRES *R, UDM_VARLIST *S, size_t row)
{
  const char *sname=UdmSQLValue(R,row,1);
  const char *sval=UdmSQLValue(R,row,2);
  UdmVarListAddStr(S, sname, sval);
}


static size_t
UdmDBNum(UDM_RESULT *Res, size_t n)
{
  UDM_URLDATA *Data= &Res->URLData.Item[n + Res->first];
  return UDM_COORD2DBNUM(Data->score);
}


static int
UdmResAddURLInfoUsingIN(UDM_RESULT *Res, UDM_DB *db, size_t dbnum,
                        const char *qbuf)
{
  int rc;
  UDM_SQLRES SQLres;
  size_t j, sqlrows;
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;
    
  for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
  {
    if (UdmDBNum(Res, j) == dbnum)
    {
      size_t i;
      UDM_DOCUMENT *D=&Res->Doc[j];
      urlid_t      url_id = UdmVarListFindInt(&D->Sections, "ID", 0);
      for(i = 0; i < sqlrows; i++)
      {
        if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
          SQLResToSection(&SQLres, &D->Sections, i);
      }
    }
  }
  UdmSQLFree(&SQLres);
  return UDM_OK;
}


static int
UdmDocAddURLInfo(UDM_DOCUMENT *D, UDM_DB *db, const char *qbuf)
{
  UDM_SQLRES SQLres;
  int rc;
  size_t row;
  
  if(UDM_OK != (rc= UdmSQLQuery(db,&SQLres,qbuf)))
    return rc;

  for(row= 0; row < UdmSQLNumRows(&SQLres); row++)
    SQLResToSection(&SQLres, &D->Sections, row);
  UdmSQLFree(&SQLres);
  return rc;
}


int UdmResAddDocInfoSQL(UDM_AGENT *query, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  size_t      i;
  char        instr[1024*4]="";
  char        qbuf[1024*4];
  UDM_SQLRES  SQLres;
  int         rc;
  int         use_showcnt = !strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "PopRankUseShowCnt", "no"), "yes");
  int         use_category = UdmVarListFindStr(&query->Conf->Vars, "cat", NULL) ? 1 : 0;
  double      pr, ratio = 0.0;
  const       char *hi_priority= db->DBType == UDM_DB_MYSQL ? "HIGH_PRIORITY" : "";
  int         use_urlinfo= UdmVarListFindBool(&query->Conf->Vars, "LoadURLInfo", 1);
  int         use_urlbasicinfo= UdmVarListFindBool(&query->Conf->Vars, "LoadURLBasicInfo", 1);
  int         use_taginfo= UdmVarListFindBool(&query->Conf->Vars, "LoadTagInfo", 0);
  if(!Res->num_rows)return UDM_OK;
  if (use_showcnt) ratio = UdmVarListFindDouble(&query->Conf->Vars, "PopRankShowCntRatio", 25.0);
  
  if (use_showcnt)
    UdmLog(query, UDM_LOG_DEBUG, "use_showcnt: %d  ratio: %f", use_showcnt, ratio);
  
  for (i= 0; i < Res->num_rows; i++)
  {
    UdmVarListReplaceInt(&Res->Doc[i].Sections, "id",
                         Res->URLData.Item[i + Res->first].url_id);
  }
  
  if(db->DBSQL_IN)
  {
    size_t  j, sqlrows;
    
    /* Compose IN string and set to zero url_id field */
    for(i=0; i < Res->num_rows; i++)
    {
      if (UdmDBNum(Res, i) == dbnum)
      {
        const char *comma= instr[0] ? "," : "";
        const char *squot= db->DBType == UDM_DB_PGSQL ? "'" : "";
        sprintf(UDM_STREND(instr),"%s%s%i%s", comma, squot,
                UdmVarListFindInt(&Res->Doc[i].Sections, "ID", 0), squot);
      }
    }
    
    if (!instr[0])
      return UDM_OK;
    
    if (use_urlbasicinfo)
    {
      udm_snprintf(qbuf,sizeof(qbuf),
                  "SELECT %s"
                  " rec_id,url,last_mod_time,docsize,"
                  " next_index_time,referrer,crc32,site_id,pop_rank"
                  " FROM url WHERE rec_id IN (%s)", hi_priority,instr);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
    
      for(sqlrows= UdmSQLNumRows(&SQLres), j=0; j<Res->num_rows; j++)
      {
        if (UdmDBNum(Res, j) == dbnum)
        {
          UDM_DOCUMENT *D= &Res->Doc[j];
          urlid_t      url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
          for(i = 0; i < sqlrows; i++)
          {
            if(url_id == UDM_ATOI(UdmSQLValue(&SQLres,i,0)))
            {
              SQLResToDoc(query->Conf, D, &SQLres, i);
              if (use_showcnt &&
                  (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
                  UpdateShows(db, url_id);
              break;
            }
          }
        }
      } 
      UdmSQLFree(&SQLres);
    }
    
    if (use_category)
    {
      udm_snprintf(qbuf, sizeof(qbuf),"SELECT u.rec_id,'Category' as sname,c.path FROM url u,server s,categories c WHERE u.rec_id IN (%s) AND u.server_id=s.rec_id AND s.category=c.rec_id", instr); 
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
    
    if (use_taginfo)
    {
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT u.rec_id, 'tag', tag FROM url u, server s "
                   "WHERE  u.rec_id in (%s) AND u.server_id=s.rec_id",
                   instr); 
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
    
    if (use_urlinfo)
    {
      udm_snprintf(qbuf,sizeof(qbuf),"SELECT url_id,sname,sval FROM urlinfo WHERE url_id IN (%s)",instr);
      if (UDM_OK != (rc= UdmResAddURLInfoUsingIN(Res, db, dbnum, qbuf)))
        return rc;
    }
  }
  else
  {
    for(i=0;i<Res->num_rows;i++)
    {
      UDM_DOCUMENT  *D=&Res->Doc[i];
      urlid_t  url_id= UdmVarListFindInt(&D->Sections, "ID", 0);
      
      if (UdmDBNum(Res, i) != dbnum)
        continue;
      
      sprintf(qbuf,"SELECT rec_id,url,last_mod_time,docsize,next_index_time,referrer,crc32,site_id,pop_rank FROM url WHERE rec_id=%i", url_id);
      if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
        return rc;
      
      if(UdmSQLNumRows(&SQLres))
      {
        SQLResToDoc(query->Conf, D, &SQLres, 0);
          if (use_showcnt &&
              (pr= atof(UdmVarListFindStr(&D->Sections, "Score", "0.0"))) >= ratio)
              UpdateShows(db, url_id);
      }
      UdmSQLFree(&SQLres);
      
      if (use_category)
      {
        sprintf(qbuf,"SELECT u.rec_id,c.path FROM url u,server s,categories c WHERE rec_id=%i AND u.server_id=s.rec_id AND s.category=c.rec_id", url_id);
        if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
          return rc;
        if(UdmSQLNumRows(&SQLres))
        {
          UdmVarListReplaceStr(&D->Sections, "Category", UdmSQLValue(&SQLres, i, 1));
        }
        UdmSQLFree(&SQLres);
      }
      
      if (use_taginfo)
      {
        udm_snprintf(qbuf, sizeof(qbuf),
                     "SELECT u.rec_id, 'tag', tag FROM url u, server s "
                     "WHERE  u.rec_id=%d AND u.server_id=s.rec_id", url_id); 
        if(UDM_OK != (rc= UdmDocAddURLInfo(D, db, qbuf)))
          return rc;
      }
      
      if (use_urlinfo)
      {
        sprintf(qbuf,"SELECT url_id,sname,sval FROM urlinfo WHERE url_id=%i", url_id);
        if(UDM_OK != (rc= UdmDocAddURLInfo(D, db, qbuf)))
          return rc;
      }
    }
  }
  return(UDM_OK);
}



/******* "indexer -Ewordstat" - word statistics for suggestions *************/

static int
UdmWordStatQuery(UDM_AGENT *A, UDM_DB *db, const char *src)
{
  int rc;
  UDM_SQLRES SQLRes;
  size_t row, rows;
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, src)))
    return rc;
  
  rows=UdmSQLNumRows(&SQLRes);
  for(row=0 ; row < rows ; row++)
  {
    const char *word;
    int count;
    size_t wordlen;
    char snd[32];
    char insert[128];
    word= UdmSQLValue(&SQLRes, row, 0);
    wordlen= UdmSQLLen(&SQLRes, row, 0);
    count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
    UdmSoundex(A->Conf->lcs, snd, word, wordlen);
    if (snd[0])
    {
      sprintf(insert,
              "INSERT INTO wrdstat (word, snd, cnt) VALUES ('%s','%s',%d)",
              word, snd, count);
      if (UDM_OK!= (rc= UdmSQLQuery(db, NULL, insert)))
        return rc;
    }
  }
  UdmSQLFree(&SQLRes);
  return UDM_OK;
}


static int
UdmWordStatCreateMulti(UDM_AGENT *A, UDM_DB *db)
{
  int rc;
  size_t i;
  
  for (i=0; i <= 0xFF; i++)
  {
    char qbuf[128];
    UdmLog(A, UDM_LOG_EXTRA, "Processing table %02X", i);

    sprintf(qbuf, "SELECT word, count(*) FROM dict%02X GROUP BY word", i);
    if (UDM_OK != (rc= UdmWordStatQuery(A, db, qbuf)))
      return rc;
  }
  return UDM_OK;
}


static int
UdmWordStatCreateSingle(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  sprintf(qbuf, "SELECT word, count(*) FROM dict GROUP BY word");
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreateBlob(UDM_AGENT *A, UDM_DB *db)
{
  char qbuf[128];
  const char *funcname;
  switch(db->DBType)
  {
    case UDM_DB_ORACLE8:
      funcname= "lengthb";
      break;
    case UDM_DB_SQLITE3:
      funcname= "length";
      break;
    default:
      funcname= "octet_length";
  }
  sprintf(qbuf, "SELECT word, sum(%s(intag)) FROM bdict GROUP BY word", funcname);
  return UdmWordStatQuery(A, db, qbuf);
}


static int
UdmWordStatCreate(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  int rc;

  UdmLog(A, UDM_LOG_ERROR, "Calculating word statistics");

  rc= UdmSQLTableTruncateOrDelete(db, "wrdstat");
  if (rc != UDM_OK)
    return rc;
  
  if (db->DBMode == UDM_DBMODE_SINGLE)
  {
    rc= UdmWordStatCreateSingle(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_MULTI)
  {
    rc= UdmWordStatCreateMulti(A, db);
  }
  else if (db->DBMode == UDM_DBMODE_BLOB)
  {
    rc= UdmWordStatCreateBlob(A, db);
  }
  
  UdmLog(A, UDM_LOG_ERROR, "Word statistics done");
  return rc;
}


/********************  Suggest ************************************/

static float
UdmUCharArrayCorrelation(unsigned char *a, unsigned char *b, size_t size)
{
  float Asum= 0;
  float Bsum= 0;
  float ABsum= 0;
  size_t i;

  for (i= 0; i < size; i++)
  {
    Asum+=  (float) a[i] * (float) a[i];
    Bsum+=  (float) b[i] * (float) b[i];
    ABsum+= (float) a[i] * (float) b[i];
  }
  return ABsum / sqrt(Asum) / sqrt(Bsum);
}


static void
UdmUCharArrayFillByteStatistics(unsigned char *a, size_t size,
                                const unsigned char *src, size_t srclen)
{
  bzero((void*) a, size);
  for ( ; srclen ; src++, srclen--)
  {
    a[((unsigned char) *src) % size]++;
  }
}


static int
WordProximity(UDM_CHARSET *cs,
              const char *s1, size_t len1,
              const char *s2, size_t len2, int scale)
{
  size_t max= len1 > len2 ? len1 : len2;
  size_t min= len1 < len2 ? len1 : len2;
  if ((max - min)*3 > max)  /* Not more than 1/3 of length difference */
    return 0;
  {
    unsigned char a[256];
    unsigned char b[256];
    UdmUCharArrayFillByteStatistics(a, sizeof(a), (unsigned const char *) s1, len1);
    UdmUCharArrayFillByteStatistics(b, sizeof(b), (unsigned const char *) s2, len2);
    return (size_t) (UdmUCharArrayCorrelation(a, b, sizeof(a)) * scale);
  }
  return (1000 * min) / (max ? max : 1);
}


static int
UdmResSuggest(UDM_AGENT *A, UDM_DB *db, UDM_RESULT *Res, size_t dbnum)
{
  int rc= UDM_OK;
  size_t i, nwords;
  UDM_WIDEWORDLIST *WWList= &Res->WWList;
  UDM_CONV lcs_uni;
  
  UdmLog(A, UDM_LOG_DEBUG, "Generating suggestion list");
  UdmConvInit(&lcs_uni, A->Conf->lcs, &udm_charset_sys_int, UDM_RECODE_HTML);

  for (i=0, nwords= WWList->nwords; i < nwords; i++)
  {
    UDM_WIDEWORD *W= &WWList->Word[i];
    if (W->origin == UDM_WORD_ORIGIN_QUERY && !W->count)
    {
      char snd[32];
      char qbuf[128];
      char topstr[64], rowstr[64], limstr[64];
      UDM_SQLRES SQLRes;
      UDM_WIDEWORD sg;
      size_t row, rows;
      size_t Wlen= W->len; /* W is not valid after UdmWideWordListAdd() */
      size_t Word= W->order;
      size_t phrpos= W->phrpos;
      size_t count_sum;
      const char *Wword= W->word;
      
      UdmSQLTopClause(db, 100,
                      topstr, sizeof(topstr),
                      rowstr, sizeof(rowstr),
                      limstr, sizeof(limstr));
      
      UdmSoundex(A->Conf->lcs, snd, W->word, W->len);
      UdmLog(A, UDM_LOG_DEBUG, "Suggest for '%s': '%s'", W->word, snd);
      udm_snprintf(qbuf, sizeof(qbuf),
                   "SELECT %sword, cnt FROM wrdstat WHERE snd='%s'%s"
                   " ORDER by cnt DESC%s",
                    topstr, snd, rowstr, limstr);
      
      if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
        return rc;
      
      rows=UdmSQLNumRows(&SQLRes);
      UdmLog(A, UDM_LOG_DEBUG, "%d suggestions found", rows);
      if (rows > 0)
      {
        UdmLog(A, UDM_LOG_DEBUG, "<%s>: <%s>/<%s>/<%s>/<%s>", 
               "word", "count", "count_weight", "proximity_weight",
               "final_weight");
      }
      bzero((void*)&sg, sizeof(sg));
      
      for (count_sum= 0, row= 0; row < rows; row++)
      {
        count_sum+= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
      }
      if (!count_sum)
        count_sum= 1;

      for(row=0 ; row < rows ; row++)
      {
        size_t nbytes, proximity_weight, count_weight, final_weight;
        int tmp[128];
        int word_count_factor= 63;
        int word_proximity_factor= 256 - word_count_factor;
        
        sg.word= (char*) UdmSQLValue(&SQLRes, row, 0);
        sg.count= UDM_ATOI(UdmSQLValue(&SQLRes, row, 1));
        sg.len= UdmSQLLen(&SQLRes, row, 0);

        count_weight= (word_count_factor * sg.count) / count_sum;
        proximity_weight= WordProximity(A->Conf->lcs,
                                        Wword, Wlen,
                                        sg.word, sg.len,
                                        word_proximity_factor);
        final_weight= (count_weight + proximity_weight);
        UdmLog(A, UDM_LOG_DEBUG, "%s: %d/%d/%d/%d", 
               sg.word, sg.count, count_weight, proximity_weight, final_weight);
        sg.count= final_weight;

        nbytes= (sg.len + 1) * sizeof(int);
        if (nbytes < sizeof(tmp))
        {
          sg.order= Word;
          sg.phrpos= phrpos;
          sg.origin= UDM_WORD_ORIGIN_SUGGEST;
          UdmWideWordListAdd(WWList, &sg); /* W is not valid after this */
        }
      }
      UdmSQLFree(&SQLRes);
    }
  }
  return rc;
}


/************************* Misc *******************************************/


int
UdmResActionSQL(UDM_AGENT *Agent, UDM_RESULT *Res, int cmd, UDM_DB *db, size_t dbnum)
{
  switch(cmd)
  {
    case UDM_RES_ACTION_DOCINFO:
      return UdmResAddDocInfoSQL(Agent, db, Res, dbnum);
    case UDM_RES_ACTION_SUGGEST:
      return UdmResSuggest(Agent, db, Res, dbnum);
    default:
      UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Res Action SQL");
      return UDM_ERROR;
  }
}


int
UdmCatActionSQL(UDM_AGENT *Agent, UDM_CATEGORY *Cat, int cmd,UDM_DB *db)
{
  switch(cmd)
  {
    case UDM_CAT_ACTION_LIST:
      return UdmCatList(Agent,Cat,db);
    case UDM_CAT_ACTION_PATH:
      return UdmCatPath(Agent,Cat,db);
    default:
              UdmLog(Agent, UDM_LOG_ERROR, "Unsupported Cat Action SQL");
      return UDM_ERROR;
  }
}


int
UdmCheckUrlidSQL(UDM_AGENT *Agent, UDM_DB *db, urlid_t id)
{
  UDM_SQLRES SQLRes;
  char qbuf[128];
  unsigned int rc = 0;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT rec_id FROM url WHERE rec_id=%d", id);
  rc = UdmSQLQuery(db, &SQLRes, qbuf);
  if(UDM_OK != rc)
  {
    rc = 1;
  }
  else
  {
    if (UdmSQLNumRows(&SQLRes) != 0) rc = 1;
    else rc = 0;
  }
  UdmSQLFree(&SQLRes);
  return rc;
}


int
UdmExportSQL(UDM_AGENT *Indexer, UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  UDM_PSTR row[24];

  printf("<database>\n");
  printf("<urlList>\n");
  rc= UdmSQLExecDirect(db, &SQLRes, "SELECT * FROM url");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<url "
      "rec_id=\"%s\" "
      "status=\"%s\" "
      "docsize=\"%s\" "
      "next_index_time=\"%s\" "
      "last_mod_time=\"%s\" "
      "referrer=\"%s\" "
      "hops=\"%s\" "
      "crc32=\"%s\" "
      "seed=\"%s\" "
      "bad_since_time=\"%s\" "
      "site_id=\"%s\" "
      "server_id=\"%s\" "
      "shows=\"%s\" "
      "pop_rank=\"%s\" "
      "url=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val, row[3].val,
      row[4].val, row[5].val, row[6].val, row[7].val,
      row[8].val, row[9].val, row[10].val, row[11].val,
      row[12].val, row[13].val, row[14].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</urlList>\n");

  printf("<linkList>\n");
  rc= UdmSQLExecDirect(db, &SQLRes, "SELECT * FROM links");
  if (rc != UDM_OK) return(rc);
  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    printf(
      "<link "
      "ot=\"%s\" "
      "k=\"%s\" "
      "weight=\"%s\" "
      "/>\n",
      row[0].val, row[1].val, row[2].val);
  }
  UdmSQLFree(&SQLRes);
  printf("</linkList>\n");

  printf("</database>\n");
  return(0);
}


static int
UdmDocPerSite(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  char qbuf[1024];
  const char *s, *hostinfo= UdmVarListFindStr(&D->Sections, "Hostinfo", NULL);
  int rc, num, prevnum= UdmVarListFindInt(&D->Sections, "DocPerSite", 0);
  UDM_SQLRES SQLRes;
  
  if (!hostinfo)
    return UDM_OK;
  
  for (s= hostinfo; s[0]; s++)
  {
    /*
      Host name good characters: digits, letters, hyphen (-).
      Just check the worst characters.
    */
    if (*s == '\'' || *s == '\"')
    {
      num= 1000000;
      goto ret;
    }
  }
  udm_snprintf(qbuf, sizeof(qbuf),
               "SELECT COUNT(*) FROM url WHERE url LIKE '%s%%'", hostinfo);
  
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
    return rc;
  num= prevnum + atoi(UdmSQLValue(&SQLRes, 0, 0));
  UdmSQLFree(&SQLRes);
ret:
  UdmVarListReplaceInt(&D->Sections, "DocPerSite", num);
  return UDM_OK;
}

 
static int
UdmImportSection(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)
{
  UDM_TEXTITEM Item;
  UDM_VAR *Sec;
  UDM_VARLIST Vars;
  UDM_SQLRES SQLRes;
  UDM_DSTR d;
  int rc;
  size_t row, rows, cols;
  const char *fmt= UdmVarListFindStr(&D->Sections, "SQLImportSection", NULL);
  
  if (!fmt)
    return UDM_OK;
  
  UdmDSTRInit(&d, 1024);
  UdmVarListInit(&Vars);
  UdmVarListSQLEscape(&Vars, &D->Sections, db);
  UdmDSTRParse(&d, fmt, &Vars);
  UdmVarListFree(&Vars);
  if(UDM_OK!= (rc= UdmSQLQuery(db, &SQLRes, d.data)))
    return rc;

  cols= UdmSQLNumCols(&SQLRes);
  bzero((void*)&Item, sizeof(Item));
  for (row=0, rows= UdmSQLNumRows(&SQLRes); row < rows; row++)
  {
    size_t col;
    for (col= 0; col + 1 < cols; col+= 2)
    {
      Item.section_name= (char*) UdmSQLValue(&SQLRes, row, col);
      if ((Sec= UdmVarListFind(&D->Sections, Item.section_name)))
      {
        Item.str= (char*) UdmSQLValue(&SQLRes, row, col + 1);
        Item.section= Sec->section;
        UdmTextListAdd(&D->TextList, &Item);
      }
    }
  }
  
  UdmDSTRFree(&d);
  UdmSQLFree(&SQLRes);
  return rc;
}


static int
UdmGetReferers(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t    i,j;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  const char  *where;
  int    rc;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = UdmSQLBuildWhereCondition(Indexer->Conf, db);
  
  udm_snprintf(qbuf,sizeof(qbuf),"SELECT url.status,url2.url,url.url FROM url,url url2%s WHERE url.referrer=url2.rec_id %s %s",
    db->from, where[0] ? "AND" : "", where);
  
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  j=UdmSQLNumRows(&SQLres);
  for(i=0;i<j;i++)
  {
    if(Indexer->Conf->RefInfo)Indexer->Conf->RefInfo(
      atoi(UdmSQLValue(&SQLres,i,0)),
      UdmSQLValue(&SQLres,i,2),
      UdmSQLValue(&SQLres,i,1)
    );
  }
  UdmSQLFree(&SQLres);
  return rc;
}


static int
UdmGetDocCount(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  char    qbuf[200]="";
  UDM_SQLRES  SQLres;
  int    rc;
  
  sprintf(qbuf,NDOCS_QUERY);
  if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
  
  if(UdmSQLNumRows(&SQLres))
  {
    const char * s;
    s=UdmSQLValue(&SQLres,0,0);
    if(s)Indexer->doccount += atoi(s);
  }
  UdmSQLFree(&SQLres);
  return(UDM_OK);
}


int
UdmStatActionSQL(UDM_AGENT *Indexer,UDM_STATLIST *Stats,UDM_DB *db)
{
  size_t    i,j,n;
  char    qbuf[2048];
  UDM_SQLRES  SQLres;
  int    have_group= (db->flags & UDM_SQL_HAVE_GROUPBY);
  const char  *where;
  int    rc=UDM_OK;
  
  if(db->DBType==UDM_DB_IBASE)
    have_group=0;

  UDM_LOCK_CHECK_OWNER(Indexer, UDM_LOCK_CONF);
  where = UdmSQLBuildWhereCondition(Indexer->Conf, db);

  if(have_group)
  {
    char func[128];
    
    switch(db->DBType)
    {
      case UDM_DB_MYSQL:
        udm_snprintf(func,sizeof(func)-1, "next_index_time<=%d", Stats->time);
        break;
    
      case UDM_DB_PGSQL:
      case UDM_DB_MSSQL:
      case UDM_DB_SYBASE:
      case UDM_DB_DB2:
      case UDM_DB_SQLITE:
      case UDM_DB_SQLITE3:
      default:
        udm_snprintf(func,sizeof(func)-1, "case when next_index_time<=%d then 1 else 0 end", Stats->time);
        break;

      case UDM_DB_ACCESS:
        udm_snprintf(func,sizeof(func)-1, "IIF(next_index_time<=%d, 1, 0)", Stats->time);
        break;

      case UDM_DB_ORACLE8:
      case UDM_DB_SAPDB:
        udm_snprintf(func,sizeof(func)-1, "DECODE(SIGN(%d-next_index_time),-1,0,1,1)", Stats->time);
        break;
    }

    udm_snprintf(qbuf, sizeof(qbuf) - 1,
                 "SELECT status, SUM(%s), count(*) FROM url%s %s%s GROUP BY status ORDER BY status",
                 func, db->from, where[0] ? "WHERE " : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    if((n = UdmSQLNumRows(&SQLres)))
    {
      for(i = 0; i < n; i++)
      {
        for(j=0;j<Stats->nstats;j++)
        {
          if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
          {
            Stats->Stat[j].expired += atoi(UdmSQLValue(&SQLres,i,1));
            Stats->Stat[j].total += atoi(UdmSQLValue(&SQLres,i,2));
            break;
          }
        }
        if(j==Stats->nstats)
        {
          UDM_STAT  *S;
        
          Stats->Stat=(UDM_STAT*)UdmRealloc(Stats->Stat,(Stats->nstats+1)*sizeof(Stats->Stat[0]));
          S=&Stats->Stat[Stats->nstats];
          S->status=atoi(UdmSQLValue(&SQLres,i,0));
          S->expired=atoi(UdmSQLValue(&SQLres,i,1));
          S->total=atoi(UdmSQLValue(&SQLres,i,2));
          Stats->nstats++;
        }
      }
    }
    UdmSQLFree(&SQLres);
    
  }else{
/*  
  FIXME: learn how to get it from SOLID and IBASE
  (HAVE_IBASE || HAVE_SOLID || HAVE_VIRT )
*/
    
    udm_snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,next_index_time FROM url%s %s%s ORDER BY status",
      db->from, where[0] ? "WHERE " : "", where);

    if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
    
    for(i=0;i<UdmSQLNumRows(&SQLres);i++)
    {
      for(j=0;j<Stats->nstats;j++)
      {
        if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0)))
        {
          if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
            Stats->Stat[j].expired++;
          Stats->Stat[j].total++;
          break;
        }
      }
      if(j==Stats->nstats)
      {
        Stats->Stat=(UDM_STAT *)UdmRealloc(Stats->Stat,sizeof(UDM_STAT)*(Stats->nstats+1));
        Stats->Stat[j].status = UDM_ATOI(UdmSQLValue(&SQLres,i,0));
        Stats->Stat[j].expired=0;
        if((time_t)UDM_ATOU(UdmSQLValue(&SQLres,i,1)) <= Stats->time)
          Stats->Stat[j].expired++;
        Stats->Stat[j].total=1;
        Stats->nstats++;
      }
    }
    UdmSQLFree(&SQLres);
  }
  return rc;
}

/******************* URL handlers **************************/

int (*udm_url_action_handlers[])(UDM_AGENT *A, UDM_DOCUMENT *D, UDM_DB *db)=
{
  UdmDeleteURL,           /* UDM_URL_ACTION_DELETE */
  UdmAddURL,              /* UDM_URL_ACTION_ADD */
  UdmUpdateUrl,           /* UDM_URL_ACTION_SUPDATE */
  UdmLongUpdateURL,       /* UDM_URL_ACTION_LUPDATE */
  UdmDeleteWordsAndLinks, /* UDM_URL_ACTION_DUPDATE */
  UdmUpdateClone,         /* UDM_URL_ACTION_UPDCLONE */
  UdmRegisterChild,       /* UDM_URL_ACTION_REGCHILD */
  UdmFindURL,             /* UDM_URL_ACTION_FINDBYURL */
  UdmFindMessage,         /* UDM_URL_ACTION_FINDBYMSG */
  UdmFindOrigin,          /* UDM_URL_ACTION_FINDORIG */
  UdmMarkForReindex,      /* UDM_URL_ACTION_EXPIRE */
  UdmGetReferers,         /* UDM_URL_ACTION_REFERERS */
  UdmGetDocCount,         /* UDM_URL_ACTION_DOCCOUNT */
  UdmDeleteLinks,         /* UDM_URL_ACTION_LINKS_DELETE */
  UdmAddLink,             /* UDM_URL_ACTION_ADD_LINK */
  UdmGetCachedCopy,       /* UDM_URL_ACTION_GET_CACHED_COPY */
  UdmWordStatCreate,      /* UDM_URL_ACTION_WRDSTAT */
  UdmDocPerSite,          /* UDM_URL_ACTION_DOCPERSITE */
  UdmImportSection        /* UDM_URL_ACTION_SQLIMPORTSEC */
};



#endif /* HAVE_SQL */

