#ifndef SDLTOOLS_H
#define SDLTOOLS_H

#include <SDL.h>

#include "SDLException.h"
#include "SDLCalls.h"


//----------------------------------------------------------------------------
/**
 * This namespace contains some common needed functions
 * related to SDL datastructures.
 */
namespace SDL_TOOLS
{
    //------------------------------------------------------------------------
    /**
     * @return The pixel x/y of the given surface.
     */
    Uint32 getPixel(const SDL_Surface *surface, Uint16 x, Uint16 y);

    //------------------------------------------------------------------------
    /**
     * Returns, whether the given point (x/y) lies inside the given rectangle.
     *
     * @param x The x coordinate of the point.
     * @param y The y coordinate of the point.
     * @param r The rectangle to check.
     *
     * @return true, if the point (x/y) lies inside r, else false.
     */
    bool inside(const Sint16 x, const Sint16 y, const SDL_Rect &r);

    //------------------------------------------------------------------------
    /**
     * Calculate the intersection of two rectangles.
     *
     * @param r1 A const reference to the first rectangle.
     * @param r2 A const reference to the second rectangle.
     * @param clip A reference to a rectangle,
     *             where the result will be stored.
     *
     * @return true, if there is an intersection, else false.
     */
    bool intersect(const SDL_Rect &r1, const SDL_Rect &r2, SDL_Rect &clip);

    //------------------------------------------------------------------------
    /**
     * Calculates the union of two rectangles.
     *
     * @param r1 A const reference to the first rectangle.
     * @param r2 A const reference to the second rectangle.
     * @param u A reference to a rectangle,
     *          where the result will be stored.
     */
    void unite(const SDL_Rect &r1, const SDL_Rect &r2, SDL_Rect &u);


    //------------------------------------------------------------------------
    /**
     * @param src The source surface to rotate.
     * @param angle The angle (90, 180 or 270).
     *
     * @return A pointer to the new rotated surface.
     *
     * @throw SDLException if an SDL error occured.
     */
    SDL_Surface *rotate(const SDL_Surface *src, Uint16 angle)
        throw (SDLException);

    //------------------------------------------------------------------------
    /**
     * @param s1 The first surface.
     * @param p1 The position of the first surface.
     * @param s2 The second surface.
     * @param p2 The position of the second surface.
     *
     * @return true, if there is a collision between s1 and s2, else false.
     */
    bool isCollision(const SDL_Surface *s1, const SDL_Rect &p1,
                     const SDL_Surface *s2, const SDL_Rect &p2);

    //------------------------------------------------------------------------
    /**
     * Returns the angle between the mid-points
     * of the two given surface boxes relative to the y axis,
     * counted in clockwise order and inside the value range from [0..360).
     */
    int getAngle(const SDL_Rect &p1, const SDL_Rect &p2);

    //------------------------------------------------------------------------
    /**
     * Calculates the centre of the given SDL_Rect
     * and stores the x and y coordinate into the given integer variables.
     */
    void getCentre(const SDL_Rect &r, Sint16 &x, Sint16 &y);

    //------------------------------------------------------------------------
    /**
     * Calls SDL_CALLS::LoadBMP() and sets the SDL_SRCCOLORKEY
     * to the given rgb value, before returning the surface.
     */
    SDL_Surface *loadBMPWithColorKey(const char *file,
                                     Uint8 r = 0, Uint8 g = 0, Uint8 b = 0)
        throw (SDLException);
}

#endif //SDLTOOLS_H
