#include <cassert>
#include <memory>

#include "Tools.h"
#include "MathTools.h"
#include "SDLException.h"
#include "SDLFrameRate.h"
#include "SDLCalls.h"
#include "SDLMixerCalls.h"
#include "SDLTools.h"


//============================================================================
// Implementation of SDLException.h
//============================================================================

//----------------------------------------------------------------------------
SDLException::SDLException(const char *call)
        : m_call(call),
          m_sdlError(SDL_GetError())
{
}

//----------------------------------------------------------------------------
SDLException::~SDLException()
{
    m_call = NULL;
    m_sdlError = NULL;
}

//----------------------------------------------------------------------------
std::string SDLException::toString() const
{
    std::string s;
    s.append(m_call).append(" failed: ").append(m_sdlError);
    return s;
}



//============================================================================
// Implementation of SDLFrameRate.h
//============================================================================

//----------------------------------------------------------------------------
FPSmanager SDLFrameRate::m_man;

//----------------------------------------------------------------------------
SDLFrameRate::SDLFrameRate()
{
}

//----------------------------------------------------------------------------
SDLFrameRate::~SDLFrameRate()
{
}

//----------------------------------------------------------------------------
void SDLFrameRate::setFrameRate(unsigned rate)
{
    SDL_initFramerate(&m_man);
    (void)SDL_setFramerate(&m_man, rate);
}

//----------------------------------------------------------------------------
unsigned SDLFrameRate::getFrameRate()
{
    return SDL_getFramerate(&m_man);
}

//----------------------------------------------------------------------------
void SDLFrameRate::delay()
{
    SDL_framerateDelay(&m_man);
}



//============================================================================
// Implementation of SDLCalls.h
//============================================================================

//----------------------------------------------------------------------------
void SDL_CALLS::Init(Uint32 flags) throw (SDLException)
{
    int rc = SDL_Init(flags);
    if (rc != 0)
    {
        throw SDLException("SDL_Init()");
    }
}

//----------------------------------------------------------------------------
void SDL_CALLS::BlitSurface(const SDL_Surface *src, const SDL_Rect *srcrect,
                            SDL_Surface *dst, const SDL_Rect *dstrect)
    throw (SDLException)
{
    SDL_Rect dest;
    if (dstrect)
    {
        // Copy the destination rectangle, since it may be modified
        // by SDL_BlitSurface.

        dest = *dstrect;
        dstrect = &dest;
    }

    int rc = SDL_BlitSurface(
        const_cast<SDL_Surface*>(src), const_cast<SDL_Rect*>(srcrect),
        dst, const_cast<SDL_Rect*>(dstrect));

    if (rc != 0)
    {
        throw SDLException("SDL_BlitSurface()");
    }
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_CALLS::ConvertSurface(const SDL_Surface *src,
                                       const SDL_PixelFormat *fmt,
                                       Uint32 flags)
    throw (SDLException)
{
    SDL_Surface *s = SDL_ConvertSurface(
        const_cast<SDL_Surface*>(src),
        const_cast<SDL_PixelFormat*>(fmt),
        flags);
    if (s == NULL)
    {
        throw SDLException("SDL_ConvertSurface()");
    }
    return s;
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_CALLS::CreateRGBSurface(
    Uint32 flags, int width, int height, int depth,
    Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
    throw (SDLException)
{
    SDL_Surface *s = SDL_CreateRGBSurface(flags, width, height, depth,
                                          Rmask, Gmask, Bmask, Amask);
    if (s == NULL)
    {
        throw SDLException("SDL_CreateRGBSurface()");
    }
    return s;
}

//----------------------------------------------------------------------------
void SDL_CALLS::UpdateRect(Sint32 x, Sint32 y, Sint32 w, Sint32 h)
{
    SDL_UpdateRect(SDL_GetVideoSurface(), x, y, w, h);
}

//----------------------------------------------------------------------------
void SDL_CALLS::UpdateRects(int numrects, const SDL_Rect *rects)
{
    SDL_UpdateRects(SDL_GetVideoSurface(),
                    numrects, const_cast<SDL_Rect*>(rects));
}

//----------------------------------------------------------------------------
void SDL_CALLS::Flip() throw (SDLException)
{
    int rc = SDL_Flip(SDL_GetVideoSurface());
    if (rc != 0)
    {
        throw SDLException("SDL_Flip()");
    }
}

//----------------------------------------------------------------------------
void SDL_CALLS::FreeSurface(SDL_Surface *surface)
{
    SDL_FreeSurface(surface);
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_CALLS::LoadBMP(const char *file) throw (SDLException)
{
    SDL_Surface *s = SDL_LoadBMP(file);
    if (s == NULL)
    {
        throw SDLException("SDL_LoadBMP()");
    }
    return s;
}

//----------------------------------------------------------------------------
void SDL_CALLS::SetAlpha(SDL_Surface *surface, Uint32 flag, Uint8 alpha)
    throw (SDLException)
{
    int rc = SDL_SetAlpha(surface, flag, alpha);
    if (rc != 0)
    {
        throw SDLException("SDL_SetAlpha()");
    }
}

//----------------------------------------------------------------------------
void SDL_CALLS::SetColorKey(SDL_Surface *surface, Uint32 flag, Uint32 key)
    throw (SDLException)
{
    int rc = SDL_SetColorKey(surface, flag, key);
    if (rc != 0)
    {
        throw SDLException("SDL_SetColorKey()");
    }
}

//----------------------------------------------------------------------------
void SDL_CALLS::SetPalette(SDL_Surface *surface, int flags,
                           SDL_Color *colors, int firstcolor, int ncolors)
    throw (SDLException)
{
    int rc = SDL_SetPalette(surface, flags, colors, firstcolor, ncolors);
    if (rc != 1)
    {
        throw SDLException("SDL_SetPalette()");
    }
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_CALLS::SetVideoMode(int width, int height,
                                     int bpp, Uint32 flags)
    throw (SDLException)
{
    SDL_Surface *s = SDL_SetVideoMode(width, height, bpp, flags);
    if (s == NULL)
    {
        throw SDLException("SDL_SetVideoMode()");
    }
    return s;
}


//----------------------------------------------------------------------------
void SDL_CALLS::FillRect(SDL_Surface *surface,
                         const SDL_Rect *rect,
                         Uint32 color)
    throw (SDLException)
{
    SDL_Rect dest;
    if (rect)
    {
        // Copy the destination rectangle, since it may be modified
        // by SDL_BlitSurface.

        dest = *rect;
        rect = &dest;
    }

    int rc = SDL_FillRect(surface, const_cast<SDL_Rect*>(rect), color);
    if (rc != 0)
    {
        throw SDLException("SDL_FillRect()");
    }
}


//----------------------------------------------------------------------------
SDL_Surface *SDL_CALLS::rotozoomSurface(const SDL_Surface *src, double angle,
                                        double zoom, int smooth)
    throw (SDLException)
{
    SDL_Surface *s = ::rotozoomSurface(
        const_cast<SDL_Surface*>(src), angle, zoom, smooth);
    if (s == NULL)
    {
        throw SDLException("rotozoomSurface()");
    }
    return s;
}

//----------------------------------------------------------------------------
void SDL_CALLS::PushEvent(const SDL_Event &event)
    throw (SDLException)
{
    int rc = SDL_PushEvent(&const_cast<SDL_Event&>(event));
    if (rc != 0)
    {
        throw SDLException("SDL_PushEvent()");
    }
}



//============================================================================
// Implementation of SDLMixerCalls.h
//============================================================================

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::OpenAudio(int rate, Uint16 format,
                                int channels, int buffers)
    throw (SDLException)
{
    int rc = Mix_OpenAudio(rate, format, channels, buffers);
    if (rc != 0)
    {
        throw SDLException("Mix_OpenAudio()");
    }
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::QuerySpec(int* rate, Uint16* format, int* channels)
    throw (SDLException)
{
    int rc = Mix_QuerySpec(rate, format, channels);
    if (rc == 0)
    {
        throw SDLException("Mix_QuerySpec()");
    }
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::CloseAudio()
{
    Mix_CloseAudio();
}

//----------------------------------------------------------------------------
int SDL_MIXER_CALLS::AllocateChannels(int num)
{
    return Mix_AllocateChannels(num);
}

//----------------------------------------------------------------------------
Mix_Chunk* SDL_MIXER_CALLS::LoadWAV(const char* file) throw (SDLException)
{
    Mix_Chunk *chunk = Mix_LoadWAV(file);
    if (chunk == NULL)
    {
        throw SDLException("Mix_LoadWAV()");
    }

    return chunk;
}

//----------------------------------------------------------------------------
int SDL_MIXER_CALLS::PlayChannel(const Mix_Chunk* chunk, int loops)
    throw (SDLException)
{
    int rc = Mix_PlayChannel(-1, const_cast<Mix_Chunk*>(chunk), loops);
    if (rc == -1)
    {
        throw SDLException("Mix_PlayChannel()");
    }

    return rc;
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::HaltChannel(int channel) throw (SDLException)
{
    (void)Mix_HaltChannel(channel);  // Always returns 0.
}

//----------------------------------------------------------------------------
Mix_Music* SDL_MIXER_CALLS::LoadMUS(const char* track) throw (SDLException)
{
    Mix_Music *m = Mix_LoadMUS(track);
    if (m == NULL)
    {
        throw SDLException("Mix_LoadMUS()");
    }

    return m;
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::PlayMusic(const Mix_Music* music, int loops)
    throw (SDLException)
{
    int rc = Mix_PlayMusic(const_cast<Mix_Music*>(music), loops);
    if (rc == -1)
    {
        throw SDLException("Mix_PlayMusic()");
    }
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::HookMusicFinished(void(*hook)())
{
    Mix_HookMusicFinished(hook);
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::HaltMusic()
{
    (void)Mix_HaltMusic(); // always returns 0
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::FreeChunk(Mix_Chunk* chunk)
{
    Mix_FreeChunk(chunk);
}

//----------------------------------------------------------------------------
void SDL_MIXER_CALLS::FreeMusic(Mix_Music* music)
{
    Mix_FreeMusic(music);
}



//============================================================================
// Implementation of SDLTools.h
//============================================================================

//----------------------------------------------------------------------------
Uint32 SDL_TOOLS::getPixel(const SDL_Surface *surface, Uint16 x, Uint16 y)
{
    int bpp = surface->format->BytesPerPixel;
    assert(bpp == 1 || bpp == 2 || bpp == 4);

    Uint8 *p = (Uint8*)surface->pixels + y * surface->pitch + x * bpp;

    switch (bpp)
    {
    case 1:
        return *p;

    case 2:
        return *(Uint16*)p;

    case 4:
        return *(Uint32*)p;

    default:
        return 0;
    }
}

//----------------------------------------------------------------------------
bool SDL_TOOLS::inside(const Sint16 x, const Sint16 y, const SDL_Rect &r)
{
    return (r.x <= x) && (x < r.x + r.w) && (r.y <= y) && (y < r.y + r.h);
}

//----------------------------------------------------------------------------
bool SDL_TOOLS::intersect(const SDL_Rect &r1, const SDL_Rect &r2,
                          SDL_Rect &clip)
{
    // These pre-calculations will reduce the execution time of this function.
    Sint16 r1xw = r1.x + r1.w;
    Sint16 r1yh = r1.y + r1.h;
    Sint16 r2xw = r2.x + r2.w;
    Sint16 r2yh = r2.y + r2.h;
    Sint16 r1x = r1.x;
    Sint16 r1y = r1.y;
    Sint16 r2x = r2.x;
    Sint16 r2y = r2.y;

    if ((r1xw <= r2x) || (r1yh <= r2y) ||
        (r2xw <= r1x) || (r2yh <= r1y))
    {
        return false;
    }

    clip.x = MAX(r1x, r2x);
    clip.y = MAX(r1y, r2y);
    clip.w = MIN(r1xw, r2xw) - clip.x;
    clip.h = MIN(r1yh, r2yh) - clip.y;

    return true;
}

//----------------------------------------------------------------------------
void SDL_TOOLS::unite(const SDL_Rect &r1, const SDL_Rect &r2, SDL_Rect &u)
{
    // These helper variables will reduce the execution time a bit.
    Sint16 r1x = r1.x;
    Sint16 r1y = r1.y;
    Sint16 r2x = r2.x;
    Sint16 r2y = r2.y;

    if (r1.w == 0 && r1.h == 0)
    {
        u = r2;
    }
    else if (r2.w == 0 && r2.h == 0)
    {
        u = r1;
    }
    else
    {
        u.x = MIN(r1x, r2x);
        u.y = MIN(r1y, r2y);

        u.w = MAX(r1x+r1.w, r2x+r2.w) - u.x;
        u.h = MAX(r1y+r1.h, r2y+r2.h) - u.y;
    }
}



//----------------------------------------------------------------------------
static SDL_Surface *rotate8bpp_(const SDL_Surface *src, Uint16 angle)
    throw (SDLException)
{
    std::auto_ptr<SDL_Surface> dst;

    Uint8 *srcPixels = (Uint8*)src->pixels;
    Uint8 *dstPixels = NULL;

    Uint8 r, g, b;

    switch (angle)
    {
      case 0:
        dst.reset(SDL_CALLS::CreateRGBSurface(
                      SDL_SWSURFACE, src->w, src->h, 32));
        dstPixels = (Uint8*)dst->pixels;
        for (Sint32 y=0; y<src->h; y++)
        {
            for (Sint32 x=0; x<src->w; x++)
            {
                SDL_GetRGB(
                    *(srcPixels + y*src->pitch + x), src->format, &r, &g, &b);

                *(Uint32*)(dstPixels + y*dst->pitch + x*4) =
                    SDL_MapRGB(dst->format, r, g, b);
            }
        }
        break;

      case 90:
        dst.reset(SDL_CALLS::CreateRGBSurface(
                      SDL_SWSURFACE, src->h, src->w, 32));
        dstPixels = (Uint8*)dst->pixels;
        for (Sint32 y=0; y<src->h; y++)
        {
            for (Sint32 x=0; x<src->w; x++)
            {
                SDL_GetRGB(
                    *(srcPixels + y*src->pitch + x), src->format, &r, &g, &b);

                *(Uint32*)(dstPixels + x*dst->pitch + (dst->w-y-1)*4) =
                    SDL_MapRGB(dst->format, r, g, b);
            }
        }
        break;

      case 180:
        dst.reset(SDL_CALLS::CreateRGBSurface(
                      SDL_SWSURFACE, src->w, src->h, 32));
        dstPixels = (Uint8*)dst->pixels;
        for (Sint32 y=0; y<src->h; y++)
        {
            for (Sint32 x=0; x<src->w; x++)
            {
                SDL_GetRGB(
                    *(srcPixels + y*src->pitch + x), src->format, &r, &g, &b);

                *(Uint32*)(dstPixels + (dst->h-y-1)*dst->pitch + (dst->w-x-1)*4) =
                    SDL_MapRGB(dst->format, r, g, b);
            }
        }
        break;

      case 270:
        dst.reset(SDL_CALLS::CreateRGBSurface(
                      SDL_SWSURFACE, src->h, src->w, 32));
        dstPixels = (Uint8*)dst->pixels;
        for (Sint32 y=0; y<src->h; y++)
        {
            for (Sint32 x=0; x<src->w; x++)
            {
                SDL_GetRGB(
                    *(srcPixels + y*src->pitch + x), src->format, &r, &g, &b);

                *(Uint32*)(dstPixels + (dst->h-x-1)*dst->pitch + y*4) =
                    SDL_MapRGB(dst->format, r, g, b);
            }
        }
        break;

      default:
        throw SDLException("Unsupported angle");
    }

    return dst.release();
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_TOOLS::rotate(const SDL_Surface *src, Uint16 angle)
    throw (SDLException)
{
    switch (src->format->BitsPerPixel)
    {
      case 8:
        return rotate8bpp_(src, angle);

      default:
        throw SDLException("Unsupported source bpp");
    }
}

//----------------------------------------------------------------------------
bool SDL_TOOLS::isCollision(const SDL_Surface *s1, const SDL_Rect &p1,
                            const SDL_Surface *s2, const SDL_Rect &p2)
{
    SDL_Rect clip;
    if (SDL_TOOLS::intersect(p1, p2, clip))
    {
        // Only check the pixels in the intersection rectangle
        // of the two position rectangles.

        Uint32 xOffset1 = clip.x - p1.x;
        Uint32 yOffset1 = clip.y - p1.y;
        Uint32 xOffset2 = clip.x - p2.x;
        Uint32 yOffset2 = clip.y - p2.y;

        for (Uint16 y=0; y<clip.h; y++)
        {
            for (Uint16 x=0; x<clip.w; x++)
            {
                // @todo Calling getPixel() is slow, since internally,
                // the offset for s->pixels is calculated for every call.
                // Since this shouldn't be much of a bottleneck, we ignore it.

                Uint32 p1 = getPixel(s1, xOffset1 + x, yOffset1 + y);
                Uint32 p2 = getPixel(s2, xOffset2 + x, yOffset2 + y);
                if (p1 != 0 && p2 != 0)
                {
                    return true;
                }
            }
        }
    }

    return false;
}

//----------------------------------------------------------------------------
int SDL_TOOLS::getAngle(const SDL_Rect &p1, const SDL_Rect &p2)
{
    Sint16 x1 = p1.x + p1.w / 2;
    Sint16 y1 = p1.y + p1.h / 2;
    Sint16 x2 = p2.x + p2.w / 2;
    Sint16 y2 = p2.y + p2.h / 2;

    return MATH_TOOLS::getAngle(x1, y1, x2, y2);
}

//----------------------------------------------------------------------------
void SDL_TOOLS::getCentre(const SDL_Rect &r, Sint16 &x, Sint16 &y)
{
    x = r.x + r.w / 2;
    y = r.y + r.h / 2;
}

//----------------------------------------------------------------------------
SDL_Surface *SDL_TOOLS::loadBMPWithColorKey(const char *file,
                                            Uint8 r, Uint8 g, Uint8 b)
    throw (SDLException)
{
    SDL_Surface *s = SDL_CALLS::LoadBMP(file);
    SDL_CALLS::SetColorKey(
        s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, r, g, b));
    return s;
}

